/*
 * This is a VERY simple DDLINK compatible setup which should make a good
 * base for designing your own code.
 *
 * It is fairly minimal, perforing only the protocol/transfer functions.
 *
 * Compile "as is" with DDS Micro-C/DOS: cc DDLCMD -pof
 *
 * Dave Dunfield   -   https://dunfield.themindfactory.com
 */
#include <stdio.h>
#include <file.h>
// Debugging macros: easly add/remove debugging by add/delete '1'
//		Debug((...))	becomes:	//<nothing>
//		Debug1((...))	becomes:	printf(...);
#define	Debug(a)	//printf a;
#define	Debug1(a)	printf a;

// If your compiler required different definitions for variables,
// You can change these!
#define	Ushort	unsigned short	// 16 bit
#define	Uchar	unsigned char	// 8 bit
#define	ALL1	0xFFFF			// all 1 bits in Ushort

#define	PROTVER		4			// DDLINK protocol version
#define	BUFFMAX		4096		// Max data transfer buffer size

#define	O_SERVER	0x01

Ushort
	Sh, Sl, At, Ti, Da,			// FindFile info
	RDtop,						// Top of Remote DIR
	RXlen,						// Length of received packet
	DDlinkBsize = 1024,			// Line buffer size
	DDlinkRstart = 0x90,		// Paket Rx start char
	DDlinkTstart = 0x90;		// Paket TX start char
FILE
	*fp;
Uchar
	*Ptr,						// General pointer
	*Ptr1,						// ""
	Opt,
	Temp[128],					// Transient
	Drives[32],					// List of available drives
	Rdir[128],					// Remote directory
	Buffer[BUFFMAX+1];			// Transfer buffer (+1 for Zterm)

// DDLINK command codes - Refer to DDLINK.TXT for meanings!
#define	LINK_INIT	0x00		// 1:Xsize
#define	FIND_FIRST	0x01		// 1:mattrs-2,3:name...
#define	FIND_NEXT	0x02		// nothing
#define	FIND_REPLY	0x03		// 1:count-2,3:size-4,7:attr-2,9:time-2,11:date-2,13:name-13
#define	OPEN		0x04		// 1:mode-2,3:attr-2,5:time-2,7:date-2,9:name...
#define	OPEN_REPLY	0x05		// 1:handle-2
#define	CLOSE		0x06		// 1:handle-2
#define	CLOSE_REPLY	0x07		// 1:result-2
#define	READ		0x08		// 1:handle-2,3:size-2
#define	READ_REPLY	0x09		// 1:handle-2,3:size-2,5:data...
#define	WRITE		0x0A		// 1:handle-2,3:size-2,5:data...
#define	WRITE_REPLY	0x0B		// 1:size-2
#define	CHD			0x0C		// 1:path...
#define	CHD_REPLY	0x0D		// 1:result-2
#define	GETD		0x0E		// nothing
#define	GETD_REPLY	0x0F		// 1:result-2,2:path...
#define	DEL			0x10		// 1:path...
#define	DEL_REPLY	0x11		// 2:result
#define	REN			0x12		// 1:oldpath 257:newpath
#define	REN_REPLY	0x13		// 1:result-2
#define	RMDIR		0x14		// 1:path
#define	RMDIR_REPLY	0x15		// 1:result-2
#define	MKDIR		0x16		// 1:path
#define	MKDIR_REPLY	0x17		// 1:result-2
#define	SEEK		0x18		// 1:handle-2,3:high-2,5:low-2,7:mode-2
#define	SEEK_REPLY	0x19		// 1:result-2
#define	DRIVE		0x20		// nothing
#define	DRIVE_REPLY	0x21		// 1:result-2,3:drives...
#define	SELDR		0x22		// 1:drive-2
#define	SELDR_REPLY	0x23		// 1:Status
#define	LINK_CLOSE	0x24		// nothing

Uchar Help[] = {
	117,115,101,58,132,68,68,76,67,77,68,32,99,112,116,32,91,99,115,112,93,
	32,91,111,112,116,105,111,110,115,93,10,10,99,112,116,58,136,67,111,
	109,32,80,111,114,84,32,116,111,32,117,115,101,32,40,49,45,52,41,10,
	99,115,112,100,58,135,67,111,109,32,83,112,101,101,68,32,40,98,97,117,
	100,114,97,116,101,41,164,91,49,49,53,50,48,48,93,10,111,112,116,115,
	58,131,45,83,130,111,112,101,114,97,116,101,32,97,115,32,83,101,114,
	118,101,114,32,40,111,116,104,101,114,119,105,115,101,32,99,108,105,
	101,110,116,41,10,10,86,101,114,121,32,115,105,109,112,108,101,32,
	115,101,114,105,97,108,45,111,110,108,121,32,99,111,109,109,97,110,
	100,32,108,105,110,101,32,100,101,109,111,110,115,116,114,97,116,105,
	111,110,32,111,102,32,68,68,76,73,78,75,33,10,10,68,97,118,101,32,
	68,117,110,102,105,101,108,100,131,45,131,104,116,116,112,115,58,47,
	47,100,117,110,102,105,101,108,100,46,116,104,101,109,105,110,100,102,
	97,99,116,111,114,121,46,99,111,109,10,0 };

// Simple console output
void Pc(Uchar c)	{	putc(c, stdout);	}
void Ps(Uchar *p)	{	while(*p) Pc(*p++);	}
void Nl(void)		{	Pc('\n');			}

void Exit(void);	// Fwd ref: called in next section

/*
 * ----------------------------------------------------------------------
 * Serial communications is simplest!
 * - adjust as wanted to perform ethernet or parallel!
 * These functions are very specific to the serial communication library built
 * into my Micro-C/DOS compiler - you will need to essentially rewrite these
 * for a different system/compiler/library/
 */
#include <comm.h>
#define	DebugS(a)	//printf a;

Ushort
	TOcount,			// Timeout counter
	Cport,				// COM port to use
	Cspeed,				// COM baudrate
	Ctime = 18,			// Approx 1 second (55ms ticks)
	Crc,				// Computer CRC value
	CrcTable[256];		// pre-computed CRC calculation table

// Reduce timeout counter and Error if excessive
void Timeout(Ushort i)
{
	if(!--TOcount) {
		printf("?Timeout(%u)", i);
		Exit(); }
}

// Get character from serial port with timeout (Ctime)
Ushort CgetT(void)
{
	Ushort c, i, j, t;
	i = 0;
	for(;;) {
		if((c = Ctestc()) != -1) {	// Character received
			return c; }
		j = peekw(0x40, 0x6C) | 0x8000;	// Get BIOS tick
		if(!i) {						// First time
			i = j;						// indicate not first
			t = Ctime; }				// and get initial tick
		if(j != i) {					// Tick has occured
			if(!--t)					// count to second
				return ALL1;				// Indicate timeout
			i = j; } }					// Reset saved tick count
}

// Send a DDlink message - <start>{length}<data...>{CRC}
void DDlinkSend(Uchar data[], Ushort l)
{
	Ushort c, i;
	TOcount = 5;
a1:	DebugS(("T:%02x %u", DDlinkTstart, l))
	Crc = ALL1;
	Cputc(DDlinkTstart);
	Cputc(l >> 8);
	Cputc(l & 255);
	i = 0;
	while(i < l) {
		Cputc(c = data[i++]);
		DebugS((" %02x", c))
		Crc = CrcTable[Crc >> 8] ^ (Crc << 8) ^ c; }
	DebugS(("[%04x]", Crc))
	Cputc(Crc >> 8);
	Cputc(Crc & 255);
	// Wait for ack/nak
	for(;;) switch(c = CgetT()) {
		case ALL1	:	// Timeout
			Timeout(1);
			DebugS(("T\n"))
			goto a1;
		case 0x92:	//		ACK
		case 0x93:	//	or  NAK
			if((DDlinkTstart+2) == c) {	// ACK
				DDlinkTstart ^= 1;
				DebugS(("A\n"))
				return; }
			// Otherwise NAK
			Timeout(2);
			DebugS(("N\n"))
			goto a1; }
}

// Receive a DDlink message
Ushort DDlinkReceive(Uchar data[], Ushort t)
{
	Ushort s, c, i, l;
	TOcount = 5;
a1:	switch(s = CgetT()) {
	case ALL1:		// Timeout
		if(!--t)
			return 0;
	default	:
		goto a1;
	case 0x90:		// valid..
	case 0x91: ; }	// start
	DebugS(("R:%04x", s))
	if((c = CgetT()) == ALL1) goto nk;
	l = c << 8;	// Length.1
	if((c = CgetT()) == ALL1) goto nk;
	l |= c;		// Length.0
	DebugS((" %u", l))
	if(l >= DDlinkBsize)
		goto nk;
	Crc = ALL1;	// Initial CRC
	i = 0;
	while(i < l) {	// Get l data bytes
		if((c = CgetT()) == ALL1)	goto nk;
		Debug((" %02x", c))
		data[i++] = c;									// Store in buffer
		Crc = CrcTable[Crc >> 8] ^ (Crc << 8) ^ c; }	// Add to CRC
	DebugS(("[%04x", Crc))
	if((c = CgetT()) == ALL1) goto nk;
	i = c << 8;		// CRC.1
	if((c = CgetT()) == ALL1) goto nk;
	i |= c & 255;	// CRC.0
	DebugS(("=%04x]", i))
	if(i != Crc) {	// CRC mismatch - comm error
nk:		Timeout(3);
		DebugS(("!NK\n"))
n1:		l = 2;					// Wait for no data to
		i = peekw(0x40, 0x6C);	// clear input stream
n2:		s = i;					// Before sending NAK
		while(l) {
			if(Ctestc() != -1)
				goto n1;
			if((i = peekw(0x40, 0x6C)) != s) {
				--l;
				goto n2; } }
		Cputc((DDlinkRstart ^ 1) + 2);	// Send wrong ACK = NAK
		goto a1; }
	if(!*data)
		DDlinkRstart = s;
	if(DDlinkRstart != s)				// Wrong START
		goto nk;
	DebugS(("OK\n"))
	Cputc(DDlinkRstart + 2);			// Send ACK
	DDlinkRstart ^= 1;					// Toggle 0x90/0x91 start
	return l;
}

// Open the serial port for subsequent access
void DDlinkOpen(void)
{
	Ushort c, i, j;

	// Build the pre-computed CRC table
	for(i=0; i < 256; ++i) {
		c = i << 8;
		for(j=0; j < 8; ++j)
			c = ((c & 0x8000) ? 0x1021 : 0) ^ (c << 1);
		CrcTable[i] = c; }

	if(Copen(Cport, Cspeed, PAR_NO|DATA_8|STOP_1, SET_DTR|SET_RTS|OUTPUT_2)) {
		printf("COM%u: fail", Cport);
		Exit(); }
	Cport = 0;	// 0=invalid, so we know it was opened!
	disable();
	Cflags |= TRANSPARENT;	// Set for pure binary transfer (no XON/XOFF etc.)
	enable();
}

void DDlinkClose(void)
{
	if(!Cport)	// Port was successfully opened
		Cclose();
}

/*
 * ----------------------------------------------------------------------
 */

// Terminate and close communications
void Exit(void)
{
	Nl();
	DDlinkClose();
	exit(-1);
}

// Expect a certain type of message
void ExpectRX(Ushort t)
{
	RXlen = DDlinkReceive(Ptr = Buffer, 3);
	Debug(("[RX%02x %02x %u]", *Ptr, t, RXlen))
	if(*Ptr != t) {
		printf("Expected[%02x] got [%02x]", t, *Ptr);
		Exit(); }
	*(Ptr1 = Ptr++ + RXlen) = 0;
}
// Retreive 16-bit word from message
Ushort RXword(void)	// Retrieve word(16)
{
	Ushort v;
	v = (Ptr[1] << 8) | *Ptr;
	Ptr += 2;
	return v;
}
// Retrieve a string from message
void RXstring(Uchar *p)
{
	while(*p++ = *Ptr++);
}
// Retrieve a set number of bytes from message
void RXbytes(Uchar *p, Ushort l)
{
	while(l) {
		*p++ = *Ptr++;
		--l; }
}

// Start setting up a transmission
void StartTX(Uchar t)
{
	Ptr = Ptr1 = Buffer;
	*Ptr++ = t;
}
// Write a 16-bit word into transmit message
void TXword(Ushort v)
{
	*Ptr = v & 255;
	Ptr[1] = v >> 8;
	Ptr += 2;
}
// Write a zero-term string into transmit message
void TXstring(Uchar *p)
{
	do {
		*Ptr++ = *p; }
	while *p++;
}
// Write a set number of bytes into the transmit buffer
void TXbytes(Uchar *p, Ushort l)
{
	while(l) {				// This is used for...
		if(*Ptr++ = *p)		// fixed lenght strings
			++p;			// So stay at final zero!
		--l; }
}
// Transmit the message and receive response (if requested)
void EndTX(Ushort r)
{
	Debug(("[TX%02x]", *Buffer))
	DDlinkSend(Buffer, Ptr - Ptr1);
	if(r)
		ExpectRX(r);
}


// Skip to non-blank in input stream
int Skip(void)
{
	while(isspace(*Ptr))
		++Ptr;
	return *Ptr;
}

//Parse string from input stream
Ushort Parse(Uchar *p, Uchar *de)
{
	Ushort i;

	Skip(); i = 0;
a1:	switch(p[i] = toupper(*Ptr)) {
	default: ++Ptr; ++i;	goto a1;
	case ' ' :
	case '\t':
	case 0 : p[i] = 0; }
	if(de && !i) {
		if(*de == '!') {
			Ps(de+1);
			Nl();
			goto ex; }
		strcpy(p, de);
		i = strlen(p); }
ex:	return i;
}

// Show a "long" number value
// As Micro-C doesn't do 32 bit natively, use my "longdiv" from lubrary!
extern Ushort Longreg[];	// Has remainder after longdiv()
void ShowLN(Ushort h, Ushort l)
{
	Ushort i, lv[2];
	Uchar buf[13];
	static Ushort L10[] = { 10, 0 };	// "long" value 10 for longdiv()
	buf[i = sizeof(buf)-1] = 0;
	lv[1] = h;
	*lv = l;
	do {
		longdiv(lv, L10);
		buf[--i] = *Longreg + '0'; }
	while longtst(lv);
	while(i) buf[--i] = ' ';
	Ps(buf);
}

// Show a (DOS format) Date and Time
void ShowDT(Ushort da, Ushort ti)
{
	printf("%04u-%02u-%02u", (da>>9)+1980,	(da>>5)&15,		da&31);
	printf(" %3u:%02u:%02u", ti>>11, 		(ti>>5)&63,		(ti&31)*2);
}

// Detect/Show response error code
Ushort ShowERR(void)
{
	Ushort i;
	if(i = RXword())
		printf("Error(%u)\n", i);
	return i;
}

// Get current directory from remote end
void GetDir(void)
{
	StartTX(GETD);
	EndTX(GETD_REPLY);
	if(!ShowERR())
		strcpy(Rdir, Ptr);
	RDtop = strlen(Rdir);
}

// Cheak for valid filename (no PATH components)
Ushort CkFn(Uchar *p)
{
a1:	switch(*p++) {
	case ':':
	case'\\':
		Ps("Use \"drive:\" and \"CD ...\" - filename must not have \":\\\"\n");
		return 7;
	default	:	goto a1;
	case 0	:	; }
	return 0;
}

// Get attributes & timestamp from file on local system
Ushort GetLclFileInfo(Uchar *fn)
{
	Uchar tmp[128];
	return find_first(fn, 0x3F, tmp, &Sh, &Sl, &At, &Ti, &Da);
}

// Get attributes & timestamp from file on remote system
Ushort GetRmtFileInfo(Uchar *fn)
{
	Ushort i;
	Debug(("Grf'%s'", fn))
	StartTX(FIND_FIRST);
	TXword(0x3F);
	TXstring(fn);
	EndTX(FIND_REPLY);
	Sh = Sl = At = Ti = Da = 0;
	if(i = RXword()) {
		Sh = RXword();
		Sl = RXword();
		Ti = RXword();
		Da = RXword();
		At = *Ptr++; }
	Debug(("=%u'%s'%04x%04x %04x %04x %04x\n", i, Ptr, Sh, Sl, At, Ti, Da))
	return i;
}

Uchar *Commands[] = {
	"EXIT",			// 1
	"DRIVES",		// 2
	"DIRECTORY",	// 3
	"CD",			// 4
	"GET",			// 5
	"PUT",			// 6
	"DELETE",		// 7
	"RENAME",		// 8
	0 };

void DDlinkClient(void)
{
	Ushort i, j;
	Uchar c;

	Ps("DDLINK Client");
	StartTX(LINK_INIT);
	TXword(PROTVER);
	TXword(DDlinkBsize);
	EndTX(0);
	ExpectRX(LINK_INIT);
	if(RXword() != PROTVER) {
		printf("Protocol Versionm Mismatch!");
		Exit(); }
	if((i = RXword()) < DDlinkBsize)
		DDlinkBsize = i;

	GetDir();

	// Get list of drives on remote
	StartTX(DRIVE);
	EndTX(DRIVE_REPLY);
	Ps(" Drives: ");
	if(!ShowERR()) {
		strcpy(Drives, Ptr);
dr:		Ps(Drives);
		Nl(); }

	// Prompt for and perform a command
cm:	strcpy(Buffer, Rdir);
	if(RDtop > 3)	// Drop training '\\'
		Buffer[RDtop-1] = 0;
	Pc('<');	// So we can easily tell from std cmd prompt
	Ps(Buffer);
	Pc('>');
	fgets(Ptr = Buffer, 128, stdin);
	if(!Parse(Temp, 0))
		goto cm;		// Blank command

	// If "drive:" - set drive to access
	if((Temp[1] == ':') && !Temp[2]) {
		c = *Temp;
		i = 0;
		while(j = Drives[i++]) {
			if(j == c) {
				strcpy(Rdir, "::\\");
				*Rdir = c;
				RDtop = strlen(Rdir);
				goto cm; } }
		Ps("Bad drive\n");
		goto cm; }

	// Lookup command
	i = j = 0;
	while(Ptr1 = Commands[i++]) {
		if(strbeg(Ptr1, Temp)) {
			if(j) goto nc;	// Matches more than 1
			j = i; } }
	switch(j) {
	default	:
	// Command is not found
nc:		Ps("Unknown command \"");
		Ps(Temp);
		Ps("\" use:\n\t<drive>:\n");
		for(i = 0; Ptr = Commands[i]; ++i) {
			Pc('\t');
			Ps(Ptr);
			Nl(); }
		goto cm;
	case 2	:	// DRIVES
		goto dr;
	case 3	:	// DIRECTORY
		Parse(Temp, "*.*");	// Get filename, assume *.*
		if(CkFn(Temp))
			goto cm;
		strcpy(Rdir+RDtop, Temp);	// Add to remove directory

		StartTX(FIND_FIRST);	// Lookup matching files on remote
		TXword(0x3F);
		TXstring(Rdir);
		Rdir[RDtop] = 0;		// ZeroTerminate
a1:		EndTX(FIND_REPLY);
		if(i = RXword()) {		// Files to list
			while(i--) {		
				Sh = RXword();			// Get SizeHigh
				Sl = RXword();			// Get SizeLow
				Ti = RXword();			// Get Timestamp
				Da = RXword();			// Get Datestamp
				At = *Ptr++;			// Get attributes
				RXbytes(Temp, 13);		// GEt filename
				if(!(At & VOLUME)) {	// Don't show VOLUME entries
					ShowDT(Da, Ti);
					if(At & DIRECTORY)	// Indicate if directory
						printf("%-12s", " <DIR>");
					else
						ShowLN(Sh, Sl);	// Otherwise file (size)
					Pc(' ');
					Ps(Temp);			// Show the name
					Nl(); } }
			StartTX(FIND_NEXT);			// Get more if available
			goto a1; }
		goto cm;
	case 4	:	// CD
		if(Parse(Temp, 0)) {	// New name was given
			StartTX(CHD);
			TXstring(Temp);
			EndTX(CHD_REPLY);
			ShowERR(); }
		GetDir();				// Get actual new dir from remote
		goto cm;
	case 5	:	// GET (copy from) a file
		if(!Parse(Temp, "!Filename?"))
			goto cm;
		if(CkFn(Temp))
			goto cm;
		if(GetRmtFileInfo(Temp) != 1)
			goto cm;
		if(!(fp = fopen(Temp, "wbv")))
			goto cm;
		strcpy(Rdir+RDtop, Temp);
		StartTX(OPEN);
		TXword(0x0001);		// READ
		TXword(At);			// Attributes	\
		TXword(Ti);			// Timeime		 > In case write
		TXword(Da);			// Date			/
		TXstring(Rdir);		// Remote file path
		Rdir[RDtop] = 0;	// ZeroTerminate
		EndTX(OPEN_REPLY);
		if(!(Sh = RXword())) {	// !Handle == OpenFail
			Ps("Cannot open\n");
			goto cm; }
		j = DDlinkBsize-5;	// Max available transfer buffer
		do {
			StartTX(READ);
			TXword(Sh);		// Remote handle
			TXword(j);		// Requested bytes
			EndTX(READ_REPLY);
			RXword();			// Discard returned Handle
			if(Sl = RXword()) {	// #bytes
				fput(Ptr, Sl, fp);
				Pc('.'); }		// Indicatge progress
		} while(Sl == j);
		fclose(fp);
//		update_attributes(At & 0xFF);
//		update_timestamp(Da, Ti);
		StartTX(CLOSE);
		TXword(Sh);
		EndTX(CLOSE_REPLY);
		ShowERR();
		Nl();
		goto cm;
	case 6	:	// PUT (copy to) a file
		if(!Parse(Temp, "!Filename?"))
			goto cm;
		if(CkFn(Temp))
			goto cm;
		if(GetLclFileInfo(Temp))
			goto cm;
		if(!(fp = fopen(Temp, "rbv")))
			goto cm;
		strcpy(Rdir+RDtop, Temp);
		StartTX(OPEN);
		TXword(0x0002);		// WRITE
		TXword(At|0xC000);	// Tell Rmt to update attrs/time
		TXword(Ti);
		TXword(Da);
		TXstring(Rdir);
		Rdir[RDtop] = 0;	// ZeroTerminate
		EndTX(OPEN_REPLY);
		if(!(Sh = RXword())) {	// !Handle == OpenFail
			Ps("Cannot open\n");
			goto cm; }
		j = DDlinkBsize-5;	// Max available transfer buffer
		do {
			if(Sl = fget(Buffer+5, j, fp)) {	// Bytes from file
				StartTX(WRITE);
				TXword(Sh);		// Remote handle
				TXword(Sl);		// Bytes to write
				Ptr += Sl;		// Include data in TX buffer
				EndTX(WRITE_REPLY);
				if(ShowERR())
					break;
				Pc('.'); }		// Indicate progress
		} while(Sl == j);
		fclose(fp);
		StartTX(CLOSE);
		TXword(Sh);
		EndTX(CLOSE_REPLY);
		ShowERR();
		Nl();
		goto cm;
	case 7	:	// DELETE
		if(!Parse(Temp, "!Filename?"))
			goto cm;
		if(CkFn(Temp))
			goto cm;
		strcpy(Rdir+RDtop, Temp);
		StartTX(DEL);
		TXstring(Rdir);		// Remote path
		Rdir[RDtop] = 0;	// ZeroTerminate
		EndTX(DEL_REPLY);
		ShowERR();
		goto cm;
	case 8	:	// RENAME
		if(!Parse(Buffer+2048, "!Filename?"))
			goto cm;
		if(!Parse(Temp, "!NewName?"))
			goto cm;
		if(CkFn(Buffer+2048) || CkFn(Temp))
			goto cm;
		strcpy(Rdir+RDtop, Buffer+2048);
		StartTX(REN);
		TXstring(Rdir);
		TXstring(Temp);
		Rdir[RDtop] = 0;		// ZeroTerminate
		EndTX(REN_REPLY);
		ShowERR();
		goto cm;
	case 1	:	// EXIT
		StartTX(LINK_CLOSE);
		EndTX(0); }
}

// Wait for a received message, allow keyboard ESC to terminate server
Ushort WaitRX(Ushort t)
{
	while(!(RXlen = DDlinkReceive(Ptr = Buffer, t))) {
a1:		switch(kbtst()) {
		case 0x1B:				// ESC
			Ps("OperatorAbort!");
			Exit();
		default	:	goto a1;	// Any other key
		case 0	:	; } }		// No key
	return *Ptr++;
}

void DDlinkServer(void)
{
	Ushort i, j;

a1:	Ps("DDLINK Server (ESC to exit)");
	while(WaitRX(3));		// WAit for LINK_INIT
	i = RXword();			// Get protocol version
	DDlinkBsize = RXword();	// Get desired block size
	printf("Start %u %u", i, DDlinkBsize);

	StartTX(LINK_INIT);
	TXword(PROTVER);		// Send OUR protocol version
	TXword(DDlinkBsize);
	EndTX(0);
	if(i != PROTVER) {
		printf("ProtocolVersion!%u\n", PROTVER); 
		goto a1; }
	Nl();

	// Wait for server command & process
c1:	switch(i = WaitRX(3)) {
	case FIND_FIRST:
		i = RXword();		// Search attributes
		RXstring(Temp);		// Path
		printf("Find%x'%s'", i, Temp);
		j = 0;				// Count of files in message
		Ptr = Buffer + 3;				// Data block location
		Ptr1 = Buffer + DDlinkBsize;	// End of buffer
		if(!find_first(Temp, i, Temp, &Sh, &Sl, &At, &Ti, &Da)) {
c2:			TXword(Sh);
			TXword(Sl);
			TXword(Ti);
			TXword(Da);
			*Ptr++ = At;
			TXbytes(Temp, 13);
			++j;
			Debug(("\n%04x %04x'%s'%04x", Ptr-22, Ptr+22, Temp, Ptr1))
			if((Ptr + 22) < Ptr1) {		// Is buffer full?
				if(!find_next(Temp, &Sh, &Sl, &At, &Ti, &Da))
					goto c2; } }
c3:		*(Ptr1 = Buffer) = FIND_REPLY;	// Set up start of message...
		Buffer[1] = j & 255;			// Ptr1 so corrent TX lencth
		Buffer[2] = j >> 8;				// Send count of files
		EndTX(0);						// Data block already read
		printf("=%u\n", j);
		goto c1;
	case FIND_NEXT:
		Ps("FindNext");
		j = 0;
		Ptr = Buffer+3;
		Ptr1 = Buffer+DDlinkBsize;
		if(!find_next(Temp, &Sh, &Sl, &At, &Ti, &Da))
			goto c2;
		goto c3;
	case OPEN:
		i = RXword();		// Open mode
		At = RXword();		// attributes	\
		Ti = RXword();		// Time			 > in case Write
		Da = RXword();		// Date			/
		RXstring(Temp);		// Get filename
		Ptr = Buffer;
		if(i & 1) *Ptr++ = 'r';
		if(i & 2) *Ptr++ = 'w';
		*Ptr++ = 'b';
		if(i & 4) *Ptr++ = 'a';
		*Ptr++ = 'v';
		*Ptr = 0;
		printf("Open'%s'%s'", Temp, Buffer);
		fp = fopen(Temp, Buffer);
		printf("=%x\n", fp);
		StartTX(OPEN_REPLY);
		TXword(fp);
		EndTX(0);
		goto c1;
	case CLOSE:
		i = 7;
		if(fp = RXword()) {
			fclose(fp);
		//	if(At & 0x8000) update_attributes(At & 0xFF);
		//	if(At & 0x4000) update_timestamp(Da, Ti);
			i = 0; }
		StartTX(CLOSE_REPLY);
		TXword(i);
		EndTX(0);
		printf("Close %x=%u\n", fp, i);
		goto c1;
	case READ:
		fp = RXword();		// Handle
		i = RXword();		// requested #bytes
		StartTX(READ_REPLY);
		TXword(fp);			// Send handle
		TXword(j = fget(Buffer+5, i, fp));	// Read data, send length
		Ptr += j;			// Insure data gets sent
		EndTX(0);
		printf("Read %x(%u)=%u\n", fp, i, j);
		goto c1;
	case WRITE:
		fp = RXword();		// Handle
		i = RXword();		// Number of bytes to write
		j = fput(Buffer+5, i, fp);	// Write & get length
		printf("Write %x(%u)=%u\n", fp, i, j);
		StartTX(WRITE_REPLY);
		TXword(i != j);		// Error if not same
		EndTX(0);
		goto c1;
	case CHD:
		RXstring(Temp);			// Get new path
		if(Temp[1] == ':') {	// Drive was specified
			set_drive(i = toupper(*Temp) - 'A');
			if(get_drive() != i) {
				i = 7;			// Couldn't set drive
				goto cd1; } }
		i = cd(Temp);
cd1:	printf("CD'%s'=%u\n", Temp, i);
		StartTX(CHD_REPLY);
		TXword(i);
		EndTX(0);
		goto c1;
	case GETD:
		strcpy(Temp, "::\\");		// Include drive
		*Temp = get_drive() + 'A';	// ""
		getdir(Temp+3);				// ""
		printf("GetDir'%s'\n", Temp);
		StartTX(GETD_REPLY);
		TXstring(Temp);
		EndTX(0);
		goto c1;
	case DEL:
		RXstring(Temp);
		i = delete(Temp);
		printf("Del'%s'=%u\n", Temp, i);
		StartTX(DEL_REPLY);
		TXword(i);
		EndTX(0);
		goto c1;
	case REN:
		RXstring(Temp);
		RXstring(Buffer);
		i = j = 0;
r1:		switch(Buffer[i++]) {
		case ':':					// Remove leading PATH
		case'\\':	j = i;			// This demo does not..
		default	:	goto r1;		// MOVE files on rename!
		case 0	:	; }
		i = rename(Temp, Buffer+j);
		printf("Ren'%s'%s'=%u\n", Temp, Buffer+j, i);
		StartTX(REN_REPLY);
		TXword(i);
		EndTX(0);
		goto c1;
	case RMDIR:
		RXstring(Temp);
		i = rmdir(Temp);
		printf("RmDir'%s'=%u\n", Temp, i);
		StartTX(RMDIR_REPLY);
		TXword(i);
		EndTX(0);
		goto c1;
	case MKDIR:
		RXstring(Temp);
		i = mkdir(Temp);
		printf("MkDir'%s'=%u\n", Temp, i);
		StartTX(MKDIR_REPLY);
		TXword(i);
		EndTX(0);
		goto c1;
	case SEEK:
		fp = RXword();
		Sh = RXword();
		Sl = RXword();
		i = RXword();
		j = fseek(fp, Sh, Sl, i);
		printf("Seek %x(%04x%04x)%u=%u\n", fp, Sh, Sl, i, j);
		StartTX(SEEK_REPLY);
		TXword(j);
		EndTX(0);
		goto c1;
	case DRIVE:
		Ptr = Temp;
		j = get_drive();
		for(i=0; i < 26; ++i) {
			set_drive(i);
			if(get_drive() == i)
				*Ptr++ = i + 'A'; }
		set_drive(j);
		*Ptr = 0;
		printf("Drives'%s'\n", Ptr);
		StartTX(DRIVE_REPLY);
		TXword(0);
		TXstring(Temp);
		EndTX(0);
		goto c1;
	case SELDR:
		set_drive(i = toupper(RXword()) - 'A');
		j = get_drive() != i;
		printf("SelDir(%u)=%u\n", i, j);
		StartTX(SELDR_REPLY);
		TXword(j);
		EndTX(0);
		goto c1;
	case LINK_CLOSE:
		goto a1;	// Return to waiting for INIT
	} printf("Unknown request %02x\n", i);
	goto c1;
}

main(int argc, char *argv[])
{
	Ushort i;
	Uchar c;
	i = 0;
	while(++i < argc) {
		Ptr = Ptr1 = argv[i];
		if(isdigit(*Ptr)) {
			if(!Cport) {
				Cport = atoi(Ptr);
				continue; }
			if(!Cspeed) {
				while(*Ptr1) ++Ptr1;	// Drop last
				*(Ptr1-1) = 0;			// char. to keep
				Cspeed = atoi(Ptr);		// within 16bi
				continue; } }
		switch((*Ptr << 8) | toupper(Ptr[1])) {
		default	:	goto he;
		case '-S':
		case '/S':	c = O_SERVER;
		} Opt ^= c; }
/*ChtTxt R:\\Help.H
use:	DDLCMD cpt [csp] [options]

cpt:		Com PorT to use (1-4)
csp:		Com SPeed (baudrate)									[115200]
opts:	-S	operate as Server (otherwise client)

Very simple serial-only command line demonstration of DDLINK!

Dave Dunfield   -   https://dunfield.themindfactory.com
*/

	if((Cport > 4) || !Cport) {
he:		Ptr = Help;
		while(c = *Ptr++) {
			if(c & 0x80) {
				while(c-- & 0x7F)
					Pc(' ');
				continue; }
			Pc(c); }
		return; }

	if(Cspeed)
		Cspeed = 11520 / Cspeed;
	else
		Cspeed = _115200;

	printf("COM%u: %u0\n", Cport, 11520/Cspeed);

	DDlinkOpen();
	if(Opt & O_SERVER)
		DDlinkServer();
	else
		DDlinkClient();
	DDlinkClose();
}
