/*
 * Emulator & PC EXection TeST
 *
 * This is a little "quick and dirty" tool I wrote while evaluating some 8086
 * PC/DOS emulators, to get a feel for how closely they could approximate the
 * "real time" of my actual PC systems.
 *
 * Compie using	 > cht EPCEXTST			<- Build Help.h header file
 * DDS Micro-C/PC>  cc EPCEXTST -pof	<- Compile to EPCEXTST.COM
 *
 * Dave Dunfield   -   https://dunfield.themindfactory.com
 */
#include <stdio.h>
#include <window.h>
#define	Debug(a)	//printf a;
#define	Debug1(a)	printf a;

#define	LOGFILE	"EPCEXTST.LOG"
#define	MAXT	256		// Max ticks

#define	CT_DIG	0x10	// Char is a decimal digit	0-9
#define	CT_HEX	0x20	// Char is a hex digit		A-F

#define	O_TICKS 0x01	// Show ticks
#define	O_SUMM	0x02	// "" summary
#define	O_INST	0x04	// Add extra instructuons
#define	O_WLOG	0x08	// Write log

unsigned
	H, L,			// High/Low count
	XTword,			// If needed by XTtick fns
	COMadr,			// COM port address
	COMspd = 1152,	// 115200/baud
	Ntime,			// Number of seconds to count
	TimeT,			// Time top of list
	TimeH[MAXT],	// High times
	TimeL[MAXT];	// Low times
FILE
	*fpo;			// Output fp
unsigned char
	*Ptr,			// General pointer
	*Sptr,			// System name pointer
	*Optr,			// Enter options
	*Wfile = "wvq",	// Write File
	Opt = O_TICKS|O_SUMM,	// Command options
	Xtest = 1,		// Exec test counter
	Rtest,			// Reduce test
	XTbyte,			// If needed by XTtick fns
	CT[256],		// CharType table
	OPtxt[128],		// Option text
	Sname[128];		// System name
unsigned
	L10[] = { 10, 0 },
extern unsigned Longreg[];

//ChtTxt R:\Help.h
#include "R:\\Help.h"

// Output Char/String/Newline to stdout (& fpo)
void Pc(unsigned char c)
{
	putc(c, stdout);
	if(fpo) putc(c, fpo);
}
void Ps(unsigned char *p)	{	while(*p) Pc(*p++);	}
void Nl(void)				{	Pc('\n');			}

void DOhelp(void)
{
	unsigned char *p, c;
	p = Chelp;
	while(c = *p++) {
		if(c & 0x80) {
			while(c-- & 0x7F)
				Pc(' ');
			continue; }
		Pc(c); }
	exit(0);
}

//Print error message and terminate
register Error(unsigned args)
{
	unsigned char buf[200];
	_format_(nargs()*2+&args, buf);
	wclose();
//	if(Line) printf("%u: ", Line);
	Ps(buf);
	exit(-1);
}

// Set character type table entries - main used for Dec-Hex conversion
void SetCT(unsigned s, unsigned e, unsigned char c)
{
	while(s <= e) {
		if((CT[s++] = c) & (CT_DIG|CT_HEX))
			++c; }
}

// Get a value (CT_DIG[|CT_HEX])
unsigned Value(unsigned char f)
{
	unsigned v, m;
	unsigned char *p, c;
	p = Ptr;
	m = (f & CT_HEX) ? 16 : 10;
	v = 0;
	while((c = CT[*Ptr]) & f) {
		Debug(("(%04x*", v))
		v = (v * m) | (c & 15);
		Debug(("%u)+%u=%04x %u\n", m, (c & 15), v, v))
		++Ptr; }
	if((Ptr == p) || *Ptr)
		Error("?Value(%u)%s",m, p);
	return v;
}

// Initialize COM port
unsigned COMinit(void) asm
{
		MOV		DX,_COMadr			; Get address
		CLI							; Inhibit interrupts
; Setup the uart
		DEC		DX					; Backup ...
		DEC		DX					; to line control register (FB)
		IN		AL,DX				; Read current value
		OR		AL,80h				; Enable baud rate register
		OUT		DX,AL				; Write it
		MOV		AX,DGRP:_COMspd		; Get baud rate		//?
		MOV		BX,AX				; Save for test
		SUB		DX,3				; Point to baud rate LSB (F8)
		OUT		DX,AL				; Write it
		INC		DX					; Advance to MSB (F9)
		MOV		AL,AH				; Get MSB
		OUT		DX,AL				; Write it
		DEC		DX					; Backup to LSB (F8)
		IN		AL,DX				; Re-read LSB
		MOV		AH,AL				; Copy for later
		INC		DX					; Back to MSB (F9)
		IN		AL,DX				; Re-read MSB
		XCHG	AH,AL				; Swap for multi
		SUB		AX,BX				; Does it match			115200/baud
		JNZ		cfail				; No
		MOV		AL,3				; Get mode				PAR_NO|STOP_1|DATA_8=3
		INC		DX					; Advance...
		INC		DX					; to line control register (FB)
		OUT		DX,AL				; Write it
		DEC		DX					; Backup ...
		DEC		DX					; to Interrupt enable register (F9)
		MOV		AL,11				; Get modem control		RTS|DTR|OUT2=11
		ADD		DX,3				; Point to modem control register (FC)
		OUT		DX,AL				; Write it
; Clear out any pending characters
		SUB		DX,4				; Point to data register (F8)
		IN		AL,DX				; Read to clear interrupt
		IN		AL,DX				; Read to clear interrupt
		XOR		AX,AX				; Success
cfail:	STI
}

// Flush any pending RX from COM port
void COMflush(void) asm
{
cf1:	XOR		AH,AH				; count 256
cf2:	MOV		DX,DGRP:_COMadr		; Get COM port
		IN		AL,DX				; Get status
		TEST	AL,00000001b		; RX ready?
		JZ		cf3					; No
		SUB		DX,4				; Point to data
		IN		AL,DX				; Clear RX char
		JMP short cf1				; Reset & for next
cf3:	DEC		AH					; Reduce count
		JNZ		cf2					; Keep clearing
}

// TickFunctions to guage passage of 1 second
asm {
; Read RTC seconds
XTrtc:	MOV		DX,0070h			; RTS control
		XOR		AX,AX				; Seconds
		OUT		DX,AL				; Select
		INC		DX					; RTC data
		IN		AL,DX				; Read seconds
		TEST	AL,80h				; Invalid?
//		JNZ		XTrtc				; Ignore
		JNZ		r0					; Ignore
		CMP		AL,DGRP:_XTbyte		; Changed?
		JZ		r0					; No
		MOV		DGRP:_XTbyte,AL		; Save updated
		MOV		AL,7				; !0
		RET
; Read BIOS clock ticl E000:006C
XTbios:	MOV		AX,0040h			; BIOS data seg
		MOV		ES,AX				; Address it
		MOV		BX,006Ch			; BIOS clock tick
		MOV		AX,ES:[BX]			; Read ""
		CMP		AX,DGRP:_XTword		; Changed?
		JZ		r0					; No
		MOV		DGRP:_XTword,AX		; Save update
		DEC		byte ptr DGRP:_XTbyte ; Count 18/seg
		JNZ		r0					; Not yet
		MOV		AL,18				; Reset count (&!0)
		MOV		DGRP:_XTbyte,AL		; Resave
		RET
r0:		XOR		AX,AX				; No "tick" return0
		RET
; Read DOS clock
XTdos:	MOV		AH,2Ch				; get TIME
		INT		21h					; Ask DOS
		CMP		DH,DGRP:_XTbyte		; Changed?
		JZ		r0					; No
		MOV		DGRP:_XTbyte,DH		; Save update
		MOV		AL,7				; !0
		RET
; Read COM loopback (100 baud = 10/sec)
XTcom:	MOV		DX,DGRP:_COMadr		; Get COM port
		IN		AL,DX				; Get status
		MOV		AH,AL				; Copy
		SUB		DX,5				; Point to data
		TEST	AL,00100000b		; TX ready?
		JZ		XTcom1				; No
		MOV		AL,055h				; send char
		OUT		DX,AL				; Write
XTcom1:	TEST	AH,00000001b		; RX ready?
		JZ		r0					; No
		IN		AL,DX				; Clear RX char
		DEC		byte ptr DGRP:_XTbyte ;Count 10/sec
		JNZ		r0					; Not yet
		MOV		AL,10				; Reset
		MOV		DGRP:_XTbyte,AL		; Reset count
		RET
}

// Test for time "tick"
unsigned TestTick(void) asm
{
		DEC		DGRP:_Xtest			; Not reduced?
		JZ		tt1					; Perform test
		XOR		AX,AX				; Fake 0
		POP		BP
		RET
tt1:	MOV		AL,DGRP:_Rtest		; Get set -R
		MOV		DGRP:_Xtest,AL		; Reset
tt2:	DB		0E8h,0,0			; CALL (patched)
}

// Returns !0 if TickFunction set
unsigned TFtst(void) asm
{
	MOV	SI,offset DGRP:tt2		// CALL instruction address
	MOV	AX,1[SI]				// Return operand (offset)
}
// Set TickFunction (to address in AX)
void TFset(void)
{
	unsigned xa;
	unsigned char *xp;
	xa = nargs()-3;					// eXecuton address (adjust for CALL size)
	asm" MOV AX,offset DGRP:tt2";	// CALL address
	xp = nargs();
	if(TFtst()) DOhelp();			// Already set
	xp[1] = (xa -= xp);				// Calc offset & patch low byte
	xp[2] = xa>>8;					// Patch high byte
}

// Perform misc. extra 8086 instructions
// Just to perform misc. instructions - no meaningful result
unsigned extra_inst(void)
{
	unsigned a, b, c;
	a = 5;
	b = ((a+5) * 7)<<5;
	c = (b / (a+8))>>(a-2);
	a = a ^ b ^ c;
	return (TimeH[TimeT & a] | a) + (TimeL[TimeT & a] & a);
}

// Display a 32-bit "long" number
void ShowLN(unsigned h, unsigned l)
{
	unsigned i, lv[2];
	unsigned char buf[13];
	buf[i = sizeof(buf)-1] = 0;
	lv[1] = h;
	*lv = l;
	do {
		longdiv(lv, L10);
		buf[--i] = *Longreg + '0'; }
	while longtst(lv);
	while(i) buf[--i] = ' ';
	Ps(buf);
}

main(int argc, char *argv[])
{
	unsigned i, j, k, l1[2], l2[2];
	unsigned char c;

	SetCT('0', '9', CT_DIG);
	SetCT('A', 'F', CT_HEX|0x0A);
	SetCT('a', 'f', CT_HEX|0x0A);
	Sptr = Sname;
	Optr = OPtxt;
	i = 0;
	while(++i < argc) {
		if(*(Ptr = argv[i]) == '-') {
			++Ptr;
o1:			j = 0;
			while(*Optr) ++Optr;
			switch(toupper(*Ptr++)) {
			default	:	DOhelp();
/*ChtTxt Chelp
Emulator / PC  EXecution TeST

use:	EPCEXTST	[name...] [-options]
  opts:	 -1		count seconds by COM1		\
		 -2				""		  ""2		 >
		 -3				""		  ""3		 >	!MUST! select!
		 -4				""		  ""4		 > ONE and ONLY ONE
		 -B				""		 BIOS tick	 >		method
		 -D				""		 DOS time	 >
		 -H				""		 HardwareRTC/
		 -I		extra Instruction/loop		-S		don't show	Summary
											-T			""		Ticks
		 -L		write Log: EPCEXTST.LOG		-Nval	Number of seconds/tests[10]
		 -A		Append		  ""			-Rval	Reduce time test freq.  [1]
Performs a fairly simple loop test to see how many "loops" can be executed
 per second on an 8086 PC/DOS emulator (or real PC).

NOTE: This was a "quick and dirty" test I wrote while evaluating a few PC/DOS
 emulators - The calculated loop counts have NO MEANING other than to compare
 with those from different "systems" run with the SAME command options!
				*** For more information:  read  EPCEXTST.TXT ***

Dave Dunfield   -   https://dunfield.themindfactory.com~
*/
			case 'B':	*Optr = 'B';
				asm" MOV AX,offset DGRP:XTbios";
				TFset();
				XTbyte = 7;
				goto o3;
			case 'D':	*Optr = 'D';
				asm" MOV AX,offset DGRP:XTdos";
				TFset();
				goto o3;
			case 'H':	*Optr = 'H';
				asm" MOV AX,offset DGRP:XTrtc";
				TFset();
				goto o3;
			case '1':	*Optr = '1';
				j = 0x3FD;
				goto o2;
			case '2':	*Optr = '2';
				j = 0x2FD;
				goto o2;
			case '3':	*Optr = '3';
				j = 0x3ED;
				goto o2;
			case '4':	*Optr = '4';
				j = 0x2ED;
o2:				asm" MOV AX,offset DGRP:XTcom";
				TFset();
				XTbyte = 7;
o3:				COMadr = j;
				break;
			case 'N':
				if(Ntime) DOhelp();
				Ntime = Value(CT_DIG);
				continue;
			case 'R':
				if(Rtest) DOhelp();
				Rtest = Value(CT_DIG);
				sprintf(Optr, "R%u", Rtest);
				continue;
			case 'I':	*Optr = 'I';
				c = O_INST;
				goto o4;
			case 'S':	c = O_SUMM;		goto o4;
			case 'T':	c = O_TICKS;	goto o4;
			case 'A':	Wfile = "wavq";
			case 'L':	c = O_WLOG;
o4:				Opt ^= c; }
			if(*Ptr) goto o1;
			continue; }
		if(*Sname) *Sptr++ = ' ';
		strcpy(Sptr, Ptr);
		while(*++Sptr); }

	if(!TFtst()) {			// No method set
		Ptr = Chelp;
		while(c = *Ptr++) {
			if(c & 0x80) {
				while(c-- & 0x7F)
					putc(' ', stdout);
				continue; }
			putc(c, stdout); }
		return; }

	if(!Ntime)			Ntime = 10;
	if(!Rtest)			Rtest = 1;
	if(Ntime > MAXT)	Ntime = MAXT;

	Debug(("[E:%x C:%x N:%u R:%u]", XTexe, COMadr, Ntime, Rtest))

	if(COMadr) {			// Open and flush COM port
		if(COMinit())
			Error("?ComPort");
		COMflush(); }

	if(Opt & O_TICKS) {		// Temp screen for tick display
		wopen(0, 0, 80, 25, WSAVE|WCOPEN|WWRAP|0x07);
		wprintf("EPCEXTST: Counting %u seconds ESC to abort", Ntime);
		wupdatexy(); }

	// Wait a few "ticks" to insure it's stable
	i = 3;
	c = 55;
	do {
		if(TestTick()) {
			--i;
			goto a1; }
		else if((j = peekw(0x40, 0x6C)) != k) {
			k = j;
			if(!--c)
				Error("?NoTick");
a1:			if(kbtst() == 0x1B) {
				wclose();
				return; } }
	} while(i);

	if(Opt & O_TICKS) {	// Newline to show when ticks actually start
		wputc('\n');
		wupdatexy(); }

	for(;;) {				// Loop for Ntime seconds
		if(TestTick()) {		// One second has passed
			if(Opt & O_TICKS)		// Show ticks
				wputc('*');
			if(TimeT >= Ntime)		// Finished
				break;
			TimeH[TimeT]	= H;	// Save this
			TimeL[TimeT++]	= L;	// seconds count
			H = L = 0;
			if(kbtst() == 0x1B)		// Abort?
				break;
			continue; }
		if(Opt & O_INST)			// Extra instructions
			extra_inst();
		if(!++L) ++H; }				// Advance count
	wclose();

	if(Opt & O_WLOG)		// Writing log
		fpo = fopen(LOGFILE, Wfile);
	Ps("EPCEXTST:");		// Show title in log
	if(*Sname) {
		Pc(' ');
		Ps(Sname); }
	Nl();

	longset(l1, 0);			// Walk saves counts and average
	for(i=0; i < TimeT; ++i) {
		l2[1] = TimeH[i];
		*l2 = TimeL[i];
		if(Opt & O_SUMM) {		// Show summary
			ShowLN(l1[1], *l1);
			Pc('+');
			ShowLN(l2[1], *l2);
			Nl(); }
		longadd(l1, l2); }
	longset(l2, TimeT);
	ShowLN(l1[1], *l1);			// Total
	Pc('/');
	ShowLN(l2[1], *l2);			// Ntime
	longdiv(l1, l2);
	Pc('=');
	ShowLN(l1[1], *l1);			// Average
	Ps("  -");
	Ps(OPtxt);
	Nl();

	if(fpo)
		fclose(fpo);
}
