/*
 * CALLS - Display graph of function calls within a C program.
 *
 * This program examines a C source file, and records the function calls
 * made by every function. It then displays the function(s) you request
 * in the form of a tree with the calls made by each function nested
 * within it's definition.
 *
 * The parsing of the source file is fairly simple. Function calls through
 * pointers or other complex expressions are not tracked.
 *
 * It is assumed that the C source file being examined is error-free, ie:
 * compiles without error. In particular: duplicate function names, mis-
 * matched () & {}, unterminated comments, character values and strings
 * are not detected (but will invalidate the output).
 *
 * Copyright 1997-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc calls
 */
#include <stdio.h>

/* Fixed design parameters */
#define	MAX_FILES	25			/* Maximum number of files on command line */
#define	MAX_FUNCS	25			/* Maximum number of funcs on command line */
#define	MAX_GRAPH	5000		/* Maximum # entries in the call graph */
#define	MAX_NAMES	2000		/* Maximum number of function names */
#define	NAME_SPACE	25000		/* Storage space for function names */
#define	NEST_DEPTH	100			/* Maximum nesting of functions */
#define	UNGET_DEPTH	10			/* Maximum depth to unget characters */
#define	SYMBOL_SIZE	50			/* Maximum size of a symbol name */

/*
 * Call graph, format:
 *  position[0]  = Count of called functions
 *  position[1]  = Index into names[] of this function
 *  position[2+] = Index into names[] of each called function
 *  ... = Repeats for each function
 */
char call_graph[MAX_GRAPH];
unsigned call_top = 0, current_node;

/*
 * List of function names & text storage area + housekeeping pointers
 */
char *names[MAX_NAMES], pool[NAME_SPACE], *pptr = &pool;
unsigned name_top = 0;

/*
 * Unget character pending buffer + pending count
 */
char pending_input[UNGET_DEPTH], pcount = 0;

/*
 * Misc housekeeping variables
 */
char in_function = 0;
unsigned brace_count = 0, line_count = 0;

/*
 * Command line options
 */
char dup = -1, line = -1, verbose = 0, debug = 0, cppc = -1;
unsigned column = 40, indent = 2;
FILE *input_fp;

/*
 * List of reserved words used by Micro-C itself
 */
static char *reserved[] = {
	"int", "unsigned", "char", "static", "extern", "register",
	"if", "else", "while", "do", "for", "switch", "case",
	"default", "return", "break", "continue", "goto", "sizeof",
	"asm", "struct", "union", "void" };			/* end of table */

/*
 * Command line help text
 */
static char help_text[] = { "\
Use:	CALLS <filenames...> [options]\n\n\
Opts:	-B	- Display call graph building records\n\
	-C	- Disable C++ style // comment detection\n\
	-D	- Remove duplicate calls within a function\n\
	-L	- Do not display line numbers\n\
	-V	- Verbose: Expand all call nodes (no back references)\n\
	C=n	- Start comments in column n (0=Free format, default=40)\n\
	F=name	- Display call graph for function 'name' (default=main)\n\
	I=n	- Indent call displays by n (default=2)\n" };

/*
 * Skip to the end of a comment in the input stream
 */
void skip_comment()
{
	int c;
	for(;;) switch((pcount) ? pending_input[--pcount] : getc(input_fp)) {
		case '*' :
			c = (pcount) ? pending_input[--pcount] : getc(input_fp);
			if(c == '/')
				return;
			pending_input[pcount++] = c;
			continue;
		case '\n' : ++line_count; continue;
		case EOF : return; }
}

/*
 * Skip a string in the input stream looking for specific terminator char,
 * and ignoring "escaped" (\c) characters.
 */
void skip_escape(char t)
{
	int c;
	for(;;) switch(c = (pcount) ? pending_input[--pcount] : getc(input_fp)) {
		case '\n' : ++line_count; continue;
		case '\\' :
			c = (pcount) ? pending_input[--pcount] : getc(input_fp);
			if(c == '\n')
				++line_count;
			continue;
		case EOF :
			return;
		default:
			if(c == t)
				return; }
}

/*
 * Skip a block that can nest
 */
void skip_nest(char s, char e)
{
	int c;
	for(;;) switch(c = (pcount) ? pending_input[--pcount] : getc(input_fp)) {
		case '\n' : ++line_count; continue;
		case '{' : ++brace_count; continue;
		case '}' : --brace_count; continue;
		case '\'' :
		case '\"' :
			skip_escape(c);
			continue;
		case EOF :
			return;
		default:
			if(c == s) {
				skip_nest(s, e);
				continue; }
			if(c == e)
				return; }
}

/*
 * Get a character from the input stream...
 * Ignore comments & characters in quotes
 * keep track of lines and braces
 */
int get_char()
{
	int c, d;
reload:
	c = (pcount) ? pending_input[--pcount] : getc(input_fp);
	switch(c) {
		case '/' :		/* Comment, skip to next */
			d = (pcount) ? pending_input[--pcount] : getc(input_fp);
			if(d == '*') {				/* Standard comment */
				skip_comment();
				goto reload; }
			if((d == '/') && cppc) {	/* C++ style comment */
				do d=(pcount) ? pending_input[--pcount] : getc(input_fp);
				while(d != '\n');
				goto reload; }
			pending_input[pcount++] = d;
			break;
		case '\'' :		/* Single quote, skip to next */
		case '\"' :		/* Double quote, skip to next */
			skip_escape(c);
			goto reload;
		case '\n' : ++line_count; break;
		case '{' : ++brace_count; break;
		case '}' : --brace_count; }
	return c;
}

/*
 * Return a character to the input stream
 */
void unget_char(char c)
{
	switch(c) {
		case '\n' : --line_count; break;
		case '{' : --brace_count; break;
		case '}' : ++brace_count; }
	pending_input[pcount++] = c;
}

/*
 * Get next symbol name from the input stream
 */
int get_next_symbol(char *ptr)
{
	int c;

	while((c = get_char()) != EOF) {
		if(isalpha(c) || (c == '_')) {
			do
				*ptr++ = c;
			while(isalnum(c = get_char()) || (c == '_'));
			unget_char(c);
			*ptr = 0;
			return -1; } }
	return 0;
}

/*
 * Get next non-blank character from the input stream
 */
int get_next_char()
{
	int c;
	for(;;) switch(c = get_char()) {
		default:
			return c;
		case ' ' : 
		case '\t' : 
		case '\n' : }
}

/*
 * Search for a name in the name pool. If not found, add it
 * in either case return index to name.
 */
unsigned locate_name(char *name)
{
	unsigned i;
	for(i=0; i < name_top; ++i)
		if(!strcmp(name, names[i]))
			return i;

	names[name_top] = pptr;
	do; while(*pptr++ = *name++);
	return name_top++;
}

/*
 * Display a number of spaces
 */
void space(unsigned n)
{
	while(n--)
		putc(' ', stdout);
}

/*
 * Display function & called functions
 */
void show_function(unsigned f)
{
	unsigned i, j, n;
	int l;
	static unsigned nest_level = 0, func_count = 0, line_num = 0;
	static unsigned nest_func[NEST_DEPTH], func_log[MAX_NAMES],
		func_def[MAX_NAMES];

	/* Display line & function name */
	++line_num;
	if(line)
		printf("%05u: ", line_num);
	space(l = nest_level*indent);
	printf("%s", names[f]);

	/* Calculate indent width for comments */
	l = column - strlen(names[f]) - l;
	if(l < 0)
		l = 0;

	/* See if this function has been displayed before */
	for(i=j=0; j < func_count; ++j)
		if(func_log[j] == f)
			goto lookup;
	/* New definition, log it and indicate expansion required */
	func_def[func_count] = line_num;
	func_log[func_count++] = f;
	j = -1;

lookup:
	/* Locate this function definition in the call graph */
	n = call_graph[i++];
	if(call_graph[i++] != f) {
		if((i += n) < call_top)
			goto lookup;
		/* Not found, flag as an external definition */
		space(l);
		printf(" = external\n");
		return; }

	/* For verbose mode, we have to detect recursive loops */
	if(verbose) {
		for(j=0; j < nest_level; ++j)
			if(nest_func[j] == f) {
				space(l);
				printf(" = recursive call\n");
				return; } }
	/* For non-verbose mode, only display reference if already logged */
	else {
		if((j != -1) && n) {
			space(l);
			printf(" = ^ %u ^\n", func_def[j]);
			return; } }

	/* Expand function display by showing all called functions */
	printf("\n");
	nest_func[nest_level++] = f;
	while(n--)
		show_function(call_graph[i++]);
	--nest_level;
}

/*
 * Process a file & add to the call graph
 */
void process_file(char *filename)
{
	int c;
	unsigned n;
	char buffer[SYMBOL_SIZE];


	input_fp = fopen(filename, "rvq");
	in_function = brace_count = line_count = 0;

skip:
	/* Process each symbol from the file & identify functions */
	while(get_next_symbol(buffer)) {
		/* Filter out any reserved words */
		for(n=0; n < sizeof(reserved)/sizeof(reserved[0]); ++n)
			if(!strcmp(reserved[n], buffer))
				goto skip;
		/* Only a function call if followed by '(' */
		if((c = get_next_char()) != '(') {
			unget_char(c);
			continue; }
		/* If at level 0 (definition), terminate previous processing */
		if(in_function && !brace_count) {
			call_graph[current_node] = c = (call_top - current_node) - 2;
			if(debug)
				printf(" %u call(s).\n", c);
			in_function = 0; }

		/* Insert name into dictionary & skip arguments */
		n = locate_name(buffer);
		skip_nest('(', ')');

		/* If in level 0, this is a definition, begin processing */
		if(!in_function) {
			c = get_next_char();
			if((c == ';') || (c == ',')) {	/* Not an actual definition */
				unget_char(c);
				continue; }
			in_function = -1;
			current_node = call_top++;
			call_graph[call_top++] = n;
			if(debug)
				printf("Function: %s\n", names[n]);
			continue; }

		/* Filter duplicates if enabled */
		if(!dup) {
			for(c = current_node+2; c < call_top; ++c)
				if(call_graph[c] == n)
					goto skip; }

		if(debug)
			printf(" Call to: %s\n", names[n]);
		call_graph[call_top++] = n; }

	/* Terminate processing of last definition */
	if(in_function) {
		call_graph[current_node] = c = (call_top - current_node) - 2;
		if(debug)
			printf(" %u call(s).\n", c); }

	fclose(input_fp);
}

/*
 * Search for a name in the name pool
 */
main(int argc, char *argv[])
{
	unsigned i;
	char *ptr, *file_list[MAX_FILES], *func_list[MAX_FUNCS];
	static unsigned file_top = 0, func_top = 0;

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case '-B' : debug = -1;						continue;
			case '-C' : cppc = 0;						continue;
			case '-D' : dup = 0;						continue;
			case '-L' : line = 0;						continue;
			case '-V' : verbose = -1;					continue;
			case 'C=' : column = atoi(ptr);				continue;
			case 'F=' : func_list[func_top++] = ptr;	continue;
			case 'I=' : indent = atoi(ptr);				continue; }
		file_list[file_top++] = argv[i]; }

	printf("C Function Calls - Copyright 1997-2003 Dave Dunfield. All rights reserved.\n\n");

	if(!file_top)			/* If no files given, display help */
		abort(help_text);

	if(!func_top)			/* If no functions given, default to "main" */
		func_list[func_top++] = "main";

	for(i=0; i < file_top; ++i)		/* Process each file */
		process_file(file_list[i]);

	for(i=0; i < func_top; ++i)		/* Display each function */
		show_function(locate_name(func_list[i]));
}
