;
; Dunfield MicroScope - Target Kernel for: 68HC11
;
; Copyright 2001-2005 Dave Dunfield - All rights reserved.
;
; Memory addresses
INTREGS	EQU	$0000		; Internal registers
ROM	EQU	$E000		; Monitor code goes here
RAM	EQU	$0080		; Monitor data goes here
INTVEC	EQU	$7FD6		; Revector interrupts
STACK	EQU	RAM+$1F		; Stack goes here
; Internal registers
SCIBAUD	EQU	INTREGS+$2B	; SCI Baudrate
SCICCR1	EQU	INTREGS+$2C	; SCI Control reg 1
SCICCR2	EQU	INTREGS+$2D	; SCI Control reg 2
SCISTAT	EQU	INTREGS+$2E	; SCI Status Port
SCIDATA	EQU	INTREGS+$2F	; SCI Data Port
CONFIG	EQU	INTREGS+$3F	; CONFIG register
; Command codes from host to target monitor
CQUERY	EQU	$A0		; Query command
CBREAK	EQU	$A1		; Break command
CEXEC	EQU	$A2		; Execute command
CSTEP	EQU	$A3		; Single-step command
CREAD	EQU	$B0		; Read Program memory
CWRITE	EQU	$B8		; Write program memory
; Response codes from target monitor to host
RESP1	EQU	$AA		; Query response 1
RESP2	EQU	$55		; Query response 2
BRKF1	EQU	$90		; Break leadin-1
BRKF2	EQU	$A5		; Break leadin-2
BRKF3	EQU	$B9		; Break leadin-3
KID	EQU	$6811		; Kernel ID
;
	ORG	ROM		; Code goes here
;
; Beginning of monitor code
;
BEGIN:	LDS	#STACK		; Initial stack
; CONFIG of 68HC711D3 must be written at startup
	LDAA	#%00000100	; Disable COP and internal ROM
	STAA	CONFIG		; Write config register
; Initialize the SCI device
	LDAA	#%00110000	; SCP=3, SCR=0 : 9600 @ 8Mhz
	STAA	SCIBAUD		; Set up baud rate
	CLR	SCICCR1		; 8 bit data format
	LDAA	#%00001100	; No ints, enable TX & RX
	STAA	SCICCR2		; Write to SCI
; Loop, reading data till it times out
flush:	BSR	getchr		; Get data
	BRA	flush		; till timeout
;
; Breakpoint has occured
;
SWIHND:	LDAB	#1		; Adjust by 1 for breakpoint
swih0	LDAA	#BRKF1		; Breakpoint flag1
	BSR	wrchr		; Write it
	LDAA	#BRKF2		; Breakpoint flag2
	BSR	wrchr		; Write it
	LDAA	#BRKF3		; Breakpoint flag3
	BSR	wrchr		; Write it
	TBA			; Get adjust value
	BSR	wrchr		; Send it
	LDAB	#9		; Pop/send 9 bytes
swih1:	PULA			; Get byte from stack
	BSR	wrchr		; Send to host
	DECB			; Reduce count
	BNE	swih1		; Get them all
	TSX			; X = stack
	XGDX			; D = stack
	BSR	wrchr		; Send stack high
	TBA			; Get stack low
	BSR	wrchr		; Send stack low
	BRA	main		; re-enter monitor
;
; Write character to serial port
;
wrchr:	PSHA			; Save data
wrch1	LDAA	SCISTAT		; Read SCI status
	BITA	#%10000000	; TX ready?
	BEQ	wrch1		; No, wait for it
	PULA			; Restore data
	STAA	SCIDATA		; Write character
	RTS
;
; Get an address into the X register
;
getaddr	BSR	getchr		; Get character
	TAB			; B = LOW word
	BSR	getchr		; Get character
	XGDX			; X = address
	RTS
;
; Get a character from the serial port with timeout
; Re-enter monitor if timeout
;
Getchr:	LDY	#0		; Set timeout
getch1	LDAA	SCISTAT		; Get SCI status
	BITA	#%00100000	; RX ready?
	BEQ	getch2		; No, it's not
	LDAA	SCIDATA		; Read the data
	RTS			; And return
getch2	DEY			; Reduce count
	BNE	getch1		; Get next
;
; Main monitor command loop
;
main:	LDS	#STACK		; Point to stack
main1:	BSR	getchr		; Read a character
;
; Query command
;
	CMPA	#CQUERY		; Query command?
	BNE	mread		; No, try next
	LDX	#QRESP		; Point to response
qloop	LDAA	,X		; Get byte
	BEQ	main1		; All done
	BSR	wrchr		; Write it
	INX			; Skip to next
	BRA	qloop		; Do them all
qresp:	FCB	RESP1,RESP2,=KID,KID,1,0
;
; Read memory
;
mread:	CMPA	#CREAD		; Read memory?
	BNE	mwrite		; No, try next
	BSR	getaddr		; Get memory address
	BSR	getchr		; Get size
	TAB			; B = size
mread1:	LDAA	,X		; Get data
	BSR	wrchr		; Send it
	INX			; Skip to next
	DECB			; Reduce count
	BNE	mread1		; Do them all
	BRA	main1		; Exit
;
; Write memory
;
mwrite:	CMPA	#CWRITE		; Write memory?
	BNE	break		; No, try next
	BSR	getaddr		; Get memory address
	BSR	getchr		; Get size
	TAB			; B = size
mwrit1:	BSR	getchr		; Get character
	STAA	,X		; Save it
	INX			; Skip to next
	DECB			; Reduce count
	BNE	mwrit1		; Do them all
	BRA	main1		; Exit
;
; Set a breakpoint
;
break:	CMPA	#CBREAK		; Set breakpoint?
	BNE	exec		; No, try next
	BSR	getaddr		; Get address
	LDAA	,X		; Get data byte
	BSR	wrchr		; Set it
	LDAA	#$3F		; 'SWI' = breakpoint
	STAA	,X		; Write to memory
	SUBA	,X		; Test for valid
	BSR	wrchr		; Write it
	BRA	main1		; Exit
;
; Begin program execution
;
exec:	CMPA	#CEXEC		; Execute command?
	BNE	main1		; No, try next
	BSR	getaddr		; Get SP
	TXS			; Set user SP
	LDAB	#9		; Get 9 bytes
exec1:	BSR	getchr		; Get data from PC
	PSHA			; Stack it
	DECB			; Reduce count
	BNE	exec1		; Get/stack them all
	RTI			; Execute user program
;
; Interrupt handlers
;
FFD6	CLRB			; SCI
	BRA	REVECT
FFD8	LDAB	#1		; SPI
	BRA	REVECT
FFDA	LDAB	#2		; Pulse ACC Input
	BRA	REVECT
FFDC	LDAB	#3		; Pulse ACC overflow
	BRA	REVECT
FFDE	LDAB	#4		; Timer overflow
	BRA	REVECT
FFE0	LDAB	#5		; Timer IC4/OC5
	BRA	REVECT
FFE2	LDAB	#6		; Timer out comp 4
	BRA	REVECT
FFE4	LDAB	#7		; Timer out comp 3
	BRA	REVECT
FFE6	LDAB	#8		; Timer out comp 2
	BRA	REVECT
FFE8	LDAB	#9		; Timer out comp 1
	BRA	REVECT
FFEA	LDAB	#10		; Timer inp comp 3
	BRA	REVECT
FFEC	LDAB	#11		; Timer inp comp 2
	BRA	REVECT
FFEE	LDAB	#12		; Timer inp comp 1
	BRA	REVECT
FFF0	LDAB	#13		; Real-Time
	BRA	REVECT
FFF2	LDAB	#14		; IRQ
	BRA	REVECT
FFF4	LDAB	#15		; XIRQ
	BRA	REVECT
FFF6	LDAB	#16		; SWI
	BRA	REVECT
FFF8	LDAB	#17		; Illegal opcode
;
; ReVector interrupt
;
REVECT:	LDX	#INTVEC		; Point to vectors
	ABX			; Offset to vector
	ABX			; Offset to vector
	LDX	,X		; Get address
	BEQ	UNEXINT		; Unexpected interrupt
	PSHX			; Stack for return
	TSX			; Point to stack
	LDAB	3,X		; Restore B
	LDX	5,X		; Restore X
	RTS			; Execute handler
; Interrupt was not expected - enter monitor
UNEXINT	LDAB	#0		; No offset
	JMP	swih0		; And proceed
;
; Interrupt vectors
;
	ORG	ROM+$1FD6
	FDB	FFD6		; SCI
	FDB	FFD8		; SPI
	FDB	FFDA		; Pulse ACC Input
	FDB	FFDC		; Pulse ACC overflow
	FDB	FFDE		; Timer overflow
	FDB	FFE0		; Timer IC4/OC5
	FDB	FFE2		; Timer out comp 4
	FDB	FFE4		; Timer out comp 3
	FDB	FFE6		; Timer out comp 2
	FDB	FFE8		; Timer out comp 1
	FDB	FFEA		; Timer inp comp 3
	FDB	FFEC		; Timer inp comp 2
	FDB	FFEE		; Timer inp comp 1
	FDB	FFF0		; Real-Time
	FDB	FFF2		; IRQ
	FDB	FFF4		; XIRQ
	FDB	SWIHND		; SWI (Breakpoint)
	FDB	FFF8		; Illegal opcode
	FDB	BEGIN		; FFFA COP failure
	FDB	BEGIN		; FFFC CLOCK monitor
	FDB	BEGIN		; FFFE Reset
