/*
 * Map File Generator - version 2.1
 *
 * This program reads a DDS listing file, and generates a .MAP file
 * containing information about file names, line number/address mapping,
 * and detailed information on all symbols (local & global).
 *
 * Copyright 1994-2005 Dave Dunfield
 * Freely distributable.
 *
 * Permission is granted to modify and distribute this program without
 * payment or royalty, as long as the original copyright notices are not
 * removed. ** Portability notes are identified by **
 *
 * Conditionals using _MICROC_ are provided which allow compilation with
 * either Micro-C/PC, or Turbo-C 2.0 - You may need to modify these areas
 * for use with a different compiler.
 *
 * Compile in SMALL model:
 *  Micro-C: cc mapgen2 -fop m=S
 *  Turbo-C: tcc -ms mapgen2
 */
#include <stdio.h>

/* Misc. MAPGEN parameters */
#define	LINE_SIZE	250		/* Maximum size of input line */
#define	MAX_FILES	75		/* Maximum number of files to handle */
#define	MAX_SYMB	5000	/* Maximum number of symbols */
#define	MAX_DIMS	500		/* Maximum number of array dimensions */

/* Offsets into listing lines */
#define	OPCODE	7			/* Opcode bytes begin here */
#define	SOURCE	32			/* Source text begins here */

/*
 * Bits found in the "type" entry of symbol table, also
 * used on the expression stack to keep track of element types,
 * and passed to the code generation routines to identify types.
 *
 * The POINTER definition indicates the level of pointer indirection,
 * and should contain all otherwise unused bits, in the least significant
 * (rightmost) bits of the word.
 *
 * MAPGEN reuses the REFERENCE bit to indicate a LOCAL symbol.
 */
#define REFERENCE	0x8000		/* symbol has been referenced */
#define INITED		0x4000		/* symbol is initialized */
#define ARGUMENT	0x2000		/* symbol is a function argument */
#define EXTERNAL	0x1000		/* external attribute */
#define STATIC		0x0800		/* static attribute */
#define REGISTER	0x0400		/* register attribute */
#define	TVOID		0x0200		/* void attribute */
#define UNSIGNED	0x0100		/* unsigned attribute */
#define BYTE		0x0080		/* 0=16 bit, 1=8 bit */
#define	CONSTANT	0x0040		/* constant attribute */
#define	SYMTYPE		0x0030		/* symbol type (see below) */
#define ARRAY		0x0008		/* symbol is an array */
#define POINTER		0x0007		/* level of pointer indirection */

/*
 * Symbol type designator bits (SYMTYPE field above)
 */
#define	VARIABLE	0x00		/* Symbol is a simple variable */
#define	MEMBER		0x10		/* Symbol is a structure member */
#define	STRUCTURE	0x20		/* Symbol is a structure template */
#define	FUNCGOTO	0x30		/* Symbol is a function */

/*
 * Filename table	- tracks files occuring in the listing
 */
unsigned fcount = 0,			/* Count of filenames */
	current_file = 0,			/* File currently active */
	filename[MAX_FILES];		/* File name (offset) */

/*
 * Symbol table 	- tracks symbols occuring in the listing
 */
unsigned scount,				/* Count of symbols */
	name[MAX_SYMB],				/* Symbol name (offset) */
	type[MAX_SYMB],				/* Symbol type */
	addr[MAX_SYMB],				/* Symbol address/offset */
	value1[MAX_SYMB],			/* F:StkAlloc V:Segment */
	value2[MAX_SYMB];			/* F:EndAddr  V:Array=DimIdx Struct=Size*/
unsigned char
	file[MAX_SYMB];				/* Index to source file name */

#define	UNKNOWN	0x80			/* Marks symbol as not known */

/*
 * Dimension table	- tracks dimensions of symbols with array type
 */
unsigned dim_pool[MAX_DIMS], dcount = 0;

/*
 * Command line argument switches
 */
char
	amap = -1,		// Map assembly files
	cmap = -1,		// Map C files
	quiet = 0,		// Limit output
	ucase = 0,		// Translate to upper case
	dup = 0,		// Allow duplicate line records
	equ = 0,		// Inhibit equate records
	colon = 0,		// Do not map symbols ending in ':'
	list = 0;		// Output debug listing

/*
 * Misc. global variables
 */
unsigned string_seg,			/* External string segment */
	mapseg = 0,					/* Current "map" segment in file */
	lst_line = 0,				/* Current line number in listing */
	acount = 0;					/* Count of address/line records */
char buffer[LINE_SIZE+1];		/* Line input buffer */
FILE *fpi, *fpo;				/* Input & Output file pointers */


#ifndef _MICROC_
#define	peek	peekb
#define	poke	pokeb
/*
 ** If your compiler puts '\n' on the end of lines read by fgets()
 ** (this will be evidenced by newlines after every filename in the
 ** output file)... Use this version:
 */
static char *fgets(char *dest, int size, FILE *fp)
{
	register int c;
	register char *ptr;

	ptr = dest;
	while(--size) {
		if((c = getc(fp)) < 0) {
			if(ptr == dest)
				dest = 0;
			break; }
		if(c == '\n')
			break;
		*ptr++ = c; }
	*ptr = 0;
	return dest;
}
#endif

/*
 ** Save a string into the external pool
 */
static unsigned save_string(char *name)
{
	unsigned p;
	static unsigned string_top = 0;

	p = string_top;
	do
		poke(string_seg, string_top++, *name);
	while(*name++);
	return p;
}

/*
 ** Retrieve a string from the external pool
 */
static char *get_string(unsigned i)
{
	char *ptr;
	static char temp[65];

	ptr = temp;		/* Point to our copy */
	while(*ptr++ = peek(string_seg, i++));
	return temp;
}

/*
 * Test for uppercase alphabetic
 */
static isupper(char c)
{
	return (c >= 'A') && (c <= 'Z');
}

/*
 * Test for space character
 */
static isspace(char c)
{
	return (c == ' ') || (c == '\t');
}

/*
 * Test for numeric digit
 */
static isdigit(char c)
{
	return (c >= '0') && (c <= '9');
}

/*
 * Test for a valid symbol name
 */
static issymbol(char *ptr)
{
	char c;
	c = *ptr;
	do {
		if( ((c < 'a') || (c > 'z'))
		&&	((c < 'A') || (c > 'Z'))
		&&	((c < '0') || (c > '9'))
		&&	(c != '_') ) {
			if((c != ':') || colon)
				return 0; } }
	while(!isspace(c = *++ptr));
	return 1;
}

/*
 * Test for string1 beginning with string2
 */
static strbeg(char *str1, char *str2)
{
	while(*str2)
		if(*str1++ != *str2++)
			return 0;
	return 1;
}

/*
 * Get a number in specified base with digit limit
 */
static get_number(char *ptr, unsigned base, unsigned digits)
{
	unsigned value, c;

	value = 0;
	while(isspace(*ptr))
		++ptr;
	do {
		if(isdigit(c = *ptr))			/* convert numeric digits */
			c -= '0';
		else if(c >= 'a')				/* convert lower case alphabetics */
			c -= ('a' - 10);
		else if(c >= 'A')				/* convert upper case alphabetics */
			c -= ('A' - 10);
		else							/* not a valid "digit" */
			break;
		if(c >= base)					/* outside of base */
			break;
		value = (value * base) + c;		/* include in total */
		++ptr; }
	while(--digits);					/* enforce maximum digits */
	return value;
}

/*
 * Copy string up to a delimiter
 */
static char *copy_parse(char *dest, char *source)
{
	while(isspace(*source))
		++source;
	if(ucase) {
		while(*source && !isspace(*source))
			*dest++ = toupper(*source++); }
	else {
		while(*source && !isspace(*source))
			*dest++ = *source++; }
	if((*(dest-1) == ':') && !colon)
		--dest;
	*dest = 0;
	return source;
}

/*
 * Find a global symbol
 */
static unsigned find_global(char *s)
{
	unsigned i;
	i = 0;
	while(i < scount) {
		if(!strcmp(s, get_string(name[i])))				/* found! */
			return i+1;
		if((type[i]&(SYMTYPE|REFERENCE)) == FUNCGOTO) {		/* Function - skip locals */
			while(type[++i] & REFERENCE);
			continue; }
		++i; }
	return 0;
}

/*
 * First pass... Define all files and symbols
 */
static first_pass()
{
	unsigned i, j;
	char *ptr;
	char temp[65], temp1[10];
	static unsigned last_func;

top:
	if(!fgets(buffer, LINE_SIZE, fpi))			/* End of input */
		return;
	if(*buffer<' ') goto top;					/* Blank line */
	if(strbeg(buffer, "DUNFIELD")) goto top;	/* Header line */
	if(strbeg(buffer, "SYMBOL")) return;		/* End of listing */

	if(*(buffer+SOURCE) == '*') {				/* Comment line */
		if(isdigit(*(ptr=buffer+(SOURCE+1)))) {	/* 'C' line number */
			goto top; }
		if(*ptr != '#') {						/* ASM comment */
			current_file = 0;
			goto top; }
		if(strbeg(++ptr, "file ")) {			/* C file name */
			copy_parse(temp, buffer+(SOURCE+7));
			for(current_file=0; current_file < fcount; ++current_file)
				if(!strcmp(get_string(filename[current_file]), temp))
					goto top;
			filename[fcount++] = save_string(temp);
			goto top; }
		if(strbeg(ptr, "link")) {				/* Linked file */
			current_file = 0;
			goto top; }
		if(strbeg(ptr, "map")) {				/* Map segment */
			mapseg = atoi(ptr+3);
			goto top; }
		if(strbeg(ptr, "cpu")) {				/* CPU identifier */
			goto top; }
		if(strbeg(ptr, "fun")) {				/* Function beginning */
			ptr = copy_parse(temp, ptr+3);
			name[scount] = save_string(temp);
			file[scount] = current_file;
			ptr = copy_parse(temp1, ptr);
			type[last_func = scount] = FUNCGOTO;
			value1[scount++] = atoi(temp1);
			goto top; }
		if(strbeg(ptr, "end")) {				/* Function ending */
			value2[last_func] = get_number(buffer, 16, 4) - 1;
			goto top; }
		if(strbeg(ptr, "lcl")) {				/* Local variable */
			ptr = copy_parse(temp, ptr+3);
			ptr = copy_parse(temp1, ptr);
			i = atoi(temp1) | REFERENCE;
			if((i & SYMTYPE) == MEMBER)
				*temp |= 0x80;
			name[j = scount] = save_string(temp);
			ptr = copy_parse(temp1, ptr);
			value2[scount] = addr[scount] = atoi(temp1);
			++scount;
			goto setsym; }
		if(strbeg(ptr, "gbl")) {				/* Global variable */
			ptr = copy_parse(temp, ptr+3);
			ptr = copy_parse(temp1, ptr);
			i = atoi(temp1) & ~REFERENCE;
			if((i & SYMTYPE) == MEMBER)
				*temp |= 0x80;
			ptr = copy_parse(temp1, ptr);
			value2[scount] = atoi(temp1);
			if(j = find_global(temp))
				--j;
			else
				name[j = scount++] = save_string(temp);
		setsym:
			file[j] = current_file;
			if((type[j] = i) & ARRAY) {
				ptr = copy_parse(temp1, ptr);
				value2[j] = dcount;
				dim_pool[dcount++] = i = atoi(temp1);
				while(i--) {
					ptr = copy_parse(temp1, ptr);
					dim_pool[dcount++] = atoi(temp1); } }
			goto top; } }

	/*
 	 * Global symbol - make/amend symbol table entry
	 */
	if(issymbol(ptr = buffer+SOURCE)) {
		copy_parse(temp, ptr);
		if(j = find_global(temp)) {			/* Already defined */
			addr[--j] = get_number(buffer, 16, 4);
			if((type[j] & (SYMTYPE|REFERENCE)) != FUNCGOTO)
				value1[j] = mapseg; }
		else {
			name[scount] = save_string(temp);
			file[scount] = current_file | UNKNOWN;	/* New symbol */
			addr[scount] = get_number(buffer, 16, 4);
			value1[scount] = mapseg;
			type[scount++] = 0; 	/* Unknown */ } }

	/*
	 * Set addresses for static symbols
	 */
	if(*ptr == '?') {		/* Possible static */
		ptr = buffer+(SOURCE+1);
		if(isupper(*ptr) && isupper(ptr[1]) && isdigit(ptr[2])) {
			ptr += 3;
			while(isdigit(*ptr)) ++ptr;
			if(issymbol(ptr)) {				/* Looks like a static */
				copy_parse(temp, buffer+SOURCE);
				for(i=0; i < scount; ++i)
					if(!strcmp(get_string(name[i]), temp)) {
						addr[i] = get_number(buffer, 16, 4);
						if((type[i] & (SYMTYPE|REFERENCE)) != FUNCGOTO)
							value1[i] = mapseg;
						break; } } } }
	goto top;
}

/*
 * Second pass... Generate address/line-number records
 */
static dump_address()
{
	char *ptr;
	char temp[50], cflag;
	unsigned a, l, cline, last_cline;

	mapseg = current_file = cflag = last_cline = 0;

top:
	if(!fgets(buffer, LINE_SIZE, fpi))			/* End of input */
		return;
	++lst_line;
	if(*buffer<' ') goto top;					/* Blank line */
	if(strbeg(buffer, "DUNFIELD")) goto top;	/* Header line */
	if(strbeg(buffer, "SYMBOL")) return;		/* End of listing */

	if(*(buffer+SOURCE) == '*') {				/* Comment line */
		if(isdigit(*(ptr=buffer+(SOURCE+1)))) {	/* 'C' line number */
			cline = get_number(buffer+(SOURCE+1), 10, 0);
			cflag = cmap;
			goto top; }
		if(*ptr != '#') {						/* ASM comment */
			current_file = 0;
			goto top; }
		if(strbeg(++ptr, "file ")) {			/* C file name */
			copy_parse(temp, buffer+(SOURCE+7));
			for(current_file=0; current_file < fcount; ++current_file)
				if(!strcmp(get_string(filename[current_file]), temp))
					break;
			goto top; }
		if(strbeg(ptr, "link")) {				/* Linked file */
			current_file = 0;
			goto top; }
		if(strbeg(ptr, "map")) {				/* Map segment */
			mapseg = atoi(ptr+3);
			goto top; }
		goto top; }

	if(*(buffer+OPCODE) == ' ')
		goto top;

	if(current_file) {
		if(!cflag) {
			a = get_number(buffer, 16, 4);
			fprintf(fpo, "A%04x 1 %u\n", a, lst_line);
			goto top; }
		if((cline == last_cline) && !dup)
			goto top;
		last_cline = l = cline; }
	else {
		if(!amap)
			goto top;
		l = lst_line; }
	a = get_number(buffer, 16, 4);
	if(list)
		printf("@%04x: F:%03u L:%u\n", a, current_file, l);
	++acount;

/**************************************************/
/* Dump an address/line record to the output file */
/**************************************************/
	fprintf(fpo, "A%04x %u %u\n",
		a, current_file+1, l);

	cflag = 0;
	goto top;
}

/**********************************************/
/* Dump symbol definitions to the output file */
/**********************************************/
static dump_symbols()
{
	int i, j, k, t, st, func;
	char *ptr, c;
	static char types[] = { "VMSF" };

	for(i=0; i < scount; ++i) {
		ptr = get_string(name[i]);
		*ptr &= 0x7f;
		st = (t = type[i]) & SYMTYPE;
		j = file[i];
		if(list)
			printf("S%-4u: F:%03u T:%04x A:%04x 1:%04x 2:%04x %s\n",
				i, j & ~UNKNOWN, t, addr[i], value1[i], value2[i], ptr);
		if((j & UNKNOWN) && equ)		/* Do not dump unknown's */
			continue;
		if(*ptr == '?') {				/* Trim prefix on static symbol */
			ptr += 3;
			while(isdigit(*ptr)) ++ptr; }
		c = (t & REGISTER) ? '3' : '2';
		if(j & UNKNOWN)
			fprintf(fpo, "C%s %u %04x ;Unknown\n", ptr, j & ~UNKNOWN, addr[i]);
		else switch(st) {
		case MEMBER :
			fprintf(fpo, "C%s %u %u ;Member\n", ptr, j, value2[i]);
			break;
		case STRUCTURE :
//			fprintf(fpo, "%c%s %u 04x ;Structure", c, ptr, j, value2[i]);
//			break;
		case VARIABLE :
			if(t & REFERENCE) {			// LOCAL
				if(t & STATIC) {
					fprintf(fpo, "%c%s %u %04x %04x %04x ;Static local\n", c, ptr, j,
						addr[i], addr[func], value2[func]); }
				else {
					fprintf(fpo, "S%s %u %04x %04x %04x ;Stack local\n", ptr, j,
						addr[i], addr[func], value2[func]); } }
			else {						// GLOBAL
				fprintf(fpo, "%c%s %u %04x ;Global\n", c, ptr, j, addr[i]); }
			break;
		case FUNCGOTO :
			if(t & REFERENCE) {			// GOTO
				fprintf(fpo, "1%s %u %04x %04x %04x ;Goto\n", ptr, j, addr[i],
					addr[func], value2[func]); }
			else {
				func = i;
				fprintf(fpo, "1%s %u %04x ;Func\n", ptr, j, addr[i]); } } }

}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i;
	char *ptr, c;

	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {	/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'A' : amap = 0;	break;
				case 'C' : cmap = 0;	break;
				case 'D' : dup = -1;	break;
				case 'E' : equ = -1;	break;
				case 'L' : list = -1;	break;
				case 'Q' : quiet = -1;	break;
				case 'U' : ucase = -1;	break;
				case ':' : colon = -1;	break;
				default: goto badopt; }
			continue; }
	badopt:
		printf("Invalid option: %s\n", argv[i]);
		exit(-1); }

	if(!quiet)
		printf("Map file generator, version 2.0\n");

	if(argc < 2) {
		printf("\nUse: MAPGEN <filename> [-acdelqu]\n\nCopyright 1994-2005 Dave Dunfield\nFreely distributable.\n\n");

/*
 ** Please provide indication of MAP file supported below
 */
		printf("Modified by Dave Dunfield to support: DDS MicroScope\n");
		exit(-1); }

/*
 ** Allocate a 64K external segment for string storage
 */
#ifdef	_MICROC_
	if(!(string_seg = alloc_seg(4096))) {
#else
	if(allocmem(4096, &string_seg) != -1) {
#endif
		printf("Out of memory\n");
		exit(-1); }

	/* Parse filename & extension from passed argument */
	copy_parse(ptr = buffer, argv[1]);
	while(c = *ptr) {
		if(c == '.')
			goto noext;
		++ptr; }
	copy_parse(ptr, ".LST");
noext:

	/* Parse filenames */
	filename[fcount++] = save_string(buffer);

/*
 * Open input and output files
 *
 ** Most 'C' compilers do not support MICRO-Cs 'v'erbose and 'q'uit
 ** options on fopen().
 **
 ** If your output .MAP file is in BINARY format, don't forget to
 ** add the 'b'inary option to the second fopen() statement.
 */
#ifdef _MICROC_
	fpi = fopen(buffer, "rvq");
	copy_parse(ptr, ".MAP");
	fpo = fopen(buffer, "wvq");
#else
	if(!(fpi = fopen(buffer, "r"))) {
		printf("Cannot open input file: %s\n", buffer);
		exit(-2); }
	copy_parse(ptr, ".MAP");
	if(!(fpo = fopen(buffer, "w"))) {
		printf("Cannot open output file: %s\n", buffer);
		exit(-2); }
#endif

	first_pass();		/* Evaluate all symbols */

	rewind(fpi);

/************************************************************************/
/* Output .MAP file information. These steps can be moved around if you */
/* need the files/symbols/address records output in a different order   */
/************************************************************************/

	/* Write pre-amble needed for file */
	fprintf(fpo, ";Example ASCII MAP file\n");

	/* Dump filenames to the output file */
	for(i=0; i < fcount; ++i) {
		if(list)
			printf("F%-4u: %s\n", i, get_string(filename[i]));
		fprintf(fpo, "F%s\n", get_string(filename[i])); }

	/* Dump symbol definitions to the output file */
	dump_symbols();

	/* Dump address records to the output file */
	dump_address();

	fclose(fpo);
	fclose(fpi);

	if(!quiet)
		printf("%u files, %u symbols, %u dimensions, %u address entries\n",
			fcount, scount, dcount, acount);
}
