/*
 * ArmOS - Hardware layout
 *
 * Dave Dunfield - 98/10/04
 *	03/10/29 - Added GPIOs
 *  03/12/09 - New PLDs - updated PCBREV/ID type & locations
 */

#ifndef BOOT_MODE
	#define	BOOT_MODE	1
#endif

#ifndef __HARDWARE_H_
	#define __HARDWARE_H_

	/*
	 * Main data memory
	 */
	#define	DATA_RAM_START	0x00804000

	/*
	 * ARM System registers
	 */
	#define SYSCON1	*(volatile unsigned		*)0x80000100
	#define SYSCON2	*(volatile unsigned		*)0x80001100
	#define SYSCON3	*(volatile unsigned		*)0x80002200
	#define	SYSFLG1	*(volatile unsigned		*)0x80000140
	#define SYNCIO	*(volatile unsigned		*)0x80000500
	#define MEMCFG1	*(volatile unsigned		*)0x80000180
	#define MEMCFG2	*(volatile unsigned		*)0x800001C0
	#define SDCONF	*(volatile unsigned		*)0x80002300
	#define SDRFPR	*(volatile unsigned		*)0x80002340
	#define UBRLCR1	*(volatile unsigned		*)0x800004C0
	#define	UARTD	*(volatile unsigned char*)0x80000480
	#define	PLLW	*(volatile unsigned		*)0x80002610
	
	/*
	 * ARM GPIO pins
	 */
	#define PADR	*(volatile unsigned char*)0x80000000
	#define PBDR	*(volatile unsigned char*)0x80000001
	#define PDDR	*(volatile unsigned char*)0x80000003
	#define PADDR	*(volatile unsigned char*)0x80000040
	#define PBDDR	*(volatile unsigned char*)0x80000041
	#define PDDDR	*(volatile unsigned char*)0x80000043
	#define PEDR	*(volatile unsigned char*)0x80000083
	#define PEDDR	*(volatile unsigned char*)0x800000C3
	
	/*
	 * ARM Timer control registers
	 */
	#define	TC1D	*(volatile unsigned		*)0x80000300
	#define	TC2D	*(volatile unsigned		*)0x80000340
	#define	RTCDR	*(volatile unsigned		*)0x80000380
	#define	RTCMR	*(volatile unsigned		*)0x800003C0
	
	/*
	 * ARM Interrupt control registers
	 */
	#define	INTSR1	*(volatile unsigned		*)0x80000240
	#define	INTSR2	*(volatile unsigned		*)0x80001240
	#define	INTSR3	*(volatile unsigned		*)0x80002240
	#define	INTMR1	*(volatile unsigned		*)0x80000280
	#define	INTMR2	*(volatile unsigned		*)0x80001280
	#define	INTMR3	*(volatile unsigned		*)0x80002280
	#define	BLEOI	*(volatile unsigned		*)0x80000600
	#define	MCEOI	*(volatile unsigned		*)0x80000640
	#define	TEOI	*(volatile unsigned		*)0x80000680
	#define	TC1EOI	*(volatile unsigned		*)0x800006C0
	#define	TC2EOI	*(volatile unsigned		*)0x80000700
	#define	RTCEOI	*(volatile unsigned		*)0x80000740
	#define	UMSEOI	*(volatile unsigned		*)0x80000780
	#define	COEOI	*(volatile unsigned		*)0x800007C0
	#define	KBDEOI	*(volatile unsigned		*)0x80001700
	#define	SRXEOF	*(volatile unsigned		*)0x80001600
	
	/*
	 * ARM chip selects
	 *
	 * These three chip selects ALL reference the same external 16 bit data
	 * bus, allowing you to access any of the attached hardware with any of
	 * three types of selects. They are documented here from a SOFTWARE point
	 * of view, which differs from the hardware description (see note below).
	 *
	 * CS8	-  8 bit transfer, single  8-bit ARM cycle, 1 bus access
	 * CS16 - 16 bit transfer, single 32-bit ARM cycle, 1 bus access **
	 * CS32 - 32 bit transfer, single 32-bit ARM cycle, 2 bus accesses
	 *
	 * ** Since the ARM does not support 16 bit transfers, this looks like a
	 *    32 bit transfer to the CPU, hoever the top 16 bits are lost on
	 *    writes, and yield indeterminate data on reads.
	 *
	 * In the hardware documentation, CS16 is called the 32 bit select because
	 * the ARM is programmed for a 32 bit bus. CS32 is called a 16 bit select
	 * because the ARM is programmed for a 16 bit bus.
	 *
	 * The following macros provide direct access to memory by select type as
	 * an offset into the external bus address range. These macros should be
	 * used for ALL hardware access as they will automatically reconfigure the
	 * address as required for both NORMAL an SERIAL BOOTSTRAP modes.
	 *
	 * NOTE: In addition to configuring BOOT_MODE, you must also change the
	 * following to move the load address of a program:
	 *		CODE base address: linker "-base" in LINK.VIA
	 *		DATA base address: linker "-data" in LINK.VIA
	 *		STACK top address: "LDR SP" at beginning of BEGIN.S
	 */
	#if BOOT_MODE == 1			// Normal (flash) startup
		#define	CS8(a)	(*(volatile unsigned char   *)(0x10000000+a))
		#define	CS16(a)	(*(volatile unsigned		*)(0x20000000+a))
		#define	CS32(a)	(*(volatile unsigned		*)(0x30000000+a))
	#elif BOOT_MODE == 2		// Boot (serial) startup
		#define	CS8(a)	(*(volatile unsigned char   *)(0x60000000+a))
		#define	CS16(a)	(*(volatile unsigned		*)(0x50000000+a))
		#define	CS32(a)	(*(volatile unsigned		*)(0x40000000+a))
	#else
		#error BOOT_MODE must be set: 1=NORMAL/FLASH 2=SERIAL
	#endif
		
	
	/*
	 * ARM chip select addresses
	 */
	#define BLATCH		CS32(0x020000)	// Bottom board latch
	#define TLATCH0		CS32(0x120000)	// Top board latch (cva)
	#define	TLATCH1		CS32(0x020004)	// Top board latch (vs)
	#define	BPCBID		CS8(0x040006)	// Bottom PCB card ID
	#define BPCBREV		CS8(0x040004)	// Bottom PCB revision
	#define	TPCBID		CS8(0x04000E)	// Top    PCB card ID
	#define	TPCBREV		CS8(0x04000A)	// Top    PCB revision
	#define IRQS		CS32(0x040000)	// IRQ status
	#define DNLD		CS8(0x160000)	// Download port
	#define RTL8019		CS8(0x1A0000)	// Ethernet
	#define SM_DATA		CS8(0x080000)	// SmartMedia data
	#define SM_INS		CS8(0x080002)	// SmartMedia inserted
	#define WDOG		CS8(0x0E0000)	// Watch dog reset
	#define WDOGE		CS8(0x0E0002)	// Watch dog enable
	#define	TOPHOOK		CS16(0x40008)	// Top hookswitch status
	
#endif
