/*
 * ArmOS library functions
 *
 * Dave Dunfield - 98/10/04
 *	99/04/29 + memcmp, strncpy, strncat, strncmp, strbeg, strstr, abs, size_t
 *	01/25/20 - changed testc, getc, gets, printf to raw* diagnostic functions.
 *			 + Added new printf, Iprintf for official console output
 *  03/12/08 - Added strupr
 *			 + Improved parse() functionality
 *			 + Fixed rawgets() to enforce length
 *	04/01/22 - Modified _format_ to be bounded
 *			 + Added snprintf()
 *			 + Added "safe" OS functions.
 */
#define	BOOT_MODE 1
#include "hardware.h"
#include "os.h"

#define	MMPRINTF	50			// Minimum message blocks for printf

#ifdef OO_DEV
	extern unsigned char OS_last_rel;
#endif

/*
 * ----- Hardware Interface functions -----
 */

/*
 * Test for character from serial port
 * Return 0-255 for ASCII character code, or 0xFFFFFFFF for no character
 */
unsigned rawtestc(void)
{
	unsigned i;
	WDOG = 0;	// Kick dog
	if(SYSFLG1 & 0x00400000)
		return 0xFFFFFFFF;
	i = UARTD;
	return i;
}

/*
 * Get a character from the serial port
 */
unsigned rawgetc(void)
{
	unsigned v;
	while((v = rawtestc()) == 0xFFFFFFFF);
	return v;
}

/*
 * Write a character to the serial port
 */
void rawputc(unsigned char c)
{
	if(c == '\n')
		rawputc('\r');
	WDOG = 0;
	while(SYSFLG1 & 0x00800000);
	UARTD = c;
/*
	switch(OS_irq_active) {
	case 0xAA : c = 0xB2; break;
	case 0x00 : c = 0xB0; break;
	default:	c = 0xB1; };
	while(SYSFLG1 & 0x00800000);
	UARTD = c; */
}

/*
 * Write a string to the serial port
 */
void rawputs(void *st)
{
	char *s = (char*)st;
	while(*s)
		rawputc(*s++);
}

/*
 * Test for character from system console
 */
unsigned testc(void)
{
	unsigned char c;
	OS_swap();
	if(OS_call(OS_CONSOLE_RX, &c, 1) == 1)
		return c;
	return 0xFFFFFFFF;
}

/*
 * Read a string from the serial port
 */
unsigned rawgets(unsigned char *dest, unsigned length)
{
	unsigned c, l=0;
	for(;;) switch(c = rawgetc()) {
	case '\b' :
		if(l) {
			--l;
			rawputs("\b \b"); }
			continue;
	case '\r' :
		dest[l] = 0;
		return l;
	default:
		if(l < length) {
			if((c >= ' ') && (c < 0x7F))
				rawputc(dest[l++] = c); } }
}

/*
 * ----- General library functions -----
 */

/*
 * Set a memory region to a constant value (byte)
 */
void byteset(void *address, unsigned char value, unsigned size)
{
	unsigned char *a = (unsigned char*)address;
	while(size) {
		*a++ = value;
		--size; }
}

/*
 * Set a memory region to a constant value (word)
 */
void wordset(void *address, unsigned value, unsigned size)
{
	unsigned *a = (unsigned*)address;
	while(size) {
		*a++ = value;
		--size; }
}

/*
 * Set a memory region to constant value (byte or word)
 */
void memset(void *address, unsigned char value, unsigned size)
{
	unsigned v;
	if(((unsigned)address | size) & 3)
		byteset(address, value, size);
	else {
		v = (value << 8) | value;
		wordset(address, (v << 16) | v, size >> 2); }
}

/*
 * Copy one memory region to another (byte)
 */
void bytecpy(void *dest, const void *source, unsigned size)
{
	unsigned char *d = (unsigned char*)dest, *s = (unsigned char *)source;
	while(size) {
		*d++ = *s++;
		--size; }
}

/*
 * Copy one memory region to another (word)
 */
void wordcpy(void *dest, const void *source, unsigned size)
{
	unsigned *d = (unsigned*)dest, *s = (unsigned*)source;
	while(size) {
		*d++ = *s++;
		--size; }
}

/*
 * Copy memory region to another (byte or word)
 */
void memcpy(void *dest, const void *source, unsigned size)
{
	if(((unsigned)dest | (unsigned)source | size) & 3)
		bytecpy(dest, source, size);
	else
		wordcpy(dest, source, size >> 2);
}

/*
 * Compare two blocks of memory
 */
int memcmp(const void *block1, const void *block2, unsigned size)
{
	char *b1 = (char*)block1, *b2 = (char*)block2;
	while(size--) {
		if(*b1 > *b2)
			return 1;
		if(*b1++ < *b2++)
			return -1; }
	return 0;
}

/*
 * Determine length of a string
 */
unsigned strlen(const void *string)
{
	unsigned char *s = (unsigned char*)string;
	while(*s++);
	return ((unsigned)s - (unsigned)string)-1;
}

/*
 * Compare two strings
 */
int strcmp(const void *string1, const void *string2)
{
	char *s1 = (char*)string1, *s2 = (char*)string2;
	do {
		if(*s1 > *s2)		/* String1 > String2 */
			return 1;
		if(*s1 < *s2++)		/* String1 < String2 */
			return -1; }
	while(*s1++);
	return 0;				/* String 1 == String 2 */
}

/*
 * Copy a string
 */
unsigned char *strcpy(void *dest, const void *source)
{
	unsigned char *d = (unsigned char*)dest, *s = (unsigned char*)source;
	while(*d = *s++)
		++d;
	return d;
}

/*
 * Concatinate to a string
 */
unsigned char *strcat(void *dest, const void *source)
{
	unsigned char *d = (unsigned char*)dest;
	while(*d)
		++d;
	return strcpy(d, source);
}

/*
 * Compare portions of two strings
 */
int strncmp(const void *string1, const void *string2, unsigned length)
{
	char *s1 = (char*)string1, *s2 = (char*)string2;
	do {
		if(*s1 > *s2)		/* String1 > String2 */
			return 1;
		if(*s1 < *s2++)		/* String1 < String2 */
			return -1; }
	while(*s1++ && --length);
	return 0;				/* String 1 == String 2 */
}

/*
 * Copies a variable length string into a fixed length field.
 * The destination field will be padded with zero (0) chars
 * if it is longer than the source string.
 */
unsigned char *strncpy(void *dest, const void *source, unsigned length)
{
	unsigned char *d = (unsigned char*)dest, *s = (unsigned char*)source;
	while(length--) {
		if(*d++ = *s)
			++s; }
	return d;
}

/*
 * Concatenates a string to another string, insuring that at most
 * "length" characters are copied.
 */
unsigned char *strncat(void *dest, const void *source, unsigned length)
{
	unsigned char *d = (unsigned char*)dest, *s = (unsigned char*)source;
	while(*d)					/* Find end of source */
		++d;
	while(*s && length--)
		*d++ = *s++;
	*d = 0;
	return d;
}

/*
 * Test for string1 beginning with string2
 */
int strbeg(const void *string1, const void *string2)
{
	unsigned char *s1 = (unsigned char*)string1, *s2 = (unsigned char*)string2;
	while(*s2)
		if(*s1++ != *s2++)
			return 0;
	return 1;
}

/*
 * Search for occurance of string2 in string1
 */
unsigned char *strstr(const void *string1, const void *string2)
{
	unsigned char *s1 = (unsigned char*)string1, *s2 = (unsigned char*)string2;
	while(*s1) {
		if(strbeg(s1, s2))
			return s1;
		++s1; }
	return 0;
}

/*
 * Format to string routine, format operands are passed
 * as a pointer to the calling functions argument lists.
 */
unsigned _format_(char *output, unsigned len, const char *format, unsigned *optr)
{
	unsigned width, value, i;
	char justify, zero, minus, comma, c, *ptr, *o;
	char outstk[43];

	o = output;
	while(c = *format++) {
		if(c == '%') {					/* format code */
			c = *format++;
			*(ptr = &outstk[42]) = justify = minus = comma = width = value = i = 0;
			zero = ' ';
			if(c == '-') {				/* left justify */
				--justify;
				c = *format++; }
			if(c == '0')					/* leading zeros */
				zero = '0';
			while((c >= '0') && (c <= '9')) {	/* Field width */
				width = (width * 10) + (c - '0');
				c = *format++; }

			value = *++optr;

again:		switch(c) {
			case 'd' :					/* decimal number */
				if(value & 0x80000000) {
					value = -value;
					++minus; }
			case 'u' :					/* unsigned number */
				i = 10;
				break;
			case 'x' :					/* hexidecimal number */
				i = 16;
				break;
			case 'o' :					/* octal number */
				i = 8;
				break;
			case 'b' :					/* binary number */
				i = 2;
				break;
			case 'c' :					/* character data */
				*--ptr = value;
				break;
			case 's' :					/* string */
				if(!(ptr = (char *) value))
					ptr = "";
				break;
			case ',' :					/* Comma delimiter */
				comma = 4;
				c = *format++;
				goto again;
			default:					/* all others */
				--optr;
				*--ptr = c; }

			if(i) do {	/* for all numbers, generate the ASCII string */
				if(comma) {
					if(!--comma) {
						*--ptr = ',';
						comma = 3; } }
				if((c = (value % i) + '0') > '9')
					c += 7;
				*--ptr = c; }
			while(value /= i);

/* output sign if any */
			if(minus) {
				if(len) {
					--len;
					*o++ = '-'; }
				if(width)
					--width; }

/* pad with 'zero' value if right justify enabled  */
			if(width && !justify) {
				for(i = strlen(ptr); i < width; ++i) {
					if(!len) break;
					--len;
					*o++ = zero; } }

/* move in data */
			i = 0;
			value = width - 1;
			while((*ptr) && (i <= value)) {
				if(!len) break;
				--len;
				*o++ = *ptr++;
				++i; }

/* pad with 'zero' value if left justify enabled */
			if(width && justify) {
				while(i < width) {
					if(!len) break;
					--len;
					*o++ = zero;
					++i; } } }
		else {
			if(!len) break;
			--len;
			*o++ = c; } }

	*o = 0;
	return (unsigned)o - (unsigned)output;
}

/*
 * Formatted print to string
 */
unsigned sprintf(void *output, const char *format, ...)
{
	return _format_((char*)output, 0xFF00, format, (unsigned*)&format);
}
unsigned snprintf(void *output, unsigned len, const char *format, ...)
{
	return _format_((char*)output, len, format, (unsigned*)&format);
}

/*
 * Formatted print directly to UART
 */
void rawprintf(const char *format, ...)
{
	char buffer[150];
	_format_(buffer, 149, format, (unsigned*)&format);
	rawputs(buffer);
}

/*
 * Formatted print to system console
 */
void printf(const char *format, ...)
{
	struct MESSAGE *m;

	while(OS_console_owner && (OS_active_tcb->Id != OS_console_owner)) {
		WDOG = 0;
		OS_swap(); }

	while(OS_free_msg_small < MMPRINTF) {
		WDOG = 0;
		OS_swap(); }

	m = (struct MESSAGE*)OS_call(OS_MSG_SMALL);
	m->Info1 = _format_((char*)m->Data, 127, format, (unsigned*)&format);
	OS_call(OS_CONSOLE_TX, m);
	OS_swap();
}

/*
 * Formatted print of process informational message
 */
void Dprintf(unsigned mask, const char *format, ...)
{
	struct MESSAGE *m;

	if((OS_debug_mask & mask) || !mask) {
		if(OS_console_owner && (OS_active_tcb->Id != OS_console_owner)) {
			return; }

		if(OS_free_msg_small < MMPRINTF) {
			return; }

		m = (struct MESSAGE*)OS_call(OS_MSG_SMALL);
		if(OS_irq_active) {
			strcpy(m->Data, "\r\n");
			m->Info1 = _format_((char*)m->Data+2, 125, format, (unsigned*)&format);
			OS_console_tx(m);
			return; }
		sprintf(m->Data, "\r\n%-8s: ", OS_active_tcb->Name);
		m->Info1 = _format_((char*)m->Data+12, 115, format, (unsigned*)&format) + 12;
		OS_call(OS_CONSOLE_TX, m); }
}

/*
 * Convert ASCII-decimal into binary
 */
unsigned atoi(const void *source)
{
	unsigned v = 0;
	unsigned char *s = (unsigned char*)source;
	while((*s >= '0') && (*s <= '9'))
		v = (v * 10) + (*s++ - '0');
	return v;
}

/*
 * Convert ASCII-hexidecimal into binary
 */
unsigned atox(const void *source)
{
	unsigned v = 0, c;
	unsigned char *s = (unsigned char*)source;
	for(;;) {
		c = *s++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a' - 10);
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A' - 10);
		else
			return v;
		v = (v << 4) | c; }
}

/*
 * Skip to non-blank in input buffer
 */
unsigned char skip_blanks(unsigned char **source)
{
	unsigned char *s = (unsigned char*)*source;
	while((*s == ' ') || (*s == '\t'))
		++s;
	*source = s;
	return *s;
}

/*
 * parse text from input buffer
 *	xxxxxx00xxxxxxxx	= Normal
 *	xxxxxx01xxxxxxxx	= 0-9 only
 *	xxxxxx10xxxxxxxx	= 0-9,A-F only
 *  xxxxxx11xxxxxxxx    = 0-9,A-Z only
 *  xxxxx1xxxxxxxxxx	= Convert TABs to spaces
 *  xxxx1xxxxxxxxxxx    = Exit in spaces
 *  xxx1xxxxxxxxxxxx	= 
 */
unsigned parse(unsigned char **source, void *dest, unsigned size)
{
	unsigned char *s = (unsigned char*)*source;
	unsigned char *d = (unsigned char*)dest;
	unsigned char c, c1, c2;
	unsigned l = size & 0xFF;

	size >>= 8;
	c1 = size >> 8;
	c2 = size >> 16;
	skip_blanks(&s);
	while(c = *s) {
		if((c == '\t') && !(size & 0x04)) c = ' ';
		if((c == ' ') && !(size & 0x08)) goto exit;

		if(c == c1) break;	// Extra character1
		if(c == c2) break;	// Extra character 2

		if(size & 0x10) {		// convert to upper case
			if((c >= 'a') && (c <= 'z'))
				c -= ('a'-'A'); }

		switch(size & 3) {
		case 3 :	// 0-9,a-z,A-Z only
			if((*s >= 'a') && (*s <= 'z'))
				break;
			if((*s >= 'A') && (*s <= 'Z'))
				break;
		case 2 :	// 0-9,a-f,A-F only
			if((*s >= 'a') && (*s <= 'f'))
				break;
			if((*s >= 'A') && (*s <= 'F'))
				break;
		case 1 :	// 0-9 only
			if((*s < '0') || (*s > '9')) {
				*d = 0;
				if(size & 0x10) goto exit;
				return 0; } }
		*d++ = c;
		++s;
		if(!--l)
			break; }
exit:
	*d = 0;
	*source = s;
	return (unsigned)d - (unsigned)dest;
}

/*
 * Parse a number in either hex or decimal format
 */
int parse_number(unsigned char **source, unsigned *dest, unsigned char hex)
{
	unsigned char c, temp[11];
	switch(c = skip_blanks(source)) {
	case '.' :	hex = 0;	goto nexts;
	case '$' :	hex = 255;
	nexts: ++*source; }
	if(!parse(source, temp, hex ? 0x20A : 0x10A))
		return 255;
	*dest = hex ? atox(temp) : atoi(temp);
	return 0;
}

/*
 * Return absolute value of a number
 */
int abs(int value)
{
	return (value < 0) ? -value : value;
}

/*
 * Convert string to upper case
 */
void strupr(void *string)
{
	unsigned char *s = (unsigned char*)string;
	while(*s) {
		if((*s >= 'a') && (*s <= 'z'))
			*s -= ('a'-'A');
		++s; }
}

/*
 * Compare portions of two strings up to token  
*/ 
int strncmptok(const void *string1, const void *string2, unsigned length, const char token) 
{
	char *s1 = (char*)string1, *s2 = (char*)string2;
	do {
		if(*s1 > *s2)		/* String1 > String2 */
			return 1;
		if(*s1 < *s2++)		/* String1 < String2 */
			return -1; }
	while(*s1 && --length && *s1++ != token);
	return 0;				/* String 1 == String 2 */
}

/*
 * "Safe" OS functions
 *
 * These functions examine the OS_irq_active flag, and translate the users
 * rquest into the appropriate OS_call(...) or OS_... direct function as
 * required.
 */
struct MESSAGE *SOS_msg_small(void)
{
	return OS_irq_active ? OS_msg_small() : (struct MESSAGE*)OS_call(OS_MSG_SMALL);
}
struct MESSAGE *SOS_msg_medium(void)
{
	return OS_irq_active ? OS_msg_medium() : (struct MESSAGE*)OS_call(OS_MSG_MEDIUM);
}
struct MESSAGE *SOS_msg_large(void)
{
	return OS_irq_active ? OS_msg_large() : (struct MESSAGE*)OS_call(OS_MSG_LARGE);
}

void SOS_msg_release(struct MESSAGE *m)
{
	if(OS_irq_active) {
#ifdef	OO_DEV
		OS_last_rel = 6;
#endif
		OS_msg_release(m); }
	else
		OS_call(OS_MSG_RELEASE, m);
}

unsigned SOS_msg_send(unsigned t, struct MESSAGE *m, unsigned char p)
{
	if(OS_irq_active)
		return OS_msg_send(t, m, p);
	else
		return OS_call(OS_MSG_SEND, t, m);
}

void SOS_proc_signal(unsigned e, unsigned v)
{
	if(OS_irq_active)
		OS_proc_signal(e, v);
	else
		OS_call(OS_PROC_SIGNAL, e, v);
}

void SOS_irq_set(unsigned mask)
{
	if(OS_irq_active)
		OS_irq_set(mask);
	else
		OS_call(OS_IRQ_SET, mask);
}

void SOS_irq_clear(unsigned mask)
{
	if(OS_irq_active)
		OS_irq_clear(mask);
	else
		OS_call(OS_IRQ_CLEAR, mask);
}

void SOS_console_tx(struct MESSAGE *m)
{
	if(OS_irq_active)
		OS_console_tx(m);
	else
		OS_call(OS_CONSOLE_TX, m);
}

unsigned SOS_ethernet_tx(unsigned char *d, unsigned s)
{
	if(OS_irq_active)
		return OS_ethernet_tx(d, s);
	else
		return OS_call(OS_ETHERNET_TX, d, s);
}

void SOS_bls(unsigned mask)
{
	if(OS_irq_active)
		OS_bls(mask);
	else
		OS_call(OS_BLS, mask);
}

void SOS_blc(unsigned mask)
{
	if(OS_irq_active)
		OS_blc(mask);
	else
		OS_call(OS_BLC, mask);
}

void SOS_tls(unsigned mask)
{
	if(OS_irq_active)
		OS_tls(mask);
	else
		OS_call(OS_TLS, mask);
}

void SOS_tlc(unsigned mask)
{
	if(OS_irq_active)
		OS_tlc(mask);
	else
		OS_call(OS_TLC, mask);
}

/* ----- debug function ---- */

void Eprintf(unsigned char a, const char *format, ...)
{
	unsigned char *p;
	struct {
		unsigned char To[6];
		unsigned char From[6];
		unsigned Type;
		unsigned char Attr;
		unsigned char Data[64]; } packet;

	if(p = (unsigned char*)OS_call(OS_ETHERNET_MAC)) {
		byteset(packet.To, 0xFF, 6);
		bytecpy(packet.From, p, 6);
		packet.Type = 0xEDEB;
		packet.Attr = a;
		_format_((char*)packet.Data, 149, format, (unsigned*)&format);
		OS_call(OS_ETHERNET_TX, &packet, sizeof(packet)); }
}
