/*
 * ArmOS - A compact, low-overhead fast real-time executive for the ARM
 *         processor.
 *
 * Dave Dunfield
 * First edition: Oct 4, 1998
 * Last   update: Feb 26, 2006
 */
#define	BOOT_MODE	1
#include "hardware.h"		// Hardware definitions
#include "library.h"		// Library prototypes and definitions

#define	TID_TIMER	255		// Msg is from Timer
#define	TID_ENET	254		// Msg is from Ethernet
#define	TID_MUTEX	253		// Msg is from Mutex
#define	TID_FREE	250		// Msg is Free

#define	SBOUND		0x12345678
#define	O_STACK		0x60009000	// OS stack
#define	I_STACK		0x6000A000	// IRQ stack
#define	S_STACK		0x6000B000	// Supervisor stack
#define	E_STACK		0x6000C000	// End of OS stack space

#define	Debug(x)	//rawprintf x;

#define	OS_BUILD
#include "os.h"
#include "irq.h"

#define	MAC_OFFSET	0xC0000006		// Offset to MAC address in flash

/*
 * Timer control block
 */
struct TIMER {
	struct TIMER	*Link;			// Link to next
	unsigned		Countdown;		// Countdown value
	unsigned		Setting;		// Set value (ms)
	unsigned		Control;		// Control value
	unsigned		Uparm;	};		// User parameter

struct MUTEX {
	struct MUTEX	*Link;			// Link to next mutex in chain
	unsigned char	Mid;			// Owner of mutex
	unsigned char	Mtype; };		// Type of mutex request

/*
 * Internal operating system registers
 */
struct TCB
	*OS_active_tcb;				// Pointer to current task
unsigned
#ifdef OO_DEV
	OS_last_rel,				// Track last message release
#endif
	OS_ram_top,					// Top of RAM
	OS_timer_tick,				// Tick count
	OS_debug_mask,				// Debug output mask
	OS_eirq_mask,				// External IRQ mask
	OS_system_config,			// System configuration
	OS_console_rx_read,			// Uart RX read  pointer
	OS_console_rx_write,		// Uart RX write pointer
#ifdef	OO_MLOW
	OS_free_min_small,			// Small message low-water mark
	OS_free_min_medium,			// Medium message low-water mark
	OS_free_min_large,			// Large message low-water mark
#endif
	OS_free_msg_small,			// Small message free count
	OS_free_msg_medium,			// Medium message free count
	OS_free_msg_large;			// Large message free count
unsigned char
	OS_hw_type,					// Hardware type
	OS_irq_active,				// Interrupt is active
	OS_hw_port,					// Hardware port output
	OS_at_owner,				// Receiver of AT commands
	OS_panic_active,			// Panic debugger active
	OS_console_echo,			// Enable echo at system console
	OS_console_owner,			// Console owner
	OS_console_buffer[81],		// Console input buffer
	OS_console_rx_buffer[2048];	// Uart RX buffer (Must be power of 2)

static unsigned
	OS_panic_name[10],
	OS_panic_address[10],
	OS_panic_top;

static struct TCB
	*Tready[PRIORITIES+1],		// Task exec ready list
	*Tnext[PRIORITIES+1],		// Link to next task
	*Signals[MAX_SIGNAL],		// Signal list
	Task_list[MAX_TASKS];		// List of task control blocks

static struct TIMER
	Timer_list[MAX_TIMER],		// Timer control blocks
	*Timer_free,				// Free list
	*Timer_base;				// First active

static struct MESSAGE
	*Small_free,				// Free message list (small)
	*Medium_free,				// Medium block free list
	*Large_free,				// Large block free list
	*Uart_tx_msg;				// Serial message

static struct MUTEX
	Mutex[MAX_MQUEUE],			// Queue of pending Mutexes
	*Mutex_free,				// Free mutex list
	*Mutex_queue[MAX_MUTEX];	// Queue of pending mutexes

static unsigned
	packet_type[MAX_PACKET],	// Types of interest
	packet_top;					// Top of packet table

static unsigned char
	Mutex_owner[MAX_MUTEX],		// Mutex owners
	nic_address[6],				// Our MAC address
	packet_pid[MAX_PACKET],		// Pid to send
	Uart_tx_position;			// Uart TX output position

static struct MESSAGE
	Small_list[MAX_SMALL];		// Small message list
static unsigned
	Medium_list[MAX_MEDIUM][(MSG_MEDIUM/4)+2],	// Medium message list
	Large_list[MAX_LARGE][(MSG_LARGE/4)+2];		// Large message list

static unsigned
	Latency,
	MaxLatency;

extern unsigned
	__OSinitram(void);			// Initalize memory

extern void
	__OSexit(void),				// Exit kernel mode
	__OSreturn(unsigned *),		// Return from debugger
	__OSsystem(void),			// Launch user in supervisor mode (stop)
	__OSboot(void),				// Restart system
	__OSdisable(void),			// Disable interrupts
	__OSenable(void),			// Enable interrupts
	__OSinvtlb(unsigned),		// Invalidate TLB
	user_init(void);			// User mode main program
//extern unsigned
//	__OSgetspsr(void);			// Get saved processor status register

static void OS_proc_unready(unsigned id, unsigned char state);

#ifdef OO_TIME
	extern unsigned char OSappID;
	extern unsigned OSappTimer;
#endif

typedef const char *ccp;

static const char hello[] = { "\r\nArmOS - Build: "__DATE__" "__TIME__
#ifdef OO_DEV
	" (Development)"
#endif
};

extern int flash_erase_block(unsigned b);
extern int flash_program_block(unsigned a, unsigned d, unsigned l);
//extern unsigned char DallasRead(unsigned a);
//extern void DallasWrite(unsigned a, unsigned char d);

#ifdef	OO_DEV
	void OSexit(void) { OS_irq_active = 0; __OSexit(); }
	void OStest(char *f) { if(!OS_irq_active) OS_panic("OS_%s() - direct", f); }
	void OSmfree(struct MESSAGE *m, char *x)
	{
		if(m->Info == TID_FREE)
			OS_panic("%s FREE %08x", x, m);
	}
#else
	#define	OSexit		__OSexit
	#define	OStest(f)
	#define	OSmfree(m,x)
#endif

unsigned
	OS_blatch,		// Mirror for bottom latch
	OS_tlatch;		// Mirror for top latch

// Set a bit in the bottom latch
void OS_bls(unsigned mask)
{
	OStest("bls");
	OS_blatch |= mask;
	BLATCH = OS_blatch;
}

// Clear a bit in the bottom latch
void OS_blc(unsigned mask)
{
	OStest("blc");
	OS_blatch &= ~mask;
	BLATCH = OS_blatch;
}

// Set a bit in the top latch
void OS_tls(unsigned mask)
{
	OStest("tls");
	OS_tlatch |= mask;
	if(OS_hw_type < 2)
		TLATCH0 = OS_tlatch;
	else
		TLATCH1 = OS_tlatch;
}

// Clear a bit in the top latch
void OS_tlc(unsigned mask)
{
	OStest("tlc");
	OS_tlatch &= ~mask;
	if(OS_hw_type < 2)
		TLATCH0 = OS_tlatch;
	else
		TLATCH1 = OS_tlatch;
}

/*
 * Dump a block of memory
 */
static int dump(void *address, unsigned size, void (*output)(const char*,...), unsigned (*test)(void))
{
	unsigned i, j;
	unsigned char *a, *o, buffer[81];

	a = (unsigned char*)address;
	for(i=0; i < size; i += 16) {
		if(test() == 0x1B) return 255;
		o = buffer;
		o += sprintf(o, "\r\n%08x", a);
		for(j=0; j < 16; ++j) {
			if(!(j & 3)) *o++ = ' ';
			if((i+j) < size)
				o += sprintf(o, " %02x", a[j]);
			else
				o += sprintf(o, "   "); }
		o += sprintf(o, "  ");
		for(j=0; j < 16; ++j) {
			if((i+j) < size)
				*o++ = (a[j] >= ' ') && (a[j] <0x7F) ? a[j] : '.';
			else
				*o++ = ' '; }
		*o = 0;
		output("%s", buffer);
		a += 16; }
	return 0;
}


#ifdef OO_TIME
volatile unsigned
	OST_position,					// Real time position
	OST_real_time;					// Real time counter
volatile unsigned char
	OST_busy;						// RTC display in progress
static unsigned
	OST_irq_time[MAX_IRQ][OO_TIME],	// IRQ timestamps
	OST_irq_runlong[MAX_IRQ],		// Longest run seen
	OST_irq_seclong[MAX_IRQ],		// Longest second seen
	OST_tbase;						// Task timestamp base
unsigned
	OST_ibase;						// Interrupt timestamp base

/*
 * Read real-time clock
 */
unsigned OS_read_real_time(void)
{
	unsigned t, r;
	t = OST_real_time;
	r = TC2D & 0xFFFF;
	if(INTSR1 & 0x0200) {
//if(!(r & 0x8000)) rawprintf("\nIRQ+: t=%u r=%u", t, r);
		if(r)
			++t; }
	return (t * 65536) + (65535 - r);
}

/*
 * Exit from an interrupt - record time taken, and adjust
 * global timers to remove time taken by irq
 */
void OS_irq_exit(unsigned irq)
{
	unsigned t;
	t = OS_read_real_time() - OST_ibase;
	if(t > OST_irq_runlong[irq])
		OST_irq_runlong[irq] = t;
	OST_irq_time[irq][OST_position] += t;
	OST_tbase += t;
	OST_ibase += t;
}

/*
 * Real-time clock interrupt
 * - Increment real time clock extension
 * - Compute time taken during last second and
 *   adjust maximum time values if greater.
 * - Advance to next second slot and zero counters
 */
void IRQ_timer2(void)
{
	unsigned i, p, *pp;
	struct TCB *t;
	struct MESSAGE *m;
	if(!(++OST_real_time & (OO_RT_MASK-1))) {
		if(OST_busy == 0x55) {		// Network TX
			m = OS_msg_medium();
			memset(m->Data, 0xFF, 6);
			memcpy(m->Data+6, nic_address, 6);
			pp = (unsigned*)(m->Data+12);
			*pp++ = 0x1616;
			*pp++ = OSappID;
			*pp++ = OSappTimer;
			for(i=0; i < MAX_TASKS; ++i) {
				t = &Task_list[i];
				*pp++ = t->TT_runlong;
				t->TT_runlong = 0;
				*pp++ = t->TT_time[0];
				t->TT_time[0] = 0; }
			for(i=0; i < MAX_IRQ; ++i) {
				*pp++ = OST_irq_runlong[i];
				OST_irq_runlong[i] = 0;
				*pp++ = OST_irq_time[i][0];
				OST_irq_time[i][0] = 0; }
			OS_ethernet_tx(m->Data, (MAX_TASKS*8)+(MAX_IRQ*8)+22);
			OS_msg_release(m); }
		else {
			p = OST_position;
			if(!OST_busy)
				OST_position = (p+1) & (OO_TIME-1);
			for(i=0; i < MAX_TASKS; ++i) {
				t = &Task_list[i];
				if(t->TT_time[p] > t->TT_seclong)
					t->TT_seclong = t->TT_time[p];
				t->TT_time[OST_position] = 0; }
			for(i=0; i < MAX_IRQ; ++i) {
				if(OST_irq_time[i][p] > OST_irq_seclong[i])
					OST_irq_seclong[i] = OST_irq_time[i][p];
				OST_irq_time[i][OST_position] = 0; } } }
	TC2EOI = 0;
	OS_irq_exit(XIRQ_timer2);
}
#endif

/*
 * Display process information on specified output function
 */
static void ps(unsigned char *cptr, void (*output)(const char *,...))
{
	unsigned i, j;
	unsigned char f, h, ti, rf;
	struct TCB *t;
	unsigned char *p, name[9], temp[80];
#ifdef OO_TIME
	unsigned r;
	static const ccp irqnames[] = {
		"Timer1", "Timer2", "UartRX", "UartTX", "Enet" };
#endif


	*name = f = h = ti = rf = 0;
	while(parse(&cptr, p=temp, 0x1000+8)) {
		switch(*p++) {
		case '-' :
		case '/' :
			while(*p) switch(*p++) {
#ifdef OO_TIME
				case 'I' :	ti = 255; goto dotime;
				case 'T' :	f = 1;
				dotime:		if(OST_busy) return;				continue;
				case 'N' :	OST_busy = 0x55; OST_position = 0;	return;
				case 'C' :	OST_busy = 0;						return;
#endif
#ifdef OO_MHIGH
				case 'M' :	f = 2;								continue;
#endif
				case 'H' : h = 1;								continue;
				case 'R' : rf = 255;							continue;
				default: output("\r\nBad switch");
				return; }
			continue; }
		strcpy(name, temp); }

#ifdef OO_TIME
	if((f == 1) || ti) {
		OST_busy = 255;
		r = OST_position; }

	if(ti) {
		if(!h) {
			output("\r\n#  Name     MaxSlot --- History 1-7 ----");
			output("\r\n   MaxRun   ----------- History 8-15 ---"); }
		for(i=0; i < MAX_IRQ; ++i) {
			if(*name) {
				strcpy(temp, irqnames[i]);
				strupr(temp);
				if(!strbeg(temp, name))
					continue; }
			p = temp;
			for(j=1; j < OO_TIME; ++j) {
				if(j == 8) {
					*p = 0;
					output("\r\n%-3u%-9s%-8u%s", i, irqnames[i], OST_irq_seclong[i], p=temp); }
				sprintf(p, "%-8u", OST_irq_time[i][(r-j) & (OO_TIME-1)]);
				p += 8; }
			*p = 0;
			output("\r\n   %-9u%s", OST_irq_runlong[i], temp);
			if(rf)
				OST_irq_runlong[i] = OST_irq_seclong[i] = 0; }
		if(f != 1) {
			OST_busy = 0;
			return; } }
#endif
	if(!h) {
		switch(f) {
		case 0 : output("\r\n#  Name     Sta Pri Tcb      SP-Low   SP-Cur.  Free  M-Queue  T-Link"); break;
#ifdef	OO_TIME
		case 1 :output("\r\n#  Name     MaxSlot --- History 1-7 ----");
				output("\r\n   MaxRun   ----------- History 8-15 ---"); break;
#endif
#ifdef	OO_MHIGH
		case 2 : output("\r\n#  Name     Current Highest Limit");
#endif
		} }
	for(i=0; i < MAX_TASKS; ++i) {
		t = &Task_list[i];
		if(*name) {
			strcpy(temp, t->Name);
			strupr(temp);
			if(!strbeg(temp, name))
				continue; }
		if(t->State) {
			++h;
			switch(f) {
			case 0 :
				for(j=0; t->SpLow[j] == SBOUND; ++j);
				output("\r\n%-3u%-9s%-4u%-4u%08x %08x %08x %-6u%08x %08x",
					t->Id, t->Name, t->State, t->Priority, t, t->SpLow,
					t->Sp, j, t->Mqueue, t->Next);
				continue;
#ifdef OO_TIME
			case 1 :
				p = temp;
				for(j=1; j < OO_TIME; ++j) {
					if(j == 8) {
						*p = 0;
						output("\r\n%-3u%-9s%-8u%s", t->Id, t->Name, t->TT_seclong, p=temp); }
					sprintf(p, "%-8u", t->TT_time[(r-j) & (OO_TIME-1)]);
					p += 8; }
				*p = 0;
				output("\r\n   %-9u%s", t->TT_runlong, temp);
				continue;
#endif
#ifdef OO_MHIGH
			case 2 :
				output("\r\n%-3u%-9s%-8u%-8u%u", t->Id, t->Name,
					t->Mqueue_size, t->Mqueue_high, t->Mqueue_limit);
				if(rf)
					t->Mqueue_high = t->Mqueue_size;
#endif
		} } }

	if(f == 1) {
#ifdef OO_TIME
		if(rf) for(i=0; i < MAX_TASKS; ++i) {
			t = &Task_list[i];
			t->TT_runlong = t->TT_seclong = 0; }
		OST_busy = 0;
#endif
	}
}



/*
 * Display system information on specified function
 */
static void info(void (*output)(const char *,...))
{
	unsigned i, j;
	i = OS_free_msg_small;
	output((char*)hello);
	output("\r\nHardware Type      : %x", OS_hw_type);
	output("\r\nSystem config      : %08x", OS_system_config);
	output("\r\nFree message blocks: %u/%u/%u", i, OS_free_msg_medium, OS_free_msg_large);
#ifdef OO_MLOW
	output("\r\nLowest free seen   : %u/%u/%u", OS_free_min_small, OS_free_min_medium, OS_free_min_large);
#endif
	output("\r\nTimer tick         : %08x", OS_timer_tick);
	output("\r\nExternal IRQ mask  : %08x", OS_eirq_mask);
	output("\r\nDebug output mask  : %08x", OS_debug_mask);
	output("\r\nLatency/max        : %u/%u", Latency, MaxLatency);
	output("\r\nTop of RAM         : %08x", OS_ram_top);
	for(i=0; ((unsigned*)S_STACK)[i] == SBOUND; ++i);
	for(j=0; ((unsigned*)I_STACK)[j] == SBOUND; ++j);
	output("\r\nS/I stack avail.   : %u/%u", i, j);
	output("\r\nBottom/Top latch   : %08x %08x", OS_blatch, OS_tlatch);
}

/*
 * Display packet/network information on specified function
 */
static void packet(void (*output)(const char*,...))
{
	unsigned i;
	output("\r\nMac: %02x%02x%02x%02x%02x%02x - %u types registered.",
		nic_address[0], nic_address[1], nic_address[2],
		nic_address[3], nic_address[4], nic_address[5], packet_top);
	for(i=0; i < packet_top; ++i)
		output("\r\n%08x -> %u", packet_type[i], packet_pid[i]);
}

#include "panic.c"

/*
 * Divide by zero exception
 */
void dividebyzero(void)
{
	OS_panic("Divide by zero");
}

/*
 * Wait for approximately microseconds
 */
static void wait(unsigned t)
{
	unsigned i, j;
	for(i=0; i < t; ++i) {
		for(j=0; j < 8; ++j)
			; }
}

/*
 * ----- Message list management functions -----
 */

struct MESSAGE *OS_msg_large(void)
{
	struct MESSAGE *m;
	OStest("msg_large");
	if(!(m = Large_free))		// Allocate a message block
		OS_panic("No message blocks");
	#ifdef OO_DEV
		if(m->Info != TID_FREE)
			OS_panic("Allocate LARGE !FREE %08x", m);
	#endif
	Large_free = m->Link;
	m->Link = 0;
	m->Info = OS_active_tcb->Id;
	--OS_free_msg_large;
#ifdef OO_MLOW
	if(OS_free_min_large > OS_free_msg_large)
		OS_free_min_large = OS_free_msg_large;
#endif
	return m;
}

struct MESSAGE *OS_msg_medium(void)
{
	struct MESSAGE *m;
	OStest("msg_medium");
	if(!(m = Medium_free))		// Allocate a message block
		return OS_msg_large();
	#ifdef OO_DEV
		if(m->Info != TID_FREE)
			OS_panic("Allocate MEDIUM !FREE %08x", m);
	#endif
	Medium_free = m->Link;
	m->Link = 0;
	m->Info = OS_active_tcb->Id;
	--OS_free_msg_medium;
#ifdef OO_MLOW
	if(OS_free_min_medium > OS_free_msg_medium)
		OS_free_min_medium = OS_free_msg_medium;
#endif
	return m;
}

/*
 * Allocate a message from the free list
 */
struct MESSAGE *OS_msg_small(void)
{
	struct MESSAGE *m;
	OStest("msg_small");
	if(!(m = Small_free))		// Allocate a message block
		return OS_msg_medium();
	#ifdef OO_DEV
		if(m->Info != TID_FREE)
			OS_panic("Allocate SMALL !FREE %08x", m);
	#endif
	Small_free = m->Link;
	m->Link = 0;
	m->Info = OS_active_tcb->Id;
	--OS_free_msg_small;
#ifdef OO_MLOW
	if(OS_free_min_small > OS_free_msg_small)
		OS_free_min_small = OS_free_msg_small;
#endif
	return m;
}

/*
 * Return a message to the free list
 */
void OS_msg_release(struct MESSAGE *m)
{
	Debug(("Release_msg: %08x\n", m))
	OStest("msg_release");
	#ifdef OO_DEV
		OSmfree(m, "RELEASE");
		m->Data[MSG_SMALL-1] = m->Info;
		m->Data[MSG_SMALL-2] = OS_active_tcb->Id;
		m->Data[MSG_SMALL-3] = OS_irq_active;
		m->Info = TID_FREE;
		m->Data[MSG_SMALL-4] = OS_last_rel;
		OS_last_rel = 0;
	#endif

	if((unsigned)m < (unsigned)Medium_list) {
		m->Link = Small_free;
		Small_free = m;
		++OS_free_msg_small;
		return; }
	if((unsigned)m < (unsigned)Large_list) {
		m->Link = Medium_free;
		Medium_free = m;
		++OS_free_msg_medium;
		return; }
	m->Link = Large_free;
	Large_free = m;
	++OS_free_msg_large;
}

unsigned OS_msg_send(unsigned Tid, struct MESSAGE *m, unsigned char info)
{
	struct TCB *t;
	struct MESSAGE *n;
	unsigned char p;

	OStest("msg_send");
	if(Tid >= MAX_TASKS)
		return 3;			// Bad Task ID

	t = &Task_list[Tid];
	if(!t->State)
		return 2;			// Task is dead

	OSmfree(m, "SEND");
	m->Link = 0;			// End of list
	m->Info = info;			// Set sender

	// If task is received blocked - pass message directly
	if(t->State == TASK_RECEIVE) {
		Debug(("MakeReady: %u (%08x)\n", Tid, t))
		t->Sp[0] = (unsigned)m;
		p = t->Priority;
		t->Next = Tready[p];
		Tready[p] = t;
		t->State = TASK_READY;
		return 0; }

	// Task is in another state - queue message
	if(t->Mqueue_size >= t->Mqueue_limit)
		return 4;			// Task is over limit
	if(n = t->Mqueue) {
		while(n->Link)		// Find end of chain
			n = n->Link;
		n->Link = m; }		// Link this one in
	else
		t->Mqueue = m;
	++t->Mqueue_size;
#ifdef OO_MHIGH
	if(t->Mqueue_size > t->Mqueue_high)
		t->Mqueue_high = t->Mqueue_size;
#endif
	return 1;
}

static void OS_msg_requeue(struct MESSAGE *h, struct MESSAGE *t)
{
	t->Link = OS_active_tcb->Mqueue;
	OS_active_tcb->Mqueue = h;
	++OS_active_tcb->Mqueue_size;
	while(h != t) {
		h = h->Link;
		++OS_active_tcb->Mqueue_size; }
#ifdef OO_MHIGH
	if(OS_active_tcb->Mqueue_size > OS_active_tcb->Mqueue_high)
		OS_active_tcb->Mqueue_high = OS_active_tcb->Mqueue_size;
#endif
}

static struct MESSAGE *OS_msg_check(void)
{
	struct MESSAGE *m;
	if(m = OS_active_tcb->Mqueue) {
		OSmfree(m, "RX");
		OS_active_tcb->Mqueue = m->Link;
		--OS_active_tcb->Mqueue_size;
		return m; }
	return 0;
}

static struct MESSAGE *OS_msg_receive(void)
{
	struct MESSAGE *m;
	for(;;) {
		if(m = OS_active_tcb->Mqueue) {
			OSmfree(m, "RX");
			OS_active_tcb->Mqueue = m->Link;
			--OS_active_tcb->Mqueue_size;
			return m; }
		OS_proc_unready(OS_active_tcb->Id, TASK_RECEIVE);
		OSexit();
	}
}

static void OS_proc_wait(unsigned signal)
{
	struct TCB *t;
	if(signal >= MAX_SIGNAL)
		OS_panic("Signal %u", signal);
	t = OS_active_tcb;
	OS_proc_unready(OS_active_tcb->Id, signal+TASK_SIGNAL);
	t->Next = Signals[signal];
	Signals[signal] = t;
	OSexit();
}

void OS_proc_signal(unsigned signal, unsigned value)
{
	struct TCB *t, *tn;
	OStest("proc_signal");
	if(signal >= MAX_SIGNAL)
		OS_panic("Signal %u", signal);
	tn = Signals[signal];
	while(t = tn) {
		tn = t->Next;
		OS_proc_ready(t->Id, value); }
	Signals[signal] = 0;
}

void OS_irq_set(unsigned mask)
{
	OStest("irq_set");
	OS_eirq_mask |= mask;
	IRQS = OS_eirq_mask;
}

void OS_irq_clear(unsigned mask)
{
	OS_eirq_mask &= ~mask;
	IRQS = OS_eirq_mask;
}

unsigned OS_mutex_request(unsigned mutex, unsigned type)
{
	struct MUTEX *m1, **mq;
#ifdef OO_DEV
	if(mutex >= MAX_MUTEX)
		OS_panic("Mutex (%u) out of range", mutex);
#endif
	if(!Mutex_owner[mutex]) {
		Mutex_owner[mutex] = OS_active_tcb->Id;
		return 1; }
	if(type) {
		// Find end of mutex queue
		mq = &Mutex_queue[mutex];
		while(m1 = *mq)
			mq = &m1->Link;
		// Allocate new mutex block
		if(!(*mq = m1 = Mutex_free))
			OS_panic("Mutex queue exhausted");
		Mutex_free = m1->Link;
		m1->Link = 0;
		m1->Mid = OS_active_tcb->Id;
		m1->Mtype = type;
		if(type != 1) {		// Block
			OS_proc_unready(OS_active_tcb->Id, TASK_MUTEX);
			OSexit(); } }
	return 0;
}

void OS_mutex_release(unsigned mutex)
{
	struct MUTEX *m1;
	struct MESSAGE *m;
#ifdef OO_DEV
	if(mutex >= MAX_MUTEX)
		OS_panic("Mutex (%u) out of range", mutex);
#endif
	if(Mutex_owner[mutex] != OS_active_tcb->Id)
		OS_panic("Release unowned mutex %u", mutex);
	if(m1 = Mutex_queue[mutex]) {	// Queued
		Mutex_queue[mutex] = m1->Link;
		Mutex_owner[mutex] = m1->Mid;
		m1->Link = Mutex_free;
		Mutex_free = m1;
		if(m1->Mtype == 1) {	// Send message
			m = OS_msg_small();
			m->Info1 = mutex;
			if(OS_msg_send(m1->Mid, m, TID_MUTEX)  > 1)
				OS_msg_release(m);
			return; }
		// Unblock task
		if(Task_list[m1->Mid].State == TASK_MUTEX) {
			OS_proc_ready(m1->Mid, 2);
			return; } }
	Mutex_owner[mutex] = 0;
}

/*
 * ----- Kernel process management functions -----
 */

/*
 * Get parameter from active function
 */
static unsigned get_parm(unsigned p)
{
	return OS_active_tcb->Sp[p];
}

static void send_result(unsigned r)
{
	OS_active_tcb->Sp[0] = r;
}

/*
 * Obtain a task control block
 */
static struct TCB *get_tcb(void)
{
	unsigned i;
	struct TCB *t;
	for(i=0; i < MAX_TASKS; ++i) {
		t = &Task_list[i];
		if(t->State == TASK_DEAD) {
			t->Id = i;
			Debug(("Allocated TCB %u %08x\n", i, t))
			return t; } }
	OS_panic("No TCBs");
}

/*
 * Dispatch the next task waiting to run
 */
void __OSdispatch_next(void)
{
	unsigned i;
#ifdef OO_TIME
	unsigned t, intmr1;
#endif
#ifdef OO_DEV
	if(OS_irq_active == 255)
		OS_panic("OS_swap() in interrupt");
	if(*(OS_active_tcb->SpLow) != SBOUND)
		OS_panic("Stack base corrupt");
#endif
#ifdef OO_TIME
	intmr1 = INTMR1;
	INTMR1 = 0;
	t = OS_read_real_time();
	if((i = t - OST_tbase) & 0x80000000) {
//rawprintf("\nOST_tbase=%u t=%u %x", OST_tbase, t, OS_irq_active);
		i = 0; }
	if(i > OS_active_tcb->TT_runlong)
		OS_active_tcb->TT_runlong = i;
	OS_active_tcb->TT_time[OST_position] += i;
#endif
	for(i=0; i < (PRIORITIES+1); ++i) {
		if(OS_active_tcb = Tnext[i]) {
			Tnext[i] = OS_active_tcb->Next;
#ifdef OO_TIME
			OST_tbase = t;
			INTMR1 = intmr1;
#endif
			return; }
		Tnext[i] = Tready[i]; }
	OS_panic("No task ready!");
}

/*
 * Start a task
 */
static unsigned OS_proc_start(unsigned execaddr, unsigned char priority, unsigned *stack, unsigned ssize, unsigned char *name, unsigned mlimit)
{
	struct TCB *t;

	Debug(("Start E:%08x P:%u S:%08x '%-8s'\n", execaddr, priority, stack, name))
	t = get_tcb();
	wordset(t->SpLow = stack, SBOUND, ssize - 14);
	stack += ssize;
	// Setup task initial stack
	*--stack = execaddr;		// PC
	*--stack = 0;				// R12
	*--stack = 0;				// R11
#ifdef OO_DEV
	*--stack = ((unsigned)t->SpLow) + 256;	// R10 (Stack Limit)
#else
	*--stack = 0;				// R10
#endif
	*--stack = 0;				// R9
	*--stack = 0;				// R8
	*--stack = 0;				// R7
	*--stack = 0;				// R6
	*--stack = 0;				// R5
	*--stack = 0;				// R4
	*--stack = 0;				// R3
	*--stack = 0;				// R2
	*--stack = 0;				// R1
	*--stack = 0;				// R0
	t->Sp = stack;				// Set initial stack
	t->Priority = priority;		// Set priority
	t->Next = Tready[priority];	// Get next in list
	t->Mqueue = 0;				// Zero message queue
	t->State = 1;				// Set state
	t->Name = name;				// Set task name
	t->Mqueue_size = 0;			// Zero message queue
	t->Mqueue_limit = mlimit;		// Default message limit
#ifdef OO_MHIGH
	t->Mqueue_high = 0;			// Reset high-water mark
#endif
	Tready[priority] = t;		// Link into list
	return t->Id;				// Send back Task ID
}

/*
 * Remove a task from the ready list
 */
static void OS_proc_unready(unsigned id, unsigned char state)
{
	struct TCB *t, *l, **ll;
	unsigned char p;

	t = &Task_list[id];		// Get task pointer
	Debug(("Unready %u (%08x) [%08x] = %u\n", id, t, t->Next, state))

	ll = &Tready[p = t->Priority];
	while(l = *ll) {
		Debug(("%08x:%08x: ll=%08x *ll=%08x l=%08x t=%08x  \n", GetSp(), OS_active_tcb, ll,  *ll, l, t))
		if(l == t) {		// We have located task
		Debug(("->%08x\n", t->Next))
			*ll = t->Next;	// Remove us from link
			break; }		// Exit loop
		ll = &l->Next; }
	t->State = state;

	if(state == TASK_DEAD) {		// Task is Terminating
		if(OS_console_owner == id)
			OS_console_owner = 0;		// Release console
		if(OS_at_owner == id)
			OS_at_owner = 0; }			// Release AT redirection

	if(t == OS_active_tcb)				// Executing - swap out to halt
		__OSdispatch_next();
	else if(Tnext[p] == t)				// Next-up - skip to prevent exec
		Tnext[p] = t->Next;
}

/*
 * Ready a task for execution
 */
void OS_proc_ready(unsigned id, unsigned value)
{
	struct TCB *t;
	unsigned char p;
	OStest("proc_ready");
	t = &Task_list[id];
	Debug(("MakeReady: %u (%08x)\n", id, t))
	if(t->State > 1) {
		t->Sp[0] = value;
		p = t->Priority;
		t->Next = Tready[p];
		Tready[p] = t;
		t->State = TASK_READY; }
}

/*
 * Establish a timer
 */
static unsigned OS_timer_start(unsigned count, unsigned set, unsigned control, unsigned up)
{
	struct TIMER *t;
	if(!(t = Timer_free))
		OS_panic("No timer blocks");
	Debug(("SetTimer: %08x %u %u %08x\n", t, count, set, control))
	Timer_free = t->Link;
	t->Countdown = count;
	t->Setting = set;
	t->Control = control ? control : (OS_active_tcb->Id+1);
	t->Uparm = up;
	t->Link = Timer_base;
	Timer_base = t;
	return ((unsigned)t - (unsigned)&Timer_list) / sizeof(Timer_list[0]);
}

/*
 * End a timer
 */
static void OS_timer_stop(unsigned index)
{
	struct TIMER *t, **t1, *t2;
	t1 = &Timer_base;
	t2 = &Timer_list[index];
	while(t = *t1) {
		if(t == t2) {
			*t1 = t->Link;
			t->Link = Timer_free;
			Timer_free = t;
			return; }
		t1 = &t->Link; }
}

/*
 * Queue a message for console output
 */
void OS_console_tx(struct MESSAGE *m)
{
	struct MESSAGE **p;
	OStest("console_tx");
	OSmfree(m, "ConTX");
	p = &Uart_tx_msg;
	while(*p)		// Walt to end of list
		p = &(*p)->Link;
	m->Link = 0;
	*p = m;
	INTMR1 |= 0x00001000;	// Enable the interrupt
}

static unsigned OS_console_rx(unsigned char *data, unsigned length)
{
	unsigned i = 0;

	while((OS_console_rx_read != OS_console_rx_write) && (i < length)) {
		data[i++] = OS_console_rx_buffer[OS_console_rx_read];
		OS_console_rx_read = (OS_console_rx_read + 1) & (sizeof(OS_console_rx_buffer)-1); }
	return i;
}

/*
 * Timer interrupt handler - dispatch requested messages/callbacks
 * This function runs in "interrupt" mode
 */
void IRQ_timer1(void)
{
	unsigned i;
	struct TIMER *t, **tl;
	struct MESSAGE *m;

	++OS_timer_tick;

	tl = &Timer_base;
	while(t = *tl) {
		if(!--t->Countdown) {
			if(i = t->Control) {
				if(i > MAX_TASKS)				// Function
					((PFV)i)(t->Uparm);
				else {							// Message
					m = OS_msg_small();
					m->Info1 = ((unsigned)t - (unsigned)&Timer_list) / sizeof(Timer_list[0]);
					*((unsigned*)m->Data) = t->Uparm;
					if(OS_msg_send(i-1, m, TID_TIMER) > 1) {
#ifdef OO_DEV
						OS_last_rel = 1;
#endif
						OS_msg_release(m);
					}
				}
			}
			if(!(t->Countdown = t->Setting)) {	// One-shot - delete
				*tl = t->Link;
				t->Link = Timer_free;
				Timer_free = t;
				continue; }
		}
		tl = &t->Link;		// Skip to next timer
	}
	TC1EOI = 0;
#ifdef OO_TIME
	OS_irq_exit(XIRQ_timer1);
#endif
}

/*
 * Uart transmit interrupt
 */
void IRQ_uart_tx(void)
{
	struct MESSAGE *m;

	if(!(m = Uart_tx_msg)) {	// No message available
		INTMR1 &= ~0xFFFF1000;	// Disable UART TX
#ifdef OO_TIME
		OS_irq_exit(XIRQ_uart_tx);
#endif
		return; }

	while(!(SYSFLG1 & 0x00800000)) {
		UARTD = m->Data[Uart_tx_position];
		if(++Uart_tx_position >= m->Info1) {
			Uart_tx_position = 0;
			Uart_tx_msg = m->Link;
#ifdef OO_DEV
			OS_last_rel = 2;
#endif
			OS_msg_release(m);
			if(!(m = Uart_tx_msg)) {
				INTMR1 &= ~0xFFFF1000;	// Disable UART TX
#ifdef OO_TIME
			OS_irq_exit(XIRQ_uart_tx);
#endif
				return; } } }
#ifdef	OO_TIME
	OS_irq_exit(XIRQ_uart_tx);
#endif
}

void IRQ_uart_rx(void)
{
//	unsigned char c;
	while(!(SYSFLG1 & 0x00400000)) {		// Data available
		OS_console_rx_buffer[OS_console_rx_write] = /* c = */ UARTD;
		OS_console_rx_write = (OS_console_rx_write+1) & (sizeof(OS_console_rx_buffer)-1);
		/* if(c == 0x1A) OS_panic("^Z"); */ }
#ifdef OO_TIME
	OS_irq_exit(XIRQ_uart_rx);
#endif
}

#include "osnet.c"
#include "osfile.c"

/*
 * Application call to OS
 */
void __OSentry(void)
{
	Debug(("ENTRY-%08x: %u %08x %08x %08x %08x\n", OS_active_tcb, get_parm(0), get_parm(1), get_parm(2), get_parm(3), get_parm(14)))

#ifdef OO_DEV
	if(OS_irq_active)
		OS_panic("OS_call(%u) from INT", get_parm(0));
	OS_irq_active = 0x55;
#endif
	switch(get_parm(0)) {
	case OS_PROC_START :		// Start
		send_result(OS_proc_start(get_parm(1), (unsigned char)get_parm(2), (unsigned *)get_parm(3), (unsigned)get_parm(14), (unsigned char*)get_parm(15), (unsigned)get_parm(16)));
		OSexit();
	case OS_PROC_UNREADY :		// Unready
		OS_proc_unready(get_parm(1), get_parm(2));
		OSexit();
	case OS_PROC_READY :		// Make ready
		OS_proc_ready(get_parm(1), get_parm(2));
		OSexit();
	case OS_PROC_WAIT :			// Wait on event
		OS_proc_wait(get_parm(1));
	case OS_PROC_SIGNAL :		// Signal an event
		OS_proc_signal(get_parm(1), get_parm(2));
		OSexit();
	case OS_MSG_SMALL:			// Allocate small  message
		send_result((unsigned)OS_msg_small());
		OSexit();
	case OS_MSG_MEDIUM:			// Allocate medium message
		send_result((unsigned)OS_msg_medium());
		OSexit();
	case OS_MSG_LARGE:			// Allocate large message
		send_result((unsigned)OS_msg_large());
		OSexit();
	case OS_MSG_RELEASE:		// Release message
#ifdef OO_DEV
		OS_last_rel = 3;
#endif
		OS_msg_release((struct MESSAGE*)get_parm(1));
		OSexit();
	case OS_MSG_SEND :			// Send message
		send_result(OS_msg_send(get_parm(1), (struct MESSAGE *)get_parm(2), OS_active_tcb->Id));
		OSexit();
	case OS_MSG_REQUEUE:		// Requeue list of messages
		OS_msg_requeue((struct MESSAGE*)get_parm(1), (struct MESSAGE*)get_parm(2));
		OSexit();
	case OS_MSG_RECEIVE:		// Receive message
		send_result((unsigned)OS_msg_receive());
		OSexit();
	case OS_MSG_CHECK:			// Check for message
		send_result((unsigned)OS_msg_check());
		OSexit();
	case OS_TIMER_START:		// Start a timer
		send_result(OS_timer_start(get_parm(1), get_parm(2), get_parm(3), get_parm(14)));
		OSexit();
	case OS_TIMER_STOP:			// Stop a timer
		OS_timer_stop(get_parm(1));
		OSexit();
	case OS_IRQ_DISABLE:		// Disable interrupts
		//rawprintf("ENABLE: %08x", __OSgetspsr());
		__OSdisable();
		//rawprintf(" %08x\n", __OSgetspsr());
		OSexit();
	case OS_IRQ_ENABLE:			// Enable interrupts
		//rawprintf("DISABLE: %08x", __OSgetspsr());
		__OSenable();
		//rawprintf(" %08x\n", __OSgetspsr());
		OSexit();
	case OS_IRQ_SET:			// Set external IRQ
		OS_eirq_mask |= get_parm(1);
		IRQS = OS_eirq_mask;
		OSexit();
	case OS_IRQ_CLEAR:			// Clear external IRQ
		OS_eirq_mask &= ~get_parm(1);
		IRQS = OS_eirq_mask;
		OSexit();
	case OS_SYS_STOP:			// Halt system
		__OSsystem();
	case OS_SYS_RESTART:		// Restart system
		__OSboot();
	case OS_CONSOLE_TX:			// Transmit message to console
		if(OS_console_owner && (OS_active_tcb->Id != OS_console_owner))
			goto sendfail;
		OS_console_tx((struct MESSAGE*)get_parm(1));
	sendok:
		send_result(0);
		OSexit();
	case OS_CONSOLE_RX:			// Receive character
		if(OS_active_tcb->Id == OS_console_owner) {
			send_result(OS_console_rx((unsigned char*)get_parm(1), get_parm(2)));
			OSexit(); }
	sendfail:
		send_result(0xFFFFFFFF);
		OSexit();
	case OS_CONSOLE_ALLOC:		// Allocate console
		if(OS_console_owner) goto sendfail;
		OS_console_owner = OS_active_tcb->Id;
		goto sendok;
	case OS_CONSOLE_RELEASE:	// Release console
		if(OS_console_owner != OS_active_tcb->Id) goto sendfail;
		OS_console_owner = 0;
		goto sendok;
	case OS_CONSOLE_AT:			// Allocate AT commands
		if(OS_at_owner && (OS_at_owner != OS_active_tcb->Id))
			OS_panic("Two AT owners");
		OS_at_owner = OS_active_tcb->Id;
		OSexit();
	case OS_ETHERNET_TX:		// Ethernet transmit
		if(OS_system_config & CONFIG_NIC) {
			send_result(OS_ethernet_tx((unsigned char*)get_parm(1), get_parm(2)));
			OSexit(); }
		OS_panic("NoNIC");
	case OS_ETHERNET_MAC:		// Get ethernet address
		if(OS_system_config & CONFIG_NIC) {
			send_result((unsigned)nic_address);
			OSexit(); }
		goto sendok;
	case OS_ETHERNET_REGISTER:	// Register for packet type
		OS_ethernet_register(get_parm(1));
		OSexit();
	case OS_ETHERNET_RELEASE:	// Release packet type
		OS_ethernet_release(get_parm(1));
		OSexit();
	case OS_STORE_READ:			// Read bulk storage
		send_result(read_file(get_parm(1), get_parm(2), (unsigned char*)get_parm(3)));
		OSexit();
	case OS_STORE_WRITE:			// Write bulk storage
		send_result(write_file(get_parm(1), get_parm(2), (unsigned char*)get_parm(3)));
		OSexit();
	case OS_STORE_ERASE:			// Erase bulk storage
		send_result(erase_file(get_parm(1), get_parm(2)));
		OSexit();
	case OS_MUTEX_REQUEST:			// Request MUTEX
		send_result(OS_mutex_request(get_parm(1), get_parm(2)));
		OSexit();
	case OS_MUTEX_RELEASE:			// Release MUTEX
		OS_mutex_release(get_parm(1));
		OSexit();
	case OS_BLS:					// Set bottom latch
		OS_bls(get_parm(1));
		OSexit();
	case OS_BLC:					// Clear bottom latch
		OS_blc(get_parm(1));
		OSexit();
	case OS_TLS:					// Set top latch
		OS_tls(get_parm(1));
		OSexit();
	case OS_TLC:					// Clear top latch
		OS_tlc(get_parm(1));
		OSexit();
	case OS_PANIC_REGISTER:
//	#ifdef OO_DEV
		OS_panic_register(get_parm(1), get_parm(2));
//	#endif
		OSexit();
	case OS_SYS_SERV:
		switch(get_parm(1)) {
case 1 :
	rawprintf("\n%u %u %u %u %u %u %u %u",
		get_parm(1), get_parm(2), get_parm(3), get_parm(14),
		get_parm(15), get_parm(16), get_parm(17), get_parm(18));
	OSexit();
			case SS_INV_TLB :
				__OSinvtlb(0);
				OSexit(); }
	} OS_panic("Bad kernel call (%u)", get_parm(0));
}

/*
 * Scan data memory to locate top of RAM
 */
unsigned locate_ram_top(void)
{
	unsigned a, b;
	volatile unsigned *p;
	p = (unsigned*)DATA_RAM_START;
	do {
		p += 1024;			// 4k chunk
		*p = 0x12345678;
		a = *p;
		*p = 0x98765432;
		b = *p;
		WDOG = *p = 0; }
	while((a == 0x12345678) && (b == 0x98765432) && !*p);
	return (unsigned)p;
}

// MMU may segment entry templates for each main address boundary
// init_mmu() builds 16 MMU segment entries from each template.
const unsigned mmu_init_table[] = {
	0xC0000C0E,		// bank 0 - SDRAM	(Cached & buffered)
	0x10000C02,		// Bank 1 - 8-bit I/O
	0x20000C02,		// Bank 2 - 16 bit I/O
	0x30000C02,		// Bank 3 - 32 bit I/O
	0x40000C02,		// Bank 4
	0x50000022,		// Bank 5
	0x60000C0E,		// Bank 6 - I-SRAM	(Cached & buffered)
	0x70000C02,		// Bank 7 - Internal ROM
	0x80000C02,		// Bank 8 - System registers
	0x90000022,		// Bank 9
	0xA0000022,		// Bank A
	0xB0000022,		// Bank B
	0x00000C02,		// Bank C - FLASH	!(Cached & Buffered)
	0xD0000022,		// Bank D
	0xE0000022,		// Bank E
	0xF0000022 };	// Bank F *

/*
 * Re-build MMU mapping table in RAM with updated protection information
 * The invalidate the TLB to force loading of the new table
 */
void init_mmu()
{
	unsigned i, j, *d;
	rawprintf("\nInitializing MMU and enabling memory protection");
	// Build MMU segment mapping table
	d = (unsigned*)0x800000;
	for(i=0; i < 4096; ++i) {
		j = mmu_init_table[i >> 8] | ((i & 0xFF) << 20);
		if(i < 8)	// Write protect only code space in SRAM
			j &= ~0xC00;
		d[i] = j; }
	OS_call(OS_SYS_SERV, SS_INV_TLB);
}

void __OSmain()
{
	unsigned i, j, k;
	unsigned char *cptr, c, temp[33];
	struct MESSAGE *m;
	static const ccp words[] = {
		"DEBUG",		// 0
		"DUMP",			// 1
		"HALT",			// 2
		"INFO",			// 3
		"PACKET",		// 4
		"POKE",			// 5
		"PS",			// 6
		"SIGNAL",		// 7
		"TX",			// 8
		"READ",			// 9
		"ERASE",		// 10
		"WRITE",		// 11
		"PROTECT",		// 12
//		"DALLAS",		// 12
//"X",
		0 };

	WDOGE	= 1;				// Enable Watchdog

	// Configure system registers
	SYSCON1 = 0x000301B2;		// Timer-1 auto-reload, 512khz
#ifdef OO_90MHZ
	PLLW	= 0x31000000;		// Configure for 90Mhz
	UBRLCR1	= 0x00078004;		//  56k @ 90Mhz
#else
//	PLLW	= 0x28000000;		// Configure for 74Mhz
//	UBRLCR1 = 0x00078001;		// 115k FIFO enabled
	UBRLCR1 = 0x00078003;		//  56k FIFO enabled
//	UBRLCR1 = 0x00078017;		// 9600 FIFO enabled
//	UBRLCR1 = 0x00068017;		// 9600 FIFO disabled
#endif

	PADDR = 0xF0;				// Set PORT-A direction
	PADR = 0x30;				// Set PORT-A data
	IRQS = 0;					// Reset enabled interrupts

	// Zero OS data segment & Stack space
	wordset((unsigned*)0x60000000, 0, 0xBF80/4);

	rawputs((char*)hello);
	rawputs("\nDRAM size: ");
	OS_ram_top = __OSinitram();
	rawprintf("%,u bytes", OS_ram_top);

	// Initialize system latches
	BLATCH = OS_blatch	= 0x1F400CF0;
	TLATCH0 = TLATCH1 = OS_tlatch	= 0x00000CF0;

#ifdef OO_DEV
	OS_irq_active = 0xAA;
#endif
	// Establish inital message queue
	rawputs("\nEstablishing OS data structures & initializing kernel");
	for(i=0; i < (MAX_SMALL-1); ++i) {
		#ifdef OO_DEV
			Small_list[i].Info = TID_FREE;
		#endif
		Small_list[i].Link = &Small_list[i+1]; }
	for(i=0; i < (MAX_MEDIUM-1); ++i) {
		#ifdef OO_DEV
			((struct MESSAGE*)(Medium_list[i]))->Info = TID_FREE;
		#endif
		((struct MESSAGE*)(Medium_list[i]))->Link = (struct MESSAGE*)&Medium_list[i+1]; }
	for(i=0; i < (MAX_LARGE-1); ++i) {
		#ifdef OO_DEV
			((struct MESSAGE*)(Large_list[i]))->Info = TID_FREE;
		#endif
		((struct MESSAGE*)(Large_list[i]))->Link = (struct MESSAGE*)&Large_list[i+1]; }
	Small_free = &Small_list[0];
	Medium_free = (struct MESSAGE*)Medium_list;
	Large_free = (struct MESSAGE*)Large_list;
	Small_list[MAX_SMALL-1].Link = 0;
	((struct MESSAGE*)Medium_list[MAX_MEDIUM-1])->Link = 0;
	((struct MESSAGE*)Large_list[MAX_LARGE-1])->Link = 0;
	#ifdef OO_DEV
		Small_list[MAX_SMALL-1].Info = TID_FREE;
		((struct MESSAGE*)Medium_list[MAX_MEDIUM-1])->Info = TID_FREE;
		((struct MESSAGE*)Large_list[MAX_LARGE-1])->Info = TID_FREE;
	#endif
	OS_free_msg_small  = MAX_SMALL;
	OS_free_msg_medium = MAX_MEDIUM;
	OS_free_msg_large  = MAX_LARGE;
#ifdef OO_MLOW
	OS_free_min_small  = MAX_SMALL;
	OS_free_min_medium = MAX_MEDIUM;
	OS_free_min_large  = MAX_LARGE;
#endif

	for(i=0; i < (MAX_MUTEX-1); ++i)
		Mutex[i].Link = &Mutex[i+1];
	Mutex_free = &Mutex[0];


	// Establish initial timer queue
	for(i=0; i < (MAX_TIMER-1); ++i)
		Timer_list[i].Link = &Timer_list[i+1];
	Timer_list[MAX_TIMER-1].Link = 0;
	Timer_free = &Timer_list[0];

	// Establish this thread as the background "idle" task
	Tready[PRIORITIES] = Tnext[PRIORITIES] = OS_active_tcb = get_tcb();
	OS_active_tcb->Name = (unsigned char*)"OS";
	OS_active_tcb->Priority = PRIORITIES;
	OS_active_tcb->State = 1;
	OS_active_tcb->Next = OS_active_tcb;
	OS_active_tcb->Mqueue_size = 0;
	OS_active_tcb->Mqueue_limit = 50;
#ifdef OO_MHIGH
	OS_active_tcb->Mqueue_high = 0;
#endif
	wordset(OS_active_tcb->SpLow = (unsigned*)O_STACK, SBOUND, E_STACK-O_STACK);

	init_file();
	m = OS_msg_medium();
	bytecpy(nic_address, (unsigned char*)MAC_OFFSET, 6);
#ifdef OO_DEV
	OS_last_rel = 4;
#endif
	OS_msg_release(m);

	rawputs("\nDetected hardware:");
	i = BPCBID & 15;
	j = BPCBREV & 15;
	OS_system_config = (i << 4) | j;
	switch(OS_system_config & 0xF0) {
	case 0x30 :
		if(j < 2) {
			rawputs(" 2X8");
			OS_hw_type = 1;
			if(IRQS & (1 << 19)) {
				rawputs("+T");
				OS_system_config |= CONFIG_TOP; } }
		else {
			while(SYSFLG1 & 0x00000800);	// Wait for UART TX complete
			PLLW	= 0x31000000;			// Configure for 90Mhz
			UBRLCR1	= 0x00078004;			// 56k @ 90Mhz
			rawputs("2xN");
			OS_hw_type = 2; }
		break;
	default:
		PADDR = 0xF1;		// Bit-0 must be output on 4x8
		rawputs(" 4X8"); }

	// Test for ethernet/top board
	if(!init_network()) {
		OS_system_config |= CONFIG_NIC;
		OS_eirq_mask |= IRQ_ETHERNET;
		rawputs(" NIC"); }

	rawprintf("\nSystem board: %x", OS_system_config & 255);
	rawputs  ("\nTop    board: ");
	if(OS_system_config & CONFIG_TOP)
		rawprintf("%02x", (OS_system_config >> 8) & 255);
	else
		rawputs("Not present");

#ifdef OO_DEV
	OS_irq_active = 0;
	init_mmu();
#endif
	rawputs("\nInitializing timer & enabling system interrupts");

	// Establish timer interval
#ifdef OO_90MHZ
	TC1D = 784;					// = 1.25ms tick (627.2 Khz)
#else
	TC1D = (OS_hw_type >= 2) ? 784 : 640;	// = 1.25ms tick (512khz)
#endif

#ifdef OO_TIME
	INTMR1 = 0x00002320;		// Enable timer-1 interrupt
#else
	INTMR1 = 0x00002120;		// Enable timer-1 interrupt
#endif
	IRQS = OS_eirq_mask;		// Enable extended interrupts
	OS_console_echo = 255;		// Enable console echo

	rawprintf("\nTransitioning to user mode & launching application");
	user_init();
	printf("\r\nEntering console command processor");

//	l = ml = 0;
recmd:
	WDOG = 0;
	for(i=0; i < 50; ++i) {
		j = OS_timer_tick;
		OS_swap();
		if((Latency = OS_timer_tick - j) > MaxLatency)
			MaxLatency = Latency; }
	*OS_console_buffer = 0;
reprompt:
	if(OS_console_echo)
		printf("\r\n%s", OS_console_buffer);
	else
		printf("\r\n");
	i = strlen(OS_console_buffer);
	for(;;) {
		j = OS_timer_tick;
		OS_swap();

		if((Latency = OS_timer_tick - j) > MaxLatency)
			MaxLatency = Latency;
		WDOG = 0;
		if(m = OS_active_tcb->Mqueue) {		// Received a message
			OS_active_tcb->Mqueue = m->Link;
			--OS_active_tcb->Mqueue_size;
			printf("\r\nMessage from: %u %02x %04x %08x", m->Info, m->Info1, m->Info2, m->Link);
			dump(m->Data, sizeof(m->Data), &printf, &testc);
			OS_call(OS_MSG_RELEASE, m);
			goto reprompt; }
		if(OS_call(OS_CONSOLE_RX, &c, 1) == 1) switch(c) {
			default:
				if((c >= ' ') && (c <= 0x7F) && (i < (sizeof(OS_console_buffer)-1))) {
					if(OS_console_echo)
						printf("%c", c);
					OS_console_buffer[i++] = c;
					OS_console_buffer[i] = 0; }
				continue;
			case 0x1B :
				OS_console_buffer[i] = 0;
				goto reprompt;
			case '\b' :
				if(i) {
					if(OS_console_echo)
						printf("\b \b");
					--i; }
				continue;
			case '\r' :
				OS_console_buffer[i] = 0;
				cptr = OS_console_buffer;
				if(!skip_blanks(&cptr)) goto recmd;
				j = (unsigned)cptr;
				parse(&cptr, temp, 0x1000+32);
				if(strbeg(temp, "AT|") && OS_at_owner) {
					m = (struct MESSAGE*)OS_call(OS_MSG_SMALL);
					strcpy(m->Data, (unsigned char*)(j+3));
					if(OS_call(OS_MSG_SEND, OS_at_owner, m) > 1)
						OS_call(OS_MSG_RELEASE, m);
					goto recmd; }
				for(i=0; words[i]; ++i) {
					if(!strcmp((void*)words[i], temp))
						goto found; }
				for(i=0; i < MAX_TASKS; ++i) {
					if(Task_list[i].State) {
						if(!strcmp(Task_list[i].Name, temp)) {
							if(!skip_blanks(&cptr)) {
								printf("\r\nParameter required");
								goto recmd; }
							m = (struct MESSAGE *)OS_call(OS_MSG_SMALL);
							if(*cptr == '~') {
								++cptr;
								k = 0;
								while(!parse_number(&cptr, &j, 255))
									m->Data[k++] = j; }
							else
								strcpy(m->Data, cptr);
							if(OS_call(OS_MSG_SEND, Task_list[i].Id, m) > 1)
								OS_call(OS_MSG_RELEASE, m);
							goto recmd; } } }
				printf("\r\n'%s' unknown, use:", temp);
				for(i=0; words[i]; ++i) {
					if(!(i & 3)) printf("\r\n");
					printf(" %-8s", words[i]); }
				printf("\r\nuse 'proc-name text' to send message to process");
			case 0x03:		// Control-C
				goto recmd;
		found:	switch(i) {
				case 0 :	// Debug
					parse_number(&cptr, &OS_debug_mask, 255);
					goto recmd;
				case 1 :	// Dump
					if(parse_number(&cptr, &i, 255)) goto recmd;
					if(parse_number(&cptr, &j, 255)) j = i + 255;
					if(j < i) continue;
					dump((void*)i, (j-i)+1, &printf, &testc);
					goto recmd;
				case 2 :	// Halt
					OS_panic("System halted");
					goto recmd;
				case 3 :	// Info
					if(!parse_number(&cptr, &i, 255)) {
						OS_eirq_mask = 0;
						OS_call(OS_IRQ_SET, i); }
					info(&printf);
					MaxLatency = 0;
#ifdef OO_MLOW
					OS_free_min_small  = OS_free_msg_small;
					OS_free_min_medium = OS_free_msg_medium;
					OS_free_min_large  = OS_free_msg_large;
#endif
					goto recmd;
				case 4 :	// PACKET
					if(!parse_number(&cptr, &i, 255)) {
						if(parse_number(&cptr, &j, 255))
							OS_call(OS_ETHERNET_REGISTER, i);
						else
							OS_ethernet_multi(i, j); }
					packet(&printf);
					netinfo(&printf);
					goto recmd;
				case 5 :	// poke
					if(!parse_number(&cptr, &i, 255)) {
						while(!parse_number(&cptr, &j, 255)) {
							*(unsigned char*)i = j;
							++i; } }
					goto recmd;
				case 6 :	// ps
					ps(cptr, &printf);
					goto recmd;
				case 7 :	// SIGNAL
					if(!parse_number(&cptr, &i, 0)) {
						j = 0;
					 	parse_number(&cptr, &j, 0);
						OS_call(OS_PROC_SIGNAL, i, j); }
					goto recmd;
				case 8 :	// TX
					if(parse_number(&cptr, &j, 255)) goto recmd;
					skip_blanks(&cptr);
					m = (struct MESSAGE*)OS_call(OS_MSG_SMALL);
					memset(m->Data, 0xFF, 6);
					memcpy(m->Data+6, (char*)OS_call(OS_ETHERNET_MAC), 6);
					m->Data[12] = j >> 8;
					m->Data[13] = j;
					memset(m->Data+16, 0x55, sizeof(m->Data)-16);
					strcpy(m->Data+16, cptr);
					OS_call(OS_ETHERNET_TX, m->Data, sizeof(m->Data));
					OS_call(OS_MSG_RELEASE, m);
					goto recmd;
				case 9 :	// READ
					c = 0;
					if(skip_blanks(&cptr) == '~') {
						c = 255;
						++cptr; }
					if(parse_number(&cptr, &i, k = 0)) goto recmd;
					if(skip_blanks(&cptr) == '+') k = 256;
					m = (struct MESSAGE*)OS_call(OS_MSG_MEDIUM);
					j = OS_call(OS_STORE_READ, c, i, m->Data);
					printf("\r\nResult: %d", j);
					dump(m->Data+k, 256, &printf, &testc);
					OS_call(OS_MSG_RELEASE, m);
					goto recmd;
				case 10 :	// ERASE
					c = 0;
					if(skip_blanks(&cptr) == '~') {
						c = 255;
						++cptr; }
					if(parse_number(&cptr, &i, 0)) goto recmd;
					if((i >> 28) == 0x0C) {
						printf("\r\n%u", flash_erase_block(i));
						goto recmd; }
					k = i;
					parse_number(&cptr, &k, c);
					while(i <= k) {
						WDOG = 0;
						printf("\r%u", i);
			retry:		if(j = OS_call(OS_STORE_ERASE, 0, i)) {
							if(j == 1) goto retry;
							printf("\r\nError: %x", j);
							goto recmd; }
						++i; }
					goto recmd;
				case 11 :	// Write
					c = 0;
					if(skip_blanks(&cptr) == '~') {
						c = 255;
						++cptr; }
					if(parse_number(&cptr, &i, k=0)) goto recmd;
					m = (struct MESSAGE*)OS_call(OS_MSG_MEDIUM);
					memset(m->Data, 0xFF, 512);
					while(!parse_number(&cptr, &j, 255))
						m->Data[k++] = j;
					if((i >> 28) == 0x0C)
						j = flash_program_block(i, (int)m->Data, k);
					else
						j = OS_call(OS_STORE_WRITE, c, i, m->Data);
					printf("\r\nResult: %d", j);
					OS_call(OS_MSG_RELEASE, m);
					goto recmd;
				case 12 :		// PROTECT
					parse(&cptr, temp, 0x1000+32);
					if(strbeg(temp, "/D"))
						OS_panic_active = 0x55;
					init_mmu();
					goto recmd;
/*				case 12 :		// Dallas
					if(parse_number(&cptr, &i, 255)) goto recmd;
					if(parse_number(&cptr, &j, 255))
						printf("\r\nDallas %02x: %02x", i, DallasRead(i));
					else
						DallasWrite(i, j);
					goto recmd;
				case 13:
					Eprintf(1, "test1");
					Eprintf(2, "test2");
					Eprintf(3, "test3");
					goto recmd; */
	} } }
}
