/*
 * ArmOS - System definitions.
 *
 * Dave Dunfield - 98/10/04
 *	03/11/02 - Added OS_IRQ_* services
 *	03/11/03 - Added OS_CONSOLE services, OS_console_owner & OS_debug_mask
 *	03/11/05 - Added OS_PROC_WAIT/OS_PROC_SIGNAL kernel services
 *	03/11/09 - Added OS_ETHERNET_* servces
 *	03/11/10 - Added Medium and Large message blocks
 *	03/11/14 - Added OS_STORE_* services (Temporary)
 *  03/12/08 - Added CONFIG_* and OS_system_config
 *	03/12/18 - Added OS_* prototypes
 */

#ifndef __OS_H_
	#define __OS_H_

	// OS option flags (require full recompile when changed)
#ifdef TS_DEBUG
	#define	OO_DEV				// Development version
	#define	OO_MLOW				// Track message free pool low-water
	#define	OO_MHIGH			// Track task message queue high-water
	// OO_TIME value is number of seconds history - must be power of two.
	// If you undef OO_TIME in debug build, don't forget to disable OO_TIME
	// in BEGIN.S.
	#define	OO_TIME	16			// Monitor max task CPU time
	// OO_RT_MASK controls how often performance reports are generated
	// It MUST be a power-of-2 and represents ticks of 128us.
	// ie: 8=1.024 sec, 64=8.192 seconds
	#define	OO_RT_MASK 64
#endif

	// General OS parameters
	#define	PRIORITIES				3		// Number of user priorities
	#define	MAX_TASKS				25		// Maximum # of processes (tasks)
	#define	MAX_IRQ					5		// Maximum # of IRQ handlers
	#define	MAX_TIMER				50		// Maximum # of timers
	#define	MAX_SIGNAL				10		// Maximum # of signal events
	#define	MAX_MUTEX				10		// Maximum # mutexes
	#define	MAX_MQUEUE				50		// Maximum # pending mutexes
	#define	MAX_SMALL				500		// Maximum # of small  messages
	#define	MAX_MEDIUM				200		// Maximum # of medium messages
	#define	MAX_LARGE				25		// Maximum # of large  messages
	#define	MAX_PACKET				10		// Maximum # of registered packet types
	#define	MAX_PANIC				10		// Maximum # panic extensions
	#define	MSG_SMALL				128		// Size of small  message data area
	#define	MSG_MEDIUM				512		// Size of medium message data area
	#define	MSG_LARGE				1536	// Size of large  message data area
	
	// Process execution states
	#define	TASK_DEAD				0		// Process does not exist
	#define	TASK_READY				1		// Process is ready to execute
	#define	TASK_RECEIVE			2		// Process is receive blocked
	#define	TASK_MUTEX				3		// Process is mutex blocked
	#define	TASK_SIGNAL				4		// Process is signal blocked
	
	// Kernel service identification codes
	#define	OS_PROC_START			0		// Start a process
	#define	OS_PROC_UNREADY			1		// Unready a process
	#define	OS_PROC_READY			2		// Make process ready
	#define	OS_PROC_WAIT			3		// Wait on event
	#define OS_PROC_SIGNAL			4		// Signal event
	#define	OS_MSG_SMALL			5		// Allocate small  message
	#define	OS_MSG_MEDIUM			6		// Allocate medium message
	#define	OS_MSG_LARGE			7		// Allocate large  message
	#define	OS_MSG_RELEASE			8		// Release message
	#define	OS_MSG_SEND				9		// Send message
	#define	OS_MSG_RECEIVE			10		// Receive message
	#define	OS_MSG_CHECK			11		// Check for a message
	#define	OS_TIMER_START			12		// Start a timer
	#define	OS_TIMER_STOP			13		// Stop a timer
	#define	OS_IRQ_DISABLE			14		// Disable ALL interrupts
	#define	OS_IRQ_ENABLE			15		// Enable ALL interrupts
	#define	OS_IRQ_SET				16		// Set external IRQ enable
	#define	OS_IRQ_CLEAR			17		// Clear external IRQ enable
	#define	OS_SYS_STOP				18		// Stop system
	#define	OS_SYS_RESTART			19		// Restart system
	#define	OS_CONSOLE_TX			20		// Transmit to console
	#define	OS_CONSOLE_RX			21		// Test for character
	#define	OS_CONSOLE_ALLOC		22		// Allocate console
	#define	OS_CONSOLE_RELEASE		23		// Release console
	#define	OS_CONSOLE_AT			24		// Allocate AT commands
	#define	OS_ETHERNET_TX			25		// Ethernet transmit
	#define	OS_ETHERNET_MAC			26		// Get ethernet address
	#define	OS_ETHERNET_REGISTER	27		// Register interest in a packet type
	#define	OS_ETHERNET_RELEASE		28		// Release  interest in a packet type
	#define	OS_STORE_READ			29		// Read bulk device  (ECC)
	#define	OS_STORE_WRITE			30		// Write bulk device (ECC)
	#define	OS_STORE_ERASE			31		// Erase bulk device
	#define	OS_MUTEX_REQUEST		32		// Allocate mutex
	#define	OS_MUTEX_RELEASE		33		// Release mutex
	#define	OS_BLS					34		// Set bottom latch bits
	#define	OS_BLC					35		// Clear bottom latch bits
	#define	OS_TLS					36		// Set top latch bits
	#define	OS_TLC					37		// Clear top latch bits
	#define	OS_MSG_REQUEUE			34		// Requeue messages
	#define	OS_PANIC_REGISTER		35		// Register panic handler
	#define	OS_SYS_SERV				36		// System services

	// Special system services
	#define	SS_INV_TLB				0		// Invalidate ARM TLB
	
	// System configuration flags (OS_system_config)
	#define	CONFIG_NIC			0x10000000	// Nic is present
	#define	CONFIG_SMCARD		0x04000000	// SmartMedia card is present
	#define	CONFIG_SVER			0x000000FF	// System board revision
	#define	CONFIG_TVER			0x0000FF00	// Top    board revision
	
	/*
	 * Message block
	 */
	struct MESSAGE {
		struct MESSAGE	*Link;			// Link to next
		unsigned char	Info;			// Information byte (usually owner/from)
		unsigned char	Info1;			// Information byte (general)
		unsigned short	Info2;			// Information word (general)
		unsigned char	Data[MSG_SMALL]; // Data block
		};
	
	/*
	 * Task control block
	 */
	struct TCB {
		unsigned 		*Sp;				// Task SP
		unsigned		*SpLow;				// SP low boundary marker
		struct TCB 		*Next;				// Link to next task
		struct MESSAGE	*Mqueue;			// Message queue
		unsigned char	*Name;				// Pointer to task name
		unsigned char	Id;					// Task ID
		unsigned char	State;				// Task state
		unsigned char	Priority;			// Task priority
		unsigned		Mqueue_size;		// Number of messages queued
		unsigned		Mqueue_limit;		// Maximum number of msgs to allow
#ifdef	OO_MHIGH
		unsigned		Mqueue_high;		// Message queue high-water mark.
#endif
#ifdef OO_TIME
		unsigned		TT_runlong;			// Longest dispatch seen
		unsigned		TT_seclong;			// Longest second seen
		unsigned		TT_time[OO_TIME];	// Task timer for last 8 secs
#endif
	};
	
	/*
	 * Operating system entry points
	 */
	unsigned OS_call(unsigned service, ...);
	void OS_swap(void);
	void OS_panic(char *format, ...);
	
	// Direct entry points (Call from privleged mode ONLY!)
	struct MESSAGE *OS_msg_small(void);
	struct MESSAGE *OS_msg_medium(void);
	struct MESSAGE *OS_msg_large(void);
	void OS_msg_release(struct MESSAGE *m);
	unsigned OS_msg_send(unsigned Tid, struct MESSAGE *m, unsigned char info);
	void OS_proc_signal(unsigned signal, unsigned value);
	void OS_proc_ready(unsigned id, unsigned value);
	void OS_irq_set(unsigned mask);
	void OS_irq_clear(unsigned mask);
	void OS_console_tx(struct MESSAGE *m);
	unsigned OS_ethernet_tx(unsigned char *p, unsigned size);
	void OS_bls(unsigned mask);
	void OS_blc(unsigned mask);
	void OS_tls(unsigned mask);
	void OS_tlc(unsigned mask);
	
	#ifndef OS_BUILD		// Do not include for native build
	
	/*
	 * Operating system global variables
	 */
	extern const struct TCB
		*OS_active_tcb;				// Points to active process TCB
	extern volatile const unsigned
		OS_timer_tick,				// Ticks for every timer interrupt
		OS_debug_mask,				// Debug output mask
		OS_eirq_mask,				// External interrupt mask
		OS_system_config,			// System configuration
		OS_free_msg_small,			// Indicates # of available small  msg blocks
		OS_free_msg_medium,			// Indicates # of available medium msg blocks
		OS_free_msg_large,			// Indicates # of available large  msg blocks
		OS_blatch,					// Bottom board latch mirror
		OS_tlatch;					// Top board latch mirror
	extern volatile const unsigned char
		OS_hw_type,					// Hardware type
		OS_irq_active,				// Interrupt is active
		OS_console_owner,			// Owner of system console
		OS_at_owner;				// Owner of AT commands
	extern volatile unsigned char
		OS_console_echo;			// Enable echo at system console
	
	#endif

#endif
