#define	ESM	0x10		// Smart media enable
#define	EBF	0x20		// Bulk flash enable
#define	CLE	0x40		// Command latch enable
#define	ALE	0x80		// Address latch enable

unsigned char
	Bbusy,				// Device is busy
	Bdevice;			// Selected bulk device
	
/*
 * Parity table for ECC calculation
 * Bits 0-6 = Londitudinal parity, Bit 7 = Latitudinal parity
 */
static const unsigned char ecc_parity_table[] = {
0x00, 0x95, 0x96, 0x03, 0x99, 0x0C, 0x0F, 0x9A, 0x9A, 0x0F, 0x0C, 0x99,
0x03, 0x96, 0x95, 0x00, 0xA5, 0x30, 0x33, 0xA6, 0x3C, 0xA9, 0xAA, 0x3F,
0x3F, 0xAA, 0xA9, 0x3C, 0xA6, 0x33, 0x30, 0xA5, 0xA6, 0x33, 0x30, 0xA5,
0x3F, 0xAA, 0xA9, 0x3C, 0x3C, 0xA9, 0xAA, 0x3F, 0xA5, 0x30, 0x33, 0xA6,
0x03, 0x96, 0x95, 0x00, 0x9A, 0x0F, 0x0C, 0x99, 0x99, 0x0C, 0x0F, 0x9A,
0x00, 0x95, 0x96, 0x03, 0xA9, 0x3C, 0x3F, 0xAA, 0x30, 0xA5, 0xA6, 0x33,
0x33, 0xA6, 0xA5, 0x30, 0xAA, 0x3F, 0x3C, 0xA9, 0x0C, 0x99, 0x9A, 0x0F,
0x95, 0x00, 0x03, 0x96, 0x96, 0x03, 0x00, 0x95, 0x0F, 0x9A, 0x99, 0x0C,
0x0F, 0x9A, 0x99, 0x0C, 0x96, 0x03, 0x00, 0x95, 0x95, 0x00, 0x03, 0x96,
0x0C, 0x99, 0x9A, 0x0F, 0xAA, 0x3F, 0x3C, 0xA9, 0x33, 0xA6, 0xA5, 0x30,
0x30, 0xA5, 0xA6, 0x33, 0xA9, 0x3C, 0x3F, 0xAA, 0xAA, 0x3F, 0x3C, 0xA9,
0x33, 0xA6, 0xA5, 0x30, 0x30, 0xA5, 0xA6, 0x33, 0xA9, 0x3C, 0x3F, 0xAA,
0x0F, 0x9A, 0x99, 0x0C, 0x96, 0x03, 0x00, 0x95, 0x95, 0x00, 0x03, 0x96,
0x0C, 0x99, 0x9A, 0x0F, 0x0C, 0x99, 0x9A, 0x0F, 0x95, 0x00, 0x03, 0x96,
0x96, 0x03, 0x00, 0x95, 0x0F, 0x9A, 0x99, 0x0C, 0xA9, 0x3C, 0x3F, 0xAA,
0x30, 0xA5, 0xA6, 0x33, 0x33, 0xA6, 0xA5, 0x30, 0xAA, 0x3F, 0x3C, 0xA9,
0x03, 0x96, 0x95, 0x00, 0x9A, 0x0F, 0x0C, 0x99, 0x99, 0x0C, 0x0F, 0x9A,
0x00, 0x95, 0x96, 0x03, 0xA6, 0x33, 0x30, 0xA5, 0x3F, 0xAA, 0xA9, 0x3C,
0x3C, 0xA9, 0xAA, 0x3F, 0xA5, 0x30, 0x33, 0xA6, 0xA5, 0x30, 0x33, 0xA6,
0x3C, 0xA9, 0xAA, 0x3F, 0x3F, 0xAA, 0xA9, 0x3C, 0xA6, 0x33, 0x30, 0xA5,
0x00, 0x95, 0x96, 0x03, 0x99, 0x0C, 0x0F, 0x9A, 0x9A, 0x0F, 0x0C, 0x99,
0x03, 0x96, 0x95, 0x00 };

/*
 * Calculate 22 bit ECC code for 256 byte data block
 */
void generate_ecc(unsigned char *data, unsigned char ecc[])
{
	unsigned char i, t, x1, x2, x3;

	// Generate londitudinal parity
	ecc[0] = ecc[1] = ecc[2] = x1 = x2 = x3 = t = 0;
	do {
		x1 ^= (i = ecc_parity_table[*data++]) & 0x7F;
		if(i & 0x80) {		// Odd Latitudinal?
			x2 ^= t;
			x3 ^= ~t; } }
	while(++t);
	ecc[2] = (~x1 << 2) | 3;

	// Generate latitudinal parity
	i = x1 = 0x80;
	do {
		if(x2 & i)
			t |= x1;
		x1 >>= 1;
		if(x3 & i)
			t |= x1;
		x1 >>= 1; }
	while((i >>= 1) != 0x08);
	ecc[0] = ~t;

	t = 0;
	x1 = 0x80;
	do {
		if(x2 & i)
			t |= x1;
		x1 >>= 1;
		if(x3 & i)
			t |= x1;
		x1 >>= 1; }
	while(i >>= 1);
	ecc[1] = ~t;
}

/*
 * Detect and correct a 1 bit error for 256 byte data block
 * Returns:	0  = No errors were found
 *			1  = Single bit data error was corrected
 *			2  = Single bit ecc  error was ignored (data is correct)
 *			-1 = Multiple errors - could not be recovered
 *	data = 256 byte data block as read from device
 *	decc = Data ECC		(read from device)
 *	cecc = Computed ECC	(calculated from data content)
 */
int validate_data(unsigned char data[], unsigned char decc[], unsigned char cecc[])
{
	unsigned char a, b, c, i, x1, x2, x3;

	x1 = decc[0] ^ cecc[0];		// Differences in byte 0
	x2 = decc[1] ^ cecc[1];		// Differences in byte 1
	x3 = decc[2] ^ cecc[2];		// Differences in byte 2

	if(!(x1 | x2 | x3))			// No differences
		return 0;				// = No errors

	// Check for single bit data error and correct
	i = 0;
	if(	((((x1 >> 1) ^ x1) & 0x55) == 0x55)
	&&	((((x2 >> 1) ^ x2) & 0x55) == 0x55)
	&&	((((x3 >> 1) ^ x3) & 0x54) == 0x54) ) {
		// First compute offset of affected byte
		a = c = 0x80;
		do {
			if(x1 & c)
				i |= a;
			c >>= 2; }
		while((a >>= 1) != 0x08);
		c = 0x80;
		do {
			if(x2 & c)
				i |= a;
			c >>= 2; }
		while(a >>= 1);
		// Then compute bit within the byte
		b = 0x04;
		c = 0x80;
		do {
			if(x3 & c)
				a |= b;
			c >>= 2; }
		while(b >>= 1);
		data[i] ^= 1 << a;		// Toggle bad bit to correct
		return 1; }

	// Either single-bit ECC error or uncorrectable (multiple bit errors)
	do {
		i += x1 & 1; }
	while(x1 >>= 1);
	do {
		i += x2 & 1; }
	while(x2 >>= 1);
	do {
		i += x3 & 1; }
	while(x3 >>= 1);
	return (i == 1) ? 2 : -1;
}

/*
 * --- Smart Media Access functions ---
 */

// Send command to bulk device
static void smcmd(unsigned char c)
{
	PADR = (PADR | CLE) & ~Bdevice;
	SM_DATA = c;
	wait(5);
	PADR &= ~CLE;
}

static void smoff(void)
{
	PADR = ESM | EBF;
}

// Send four byte address to bulk device
static void smaddr4(unsigned a)
{
// rawprintf("\na3: %u %08x", a, a);
	PADR = (PADR | ALE) & ~Bdevice;
	SM_DATA = a;
	SM_DATA = a >> 8;
	SM_DATA = a >> 16;
	SM_DATA = a >> 24;
	PADR &= ~ALE;
}

// Send three byte address to bulk device
static void smaddr3(unsigned a)
{
// rawprintf("\na2: %u %08x", a, a);
	PADR = (PADR | ALE) & ~Bdevice;
	SM_DATA = a;
	SM_DATA = a >> 8;
	SM_DATA = a >> 16;
	PADR &= ~ALE;
}

// Send one byte address to bulk device
static void smaddr1(unsigned a)
{
	PADR = (PADR | ALE) & ~Bdevice;
	SM_DATA = a;
	PADR &= ~ALE;
}

void select_file(unsigned char device)
{
	if(Bbusy) {	// Device is busy
		smcmd(0x70);		// Issue status
		if(!(SM_DATA & 0x40)) {	// Device is busy
// rawprintf("Busy");
			smoff();
			send_result(1);
			OSexit(); }
		Bbusy = 0; }
//	smoff();
	Bdevice = device ? ESM : EBF;
// rawprintf("\nBdevice=%02x", Bdevice);
}

static unsigned read_file(unsigned char device, unsigned page, unsigned char *data)
{
	unsigned i;
	unsigned char extra[16], ecc[3];
// rawprintf("\rRead: %u %u %08x", device, page, data);
	select_file(device);
	smcmd(0x00);		// Read from block=0
	smaddr4(page << 8);
	wait(200);
	for(i=0; i < 512; ++i)
		data[i] = SM_DATA;
	for(i=0; i < 16; ++i)
		extra[i] = SM_DATA;
	smoff();
	generate_ecc(data, ecc);
// rawprintf("\nECC0: %02x%02x%02x %02x%02x%02x", ecc[0], ecc[1], ecc[2], extra[0], extra[1], extra[2]);
	i = validate_data(data, extra, ecc);
// rawprintf("\nVal: %u", i);
	if(i == -1)
		return -1;
	generate_ecc(data+256, ecc);
// rawprintf("\nECC1: %02x%02x%02x %02x%02x%02x", ecc[0], ecc[1], ecc[2], extra[3], extra[4], extra[5]);
	i = validate_data(data+256, extra+3, ecc);
// rawprintf("\nVal: %u", i);
	if(i == -1)
		return -1;
	return 0;
}

static unsigned write_file(unsigned char device, unsigned page, unsigned char *data)
{
	unsigned i;
	unsigned char extra[16];
// rawprintf("\rWrite: %u %u %08x", device, page, data);
	select_file(device);
	smcmd(0x80);
	smaddr4(page << 8);
	memset(extra, 0xFF, sizeof(extra));
	generate_ecc(data, extra);
	generate_ecc(data+256, extra+3);
	for(i=0; i < 512; ++i)
		SM_DATA = data[i];
	for(i=0; i < 16; ++i)
		SM_DATA = extra[i];
	smcmd(0x10);
	Bbusy = 1;		// Indicate programming
	smoff();
	return 0;
}

static unsigned erase_file(unsigned char device, unsigned block)
{
// rawprintf("\nErase: %u %u", device, block);
	select_file(device);
	smcmd(0x60);
	smaddr3(block << 5);
	smcmd(0xD0);
	Bbusy = 2;
	smoff();
	return 0;
}

// Initialize file system
static void init_file(void)
{
	unsigned char t;
	rawprintf("\nInitializing storage device(s),");
//	PADDR = ALE | CLE | EBF | ESM;	// Set outputs (now done by OS)
	PADR = ESM | EBF;				// Both selects high
//	blc(22);						// Reset devices
//	wait(0x4000);
	OS_bls(1<<22);					// Un-reset device

	Bdevice = EBF;					// Default is bulk flash
	wait(0x8000);
	smcmd(0xFF);					// Issue RESET command
	wait(10);
	smcmd(0x90);
	smaddr1(0x00);
	t = SM_DATA;
	rawprintf(" FlashType: %02x-%02x, ", t, SM_DATA);

	if(SM_INS & 0x10)
		rawputs("No Card inserted.");
	else {
		smoff();
		Bdevice = ESM;					// Switch to SmartMedia card
		wait(0x8000);
		smcmd(0xFF);					// Issue RESET command
		smcmd(0x90);
		smaddr1(0x00);
		t = SM_DATA;
		OS_system_config |= CONFIG_SMCARD;
		rawprintf("CardType: %02x-%02x", t, SM_DATA); }
	Bdevice = EBF;
	smoff();
}
