#include "8390.h"				// NIC definitions
//#define	EEDEBUG				// Debug EEPROM simulation
#define	ALIGN_PACKET			// Expand type to 32 bit word
#define	ETX_QUEUE	4			// Size of ethernet TX queue

// Define minimum message requirements for ethernet acceptance
// of a packet
#define	MMsmall			15
#define	MMmedium		10
#define	MMlarge			5

/* --- Network card Interfacing functions --- */

// Ethernet Hardware parameters
#define	PNP_ADDRESS		0x1A0279	// PnP Addresss register
#define	PNP_DATA		0x1A0A79	// PnP Write data register
#define	NE_BASE			0x1A0280	// Ethernet Controller base
#define	NE_RESET		0x1F		// Offset to RESET port
#define	NE_DATA			0x11		// Offset to DATA port
#define	IO				0x280		// Ethernet I/O address
#define	IRQ				11			// Ethernet interrupt
#define	READ_DATA		0x393		// Ethernet read data address
#define	CSN				0x23		// Ethernet serial number
#define	SM_TSTART_PG	0x40		// First page of TX buffer
#define	SM_RSTART_PG	0x46		// Starting page for RX ring
#define	SM_RSTOP_PG		0x5F		// Last page (+1) for RX ring

// Network statistics monitor index values
#define	NS_tx		0				// TX packets
#define	NS_tfail	1				// TX failed
#define	NS_tqueue	2				// TX queued
#define	NS_tdrop	3				// TX dropped
#define	NS_rx		4				// RX packets
#define	NS_norx		5				// No receiver
#define	NS_rfail	6				// Receiver failed
#define	NS_rovrn	7				// RX overrun
#define	NS_rfcrc	8				// CRC error
#define	NS_rffae	9				// Frame alignment error
#define	NS_rffo		10				// FiFo overrun
#define	NS_rfmpa	11				// Missed packet
#define	NS_rfcol	12				// Collision
#define	NS_rinc		13				// RX incomplete
#define	NS_nomsg	14				// No message available
#define	NS_toobig	15				// Message too largs
#define	NS_STATS	16				// Number of stats

static unsigned
	next_packet,					// NIC RX queue DMA pointer
	etxqr,							// TX queue read pointer
	etxqw,							// TX queue write pointer
	etxqs[ETX_QUEUE],				// TX queue size
	net_stats[NS_STATS];			// Statistics monitor values

static unsigned char
	etx_queue[ETX_QUEUE][MSG_LARGE];// TX queue

#define	EEC	0x01					// EEPROM clock bit in PORT-A
#define	EES	0x02					// EEPROM select bit in PORT-A
#define	EEI	0x08					// EEPROM data=in bit in PORT-A
#define	EEO	0x04					// EEPROM data-out bit in PORT-A
#define	EHC	0x10					// EEPROM clock in high-nibble
#define	EHS	0x20					// EEPROM select in high-nibble
#define	EHI	0x80					// EEPROM data-in in high-nibble

/*
 * Handle a single EEPROM clock bit.
 * Write the desired output, then wait for and record the
 * states at the low-to-high and high-to-low transitions.
 * Returns: Ir Or Sr Cr If Of Sf Cf
 */
unsigned eebit(unsigned bit)
{
	unsigned t;
	unsigned c, d;

	WDOG = 0;

	PADR = (bit) ? (0x30|EEO) : 0x30;

	for(t=0; t < 0x40; ++t) {		// Wait for clock high & record
		c = PADR & (EEC|EES|EEI);
		if(c & EEC)
			goto clock1; }
	return 0;
clock1:
	for(t=0; t < 0x40; ++t) {		// Wait for clock low & record
		d = PADR & (EEC|EES|EEI);
		if(!(d & EEC))
			return (c << 4) | d; }
	return 0;
}

#ifdef EEDEBUG
	#define	EERET	goto dumpee
#else
	#define	EERET	return
#endif

/*
 * Simulate the 9316 EEPROM by watching for the clock bits
 * and outputting data as required.
 */
void eesimulate(void)
{
	unsigned c, i, v;
	int vc;
#ifdef EEDEBUG
	unsigned t;
	static unsigned char eedata[512];
	t = 0;
#endif

eetop:
	while(c = eebit(0)) {
		if(c == (EHC|EHS  | EES|EEI))	// C:1->0 S:1 D:0->1
			goto eecmd; }
	EERET;
eecmd:
	for(i=c=v=0; i < 8;) switch(eebit(0)) {
		case 0 : EERET;			// Timeout - terminate
		case EHC		 | EES : 		// C:1->0 S:0->1	- New command
			goto eetop;
		case EHC|EHS|EHI | EES|EEI :	// C:1->0 S:1 I:1
		case EHC|EHS 	 | EES|EEI :	// C:1->0 S:1 I:0->1
			c = (c << 1) | 1;
			++i;
			continue;
		case EHC|EHS|EHI | EES :		// C:1->0 S:1 I:1->0
		case EHC|EHS	 | EES :		// C:1->0 S:1 I:0
			c <<= 1;
			++i; }
#ifdef EEDEBUG
	eedata[t++] = c;
#endif
// Returns: Ir Or Sr Cr If Of Sf Cf
	switch(c & 0x3F) {
	case 0x01 :	v = 0x80000040; }	// Reg-1 Stop here + set FD bit
//	case 0x01 :	v = 0x80000000; }	// Reg-1 Stop here + set FD bit
	i = v;
	vc = 17;
	for(;;) {
		if(vc < 17) {
			c = (v & 0x8000) ? 1 : 0;
			v <<= 1; }
		else
			c = 0;
		--vc;
		switch(eebit(c)) {
		case 0 : EERET;			// Timeout
		case EHC		 | EES : 		// C:1->0 S:0->1	- New command
			goto eetop;
#ifdef EEDEBUG
		case EHC|EHS|EHI | EES|EEI :	// C:1->0 S:1 I:1
		case EHC|EHS 	 | EES|EEI :	// C:1->0 S:1 I:0->1
		case EHC|EHS|EHI | EES :		// C:1->0 S:1 I:1->0
		case EHC|EHS	 | EES :		// C:1->0 S:1 I:0
			eedata[t++] = c;
			continue;
#endif
		case EHC|EHS :					// C:1->0 S:1->0	- End of command
			if(i & 0x80000000)
				EERET; } }

#ifdef EEDEBUG
// Dump EEPROM data output
dumpee:
	rawprintf("\nEEDUMP:");
	for(i=c=0; i < t; ++i) {
		if(eedata[i] & 0x80) {
			rawprintf("\nC:%02x", eedata[i]);
			c = 0;
			continue; }
		rawprintf(" %x", eedata[i]);
		++c; }
#endif
}

/*
 * Write ISA PnP configuration register
 */
static void writePNP(unsigned char address, unsigned char data)
{
	CS8(PNP_ADDRESS) = address;
	CS8(PNP_DATA) = data;
}

#define writeNE(offset, data) CS8(NE_BASE+offset) = data
#define	readNE(offset) (CS8(NE_BASE+offset))

/*
 * Initialize NIC device
 */
static unsigned char reset_nic(void)
{
	unsigned i;
	unsigned char c;

	// Initialize the NE2000 chip
	next_packet = SM_RSTART_PG+1;				// Init DMA pointer
	c = readNE(NE_RESET);						// Issue read for reset
	wait(800);									// Give time to reset
	writeNE(NE_RESET, c);						// Write to complete cycle

	// Page 0 config
	writeNE(EN_CCMD, ENC_NODMA+ENC_STOP+ENC_PAGE0);
	writeNE(EN0_DCFG, ENDCFG_FT10+ENDCFG_BMS);	// Data config
	writeNE(EN0_RXCR, ENRXCR_MON);				// RX control
	writeNE(EN0_TXCR, ENTXCR_LOOP);				// TX control
	writeNE(EN0_STARTPG, SM_RSTART_PG);			// Set lower limit
	writeNE(EN0_BOUNDARY, SM_RSTART_PG);		// Set queue tail
	writeNE(EN0_STOPPG, SM_RSTOP_PG);			// Set upper limit
	writeNE(EN0_ISR, 0xFF);						// Clear pending IRQ
	writeNE(EN0_IMR, 0x00);						// Disable all IRQ's

	// Page 1 config
	writeNE(EN_CCMD, ENC_NODMA+ENC_STOP+ENC_PAGE1);
	writeNE(EN1_CURPAG, SM_RSTART_PG+1);		// Init queue head
	for(i=0; i < 6; ++i)
		writeNE(EN1_PHYS+i, nic_address[i]);	// Set physical address
	for(i=0; i < 8; ++i)
		writeNE(EN1_MULT+i, 0);					// Clear multicast
//		writeNE(EN1_MULT+i, 0xFF);				// All multicast
	// Enable our multicast mask bit
//	writeNE(EN1_MULT+0, 0xFF);
//	writeNE(EN1_MULT+1, 0xFF);
//	writeNE(EN1_MULT+2, 0xFF);
//	writeNE(EN1_MULT+3, 0xFF);
	writeNE(EN1_MULT+4, 0x04);
//	writeNE(EN1_MULT+5, 0xFF);
//	writeNE(EN1_MULT+6, 0xFF);
//	writeNE(EN1_MULT+7, 0xFF);

	// Read back our NIC address & verify
	for(c=i=0; i < 6; ++i) {
		if(readNE(EN1_PHYS+i) != nic_address[i])
			c = 255; }

	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
	writeNE(EN0_TXCR, 0);						// Normal TX
	writeNE(EN0_RXCR, ENRXCR_BCST|ENRXCR_MULTI);// Turn on RX
//	writeNE(EN0_IMR, 0x15);						// OVL, RXE, PRX
	writeNE(EN0_IMR, 0x17);						// OVL, RXE, PTX, PRX

#ifdef EEDEBUG
	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE3);
	rawprintf("\nCF1=%08b ", readNE(EN3_CFG1));
	rawprintf(" CF2=%08b", readNE(EN3_CFG2));
	rawprintf(" CF3=%08b", readNE(EN3_CFG3));
	rawprintf(" CF4=%08b", readNE(EN3_CFG4));
	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
#endif

	return c;
}

/*
 * Initialize the network interface
 */
static unsigned char init_network(void)
{
	unsigned i;

	unsigned RTkey[] =
	{		/* Always works (Realtek device only) */
		0xDA, 0x6D, 0x36, 0x1B, 0x8D, 0x46, 0x23, 0x91,
		0x48, 0xA4, 0xD2, 0x69, 0x34, 0x9A, 0x4D, 0x26,
		0x13, 0x89, 0x44, 0xA2, 0x51, 0x28, 0x94, 0xCA,
		0x65, 0x32, 0x19, 0x0C, 0x86, 0x43, 0xA1, 0x50
	};


	// Reset card
	switch(OS_hw_type) {
	case 0 :		// 4x8
		PADDR = 0xF1|EEO;			// Set PORT-A direction
		PADR = 0x30|EEO;			// Set PORT-A data
		OS_tlc(1<<13);				// Set for  8-BIT access mode
		OS_tls(1<<14);				// Reset NIC
		wait(10000);				// Wait for reset
		OS_tlc(1<<14);				// Release reset
		break;
	case 1 :		// 2x8
	case 2 :		// 2xN
		PADDR = 0xF0|EEO;			// Set PORT-A direction
		PADR = 0x30|EEO;			// Set PORT-A data
		OS_blc(1<<14);				// Set for 8-bit access mode
		OS_bls(1<<15);				// Reset NIC
		wait(10000);				// Wait for reset
		OS_blc(1<<15); }				// Release reset

	if(OS_hw_type == 2)
		eesimulate();		// Load EEPROM data

	// Reset card & send PnP key
	writePNP(0x02, 0x07);	//--- reset pnp configuration
	wait(500);
	for(i=0; i < sizeof(RTkey); ++i) {
		CS8(PNP_ADDRESS) = RTkey[i];
		wait(1); }

	// Select the card
	wait(500); writePNP(0x03, 0x00);			// Enter Isolation state
	// Simce we have only one device, we don't actually have to perform the
	// P&P isolation step - we already have only one device responding.
	wait(500); writePNP(0x00, READ_DATA >> 2);	// Set PNP read data address
	wait(500); writePNP(0x06, CSN);				// Assign CSN to select

	// Write I/O address and IRQ
	wait(500); writePNP(0x60, IO >> 8);			// High byte of I/O
	wait(500); writePNP(0x61, IO);				// Low byte of I/O
	wait(500); writePNP(0x70, IRQ);				// Interrupt

	wait(500); writePNP(0x30, 0x01);			// Activate card

	return reset_nic();
}

void OS_ethernet_multi(unsigned mh, unsigned ml)
{
	writeNE(EN_CCMD, ENC_NODMA+ENC_STOP+ENC_PAGE1);
	writeNE(EN1_MULT+0, mh >> 24;);
	writeNE(EN1_MULT+1, mh >> 16);
	writeNE(EN1_MULT+2, mh >> 8);
	writeNE(EN1_MULT+3, mh);
	writeNE(EN1_MULT+4, ml >> 24);
	writeNE(EN1_MULT+5, ml >> 16);
	writeNE(EN1_MULT+6, ml >> 8);
	writeNE(EN1_MULT+7, ml);
	writeNE(EN_CCMD, ENC_NODMA+ENC_STOP+ENC_PAGE0);
}

/*
 * Physically transmit packet over Ethernet
 */
static void ethernet_tx(unsigned char *p, unsigned size)
{
	unsigned i;

	writeNE(EN0_ISR, ENISR_RDC);
#ifdef ALIGN_PACKET
	i = size-2;
	writeNE(EN0_TCNTLO, i);
	writeNE(EN0_TCNTHI, i >> 8);
	writeNE(EN0_RCNTLO, i);
	writeNE(EN0_RCNTHI, i >> 8);
#else
	writeNE(EN0_TCNTLO, size);
	writeNE(EN0_TCNTHI, size >> 8);
	writeNE(EN0_RCNTLO, size);
	writeNE(EN0_RCNTHI, size >> 8);
#endif

	writeNE(EN0_RSARLO, 0);
	writeNE(EN0_RSARHI, SM_TSTART_PG);

	writeNE(EN_CCMD, ENC_RWRITE+ENC_START);

#ifdef ALIGN_PACKET
	for(i=0; i < 14; ++i)	// Write packet header
		writeNE(NE_DATA, *p++);
	p += 2;					// Skip high half of 32 bit type
	size -= 16;				// Adjust size by total header (incl skipped)
#endif
	do {
		writeNE(NE_DATA, *p++);
	} while(--size);

	size = 65536;
	do {
		if(readNE(EN0_ISR) & ENISR_RDC) {
			++net_stats[NS_tx];
			goto nofail; }
	} while(--size);
//	Dprintf(0, "Send failed");
	++net_stats[NS_tfail];
nofail:
	writeNE(EN0_TPSR, SM_TSTART_PG);					// Set start page
	writeNE(EN_CCMD, ENC_NODMA+ENC_TRANS+ENC_START);	// Start TX
//	OS_msg_release(m);
}

/*
 * Request to transmit a packet over Ethernet
 * TX immediately if idle, else queue packet.
 */
unsigned OS_ethernet_tx(unsigned char *p, unsigned size)
{
	unsigned nw, i;
	OStest("ethernet_tx");

	nw = (etxqw + 1) & (ETX_QUEUE-1);			// Write slot

	i = 0;										// Initial write position
recmd:
	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);

	for(;;) {
		if(!(readNE(EN_CCMD) & ENC_TRANS)) {	// TX is ready
			if(etxqr == etxqw) {				// Queue is empty
				ethernet_tx(p, size);
				return 1; }
			// Send next packet from queue
			ethernet_tx(etx_queue[etxqr], etxqs[etxqr]);
			etxqr = (etxqr + 1) & (ETX_QUEUE-1);
			goto recmd; }

		if(nw == etxqr) {						// Queue is full
			++net_stats[NS_tdrop];
			return 0; }

		// Copy another byte from our packet & exit if we copy it all
		etx_queue[etxqw][i] = p[i];
		if(++i >= size) {						// Packet has been queued
			++net_stats[NS_tqueue];				// Increment queued count
			etxqs[etxqw] = size;				// Record size
			etxqw = nw;							// Set new write pointer
			return 2; } }
}

/*
 * Test for and receive a packet (if available)
 * Returns:
 *	0x000 - Packet successfully received (in RpackeT)
 *	0x100 - No packet available
 *  0x2ss - Overrun or receive error has occured (ss = NIC status)
 *  0x3ss - Receive operation failed (ss = NIC status)
 *  0x400 - Received packet too large
 *  0x500 - Received packet was wrong type
 */
void IRQ_ethernet(void)
{
	unsigned char rcv_header[4];	// Received packet header
	unsigned char status;			// Status byte from packet header
	unsigned char next;				// Next page from packet header
	unsigned char save_curr;		// Copy of current position
	unsigned size;					// Size of packet received
	int i;							// General integer
	unsigned char *p;				// Data write pointer
	struct MESSAGE *m;				// Message

poll_net:
	for(;;) {
		status = readNE(EN0_ISR);
		if(status & (ENISR_OVER|ENISR_RX_ERR)) {		// Receive error
			i = readNE(EN0_RSR);
			if(status & ENISR_RX_ERR)	++net_stats[NS_rfail];
			if(status & ENISR_OVER)		++net_stats[NS_rovrn];
			if(i & ENRSR_CRC)			++net_stats[NS_rfcrc];
			if(i & ENRSR_FAE)			++net_stats[NS_rffae];
			if(i & ENRSR_FO)			++net_stats[NS_rffo];
			if(i & ENRSR_MPA)			++net_stats[NS_rfmpa];
			if(i & ENRSR_DEF)			++net_stats[NS_rfcol];
			reset_nic();
#ifdef OO_TIME
			OS_irq_exit(XIRQ_network);
#endif
			break; }

		writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE1);
		save_curr = readNE(EN1_CURPAG);
		writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
		writeNE(EN0_ISR, ENISR_RX+ENISR_TX+ENISR_RX_ERR+ENISR_OVER);	// Clear int/status

		if(save_curr == next_packet) {					// No packet available
#ifdef OO_TIME
			OS_irq_exit(XIRQ_network);
#endif
			break; }

		writeNE(EN0_RCNTLO, sizeof(rcv_header));		// Count low
		writeNE(EN0_RCNTHI, sizeof(rcv_header) >> 8);	// Count high
		i = sizeof(rcv_header);
		writeNE(EN0_RSARLO, 0);							// DMA address
		writeNE(EN0_RSARHI, next_packet);				// DMA address high
		p = rcv_header;
		writeNE(EN_CCMD, ENC_RREAD+ENC_START);			// Start read operation

		do {	// Read NIC header
			*p++ = readNE(NE_DATA);
		} while(--i);

		status	= rcv_header[EN_RBUF_STAT];			// Extract status
		next	= rcv_header[EN_RBUF_NXT_PG];		// Extract next page
		size	= rcv_header[EN_RBUF_SIZE_HI] << 8;	// Extract size high
		size   |= rcv_header[EN_RBUF_SIZE_LO];		// Include size low
		size   -= EN_RBUF_NHDR;						// Remove header

		if(!(status & ENRSR_RXOK)) {				// Packet RX failed
			++net_stats[NS_rinc];
#ifdef OO_TIME
			OS_irq_exit(XIRQ_network);
#endif
			break; }

#ifdef ALIGN_PACKET
		if(size <= (MSG_SMALL-2)) {
			if(OS_free_msg_small < MMsmall) { ++net_stats[NS_nomsg]; goto drop_packet; }
			m = OS_msg_small(); }
		else if(size <= (MSG_MEDIUM-2)) {
			if(OS_free_msg_medium < MMmedium) { ++net_stats[NS_nomsg]; goto drop_packet; }
			m = OS_msg_medium(); }
		else if(size <= (MSG_LARGE-2)) {
			if(OS_free_msg_large < MMlarge) { ++net_stats[NS_nomsg]; goto drop_packet; }
			m = OS_msg_large(); }
#else
		if(size <= MSG_SMALL) {
			if(OS_free_msg_small < MMsmall) goto drop_packet;
			m = OS_msg_small(); }
		else if(size <= MSG_MEDIUM) {
			if(OS_free_msg_medium < MMmedium) goto drop_packet;
			m = OS_msg_medium(); }
		else if(size <= MSG_LARGE) {
			if(OS_free_msg_large < MMlarge) goto drop_packet;
			m = OS_msg_large(); }
#endif
		else {		// Message is too large to handle - ignore it
			++net_stats[NS_toobig];
drop_packet:
			writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
			writeNE(EN0_BOUNDARY, next_packet);
			next_packet = next;
			goto poll_net; }

		writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
		writeNE(EN0_RCNTLO, size);					// Count low
		writeNE(EN0_RCNTHI, size >> 8);				// Count high
		writeNE(EN0_RSARLO, EN_RBUF_NHDR);			// DMA address low
		writeNE(EN0_RSARHI, next_packet);			// DMA address high
		writeNE(EN_CCMD, ENC_RREAD+ENC_START);		// Start READ operation

#ifdef ALIGN_PACKET
		m->Info2 = size+2;
		size -= 14;
		p = m->Data;								// Point to data
		for(i=0; i < 14; ++i)		// Read header
			*p++ = readNE(NE_DATA);
		*p++ = 0;					// Pad type
		*p++ = 0;					// Pad type
#else
		m->Info2 = size;
		p = m->Data;								// Point to data
#endif
		do {
			*p++ = readNE(NE_DATA);
		} while(--size);

		writeNE(EN0_BOUNDARY, next_packet);
		next_packet = next;

		size = (m->Data[12] << 8) | m->Data[13];	// Get packet type

		for(i=0; i < packet_top; ++i) {
			if(packet_type[i] == size) {
// rawprintf("\nSend %08x to %u", m, packet_pid[i]);
				if(OS_msg_send(packet_pid[i], m, TID_ENET) > 1)
					break;
				++net_stats[NS_rx];
				goto poll_net; } }

		++net_stats[NS_norx];
#ifdef	OO_DEV
		OS_last_rel = 5;
#endif
		OS_msg_release(m); }

	// Check for transmit packet pending
	if(etxqr != etxqw) {							// Packet is queued
		writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
		if(!(readNE(EN_CCMD) & ENC_TRANS)) {		// TX is ready
			ethernet_tx(etx_queue[etxqr], etxqs[etxqr]);
			etxqr = (etxqr + 1) & (ETX_QUEUE-1); } }
}

/*
 * Register for reception of ethernet packet type
 */
static void OS_ethernet_register(unsigned type)
{
	unsigned i;

	for(i=0; i < packet_top; ++i) {
		if(packet_type[i] == type) {
			if(packet_pid[i] == OS_active_tcb->Id)
				return;
			OS_panic("Packet type %04x allocated by %02x and %02x",
				type, packet_pid[i], OS_active_tcb->Id); } }

	if(packet_top >= MAX_PACKET)
		OS_panic("Packet type table full");

	packet_pid[packet_top] = OS_active_tcb->Id;
	packet_type[packet_top++] = type;
}

/*
 * Release allocation of packet type
 */
static void OS_ethernet_release(unsigned type)
{
	unsigned i;
	for(i=0; i < packet_top; ++i) {
		if((packet_type[i] == type) && (packet_pid[i] == OS_active_tcb->Id)) {
			--packet_top;
			while(i < packet_top) {
				packet_type[i] = packet_type[i+1];
				packet_pid[i] = packet_pid[i+1];
				++i; }
			return; } }
}

/*
 * Display ethernet information/statistics
 */
static void netinfo(void (*output)(const char *,...))
{
	output("\r\nTXok  : %-11uTXque : %-11uTXdrop: %-11uTXfail: %u",
		net_stats[NS_tx], net_stats[NS_tqueue], net_stats[NS_tdrop], net_stats[NS_tfail]);
//	output("\r\nRXok  : %-11uRXfail: %-11uRX2big: %-11uRXnmsg: %u",
//		net_stats[NS_rx], net_stats[NS_rfail], net_stats[NS_toobig], net_stats[NS_nomsg]);
//	output("\r\nRXnorx: %u", net_stats[NS_norx]);
	output("\r\nRXok  : %-11uRXfail: %-11uRXovrn: %-11uRXinc : %u",
		net_stats[NS_rx], net_stats[NS_rfail], net_stats[NS_rovrn], net_stats[NS_rinc]);
	output("\r\nRXfcrc: %-11uRXfae : %-11uRXfifo: %-11uRXfmpa: %u",
		net_stats[NS_rfcrc], net_stats[NS_rffae], net_stats[NS_rffo], net_stats[NS_rfmpa]);
	output("\r\nRXcoll: %-11uRX2big: %-11uRXnmsg: %-11uRXnorx: %u",
		net_stats[NS_rfcol], net_stats[NS_toobig], net_stats[NS_nomsg], net_stats[NS_norx]);

	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE3);
	output("\r\nNIC is operating in %s duplex.",
		(readNE(EN3_CFG3) & 0x40) ? "Full" : "Half");
	writeNE(EN_CCMD, ENC_NODMA+ENC_START+ENC_PAGE0);
}
