/*
 * ArmOS - Panic debugger
 *
 * Dave Dunfield
 * First edition: May 5, 2003
 * Last   update: Feb 26, 2006
 *
 * This code is #included in OS.C to enable it to have access to
 * operating system statics, however it is presented in a separate
 * source file to keep it logically distinct.
 *
 * The panic debugger is activated by an ARM ABORT exception, and
 * runs in an independant context from the remainder of the system
 * software.
 */
#define	NUMBRK	8				// Number of breakpoints
#define	BREAKOP	0xE7F000F0		// Breakpoint opcode

static unsigned
	*panic_save,				// Pointer to ARM state save area
	breakaddr[NUMBRK+2],		// Breakpoint addresses
	breakop[NUMBRK+2];			// Breakpoint saved opcodes
static unsigned char
	*panic_cptr;				// Command line parse pointer

/*
 * ARM state display text table
 */
static const ccp reg_prompts[] = {					// CPSR 0
	"\nUSR SPSR", " SP", " LR",						// 1 2 3
	"\nSVC SPSR", " SP", " LR",						// 4 5 6
	"\nIRQ SPSR", " SP", " LR",						// 7 8 9
	"\nFIQ SPSR", " R8", " R9", " R10",				// 10 11 12 13
	"\n    R11", " R12", " SP", "  LR",				// 14 15 16 17
	"\n R0", "  R1", "  R2", "  R3", "  R4",		// 18 19 20 21 22
	"\n R5", "  R6", "  R7", "  R8", "  R9",		// 23 24 25 26 27
	"\nR10", " R11", " R12",						// 28 29 30 (+LR)
	0 };
#define	SPC	save[31]

/*
 * Panic debugger command table
 */
static const ccp panic_words[] = {
	"BOOT",			// 0
	"BREAK",		// 1
	"BTRACE",		// 2
	"CALC",			// 3
	"CAUSE",		// 4
	"CONSOLE",		// 5
	"CONT",			// 6
	"DI",			// 7
	"DUMP",			// 8
	"INFO",			// 9
	"MSG",			// 10
	"MUTEX",		// 11
	"PACKET",		// 12
	"PS",			// 13
	"READY",		// 14
	"RESET",		// 15
	"STB",			// 16
	"STW",			// 17
	"TIME",			// 18
	"CPSR",			// 19 - Register names
	"USPSR", "USP", "ULR",
	"SSPSR", "SSP", "SLR",
	"ISPSR", "ISP", "ILR",
	"FSPSR", "FR8", "FR9", "FR10", "FR11", "FR12", "FSP", "FLR",
	"R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7", "R8", "R9",
	"R10", "R11", "R12", "PC",
	0 };
#define	REGMARK	19	// Register names begin here

/*
 * Data processing instruction names
 */
static const ccp dataproc1[] = {
	"AND", "EOR", "SUB", "RSB", "ADD", "ADC", "SBC", "RSC",
	"TST", "TEQ", "CMP", "CMN", "ORR", "MOV", "BIC", "MVN" };

// 3 3 2 2  2 2 2 2  2 2 2 2  1 1 1 1    1 1 1 1  1 1 0 0  0 0 0 0  0 0 0 0
// 1 0 9 8  7 6 5 4  3 2 1 0  9 8 7 6    5 4 3 2  1 0 9 8  7 6 5 4  3 2 1 0
// - - - -  - - - -  - - - -  - - - -    - - - -  - - - -  - - - -  - - - -
/*
 * Level-1 object code classification table
 * Type:
 *	Bits  7-0 = Disassembler decode type
 *  Bits 15-8 = Single-step special handling type
 */
static const otable[] = {
// #include "itable.h"
0x0FB000F0, 0x01200000, 0x00000006, //Move register to status
0x0E000010, 0x00000000, 0x00000001, //Data processing immediate shift
0x0E000090, 0x00000010, 0x00000001, //Data processing register shift
0x0E000000, 0x02000000, 0x00000001, //Data processing immediate
0x0E000000, 0x04000000, 0x00000002, //Load/Store immediate offset
0x0E000010, 0x06000000, 0x00000002, //Load/Store register offset
0x0E000000, 0x0A000000, 0x00000103, //B/BL
0x0E000000, 0x08000000, 0x00000004, //Load/Store multiple
0x0F000000, 0x0F000000, 0x00000005, //SWI
0x0FB00000, 0x03200000, 0x00000006, //Move immediate to status
0x0F000010, 0x0E000010, 0x00000007, //Coproc register transfers
0x0F900010, 0x01000000, 0x00000063, //Misc. instructions
0x0F900090, 0x01000010, 0x00000063, //Misc. instructions
0x0E000090, 0x00000090, 0x00000063, //Multiplies, extra loads/stores
0xFE000000, 0xFA000000, 0x00000063, //B/BL & change to thumb
0x0E000000, 0x0C000000, 0x00000063, //Coproc & Double-reg transfers
0x0F000010, 0x0E000000, 0x00000063, //Coproc data processing
0 };

/* -------- Single-Step -------- */
// Install breakpoint
static void step1(unsigned addr, unsigned index)
{
	unsigned i;
	for(i=0; i < NUMBRK; ++i) {
		if(breakaddr[i] == addr) {
			return; } }
	breakaddr[index] = addr;
	breakop[index] = *(unsigned*)addr;
	*(unsigned*)addr = BREAKOP;
}
	
static unsigned step2(unsigned PC)
{
	unsigned i, k, OP;

	OP = *(unsigned*)PC;
	for(i=0; otable[i]; i += 3) {
		if((otable[i] & OP) == otable[i+1])
			k = otable[i+2];
			goto found; }

	return 0;

found:
	// Possibly 
	switch((k >> 8) & 255) {
	default: return 0;
	case 0x01 :						// B/BL
		i = (OP & 0x00FFFFFF) * 4;
		if(i & 0x02000000) i |= 0xFE000000;
		PC = PC + i + 8;
		break; }

	// Install breakpoint
rawprintf(" TB:%08x\n", PC);
	step1(PC, NUMBRK+1);
	return 255;
}

/* -------- Disassembler -------- */

static unsigned
	Disadr,						// Disassembly address
	Disop,						// Disassembly opcode
	Dislen,						// Disassembly length
	Dispos;						// Output buffer position
static char
	Disout[81];					// Output buffer

/*
 * Performa rotate-right operation
 */
static unsigned rr(unsigned v, unsigned s)
{
	if(s)
		return (v >> s) | (v << (32-s));
	return v;
}

/*
 * Display character to disassembly output buffer
 */
static void dc(char c)
{
	Disout[Dispos++] = c;
}

/*
 * Display "printf" to disassembly output buffer
 */
static void dp(const char *format, ...)
{
	_format_(&Disout[Dispos], sizeof(Disout)-Dispos, format, (unsigned*)&format);
	while(Disout[Dispos])
		++Dispos;
}

/*
 * Display register name from index
 */
static void dr(unsigned r)
{
	switch(r &= 15) {
	case 15 : dp("pc");	return;
	case 14 : dp("lr");	return;
	case 13 : dp("sp");	return; }
	dp("r%u", r);
}

/*
 * Display register from 4-bit field at offset in opcode
 */
static void drx(unsigned s)
{
	dr(Disop >> s);
}

/*
 * Display condition code to disassembly output
 */
static void discond(void)
{
	static const ccp cnames[] = {
	"EQ", "NE", "CS", "CC", "MI", "PL", "VS", "VC",
	"HI", "LS", "GE", "LT", "GT", "LE", "", "??" };
	dp(cnames[Disop >> 28]);
}

/*
 * Display translated data string to disassembly output
 *
 * Specal Bit functions in translation string:
 * 100b bbbb  = Bit must be set
 * 101b bbbb  = Bit must be clear
 * 110b bbbb  = & bit must be set
 * 111b bbbb  = & bit must be clear
 */
static void dt(char *t)
{
	unsigned char c, bf;
	unsigned i, j;

	while(c = *t++) {

	if(c & 0x80) {							// Bit conditional: 1xxbbbbb
		i = (Disop >> (c & 0x1F)) & 1;
		switch(c & 0x60) {
		case 0x00 :	bf = i ? 0x81 : 0x80;	continue;	// Bit must be set
		case 0x20 : bf = i ? 0x80 : 0x81;	continue;	// Bit must be clear
		case 0x40 :
			if(bf & 1) bf = i ? 0x81 : 0x80;	continue;
		case 0x60 :
			if(bf & 1) bf = i ? 0x80 : 0x81;	continue; } }

	switch(c) {			// Conditional switches
	case ';' : bf = 0;		continue;
	case ':' : bf ^= 0x01;	continue; }

	if(bf & 0x80) {		// Conditional active
		if(!(bf & 1))
			continue; }

	switch(c) {
	case '~' : dc(*t++);						continue;	// Protect
	case 'b' : if(Disop & 0x00400000) dc('B');	continue;	// Byte indicator
	case 'c' : discond();						continue;	// Condition code
	case 'd' : drx(12);							continue;	// Dest register
	case 'n' : drx(16);							continue;	// iNdex register
	case 'm' : drx(0);							continue;	// M register
	case 's' :												// Status register
		dp((Disop&0x01400000) ? "SPSR" : "CPSR");
		continue;
	case 'f' :												// SR fields
		if(Disop & 0x000F0000) dc('_');
		if(Disop & 0x00010000) dc('c');
		if(Disop & 0x00020000) dc('x');
		if(Disop & 0x00040000) dc('s');
		if(Disop & 0x00080000) dc('f');
		continue;
	case 'i' :												// <<Immediate
		if(Disop & 0x02000000) {	// Immediate
			i = Disop & 0xFF;
			j = ((Disop >> 8) & 15) * 2;
			dp("#%x", rr(i,j));
			continue; }
		drx(0);
		i = (Disop >> 7) & 0x1F;
		switch(Disop & 0x70) {
		case 0x00 : if(i) dp(" LSL %u", i); 				continue;
		case 0x10 : dp(" LSL "); drx(8);					continue;
		case 0x20 : dp(" LSR %u", i);						continue;
		case 0x30 : dp(" LSR "); drx(8);					continue;
		case 0x40 : dp(" ASR %u", i);						continue;
		case 0x50 : dp(" ASR "); drx(8);					continue;
		case 0x60 : dp(" ROR %u", i);						continue;
		case 0x70 : dp(" ROR "); drx(8);					continue; }
	case 'j' :	dp("%x", Disop & 0xFFF);		continue;	// 12-bit immediate
	case '+' :	dc((Disop&0x0080000)?'+':'-');	continue;	// U +/-
	} dc(c); }
}

/*
 * Disassemble ARM opcode and display
 */
static unsigned disasm(unsigned *addr)
{
	unsigned i, j, k;

	Disadr = (unsigned)addr;
	Disout[Dislen = Dispos = 0] = 0;
	Disop = addr[Dislen++];

	for(i=k=0; otable[i]; i += 3) {
		if((Disop & otable[i]) == otable[i+1]) {
			k = otable[i+2];
			break; } }
	switch(k & 0xFF) {
	case 0 : dp("???"); goto dpend;		// Undefined

	case 0x01 :		// Data processing
		dp(dataproc1[i=(Disop >> 21) & 15]);
		// 10010100 94 - Bit 20 'S'
		switch(i) {
		case 0x0D :	// MOV
		case 0x0F : // MVN
			dt("c\x94S; d,i");
			goto dpend;
		case 0x08 : // TST
		case 0x09 : // TEQ
		case 0x0A : // CMP
		case 0x0B : // CMN
			dt("c n,i");
			goto dpend;
		default: dt("c\x94S; d,n,i");
			goto dpend; }

	case 0x02 :			// LD/ST immediate/register
		dp((Disop & 0x00100000) ? "LD" : "ST");
		// 10011000 98 - Bit 24 set
		// 10111000	B8 - Bit 24 clear
		// 11010110	D6 - & Bit 22 set
		// 11110110	F6 - & Bit 22 clear
		if(!(Disop & 0x02000000))
			dt("R\xB8\xD6T;b d,[n\xB8];,#+j\x98]\xD6!;");
		else
			dt("R\xB8\xD6T;b d,[n\xB8];,+i]\xD6!;");
		goto dpend;

	case 0x03:			// B & BL
		dc('B');
		if(Disop & 0x01000000) dc('L');
		discond();
		i = (Disop & 0x00FFFFFF) * 4;
		if(i & 0x02000000) i |= 0xFE000000;
		dp(" %08x", (unsigned)addr + i + 8);
		goto dpend;


	case 0x04 :			// LDM/STM
		dp((Disop & 0x00100000) ? "LD" : "ST");
		// 10010111	97 = 23 'U' set
		// 10110111 B7 = 23 'U' clear
		// 10011000 98 = 24 'P' set
		// 10111000 B8 = 24 'P' clear
		// 10010101 95 = 21 'W' set
		dt("M\x97I:D;\x98:A:B; n\x95!;,{");
		j = 255;		// No last regsiter
		for(i=0; i < 16; ++i) {
			if((1 << i) & Disop) {	// Register is included
				if(j != 255)
					dc(',');
				dr(i);
				j = i;
		loop:	if(++j < 16) {
					if((1 << j) & Disop)
						goto loop; }
				if((j - i) > 2) {
					dc('-');
					dr(i = j-1); } } }
		dt("}\x96^;");	// S bit
		goto dpend;

	case 0x05 :		// SWI
		dp("SWI %u", (Disop & 0x00FFFFFF));
		goto dpend;

	case 0x06 :		// Immediate to status
		dt("MSRc sf,i");
		goto dpend;

	case 0x07 :		// Coprocessor
		dp((Disop & 0x00100000) ? "MRC" : "MCR");
		discond();
		dc(' ');
		dp("p%u,%u,", (Disop >> 8) & 15, (Disop >> 21) & 7);
		drx(12);
		dp(",c%u,c%u,%u", (Disop >> 16) & 15, Disop & 15, (Disop >> 5) & 7);
		goto dpend;

	} dp("Unimplemented %08x", k);
dpend:
	rawprintf("\n%08x %08x ", addr, *addr);
	k =  8;
	for(i=j=0; i < Dispos; ++i) {
		if(Disout[i] == ' ') {
			do {
				rawputc(' '); }
			while(++j % k);
			k = 1;
			continue; }
		rawputc(Disout[i]);
		++j; }
	return Dislen;
}

/* -------- Panic Debugger -------- */

/*
 * Display task information from TCB
 */
static void show_task(struct TCB *t)
{
	if(t) {
		rawprintf("%-3u %-8s %-3u %-3u %08x %08x %08x %08x %08x",
			t->Id, t->Name, t->State, t->Priority, t, t->SpLow,
			t->Sp, t->Mqueue, t->Next); }
}

/*
 * Register a panic handler
 */
static void OS_panic_register(unsigned name, unsigned address)
{
	if(OS_panic_top >= MAX_PANIC)
		OS_panic("Too many PANIC handlers");
	OS_panic_name[OS_panic_top] = name;
	OS_panic_address[OS_panic_top++] = address;
}

/*
 * Get value element from command line
 */
static unsigned getval(unsigned *result)
{
	unsigned i;
	unsigned char temp[33], *sp;
	unsigned panic_value(unsigned *result);

	skip_blanks(&panic_cptr);
	switch(*panic_cptr++) {
	case '-' : if(getval(&i)) return 255; *result = -i; return 0;
	case '~' : if(getval(&i)) return 255; *result = ~i; return 0;
	case '*' : if(getval(&i)) return 255; *result = *(unsigned*)i; return 0;
	case '@' : if(getval(&i)) return 255; *result = *(unsigned char*)i; return 0;
	case '(' : if(panic_value(&i)) return 255; *result = i; return 0; }

	sp = --panic_cptr;
	parse(&panic_cptr, temp, 0x1300+32);
	for(i=REGMARK; panic_words[i]; ++i) {
		if(!strcmp((void*)panic_words[i], temp)) {
			*result = panic_save[i-REGMARK];
			return 0; } }

	panic_cptr = sp;
	i = 0;
	if(*panic_cptr == '.') {
		i = 255;
		++panic_cptr; }

	if(!parse(&panic_cptr, temp, i ? 0x110A : 0x120A)) return 255;
	*result = i ? atoi(temp) : atox(temp);
	return 0;
}

/*
 * Calculate an expression from the command line
 */
unsigned panic_value(unsigned *result)
{
	unsigned v, v1;
	unsigned char o;
	if(getval(&v)) return 255;
next:
	skip_blanks(&panic_cptr);
	switch(o=*panic_cptr++) {
		case '>' :
		case '<' : if(*panic_cptr != o) return 255;
			++panic_cptr;
		case '+' :
		case '-' :
		case '*' :
		case '/' :
		case '%' :
		case '&' :
		case '|' :
		case '^' :
			if(getval(&v1)) return 255;
			switch(o) {
			case '+' : v += v1;	goto next;
			case '-' : v -= v1;	goto next;
			case '*' : v *= v1;	goto next;
			case '/' : v /= v1;	goto next;
			case '%' : v %= v1;	goto next;
			case '&' : v &= v1;	goto next;
			case '|' : v |= v1;	goto next;
			case '^' : v ^= v1;	goto next;
			case '<' : v<<= v1;	goto next;
			case '>' : v>>= v1; goto next; }
		default:
			--panic_cptr;
		case ')' : ; }
	*result = v;
	return 0;
}

/*
 * Main panic debugger
 * Entry:
 *	vector	= Vector type causing exception (0=manual panic)
 *	save	= ARM state save area (see reg_prompts)
 *	addr	= Memory address for data abort
 */
void __OSpanic(unsigned vector, unsigned *save, unsigned addr)
{
	unsigned i, j, k, *sp, *wp;
	struct TCB *t;
	struct TIMER *u;
	struct MESSAGE *m;
	char bf, msg[81];
	unsigned char cmd[81], temp[33];

	panic_save = save;		// Make save area globally available

	bf = 255;				// Assume not a breakpoint
	switch(vector) {
		case 0x04 :			// Bad instruction - could be breakpoint
			j = SPC - 4;
			for(i=0; i < (NUMBRK+2); ++i) {
				if(breakaddr[i] == j) {
					sprintf(msg, "Breakpoint %u", bf = i);
					goto breakout; } }
			strcpy(msg, "Illegal instruction");						break;
		case 0x0C :	strcpy(msg, "Prefetch abort");					break;
		case 0x1C : strcpy(msg, "FIQ");								break;
		case 0x10 :			// Data abort
			sprintf(msg, "Access violation at %08x", addr);
			SPC -= 4;		// Special case - remove extra prefetch
			break;
		case 0 :			// Manual Panic
			switch(*save & 0x1F) {	// Determine location of stacked args
			case 0x10 :	sp=(unsigned*)save[2];	SPC=save[3];	break;	// User
			case 0x11 : sp=(unsigned*)save[16];	SPC=save[17];	break;	// FIQ
			case 0x12 : sp=(unsigned*)save[8];	SPC=save[9];	break;	// IRQ
			case 0x13 : sp=(unsigned*)save[5];	SPC=save[6];	break;	// SVC
			} sp -= 4;
			_format_(msg, 80, (char*)*sp, sp); }

breakout:
	SPC -= 4;				// Adjust Program Counter to remove ARM prefetch
	// Restore breakpointed opcodes to memory
	for(i=0; i < (NUMBRK+2); ++i) {
		if(j = breakaddr[i]) {
			*(unsigned*)j = breakop[i]; } }

	breakaddr[NUMBRK] = breakaddr[NUMBRK+1] = 
	breakaddr[0] = 0;		// Breakpoint-0 "disappears"

	if((bf == NUMBRK) || (bf == (NUMBRK+1))) {
		for(i=0; i < NUMBRK; ++i) {
			if(j=breakaddr[i]) {
//				rawprintf("\nInstalling breakpoint %u at %08x", i, j);
				breakop[i] = *(unsigned*)j;
				*(unsigned*)j = BREAKOP; } }
		__OSreturn(save); }

top:
	rawprintf("\nPANIC: %s", msg);
	i = *save;	// Display PC and CPSR (with interpretation)
	rawprintf("\nPC=%08x CPSR=%08x Flags:%c%c%c%c%c-%c%c%c Mode:", SPC, i,
		(i & 0x80000000) ? 'N' : 'n',
		(i & 0x40000000) ? 'Z' : 'z',
		(i & 0x20000000) ? 'C' : 'c',
		(i & 0x10000000) ? 'V' : 'v',
		(i & 0x08000000) ? 'Q' : 'q',
		(i & 0x00000080) ? 'I' : 'i',
		(i & 0x00000040) ? 'F' : 'f',
		(i & 0x00000020) ? 'T' : 't' );
	switch(i & 0x1F) {
	case 0x10 : rawprintf("USR");	break;
	case 0x11 : rawprintf("FIQ");	break;
	case 0x12 : rawprintf("IRQ");	break;
	case 0x13 : rawprintf("SUPER");	break;
	case 0x17 : rawprintf("ABORT");	break;
	case 0x1F : rawprintf("SYS");	break;
	default: rawprintf("???"); }

	// Display saved ARM state
	for(i=0; reg_prompts[i]; ++i)
		rawprintf("%s=%08x", reg_prompts[i], save[i+1]);

#ifndef OO_DEV
	// Release - if panic not active, wait 10 seconds for
	// ESC*3 and then perform hardware reset.
	if(OS_panic_active != 0x55) {
		rawputc(' ');
		for(i=j=0; i < 0x00800000; ++i) {
			if(!(i & 0xFFFFF))
				rawputc('*');
			WDOG = 0;
			if(rawtestc() == 0x1B) {
				if(++j == 3) {
					OS_panic_active = 0x55;
					goto cmd; } } }
		goto reset; }
#endif

	// Main command prompt
cmd: for(;;) {
		rawputs("\nPanic> ");
		rawgets(panic_cptr = cmd, sizeof(cmd)-1);
		if(!skip_blanks(&panic_cptr)) continue;
		parse(&panic_cptr, temp, 0x1000+32);
		for(i=0; panic_words[i]; ++i) {		// Lookup panic command
			if(!strcmp((void*)panic_words[i], temp))
				goto found; }
		for(i=0; i < OS_panic_top; ++i) {	// Lookup panic handler
			if(!strcmp((void*)OS_panic_name[i], temp)) {
				i = OS_panic_address[i];
				((PFV)i)((unsigned)&panic_cptr);
				goto cmd; } }
		// Not found - display available commands and handlers
		rawprintf("\n'%s' unknown, use:", temp);
		for(i=j=0; panic_words[i]; ++i) {
			if(!(j++ & 7)) rawputc('\n');
			rawprintf(" %-8s", panic_words[i]); }
		for(i=0; i < OS_panic_top; ++i) {
			if(!(j++ & 7)) rawputc('\n');
			rawprintf(" %-8s", OS_panic_name[i]); }
		continue;

	// Internal PANIC command - handle it
found: switch(i) {
		case 0 :
			INTMR1 = 0;
			while(SYSFLG1 & 0x00000800);	// Wait for UART TX complete
			PLLW	= 0x28000000;			// Configure for 74Mhz
			UBRLCR1	= 0x00078003;			// 56k @ 74Mhz
			__OSboot();		// BOOT

		case 1 :								// BREAK
			if(parse_number(&panic_cptr, &i, 0)) {
				for(i=0; i < NUMBRK; ++i)
					rawprintf("%c%u:%08x", (i&3) ? ' ' : '\n', i, breakaddr[i]);
				continue; }
			if(panic_value(&j) || (i >= 10) || (j & 3)) goto badop;
			for(k=0; k < NUMBRK; ++k) {
				if(breakaddr[k] && (breakaddr[k] == j) && (k != i))
					goto badop; }
			breakaddr[i] = j;
			continue;

		case 2 :								// BTRACE
			switch(*save & 0x1F) {
			case 0x10 :	sp=(unsigned*)save[2];	break;	// User
			case 0x11 : sp=(unsigned*)save[16];	break;	// FIQ
			case 0x12 : sp=(unsigned*)save[8];	break;	// IRQ
			case 0x13 : sp=(unsigned*)save[5];	break;	// SVC
			} if(panic_value(&i))
				i = save[29];
			do {
				wp = (unsigned*)(i-12);
				rawprintf("\n%08x fp:%08x sp:%08x lr:%08x cp:%08x", i,
					*wp, wp[1], wp[2], wp[3]);
				i = *wp;
				if((i - (unsigned)sp) > 0x10000) rawputs(" ?fp");
				if(((unsigned)wp[1] - (unsigned)sp) > 0x10000) rawputs(" ?sp");
				if(wp[2] > 0x00800000) rawputs(" ?lr");
				if(wp[3] > 0x00800000) rawputs(" ?cp"); }
			while(rawgetc() == ' ');
			continue;

		case 3 :								// CALC
			do {
				if(panic_value(&i)) goto badop;
				rawprintf("\n=%08x ", k=i);
				for(j=0; j < 8; ++j) {
					if(j) rawputc('-');
					rawprintf("%04b", k >> 28);
					k <<= 4; }
				rawprintf(" .%u", i);
				if(i & 0x80000000) rawprintf(" %d", i); }
			while(skip_blanks(&panic_cptr));
			continue;

		case 4 :		goto top;				// CAUSE

		case 5 :								// CONSOLE
			rawprintf("\nOwn:%u RW:%u RR:%u %08x[%u]", OS_console_owner,
				OS_console_rx_write, OS_console_rx_read,
				m = Uart_tx_msg, Uart_tx_position);
			if(m)
				rawprintf("\nAddress  Info Info1 Link");
			while(m) {
				if(rawtestc() == 0x1B) goto cmd;
				rawprintf("\n%08x %-5u%-6u%08x", m, m->Info, m->Info1, m->Link);
				m = m->Link; }
			continue;

		case 6 :								// CONT
			for(i=k=0; i < NUMBRK; ++i) {
				if(j=breakaddr[i]) {
					if((i == bf) && (SPC == j)) {
						k = i + 1; }
					else {
						rawprintf("\nInstalling breakpoint %u at %08x", i, j);
						breakop[i] = *(unsigned*)j;
						*(unsigned*)j = BREAKOP; } } }
			if(bf >= (NUMBRK+2)) SPC += 4;
			if(k) {
				rawprintf("\nStepping over breakpoint %u at %08x", k-1, SPC);
				step2(SPC);
				step1(SPC+4, NUMBRK); }
			__OSreturn(save);
			continue;

		case 7 :								// DI
			if(panic_value(&i)) goto badop;
			do {
					disasm((unsigned*)i);
					i += 4; }
			while(rawgetc() == ' ');
			continue;

		case 8 :								// DUMP
			if(!panic_value(&i)) {
				if(panic_value(&j)) j = i + 255;
				if(j < i) goto badop;
				dump((void*)i, (j-i)+1, &rawprintf, &rawtestc);
				continue; }
		badop: rawputs("\n?");
			continue;

		case 9 :								// INFO
			info(&rawprintf);
			continue;

		case 10 :								// MSG
			rawprintf("\nS:%u/%08x M:%u/%08x L:%u/%08x",
				OS_free_msg_small, Small_free,
				OS_free_msg_medium,	Medium_free,
				OS_free_msg_large, Large_free);
			if(parse_number(&panic_cptr, &i, 0)) continue;
			if(i < MAX_TASKS) {
				if(m = Task_list[i].Mqueue) {
					do {
						rawprintf("\n-->%08x I:%02x I1:%02x I2:%04x L:%08x", m, m->Info, m->Info1, m->Info2, m->Link);
						if(dump(m->Data, MSG_SMALL, &rawprintf, &rawtestc))
							goto cmd; }
					while(m = m->Link); } }
			continue;

		case 11:								// MUTEX
			rawprintf("\n# Own - Addr     Pid Typ Link");
			for(i=0; i < MAX_MUTEX; ++i) {
			rawprintf("\n%u %u", i, Mutex_owner[i], j = (unsigned)Mutex_queue[i]);
				while(j) {
					rawprintf("\n%8s%08x %-4u%-4u%08x", "", j, ((struct MUTEX*)j)->Mid,
						((struct MUTEX*)j)->Mtype, ((struct MUTEX*)j)->Link);
					j = (unsigned)((struct MUTEX*)j)->Link; } }
			continue;

		case 12 :								// PACKET
			packet(&rawprintf);
			continue;

		case 13 :								// PS
			ps(panic_cptr, &rawprintf);
			continue;

		case 14 :								// READY
			rawputs("\nActive: "); show_task(OS_active_tcb);
			for(i=0; i <= PRIORITIES; ++i) {
				t = Tready[i];
				rawprintf("\nPriority %u %08x %08x:%s", i, t, Tnext[i], t ? "" : " empty");
				while(t) {
					if(rawtestc() == 0x1B) goto cmd;
					rawputs("\n   ");
					show_task(t);
					if(i == PRIORITIES)
						break;
					t = t->Next; } }
			continue;

		case 15 :								// RESET
#ifndef OO_DEV
reset:
#endif
			rawputc('\n');
			INTMR1 = 0;
			i = *(unsigned*)0xC0000000;
			for(;;);

		case 16 :								// STW
			if(panic_value(&i)) goto badop;
			do {
				if(panic_value(&j)) goto badop;
				*(unsigned char*)i = j;
				++i; }
			while(skip_blanks(&panic_cptr));
			continue;

		case 17 :								// STB
			if(panic_value(&i)) goto badop;
			if(i & 3) goto badop;
			do {
				if(panic_value(&j)) goto badop;
				*(unsigned*)i = j;
				i += 4; }
			while(skip_blanks(&panic_cptr));
			continue;

		case 18 :								// TIME
			rawprintf("\nTick:%08x Base:%08x Free:%08x", OS_timer_tick, u = Timer_base, Timer_free);
			rawprintf("\nCountdown  Setting    Control  Link     Uparm");
			while(u) {
				if(rawtestc() == 0x1B) goto cmd;
				rawprintf("\n%-11u%-11u%08x %08x %08x", u->Countdown,u->Setting, u->Control, u->Link, u->Uparm);
				u = u->Link; }
			continue;

		default:								// register-name
			switch(skip_blanks(&panic_cptr)) {
			case 0 :
			case '?' : rawprintf("\n=%08x", save[i-REGMARK]); continue;
			case '=' : ++panic_cptr;
				if(panic_value(&j)) goto badop;
				save[i-REGMARK] = j;
				continue; }
			goto badop;

	} }
}
