/*
 * ArmOS - Demonstration program
 *
 * This program demonstrates some of the capabilities currently
 * available in the operating system.
 *
 * The demo starts 5 timers flashing the front panel lights - duty
 * cycle is slightly different for each one so that interesting
 * patterns will appear.
 *
 * Three demo processes are started:
 *
 * RXMSG - Simply displays an informational message when it receives
 *         a message.
 *
 * ECHO  - Sends any message it receives back to the sender.
 *
 * TIMER - Starts a 30 second timer, also displays any messages it
 *         receives.
 *
 * Things to do:
 *	Try the various CCP commands.
 *	Use 'DUMP'	to look at memory
 *	Use 'INFO'	to display some system information.
 *	Use 'PS'	to display active tasks.
 *	Use 'HALT'	to stop the system.
 *	USe 'DEBUG'	to change the debug output mask.
 *
 * Note that all processes (except OS) initially indicate a task state of
 * '5', which means that they are blocked waiting for system event 2.
 * (task-state = event+2).
 *
 * Issue the command: SIGNAL 2
 * This will unblock all of the tasks, which will output unconditional
 * Debug messages indicating that they are active.
 *
 *	Set: DEBUG 7
 *	to enable all debug output from all three tasks.
 *	Try> RXMSG hello world
 *		You will see an informational message from RXMSG that it has
 *		received your mesage.
 *
 * Try> ECHO hello world
 *		You will see an informational message from ECHO that it has
 *		received your message - you will also see a message dump
 *		indicating that you have received a message back.
 *
 *	Try> TIMER hello world
 *		You will see an informational message from TIMER that it has
 *		received your mesage. You will also see informational messages
 *		every 30 seconds indicating that the timer message has been
 *		received.
 *
 *	Set: DEBUG 0
 *	to disable all output. - try the above again - note that you do not
 *	see any informational messages, however you will get the message back
 *	from ECHO indicating that it is still running.
 *
 *	Use: 'DEBUG 1' to enable only RXMSG informational text.
 *	Use: 'DEBUG 2' to enable only ECHO  informational text.
 *	Use: 'DEBUG 4' to enable only TIMER informational text.
 *  -- Try bit combinations to enable 2 out of 3 --
 *
 * Connect a CO (1-4) to an extension of another phone system and ring it.
 * This will activate the CO ring interrupt handler, which will signal a
 * kernel panic (see dummy interrupt handlers at end of this file).
 *
 * Try the whole excercise again - note that until you: SIGNAL 2
 * the tasks will remain blocked, and no messages will be processed (they
 * will queue and get processed when SIGNAL 2 occurs).
 */

#include "os.h"					// Operating system definitions
#include "library.h"			// Library   definitions
#include "irq.h"				// Interrupt definitions
#include "hardware.h"


/*
 * Global process handles - so all tasks can read them to send messages
 */
unsigned
	p1,		// Process 1 handle
	p2,		// Process 2 handle
	p3,		// Process 3 handle
	p4;		// Process 4 handle

unsigned
	t1,		// Timer 1 handle
	t2,		// Timer 2 handle
	t3,		// Timer 3 handle
	t4,		// timer 4 handle
	t5;		// Timer 5 handle

/*
 * Simple demonstration timer callback functions:
 */

// This function simply toggles the state of LED1 each time it is called
void flash1(unsigned x)
{
	static unsigned char f;
//	static unsigned count;
	if(f = f ? 0 : 255)
		OS_blc(1<<24);
	else
		OS_bls(1<<24);
/*	if(++count > 10) {
		rawprintf("Flash1=%u\n\r", x);
		count = 0; } */
}

// This function simply toggles the state of LED1 each time it is called
void flash2(void)
{
	static unsigned char f;
	if(f = f ? 0 : 255)
		OS_blc(1<<25);
	else
		OS_bls(1<<25);
}

// This function simply toggles the state of LED1 each time it is called
void flash3(void)
{
	static unsigned char f;
	if(f = f ? 0 : 255)
		OS_blc(1<<26);
	else
		OS_bls(1<<26);
}

// This function simply toggles the state of LED1 each time it is called
void flash4(void)
{
	static unsigned char f;
	if(f = f ? 0 : 255)
		OS_blc(1<<27);
	else
		OS_bls(1<<27);
}

// This function simply toggles the state of LED1 each time it is called
void flash5(void)
{
	static unsigned char f;
	if(f = f ? 0 : 255)
		OS_blc(1<<28);
	else
		OS_bls(1<<28);
}

/*
 * Simple demonstration processes
 */

unsigned get_char(unsigned timeout)
{
	unsigned i;
	unsigned char c;
	for(i=0; i < timeout; ++i) {
		OS_swap();
		if(OS_call(OS_CONSOLE_RX, &c, 1))
			return c; }
	return 0xFFFFFFFF;
}

/*
 * This process blocks waiting for a message, and when it is received,
 * displays the sender and content of the message.
 */
void process1(void)
{
	unsigned i, j;
	struct MESSAGE *m;
	static char record[1029];

//	Dprintf(0, "Received signal=%u", OS_call(OS_PROC_WAIT, 2));
	OS_call(OS_CONSOLE_AT);
	for(;;) {
		Dprintf(1, "Waiting for message.");
		m = (struct MESSAGE*)OS_call(OS_MSG_RECEIVE);
		Dprintf(1, "Msg from %u : '%s'", m->Info, m->Data);
		OS_call(OS_MSG_RELEASE, m);
		strupr(m);
		OS_call(OS_CONSOLE_ALLOC);
		if(strcmp(m->Data, "!ZFR"))		// Send OK for everything
			printf("\n\rOK");
		else {
			printf("\n\rCONNECT\r\n");
		top:
			for(;;) {
				record[i] = get_char(0x80000);
				for(i=1; i < 1029; ++i) {
					j = get_char(0x10000);
					if(j & 0xFFFF0000) {
						while(get_char(0x10000) != 0xFFFFFFFF);
						printf("ACK2\n\r");
						goto top; }
					record[i] = j; };
				if(record[1024])
					printf("ACK1\r\n");
				else
					printf("ACK0\r\n"); }
		} OS_call(OS_CONSOLE_RELEASE); }

}

/*
 * This process blocks waiting for a message, and when it is received,
 * displays the sender and content of the message, then sends it back
 * the the sender.
 */
void process2(void)
{
	struct MESSAGE *m;
	Dprintf(0, "Received signal=%u", OS_call(OS_PROC_WAIT, 2));
	for(;;) {
		Dprintf(2, "Waiting for message.");
		m = (struct MESSAGE*)OS_call(OS_MSG_RECEIVE);
		Dprintf(2, "msg from %u : '%s'", m->Info, m->Data);
		OS_call(OS_MSG_SEND, m->Info, m); }
}

/*
 * This process just waits for and displays messages line process2,
 * however before entering it's main loop, it establishes a timer
 * which will cause it to receive messages (and unblock) at regular
 * intervals.
 */
void process3(void)
{
	unsigned ft, i;
	struct MESSAGE *m;

	Dprintf(0, "Received signal=%u", OS_call(OS_PROC_WAIT, 2));
	Dprintf(0, "10 second timer=%u", OS_call(OS_TIMER_START, 10000, 0, 0, 99990010));
	Dprintf(0, "15 second timer=%u", OS_call(OS_TIMER_START, 15000, 0, 0, 99990015));
	Dprintf(0, "20 second timer=%u", ft = OS_call(OS_TIMER_START, 20000, 30000, 0, 99990020));
	for(;;) {
		Dprintf(4, "Waiting for message.");
		m = (struct MESSAGE*)OS_call(OS_MSG_RECEIVE);
		if(m->Info == 255) {	// Timer message
			Dprintf(4, "Timer %u tick [%u]", m->Info1, *(unsigned*)m->Data);
//			for(i=0; i < 0x1000000; ++i)
				; }
		else
			Dprintf(4, "msg from %u : '%s'", m->Info, m->Data);
		OS_call(OS_MSG_RELEASE, m); }
}

void process4(void)
{
	struct MESSAGE *head, *tail, *m;
top:
	head = 0;
	// First wait for message we want
	for(;;) {
		m = (struct MESSAGE*)OS_call(OS_MSG_RECEIVE);
		if(m->Data[0] == '$')
			break;
		if(head) {
			tail->Link = m;
			tail = m; }
		else {
			head = tail = m; }
		Dprintf(0, "Queueing: %s", m->Data); }
	Dprintf(0, "RX1: %s", m->Data);
	OS_call(OS_MSG_RELEASE, m);
	OS_call(OS_MSG_REQUEUE, head, tail);
	while(m = (struct MESSAGE*)OS_call(OS_MSG_CHECK)) {
		Dprintf(0, "RX2: %s\n", m->Data);
		OS_call(OS_MSG_RELEASE, m); }
	goto top;
}

/*
 * This is the user main module.
 *
 * This initializes application tasks and timers, then returns to the
 * system where the process evolves into the CCP.
 */
void user_init(void)
{
	printf("\n\rApplication initializing.");	// Announce that we are alive!

	// Establish 5 timers with callback functions to flash the front panel
	// Led's - offset the timer intervals by just a little, to create
	// interesting patterns on the LED (after a while).
	printf("\n\rTimer1=%u", t1 = OS_call(OS_TIMER_START, 1000, 1000, &flash1, 1234));
	printf("\n\rTimer2=%u", t2 = OS_call(OS_TIMER_START, 1010, 1010, &flash2, 0));
	printf("\n\rTimer3=%u", t3 = OS_call(OS_TIMER_START, 1020, 1020, &flash3, 0));
	printf("\n\rTimer4=%u", t4 = OS_call(OS_TIMER_START, 1030, 1030, &flash4, 0));
	printf("\n\rTimer5=%u", t5 = OS_call(OS_TIMER_START, 1040, 1040, &flash5, 0));

	// Start the demonstration processes
	printf("\n\rProc1=%u", p1 = OS_call(OS_PROC_START, &process1, 0, 0x0002E000, 1024, "RXMSG", 99));
	printf("\n\rProc2=%u", p2 = OS_call(OS_PROC_START, &process2, 0, 0x0002D000, 1024, "ECHO", 99));
	printf("\n\rProc3=%u", p3 = OS_call(OS_PROC_START, &process3, 0, 0x0002C000, 1024, "TIMER", 99));
	printf("\n\rProc4=%u", p4 = OS_call(OS_PROC_START, &process4, 0, 0x0002B000, 1024, "QUEUE", 99));
}

/*
 * Dummy interrupt handlers
 * If these interrupts get enabled and occur, we will know it!
 */
void IRQ_co_ring(unsigned i){ OS_panic("CO1 ring %08x", i);	}
void IRQ_bdsp(void)			{ OS_panic("Bottom DSP");		}
void IRQ_tdsp(void)			{ OS_panic("Top DSP");			}
void IRQ_expansion(void)	{ OS_panic("Expansion");		}
