/*
 * CAM:
 *
 * This is a very simple INTEGER BASIC interpreter that I wrote a number
 * of years ago, and subsequently ported to MICRO-C. While not a great
 * example of coding style (it was a quick and dirty hack job), It is
 * quite instructive, as a simple but fairly complete interpreter.
 *
 * Variables:
 *	260 Numeric	  variables	:	A0-A9 ... Z0-Z9
 *	260 Character variables	:	A0$-A9$ ... Z0$-Z9$
 *	260 Numeric arrays		:	A0()-A9() ... Z0()-Z9()
 *
 *	For convenience the '0' variables can be referenced by letter
 *	only. IE: A is equivalent to A0 ... Z$ is equivalent to Z0$
 *
 * Statements:
 *	BEEP freq,ms			- Generate a BEEP on the PC speaker
 *	CLEAR					- Erase variables only
 *	CLOSE#n					- Close file (0-9) opened with OPEN
 *	CLS [attribute]			- Clear video screen
 *	DATA					- Enter "inline" data statements
 *	DELAY ms				- Stops for the indicated time
 *	DIM var(size)[, ... ]	- Dimension an array
 *	DOS "comand"			- Execute a DOS program
 *	END						- Terminate program with no message
 *	EXIT					- Terminate CAM
 *	FOR v=init TO limit [STEP increment] - Perform a counted loop
 *	GOSUB line				- Call a subroutine
 *	GOTO  line				- Jump to line
 *	GOTOXY x,y				- Position video cursor
 *	IF test THEN line		- Conditional goto
 *	IF test THEN statement	- Conditional statement (next statement only)
 *	INPUT var				- Get value for variable
 *	INPUT "prompt",var		- Get value of variable with prompt
 *		prompt must be a constant string, however you can use a char variable
 *		in prompt by concatinating it to such a string: INPUT ""+a$,b$
 *	INPUT#n,var				- Get value for variable from file (0-9)
 *	LET (default)			- variable = expression
 *	LIF test THEN statements- LONG IF (all statements to end of line)
 *	LIST [start,[end]]		- List program lines
 *	LIST#n ...				- List program to file (0-9)
 *	LOAD "name"				- Load program from disk file
 *		When LOAD is used within a program, execution continues with the
 *		first line of the newly loaded program. In this case, the user
 *		variables are NOT cleared. This provides a means of chaining
 *		to a new program, and passing information to it.
 *		Also note that LOAD must be the LAST statement on a line.
 *	NEW						- Erase program and variables
 *	NEXT [v]				- End counted loop
 *	OPEN#n,"name","opts"	- Open file (0-9), opts are same as "fopen()"
 *	ORDER line				- Position data read pointer
 *	OUT port,expr			- Write a value to an output port
 *	PRINT expr[,expr ...]	- Print to console
 *	PRINT#n,...				- Print to file (0-9)
 *	READ var[,var ...]		- Read data from program statements
 *		You MUST issue an "ORDER" statement targeting a line
 *		containing a valid DATA statement before using READ
 *	RETURN					- Return from subroutine
 *	REM						- Comment... reminder of line is ignored
 *	RUN [line]				- Run program
 *	SAVE ["name"]			- Save program to disk file
 *	STOP					- Terminate program & issue message
 **** New statements for the CAM project ***
 *	OSET					- Set parallel I/O bits
 *	ON value				- Turn ON  parallel I/O bits
 *	OFF value				- Turn OFF parallel I/O bits
 *	TUNE tune				- Play a PC speaker tune (0-6)
 *	PLAY "file",mask		- Play voice file
 *		mask=0 for sequence training, mask=16 for no-abort
 *	CONTROL function		- Perform a remote control function
 *	SCREEN "file"			- Load screen image
 *	SOUND frequency			- Generate a continuous tone
 *	SAY expr[,expr ...]		- Perform text-to-speech
 *	VOICE to,vo,pi,sp		- Initialize text-to-speech
 *	MIX channel,left,right	- Set mixer/volume levels
 *
 * Operators:
 *	+						- Addition, string concatination
 *	-						- Unary minus, subtraction
 *	*, /, %,				- multiplication, division, modulus
 *	&, |, ^					- AND, OR, Exclusive OR
 *	=, <>					- Assign/test equal, test NOTequal (num or string)
 *	<, <=, >, >=			- LT, LE, GT, GE (numbers only)
 *	!						- Unary NOT
 *		The "test" operators (=, <>, <, <=, >, >=) can be used in any
 *		expression, and evaluate to 1 of the test is TRUE, and 0 if it
 *		is FALSE. The IF and LIF commands accept any non-zero value to
 *		indicate a TRUE condition.
 *
 * Functions:
 *	CHR$(value)				- Returns character of passed value
 *	STR$(value)				- Returns ASCII string of value's digits
 *	ABS(value)				- Returns absolute value of argument
 *	ASC(char)				- Returns value of passed character
 *	INP(value)				- Read an I/O port
 *	KEY()					- Test for key from keyboard
 *	LEN(var)				- Return length of variable
 *	NUM(string)				- Convert string to number
 *	RND(value)				- Returns random number from 0 to (value-1)
 **** New functions for the CAM project ***
 *	BUTTON(mask)			- test/wait for button press
 *	JOYSTICK(index)			- Read joystick port
 *  TIME(index)				- Read current time
 *
 * Notes on the CAM extensions:
 *
 * 'mask' refers to a button mask, which controls what buttons are to be
 * enabled: Button1=1, Button2=2, Button3=4, Button4=8
 * Add up values for all buttons that you wish to be enabled.
 * Add 256 to force button() to wait.
 *
 * ?COPY.TXT 1982-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc basic -fop
 */
#include <stdio.h>
#include <comm.h>
#include <setjmp.h>

#define	ABORT		0x03	/* Abort character */
char break_flag = 0;		/* Break pending flag */

#include "cam.h"

/* Fixed parameters */
#define BUFFER_SIZE 100		/* input buffer size */
#define NUM_VAR 	260		/* number of variables */
#define SA_SIZE 	100		/* string accumulator size */
#define	STACK_SIZE	50		/* control stack size */

/* Control stack constant identifiers */
#define _FOR		1000	/* indicate FOR statement */
#define _GOSUB		_FOR+1	/* indicate GOSUB statement */

/* Primary keywords */
#define	LET		1
#define	EXIT	2
#define	LIST	3
#define	NEW		4
#define	RUN		5
#define	CLEAR	6
#define	GOSUB	7
#define	GOTO	8
#define	RETURN	9
#define	PRINT	10
#define	FOR		11
#define	NEXT	12
#define	IF		13
#define	LIF		14
#define	REM		15
#define	STOP	16
#define	END		17
#define	INPUT	18
#define	OPEN	19
#define	CLOSE	20
#define	DIM		21
#define	ORDER	22
#define	READ	23
#define	DATA	24
#define	SAVE	25
#define	LOAD	26
#define	DELAY	27
#define	BEEP	28
#define	DOS		29
#define	QDOS	30
#define	OUT		31
#define	CLS		32
#define	GOTOXY	33

/* Special CAM functions */
#define	PON		34
#define	POFF	35
#define	OSET	36
#define	PLAY	37
#define	TUNE	38
#define	CONTROL	39
#define	SCREEN	40
#define	SOUND	41
#define	SAY		42
#define	VOICE	43
#define	MIX		44

/* Secondary keywords */
#define	TO		45	/* Also used as marker */
#define	STEP	46
#define	THEN	47

/* Operators and functions */
#define	ADD		48	/* Also used as marker */
#define	SUB		49
#define	MUL		50
#define	DIV		51
#define	MOD		52
#define	AND		53
#define	OR		54
#define	XOR		55
#define	EQ		56
#define	NE		57
#define	LE		58
#define	LT		59
#define	GE		60
#define	GT		61
#define CHR		62
#define	STR		63
#define	ASC		64
#define	ABS		65
#define	NUM		66
#define	RND		67
#define	KEY		68
#define	INP		69
#define	LEN		70

/* Special CAM functions */
#define	BUTTON	71
#define	JOYSTK	72
#define	TIME	73

#define	RUN1	99

/* Make sure this token table matches the above definitions */
static char *reserved_words[] = {
	"LET", "EXIT", "LIST", "NEW", "RUN", "CLEAR", "GOSUB", "GOTO",
	"RETURN", "PRINT", "FOR", "NEXT", "IF", "LIF", "REM", "STOP",
	"END", "INPUT", "OPEN", "CLOSE", "DIM", "ORDER", "READ", "DATA",
	"SAVE", "LOAD", "DELAY", "BEEP", "DOS", "QDOS", "OUT", "CLS",
	"GOTOXY", "ON", "OFF", "OSET", "PLAY", "TUNE", "CONTROL", "SCREEN",
	"SOUND", "SAY", "VOICE", "MIX",
	"TO", "STEP", "THEN",
	"+", "-", "*", "/", "%", "&", "|", "^",
	"=", "<>", "<=", "<", ">=", ">",
	"CHR$(", "STR$(", "ASC(", "ABS(", "NUM(", "RND(", "KEY(", "INP(",
	"LEN(", "BUTTON(", "JOYSTICK(", "TIME(",
	0 };

/* Table of operator priorities */
static char priority[] = { 0, 1, 1, 2, 2, 2, 3, 3, 3, 1, 1, 1, 1, 1, 1 };

/* Table of error messages */
static char *error_messages[] = {
	"Syntax",				/* 0 */
	"Illegal program",		/* 1 */
	"Illegal direct",		/* 2 */
	"Line number",			/* 3 */
	"Wrong type",			/* 4 */
	"Divide by zero",		/* 5 */
	"Nesting",				/* 6 */
	"File not open",		/* 7 */
	"File already open",	/* 8 */
	"Input",				/* 9 */
	"Dimension",			/* 10 */
	"Data",					/* 11 */
	"Out of memory",		/* 12 */
/* Additional error messages for the CAM project */
	"Out of range",			/* 13 */
	"Interface"				/* 14 */
	};

struct line_record {
	unsigned Lnumber;
	struct line_record *Llink;
	char Ltext[]; };

char sa1[SA_SIZE], sa2[SA_SIZE];		/* String accumulators */
struct line_record *pgm_start,			/* Indicates start of program */
	*runptr,							/* Line we are RUNnning */
	*readptr;							/* Line we are READing */

unsigned dim_check[NUM_VAR];			/* Check dim sizes for arrays */

FILE *filein, *fileout;					/* File I/O active pointers */

jmp_buf savjmp;							/* Save area for set/longjmp */

/* Misc. global variables */
char *cmdptr,							/* Command line parse pointer */
	*dataptr,							/* Read data pointer */
	buffer[BUFFER_SIZE],				/* General input buffer */
	mode = 0,							/* 0=Stopped, !0=Running */
	expr_type,							/* Type of last expression */
	nest;								/* Nest level of expr. parser */
unsigned line,							/* Current line number */
	key_test = 0,						/* Key test data */
	ctl_ptr = 0,						/* Control stack pointer */
	ctl_stk[STACK_SIZE];				/* Control stack */

unsigned video_seg = 0xB800;

/*
 * The following variables must be iniitialized to zero. If your
 * compiler does not automatically zero uninitialized global
 * variables, modify these declarations to initialize them.
 */
char filename[65];						/* Name of program file */
FILE *files[10];						/* File unit numbers */
int num_vars[NUM_VAR];					/* Numeric variables */
int *dim_vars[NUM_VAR];					/* Dimensioned arrays */
char *char_vars[NUM_VAR];				/* Character variables */

extern unsigned RAND_SEED;				/* Random number seed */
asm " EXTRN ?heap:NEAR";
/*
 * Test for end of expression
 */
int is_e_end(char c)
{
	if((c >= (-128+TO)) && (c < (-128+ADD)))
		return(1);
	return (c == '\0') || (c == ':') || (c == ')') || (c == ',');
}

/*
 * Test for end of statement
 */
int is_l_end(char c)
{
	return (c == '\0') || (c == ':');
}

/*
 * Test for terminator character
 */
int isterm(char c)
{
	return (c == ' ') || (c == '\t');
}

/*
 * Advance to next non-blank & retreive data
 */
char skip_blank()
{
	while(isterm(*cmdptr))
		++cmdptr;
	return *cmdptr;
}

/*
 * Advance to., retrieve and skip next non blank
 */
char get_next()
{
	char c;

	while(isterm(c=*cmdptr))
		++cmdptr;
	if(c)
		++cmdptr;
	return c;
}

/*
 * Test for a specific character occuring next & remove if found
 */
int test_next(int token)
{
	if(skip_blank() == token) {
		++cmdptr;
		return -1; }
	return 0;
}

/*
 * Expect a specific token - syntax error if not found
 */
expect(int token)
{
	if(get_next() != token)
		error(0);
}

/*
 * Lookup up word from command line in table
 */
unsigned lookup(char *table[])
{
	unsigned i;
	char *cptr, *optr;

	optr = cmdptr;
	for(i=0; cptr = table[i]; ++i) {
		while((*cptr) && (*cptr == toupper(*cmdptr))) {
			++cptr;
			++cmdptr; }
		if(!*cptr) {
			if(! (isalnum(*(cptr-1)) && isalnum(*cmdptr)) ) {
				skip_blank();
				return i+1; } }
		cmdptr = optr; }
	return 0;
}

/*
 * Get a number from the input buffer
 */
unsigned get_dec()
{
	unsigned value;
	char c;
	value = 0;
	while(isdigit(c = *cmdptr)) {
		++cmdptr;
		value = (value * 10) + (c - '0'); }
	return value;
}
unsigned get_hex()
{
	unsigned value;
	char c;
	value = 0;
	for(;;) {
		if(isdigit(c = toupper(*cmdptr)))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else
			break;
		++cmdptr;
		value = (value << 4) + c; }
	return value;
}

/*
 * Allocate memory and zero it
 */
char *allocate(unsigned size)
{
	char *ptr;
	if(!(ptr = malloc(size)))
		error(12);
	memset(ptr, 0, size);
	return ptr;
}

/*
 * Delete a line from the program
 */
delete_line(unsigned lino)
{
	struct line_record *cptr, *bptr;

	if(!(cptr = pgm_start))					/* no lines in pgm */
		return;
	do {
		if(lino == cptr->Lnumber) {			/* we have line to delete */
			if(cptr == pgm_start)			/* first line in pgm */
				pgm_start = cptr->Llink;
			else
				bptr->Llink = cptr->Llink;	/* skip it in linked list */
			free(cptr);
			return; }
		bptr = cptr; }
	while(cptr = cptr->Llink);
}

/*
 * Insert a line into the program
 */
insert_line(unsigned lino)
{
	unsigned i;
	struct line_record *cptr, *bptr, *optr;
	char *ptr;

	ptr = cmdptr;
	for(i=5; *ptr; ++i)
		++ptr;
	bptr = allocate(i);
	bptr->Lnumber = lino;
	for(i=0; *cmdptr; ++i)
		bptr->Ltext[i] = *cmdptr++;
	bptr->Ltext[i] = 0;
	if((!(cptr = pgm_start)) || (lino < cptr->Lnumber)) {	/* at start */
		bptr->Llink = pgm_start;
		pgm_start = bptr; }
	else {				/* inserting into main part of pgm */
		for(;;) {
			optr = cptr;
			if((!(cptr = cptr->Llink)) || (lino < cptr->Lnumber)) {
				bptr->Llink = optr->Llink;
				optr->Llink = bptr;
				break; } } }
}

/*
 * Tokenize input line and Add/Replace a source line if suitable
 */
edit_program()
{
	unsigned value;
	char *ptr, c;

	cmdptr = ptr = buffer;
	/* Translate special tokens into codes */
	while(c = *cmdptr) {
		if(value = lookup(reserved_words))
			*ptr++ = value | 0x80;
		else {
			*ptr++ = c;
			++cmdptr;
			if(c == '"') {		/* double quote */
				while((c = *cmdptr) && (c != '"')) {
					++cmdptr;
					*ptr++ = c; }
				*ptr++ = *cmdptr++; } } }
	*ptr = 0;
	cmdptr = buffer;

	if(isdigit(skip_blank())) {	/* Valid source line */
		value = get_dec();		/* Get line number */
		delete_line(value);		/* Delete the old */
		if(skip_blank())
			insert_line(value);
		return -1; }			/* Insert the new */
	return 0;
}

/*
 * Locate given line in source
 */
struct line_record *find_line(unsigned line)
{
	struct line_record *cptr;

	for(cptr = pgm_start; cptr; cptr = cptr->Llink)
		if(cptr->Lnumber == line)
			return cptr;
	error(3);
}

/*
 * Compute variable address for assignment
 */
unsigned *address()
{
	unsigned i, j, *dptr;
	i = get_var();
	if(expr_type)
		return &char_vars[i];
	else {
		if(test_next('(')) {	/* Array */
			if(expr_type) error(0);
			if(!(dptr = dim_vars[i]))
				error(10);
			nest = 0;
			if((j = eval_sub()) >= dim_check[i])
				error(10);
			return &dptr[j]; } }
	return &num_vars[i];
}

/*
 * Execute a BASIC commands
 */
struct line_record *execute(char cmd)
{
	unsigned i, j, k, *dptr;
	int ii, jj;
	struct line_record *cptr;
	char c;

	switch(cmd & 0x7F) {
		case LET :
			dptr = address();
			j = expr_type;

			expect(-128+EQ);

			k = eval();

			if(j != expr_type)
				error(0);
			if(!expr_type)		/* numeric assignment */
				*dptr = k;
			else {				/* character assignment */
				if(*dptr)
					free(*dptr);
				if(*sa1)
					strcpy(*dptr = allocate(strlen(sa1)+1), sa1);
				else
					*dptr = 0; }
			break;
		case EXIT :
			unload_ct_voice();
			Cclose();
			exit(0);
		case LIST :
			chk_file(1);
			if(!isdigit(skip_blank())) {
				i=0; j=-1; }
			else {
				i = get_dec();
				if(get_next() == ',') {
					if(isdigit(skip_blank()))
						j=get_dec();
					else
						j = -1; }
				else
					j=i; }
			disp_pgm(fileout,i,j);
			break;
		case NEW :
			clear_vars();
			clear_pgm();
			*filename = 0;
			longjmp(savjmp, 1);
		case RUN :
			direct_only();
			clear_vars();
		case RUN1 :		/* No clearing */
			if(is_e_end(skip_blank()))
				runptr = pgm_start;
			else
				runptr = find_line(eval_num());
			--mode;			/* indicate running */
newline:
			while(runptr) {
				cmdptr = runptr->Ltext;
				line = runptr->Lnumber;
				do {
					if(i = kbtst()) {
						if(i == ABORT)
							break_flag = -1;
						else
							key_test = i; }
					if(break_flag)
						goto stop;
					if((cmd = skip_blank()) < 0) {
						++cmdptr;
						if(dptr=execute(cmd)) {
							runptr = dptr;
							goto newline; } }
					else
						execute(1); }
				while((c = get_next()) == ':');
				if(c)
					error(0);
				runptr = runptr->Llink; }
			mode = 0;
			break;
		case CLEAR :
			clear_vars();
			break;
		case GOSUB :
			if(ctl_ptr > (STACK_SIZE-3)) error(6);
			ctl_stk[ctl_ptr++] = runptr;
			ctl_stk[ctl_ptr++] = cmdptr;
			ctl_stk[ctl_ptr++] = _GOSUB;
		case GOTO :
			pgm_only();
			return find_line(eval_num());
		case RETURN :
			pgm_only();
			if(ctl_stk[--ctl_ptr] != _GOSUB)
				error(6);
			cmdptr = ctl_stk[--ctl_ptr];
			runptr = ctl_stk[--ctl_ptr];
			line = runptr->Lnumber;
			skip_stmt();
			break;
		case PRINT :
			chk_file(1);
			j = 0;
			do {
				if(is_l_end(skip_blank()))
					--j;
				else {
					i = eval();
					if(!expr_type) {
						num_string(i, sa1);
						putc(' ',fileout); }
					fputs(sa1, fileout); } }
			while(test_next(','));
			if(!j)
				putc('\n', fileout);
			break;
		case FOR :
			pgm_only();
			ii = 1;			/* default step value */
			i = get_var();
			if(expr_type) error(0);
			expect(-128+EQ);
			num_vars[i] = eval();
			if(expr_type) error(0);
			expect(-128+TO);
			jj = eval();
			if(test_next(-128+STEP))
				ii = eval();
			skip_stmt();
			if(ctl_ptr > (STACK_SIZE-6)) error(6);
			ctl_stk[ctl_ptr++] = runptr;	/* line */
			ctl_stk[ctl_ptr++] = cmdptr;	/* command pointer */
			ctl_stk[ctl_ptr++] = ii;		/* step value */
			ctl_stk[ctl_ptr++] = jj;		/* limit value */
			ctl_stk[ctl_ptr++] = i;			/* variable number */
			ctl_stk[ctl_ptr++] = _FOR;
			break;
		case NEXT :
			pgm_only();
			if(ctl_stk[ctl_ptr-1] != _FOR)
				error(6);
			i = ctl_stk[ctl_ptr-2];
			if(!is_l_end(skip_blank()))
				if(get_var() != i) error(6);
			jj = ctl_stk[ctl_ptr-3];	/* get limit */
			ii = ctl_stk[ctl_ptr-4];	/* get step */
			num_vars[i] += ii;
			if((ii < 0) ? num_vars[i] >= jj : num_vars[i] <= jj) {
				cmdptr = ctl_stk[ctl_ptr-5];
				runptr = ctl_stk[ctl_ptr-6];
				line = runptr->Lnumber; }
			else
				ctl_ptr -= 6;
			break;
		case IF :
			i = eval_num();
			expect(-128+THEN);
			if(i) {
				if(isdigit(cmd = skip_blank()))
					return find_line(eval_num());
				else if(cmd < 0) {
					++cmdptr;
					return execute(cmd); }
				else
					execute(1); }
			else
				skip_stmt();
			break;
		case LIF :
			i = eval_num();
			expect(-128+THEN);
			if(i) {
				if((cmd = skip_blank()) < 0) {
					++cmdptr;
					return execute(cmd); }
				else
					execute(1);
				break; }
		case DATA :
			pgm_only();
		case REM :
			if(mode) {
				if(cptr = runptr->Llink)
					return cptr;
				longjmp(savjmp, 1); }
			break;
		case STOP :
		stop:
			pgm_only();
			printf("STOP in line %u\n",line);
		case END :
			pgm_only();
			longjmp(savjmp, 1);
		case INPUT :
			key_test = 0;
			ii = chk_file(1);
			if(skip_blank() == '"') {		/* special prompt */
				eval();
				expect(','); }
			else
				strcpy(sa1, "? ");
			dptr = address();
			cptr = cmdptr;
input:		if(ii == -1)
				fputs(sa1, stdout);
			cmdptr = fgets(buffer, sizeof(buffer)-1, filein);
			if(break_flag) {
				if(mode) goto stop;
				return 0; }
			if(expr_type) {
				if(*dptr) free(*dptr);
				strcpy(*dptr = allocate(strlen(buffer)+1), buffer); }
			else {
				k = 0;
				if(test_next('-'))
					--k;
				if(!isdigit(*cmdptr)) {
					if(ii != -1) error(9);
					fputs("Input error\n",stdout);
					goto input; }
				j = get_dec();
				*dptr = (k) ? 0-j : j; }
			cmdptr = cptr;
			break;
		case OPEN :
			if(skip_blank() != '#') error(0);
			if(files[i = chk_file(0)]) error(8);
			eval_char();
			strcpy(buffer, sa1);
			expect(',');
			eval_char();
			files[i] = fopen(buffer,sa1);
			break;
		case CLOSE :
			if((i = chk_file(1)) == -1) error(0);
			if(!filein) error(8);
			fclose(files[i]);
			files[i] = 0;
			break;
		case DIM :
			do {
				if(dptr = dim_vars[i = get_var()])
					free(dptr);
				dim_vars[i] = allocate((dim_check[i] = eval_num()+1) * 2); }
			while(test_next(','));
			break;
		case ORDER :
			readptr = find_line(eval_num());
			dptr = cmdptr;
			cmdptr = readptr->Ltext;
			if(get_next() != -128+DATA)
				error(11);
			dataptr = cmdptr;
			cmdptr = dptr;
			break;
		case READ :
			do {
				dptr = address();
				j = expr_type;
				cptr = cmdptr;
				cmdptr = dataptr;
				ii = line;
				if(!skip_blank()) {		/* End of line */
					readptr = readptr->Llink;
					cmdptr = readptr->Ltext;
					if(get_next() != -128+DATA)
						error(11); }
				line = readptr->Lnumber;
				k = eval();
				test_next(',');
				dataptr = cmdptr;
				cmdptr = cptr;
				line = ii;
				if(j != expr_type)
					error(11);
				if(!expr_type)		/* numeric assignment */
					*dptr = k;
				else {				/* character assignment */
					if(*dptr)
						free(*dptr);
					if(*sa1)
						strcpy(*dptr = allocate(strlen(sa1)+1), sa1);
					else
						*dptr = 0; } }
			while(test_next(','));
			break;
		case PLAY :
			eval_char();
			i = BUTTONS + 1;
			j = 0;
			if(test_next(',')) {
				i = eval_num();
				if(test_next(','))
					j = eval_num(); }
			play_file(sa1, i, j);
			break;
		case TUNE :
			if((i = eval_num()) >= (sizeof(tunes)/sizeof(tunes[0])))
				error(13);
			play_tune(i);
			break;
		case CONTROL :
			if(!COM_port) error(14);
			i = eval();
			if(expr_type) {
				if(filein = fopen(sa1, "rvb")) {
					IR_top = fget(IR_buffer, sizeof(IR_buffer), filein);
					fclose(filein);
					break; }
				error(7); }
			if(i = transmit(i))
				error(i);
			break;
		case SCREEN :
			eval_char();
			if(filein = fopen(sa1, "rvb")) {
				i = 0;
				while(j = fget(buffer, sizeof(buffer), filein)) {
					copy_seg(video_seg, i, get_ds(), buffer, j);
					if((i += j) > 4000)
						break; }
				fclose(filein);
				break; }
			error(7);
		case SOUND :
			if(i = eval_num())
				sound(i);
			else
				sound_off();
			break;
		case SAY :
			do {
				i = eval();
				if(!expr_type)
					num_string(i, sa1);
				if(sbtalk(sa1))
					error(14); }
			while(test_next(','));
			break;
		case VOICE :
			i = eval_num();		/* Get tone */
			expect(',');
			j = eval_num();		/* Get volume */
			expect(',');
			k = eval_num();		/* Get pitch */
			expect(',');
			if(init_sbtalk(i, j, k, eval_num()))
				error(14);
			break;
		case MIX :
			i = eval_num();
			expect(',');
			j = k = eval_num();
			if(test_next(','))
				k = eval_num();
			if(i = mix_level(i, j, k))
				error(i);
			break;
		case DELAY :
			delay(eval_num());
			break;
		case BEEP :
			i = eval_num();
			expect(',');
			beep(i, eval_num());
			break;
		case DOS :
			eval_char();
			unload_ct_voice();
			system(sa1);
			load_ct_voice();
			break;
		case QDOS :
			eval_char();
			system(sa1);
			break;
		case OUT :
			i = eval_num();
			expect(',');
			out(i, eval_num());
			break;
		case CLS :
			i = 0x07;
			if(!is_l_end(skip_blank()))
				i = eval_num();
			i;
			asm " MOV BH,AL";
			asm " MOV AX,0600h";
			asm " XOR CX,CX";
			asm " MOV DX,1950h";
			asm " INT 10h";
			asm " XOR AX,AX";
			goto move_cursor;
		case GOTOXY :
			i = eval_num();
			expect(',');
			(eval_num() << 8) + i;
		move_cursor:
			asm " MOV DX,AX";
			asm " MOV AH,02h";
			asm " XOR BH,BH";
			asm " INT 10h";
			break;
		case PON :
			LPT_value |= eval_num();
			goto write_lpt;
		case POFF :
			LPT_value &= ~eval_num();
			goto write_lpt;
		case OSET :
			LPT_value = eval_num();
		write_lpt:
			debug("pout=%04x\n", i = LPT_value ^ LPT_mask);
			out(LPT_port, i);
			out(LPT_port+2, i >> 8);
			break;
		case SAVE :
			direct_only();
			if(skip_blank()) {
				eval_char();
				concat(filename, sa1, ".BAS"); }
			if(!*filename) error(7);
			if(fileout = fopen(filename, "wv")) {
				disp_pgm(fileout, 0, -1);
				fclose(fileout); }
			break;
		case LOAD :
			eval_char();
			concat(filename, sa1, ".BAS");
			if(filein = fopen(filename, "rv")) {
				if(!mode) clear_vars();
				clear_pgm();
				while(fgets(buffer, sizeof(buffer)-1, filein))
					edit_program();
				fclose(filein);
				return pgm_start; }
			longjmp(savjmp, 1);
		default :			/* unknown */
			error(0); }
		return 0;
}

/*
 * Test for file operator, and set up pointers
 */
int chk_file(char flag)
{
	unsigned i;

	i = -1;
	if(test_next('#')) {
		if(9 < (i = eval_num())) error(7);
		test_next(',');
		filein = fileout = files[i];
		if(flag && (!filein))
			error(7); }
	else {
		filein = stdin;
		fileout = stdout; }
	return i;
}

/*
 * Display program listing
 */
disp_pgm(FILE *fp, unsigned i, unsigned j)
{
	unsigned k;
	struct line_record *cptr;
	char c;

	for(cptr = pgm_start; cptr; cptr = cptr->Llink) {
		if(break_flag)
			return;
		k = cptr->Lnumber;
		if((k >= i) && (k <= j)) {
			fprintf(fp,"%u ",k);
			for(k=0; c = cptr->Ltext[k]; ++k)
				if(c < 0) {
					c = c & 127;
					fputs(reserved_words[c - 1], fp);
					if(c < ADD)
						putc(' ',fp); }
				else
					putc(c,fp);
			putc('\n', fp); } }
}

/*
 * Test for program only, and error if interactive
 */
pgm_only()
{
	if(!mode) error(2);
}

/*
 * Test for direct only, and error if running
 */
direct_only()
{
	if(mode) error(1);
}

/*
 * Skip rest of statement
 */
skip_stmt()
{
	char c;

	while((c=*cmdptr) && (c != ':')) {
		++cmdptr;
		if(c == '"') {
			while((c=*cmdptr) && (c != '"'))
				++cmdptr;
			if(c) ++cmdptr; } }
}

/*
 * Dislay error message
 */
error(unsigned en)
{
	printf("%s error", error_messages[en]);
	if(mode)
		printf(" in line %u", line);
	putc('\n',stdout);
	longjmp(savjmp, 1);
}

/*
 * Evaluate number only (no character)
 */
int eval_num()
{
	unsigned value;

	value = eval();
	if(expr_type)
		error(4);
	return value;
}

/*
 * Evaluate character only (no numeric)
 */
eval_char()
{
	eval();
	if(!expr_type)
		error(4);
}

/*
 * Evaluate an expression (numeric or character)
 */
int eval()
{
	unsigned value;

	nest = 0;
	value = eval_sub();
	if(nest != 1) error(0);
	return value;
}

/*
 * Evaluate a sub expression
 */
int eval_sub()
{
	unsigned value, nstack[10], nptr, optr;
	char c, ostack[10];

	++nest;								/* indicate we went down */

/* establish first entry on number and operator stacks */
	ostack[optr = nptr = 0] = 0;		/* add zero to init */

	nstack[++nptr] = get_value();		/* get next value */
/* string operations */
	if(expr_type) {						/* string operations */
		while(!is_e_end(c = skip_blank())) {
			++cmdptr;
			c &= 0x7F;
			get_char_value(sa2);
			if(c == ADD)				/* String concatination */
				strcat(sa1, sa2);
			else {
				if(c == EQ)				/* String EQUALS */
					value = !strcmp(sa1, sa2);
				else if(c == NE)		/* String NOT EQUALS */
					value = strcmp(sa1, sa2) != 0;
				else
					error(0);
				nstack[nptr] = value;
				expr_type = 0; } } }

/* numeric operations */
	else {
		while(!is_e_end(c = skip_blank())) {
			++cmdptr;
			c = (c & 0x7F) - (ADD-1);	/* 0 based priority table */
			if(priority[c] <= priority[ostack[optr]]) {	/* execute operand */
				value = nstack[nptr--];
				nstack[nptr] = do_arith(ostack[optr--], nstack[nptr], value); }
			nstack[++nptr] = get_value();		/* stack next operand */
			if(expr_type) error(0);
			ostack[++optr] = c; }
		while(optr) {				/* clean up all pending operations */
			value = nstack[nptr--];
			nstack[nptr] = do_arith(ostack[optr--], nstack[nptr], value); } }
	if(c == ')') {
		--nest;
		++cmdptr; }
	return nstack[nptr];
}

/*
 * Get a value element for an expression
 */
int get_value()
{
	unsigned value, v, *dptr, j[4];
	char c, *ptr;
	static unsigned year, month, day, hour, minute, second, tline;
	static unsigned char dow, dmask;

	expr_type = 0;
	if(isdigit(c = skip_blank()))
		value = get_dec();
	else {
		++cmdptr;
		switch(c) {
			case '(' :			/* nesting */
				return eval_sub();
			case '!' :			/* not */
				return ~get_value();
			case -128+SUB :	/* negate */
				return -get_value();
			case -128+ASC :	/* Convert character to number */
				eval_sub();
				if(!expr_type) error(4);
				expr_type = 0;
				return *sa1 & 255;
			case -128+NUM :	/* Convert string to number */
				eval_sub();
				if(!expr_type) error(4);
				expr_type = 0;
				return atoi(sa1);
			case -128+ABS :	/* Absolute value */
				if((value = eval_sub()) > 32767)
					value = -value;
				goto number_only;
			case -128+RND :	/* Random number */
				value = random(eval_sub());
				goto number_only;
			case -128+KEY :	/* Keyboard test */
				expect(')');
				value = key_test;
				key_test = 0;
				break;
			case -128+INP :	/* Read from port */
				value = in(eval_sub());
				goto number_only;
			case -128+LEN :	/* Get length of variable */
				v = get_var();
				expect(')');
				if(!expr_type)
					return dim_check[v];
				expr_type = 0;
				return (ptr = char_vars[v]) ? strlen(ptr) : 0;
			case -128+BUTTON :	/* Input button test */
				value = test_button(eval_sub());
				goto number_only;
			case -128+JOYSTK :	/* Joystick input */
				switch(value = eval_sub()) {
					default: error(10);
					case 0 : value = (joystick(j, -1) >> 4) ^ 0x0F;	break;
					case 1 :
					case 2 :
					case 3 :
					case 4 :
						v = 1 << (value - 1);
						if(joystick(j, ~v) & v)
							error(14);
						value = j[value-1]; }
			number_only:
				if(expr_type) error(4);
				break;
			case -128+TIME :		/* Time functions */
				v = (1 << (value = eval_sub())) & 0xFE;
				if(expr_type) error(4);
				if(!value)
					return get_ticks();
				if(tline != line) {
					dmask = 0;
					tline = line; }
				debug("dmask=%02x v=%02x ", dmask, v);
				if(dmask & v)
					dmask &= ~v;
				else {
					dow = get_date(&day, &month, &year);
					get_time(&hour, &minute, &second);
					dmask = ~v; }
				switch(value) {
					case 1 : return second;
					case 2 : return minute;
					case 3 : return hour;
					case 4 : return day;
					case 5 : return month;
					case 6 : return year;
					case 7 : return dow;
					case 8 : return (hour * 720) + (minute * 12) + (second/5);
					default: dmask=0; error(10); }
			case '$' :		/* Hex number */
				return get_hex();
			default:			/* test for character expression */
				--cmdptr;
				if(isalpha(c)) {		/* variable */
					value = get_var();
					if(expr_type) {		/* char */
						strcpy(sa1, (ptr = char_vars[value]) ? ptr : "");
						if(test_next('(')) {
							value = strlen(sa1);
							if((v = eval_sub()) >= value)
								error(10);
							expr_type = 1;
							*sa1 = sa1[v];
							sa1[1] = 0; } }
					else {
						if(test_next('(')) {	/* Array */
							if(!(dptr = dim_vars[value]))
								error(10);
							if((v = eval_sub()) >= dim_check[value])
								error(10);
							value = dptr[v]; }
						else						/* Std variable */
							value = num_vars[value]; } }
				else
					get_char_value(sa1); } }
	return value;
}

/*
 * Get character value
 */
get_char_value(char *ptr)
{
	unsigned i;
	char c, *st;

	if((c = get_next()) == '"') {	/* character value */
		while((c = *cmdptr++) != '"') {
			if(!c) error(0);
			*ptr++ = c; }
		*ptr = 0; }
	else if(isalpha(c)) {			/* variable */
		--cmdptr;
		i = get_var();
		if(!expr_type)
			error(0);
		if(st = char_vars[i])
			strcpy(ptr,st);
		else
			strcpy(ptr,""); }
	else if(c == -128+CHR) {		/* Convert number to character */
		*ptr++ = eval_sub();
		if(expr_type)
			error(4);
		*ptr = 0; }
	else if(c == -128+STR) {		/* Convert number to string */
		num_string(eval_sub(), ptr);
		if(expr_type)
			error(4); }
	else
		error(0);
	expr_type = 1;
}

/*
 * Perform an arithmetic operation
 */
int do_arith(char opr, unsigned op1, unsigned op2)
{
	unsigned value;

	switch(opr) {
		case ADD-(ADD-1) :		/* addition */
			value = op1 + op2;
			break;
		case SUB-(ADD-1) :		/* subtraction */
			value = op1 - op2;
			break;
		case MUL-(ADD-1) :		/* multiplication */
			value = op1 * op2;
			break;
		case DIV-(ADD-1) :		/* division */
			value = op1 / op2;
			break;
		case MOD-(ADD-1) :		/* modulus */
			value = op1 % op2;
			break;
		case AND-(ADD-1) :		/* logical and */
			value = op1 & op2;
			break;
		case OR-(ADD-1) :		/* logical or */
			value = op1 | op2;
			break;
		case XOR-(ADD-1) :		/* exclusive or */
			value = op1 ^ op2;
			break;
		case EQ-(ADD-1) :		/* equals */
			value = op1 == op2;
			break;
		case NE-(ADD-1) :		/* not-equals */
			value = op1 != op2;
			break;
		case LE-(ADD-1) :		/* less than or equal to */
			value = op1 <= op2;
			break;
		case LT-(ADD-1) :		/* less than */
			value = op1 < op2;
			break;
		case GE-(ADD-1) :		/* greater than or equal to */
			value = op1 >= op2;
			break;
		case GT-(ADD-1) :		/* greater than */
			value = op1 > op2;
			break;
		default:
			error(0); }
	return value;
}

/*
 * Convert a number to a string, and place in memory
 */
num_string(unsigned value, char *ptr)
{
	char cstack[5], cptr;

	cptr = 0;

	if(value > 32767) {
		*ptr++ = '-';
		value = -value; }
	do
		cstack[cptr++] = (value % 10) + '0';
	while(value /= 10);
	while(cptr)
		*ptr++ = cstack[--cptr];
	*ptr = 0;
}

/*
 * Clear program completely
 */
clear_pgm()
{
	for(runptr = pgm_start; runptr; runptr = runptr->Llink)
		free(runptr);
	pgm_start = 0;
}

/*
 * Clear all variables to zero
 */
clear_vars()
{
	unsigned i;
	char *ptr;

	for(i=0; i < NUM_VAR; ++i) {
		num_vars[i] = 0;
		if(ptr = char_vars[i]) {	/* Character variables */
			free(ptr);
			char_vars[i] = 0; }
		if(ptr = dim_vars[i]) {		/* Dimensioned arrays */
			free(ptr);
			dim_vars[i] = 0; } }
}

/*
 * Get index for variable from its name
 */
int get_var()
{
	unsigned index;
	char c;

	if(!isalpha(c = get_next()))
		error(0);
	index = ((c - 'A') & 0x1F) * 10;
	if(isdigit(c = *cmdptr)) {
		index += (c - '0');
		c = *++cmdptr; }
	if(c == '$') {
		++cmdptr;
		expr_type = 1; }
	else
		expr_type = 0;

	return index;
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	int i, j;

	pgm_start = j = 0;
	RAND_SEED = get_ticks();
	for(i=1; i < argc; ++i) {
		cmdptr = dataptr = argv[i];
		switch((toupper(*cmdptr++)<<8) | toupper(*cmdptr++)) {
			case '/B' : debounce_count = get_dec();	break;
			case '/D' :	debug_flag = -1; 			break;
			case '/I' : COM_port = get_dec();		break;
			case '/K' : KPD_port = get_hex();		break;
			case '/P' : LPT_port = get_hex();
				if(*cmdptr++ == ',')
					LPT_mask = get_hex();
				break;
			case '/S' : RAND_SEED = get_dec();		break;
			case '/?' :
			case '?'<<8 :
				abort("\nUse: CAM [/Bcount /D /Icom /Kadr /Padr,inv /Sseed] [program [parms]]\n");
			default:
				char_vars[j++] = dataptr; } }

	/* Get the video segment */
	asm " MOV AH,0Fh";
	asm " INT 10h";
	asm " CMP AL,07h";
	asm " JNE nomono";
	asm " MOV word ptr DGRP:_video_seg,0B000h";
	asm "nomono: MOV AX,OFFSET DGRP:?heap";
	i = nargs();

	/* Read in IR key definition file */
	if(COM_port) {
		/* Open COMM port for binary I/O activity */
		if(Copen(COM_port, _9600, PAR_NO|DATA_8|STOP_1,
			SET_DTR|SET_RTS|OUTPUT_2))
			abort("Cannot open comm port");
		Cflags |= TRANSPARENT; }

	debug("Seed=%u Top=%04x\n", RAND_SEED, i);
	load_ct_voice();

/*
 * If a name is given on the command line, load it as a program and
 * run immediately. If the program does not explicitly EXIT, we will
 * then proceed to an interactive session
 */
	if(j) {
		concat(filename, char_vars[0], ".BAS");
		if(filein = fopen(filename, "rv")) {
			while(fgets(buffer, sizeof(buffer)-1, filein))
				edit_program();
			fclose(filein);
			if(!setjmp(savjmp))
				execute(RUN1);
			return; } }

/*
 * Display header AFTER running command line, so that programs run as
 * "batch" commands terminating with EXIT do not get "noise" output.
 */
	printf(HEADER);
	asm {
		PUSH	DS			; Save DS
		MOV		AX,CS		; Get code segment
		MOV		DS,AX		; save it
		MOV		DX,OFFSET cch;
		MOV		AX,2523h	; Set interrupt
		INT		21h			; Transfer to MS-DOS
	}

	setjmp(savjmp);
	for(;;) {						/* Main interactive loop */
		if(break_flag)
			while(kbtst());
		fputs("Ready\n", stdout);
	noprompt: mode = break_flag = ctl_ptr = key_test = 0;
		fgets(buffer, sizeof(buffer)-1, stdin);
		if(edit_program())			/* Tokenize & edit if OK */
			goto noprompt;
		if((i = *cmdptr) < 0) {		/* Keyword, execute command */
			++cmdptr;
			execute(i); }
		else if(i)					/* Unknown, assume LET */
			execute(LET); }
}

asm {
cch:	PUSH	DS			; Save DS
; Patch CAM break flag
		PUSH	CS			; Get CS
		POP		DS			; Set new DS
		MOV		DGRP:_break_flag,-1
; Clear keyboard buffer & stuff a single ENTER key
		PUSH	BX			; Save BX
		MOV		BX,40h		; Address BIOS segment
		MOV		DS,BX		; Set DS
		MOV		BX,001Ah	; Point to table
		CLI					; Disable ints
		MOV		[BX],WORD PTR 001Eh
		MOV		2[BX],WORD PTR 0020h
		MOV		4[BX],WORD PTR 1C0Dh
		STI					; Enable ints
		POP		BX			; Restore BX
; Restore DS and return
		POP		DS			; Restore DS
		IRET
}
