/*
 * FIND command for MS-DOS
 *
 * This version of FIND is similar to the original MS-DOS FIND command
 * except that it allows multiple search strings and filenames, allows
 * wildcards in the search filenames, will search subdirectories,  and
 * has many other additional options and features.
 *
 * ?COPY.TXT 1998-2003 Dave Dunfield
 * Freely distributable.
 *
 * Compile command: cc find -fop
 */
#include <stdio.h>
#include <file.h>
// #define PATHDEBUG			/* Debug path/pattern detection logic */

#define	MAXPATH		80			/* Maximum size of a pathname */
#define	MAXPAT		25			/* Maximum number of patterns */
#define	MAXFILE		25			/* Maximum number of files */
#define	LINE_SIZE	2000		/* Maximum length of line */
#define	ALLFILES	0x3F		/* findfirst mask for all files */
#define	MAXSTACK	1820		/* Maximum # stacked filenames */

/* DOS return codes */
#define	RC_NOFILE	2			/* File not found */
#define	RC_NODIR	3			/* Directory not found */
#define	RC_NOMORE	18			/* No more files */

extern int ARGC;				/* Global access to argc */
extern char *ARGV[];			/* Global access to argv */

/*
 * Structure of file/directory information block
 */
struct FI_block {
	char		FI_attrib;			/* file Attributes */
	unsigned	FI_time;			/* file Time stamp */
	unsigned	FI_date;			/* file Date stamp */
	char		FI_name[13]; };		/* Name of file */

/*
 * Source/Dest path/pattern variables
 */
char
	source[MAXPATH+1],		/* Source path */
	*source_ptr,			/* Pointer to current level along source path */
	sourcefile[13],			/* Source file matching pattern */
	*patterns[MAXPAT],		/* Patterns to look for */
	*filenames[MAXFILE];	/* Filenames to search */

/*
 * Command line options
 */
char
	icase,			/* Ignore case in search */
	notfnd,			/* Display lines where string is NOT found */
	lcount,			/* Display count of lines only */
	lnum,			/* List line numbers */
	subdir,			/* Search subdirectories */
	hidden,			/* Include hidden and system files */
	date,			/* Search by date: (1=After, 2=Before) */
	econt;			/* Continue, even if errors */

/*
 * Misc. global variables
 */
unsigned
	xdate,			/* Date for date operations */
	pcount,			/* Count of patterns */
	fcount,			/* Count of files */
	rc,				/* Return code for DOS operations */
	tmatch[MAXPAT];	/* Total match table */
FILE
	*fp;			/* Open file to search */

/*
 * Stack to hold processed file/dir names
 * Filenames build up from the bottom, dir names grow down from the top
 */
struct FI_block file_stack[MAXSTACK];
unsigned sfile, sdir = MAXSTACK;

/*
 * Help text
 */
char help[] = { "Search for text string(s) in a file or files.\n\n\
FIND \"string\"... [drive:path\\filespec...] [options]\n\n\
 /C          Display only the count of lines containing the string(s)\n\
 /Ddd/mm/yy  Search files changed on or after this date (default: today)\n\
 /D-dd/mm/yy Search files changed before or on this date\n\
 /H          Include hidden/system files in the search\n\
 /I          Ignore the case of characters when searching\n\
 /K          Keep going, even if errors occur\n\
 /N          Display line numbers preceeding the text\n\
 /S          Search subdirectories\n\
 /V          Display lines NOT containing at least one search string\n\
 \"string\"    Specifies the text string(s) to find\n\
 drive:path\\filespec  Specifies file(s) to search\n\
\nIf no pathnames are specified, FIND searches the standard input stream\n\
\n?COPY.TXT 1998-2003 Dave Dunfield - Freely distributable.\n" };

/*
 * Re-parse the command line arguments to obtain values in quotations
 * and place them in the pattern stack.
 */
parse_quoted_args()
{
	unsigned char *ptr, *aptr;
	unsigned l;
	char c, flag;

	ptr = 0x80;
	aptr = ARGV[ARGC=1];
	l = *ptr++;
	flag = 0;
	while(l--) {
		c = *ptr++;
		if(!flag) {		/* Looking for next argument */
			if(isspace(c))
				continue;
			if(c == '"') {	/* Quoted args go directly into patterns */
				flag = 2;
				patterns[pcount++] = aptr;
				continue; }
			ARGV[ARGC++] = aptr;
			flag = 1; }
		switch(c) {
			case '"' :
				if(flag != 2)		/* Not processing within quotes */
					goto norchar;
				if(*ptr == '"') {	/* Double quotes */
					++ptr;
					--l;
					goto norchar; }
			case 0x0D :
				flag = 1;
			case ' ' :
			case '\t' :
				if(flag == 1) {
					*aptr++ = flag = 0;
					continue; }
			default:
			norchar:
				*aptr++ = c; } }
	*aptr = 0;
	return ARGC;
}

/*
 * Parse an option string, allowing multiple options within one string
 */
parse_option(char *ptr)
{
	unsigned y, m, d;

	while(*ptr) {
		if(*ptr++ != '/')
			abort("Invalid switch");
		switch(toupper(*ptr++)) {
			case 'C' : lcount = -1;	continue;
			case 'H' : hidden = -1;	continue;
			case 'I' : icase = -1;	continue;
			case 'K' : econt = -1;	continue;
			case 'N' : lnum = -1;	continue;
			case 'S' : subdir = -1;	continue;
			case 'V' : notfnd = -1;	continue;
			case 'D' :
				get_date(&d, &m, &y);
				date = 1;
				if(*ptr == '-') { date = 2; ++ptr; }
				if(isdigit(*ptr)) {
					d = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(*ptr++ != '/') abort("Invalid date");
					m = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(*ptr++ != '/') abort("Invalid date");
					y = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(y < 80)
						y += 2000;
					else if(y < 100)
						y += 1900; }
				xdate = ((y-1980) << 9) | (m << 5) | d;
				continue;
			case '?' : abort(help); }

		abort("Invalid switch"); }
}

/*
 * Report an error
 */
error(char *string)
{
	char f;
	f = *string++;
	printf("Error : %s", string);
	if(f & 0x04) printf(", DosErrorCode=%u", rc);
	putc('\n', stdout);
	if(f & 0x01) stpcpy(source_ptr, sourcefile);
	if(f & 0x01) printf("\t%s\n", source);
	if((f & 0x08) || !econt) exit(rc);
}

/*
 * Test for a string containing wildcards
 */
iswild(char *p)
{
	while(*p) switch(*p++) {
		case '*' :
		case '?' :
			return -1; }
	return 0;
}

/*
 * Match filename to source file matching pattern
 */
fmatch(char *f)
{
	char *p, c;
	p = sourcefile;
	for(;;) switch(c = *p++) {
		case '?' :		/* Match (skip) one character from filename */
			if((*f != '.') && *f)
				++f;
			break;
		case '*' :		/* Match (skip) remaining characters to . or end */
			if(!*p)
				return 1;
			while((*f != '.') && *f)
				++f;
			continue;
		case 0 :		/* End, match only if also end of filename */
			return !*f;
		case '.' :		/* Don't count '.' if it is at the end */
			if(!*f) continue;
		default:		/* Character in filename must match */
			if(c != *f++)
				return 0; }
}

/*
 * Put an entry on the file information stack
 */
stack(unsigned entry, struct FF_block *f)
{
	struct FI_block *d;
	d = file_stack[entry];
	stpcpy(d->FI_name, f->FF_name);
	d->FI_attrib = f->FF_attrib;
	d->FI_time = f->FF_time;
	d->FI_date = f->FF_date;
}

/*
 * Rapid search of a line for a substring
 */
search_line(line, text) asm
{
		XOR		AH,AH			; Zero high
		MOV		CH,AH			; Zero Chigh
		MOV		CL,DGRP:_icase	; Get "ignore case" flag
		MOV		SI,6[BP]		; Get line pointer
sl1:	MOV		DI,4[BP]		; Get string pointer
sl2:	MOV		AL,[SI]			; Get char from line
		AND		AL,AL			; End of line?
		JZ		sl5				; Yes, exit
		JCXZ	sl21			; No ignore case
		CMP		AL,'a' 			; In range?
		JB		sl21			; No, leave alone
		CMP		AL,'z'			; In range?
		JA		sl21			; No, leave alone
		AND		AL,0DFh			; convert to upper case
sl21:	CMP		AL,[DI]			; Does it match?
		LEA		SI,1[SI]		; Skip to next
		JNZ		sl2				; Keep looking
		MOV		BX,SI			; Save source position
; We found a character
sl3:	INC		DI				; Next in destination
		MOV		AL,[DI]			; Get char from dest
		AND		AL,AL			; End of string?
		JZ		sl4				; Yes, we found it
		MOV		AL,[SI]			; Get char from source
		JCXZ	sl31			; No ignore case
		CMP		AL,'a' 			; In range?
		JB		sl31			; No, leave alone
		CMP		AL,'z'			; In range?
		JA		sl31			; No, leave alone
		AND		AL,0DFh			; convert to upper case
sl31:	CMP		AL,[DI]			; Match source?
		LEA		SI,1[SI]		; Skip to next
		JZ		sl3				; Still on target
		MOV		SI,BX			; Reset source position
		JMP SHORT sl1			; and proceed
sl4:	DEC		AX				; Indicate found
sl5:
}

/*
 * Search an individual file
 */
search_file()
{
	unsigned i, line, matches[MAXPAT];
	char buffer[LINE_SIZE], flag;

	memset(matches, 0, sizeof(matches));

	line = 0;
	while(fgets(buffer, sizeof(buffer)-1, fp)) {
		++line;
		i = flag = 0;
		do {
			if(search_line(buffer, patterns[i])) {
				++matches[i];
				flag = -1; } }
		while(++i < pcount);
		if(notfnd ? !flag : flag) {
			if(!lcount) {
				if(lnum) printf("[%u]", line);
				fputs(buffer, stdout);
				putc('\n', stdout); } } }
	fclose(fp);
	i=0;
	do {
		tmatch[i] += matches[i];
		if(lcount)
			printf(" %u", matches[i]); }
	while(++i < pcount);
	if(lcount)
		putc('\n', stdout);
}

/*
 * Search a directory and all subdirectories (if requested)
 */
search_dir()
{
	int i;
	struct FF_block f;
	struct FI_block *fi;
	unsigned sdir_hold;
	char *s;
	static unsigned level = 0;

	sdir_hold = sdir;
	stpcpy(source_ptr, "*.*");
	sfile = 0;

	/*
	 * See of any files exist
	 */
	if(rc = findfirst(source, f, ALLFILES)) {
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			error("\x15Unable to access source path");
			return; }
		if(!level)
			error("\x15File not found!"); }
	else do {
		if(f.FF_attrib & VOLUME) continue;
		if(*f.FF_name == '.') continue;
		if(f.FF_attrib & DIRECTORY) {
			if(subdir)
				stack(--sdir, f);
			continue; }
		if(!fmatch(f.FF_name)) continue;
		if((f.FF_attrib & (HIDDEN|SYSTEM)) && !hidden) continue;
		if(date) {
			if(date == 2) {
				if(f.FF_date > xdate) continue; }
			else {
				if(f.FF_date < xdate) continue; } }
		stack(sfile++, f); }
	while(!findnext(f));

	/* Report an error if no matching files and not copying subdirs */
	if((!level) && (sdir == sdir_hold) && !sfile) {
		error("\x11File not found!");
		return; }

	/* Search all the files that we found */
	for(i=0; i < sfile; ++i) {
		fi = file_stack[i];
		stpcpy(source_ptr, fi->FI_name);
		if(!(fp = fopen(source, "rv"))) {
			error("\x05Unable to READ file");
			continue; }
		printf("---------- %s", source);
		putc(lcount ? ':' : '\n', stdout);
		search_file(); }

	/* Search subdirectories by recursive call for each directory */
	i = sdir_hold;
	s = source_ptr;
	++level;
	while(i > sdir) {
		fi = file_stack[--i];
		source_ptr = stpcpy(stpcpy(s, fi->FI_name), "\\");
		search_dir(); }
	--level;
	source_ptr = s;
	sdir = sdir_hold;
}

/*
 * Main program - Parse options and launch the search engine
 */
main(int argc, char *argv[])
{
	int i;
	char *ptr, *save_arg;
	struct FF_block f;

	argc = parse_quoted_args();
	/* Parse command options */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		if(*ptr == '/') {
			parse_option(ptr);
			continue; }
		filenames[fcount++] = argv[i]; }

	if(!pcount)
		abort(help);

	IOB_size = 4096;	/* Speed up file reads with larger buffer */

	/* If ignoring case, convert all patterns to upper case */
	if(icase) for(i=0; i < pcount; ++i)
		strupr(patterns[i]);

	if(!fcount) {
		fp = stdin;
		search_file();
		return; }

	for(i=0; i < fcount; ++i) {
		save_arg = source_ptr = stpcpy(source, filenames[i]);
		/*
		 * Determine source path and file matching pattern
		 */
		strupr(source);		/* Insure uppercase for pattern matches */
		switch(*(source_ptr-1)) {
			default:		/* Unknown, lets look for it */
				/* Lookup file and determine what it is */
				if(findfirst(source, f, ALLFILES))
					break;
				/* Must be a file, as it is not a directry */
				if(!(f.FF_attrib & DIRECTORY))
					break;
				/* If source path contains wildcards, it cannot be a directory */
				if(iswild(source))
					break;
				*source_ptr++ = '\\';	/* Append trailing '\' to directory */
			case '\\' :		/* Ends with '\', must be a directory */
			case ':' :		/* Ends with ':', must be a directory */
				stpcpy(source_ptr, "*.*"); }
		/* Pull off last entry as file match pattern */
		while(source_ptr > source) {
			switch(*(source_ptr-1)) {
				case '\\' :		/* We found last entry, copy to pattern */
				case ':' :		/* Then truncate to end at '\' or ':' */
					stpcpy(sourcefile, source_ptr);
					*source_ptr = 0;
					goto source_ok; }
			--source_ptr; }
		/* This was the only entry, assume current dir & argument is pattern */
		stpcpy(sourcefile, source);
		*source = 0;
	source_ok:

#ifdef PATHDEBUG
		printf("Source: %s\n", source);
		printf("Sourcefile: %s\n", sourcefile);
		return;
#endif

		search_dir();
		*save_arg = 0; }

	printf("Total matches:");
	for(i=0; i < pcount; ++i)
		printf(" %u", tmatch[i]);
	putc('\n', stdout);
}
//01234 26/11/2019
