#include <stdio.h>
#include <window.h>
#include "help.h"

/* Fixed editor parameters */
#define	LINE_SIZE	512			/* Mex length of input line */
#define	CMD_SIZE	80			/* Maximum size of command line */
#define	LINES		25			/* Number of lines on screen */
#define	COLS		80			/* Number of columns on screen */
#define	EDIT_START	1			/* Starting offset of edit buffer */

/* Command key functions */
#define	BLI			-128		/* 000 Back line */
#define	FLI			-127		/* 001 Fwd line */
#define	BCH			-126		/* 002 Back char */
#define	FCH			-125		/* 003 Fwd char */
#define	BPG			-124		/* 004 Back page */
#define	FPG			-123		/* 005 Fwd page */
#define	SOL			-122		/* 006 Start of line */
#define	EOL			-121		/* 007 End of line */
#define	CMD			-120		/* 008 Enter command */
#define	REP			-119		/* 009 Repeat command */
#define	INS			-118		/* 010 Insert toggle */
#define	DEL			-117		/* 011 Delete character */
#define	BSD			-116		/* 012 Backspace & delete */
#define	ELD			-115		/* 013 End of line display */
#define	POS			-114		/* 014 Position information */
#define	TOP			-113		/* 015 Current line to top */
#define	TAG			-112		/* 016 Tag lines */
#define	DLI			-111		/* 017 Delete line inclusive */
#define	DLE			-110		/* 018 Delete line exclusive */
#define	IDL			-109		/* 019 Insert deleted line */
#define	FUN			-108		/* 020 Function menu */
#define	SOF			-107		/* 021 Start of file */
#define	ENF			-106		/* 022 End of file */
#define	BWR			-105		/* 023 Back word */
#define	FWR			-104		/* 024 Forward word */
#define	HLP			-103		/* 025 Help command */
#define	XXX			-102		/* 026 De-assigned key */
#define	CMDS		0x009B		/* 027 User commands */

/*
 * Function key translation table
 */
char edit_keys[] = {
/* ^A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S, T, U, V, W, X, Y, Z */
	1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,

/*	ESC, KUA, KDA, KLA, KRA, KPU, KPD, KHO, KEN, KKP, KKM, KIN, KDL, KBS */
	27,  BLI, FLI, BCH, FCH, BPG, FPG, SOL, EOL, CMD, REP, INS, DEL, BSD,

/*	FN1, FN2, FN3, FN4, FN5, FN6, FN7, FN8, FN9, FN10 */
	HLP, POS, ELD, TAG, DLI, DLE, IDL, FUN, REP, CMD,

/*	CPU, CPD, CHO, CEN, CLA, CRA */
	SOF, ENF, TOP, 0,   BWR, FWR,

/* !A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S, T, U, V, W, X, Y, Z */
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
	};

char
	*edit_key_cmds[25];			/* Edit user command table */

int
	cursor_x,					/* Current cursor X position */
	cursor_y,					/* Current cursor Y position */
	edlen,						/* Length of line in edit buffer */
	edit_length,				/* Length of original edit line */
	delete_length;				/* Length of deleted text */

unsigned
	horz,						/* Actual cursor X position */
	left_margin,				/* Position of left side of screen */
	edit_seg,					/* Edit buffer segment */
	edit_end = EDIT_START,		/* End address of edit buffer */
	edit_line = EDIT_START,		/* Position of line to edit */
	newpos,						/* New position to go to */
	screen_top = EDIT_START,	/* Position of current screen */
	tag_low, tag_high,			/* Locations of the tagged lines */
	tab_width = 4;				/* Current tab stops */

char
	edit_buffer[LINE_SIZE+1],	/* Line under cursor resides here */
	delete_buffer[LINE_SIZE+1],	/* Holds last line deleted */
	command[CMD_SIZE+1],		/* Holds line mode command */
	changed,					/* Indicates line has changed */
	CHANGED,					/* Changed flag */
	eol_display = 0,			/* End of line display flag */
	insert_flag = 1,			/* Disable inserting */
	video,						/* Indicate that video is reversed */
	refresh,					/* Screen refresh required */
	*errmsg,					/* Pointer to error message */
	hilite;						/* Hilited character data */

extern char
	filename[65],				/* Name of file being edited */
	*key_ptr;					/* Auto-key pointer */

extern unsigned char			/* Video attributes */
	Vedt_txt,					/* Normal text */
	Vedt_eof,					/* EOF marker */
	Vedt_msg,					/* Informational messages */
	Vedt_err;					/* Error messages */

extern struct WINDOW *mwin, *twin;

extern register title();

/*
 * Load and display the edit file
 */
load_edit_file(char *file)
{
	FILE *fp;
	char buffer[2048];
	unsigned size, pos;

	errmsg = 0;
	if((fp = openf(file, "r")) || (errmsg && (*errmsg == 'U'))) {
		strcpy(filename, file);
		screen_top = edit_end = edit_line = pos = EDIT_START;
		cursor_x = cursor_y = horz = video = refresh =
		left_margin = tag_low = tag_high = newpos = 0;
		if(fp) {
			while(size = fget(buffer, sizeof(buffer), fp)) {
				if(!einsert(pos, size))
					break;
				copy_seg(edit_seg, pos, get_ds(), buffer, size);
				pos += size; }
			fclose(fp); }
		else
			errmsg = ":New file: %s";
		CHANGED = changed = 0; }
}

/*
 * --- Screen Display functions ---
 */

/*
 * Display a character if within screen boundary
 */
display_char(unsigned char c)
{
	if(c < ' ') {
		*W_OPEN = (video) ? Vedt_txt : hilite;
		wputc(c + '@');
		*W_OPEN = (video) ? hilite : Vedt_txt; }
	else
		wputc(c);
	++horz;
}

/*
 * Display a line of text
 */
display_line(char *line)
{
	unsigned eol;

	eol = left_margin + W_OPEN->WINwidth;

	if((edit_line >= tag_low) && (edit_line <= tag_high)) {
		video = -1;
		*W_OPEN = hilite; }

	/* Skip any data which preceeds the horizontal scrolling window */
	while((horz < left_margin) && *line) {
		if(*line++ == '\t')
			while(++horz % tab_width);
		else
			++horz; }

	/* Output any data which is within the horizontal scrolling window */
	if(horz >= left_margin) {
		while(horz < eol) {
			if(!*line) {		/* End of line */
				if(eol_display)
					display_char('\n');
				break; }
			if(*line == '\t') {
				do
					wputc(' ');
				while((horz < eol) && (++horz % tab_width)); }
			else
				display_char(*line);
			++line; } }

	video = 0;
	*W_OPEN = Vedt_txt;		/* No special effects */

	/* If not at end, clear to end of line */
	if(horz < eol)
		wcleol();
}

/*
 * Display a screen 
 */
display_screen(unsigned text_ptr, unsigned line)
{
	unsigned save_horz, save_edit;
	char buffer[LINE_SIZE+1];

	save_horz = horz;
	save_edit = edit_line;
	edit_line = text_ptr;
	while(line < W_OPEN->WINheight) {
		wgotoxy(horz = 0, line++);
		if(text_ptr >= edit_end) {
			*W_OPEN = Vedt_eof;
			wputs("*END OF FILE*");
			*W_OPEN = Vedt_txt;
			wcleow();
			break; }
		text_ptr = eget_line(buffer, edit_line);
		display_line(buffer);
		edit_line = text_ptr; }
	horz = save_horz;
	edit_line = save_edit;
}

/*
 * Function to position the cursor at the right location
 * Return actual horizontal offset into line.
 */
int position_cursor()
{
	char *tmptr;
	int i;

	horz = 0;

	tmptr = edit_buffer;
	for(i=0; (i < cursor_x) && (i < edlen); ++i) {
		++horz;
		if(*tmptr++ == '\t')
			while(horz % tab_width)
				++horz; }

	if(horz < left_margin) {								/* Scroll right */
		update_changes();
		while(horz < left_margin)
			left_margin -= (W_OPEN->WINwidth/2);
		refresh = -1; }
	else while(horz >= (left_margin+W_OPEN->WINwidth)) {	/* Scroll left */
		update_changes();
		left_margin += (W_OPEN->WINwidth/2);
		refresh = -1; }

	return i;
}

/*
 * Reposition the cursor to the desired line
 */
reposition(unsigned addr)
{
	unsigned ptr;

	/* Calculate new 'Y' address */
	cursor_y = changed = 0;
	ptr = screen_top;
	do {
		if(ptr >= addr)
			break;
		edit_line = eget_line(edit_buffer, ptr);
		if(edit_line > addr)
			break;
		if(++cursor_y > W_OPEN->WINheight-1) { /* ???-2 */
			screen_top = -1;
			break; } }
	while((ptr = edit_line) < edit_end);

	/* Calculate new 'X' address */
	cursor_x = 0;
	edit_line = addr;
	while((edit_line > EDIT_START) && (peek(edit_seg, edit_line-1) != '\n')) {
		--edit_line;
		++cursor_x; }

	if(addr < screen_top) {
		screen_top = edit_line;
		cursor_y = 0;
		refresh = -1; }
}

/*
 * --- Text editing functions ---
 */

/*
 * Update any pending changes to the current line
 */
update_changes()
{
	if(changed > 1) {
		CHANGED = -1;
		edit_buffer[edlen++] = '\n';
		if(edit_length > edlen)
			edelete(edit_line, edit_length - edlen);
		else if(edit_length < edlen)
			if(!einsert(edit_line, edlen - edit_length))
				goto no_update;
		copy_seg(edit_seg, edit_line, get_ds(), edit_buffer, edit_length = edlen);
	no_update:
		edit_buffer[--edlen] = 0; }
	changed = 0;
}

/*
 * Move forward one character
 */
fwd_chr()
{
	if(cursor_x < edlen)
		++cursor_x;
	else if(fwd_line())
		cursor_x = 0;
}

/*
 * Move forward one line
 */
fwd_line()
{
	update_changes();
	if(edit_line >= edit_end)
		return 0;

	edit_line = eget_line(edit_buffer, edit_line);
	if(cursor_y < W_OPEN->WINheight-1) /* ???-2 */
		++cursor_y;
	else {
		screen_top = eget_line(edit_buffer, screen_top);
		--refresh; }
	return -1;
}

/*
 * Move forward one page
 */
fwd_page()
{
	unsigned i;

	update_changes();
	for(i=1; i < W_OPEN->WINheight; ++i)
		screen_top = eget_line(edit_buffer, screen_top);
	edit_line = screen_top;
	cursor_y = 0;
	refresh = -1;
}

/*
 * Move back one character
 */
back_chr()
{
	if(--cursor_x < 0) {
		cursor_x = 32767;
		back_line(); }
}

/*
 * Move back one line
 */
back_line()
{
	update_changes();
	--edit_line;
	while((edit_line >= EDIT_START) && (peek(edit_seg, --edit_line) != '\n'));
	++edit_line;
	if(--cursor_y < 0) {
		screen_top = edit_line;
		cursor_y = 0;
		refresh = -1; }
}

/*
 * Move back one page
 */
back_page(unsigned lines)
{
	unsigned i;

	update_changes();
	for(i=0; i < lines; ++i)
		while((screen_top >= EDIT_START) && (peek(edit_seg, --screen_top) != '\n'));
	edit_line = ++screen_top;
	cursor_y = 0;
	refresh = -1;
}

/*
 * Display information about cursor
 */
get_info()
{
	unsigned line, lcount;
	char buffer[LINE_SIZE+1];

	lcount = 0;
	line = EDIT_START;
	while(line < edit_line) {
		line = eget_line(buffer, line);
		++lcount; }
	w_gotoxy(0, 0, twin);
	w_printf(twin, "Line: %u, Column: %u, Char: %u  (%u of %u in file)",
		lcount+1, horz+1, cursor_x+1, line+cursor_x, edit_end);
	w_cleol(twin);
}

/*
 * Get a line of input
 */
get_input(char *prompt, char *dest, unsigned length)
{
	unsigned i;
	struct WINDOW *save;
	char c;

	i = 0;
	save = W_OPEN;
	W_OPEN = twin;
	wclwin();
	wputs(prompt);
	do {
		if((c = egetc()) == BSD) {
			if(i) {
				wputs("\b \b");
				--i; } }
		if((c >= 0) && (i < length)) {
			if((dest[i++] = c) < ' ') {
				*W_OPEN = (Vedt_msg >> 4) | (Vedt_msg << 4);
				c += '@'; }
			wputc(c);
			*W_OPEN = Vedt_msg; } }
	while(c != CMD);
	dest[i] = 0;
	W_OPEN = save;
	return i;
}

/*
 * --- Edit buffer functions ---
 */
eget_line(dest, source) asm
{
		MOV		SI,4[BP]			; Get source offset
		MOV		DI,6[BP]			; Get dest offset
		MOV		ES,DGRP:_edit_seg	; Edit segment
egetl1:	CMP		SI,DGRP:_edit_end	; Are we at end?
		JAE		egetl2				; Yes, quit
		MOV		AL,ES:[SI]			; Get byte from source
		INC		SI					; Advance
		MOV		[DI],AL				; Write to dest
		INC		DI					; Advance
		CMP		AL,0Ah				; End of line?
		JNZ		egetl1				; Copy them all
		DEC		DI					; Backup
egetl2:	XOR		AL,AL				; Get zero
		MOV		[DI],AL				; Zero terminate
		MOV		AX,SI				; Get address
}

/*
 * Delete a number of characters from the edit buffer
 */
edelete(unsigned pos, unsigned size)
{
	if(pos < edit_line) {	/* Adjust pointers if deleting below */
/*		if((edit_line -= size) < pos)
			edit_line = pos; */
		if((pos + size) > edit_line)
			edit_line = pos;
		else
			edit_line -= size;
		newpos = edit_line; }
	if(pos < screen_top) {
/*		if((screen_top -= size) < pos) {
			screen_top = pos;
			newpos = edit_line;
			refresh = -1; } } */
		if((pos + size) > screen_top) {
			screen_top = pos;
			newpos = edit_line;
			refresh = -1; }
		else
			screen_top -= size; }
	if(pos < tag_low)
		tag_low -= size;
	if(pos  < tag_high)
		tag_high -= size;

	asm {
		MOV		SI,6[BP]			; Get source
		MOV		DI,SI				; Copy to dest
		ADD		SI,4[BP]			; Adjust to offset
		MOV		ES,DGRP:_edit_seg	; Get edit segment
edel1:	CMP		SI,DGRP:_edit_end	; Are we over?
		JAE		edel2				; Yes, exit
		MOV		AL,ES:[SI]			; Get byte from source
		INC		SI					; Advance
		MOV		ES:[DI],AL			; Write to dest
		INC		DI					; Advance
		JMP		SHORT edel1			; Do them all
edel2:	MOV		DGRP:_edit_end,DI	; Save new end
	}
	CHANGED = -1;
}

/*
 * Insert a number of characters in the edit buffer
 */
einsert(unsigned pos, unsigned size)
{
	if((65534 - size) < edit_end) {
		errmsg = "\007Out of memory!";
		refresh = -1;
		return 0; }

	if(pos < screen_top)
		screen_top += size;
	if(pos < edit_line)
		edit_line += size;
	if(pos < tag_low)
		tag_low += size;
	if(pos < tag_high)
		tag_high += size;

	asm {
		MOV		SI,DGRP:_edit_end	; Get source
		MOV		DI,SI				; Copy to dest
		ADD		DI,4[BP]			; Adjust for size
		MOV		DGRP:_edit_end,DI	; Save new size
		MOV		ES,DGRP:_edit_seg	; Get edit segment
eins1:	CMP		SI,6[BP]			; Are we there
		JB		eins2				; Yes, exit
		DEC		SI					; Backup source
		MOV		AL,ES:[SI]			; Get data from source
		DEC		DI					; Backup dest
		MOV		ES:[DI],AL			; Write data to dest
		JMP		SHORT eins1			; Do them all
eins2:
	}
	return CHANGED = -1;
}

/*
 * Locate a specific line in the edit buffer
 */
find_line(line) asm
{
		MOV		SI,DGRP:_edit_end	; Point to end
		MOV		CX,4[BP]			; Get line number
		AND		CX,CX				; Special case?
		JZ		findl2				; Yes, return END
		MOV		BX,SI				; Save end marker
		MOV		SI,1				; Starting offset
		MOV		ES,DGRP:_edit_seg	; Get edit segment
		DEC		CX					; Reduce count
		JZ		findl2				; We have it
findl1:	CMP		SI,BX				; Are we over?
		JAE		findl2				; Yes, we are
		MOV		AL,ES:[SI]			; Get character
		INC		SI					; Advance
		CMP		AL,0Ah				; New line?
		JNZ		findl1				; No, its not
		LOOP	findl1				; Do them all
findl2:	MOV		AX,SI				; Get pointer value
}

/*
 * Perform a partial match with the edit buffer
 */
partial_match(string, buffer) asm
{
		MOV		SI,6[BP]			; Get string
		MOV		DI,4[BP]			; Get buffer
		MOV		ES,DGRP:_edit_seg	; Point to segment
		DEC		DI					; Predecrement
partm1:	MOV		AL,[SI]				; Get source from string
		AND		AL,AL				; End of string?
		JZ		partm2				; Yes, exit
		INC		SI					; Advance string
		INC		DI					; Advance buffer
		SUB		AL,ES:[DI]			; Does it match?
		JZ		partm1				; It matches, continue
partm2:	XOR		AH,AH				; Zero high
}

/*
 * Read a file into the edit buffer
 */
read_file(char *filename, unsigned pos)
{
	FILE *fp;
	char buffer[2048];
	unsigned size;

	if(fp = openf(filename, "r")) {
		while(size = fget(buffer, sizeof(buffer), fp)) {
			if(!einsert(pos, size))
				break;
			copy_seg(edit_seg, pos, get_ds(), buffer, size);
			pos += size; }
		fclose(fp);
		return 0; }
	return 1;
}

/*
 * Write a file from the edit buffer
 */
write_file(char *filename, unsigned start, unsigned end)
{
	FILE *fp;
	char buffer[2048];
	unsigned size;

	if(fp = openf(filename, "w")) {
		while(size = end - start) {
			if(size > sizeof(buffer))
				size = sizeof(buffer);
			copy_seg(get_ds(), buffer, edit_seg, start, size);
			fput(buffer, size, fp);
			start += size; }
		fclose(fp);
		return 0; }
	return 1;
}

/*
 * --- Main program ---
 */
edit(char interactive)
{
	unsigned i, j;
	char key;

	*(W_OPEN = mwin) = Vedt_txt;
	hilite = (Vedt_txt >> 4) | (Vedt_txt << 4);
	wcursor_line();
	if(!errmsg)
		errmsg = ":EDIT: %s";
	refresh = -1;

	while(cursor_y >= W_OPEN->WINheight)	/* Window has shrunk */
		back_line();

cmd:
	/* If new screen position is requested, move to it */
	 if(newpos) {
		reposition(newpos);
		newpos = 0; }
	if(!edit_line) {
		cursor_x = cursor_y = changed = 0;
		edit_line = screen_top; }
	if(!changed) {
		edit_length = eget_line(edit_buffer, edit_line) - edit_line;
		edlen = (edit_length) ? edit_length -1 : 0;
		changed = 1; }
	j = position_cursor();	/* Position horizontal cursor */
	if(refresh)				/* Refresh screen if requested */
		display_screen(screen_top, refresh = 0);
	if(!interactive)
		return;
	if(errmsg) {			/* Display pending error message */
		switch(*errmsg++) {
			case ':' : i = Vedt_msg; break;
			default: --errmsg; i = Vedt_err; }
		title(i, errmsg, filename);
		*twin = Vedt_msg;
		errmsg = 0; }
	wgotoxy(horz - left_margin, cursor_y);

	switch(key = egetc()) {
		default:
			cursor_x = j;
		case FLI :
		case BLI : }

	if(key <= 0) switch(key) {		/* Special command key */
		case SOF :			/* Start of file */
			update_changes();
			cursor_x = cursor_y = horz = 0;
			screen_top = edit_line = EDIT_START;
			refresh = -1;
			break;
		case ENF :			/* End of file */
			screen_top = edit_end;
			back_page(W_OPEN->WINheight);	/* ???-1 */
			newpos = edit_end;
			break;
		case FPG :			/* Forward one page */
			fwd_page();
			break;
		case FLI :			/* Forward one line */
			fwd_line();
			break;
		case FCH :			/* Forward one character */
			fwd_chr();
			break;
		case BPG :			/* Back one page */
			back_page(W_OPEN->WINheight);
			break;
		case BLI :			/* Back one line */
			back_line();
			break;
		case BCH :			/* Back one character */
			back_chr();
			break;
		case SOL :			/* Cursor to start of line */
			if(!cursor_x)
				back_line();
			cursor_x = 0;
			break;
		case EOL :			/* Cursor to end of line */
			if(cursor_x >= edlen)
				fwd_line();
			cursor_x = 32767;
			break;
		case BSD :			/* Backspace & delete */
			back_chr();
			if(!changed) {
				edit_length = eget_line(edit_buffer, edit_line) - edit_line;
				edlen = edit_length ? edit_length - 1 : 0; }
			cursor_x = position_cursor();
			wgotoxy(horz - left_margin, cursor_y);
		case DEL :			/* Delete character */
			if(cursor_x < edlen) {
				changed = CHANGED = 2;
				for(i = cursor_x; i < edlen; ++i)
					edit_buffer[i] = edit_buffer[i+1];
				--edlen;
				display_line(edit_buffer + cursor_x);
				break; }
			update_changes();
			if((edit_line + edlen + 1) < edit_end)
				edelete(edit_line + edlen, 1);
			else if(!edlen)
				edit_end = edit_line;
			refresh = -1;
			break;
		case FWR :			/* Word right */
			if(cursor_x < edlen) {
				while((edit_buffer[cursor_x] > ' ') && (cursor_x < edlen))
					++cursor_x;
				while((edit_buffer[cursor_x] <= ' ') && (cursor_x < edlen))
					++cursor_x; }
			else
				fwd_chr();
			break;
		case BWR :			/* Word left */
			if(cursor_x) {
				while(cursor_x && (edit_buffer[--cursor_x] <= ' '))
					;
				while(cursor_x && (edit_buffer[cursor_x-1] > ' '))
					--cursor_x; }
			else
				back_chr();
			break;
		case INS :			/* Toggle Insert key */
			errmsg = (insert_flag = !insert_flag) ? ":Insert" : ":Overwrite";
			break;
		case ELD :			/* Toggle EOL display */
			eol_display = !eol_display;
			update_changes();
			refresh = -1;
			break;
		case POS :			/* Position info request */
			get_info();
			break;
		case TOP :			/* Current line to top */
			update_changes();
			screen_top = edit_line;
			cursor_y = 0;
			refresh = -1;
			break;
		case TAG :			/* Tag lines */
			update_changes();
			if(!tag_low) {
				tag_low = tag_high = edit_line;
				wgotoxy(horz=0, cursor_y);
				display_line(edit_buffer); }
			else {
				if(edit_line < tag_low)
					tag_low = edit_line;
				else if(edit_line > tag_high)
					tag_high = edit_line;
				else
					tag_low = tag_high = 0;
				refresh = -1; }
			break;
		case DLI :		/* Delete to end of line inclusive */
			update_changes();
			copy_seg(get_ds(), delete_buffer, edit_seg, i=edit_line + cursor_x,
				delete_length = (edlen-cursor_x)+1);
			edelete(i, delete_length);
			display_screen(edit_line, cursor_y);
			break;
		case DLE :		/* Delete to end of line exclusive */
			update_changes();
			copy_seg(get_ds(), delete_buffer, edit_seg, i=edit_line + cursor_x,
				delete_length = edlen-cursor_x);
			edelete(i, delete_length);
			display_screen(edit_line, cursor_y);
			break;
		case IDL :		/* Insert deleted line */
			update_changes();
			if(einsert(i = edit_line+cursor_x, delete_length))
				copy_seg(edit_seg, i, get_ds(), delete_buffer, delete_length);
			display_screen(edit_line, cursor_y);
			break;
		case CMD :		/* Line mode command */
			get_input("Command: ", command, sizeof(command)-1);
		case REP :		/* Re-issue last command */
			title(Vedt_msg, command);
			update_changes();
			if(execute(command))
				return;
			if(!errmsg)
				errmsg = ":EDIT: %s";
			break;
		case FUN :		/* Main menu */
			update_changes();
			return;
		case HLP :		/* Help command */
			help(H_EDIT);
			break;
		default:
			if(key_ptr = edit_key_cmds[(unsigned)key-CMDS])
				break;
		case XXX :		/* Deassigned key */
		case 0 :		/* Unassigned key */
			errmsg = "\007Invalid key"; }
	else  {				/* Normal key pressed */
		changed = 2;
		if(insert_flag || (cursor_x >= edlen) || (key == '\n')) {
			i = ++edlen;
			while(i > cursor_x) {
				edit_buffer[i] = edit_buffer[i-1];
				--i; } }
		edit_buffer[cursor_x] = key;
		if(key == '\n') {
			update_changes();
			display_screen(edit_line, cursor_y);
			cursor_x = 0;
			fwd_line(); }
		else {
			display_line(edit_buffer + cursor_x++);
			if(edit_line >= edit_end)
				display_screen(edit_line, cursor_y+1); } }
	goto cmd;
}

execute(char *cmd)
{
	unsigned i, j, start, end, tmptr;
	int k;
	char deflg, tgflg, c, *optr, *ptr;

	start = tgflg = 0;

	/* Get input line range */
	do {
		while((c = *cmd++) == ' ');		/* Skip leading blanks */
		tmptr = edit_line;
		deflg = end = i = 0;
		switch(c) {
			case '=' :		/* Tagged lines */
				if(!(tmptr = tag_low)) {
					errmsg = "\007No tagged lines";
					return 0; }
				end = tag_high;
				tgflg = -1;
				break;
			case '/' :		/* Entire file */
				tmptr = EDIT_START;
				end = edit_end;
			case '*' :		/* Current line */
				break;
			default:		/* Unknown.. could be numeric */
				--cmd;
				if(isdigit(c)) {
					while(isdigit(*cmd))
						i = (i*10) + (*cmd++ - '0');
					tmptr = find_line(i); }
				else
					deflg = -1; }

		/* Handle '+' and '-' from range */
		while((c = *cmd++) == ' ');
		if((c == '+') || (c == '-')) {
			i = 0;
			while(isdigit(*cmd))
				i = (i*10) + (*cmd++ - '0');
			if(c == '+') {
				while((tmptr < edit_end) && i)
					if(peek(edit_seg, tmptr++) == '\n')
						--i; }
			else {
				++i;
				while((tmptr >= EDIT_START) && i)
					if(peek(edit_seg, --tmptr) == '\n')
						--i;
				++tmptr; }
			end = tmptr;
			deflg = 0;
			while((c = *cmd++) == ' '); }

		if(!start)
			start = tmptr;
		if(!end)
			end = tmptr; }
	while(c == ',');

	if(start > end) {
		tmptr = start;
		start = end;
		end = tmptr; }

	while((end < edit_end) && (peek(edit_seg, end++) != '\n'));
	j = end - start;

	/* Get command character */
	optr = cmd;
	while(*cmd == ' ')
		++cmd;
	switch(c = tolower(c)) {
		case 0 :			/* Goto line */
			newpos = start;
			break;
		case 'c' :			/* Copy lines */
		case 'm' :
			if((edit_line > start) && (edit_line < end)) {
				errmsg = "\007Invalid destination";
				break; }
			if(!einsert(edit_line, j))
				break;
			if(start >= edit_line)
				start += j;
			copy_seg(edit_seg, edit_line, edit_seg, start, j);
			refresh = -1;
			if(c != 'm')
				break;
		case 'd' :			/* Delete lines */
			edelete(start, j);
			refresh = -1;
			break;
		case '?' :			/* Find */
			if(deflg) {
				start = edit_line + cursor_x + 1;
				end = edit_end; }
			while(start < end) {
				if(!partial_match(optr, start))
					end = newpos = start;
				++start; }
			if(!newpos)
				errmsg = "Not found";
			break;
		case 's' :			/* Substitute */
			ptr = edit_buffer;
			c = *cmd++;
			while(*cmd && (*cmd != c))
				*ptr++ = *cmd++;
			*ptr = 0;
			if(!*cmd++) {
				errmsg = "\007Invalid search string";
				break; }
			i = strlen(edit_buffer);
			j = strlen(cmd);
			while(start < end) {
				if(!partial_match(edit_buffer, start)) {	/* found it */
					if(i < j) {
						if(!einsert(start, k = j - i))
							goto nosub;
						end += k; }
					else if(i > j) {
						edelete(start, k = i - j);
						end -= k; }
					copy_seg(edit_seg, start, get_ds(), cmd, j);
					newpos = start;
					start += j; }
				else
					++start; }
			if(newpos)
				refresh = CHANGED = -1;
			else
				errmsg = "\007Not found";
		nosub:
			break;
		case 't' :			/* Tag lines */
			tag_low = start;
			tag_high = end - 1;
			tgflg = 0;
			refresh = -1;
			break;
		case 'r' :			/* Read file */
			if(!read_file(cmd, start))
				refresh = CHANGED = -1;
			break;
		case 'f' :			/* File information */
			wclwin();
			i = j = 0;
			for(tmptr = EDIT_START; tmptr <= edit_end; ++tmptr) {
				if(tmptr == start)
					j = i;
				if(peek(edit_seg, tmptr) == '\n')
					++i; }
			wprintf("Filename: %s, %u Lines, %u Characters\n",
				filename, i, edit_end - EDIT_START);
			i = 0;
			for(tmptr = start; tmptr <= end; ++tmptr) {
				if(peek(edit_seg, tmptr) == '\n')
					++i; }
			wprintf("Position: %u, %u Lines, %u Characters\n",
				j+1, i, end - start);
			wprintf("There are%s unsaved changes.\n",
				CHANGED ? "" : " no");
			title(Vedt_msg, "Press any key to continue...");
			w_getc(twin);
			refresh = -1;
			break;
		case 'w' :			/* Write file */
			if(deflg) {
				start = EDIT_START;
				end = edit_end; }
			if(!*cmd)
				cmd = filename;
			if(write_file(cmd, start, end)) {
				/* errmsg = "\007Can't open output file"; */
				break; }
			CHANGED = 0;
			break;
		default:
			errmsg = "\007Unknown command"; }

	if(tgflg && !errmsg) {
		tag_low = tag_high = 0;
		refresh = -1; }

	return 0;
}

/*
 * Get a key with translations for the editor
 */
egetc()
{
	int k;

	if((k = dgetc()) & 0x80)		/* Function key */
		k = ((k &= 0x7F) < sizeof(edit_keys)) ? (int)edit_keys[k+27] : 0;
	else if(k <= 0x1B)
		k = k ? (int)edit_keys[k-1] : 0;

	return k;
}
