/*
 * Debug Scripting MONitor
 *
 * See DESMO.TXT and: DESO ?
 *
 * ?COPY.TXT 2003-2007 Dave Dunfield
 */
#include <stdio.h>
#include <setjmp.h>
#include <window.h>
#include <keys.h>
#include <comm.h>
#include "desmo.h"
#define	TICK	peekw(0x40, 0x6C)

/* Help subsystem parameters */
#define	LINK_S		('N'-0x40)			/* Start of a link */
#define	LINK_E		('O'-0x40)			/* End of a link */
#define	TAB_SIZE	4					/* Spacing of TABs */
#define	LINKS		50					/* Maximum # links per screen */

#define	GVARS	500
#define	CMDSIZE	100
#define	STACK	500

#define	STK_WHILE	0xFF00		// WHILE is stacked
#define	STK_IF		0xFF01		// IF is stacked
#define	STK_XXX		0xFF02		// Depreciated If/While is stacked

unsigned
	Cport = 1,			// Commport
	Baud,				// Baud rate
	Seg,				// Active segment
	Stop = 3,			// Segment top
	Key,				// Key code
	Line,				// Line number
	Fline,				// File line#
	FP,					// Function Pointer
	SP,					// Stack pointer
	Fbase,				// Function base pointer
	Gtop,				// Global symbol top
	pool_top,			// Top of symbol pool
	pool_base,			// Base for pool
	_Gtop,				// Base symbol top
	_pool_top,			// Base pool top
	Ctimeout,			// Comms timeout
	parms,				// Parameter pointer
	gvalue[GVARS][2],	// Global variable values
	stack[STACK],		// System stack
	Tblocks[14];		// Target access blocks

unsigned char
	*keyptr = "",		// Fkey pointer
	*Func,				// Function name pointer
	*pptr,				// Parsing pointer
	*gvars[GVARS],		// Global variable names
	gtype[GVARS],		// Variable type
	eflag,				// Global end flag
	Cflag,				// Comms flags
	xtemp[65],			// More temp storage
	lastcmd[CMDSIZE+1],	// Last command
	pool[15000];		// Global text pool

unsigned char
	One[4] = { 1 },		// Long one
	Two[4] = { 2 },		// Long two
	Four[4] = { 4 };	// Long four

char *keywords[] = {
	"R8",
	"R16",
	"R32",
	"W8",
	"W16",
	"W32",
	"PRINT",
	"END",
	"IF",
	"EXIT",
	"ELSE",
	"WHILE",
	"GOTO",
	"DELAY",
	"LR8",
	"LR16",
	"LR32",
	"LW8",
	"LW16",
	"LW32",
	"VARIABLE",
	"CONSTANT",
	"FUNCTION",
	"FILE",
	"KEY",
	"CLS",
	"CLEAR",
	"HELP",
	"RELOAD",
	"STOP",
	"VLIST",
	"CLIST",
	"FLIST",
	"DUMP",
	"TTY"
	};
#define	R8		0
#define	R16		1
#define	R32		2
#define	W8		3
#define	W16		4
#define	W32		5
#define	PRINT	6
#define	END		7
#define	IF		8
#define	EXIT	9
#define	ELSE	10
#define	WHILE	11
#define	GOTO	12
#define	DELAY	13
#define	LR8		14
#define	LR16	15
#define	LR32	16
#define	LW8		17
#define	LW16	18
#define	LW32	19
#define	VAR		20
#define	CONST	21
#define	FUNC	22
#define	INCL	23
#define	KEY		24
#define	CLS		25
#define	CLEAR	26
#define	HELP	27
#define	RELOAD	28
#define	STOP	29
#define	VLIST	30
#define	CLIST	31
#define	FLIST	32
#define	DUMP	33
#define	TTY		34
#define	NKEY	35

FILE
	*fp;
extern unsigned Longreg[];
extern int Csignals();
extern char *ARGV[];

// Application variables
jmp_buf
	gomain;		// Return to main

struct WINDOW
	*swin,		// Status window
	*mwin;		// Main window

unsigned char
	fnames[40][8],	// Function key names
	fkeys[40][81];	// Function key text

unsigned char colors[] = {
	0x70,			//0: Status window
	0x74,			//1: Status hilight
	0x17,			//2: Main window
	0x70,			//3: Help screen
	0x71,			//4: Help links
	0x17 };			//5: Help cursor
unsigned char mcolors[] = {
	0x70, 0x07, 0x07, 0x70, 0x7F, 0x07 };
#define	CO_STATUS	0			// Color index for status line
#define	CO_SHIGH	1			// Color index for status hilight
#define	CO_MAIN		2			// Color index for main screen
#define	CO_HELPS	3			// Color index for Help screen
#define	CO_HELPL	4			// Color index for help links
#define	CO_HELPC	5			// Color index for help cursor

// Mark current boundaries
mark()
{
	_Gtop = Gtop;
	_pool_top = pool_top;
}
// Return to marked position
release()
{
	if(fp)
		fclose(fp);
	fp = FP = Fbase = Key = 0;
	Gtop = _Gtop;
	pool_top = _pool_top;
	SP = parms = eflag = Func = 0;
	keyptr = "";
}

// Report an error and exit
register error(unsigned args)
{
	char buffer[81];
	_format_(nargs() * 2 + &args, buffer);
	if(!mwin) {
		fputs(buffer, stdout);
		putc('\n', stdout);
		exit(-1); }
	if(fp)
		wprintf("[%u] ", Fline);
	if(Func)
		wprintf("%s(%u): ", Func, Line);
	wprintf("%s\n", buffer);
	release();
	longjmp(gomain, 255);
}
register status(unsigned args)
{
	char buffer[81];
	_format_(nargs() * 2 + &args, buffer);
	w_gotoxy(0, 0, swin);
	w_puts(buffer, swin);
	w_cleol(swin);
}

unsigned char *rfile(unsigned char *ext)
{
	unsigned char *p1, *p2, *p3;
	p1 = ARGV[0];
	p2 = p3 = xtemp;
	for(;;) switch(*p2++ = *p1++) {
		case 0 :
			strcpy(p2-4, ext);
			return p3;
		case ':' :
		case '\\': p3 = p2; }
}

/*
 * Test for user abort within target function
 */
void Cabort()
{
	if(Cflag & CF_MESSAGE) {
		switch(wtstc()) {
		case 0x1B :
		case _KEN :
			release();
			longjmp(gomain, 255); } }
}

/*
 * Get character from target with timeout
 */
int Cgett()
{
	unsigned c, t;
	t = TICK;
	do {
		if((c = Ctestc()) != -1)
			return c;
		Cabort(); }
	while((TICK - t) <= Ctimeout);
	Cflag |= CF_ERROR;
	if(!(Cflag & CF_FAIL))
		error("Serial timeout");
}

/*
 * Get hex data from target
 */
unsigned Cgeth()
{
	unsigned char c;
	c = Cgett();
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c - ('A'-10);
	if((c >= 'a') && (c <= 'f'))
		return c - ('a'-10);
	Cflag |= CF_ERROR;
	if(!(Cflag & CF_FAIL))
		error("Bad target response: expect 0-F, receive %02x", c);
}

/*
 * Expect a character from the target
 */
void Cexpect(unsigned char c)
{
	unsigned char d;
	if((d = Cgett()) != c) {
		Cflag |= CF_ERROR;
		if(!(Cflag & CF_FAIL))
			error("Bad target response: expect %02x, receive %02x", c, d); }
}

/*
 * Write a character to the target (expect echo if enabled)
 */
void Cwrite(unsigned char c)
{
	Cputc(c);
	if(Cflag & CF_ECHO)
		Cexpect(c);
}

/*
 * Perform a target transaction
 */
void target(unsigned cc, unsigned char addr[], unsigned char data[], char write)
{
	unsigned char c, d, *cmd;
	unsigned t;

	if(!(t = Tblocks[cc]))
		error("Not supported by target");
// wprintf("%u %u : ", cc, t);
	cmd = (pool+t)-1;

	Ctimeout = 18;		// Default timeout
	Cflag = 0;

	while(c = *cmd++) {
// wprintf("%02x ", c);
		if(Cflag & CF_ERROR)
			break;

		if(c >= 0xF8) {		// Data element access
			if(write) {				// Sending
				d = data[c&3];
				goto send; }
			if(c & 4) {				// Receive ASCII
				d = Cgeth() << 4;
				if(Cflag & CF_ERROR)
					break;
				data[c&3] = Cgeth() | d;
				continue; }
			data[c&3] = Cgett();	// Receive BINARY
			continue; }

		if(c >= 0xF0) {		// Address element access
			d = addr[c&3];
		send:
			if(c & 4) {				// Send ASCII
				if((c = d >> 4) > 9)
					c += 7;
				Cwrite(c+'0');
				if(Cflag & CF_ERROR)
					break;
				if((c = d & 0x0F) > 9)
					c += 7;
				Cwrite(c+'0');
				continue; }
			Cwrite(d);				// Send BINARY
			continue; }

		if(c <= MAXDATA) {				// Write data
			do {
				Cwrite(*cmd++);
				if(Cflag & CF_ERROR)
					break; }
			while(--c);
			continue; }

		switch(c) {
		case tPRINT :				// Print message
			while(c = *cmd++)
				wputc(c);
			wputc('\n');
			continue;
		case tFILL	:				// Send fill block
			c = *cmd++;
			d = *cmd++;
			do {
				Cwrite(c); }
			while(--d);
			continue;
		case tDROP :				// Discard characters
			d = *cmd++;
			do {
				Cgett();
				if(Cflag & CF_ERROR)
					break; }
			while(--d);
			continue;
		case tEXPECT:				// Expect character
			Cexpect(*cmd++);
			continue;
		case tUNTIL:				// Wait for character
			c = *cmd++;
			while(Cgett() != c) {
				if(Cflag & CF_ERROR)
					break; }
			continue;
		case tTIME :				// Set timeout
			Ctimeout = *cmd++;
			continue;
		case tFLUSH :					// Flush
			d = *cmd++;
		loop:
			t = TICK;
			do {
				if(Ctestc() != -1)
					goto loop;
				Cabort(); }
			while((TICK - t) <= d);
			continue;
		case tFLAGS :					// Control flags
			d = *cmd++ & CF_MASK;
			if((d ^ Cflag) & CF_MESSAGE) {
				if(d & CF_MESSAGE)
					status("Press ESC to cancel.");
				else
					update_status(); }
			Cflag = d;
			t = *(unsigned*)(&Csignals+2);
			out(*(unsigned*)t - 1, (Cflag & (CF_DTR|CF_RTS)) | OUTPUT_2);
			continue;
		case tBAUD :					// Bandrate
			Copen(Cport, *cmd++, DATA_8|PAR_NO|STOP_1, (Cflag & (CF_DTR|CF_RTS)) | OUTPUT_2);
			continue;
		case tESCAPE :					// Escape
			while((t = wtstc()) != 0x1B) {
				if(t == _KEN)
					error("User abort!"); }
	} }
	if(Cflag & CF_MESSAGE)
		update_status();
}

// Test for valid symbol
issymbol(unsigned char c)
{
	return	((c >= 'a') && (c <= 'z'))
		||	((c >= 'A') && (c <= 'Z'))
		||	((c >= '0') && (c <= '9'))
		||	(c == '_');
}

// Add a symbol to the pool
unsigned char *add_pool(char *symbol)
{
	unsigned t;
	t = pool_top;
	do {
		if(pool_top >= sizeof(pool))
			error("Out of memory"); }
	while(pool[pool_top++] = *symbol++);
	return pool+t;
}

// Skip to non-blank
unsigned char skip()
{
	while(isspace(*pptr))
		++pptr;
	return *pptr;
}

// Parse string from text
char *parse(unsigned char *dest)
{
	unsigned char *d;
	d = dest;
	skip();
	while(*pptr && !isspace(*pptr))
		*d++ = toupper(*pptr++);
	*d = 0;
	return *dest;
}

// Parse a sumbol from text
char *parse_symbol(unsigned char *dest)
{
	unsigned char *d;
	d = dest;
	skip();
	while(*pptr && issymbol(*pptr))
		*d++ = toupper(*pptr++);
	*d = 0;
	return *dest;
}

// Parse symbol & report error if not found
get_symbol(unsigned char *dest)
{
	if(!parse_symbol(dest))
		error("Bad symbol");
}

// Lookup word in table
int lookup(char *word, char *list[], unsigned top)
{
	unsigned i;
	for(i=top-1; i != -1; --i) {
		if(!strcmp(word, list[i]))
			return i; }
	return -1;
}

// Get a value
get_value(unsigned char *dest)
{
	unsigned i, b, h;
	unsigned char temp[40], l[4], c, f, hf;

	switch(skip()) {
	case '(' :
		++pptr;
		value(dest);
		return;
	case '-' :
		++pptr;
		get_value(l);
		longset(dest, 0);
		longsub(dest, l);
		return;
	case '~' :
		++pptr;
		get_value(dest);
		for(i=0; i < 4; ++i)
			dest[i] ^= 0xFF;
		return;
	case '!' :
		++pptr;
		get_value(dest);
		longset(dest, longtst(dest) ? 0 : 1);
		return;
	case '#' :
		++pptr;
		switch(c = toupper(*pptr++)) {
		case 'P' : longset(dest, stack[parms]);	return;
		case 'K' : longset(dest, 0); *dest = Key; Key = 0; return; }
		c -= '1';
		i = stack[parms];
		if(SP && parms && (i > c)) {
			i = ((i-c)*2);
			longcpy(dest, &stack[parms-i]);
			return; }
		error("Bad '#' variable");
	case '.' :
		++pptr;
		b = 10;
		break;
	case '%' :
		++pptr;
		b = 2;
		break;
	case '$' :
		++pptr;
		b = 16;
		break;
	default:
		b = pptr;
		if(!parse_symbol(temp))
			error("Value required");
		if((i = lookup(temp, gvars, Gtop)) != -1) {
			if(gtype[i] > 1)
				error("Wrong symbol type");
			longcpy(dest, gvalue[i]);
			return; }
		if((i = lookup(temp, keywords, NKEY)) != -1) switch(i) {
			case R8 : i = tR8; goto rd1;
			case R16: i = tR16;
		rd1:	longset(dest, 0);
		rd2:	get_value(l);
				target(i, l, dest, 0);
				return;
			case R32 : i = tR32; goto rd2; }
		for(i=0; c = temp[i]; ++i) {
			if((c >= '0') && (c <= '9'))
				continue;
			if((c < 'A') || (c > 'F'))
				error("Not found: %s", temp); }
		pptr = b;
	case '0' :
	case '1' :
	case '2' :
	case '3' :
	case '4' :
	case '5' :
	case '6' :
	case '7' :
	case '8' :
	case '9' :
		b = 16; }
	// A calculated number
	longset(dest, f = hf = 0);
	do {
		switch(c = *pptr) {
		case '.' :
			if(hf)
				error("Too many '.'s");
			if(dest[3]|dest[2])
				error(".>16 bits");
			h = *(unsigned*)dest;
			longset(dest, 0);
			hf = -1;
		case ',' :
			continue; }
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if(c >= 'a')
			c -= ('a'-10);
		else if(c >=  'A')
			c -= ('A'-10);
		else
			break;
		if(c >= b)						/* outside of base */
			break;
		longset(l, b);
		longmul(dest, l);
		longset(l, c);
		longadd(dest, l);
		f = -1; }
	while(*++pptr);
	if(!f)
		error("Bad value element");
	if(hf) {
		if(dest[3]|dest[2])
			error(".>16 bits");
		dest[3] = h >> 8;
		dest[2] = h; }
}

// Obtain a value from an expression
value(unsigned char *dest)
{
	unsigned o, o1;
	unsigned char l[4], c;

	get_value(dest);
	for(;;) {
		c = 0;
		switch(o = skip()) {
		case ')' :
			++pptr;
		case 0 :
		default: return;
		case '<' :
			c = 255;
			switch(o1 = *++pptr) {
			case '<' :
				c = 0;
			case '=' :
				o = (o << 8) | o1;
				++pptr; }
			break;
		case '>' :
			c = 255;
			switch(o1 = *++pptr) {
			case '>' :
				c = 0;
			case '=' :
				o = (o << 8) | o1;
				++pptr; }
			break;
		case '!' :
			if(*(pptr+1) != '=')
				return;
			o = '!=';
			++pptr;
		case '=' :
			c = 255;
		case '+' :
		case '-' :
		case '*' :
		case '/' :
		case '\\':
		case '&' :
		case '|' :
		case '^' :
			++pptr; }
		get_value(l);
		if(c) o1 = longcmp(dest, l);
		switch(o) {
		case '+' :	longadd(dest, l);	continue;
		case '-' :	longsub(dest, l);	continue;
		case '*' :	longmul(dest, l);	continue;
		case '/' :	longdiv(dest, l);	continue;
		case '\\':	longdiv(dest, l);	longcpy(dest, Longreg);	continue;
		case '<' :	if(o1 == -1) goto ret1;
		ret0:		longset(dest, 0);	continue;
		case '>' :	if(o1 != 1)	goto ret0;
		ret1:		longset(dest, 1);	continue;
		case '=' :	if(o1)		goto ret0;	goto ret1;
		case '!=':	if(o1)		goto ret1;	goto ret0;
		case '<=' :	if(o1 == 1)	goto ret0;	goto ret1;
		case '>=' : if(o1 ==-1) goto ret0;	goto ret1;
		case '<<' : if(c = *l) do { longshl(dest); } while(--c); continue;
		case '>>' : if(c = *l) do { longshr(dest); } while(--c); continue;
		case '&' :	for(o=0;o<4;++o) dest[o] &= l[o];	continue;
		case '|' :	for(o=0;o<4;++o) dest[o] |= l[o];	continue;
		case '^' :	for(o=0;o<4;++o) dest[o] ^= l[o];	continue;
		} }
}

/*
 * Simple help system with links
 *
 * Input: Section number to display
 */
void help(unsigned section)
{
	int c, lx, ly, ox, oy;
	unsigned i, j, size, ncount;
	unsigned char buffer[65], *ptr, *pptr;
	unsigned char xs[LINKS], ys[LINKS], *names[LINKS], link[LINKS];
	FILE *fp;

	rfile("HLP");
	if(!(fp = fopen(xtemp, "rb"))) {
		error("Cannot open help file: %s", xtemp);
		return; }

	/* Locate the section in question */
	pptr = &pool[pool_top];
lookup:
	size = getc(fp);
	size |= getc(fp) << 8;
	if(section--) {
		fseek(fp, 0, size, 1);
		goto lookup; }

	/* Draw the screen */
	wopen(5, 2, 70, 17, WSAVE|WCOPEN|WBOX1|colors[CO_HELPS]);
	wcursor_off();
	i = ncount = ptr = 0;
	while(buffer[i++] = getc(fp));

	/* Make a title */
	memcpy(buffer+50, W_OPEN, 6);
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", buffer);
	memcpy(W_OPEN, buffer+50, 6);

	/* Display help text & record links */
	wgotoxy(0, 0);
	while(i++ < size) switch(c = getc(fp)) {
	case LINK_S :	/* Start link */
		names[ncount] = ptr = pptr;
		xs[ncount] = W_OPEN->WINcurx;
		ys[ncount] = W_OPEN->WINcury;
		*W_OPEN = colors[CO_HELPL];
		break;
	case LINK_E :	/* End link */
		link[ncount++] = getc(fp);
		*pptr++ = ptr = 0;
		*W_OPEN = colors[CO_HELPS];
		++i;
		break;
	case '\t' :		/* Horizontal TAB */
		do
			wputc(' ');
		while(W_OPEN->WINcurx % TAB_SIZE);
		break;
	case 0 :		/* End of line */
		c = '\n';
	default:
		wputc(c);
		if(ptr)
			*pptr++ = c; }

	/* Allow user to select field & link to new screen */
	i = section = 0;
	for(;;) {
		wgotoxy(xs[section], ys[section]);
		*W_OPEN = colors[CO_HELPL];
		wputs(names[section]);
		wgotoxy(xs[i], ys[i]);
		*W_OPEN = colors[CO_HELPC];
		wputs(names[section = i]);
		*W_OPEN = colors[CO_HELPS];
		switch(c = wgetc()) {		/* User keystroke */
		case _KLA :				/* Left arrow - previous field */
			i = (i ? i : ncount) - 1;
			break;
		case _KRA :				/* Right arrow - next field */
			i = (i + 1) % ncount;
			break;
		case _KUA : ox = oy = -1000; goto do_find;
		case _KDA : ox = oy = 1000;
		do_find:
			size = i;
			for(j = 0; j < ncount; ++j) {
				lx = (int)xs[j] - (int)xs[i];
				ly = (int)ys[j] - (int)ys[i];
				if(c == _KUA) {
					if((ly >= 0) || (ly < oy)) continue; }
				else {
					if((ly <= 0) || (ly > oy))	continue; }
				if(abs(lx) > abs(ox)) continue;
				size = j;
				ox = lx;
				oy = ly; }
			i = size;
			break;
		case '\n' :				/* Enter - chain to menu */
			rewind(fp);
			section = link[i];
			wclose();
			goto lookup;
		case 0x1B:				/* Escape exit */
		case _K10 :
			wclose();
			fclose(fp);
			return; } }
}

unsigned
	Daddr[2],			// Dump address
	Dpos;				// Dump position
unsigned char
	buffer[80],
	Dflag;				// Dump newline flag

/*
 * Begin a memory dump
 *
void dump_start(unsigned address[2])
{
	Daddr[1] = address[1];
	Daddr[0] = address[0] & 0xFFF0;
	Dpos = address[0] - Daddr[0];
	*buffer = Dflag = 0;
} */

/*
 * End a memory dump
 */
void dump_end(unsigned char *buffer)
{
	unsigned i;
	unsigned char c;
	if(*buffer) {
		for(i=0; i < 79; ++i)
			wputc((c=buffer[i]) ? c : ' ');
		wputc('\n');
		*buffer = Dpos = 0; }
}

/*
 * Dump a character to the memory dump
 *
void dump(unsigned char c)
{
	unsigned l[2];
	if(!*buffer) {
		if(Dflag && !(*Daddr & 0xFF))
			wputc('\n');
		memset(buffer, 0, 80);
		sprintf(buffer, "%04x.%04x", Daddr[1], Daddr[0]);
		longset(l, Dpos);
		longadd(Daddr, l); }
	sprintf(&buffer[(Dpos*3)+11+(Dpos/4)], "%02x", c);
	buffer[Dpos+63] = isprint(c) ? c : '.';
	longadd(Daddr, One);
	if(++Dpos >= 16) {
		dump_end();
		Dflag = -1; }
} */

main(int argc, char *argv[])
{
	unsigned i;
	FILE *xfp;

	for(i=1; i < argc; ++i) {
		pptr = argv[i];
		switch((toupper(*pptr++)<<8)|toupper(*pptr++)) {
		case '?'<<8 :
		case '/?' :
		case '-?' : help(0); return;
		case '/M' :
		case '-M' : memcpy(colors, mcolors, sizeof(colors));	continue;
		case 'C=' :
			Cport = atoi(pptr);
			if((Cport<1) || (Cport>4)) {
				printf("Com port (C=) must be 1-4\n");
				goto xhelp; }
			continue;
		case 'S=' :
			if(Seg = atoi(pptr)) {
				Baud = 57600 / (Seg /= 2);
				if((Baud * Seg) == 57600)
					continue; }
			printf("Invalid (S=) speed value\n");
	xhelp:	printf("Use DESMO ? for help.\n");
			exit(-1);
		case 'T=' :	Func = pptr;			continue; }
		if(!(fp = fopen(argv[1], "rv")))
			goto xhelp; }

	if(!(Seg = alloc_seg(4096)))
		error("Not enough memory");

	if(!Func) {
		if(getenv("DESMO", Func = pool))
			goto dof;
		rfile("DTC"); }
	else {
dof:	concat(xtemp, Func, ".DTC");
		if(xfp = fopen(xtemp, "rb"))
			goto fok;
		concat(rfile(""), Func, ".DTC"); }
		
	if(!(xfp = fopen(xtemp, "rb")))
		error("Cannot open target file: %s", xtemp);
fok:
	fget(&i, sizeof(i), xfp);
	if(!Baud)
		Baud = i;
	fget(Tblocks, sizeof(Tblocks), xfp);
	pool_base = _pool_top = pool_top = fget(pool, sizeof(pool), xfp);
	fclose(xfp);;

	if(Copen(Cport, Baud, DATA_8|PAR_NO|STOP_1, SET_RTS|SET_DTR|OUTPUT_2))
		error("Cannot open COM port %u", Cport);
	Cflags |= TRANSPARENT;

	swin = wopen(Func=0, 24, 80, 1, WSAVE|WCOPEN|colors[CO_STATUS]);
	if(W_BASE == 0xB000)
		memcpy(colors, mcolors, sizeof(colors));
	mwin = wopen(0, 0, 80, 24, WSAVE|WCOPEN|WSCROLL|WWRAP|colors[CO_MAIN]);
	if(!setjmp(gomain)) {
		if(Tblocks[tVERIFY])
			target(tVERIFY, xtemp, xtemp, -1);
		else
			Cflag = CF_ERROR; }
	if(Tblocks[tLOAD] && (Cflag & CF_ERROR)) {
		if(!setjmp(gomain))
			target(tLOAD, xtemp, xtemp, -1); }
	if(!(Cflag & CF_ERROR))
		wclwin();
	if(fp) {
		if(!setjmp(gomain)) {
			FP = 2;
			while(fp)
				command(); } }
	wputs("DESMO("#__DATE__"): use 'HELP' for assistance.\n");
	setjmp(gomain);
	FP = 1;
	while(FP)
		command();

	Cclose();
	wclose();
	wclose();
}

// Getc character & update fkey display
unsigned char lko;
Getc()
{
	unsigned i;
	unsigned char ko;
	if(i = *keyptr) {
		++keyptr;
		return (i==';') ? '\n' : i; }
	keyptr = "";
	wupdatexy();
	do {
		switch(peek(0, 0x417) & 0x0F) {
		case 0x00 :	ko = 0;		break;
		case 0x01 :
		case 0x02 :	ko = 10;	break;
		case 0x04 :	ko = 20;	break;
		case 0x08 : ko = 30;	break;
		default:	ko = 100; }
		if(ko != lko) {
			w_clwin(swin);
			if((lko = ko) < 31) {
				for(i=0; i < 10; ++i) {
					if(*fkeys[ko+i]) {
						*swin = colors[CO_SHIGH];
						w_printf(swin, "%u", (i+1)%10);
						*swin = colors[CO_STATUS];
						w_printf(swin, "%-7s", fnames[ko+i]); }
					else
						w_printf(swin, "%8s", ""); } }
			wupdatexy(); } }
	while(!(i = kbtst()));
	return (i == '\r') ? '\n' : i;
}

// Get input line via window
Gets(unsigned char dest[], unsigned size)
{
	unsigned c, l;
	lko = 255;
	l = 0;
	for(;;) {
		wupdatexy();
		c = Getc();
next_char:
		switch(c) {
		case _PU :
			keyptr = lastcmd + l;
			continue;
		case _PD :
			strcpy(dest+l, lastcmd+l);
			wputs(dest+l);
			goto doend;
		case _RA :
			if(c = lastcmd[l]) {
				goto next_char; }
			continue;
		case '\n' :
			dest[l] = 0;
		doend:
			wputc('\n');
			l = 0;
			while(c = dest[l++]) {
				if(!isspace(c))
					break; }
			if(c) {
				memset(lastcmd, 0, sizeof(lastcmd));
				strcpy(lastcmd, dest); }
			w_clwin(swin);
			return -1;
		case _LA :
		case '\b' :
			if(l) {
				wputs("\b \b");
				--l; }
			continue;
		case 0x1B :
			while(l) {
				wputs("\b \b");
				--l; }
			continue; }
		if((c >= 0x3B00) && (c <= 0x4400)) {
			c = (c >> 8) - 0x3B;
			goto dofunc; }
		if((c >= 0x5400) && (c <= 0x7100)) {
			c = (c >> 8) - 0x4A;
		dofunc:
//			if(*(keyptr = fkeys[c]))
//				wprintf("[%s]", fnames[c]);
			keyptr = fkeys[c];
			continue; }
		if((l < size) && (c >= ' ') && (c <= '~'))
			wputc(dest[l++] = c); }
}

update_status()
{
	if(FP > 2)
		status("Function: %s", Func);
	else
		w_clwin(swin);
}

/*
 * Get next line from the source file
 */
get_line(unsigned char *dest, unsigned char *prompt)
{
	unsigned char c;
	switch(FP) {
	case 0 :
		return 0;
	case 1 :	// Interactive
		wprintf("%s> ", prompt);
		if(!Gets(dest, CMDSIZE))
			return FP = 0;
		++Line;
		return -1;
	case 2 :	// From file
		if(!fgets(dest, CMDSIZE, fp)) {
			fclose(fp);
			return fp = FP = 0; }
		++Line;
		++Fline;
		return -1; }
	if(c = peek(Seg, FP)) {		// From function
		++Line;
		do {
			++FP;
			if(c == '\n')
				break;
			*dest++ = c; }
		while(c = peek(Seg, FP));
		*dest = 0;
		return -1; }
	return FP = 0;
}

testfunc()
{
	if(FP < 3)
		error("Must be in function");
}

// Define a user function
define_func(unsigned char *name)
{
	unsigned i, x;
	unsigned char line[CMDSIZE+1], temp[25], *p;

	Func = name;
	Line = 0;
	if((i = lookup(name, gvars, Gtop)) != -1)
		error("Name in use: %s", name);

	gvars[Gtop] = add_pool(name);
	gvalue[Gtop][0] = Stop;
	gtype[Gtop++] = 2;
	x = 0;
	while(get_line(pptr = line, name)) {
		switch(skip()) {
		case 0 :
		case ';' : continue; }
		p = pptr;
		parse_symbol(temp);
		switch(lookup(temp, keywords, NKEY)) {
		default:
		copy:
			while(*p)
				poke(Seg, Stop++, *p++);
			poke(Seg, Stop++, '\n');
			continue;
		case IF :
		case WHILE :
			++x; goto copy;
		case END :
			if(x) { --x; goto copy; }
			poke(Seg, Stop++, Func = 0);
			return;
		case FUNC:
		case INCL:
			error("Command not allowed in function"); }
	} error("EOF in function: %s", name);
}

command()
{
	unsigned a[2], l[2], i, j, k, f, ln;
	unsigned char line[CMDSIZE+1], temp[25];

	f = FP;
	ln = Line;
	if(get_line(pptr = line, "DesMo")) {
	again:
		switch(skip()) {
		case ';' :	// comment
		case 0 :	// Blank line
			return; }
		if(i = kbtst()) {
			if(i == _EN)	// Escape
				error("User abort!");
			Key = i; }
		parse_symbol(temp);
		if(skip() == ':') {
			++pptr;
			goto again; }
		switch(i = lookup(temp, keywords, NKEY)) {
		case IF :
			testfunc();
			stack[SP++] = STK_IF;
			if(eflag) { ++eflag; return; }	// Blocked
			value(l);
			if(!longtst(l)) eflag = 1;
			return;
		case WHILE :
			testfunc();
			if(!eflag) {			// Processing enabled
				value(l);
				if(longtst(l)) {	// Condition passed
					stack[SP++] = f;
					stack[SP++] = ln;
					stack[SP++] = STK_WHILE;
					return; } }
			++eflag;
			stack[SP++] = STK_XXX;
			return;
		case END :
			testfunc();
			switch(stack[SP-1]) {
			case STK_WHILE : 	// WHILE
				--SP;
				if(eflag) {		// Blocked
					SP -= 2;
					--eflag;
					return; }
				Line = stack[--SP];
				FP = stack[--SP];
				return;
			case STK_IF :	// IF
			case STK_XXX:	// Deppreciated IF/WHILE
				--SP;
				if(eflag)
					--eflag; }
			return;
		case ELSE :
			testfunc();
			if(stack[SP-1]  != STK_IF)
				error("Misplaced ELSE");
			stack[SP-1] = STK_XXX;
			switch(eflag) {
			case 0 : eflag = 1;	return;
			case 1 : eflag = 0;	}
			return; }
	if(eflag)
		return;

		switch(i) {
		case R8 :	// 8 bit read
			do {
				value(a);
				longset(l, 0);
				target(tR8, a, l, 0);
				wprintf("%04x.%04x = %02x\n", a[1], a[0], *(unsigned char*)l);
				longadd(a, One); }
			while(skip());
			return;
		case R16 :	// 16 bit read
			do {
				value(a);
				longset(l, 0);
				target(tR16, a, l, 0);
				wprintf("%04x.%04x = %04x\n", a[1], a[0], l[0]);
				longadd(a, Two); }
			while(skip());
			return;
		case R32 :	// 32 bit read
			do {
				value(a);
				target(tR32, a, l, 0);
				wprintf("%04x.%04x = %04x.%04x\n", a[1], a[0], l[1], l[0]);
				longadd(a, Four); }
			while(skip());
			return;
		case W8 :	// 8 bit write
			value(a);
			do {
				value(l);
				target(tW8, a, l, -1);
				longadd(a, One); }
			while(skip());
			return;
		case W16 :	// 16 bit write
			value(a);
			do {
				value(l);
				target(tW16, a, l, -1);
				longadd(a, Two); }
			while(skip());
			return;
		case W32 :	// 32 bit write
			value(a);
			do {
				value(l);
				target(tW32, a, l, -1);
				longadd(a, Four); }
			while(skip());
			return;
		case VAR :
			get_symbol(temp);
			longset(l, gtype[Gtop] = 0);
			if(skip()) value(l);
		dovar:
			if((i = lookup(temp, gvars, Gtop)) != -1) {
				if(!parms) error("Already defined: %s", temp);
				j = (stack[parms] * 2) + 1;
				j = stack[parms-j];
				if(i >= j) error("Already defined: %s", temp); }
			gvars[Gtop] = add_pool(temp);
			longcpy(gvalue[Gtop++], l);
		domark:
			if(FP < 3)
				mark();
			return;
		case CONST :
			get_symbol(temp);
			value(l);
			gtype[Gtop] = 1;
			goto dovar;
/*		case SET :
			get_symbol(temp);
			if((i = lookup(temp, gvars, Gtop)) == -1)
				error("Not found");
			if(gtype[i]) error("Wrong symbol type");
			value(l);
			longcpy(gvalue[i], l);
			return; */
		case PRINT :
			skip();
			j = 'W';
		printnext: switch(i = *pptr) {
			case 0 :	// End of line
				wputc('\n');
				return;
			case '{' :
				++pptr;
				j = toupper(*pptr++);
				if(*pptr++ != '}') j = 0;
				switch(j) {
				case 'B' :
				case 'W' :
				case 'H' :
				case 'D' : goto printnext; }
				error("Bad print format");
			case ' ' :
			case '\t' :
				wputc(*pptr++);
				goto printnext;
			case '\'' :
			case '"' :
			case '/' :
				++pptr;
				while(*pptr != i) {
					if(!*pptr)
						error("Unmatched: %c\n", i);
					wputc(*pptr++); }
				++pptr;
				goto printnext; }
			value(l);
			while(isspace(*(pptr-1))) --pptr;
			switch(j) {
			case 'B' : wprintf("%02x", *(unsigned char*)l); goto printnext;
			case 'H' : wprintf("%04x", l[0]); goto printnext;
			case 'D' : ltoa(l, temp, 10); wprintf("%s", temp); goto printnext; }
			wprintf("%04x.%04x", l[1], l[0]); goto printnext;
		case FUNC :
			get_symbol(temp);
			define_func(temp);
			goto domark;
		case EXIT :
			if(fp)
				fclose(fp);
			FP = fp = 0;
			return;
		case GOTO :
			testfunc();
			parse_symbol(xtemp);
			FP = Fbase;
			for(;;) {
				i = FP;
				if(!get_line(pptr = line, ""))
					error("Not found: %s", xtemp);
				parse_symbol(temp);
				if(skip() != ':')
					continue;
				if(!strcmp(temp, xtemp)) {
					FP = i;
					return; } }
		case DELAY :
			value(l);
			delay(l[0]);
			return;
		case LR8 :
		case LR16 :
		case LR32 :
			value(a);
			status("%s %04x.%04x - Press ESC to stop", keywords[i], a[1], a[0]);
			if(Tblocks[i -= (LR8-tLR8)])
				target(i, a, temp, 0);
			else {
				do target(i-(tLR8-tR8), a, temp, 0);
				while(wtstc() != 0x1B); }
			update_status();
			return;
		case LW8 :
		case LW16 :
		case LW32 :
			value(a);
			value(l);
			switch(i) {
			case LW8 :	sprintf(temp, "%02x", *(unsigned char*)l);	break;
			case LW16 :	sprintf(temp, "%04x", *l);					break;
			default:	sprintf(temp, "%04x.%04x", l[1], l[0]); }
			status("%s %04x.%04x = %s - Press ESC to stop", keywords[i], a[1], a[0], temp);
			if(Tblocks[i -= (LW8 - tLW8)])
				target(i, a, l, -1);
			else {
				do target(i-(tLW8-tW8), a, l, -1);
				while(wtstc() != 0x1B); }
			update_status();
			return;
		case KEY :
			value(l);
			i = *l-1;
			if(l[1] || (i > 39))
				error("Key must be 1-.40");
			parse(xtemp);
			if(strlen(xtemp) > 8)
				error("Name must be <= 7 chars");
			strcpy(fnames[i], xtemp);			
			skip();
			strcpy(fkeys[i], pptr);
			return;
		case INCL :
			if(!parse(xtemp)) error("Bad filename");
			stack[SP++] = fp;
			stack[SP++] = FP;
			stack[SP++] = Func;
			stack[SP++] = Line;
			stack[SP++] = Fline;
			stack[SP++] = parms;
			if(!(fp = fopen(xtemp, "r")))
				error("Unable to open: %s", xtemp);
			FP = 2;
			Line = Fline = 0;
			j = SP;
			while(FP)
				command();
			SP = j;
			parms = stack[--SP];
			Fline = stack[--SP];
			Line = stack[--SP];
			Func = stack[--SP];
			FP = stack[--SP];
			fp = stack[--SP];
			return;
		case CLS :
			wclwin();
			return;
		case CLEAR :
			_Gtop = 0;
			_pool_top = pool_base;
			error("All Clear");
		case HELP :
			status("HELP - Press ESC to exit");
			help(3);
			update_status();
			return;
		case RELOAD :
			target(tLOAD, temp, temp, -1);
			return;
		case STOP :
			release();
			longjmp(gomain, 255);
		case VLIST:	Glist(0);	return;
		case CLIST: Glist(1);	return;
		case FLIST: Glist(2);	return;
		case DUMP :
			value(a);
			longset(l, 256);
			if(skip()) value(l);
			Daddr[1] = a[1];
			Daddr[0] = a[0] & 0xFFF0;
			Dpos = a[0] - Daddr[0];
			*line = Dflag = 0;
			while(longtst(l)) {
				if(kbtst() == _EN)
					return;
				target(tR8, a, temp, 0);
				if(!*line) {
					if(Dflag && !(*Daddr & 0xFF))
						wputc('\n');
					memset(line, 0, 80);
					sprintf(line, "%04x.%04x", Daddr[1], Daddr[0]);
					longset(temp+4, Dpos);
					longadd(Daddr, temp+4); }
				sprintf(&line[(Dpos*3)+11+(Dpos/4)], "%02x", j = *temp);
				line[Dpos+63] = isprint(j) ? j : '.';
				longadd(Daddr, One);
				if(++Dpos >= 16) {
					dump_end(line);
					Dflag = -1; }
				longadd(a, One);
				longsub(l, One); }
			dump_end(line);
			return;
		case TTY :
			status("TTY: Press END to exit");
			for(;;) {
				if(j = wtstc()) {
					i = 7;
					if(j == _KEN) {
						if(mwin->WINcurx)
							wputc('\n');
						update_status();
						return; }
					if(!(j & 0xFF00))
						Cputc((j == '\n') ? '\r' : j);
					continue; }
				if((j = Ctestc()) != -1) {
					i = 7;
					wputc(j);
					continue; }
				if(i) {
					wupdatexy();
					i = 0; } }
		} if((i = lookup(temp, gvars, Gtop)) != -1) switch(gtype[i]) {
			case 0 :	// Variable
				if(*pptr++ != '=') error("Syntax error");
				value(l);
				longcpy(gvalue[i], l);
				return;
			case 2 :
				stack[SP++] = Func;
				stack[SP++] = Line;
				stack[SP++] = Fbase;
				stack[SP++] = FP;
				stack[SP++] = pool_top;
				stack[SP++] = parms;
				stack[SP++] = Gtop;
				j = SP; k = 0;
				while(skip()) {
					value(l);
					stack[SP++] = l[0];
					stack[SP++] = l[1];
					++k; }
				parms = SP;
				stack[SP++] = k;
				Func = temp;
				FP = Fbase = gvalue[i][0];
				update_status();
				Line = 0;
				while(FP)
					command();
				SP = j;
				Gtop = stack[--SP];
				parms = stack[--SP];
				pool_top = stack[--SP];
				FP = stack[--SP];
				Fbase = stack[--SP];
				Line = stack[--SP];
				Func = stack[--SP];
				update_status();
				return;
	} error("Unknown command: %s", temp); }
}

Glist(unsigned char type)
{
	unsigned i, j, t, top;
	unsigned char *p, *p1;
	unsigned index[GVARS];

	// Build index list
	for(top=i=0; i < Gtop; ++i) {
		if(gtype[i] == type)	// Matches
			index[top++] = i; }
	p = keywords[VAR+type];
	if(!top) {
		wprintf("No %sS defined\n", p);
		return; }
	wprintf("  -- %s NAMES --\n", p);
	// Sort index list
	for(i=0; i < top; ++i) {
		p = gvars[index[i]];
		t = -1;
		for(j=i+1; j < top; ++j) {
			if(strcmp(p1 = gvars[index[j]], p) > 0) {
				p = p1;
				t = j; } }
		if(t != -1) {
			j = index[i];
			index[i] = index[t];
			index[t] = j; } }
	// Display index list
	t = 0;
	while(top) {
		i = strlen(p = gvars[index[--top]]);
		if((80-t) <= i) {
			wputc('\n');
			t = 0; }
		wputs(p);
		t += i;
		do {
			if(t >= 80) {
				t = 0;
				break; }
			wputc(' '); }
		while(t++ % 8); }
	if(t)
		wputc('\n');
}
