/*
 * Tables/Functions to implement the 6502 cross disassembler
 */

#define	XBIT16	0x10
#define	PCREL	0x20

/* Text strings that the disassembler needs */
char _cpu_[]	= "6502";		/* Name of CPU */
char _cmd_[]	= "DIS02";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

unsigned char inames[] = {
	'A','D','C',	/* 80 */
	'A','N','D',	/* 81 */
	'A','S','L',	/* 82 */
	'B','I','T',	/* 83 */
	'C','M','P',	/* 84 */
	'C','P','X',	/* 85 */
	'C','P','Y',	/* 86 */
	'D','E','C',	/* 87 */
	'E','O','R',	/* 88 */
	'I','N','C',	/* 89 */
	'J','M','P',	/* 8A */
	'L','D','A',	/* 8B */
	'L','D','X',	/* 8C */
	'L','D','Y',	/* 8D */
	'L','S','R',	/* 8E */
	'O','R','A',	/* 8F */
	'R','O','L',	/* 90 */
	'R','O','R',	/* 91 */
	'S','B','C',	/* 92 */
	'S','T','A',	/* 93 */
	'S','T','X',	/* 94 */
	'S','T','Y'		/* 95 */
};

/* Table of 6502 opcodes: OPCODE, FLAG/LENGTH, TEXT */
/* FLAG = 0x80 if second opcode byte required. */
unsigned char itable[] = {
	0x69,0x02,"\x80 #\xC1",		/* ADC */
	0x6D,0x03,"\x80 \xD1",		/* ADC */
	0x65,0x02,"\x80 \xC1",		/* ADC */
	0x75,0x02,"\x80 \xC1,X",		/* ADC */
	0x7D,0x03,"\x80 \xD1,X",		/* ADC */
	0x79,0x03,"\x80 \xD1,Y",		/* ADC */
	0x61,0x02,"\x80 [\xC1,X]",		/* ADC */
	0x71,0x02,"\x80 [\xC1],Y",		/* ADC */
	0x29,0x02,"\x81 #\xC1",		/* AND */
	0x2D,0x03,"\x81 \xD1",		/* AND */
	0x25,0x02,"\x81 \xC1",		/* AND */
	0x35,0x02,"\x81 \xC1,X",		/* AND */
	0x3D,0x03,"\x81 \xD1,X",		/* AND */
	0x39,0x03,"\x81 \xD1,Y",		/* AND */
	0x21,0x02,"\x81 [\xC1,X]",		/* AND */
	0x31,0x02,"\x81 [\xC1],Y",		/* AND */
	0x0E,0x03,"\x82 \xD1",		/* ASL */
	0x06,0x02,"\x82 \xC1",		/* ASL */
	0x0A,0x01,"\x82 A",		/* ASL */
	0x16,0x02,"\x82 \xC1,X",		/* ASL */
	0x1E,0x03,"\x82 \xD1,X",		/* ASL */
	0x90,0x02,"BCC \xE1",
	0xB0,0x02,"BCS \xE1",
	0xF0,0x02,"BEQ \xE1",
	0x30,0x02,"BMI \xE1",
	0xD0,0x02,"BNE \xE1",
	0x10,0x02,"BPL \xE1",
	0x50,0x02,"BVC \xE1",
	0x70,0x02,"BVS \xE1",
	0x00,0x01,"BRK",
	0x2C,0x03,"\x83 \xD1",		/* BIT */
	0x24,0x02,"\x83 \xC1",		/* BIT */
	0x18,0x01,"CLC",
	0xD8,0x01,"CLD",
	0x58,0x01,"CLI",
	0xB8,0x01,"CLV",
	0x38,0x01,"SEC",
	0xF8,0x01,"SED",
	0x78,0x01,"SEI",
	0xEA,0x01,"NOP",
	0xC9,0x02,"\x84 #\xC1",		/* CMP */
	0xC5,0x02,"\x84 \xC1",		/* CMP */
	0xCD,0x03,"\x84 \xD1",		/* CMP */
	0xD5,0x02,"\x84 \xC1,X",		/* CMP */
	0xDD,0x03,"\x84 \xD1,X",		/* CMP */
	0xD9,0x03,"\x84 \xD1,Y",		/* CMP */
	0xC1,0x02,"\x84 [\xC1,X]",		/* CMP */
	0xD1,0x02,"\x84 [\xC1],Y",		/* CMP */
	0xE0,0x02,"\x85 #\xC1",		/* CPX */
	0xE4,0x02,"\x85 \xC1",		/* CPX */
	0xEC,0x03,"\x85 \xD1",		/* CPX */
	0xC0,0x02,"\x86 #\xC1",		/* CPY */
	0xC4,0x02,"\x86 \xC1",		/* CPY */
	0xCC,0x03,"\x86 \xD1",		/* CPY */
	0xC6,0x02,"\x87 \xC1",		/* DEC */
	0xCE,0x03,"\x87 \xD1",		/* DEC */
	0xD6,0x02,"\x87 \xC1,X",		/* DEC */
	0xDE,0x03,"\x87 \xD1,x",		/* DEC */
	0xCA,0x01,"DEX",
	0x88,0x01,"DEY",
	0x49,0x02,"\x88 #\xC1",		/* EOR */
	0x45,0x02,"\x88 \xC1",		/* EOR */
	0x4D,0x03,"\x88 \xD1",		/* EOR */
	0x55,0x02,"\x88 \xC1,X",		/* EOR */
	0x5D,0x03,"\x88 \xD1,X",		/* EOR */
	0x59,0x03,"\x88 \xD1,Y",		/* EOR */
	0x41,0x02,"\x88 [\xC1,X]",		/* EOR */
	0x51,0x02,"\x88 [\xC1],Y",		/* EOR */
	0xE6,0x02,"\x89 \xC1",		/* INC */
	0xEE,0x03,"\x89 \xD1",		/* INC */
	0xF6,0x02,"\x89 \xC1,X",		/* INC */
	0xFE,0x03,"\x89 \xD1,X",		/* INC */
	0xE8,0x01,"INX",
	0xC8,0x01,"INY",
	0x4C,0x03,"\x8A \xD1",		/* JMP */
	0x6C,0x03,"\x8A [\xD1]",		/* JMP */
	0x20,0x03,"JSR \xD1",
	0xA9,0x02,"\x8B #\xC1",		/* LDA */
	0xA5,0x02,"\x8B \xC1",		/* LDA */
	0xAD,0x03,"\x8B \xD1",		/* LDA */
	0xB5,0x02,"\x8B \xC1,X",		/* LDA */
	0xBD,0x03,"\x8B \xD1,X",		/* LDA */
	0xB9,0x03,"\x8B \xD1,Y",		/* LDA */
	0xA1,0x02,"\x8B [\xC1,X]",		/* LDA */
	0xB1,0x02,"\x8B [\xC1],Y",		/* LDA */
	0xA2,0x02,"\x8C #\xC1",		/* LDX */
	0xA6,0x02,"\x8C \xC1",		/* LDX */
	0xAE,0x03,"\x8C \xD1",		/* LDX */
	0xB6,0x02,"\x8C \xC1,Y",		/* LDX */
	0xBE,0x03,"\x8C \xD1,Y",		/* LDX */
	0xA0,0x02,"\x8D #\xC1",		/* LDY */
	0xA4,0x02,"\x8D \xC1",		/* LDY */
	0xAC,0x03,"\x8D \xD1",		/* LDY */
	0xB4,0x02,"\x8D \xC1,X",		/* LDY */
	0xBC,0x03,"\x8D \xD1,X",		/* LDY */
	0x4A,0x01,"\x8E A",		/* LSR */
	0x46,0x02,"\x8E \xC1",		/* LSR */
	0x4E,0x03,"\x8E \xD1",		/* LSR */
	0x56,0x02,"\x8E \xC1,X",		/* LSR */
	0x5E,0x03,"\x8E \xD1,X",		/* LSR */
	0x09,0x02,"\x8F #\xC1",		/* ORA */
	0x05,0x02,"\x8F \xC1",		/* ORA */
	0x0D,0x03,"\x8F \xD1",		/* ORA */
	0x15,0x02,"\x8F \xC1,X",		/* ORA */
	0x1D,0x03,"\x8F \xD1,X",		/* ORA */
	0x19,0x03,"\x8F \xD1,Y",		/* ORA */
	0x01,0x02,"\x8F [\xC1,X]",		/* ORA */
	0x11,0x02,"\x8F [\xC1],Y",		/* ORA */
	0x48,0x01,"PHA",
	0x08,0x01,"PHP",
	0x68,0x01,"PLA",
	0x28,0x01,"PLP",
	0x2A,0x01,"\x90 A",		/* ROL */
	0x26,0x02,"\x90 \xC1",		/* ROL */
	0x2E,0x03,"\x90 \xD1",		/* ROL */
	0x36,0x02,"\x90 \xC1,X",		/* ROL */
	0x3E,0x03,"\x90 \xD1,X",		/* ROL */
	0x6A,0x01,"\x91 A",		/* ROR */
	0x66,0x02,"\x91 \xC1",		/* ROR */
	0x6E,0x03,"\x91 \xD1",		/* ROR */
	0x76,0x02,"\x91 \xC1,X",		/* ROR */
	0x7E,0x03,"\x91 \xD1,X",		/* ROR */
	0x40,0x01,"RTI",
	0x60,0x01,"RTS",
	0xE9,0x02,"\x92 #\xC1",		/* SBC */
	0xE5,0x02,"\x92 \xC1",		/* SBC */
	0xED,0x03,"\x92 \xD1",		/* SBC */
	0xF5,0x02,"\x92 \xC1,X",		/* SBC */
	0xFD,0x03,"\x92 \xD1,X",		/* SBC */
	0xF9,0x03,"\x92 \xD1,Y",		/* SBC */
	0xE1,0x02,"\x92 [\xC1,X]",		/* SBC */
	0xF1,0x02,"\x92 [\xC1],Y",		/* SBC */
	0x85,0x02,"\x93 \xC1",		/* STA */
	0x8D,0x03,"\x93 \xD1",		/* STA */
	0x95,0x02,"\x93 \xC1,X",		/* STA */
	0x9D,0x03,"\x93 \xD1,X",		/* STA */
	0x99,0x03,"\x93 \xD1,Y",		/* STA */
	0x81,0x02,"\x93 [\xC1,X]",		/* STA */
	0x91,0x02,"\x93 [\xC1],Y",		/* STA */
	0x86,0x02,"\x94 \xC1",		/* STX */
	0x8E,0x03,"\x94 \xD1",		/* STX */
	0x96,0x02,"\x94 \xC1,Y",		/* STX */
	0x84,0x02,"\x95 \xC1",		/* STY */
	0x8C,0x03,"\x95 \xD1",		/* STY */
	0x94,0x02,"\x95 \xC1,X",		/* STY */
	0xAA,0x01,"TAX",
	0xA8,0x01,"TAY",
	0xBA,0x01,"TSX",
	0x8A,0x01,"TXA",
	0x9A,0x01,"TXS",
	0x98,0x01,"TYA",

	/*  This entry always matches invalid opcodes */
	0xFF, 0x00, "FCB \xC0" };

unsigned char opcode, opcode1, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 6502 uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, type;
	unsigned char c, *ptr;
	char d;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {	/* 0 length = FCB */
			length = 1;
			return; }
		++inst_ptr;		/* Skip possible second byte */
		while(*inst_ptr++); }
	length = (type = *inst_ptr++) & 0x07;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	if(dflag) {
		ptr = inst_ptr;
		while(*ptr != ' ')
			if(!*ptr++)
				return;
		while(c = *ptr++) if((c & 0xC0) == 0xC0) {
			if(c & PCREL) {
				if(c & XBIT16) {
					set_symbol(get_word(c & 0x07)+address+2, BIT16);
					continue; }
				else {
					d = memory[c & 0x07];
					set_symbol((address+2)+(int)d, BIT16);
					continue; } }
			if(c & XBIT16) {
				set_symbol(get_word(c & 0x07), BIT16);
				continue; }
			if(bflag)
				set_symbol(memory[c & 0x07], BIT8); } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	unsigned char c, *ptr;
	char d;

	/* Display instruction name */
	i = 0;
	while((c = *inst_ptr++) != ' ') {
		if(!c)
			return i;
		if(c & 0x80) {
			ptr = inames + (c & 0x7F)*3;
			putc(*ptr++, stdout);
			putc(*ptr++, stdout);
			i += 2;
			c = *ptr; }
		++i;
		putc(c, stdout); }

	/* Space if operands */
	do
		putc(' ', stdout);
	while(++i < 8);

	/* Display operands */
	while(c = *inst_ptr++) {
		if(c & 0x80) {
			if(c & PCREL) {
				if(c & XBIT16) {
					i += printv(get_word(c & 0x07)+address+2, BIT16);
					continue; }
				else {
					d = memory[c & 0x07];
					i += printv((address+2)+(int)d, BIT16);
					continue; } }
			if(c & XBIT16) {
				i += printv(get_word(c & 0x07), BIT16);
				continue; }
			i += printv(memory[c & 0x07], BIT8);
			continue; }
		++i;
		putc(c, stdout); }

	return i;
}
