/*
 * Tables/Functions to implement the 6800 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "6800";		/* Name of CPU */
char _cmd_[]	= "DIS00";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

/* Table of instruction names */
unsigned char inames[] = {
	'A','D','C',	/* 80 */
	'A','D','D',	/* 81 */
	'A','N','D',	/* 82 */
	'A','S','L',	/* 83 */
	'A','S','R',	/* 84 */
	'B','I','T',	/* 85 */
	'C','L','R',	/* 86 */
	'C','M','P',	/* 87 */
	'C','O','M',	/* 88 */
	'C','P','X',	/* 89 */
	'D','E','C',	/* 8A */
	'E','O','R',	/* 8B */
	'I','N','C',	/* 8C */
	'J','M','P',	/* 8D */
	'J','S','R',	/* 8E */
	'L','D','A',	/* 8F */
	'L','D','S',	/* 90 */
	'L','S','R',	/* 91 */
	'N','E','G',	/* 92 */
	'O','R','A',	/* 93 */
	'P','S','H',	/* 94 */
	'P','U','L',	/* 95 */
	'R','O','R',	/* 96 */
	'R','O','L',	/* 97 */
	'S','B','C',	/* 98 */
	'S','T','A',	/* 99 */
	'S','T','S',	/* 9A */	
	'S','U','B',	/* 9B */
	'T','S','T',	/* 9C */
	'L','D','X',	/* 9D */
	'S','T','X'		/* 9E */
};

/* Table of 6800 opcodes: OPCODE, SYMBOL-FLAGS/LENGTH, TEXT */
/* SYMBOL-FLAGS: 40=Relative 20=Word 10=Byte */
unsigned char itable[] = {
	0x01, 0x01, "NOP",
	0x06, 0x01, "TAP",
	0x07, 0x01, "TPA",
	0x08, 0x01, "INX",
	0x09, 0x01, "DEX",
	0x0A, 0x01, "CLV",
	0x0B, 0x01, "SEV",
	0x0C, 0x01, "CLC",
	0x0D, 0x01, "SEC",
	0x0E, 0x01, "CLI",
	0x0F, 0x01, "SEI",

	0x10, 0x01, "SBA",
	0x11, 0x01, "CBA",
	0x16, 0x01, "TAB",
	0x17, 0x01, "TBA",
	0x19, 0x01, "DAA",
	0x1B, 0x01, "ABA",

	0x20, 0x42, "BRA r",
	0x22, 0x42, "BHI r",
	0x23, 0x42, "BLS r",
	0x24, 0x42, "BCC r",
	0x25, 0x42, "BCS r",
	0x26, 0x42, "BNE r",
	0x27, 0x42, "BEQ r",
	0x28, 0x42, "BVC r",
	0x29, 0x42, "BVS r",
	0x2A, 0x42, "BPL r",
	0x2B, 0x42, "BMI r",
	0x2C, 0x42, "BGE r",
	0x2D, 0x42, "BLT r",
	0x2E, 0x42, "BGT r",
	0x2F, 0x42, "BLE r",

	0x30, 0x01, "TSX",
	0x31, 0x01, "INS",
	0x32, 0x01, "\x95A",		/* PULA */
	0x33, 0x01, "\x95B",		/* PULB */
	0x34, 0x01, "DES",
	0x35, 0x01, "TXS",
	0x36, 0x01, "\x94A",		/* PSHA */
	0x37, 0x01, "\x94B",		/* PSHB */
	0x39, 0x01, "RTS",
	0x3B, 0x01, "RTI",
	0x3E, 0x01, "WAI",
	0x3F, 0x01, "SWI",

	0x40, 0x01, "\x92A",		/* NEGA */
	0x43, 0x01, "\x88A",		/* COMA */
	0x44, 0x01, "\x91A",		/* LSRA */
	0x46, 0x01, "\x96A",		/* RORA */
	0x47, 0x01, "\x84A",		/* ASRA */
	0x48, 0x01, "\x83A",		/* ASLA */
	0x49, 0x01, "\x97A",		/* ROLA */
	0x4A, 0x01, "\x8AA",		/* DECA */
	0x4C, 0x01, "\x8CA",		/* INCA */
	0x4D, 0x01, "\x9CA",		/* TSTA */
	0x4F, 0x01, "\x86A",		/* CLRA */

	0x50, 0x01, "\x92B",		/* NEGB */
	0x53, 0x01, "\x88B",		/* COMB */
	0x54, 0x01, "\x91B",		/* LSRB */
	0x56, 0x01, "\x96B",		/* RORB */
	0x57, 0x01, "\x84B",		/* ASRB */
	0x58, 0x01, "\x83B",		/* ASLB */
	0x59, 0x01, "\x97B",		/* ROLB */
	0x5A, 0x01, "\x8AB",		/* DECB */
	0x5C, 0x01, "\x8CB",		/* INCB */
	0x5D, 0x01, "\x9CB",		/* TSTB */
	0x5F, 0x01, "\x86B",		/* CLRB */

	0x60, 0x12, "\x92 b,X",		/* NEG ,X */
	0x63, 0x12, "\x88 b,X",		/* COM ,X */
	0x64, 0x12, "\x91 b,X",		/* LSR ,X */
	0x66, 0x12, "\x96 b,X",		/* ROR ,X */
	0x67, 0x12, "\x84 b,X",		/* ASR ,X */
	0x68, 0x12, "\x83 b,X",		/* ASL ,X */
	0x69, 0x12, "\x97 b,X",		/* ROL ,X */
	0x6A, 0x12, "\x8A b,X",		/* DEC ,X */
	0x6C, 0x12, "\x8C b,X",		/* INC ,X */
	0x6D, 0x12, "\x9C b,X",		/* TST ,X */
	0x6E, 0x12, "\x8D b,X",		/* JMP ,X */
	0x6F, 0x12, "\x86 b,X",		/* CLR ,X */

	0x70, 0x23, "\x92 w",		/* NEG ext */
	0x73, 0x23, "\x88 w",		/* COM ext */
	0x74, 0x23, "\x91 w",		/* LSR ext */
	0x76, 0x23, "\x96 w",		/* ROR ext */
	0x77, 0x23, "\x84 w",		/* ASR ext */
	0x78, 0x23, "\x83 w",		/* ASL ext */
	0x79, 0x23, "\x97 w",		/* ROL ext */
	0x7A, 0x23, "\x8A w",		/* DEC ext */
	0x7C, 0x23, "\x8C w",		/* INC ext */
	0x7D, 0x23, "\x9C w",		/* TST ext */
	0x7E, 0x23, "\x8D w",		/* JMP ext */
	0x7F, 0x23, "\x86 w",		/* CLR ext */

	0x80, 0x12, "\x9BA #b",		/* SUBA # */
	0x81, 0x12, "\x87A #b",		/* CMPA # */
	0x82, 0x12, "\x98A #b",		/* SBCA # */
	0x84, 0x12, "\x82A #b",		/* ANDA # */
	0x85, 0x12, "\x85A #b",		/* BITA # */
	0x86, 0x12, "\x8FA #b",		/* LDAA # */
	0x88, 0x12, "\x8BA #b",		/* EORA # */
	0x89, 0x12, "\x80A #b",		/* ADCA # */
	0x8A, 0x12, "\x93A #b",		/* ORAA # */
	0x8B, 0x12, "\x81A #b",		/* ADDA # */
	0x8C, 0x23, "\x89 #w",		/* CPX # */
	0x8D, 0x42, "BSR r",
	0x8E, 0x23, "\x90 #w",		/* LDS # */

	0x90, 0x12, "\x9BA b",		/* SUBA dir */
	0x91, 0x12, "\x87A b",		/* CMPA dir */
	0x92, 0x12, "\x98A b",		/* SBCA dir */
	0x94, 0x12, "\x82A b",		/* ANDA dir */
	0x95, 0x12, "\x85A b",		/* BITA dir */
	0x96, 0x12, "\x8FA b",		/* LDAA dir */
	0x97, 0x12, "\x99A b",		/* STAA dir */
	0x98, 0x12, "\x8BA b",		/* EORA dir */
	0x99, 0x12, "\x80A b",		/* ADCA dir */
	0x9A, 0x12, "\x93A b",		/* ORAA dir */
	0x9B, 0x12, "\x81A b",		/* ADDA dir */
	0x9C, 0x12, "\x89 b",		/* CPX dir */
	0x9E, 0x12, "\x90 b",		/* LDS dir */
	0x9F, 0x12, "\x9A b",		/* STS dir */

	0xA0, 0x12, "\x9BA b,X",	/* SUBA ,X */
	0xA1, 0x12, "\x87A b,X",	/* CMPA ,X */
	0xA2, 0x12, "\x98A b,X",	/* SBCA ,X */
	0xA4, 0x12, "\x82A b,X",	/* ANDA ,X */
	0xA5, 0x12, "\x85A b,X",	/* BITA ,X */
	0xA6, 0x12, "\x8FA b,X",	/* LDAA ,X */
	0xA7, 0x12, "\x99A b,X",	/* STAA ,X */
	0xA8, 0x12, "\x8BA b,X",	/* EORA ,X */
	0xA9, 0x12, "\x80A b,X",	/* ADCA ,X */
	0xAA, 0x12, "\x93A b,X",	/* ORAA ,X */
	0xAB, 0x12, "\x81A b,X",	/* ADDA ,X */
	0xAC, 0x12, "\x89 b,X",		/* CPX ,X */
	0xAD, 0x12, "\x8E b,X",		/* JSR ,X */
	0xAE, 0x12,	"\x90 b,X",		/* LDS ,X */
	0xAF, 0x12, "\x9A b,X",		/* STS ,X */

	0xB0, 0x23, "\x9BA w",		/* SUBA ext */
	0xB1, 0x23, "\x87A w",		/* CMPA ext */
	0xB2, 0x23, "\x98A w",		/* SBCA ext */
	0xB4, 0x23, "\x82A w",		/* ANDA ext */
	0xB5, 0x23, "\x85A w",		/* BITA ext */
	0xB6, 0x23, "\x8FA w",		/* LDAA ext */
	0xB7, 0x23, "\x99A w",		/* STAA ext */
	0xB8, 0x23, "\x8BA w",		/* EORA ext */
	0xB9, 0x23, "\x80A w",		/* ADCA ext */
	0xBA, 0x23, "\x93A w",		/* ORAA ext */
	0xBB, 0x23, "\x81A w",		/* ADDA ext */
	0xBC, 0x23, "\x89 w",		/* CPX ext */
	0xBD, 0x23, "\x8E w",		/* JSR ext */
	0xBE, 0x23,	"\x90 w",		/* LDS ext */
	0xBF, 0x23, "\x9A w",		/* STS ext */

	0xC0, 0x12, "\x9BB #b",		/* SUBB # */
	0xC1, 0x12, "\x87B #b",		/* CMPB # */
	0xC2, 0x12, "\x98B #b",		/* SBCB # */
	0xC4, 0x12, "\x82B #b",		/* ANDB # */
	0xC5, 0x12, "\x85B #b",		/* BITB # */
	0xC6, 0x12, "\x8FB #b",		/* LDAB # */
	0xC8, 0x12, "\x8BB #b",		/* EORB # */
	0xC9, 0x12, "\x80B #b",		/* ADCB # */
	0xCA, 0x12, "\x93B #b",		/* ORAB # */
	0xCB, 0x12, "\x81B #b",		/* ADDB # */
	0xCE, 0x23, "\x9D #w",		/* LDX # */

	0xD0, 0x12, "\x9BB b",		/* SUBB dir */
	0xD1, 0x12, "\x87B b",		/* CMPB dir */
	0xD2, 0x12, "\x98B b",		/* SBCB dir */
	0xD4, 0x12, "\x82B b",		/* ANDB dir */
	0xD5, 0x12, "\x85B b",		/* BITB dir */
	0xD6, 0x12, "\x8FB b",		/* LDAB dir */
	0xD7, 0x12, "\x99B b",		/* STAB dir */
	0xD8, 0x12, "\x8BB b",		/* EORB dir */
	0xD9, 0x12, "\x80B b",		/* ADCB dir */
	0xDA, 0x12, "\x93B b",		/* ORAB dir */
	0xDB, 0x12, "\x81B b",		/* ADDB dir */
	0xDE, 0x12, "\x9D b",		/* LDX dir */
	0xDF, 0x12, "\x9E b",		/* STX dir */

	0xE0, 0x12, "\x9BB b,X",	/* SUBB ,X */
	0xE1, 0x12, "\x87B b,X",	/* CMPB ,X */
	0xE2, 0x12, "\x98B b,X",	/* SBCB ,X */
	0xE4, 0x12, "\x82B b,X",	/* ANDB ,X */
	0xE5, 0x12, "\x85B b,X",	/* BITB ,X */
	0xE6, 0x12, "\x8FB b,X",	/* LDAB ,X */
	0xE7, 0x12, "\x99B b,X",	/* STAB ,X */
	0xE8, 0x12, "\x8BB b,X",	/* EORB ,X */
	0xE9, 0x12, "\x80B b,X",	/* ADCB ,X */
	0xEA, 0x12, "\x93B b,X",	/* ORAB ,X */
	0xEB, 0x12, "\x81B b,X",	/* ADDB ,X */
	0xEE, 0x12, "\x9D b,X",		/* LDX ,X */
	0xEF, 0x12, "\x9E b,X",		/* STX ,X */

	0xF0, 0x23, "\x9BB w",		/* SUBB ext */
	0xF1, 0x23, "\x87B w",		/* CMPB ext */
	0xF2, 0x23, "\x98B w",		/* SBCB ext */
	0xF4, 0x23, "\x82B w",		/* ANDB ext */
	0xF5, 0x23, "\x85B w",		/* BITB ext */
	0xF6, 0x23, "\x8FB w",		/* LDAB ext */
	0xF7, 0x23, "\x99B w",		/* STAB ext */
	0xF8, 0x23, "\x8BB w",		/* EORB ext */
	0xF9, 0x23, "\x80B w",		/* ADCB ext */
	0xFA, 0x23, "\x93B w",		/* ORAB ext */
	0xFB, 0x23, "\x81B w",		/* ADDB ext */
	0xFE, 0x23, "\x9D w",		/* LDX ext */
	0xFF, 0x23, "\x9E w",		/* STX ext */

	/*  This entry always matches invalid opcodes */
	0xFF, 0x00, 'F','C','B',' ','o',0 };

unsigned char opcode, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 6800 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, type;
	char c;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {	/* 0 length = FCB */
			length = 1;
			return; }
		while(*inst_ptr++); }
	length = (type = *inst_ptr++) & 0x0F;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	/* Add any operand to symbol table */
	if(dflag) {
		if(type & 0x40) {			/* Relative symbol */
			c = memory[1];
			set_symbol((address+2)+(int)c, BIT16);
			return; }
		if(type & 0x20) {			/* Word symbol */
			set_symbol(get_word(1), BIT16);
			return; }
		if((type & 0x10) && bflag)	/* Byte symbol */
			set_symbol(memory[1], BIT8); }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	char c, *ptr;

	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case 'b' :		/* Byte operand */
			i += printv(memory[1], BIT8);
			break;
		case 'w' :		/* Word operand */
			i += printv(get_word(1), BIT16);
			break;
		case 'r' :		/* Relative address */
			c = memory[1];
			i += printv((address+2)+(int)c, BIT16);
			break;
		default:
			if(c & 0x80) {
				ptr = inames + (c & 0x7F)*3;
				putc(*ptr++, stdout);
				putc(*ptr++, stdout);
				i += 2;
				c = *ptr; }
			++i;
			putc(c, stdout); }

	return i;
}
