/*
 * Tables/Functions to implement the 6805 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "6805";		/* Name of CPU */
char _cmd_[]	= "DIS05";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

/* Table of instruction names and substrings */
unsigned char inames[] = {
	'J','M','P',	/* 80 */
	'J','S','R',	/* 81 */
	'A','D','C',	/* 82 */
	'A','D','D',	/* 83 */
	'A','N','D',	/* 84 */
	'B','I','T',	/* 85 */
	'C','M','P',	/* 86 */
	'C','P','X',	/* 87 */
	'E','O','R',	/* 88 */
	'L','D','A',	/* 89 */
	'L','D','X',	/* 8A */
	'O','R','A',	/* 8B */
	'S','B','C',	/* 8C */
	'S','T','A',	/* 8D */
	'S','T','X',	/* 8E */
	'S','U','B',	/* 8F */
	'A','S','R',	/* 90 */
	'C','L','R',	/* 91 */
	'C','O','M',	/* 92 */
	'I','N','C',	/* 93 */
	'D','E','C',	/* 94 */
	'L','S','L',	/* 95 */
	'L','S','R',	/* 96 */
	'N','E','G',	/* 97 */
	'R','O','L',	/* 98 */
	'R','O','R',	/* 99 */
	'T','S','T',	/* 9A */
	'B','H','C',	/* 9B */
	'S','E','T'		/* 9C */
};

/* Table of 6805 opcodes and instructions: opcode, type/length, text */
/* type: 00=None, 10=8BitData, 20=8BitAddr, 30=16Bit, 40=8BitRel */
/*		 50=8BitX 60=16BitX */
unsigned char itable[] = {
	0xBC, 0x22, "\x80 <b",		/* JMP */
	0xCC, 0x33, "\x80 w",		/* JMP */
	0xDC, 0x33, "\x80 w,X",		/* JMP */
	0xEC, 0x12, "\x80 b,X",		/* JMP */
	0xFC, 0x01, "\x80 ,X",		/* JMP */

	0xBD, 0x22, "\x81 <b",		/* JSR */
	0xCD, 0x33, "\x81 w",		/* JSR */
	0xDD, 0x33, "\x81 w,X",		/* JSR */
	0xED, 0x12, "\x81 b,X",		/* JSR */
	0xFD, 0x01, "\x81 ,X",		/* JSR */

	0xA9, 0x12, "\x82 #b",		/* ADC */
	0xB9, 0x22, "\x82 <b",		/* ADC */
	0xC9, 0x33, "\x82 w",		/* ADC */
	0xD9, 0x33, "\x82 w,X",		/* ADC */
	0xE9, 0x12, "\x82 b,X",		/* ADC */
	0xF9, 0x01, "\x82 ,X",		/* ADC */

	0xAB, 0x12, "\x83 #b",		/* ADD */
	0xBB, 0x22, "\x83 <b",		/* ADD */
	0xCB, 0x33, "\x83 w",		/* ADD */
	0xDB, 0x33, "\x83 w,X",		/* ADD */
	0xEB, 0x12, "\x83 b,X",		/* ADD */
	0xFB, 0x01, "\x83 ,X",		/* ADD */

	0xA4, 0x12, "\x84 #b",		/* AND */
	0xB4, 0x22, "\x84 <b",		/* AND */
	0xC4, 0x33, "\x84 w",		/* AND */
	0xD4, 0x33, "\x84 w,X",		/* AND */
	0xE4, 0x12, "\x84 b,X",		/* AND */
	0xF4, 0x01, "\x84 ,X",		/* AND */

	0xA5, 0x12, "\x85 #b",		/* BIT */
	0xB5, 0x22, "\x85 <b",		/* BIT */
	0xC5, 0x33, "\x85 w",		/* BIT */
	0xD5, 0x33, "\x85 w,X",		/* BIT */
	0xE5, 0x12, "\x85 b,X",		/* BIT */
	0xF5, 0x01, "\x85 ,X",		/* BIT */

	0xA1, 0x12, "\x86 #b",		/* CMP */
	0xB1, 0x22, "\x86 <b",		/* CMP */
	0xC1, 0x33, "\x86 w",		/* CMP */
	0xD1, 0x33, "\x86 w,X",		/* CMP */
	0xE1, 0x12, "\x86 b,X",		/* CMP */
	0xF1, 0x01, "\x86 ,X",		/* CMP */

	0xA3, 0x12, "\x87 #b",		/* CPX */
	0xB3, 0x22, "\x87 <b",		/* CPX */
	0xC3, 0x33, "\x87 w",		/* CPX */
	0xD3, 0x33, "\x87 w,X",		/* CPX */
	0xE3, 0x12, "\x87 b,X",		/* CPX */
	0xF3, 0x01, "\x87 ,X",		/* CPX */

	0xA8, 0x12, "\x88 #b",		/* EOR */
	0xB8, 0x22, "\x88 <b",		/* EOR */
	0xC8, 0x33, "\x88 w",		/* EOR */
	0xD8, 0x33, "\x88 w,X",		/* EOR */
	0xE8, 0x12, "\x88 b,X",		/* EOR */
	0xF8, 0x01, "\x88 ,X",		/* EOR */

	0xA6, 0x12, "\x89 #b",		/* LDA */
	0xB6, 0x22, "\x89 <b",		/* LDA */
	0xC6, 0x33, "\x89 w",		/* LDA */
	0xD6, 0x33, "\x89 w,X",		/* LDA */
	0xE6, 0x12, "\x89 b,X",		/* LDA */
	0xF6, 0x01, "\x89 ,X",		/* LDA */

	0xAE, 0x12, "\x8A #b",		/* LDX */
	0xBE, 0x22, "\x8A <b",		/* LDX */
	0xCE, 0x33, "\x8A w",		/* LDX */
	0xDE, 0x33, "\x8A w,X",		/* LDX */
	0xEE, 0x12, "\x8A b,X",		/* LDX */
	0xFE, 0x01, "\x8A ,X",		/* LDX */

	0xAA, 0x12, "\x8B #b",		/* ORA */
	0xBA, 0x22, "\x8B <b",		/* ORA */
	0xCA, 0x33, "\x8B w",		/* ORA */
	0xDA, 0x33, "\x8B w,X",		/* ORA */
	0xEA, 0x12, "\x8B b,X",		/* ORA */
	0xFA, 0x01, "\x8B ,X",		/* ORA */

	0xA2, 0x12, "\x8C #b",		/* SBC */
	0xB2, 0x22, "\x8C <b",		/* SBC */
	0xC2, 0x33, "\x8C w",		/* SBC */
	0xD2, 0x33, "\x8C w,X",		/* SBC */
	0xE2, 0x12, "\x8C b,X",		/* SBC */
	0xF2, 0x01, "\x8C ,X",		/* SBC */

	0xB7, 0x22, "\x8D <b",		/* STA */
	0xC7, 0x33, "\x8D w",		/* STA */
	0xD7, 0x33, "\x8D w,X",		/* STA */
	0xE7, 0x12, "\x8D b,X",		/* STA */
	0xF7, 0x01, "\x8D ,X",		/* STA */

	0xBF, 0x22, "\x8E <b",		/* STX */
	0xCF, 0x33, "\x8E w",		/* STX */
	0xDF, 0x33, "\x8E w,X",		/* STX */
	0xEF, 0x12, "\x8E b,X",		/* STX */
	0xFF, 0x01, "\x8E ,X",		/* STX */

	0xA0, 0x12, "\x8F #b",		/* SUB */
	0xB0, 0x22, "\x8F <b",		/* SUB */
	0xC0, 0x33, "\x8F w",		/* SUB */
	0xD0, 0x33, "\x8F w,X",		/* SUB */
	0xE0, 0x12, "\x8F b,X",		/* SUB */
	0xF0, 0x01, "\x8F ,X",		/* SUB */

	0x37, 0x22, "\x90 <b",		/* ASR */
	0x47, 0x01, "\x90A",		/* ASRA */
	0x57, 0x01, "\x90X",		/* ASRX */
	0x67, 0x12, "\x90 b,X",		/* ASR */
	0x77, 0x01, "\x90 ,X",		/* ASR */

	0x3F, 0x22, "\x91 <b",		/* CLR */
	0x4F, 0x01, "\x91A",		/* CLRA */
	0x5F, 0x01, "\x91X",		/* CLRX */
	0x6F, 0x12, "\x91 b,X",		/* CLR */
	0x7F, 0x01, "\x91 ,X",		/* CLR */

	0x33, 0x22, "\x92 <b",		/* COM */
	0x43, 0x01, "\x92A",		/* COMA */
	0x53, 0x01, "\x92X",		/* COMX */
	0x63, 0x12, "\x92 b,X",		/* COM */
	0x73, 0x01, "\x92 ,X",		/* COM */

	0x3C, 0x22, "\x93 <b",		/* INC */
	0x4C, 0x01, "\x93A",		/* INCA */
	0x5C, 0x01, "\x93X",		/* INCX */
	0x6C, 0x12, "\x93 b,X",		/* INC */
	0x7C, 0x01, "\x93 ,X",		/* INC */

	0x3A, 0x22, "\x94 <b",		/* DEC */
	0x4A, 0x01, "\x94A",		/* DECA */
	0x5A, 0x01, "\x94X",		/* DECX */
	0x6A, 0x12, "\x94 b,X",		/* DEC */
	0x7A, 0x01, "\x94 ,X",		/* DEC */

	0x38, 0x22, "\x95 <b",		/* LSL */
	0x48, 0x01, "\x95A",		/* LSLA */
	0x58, 0x01, "\x95X",		/* LSLX */
	0x68, 0x12, "\x95 b,X",		/* LSL */
	0x78, 0x01, "\x95 ,X",		/* LSL */

	0x34, 0x22, "\x96 <b",		/* LSR */
	0x44, 0x01, "\x96A",		/* LSRA */
	0x54, 0x01, "\x96X",		/* LSRX */
	0x64, 0x12, "\x96 b,X",		/* LSR */
	0x74, 0x01, "\x96 ,X",		/* LSR */

	0x30, 0x22, "\x97 <b",		/* NEG */
	0x40, 0x01, "\x97A",		/* NEGA */
	0x50, 0x01, "\x97X",		/* NEGX */
	0x60, 0x12, "\x97 b,X",		/* NEG */
	0x70, 0x01, "\x97 ,X",		/* NEG */

	0x39, 0x22, "\x98 <b",		/* ROL */
	0x49, 0x01, "\x98A",		/* ROLA */
	0x59, 0x01, "\x98X",		/* ROLX */
	0x69, 0x12, "\x98 b,X",		/* ROL */
	0x79, 0x01, "\x98 ,X",		/* ROL */

	0x36, 0x22, "\x99 <b",		/* ROR */
	0x46, 0x01, "\x99A",		/* RORA */
	0x56, 0x01, "\x99X",		/* RORX */
	0x66, 0x12, "\x99 b,X",		/* ROR */
	0x76, 0x01, "\x99 ,X",		/* ROR */

	0x3D, 0x22, "\x9A <b",		/* TST */
	0x4D, 0x01, "\x9AA",		/* TSTA */
	0x5D, 0x01, "\x9AX",		/* TSTX */
	0x6D, 0x12, "\x9A b,X",		/* TST */
	0x7D, 0x01, "\x9A ,X",		/* TST */

	0x24, 0x42, "BCC r",
	0x25, 0x42, "BCS r",
	0x27, 0x42, "BEQ r",
	0x28, 0x42, "\x9BC r",		/* BHCC */
	0x29, 0x42, "\x9BS r",		/* BHCS */
	0x22, 0x42, "BHI r",
	0x2F, 0x42, "BIH r",
	0x2E, 0x42, "BIL r",
	0x23, 0x42, "BLS r",
	0x2C, 0x42, "BMC r",
	0x2B, 0x42, "BMI r",
	0x2D, 0x42, "BMS r",
	0x26, 0x42, "BNE r",
	0x2A, 0x42, "BPL r",
	0x20, 0x42, "BRA r",
	0x21, 0x42, "BRN r",
	0xAD, 0x42, "BSR r",

	0x11, 0x02, "B\x91 0,b",		/* BCLR */
	0x13, 0x02, "B\x91 1,b",		/* BCLR */
	0x15, 0x02, "B\x91 2,b",		/* BCLR */
	0x17, 0x02, "B\x91 3,b",		/* BCLR */
	0x19, 0x02, "B\x91 4,b",		/* BCLR */
	0x1B, 0x02, "B\x91 5,b",		/* BCLR */
	0x1D, 0x02, "B\x91 6,b",		/* BCLR */
	0x1F, 0x02, "B\x91 7,b",		/* BCLR */

	0x10, 0x02, "B\x9C 0,b",		/* BSET */
	0x12, 0x02, "B\x9C 1,b",		/* BSET */
	0x14, 0x02, "B\x9C 2,b",		/* BSET */
	0x16, 0x02, "B\x9C 3,b",		/* BSET */
	0x18, 0x02, "B\x9C 4,b",		/* BSET */
	0x1A, 0x02, "B\x9C 5,b",		/* BSET */
	0x1C, 0x02, "B\x9C 6,b",		/* BSET */
	0x1E, 0x02, "B\x9C 7,b",		/* BSET */

	0x01, 0x53, "BR\x91 0,b,s",		/* BRCLR */
	0x03, 0x53, "BR\x91 1,b,s",		/* BRCLR */
	0x05, 0x53, "BR\x91 2,b,s",		/* BRCLR */
	0x07, 0x53, "BR\x91 3,b,s",		/* BRCLR */
	0x09, 0x53, "BR\x91 4,b,s",		/* BRCLR */
	0x0B, 0x53, "BR\x91 5,b,s",		/* BRCLR */
	0x0D, 0x53, "BR\x91 6,b,s",		/* BRCLR */
	0x0F, 0x53, "BR\x91 7,b,s",		/* BRCLR */

	0x00, 0x53, "BR\x9C 0,b,s",		/* BRSET */
	0x02, 0x53, "BR\x9C 1,b,s",		/* BRSET */
	0x04, 0x53, "BR\x9C 2,b,s",		/* BRSET */
	0x06, 0x53, "BR\x9C 3,b,s",		/* BRSET */
	0x08, 0x53, "BR\x9C 4,b,s",		/* BRSET */
	0x0A, 0x53, "BR\x9C 5,b,s",		/* BRSET */
	0x0C, 0x53, "BR\x9C 6,b,s",		/* BRSET */
	0x0E, 0x53, "BR\x9C 7,b,s",		/* BRSET */

	0x98, 0x01, "CLC",
	0x9A, 0x01, "CLI",
	0x42, 0x01, "MUL",
	0x9D, 0x01, "NOP",
	0x9C, 0x01, "RSP",
	0x80, 0x01, "RTI",
	0x81, 0x01, "RTS",
	0x99, 0x01, "SEC",
	0x9B, 0x01, "SEI",
	0x8E, 0x01, "STOP",
	0x83, 0x01, "SWI",
	0x97, 0x01, "TAX",
	0x9F, 0x01, "TXA",
	0x8F, 0x01, "WAIT",
	
	/*  This entry always matches invalid opcodes */
	0xFF, 0x00, 'F','C','B',' ','o',0 };

unsigned char opcode, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 6805 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, type;
	char c;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {	/* 0 length = FCB */
			length = 1;
			return; }
		while(*inst_ptr++); }
	length = (type = *inst_ptr++) & 0x0F;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	/* Add any operand to symbol table */
	if(dflag) switch(type & 0xF0) {
		case 0x10 :	/* 8 bit data */
			if(bflag)
				set_symbol(memory[1], BIT8);
			break;
		case 0x20 :	/* 8 bit address */
			set_symbol(memory[1], BIT16);
			break;
		case 0x30 :	/* 16 bit address/data */
			set_symbol(get_word(1), BIT16);
			break;
		case 0x40 :	/* 8 bit relative */
			c = memory[1];
			set_symbol((address+2)+(int)c, BIT16);
			break;
		case 0x50 :	/* direct, 8 bit rel */
			set_symbol(memory[1], BIT8);
			c = memory[2];
			set_symbol((address+3)+(int)c, BIT16); }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	char c, *ptr;

	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case 'b' :		/* Byte operand */
			i += printv(memory[1], BIT8);
			break;
		case 'w' :		/* Word operand */
			i += printv(get_word(1), BIT16);
			break;
		case 'r' :		/* Relative address */
			c = memory[1];
			i += printv((address+2)+(int)c, BIT16);
			break;
		case 's' :		/* 2nd byte relative */
			c = memory[2];
			i += printv((address+3)+(int)c, BIT16);
			break;
		default:
			if(c & 0x80) {
				ptr = inames + (c & 0x7F)*3;
				putc(*ptr++, stdout);
				putc(*ptr++, stdout);
				i += 2;
				c = *ptr; }
			++i;
			putc(c, stdout); }

	return i;
}
