/*
 * Tables/Functions to implement the 6809 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "6809";		/* Name of CPU */
char _cmd_[]	= "DIS09";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

/* Table of instruction names and substrings */
unsigned char inames[] = {
	'N','E','G',	/* 80 */
	'C','O','M',	/* 81 */
	'L','S','R',	/* 82 */
	'R','O','R',	/* 83 */
	'A','S','R',	/* 84 */
	'L','S','L',	/* 85 */
	'R','O','L',	/* 86 */
	'D','E','C',	/* 87 */
	'I','N','C',	/* 88 */
	'T','S','T',	/* 89 */
	'J','M','P',	/* 8A */
	'C','L','R',	/* 8B */
	'L','B','R',	/* 8C */
	'A','N','D',	/* 8D */
	'B','R','A',	/* 8E */
	'B','R','N',	/* 8F */
	'B','H','I',	/* 90 */
	'B','L','S',	/* 91 */
	'B','C','C',	/* 92 */
	'B','C','S',	/* 93 */
	'B','N','E',	/* 94 */
	'B','E','Q',	/* 95 */
	'B','V','C',	/* 96 */
	'B','V','S',	/* 97 */
	'B','P','L',	/* 98 */
	'B','M','I',	/* 99 */
	'B','G','E',	/* 9A */
	'B','L','T',	/* 9B */
	'B','G','T',	/* 9C */
	'B','L','E',	/* 9D */
	'L','E','A',	/* 9E */
	'P','S','H',	/* 9F */
	'P','U','L',	/* A0 */
	'S','W','I',	/* A1 */
	'S','U','B',	/* A2 */
	'C','M','P',	/* A3 */
	'S','B','C',	/* A4 */
	'B','I','T',	/* A5 */
	'L','D','A',	/* A6 */
	'E','O','R',	/* A7 */
	'A','D','C',	/* A8 */
	'O','R','A',	/* A9 */
	'A','D','D',	/* AA */
	'B','S','R',	/* AB */
	'L','D','X',	/* AC */
	'S','T','A',	/* AD */
	'J','S','R',	/* AE */
	'S','T','X',	/* AF */
	'L','D','B',	/* B0 */
	'O','R','B',	/* B1 */
	'L','D','D',	/* B2 */
	'L','D','U',	/* B3 */
	'S','T','B',	/* B4 */
	'S','T','D',	/* B5 */
	'S','T','U',	/* B6 */
	'F','C','B',	/* B7 */
	'L','B','L',	/* B8 */
	'L','B','C',	/* B9 */
	'L','B','V',	/* BA */
	'L','B','G',	/* BB */
	'L','D','Y',	/* BC */
	'S','T','Y',	/* BD */
	'L','D','S',	/* BE */
	'S','T','S'		/* BF */
};

/* Table of 6809 opcodes: OPCODE, SYMBOL-FLAGS|LENGTH, TEXT */
/* SYMBOL-FLAGS: 00=None 10=8bitData 20=8bitAddress 30=16Bitdata/address */
/* 40=8bitRel 50=16BitRel 60=Indexed */
unsigned char itable[] = {
	0x00, 0x22, "\x80 <b",		/* NEG */
	0x03, 0x22, "\x81 <b",		/* COM */
	0x04, 0x22, "\x82 <b",		/* LSR */
	0x06, 0x22, "\x83 <b",		/* ROR */
	0x07, 0x22, "\x84 <b",		/* ASR */
	0x08, 0x22, "\x85 <b",		/* LSL */
	0x09, 0x22, "\x86 <b",		/* ROL */
	0x0A, 0x22, "\x87 <b",		/* DEC */
	0x0C, 0x22, "\x88 <b",		/* INC */
	0x0D, 0x22, "\x89 <b",		/* TST */
	0x0E, 0x22, "\x8A <b",		/* JMP */
	0x0F, 0x22, "\x8B <b",		/* CLR */

	0x12, 0x01, "NOP",
	0x13, 0x01, "SYNC",
	0x16, 0x53,	"\x8CA l",		/* LBRA */
	0x17, 0x53, "L\xAB l",		/* LBSR */
	0x19, 0x01, "DAA",
	0x1A, 0x02, "ORCC #b",
	0x1C, 0x02, "\x8DCC #b",	/* ANDCC */
	0x1D, 0x01, "SEX",
	0x1E, 0x02, "EXG t",
	0x1F, 0x02, "TFR t",

	0x20, 0x42,	"\x8E r",		/* BRA */
	0x21, 0x42, "\x8F r",		/* BRN */
	0x22, 0x42,	"\x90 r",		/* BHI */
	0x23, 0x42,	"\x91 r",		/* BLS */
	0x24, 0x42,	"\x92 r",		/* BCC */
	0x25, 0x42,	"\x93 r",		/* BCS */
	0x26, 0x42,	"\x94 r",		/* BNE */
	0x27, 0x42,	"\x95 r",		/* BEQ */
	0x28, 0x42,	"\x96 r",		/* BVC */
	0x29, 0x42,	"\x97 r",		/* BVS */
	0x2A, 0x42,	"\x98 r",		/* BPL */
	0x2B, 0x42,	"\x99 r",		/* BMI */
	0x2C, 0x42,	"\x9A r",		/* BGE */
	0x2D, 0x42,	"\x9B r",		/* BLT */
	0x2E, 0x42,	"\x9C r",		/* BGT */
	0x2F, 0x42,	"\x9D r",		/* BLE */

	0x30, 0x62, "\x9EX x",		/* LEAX */
	0x31, 0x62, "\x9EY x",		/* LEAY */
	0x32, 0x62, "\x9ES x",		/* LEAS */
	0x33, 0x62, "\x9EU x",		/* LEAU */
	0x34, 0x02, "\x9FS p",		/* PSHS */
	0x35, 0x02, "\xA0S p",		/* PULS */
	0x36, 0x02, "\x9FU q",		/* PSHU */
	0x37, 0x02, "\xA0U q",		/* PULU */
	0x39, 0x01, "RTS",
	0x3A, 0x01, "ABX",
	0x3B, 0x01, "RTI",
	0x3C, 0x02, "CWAI #b",
	0x3D, 0x01, "MUL",
	0x3F, 0x01, "\xA1",			/* SWI */

	0x40, 0x01, "\x80A",		/* NEGA */
	0x43, 0x01, "\x81A",		/* COMA */
	0x44, 0x01, "\x82A",		/* LSRA */
	0x46, 0x01, "\x83A",		/* RORA */
	0x47, 0x01, "\x84A",		/* ASRA */
	0x48, 0x01, "\x85A",		/* LSLA */
	0x49, 0x01, "\x86A",		/* ROLA */
	0x4A, 0x01, "\x87A",		/* DECA */
	0x4C, 0x01, "\x88A",		/* INCA */
	0x4D, 0x01, "\x89A",		/* TSTA */
	0x4F, 0x01, "\x8BA",		/* CLRA */

	0x50, 0x01, "\x80B",		/* NEGB */
	0x53, 0x01, "\x81B",		/* COMB */
	0x54, 0x01, "\x82B",		/* LSRB */
	0x56, 0x01, "\x83B",		/* RORB */
	0x57, 0x01, "\x84B",		/* ASRB */
	0x58, 0x01, "\x85B",		/* LSLB */
	0x59, 0x01, "\x86B",		/* ROLB */
	0x5A, 0x01, "\x87B",		/* DECB */
	0x5C, 0x01, "\x88B",		/* INCB */
	0x5D, 0x01, "\x89B",		/* TSTB */
	0x5F, 0x01, "\x8BB",		/* CLRB */

	0x60, 0x62, "\x80 x",		/* NEG */
	0x63, 0x62, "\x81 x",		/* COM */
	0x64, 0x62, "\x82 x",		/* LSR */
	0x66, 0x62, "\x83 x",		/* ROR */
	0x67, 0x62, "\x84 x",		/* ASR */
	0x68, 0x62, "\x85 x",		/* LSL */
	0x69, 0x62, "\x86 x",		/* ROL */
	0x6A, 0x62, "\x87 x",		/* DEC */
	0x6C, 0x62, "\x88 x",		/* INC */
	0x6D, 0x62, "\x89 x",		/* TST */
	0x6E, 0x62, "\x8A x",		/* JMP */
	0x6F, 0x62, "\x8B x",		/* CLR */

	0x70, 0x33, "\x80 w",		/* NEG */
	0x73, 0x33, "\x81 w",		/* COM */
	0x74, 0x33, "\x82 w",		/* LSR */
	0x76, 0x33, "\x83 w",		/* ROR */
	0x77, 0x33, "\x84 w",		/* ASR */
	0x78, 0x33, "\x85 w",		/* LSL */
	0x79, 0x33, "\x86 w",		/* ROL */
	0x7A, 0x33, "\x87 w",		/* DEC */
	0x7C, 0x33, "\x88 w",		/* INC */
	0x7D, 0x33, "\x89 w",		/* TST */
	0x7E, 0x33, "\x8A w",		/* JMP */
	0x7F, 0x33, "\x8B w",		/* CLR */

	0x80, 0x12, "\xA2A #b",		/* SUBA */
	0x81, 0x12, "\xA3A #b",		/* CMPA */
	0x82, 0x12, "\xA4A #b",		/* SBCA */
	0x83, 0x33, "\xA2D #w",		/* SUBD */
	0x84, 0x12, "\x8DA #b",		/* ANDA */
	0x85, 0x12, "\xA5A #b",		/* BITA */
	0x86, 0x12, "\xA6 #b",		/* LDA */
	0x88, 0x12, "\xA7A #b",		/* EORA */
	0x89, 0x12, "\xA8A #b",		/* ADCA */
	0x8A, 0x12, "\xA9 #b",		/* ORA */
	0x8B, 0x12, "\xAAA #b",		/* ADDA */
	0x8C, 0x33, "\xA3X #w",		/* CMPX */
	0x8D, 0x42, "\xAB r",		/* BSR */
	0x8E, 0x33, "\xAC #w",		/* LDX */

	0x90, 0x22, "\xA2A <b",		/* SUBA */
	0x91, 0x22, "\xA3A <b",		/* CMPA */
	0x92, 0x22, "\xA4A <b",		/* SBCA */
	0x93, 0x22, "\xA2D <b",		/* SUBD */
	0x94, 0x22, "\x8DA <b",		/* ANDA */
	0x95, 0x22, "\xA5A <b",		/* BITA */
	0x96, 0x22, "\xA6 <b",		/* LDA */
	0x97, 0x22, "\xAD <b",		/* STA */
	0x98, 0x22, "\xA7A <b",		/* EORA */
	0x99, 0x22, "\xA8A <b",		/* ADCA */
	0x9A, 0x22, "\xA9 <b",		/* ORA */
	0x9B, 0x22, "\xAAA <b",		/* ADDA */
	0x9C, 0x22, "\xA3X <b",		/* CMPX */
	0x9D, 0x22, "\xAE <b",		/* JSR */
	0x9E, 0x22, "\xAC <b",		/* LDX */
	0x9F, 0x22, "\xAF <b",		/* STX */

	0xA0, 0x62, "\xA2A x",		/* SUBA */
	0xA1, 0x62, "\xA3A x",		/* CMPA */
	0xA2, 0x62, "\xA4A x",		/* SBCA */
	0xA3, 0x62, "\xA2D x",		/* SUBD */
	0xA4, 0x62, "\x8DA x",		/* ANDA */
	0xA5, 0x62, "\xA5A x",		/* BITA */
	0xA6, 0x62, "\xA6 x",		/* LDA */
	0xA7, 0x62, "\xAD x",		/* STA */
	0xA8, 0x62, "\xA7A x",		/* EORA */
	0xA9, 0x62, "\xA8A x",		/* ADCA */
	0xAA, 0x62, "\xA9 x",		/* ORA */
	0xAB, 0x62, "\xAAA x",		/* ADDA */
	0xAC, 0x62, "\xA3X x",		/* CMPX */
	0xAD, 0x62, "\xAE x",		/* JSR */
	0xAE, 0x62, "\xAC x",		/* LDX */
	0xAF, 0x62, "\xAF x",		/* STX */

	0xB0, 0x33, "\xA2A w",		/* SUBA */
	0xB1, 0x33, "\xA3A w",		/* CMPA */
	0xB2, 0x33, "\xA4A w",		/* SBCA */
	0xB3, 0x33, "\xA2D w",		/* SUBD */
	0xB4, 0x33, "\x8DA w",		/* ANDA */
	0xB5, 0x33, "\xA5A w",		/* BITA */
	0xB6, 0x33, "\xA6 w",		/* LDA */
	0xB7, 0x33, "\xAD w",		/* STA */
	0xB8, 0x33, "\xA7A w",		/* EORA */
	0xB9, 0x33, "\xA8A w",		/* ADCA */
	0xBA, 0x33, "\xA9 w",		/* ORA */
	0xBB, 0x33, "\xAAA w",		/* ADDA */
	0xBC, 0x33, "\xA3X w",		/* CMPX */
	0xBD, 0x33, "\xAE w",		/* JSR */
	0xBE, 0x33, "\xAC w",		/* LDX */
	0xBF, 0x33, "\xAF w",		/* STX */

	0xC0, 0x12, "\xA2B #b",		/* SUBB */
	0xC1, 0x12, "\xA3B #b",		/* CMPB */
	0xC2, 0x12, "\xA4B #b",		/* SBCB */
	0xC3, 0x33, "\xAAD #w",		/* ADDD */
	0xC4, 0x12, "\x8DB #b",		/* ANDB */
	0xC5, 0x12, "\xA5B #b",		/* BITB */
	0xC6, 0x12, "\xB0 #b",		/* LDB */
	0xC8, 0x12, "\xA7B #b",		/* EORB */
	0xC9, 0x12, "\xA8B #b",		/* ADCB */
	0xCA, 0x12, "\xB1 #b",		/* ORB */
	0xCB, 0x12, "\xAAB #b",		/* ADDB */
	0xCC, 0x33, "\xB2 #w",		/* LDD */
	0xCE, 0x33, "\xB3 #w",		/* LDU */

	0xD0, 0x22, "\xA2B <b",		/* SUBB */
	0xD1, 0x22, "\xA3B <b",		/* CMPB */
	0xD2, 0x22, "\xA4B <b",		/* SBCB */
	0xD3, 0x22, "\xAAD <b",		/* ADDD */
	0xD4, 0x22, "\x8DB <b",		/* ANDB */
	0xD5, 0x22, "\xA5B <b",		/* BITB */
	0xD6, 0x22, "\xB0 <b",		/* LDB */
	0xD7, 0x22, "\xB4 <b",		/* STB */
	0xD8, 0x22, "\xA7B <b",		/* EORB */
	0xD9, 0x22, "\xA8B <b",		/* ADCB */
	0xDA, 0x22, "\xB1 <b",		/* ORB */
	0xDB, 0x22, "\xAAB <b",		/* ADDB */
	0xDC, 0x22, "\xB2 <b",		/* LDD */
	0xDD, 0x22, "\xB5 <b",		/* STD */
	0xDE, 0x22, "\xB3 <b",		/* LDU */
	0xDF, 0x22, "\xB6 <b",		/* STU */

	0xE0, 0x62, "\xA2B x",		/* SUBB */
	0xE1, 0x62, "\xA3B x",		/* CMPB */
	0xE2, 0x62, "\xA4B x",		/* SBCB */
	0xE3, 0x62, "\xAAD x",		/* ADDD */
	0xE4, 0x62, "\x8DB x",		/* ANDB */
	0xE5, 0x62, "\xA5B x",		/* BITB */
	0xE6, 0x62, "\xB0 x",		/* LDB */
	0xE7, 0x62, "\xB4 x",		/* STB */
	0xE8, 0x62, "\xA7B x",		/* EORB */
	0xE9, 0x62, "\xA8B x",		/* ADCB */
	0xEA, 0x62, "\xB1 x",		/* ORB */
	0xEB, 0x62, "\xAAB x",		/* ADDB */
	0xEC, 0x62, "\xB2 x",		/* LDD */
	0xED, 0x62, "\xB5 x",		/* STD */
	0xEE, 0x62, "\xB3 x",		/* LDU */
	0xEF, 0x62, "\xB6 x",		/* STU */

	0xF0, 0x33, "\xA2B w",		/* SUBB */
	0xF1, 0x33, "\xA3B w",		/* CMPB */
	0xF2, 0x33, "\xA4B w",		/* SBCB */
	0xF3, 0x33, "\xAAD w",		/* ADDD */
	0xF4, 0x33, "\x8DB w",		/* ANDB */
	0xF5, 0x33, "\xA5B w",		/* BITB */
	0xF6, 0x33, "\xB0 w",		/* LDB */
	0xF7, 0x33, "\xB4 w",		/* STB */
	0xF8, 0x33, "\xA7B w",		/* EORB */
	0xF9, 0x33, "\xA8B w",		/* ADCB */
	0xFA, 0x33, "\xB1 w",		/* ORB */
	0xFB, 0x33, "\xAAB w",		/* ADDB */
	0xFC, 0x33, "\xB2 w",		/* LDD */
	0xFD, 0x33, "\xB5 w",		/* STD */
	0xFE, 0x33, "\xB3 w",		/* LDU */
	0xFF, 0x33, "\xB6 w",		/* STU */
	0xFF, 0, "\xB7 o" };		/* FCB */

unsigned char itable10[] = {
	0x21, 0x53, "\x8CN l",		/* LBRN */
	0x22, 0x53, "L\x90 l",		/* LBHI */
	0x23, 0x53, "\xB8S l",		/* LBLS */
	0x24, 0x53, "\xB9C l",		/* LBCC */
	0x25, 0x53, "\xB9S l",		/* LBCS */
	0x26, 0x53, "L\x94 l",		/* LBNE */
	0x27, 0x53, "L\x95 l",		/* LBEQ */
	0x28, 0x53, "\xBAC l",		/* LBVC */
	0x29, 0x53, "\xBAS l",		/* LBVS */
	0x2A, 0x53, "L\x98 l",		/* LBPL */
	0x2B, 0x53, "L\x99 l",		/* LBMI */
	0x2C, 0x53, "\xBBE l",		/* LBGE */
	0x2D, 0x53, "\xB8T l",		/* LBLT */
	0x2E, 0x53, "\xBBT l",		/* LBGT */
	0x2F, 0x53, "\xB8E l",		/* LBLE */

	0x3F, 0x01, "\xA12",		/* SWI2 */

	0x83, 0x33, "\xA3D #w",		/* CMPD */
	0x8C, 0x33, "\xA3Y #w",		/* CMPY */
	0x8E, 0x33, "\xBC #w",		/* LDY */

	0x93, 0x22, "\xA3D <b",		/* CMPD */
	0x9C, 0x22, "\xA3Y <b",		/* CMPY */
	0x9E, 0x22, "\xBC <b",		/* LDY */
	0x9F, 0x22, "\xBD <b",		/* STY */

	0xA3, 0x62, "\xA3D x",		/* CMPD */
	0xAC, 0x62, "\xA3Y x",		/* CMPY */
	0xAE, 0x62, "\xBC x",		/* LDY */
	0xAF, 0x62, "\xBD x",		/* STY */

	0xB3, 0x33, "\xA3D w",		/* CMPD */
	0xBC, 0x33, "\xA3Y w",		/* CMPY */
	0xBE, 0x33, "\xBC w",		/* LDY */
	0xBF, 0x33, "\xBD w",		/* STY */

	0xCE, 0x33, "\xBE #w",		/* LDS */

	0xDE, 0x22, "\xBE <b",		/* LDS */
	0xDF, 0x22, "\xBF <b",		/* STS */
	0xEE, 0x62, "\xBE x",		/* LDS */
	0xEF, 0x62, "\xBF x",		/* STS */

	0xFE, 0x33, "\xBE w",		/* LDS */
	0xFF, 0x33, "\xBF w",		/* STS */

	0xFF, 0, "\xB7 $10,o" };	/* FCB */

unsigned char itable11[] = {
	0x3F, 0x01, "\xA13",		/* SWI3 */
	0x83, 0x33, "\xA3U #w",		/* CMPU */
	0x8C, 0x33, "\xA3S #w",		/* CMPS */
	0x93, 0x22, "\xA3U <b",		/* CMPU */
	0x9C, 0x22, "\xA3S <b",		/* CMPS */
	0xA3, 0x62, "\xA3U x",		/* CMPU */
	0xAC, 0x62, "\xA3S x",		/* CMPS */
	0xB3, 0x33, "\xA3U w",		/* CMPU */
	0xBC, 0x33, "\xA3U w",		/* CMPU */
	0xFF, 0, "\xB7 $11,o" };	/* FCB */

unsigned char *treg[] = {
/*	 0   1   2   3   4   5    6   7   8   9   A    B    C   D   E   F */
	"D","X","Y","U","S","PC","?","?","A","B","CC","DP","?","?","?","?"
	};
unsigned char *preg[] = { "CC", "A", "B", "DP", "X", "Y", "U", "PC" };
unsigned char *qreg[] = { "CC", "A", "B", "DP", "X", "Y", "S", "PC" };
	
unsigned char opcode, *inst_ptr;
unsigned shift_offset;

#define	CPU_OPTION
char
	opt_ind = 0;
char cpu_help[] = { "\
	-i	- Force indexed offset size\n" };
int cpu_option(int c)
{
	switch(c) {
	default: return 0;
	case 'I' :
		opt_ind = -1;
		break; }
	return -1;
}

/*
 * Get a word from memory in the proper "endian"
 * 6809 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, type;
	char c;
	int c1;

	shift_offset = i = 1;
	inst_ptr = itable;
	switch(opcode = *memory) {		/* Record opcode for later */
		case 0x10 : inst_ptr = itable10; goto doshift;
		case 0x11 : inst_ptr = itable11;
		doshift:
			opcode = memory[i++] = getbyte();
			shift_offset = 2; }

	/* Search instruction table for this opcode */
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {	/* 0 length = FCB */
			length = shift_offset;
			return; }
		while(*inst_ptr++); }
	length = (((type = *inst_ptr++) & 0x0F) + shift_offset) - 1;

	/* Load instruction into memory */
	while(i < length)
		memory[i++] = getbyte();

	if((type & 0xF0) == 0x60) {
		switch(memory[shift_offset] & 0x9F) {
			case 0x89 :	// 16 bit offset
			case 0x99 :
			case 0x8D :	// 16 bit PCR
			case 0x9D :
			case 0x9F :	// extended indirect
				++length;
				memory[i++]  = getbyte();
			case 0x88 :	// 8 bit offset
			case 0x98 :
			case 0x8C :	// 8 bit PCR
			case 0x9C :
				++length;
				memory[i++]  = getbyte(); } }

	/* Add any operand to symbol table */
	if(dflag) switch(type & 0xF0) {
		case 0x10 :	/* 8 bit data */
			if(bflag)
				set_symbol(memory[shift_offset], BIT8);
			break;
		case 0x20 :	/* 8 bit address */
			set_symbol(memory[shift_offset], BIT16);
			break;
		case 0x30 :	/* 16 bit data/address */
			set_symbol(get_word(shift_offset), BIT16);
			break;
		case 0x40 :	/* 8 bit relative */
			c = memory[shift_offset];
			set_symbol((address+shift_offset+1)+(int)c, BIT16);
			break;
		case 0x50 :	/* 16 bit relative */
			c1 = get_word(shift_offset);
			set_symbol((address+shift_offset+2)+c1, BIT16);
			break;
		case 0x60 :	/* Indexed addressing */
			if(c == 0x9F)
				set_symbol(get_word(shift_offset+1), BIT16); }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i, j;
	char c, d, *ptr, **ptr1;
	int c1;

	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case 'b' :		/* Byte operand */
			i += printv(memory[shift_offset], BIT8);
			break;
		case 'w' :		/* Word operand */
			i += printv(get_word(shift_offset), BIT16);
			break;
		case 'r' :		/* 8 bit Relative address */
			c = memory[shift_offset];
			i += printv((address+shift_offset+1)+(int)c, BIT16);
			break;
		case 'l' :		/* 16 bit relative address */
			c1 = get_word(shift_offset);
			i += printv((address+shift_offset+2)+c1, BIT16);
			break;
		case 't' :		/* tfr/exg register set */
			c =  memory[shift_offset];
			fputs(ptr = treg[(unsigned)c >> 4], stdout);
			putc(',', stdout);
			fputs(ptr1 = treg[c & 0x0f], stdout);
			i += strlen(ptr) + strlen(ptr1) + 1;
			break;
		case 'p' :		/* PSHS/PULS */
			ptr1 = preg;
		dostack:
			c = memory[shift_offset];
			while(c) {
				if(c & 0x01) {
					fputs(*ptr1, stdout);
					i += strlen(*ptr1);
					if(c & 0xFE) {
						putc(',', stdout);
						++i; } }
				(unsigned char)c >>= 1;
				++ptr1; }
			break;
		case 'q' :		/* PSHU/PULU */
			ptr1 = qreg;
			goto dostack;
		case 'x' :		/* Indexed */
			d = "XYUS"[((c = memory[shift_offset]) >> 5) & 0x03];
			if(!(c & 0x80)) {	/* 5 bit offset */
				if((c &= 0x1F) & 0x10)
					c |= 0xE0;
				i += printf("%d,%c", c, d);
				break; }
			if(c & 0x10) {
				putc('[', stdout);
				++i; }
			switch(c & 0x0F) {
				case 0x04 :	// No offset
					j=printf(",%c", d);	break;
				case 0x08 :	// 8 bit offset
					j=printf("%s%d,%c", opt_ind ? "<" : "", (char)memory[shift_offset+1], d); break;
				case 0x09 :	// 16 bit offset
					j=printf("%s%d,%c", opt_ind ? ">" : "", get_word(shift_offset+1), d); break;
				case 0x06 :	// Accumulator A offset
					j=printf("A,%c", d);	break;
				case 0x05 :	// Accumulator B offset
					j=printf("B,%c", d);	break;
				case 0x0B :	// Accumulator D offset
					j=printf("D,%c", d);	break;
				case 0x00 :	// +
					j=printf(",%c+", d);	break;
				case 0x01 :	// ++
					j=printf(",%c++", d);	break;
				case 0x02 : // -
					j=printf(",-%c", d);	break;
				case 0x03 :	// --
					j=printf(",--%c", d);	break;
				case 0x0C :	// 8 bit offset from PC
					j=printf("%d,PC", (char)memory[shift_offset+1]); break;
				case 0x0D :	// 16 bit offset from PC
					j=printf("%d,PC", get_word(shift_offset+1) + address + length); break;
				case 0x0F :	// 16 bit indirect
					j=printv(get_word(shift_offset+1), BIT16); }
			i += j;
			if(c & 0x10) {
				putc(']', stdout);
				++i; }
			break;
/*		case '@' :		/* Specify shift offset */
			shift_offset = *inst_ptr++;
			continue; */
		default:
			if(c & 0x80) {
				ptr = inames + (c & 0x7F)*3;
				putc(*ptr++, stdout);
				putc(*ptr++, stdout);
				i += 2;
				c = *ptr; }
			++i;
			putc(c, stdout); }

	return i;
}
