/*
 * Tables/Functions to implement the 6812 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "68HC12";		/* Name of CPU */
char _cmd_[]	= "DIS12";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

/* Table of instruction names */
unsigned char inames[] = {
	'A','D','C',	/* 80 */
	'A','D','D',	/* 81 */
	'A','N','D',	/* 82 */
	'A','S','L',	/* 83 */
	'A','S','R',	/* 84 */
	'B','I','T',	/* 85 */
	'C','L','R',	/* 86 */
	'C','M','P',	/* 87 */
	'C','O','M',	/* 88 */
	'C','P','X',	/* 89 */
	'D','E','C',	/* 8A */
	'E','O','R',	/* 8B */
	'I','N','C',	/* 8C */
	'J','M','P',	/* 8D */
	'J','S','R',	/* 8E */
	'L','D','A',	/* 8F */
	'L','D','S',	/* 90 */
	'L','S','R',	/* 91 */
	'N','E','G',	/* 92 */
	'O','R','A',	/* 93 */
	'P','S','H',	/* 94 */
	'P','U','L',	/* 95 */
	'R','O','R',	/* 96 */
	'R','O','L',	/* 97 */
	'S','B','C',	/* 98 */
	'S','T','A',	/* 99 */
	'S','T','S',	/* 9A */	
	'S','U','B',	/* 9B */
	'T','S','T',	/* 9C */
	'L','D','X',	/* 9D */
	'S','T','X',	/* 9E */
	'L','D','D',	/* 9F */
	'S','T','D',	/* A0 */
	'S','E','T',	/* A1 */	
	'C','P','Y',	/* A2 */
	'L','D','Y',	/* A3 */
	'S','T','Y',	/* A4 */
	'C','P','D',	/* A5 */
	'F','C','B',	/* A6 */
	'B','E','Q',	/* A7 */
	'B','N','E',	/* A8 */
	'L','E','A',	/* A9 */
	'C','P','S',	/* AA */
	'M','O','V',	/* AB */
	'M','A','X',	/* AC */
	'M','I','N',	/* AD */
	'M','U','L',	/* AE */
	'D','I','V'		/* AF */
};

/* Table of 6812 opcodes: OPCODE, SYMBOL-FLAGS/LENGTH, TEXT */
/* SYMBOL-FLAGS: 80=Indexed 40=Relative 20=Word 10=Byte */
unsigned char itable[] = {
	0x00, "BGND",
	0x01, "MEM",
	0x02, "INY",
	0x03, "DEY",
	0x04, "v y;s",			/* DBEQ */
	0x05, "\x8D i",			/* JMP */
	0x06, "\x8D w",			/* JMP */
	0x07, "BSR r",
	0x08, "INX",
	0x09, "DEX",
	0x0A, "RTC",
	0x0B, "RTI",
	0x0C, "B\xA1 i;b",		/* BSET */
	0x0D, "B\x86 i;b",		/* BCLR */
	0x0E, "BR\xA1 i;b;r",	/* BRSET */
	0x0F, "BR\x86 i;b;r",	/* BRCLR */

	0x10, "\x82CC #b",		/* ANDCC */
	0x11, "E\xAF",
	0x12, "\xAE",
	0x13, "E\xAE",
	0x14, "ORCC #b",
	0x15, "\x8E i",			/* JSR */
	0x16, "\x8E w",			/* JSR */
	0x17, "\x8E b",			/* JSR */
	0x19, "\xA9Y i",		/* LEAY */
	0x1A, "\xA9X i",		/* LEAX */
	0x1B, "\xA9S i",		/* LEAS */
	0x1C, "B\xA1 w;b",		/* BSET */
	0x1D, "B\x86 w;b",		/* BSET */
	0x1E, "BR\xA1 w;b;r",	/* BRSET */
	0x1F, "BR\x86 w;b;r",	/* BRCLR */

	0x20, "BRA r",
	0x21, "BRN r",
	0x22, "BHI r",
	0x23, "BLS r",
	0x24, "BCC r",
	0x25, "BCS r",
	0x26, "\xA8 r",
	0x27, "\xA7 r",
	0x28, "BVC r",
	0x29, "BVS r",
	0x2A, "BPL r",
	0x2B, "BMI r",
	0x2C, "BGE r",
	0x2D, "BLT r",
	0x2E, "BGT r",
	0x2F, "BLE r",

	0x30, "\x95X",			/* PULX */
	0x31, "\x95Y",			/* PULY */
	0x32, "\x95A",			/* PULA */
	0x33, "\x95B",			/* PULB */
	0x34, "\x94X",			/* PSHX */
	0x35, "\x94Y",			/* PSHY */
	0x36, "\x94A",			/* PSHA */
	0x37, "\x94B",			/* PSHB */
	0x38, "\x95C",			/* PULC */
	0x39, "\x94C",			/* PSHC */
	0x3A, "\x95D",			/* PULD */
	0x3B, "\x94D",			/* PSHD */
	0x3D, "RTS",
	0x3E, "WAI",
	0x3F, "SWI",

	0x40, "\x92A",			/* NEGA */
	0x41, "\x88A",			/* COMA */
	0x42, "\x8CA",			/* INCA */
	0x43, "\x8AA",			/* DECA */
	0x44, "\x91A",			/* LSRA */
	0x45, "\x97A",			/* ROLA */
	0x46, "\x96A",			/* RORA */
	0x47, "\x84A",			/* ASRA */
	0x48, "\x83A",			/* ASLA */
	0x49, "\x91D",			/* LSRD */
	0x4A, "CALL w;b",
	0x4B, "CALL i;b",
	0x4C, "B\xA1 b;b",		/* BSET */
	0x4D, "B\x86 b;b",		/* BCLR */
	0x4E, "BR\xA1 b;b;r",	/* BRSET */
	0x4F, "BR\x86 b;b;r",	/* BRCLR */

	0x50, "\x92B",			/* NEGB */
	0x51, "\x88B",			/* COMB */
	0x52, "\x8CB",			/* INCB */
	0x53, "\x8AB",			/* DECB */
	0x54, "\x91B",			/* LSRB */
	0x55, "\x97B",			/* ROLB */
	0x56, "\x96B",			/* RORB */
	0x57, "\x84B",			/* ASRB */
	0x58, "\x83B",			/* ASLB */
	0x59, "\x83D",			/* ASLD */
	0x5A, "\x99A b",		/* STAA */
	0x5B, "\x99B b",		/* STAB */
	0x5C, "\xA0 b",			/* STD */
	0x5D, "\xA4 b",			/* STY */
	0x5E, "\x9E b",			/* STX */
	0x5F, "\x9A b",			/* STS */

	0x60, "\x92 i",			/* NEG */
	0x61, "\x88 i",			/* COM */
	0x62, "\x8C i",			/* INC */
	0x63, "\x8A i",			/* DEC */
	0x64, "\x91 i",			/* LSR */
	0x65, "\x97 i",			/* ROL */
	0x66, "\x96 i",			/* ROR */
	0x67, "\x84 i",			/* ASR */
	0x68, "\x83 i",			/* ASL */
	0x69, "\x86 i",			/* CLR */
	0x6A, "\x99A i",		/* STAA */
	0x6B, "\x99B i",		/* STAB */
	0x6C, "\xA0 i",			/* STD */
	0x6D, "\xA4 i",			/* STY */
	0x6E, "\x9E i",			/* STX */
	0x6F, "\x9A i",			/* STS */

	0x70, "\x92 w",			/* NEG */
	0x71, "\x88 w",			/* COM */
	0x72, "\x8C w",			/* INC */
	0x73, "\x8A w",			/* DEC */
	0x74, "\x91 w",			/* LSR */
	0x75, "\x97 w",			/* ROL */
	0x76, "\x96 w",			/* ROR */
	0x77, "\x84 w",			/* ASR */
	0x78, "\x83 w",			/* ASL */
	0x79, "\x86 w",			/* CLR */
	0x7A, "\x99A w",		/* STAA */
	0x7B, "\x99B w",		/* STAB */
	0x7C, "\xA0 w",			/* STD */
	0x7D, "\xA4 w",			/* STY */
	0x7E, "\x9E w",			/* STX */
	0x7F, "\x9A w",			/* STS */

	0x80, "\x9BA #b",		/* SUBA */
	0x81, "\x87A #b",		/* CMPA */
	0x82, "\x98A #b",		/* SBCA */
	0x83, "\x9BD #w",		/* SUBD */
	0x84, "\x82A #b",		/* ANDA */
	0x85, "\x85A #b",		/* BITA */
	0x86, "\x8FA #b",		/* LDAA */
	0x87, "\x86A",			/* CLRA */
	0x88, "\x8BA #b",		/* EORA */
	0x89, "\x80A #b",		/* ADCA */
	0x8A, "\x93A #b",		/* ORAA */
	0x8B, "\x81A #b",		/* ADDA */
	0x8C, "\xA5 #w",		/* CPD */
	0x8D, "\xA2 #w",		/* CPY */
	0x8E, "\x89 #w",		/* CPX */
	0x8F, "\xAA #w",		/* CPS */

	0x90, "\x9BA b",		/* SUBA */
	0x91, "\x87A b",		/* CMPA */
	0x92, "\x98A b",		/* SBCA */
	0x93, "\x9BD b",		/* SUBD */
	0x94, "\x82A b",		/* ANDA */
	0x95, "\x85A b",		/* BITA */
	0x96, "\x8FA b",		/* LDAA */
	0x97, "\x9CA",			/* TSTA */
	0x98, "\x8BA b",		/* EORA */
	0x99, "\x80A b",		/* ADCA */
	0x9A, "\x93A b",		/* ORAA */
	0x9B, "\x81A b",		/* ADDA */
	0x9C, "\xA5 b",			/* CPD */
	0x9D, "\xA2 b",			/* CPY */
	0x9E, "\x89 b",			/* CPX */
	0x9F, "\xAA b",			/* CPS */

	0xA0, "\x9BA i",		/* SUBA */
	0xA1, "\x87A i",		/* CMPA */
	0xA2, "\x98A i",		/* SBCA */
	0xA3, "\x9BD i",		/* SUBD */
	0xA4, "\x82A i",		/* ANDA */
	0xA5, "\x85A i",		/* BITA */
	0xA6, "\x8FA i",		/* LDAA */
	0xA7, "NOP",
	0xA8, "\x8BA i",		/* EORA */
	0xA9, "\x80A i",		/* ADCA */
	0xAA, "\x93A i",		/* ORAA */
	0xAB, "\x81A i",		/* ADDA */
	0xAC, "\xA5 i",			/* CPD */
	0xAD, "\xA2 i",			/* CPY */
	0xAE, "\x89 i",			/* CPX */
	0xAF, "\xAA i",			/* CPS */

	0xB0, "\x9BA w",		/* SUBA */
	0xB1, "\x87A w",		/* CMPA */
	0xB2, "\x98A w",		/* SBCA */
	0xB3, "\x9BD w",		/* SUBD */
	0xB4, "\x82A w",		/* ANDA */
	0xB5, "\x85A w",		/* BITA */
	0xB6, "\x8FA w",		/* LDAA */
	0xB7, "z x,y",			// UPDATE
	0xB8, "\x8BA w",		/* EORA */
	0xB9, "\x80A w",		/* ADCA */
	0xBA, "\x93A w",		/* ORAA */
	0xBB, "\x81A w",		/* ADDA */
	0xBC, "\xA5 w",			/* CPD */
	0xBD, "\xA2 w",			/* CPY */
	0xBE, "\x89 w",			/* CPX */
	0xBF, "\xAA w",			/* CPS */

	0xC0, "\x9BB #b",		/* SUBB */
	0xC1, "\x87B #b",		/* CMPB */
	0xC2, "\x98B #b",		/* SBCB */
	0xc3, "\x81D #w",		/* ADDD */
	0xC4, "\x82B #b",		/* ANDB */
	0xC5, "\x85B #b",		/* BITB */
	0xC6, "\x8FB #b",		/* LDAB */
	0xC7, "\x86B",			/* CLRB */
	0xC8, "\x8BB #b",		/* EORB */
	0xC9, "\x80B #b",		/* ADCB */
	0xCA, "\x93B #b",		/* ORAB */
	0xCB, "\x81B #b",		/* ADDB */
	0xCC, "\x9F #w",		/* LDD */
	0xCD, "\xA3 #w",		/* LDY */
	0xCE, "\x9D #w",		/* LDX */
	0xCF, "\x90 #w",		/* LDS */

	0xD0, "\x9BB b",		/* SUBB */
	0xD1, "\x87B b",		/* CMPB */
	0xD2, "\x98B b",		/* SBCB */
	0xD3, "\x81D b",		/* ADDD */
	0xD4, "\x82B b",		/* ANDB */
	0xD5, "\x85B b",		/* BITB */
	0xD6, "\x8FB b",		/* LDAB */
	0xD7, "\x9CB",			/* TSTB */
	0xD8, "\x8BB b",		/* EORB */
	0xD9, "\x80B b",		/* ADCB */
	0xDA, "\x93B b",		/* ORAB */
	0xDB, "\x81B b",		/* ADDB */
	0xDC, "\x9F b",			/* LDD */
	0xDD, "\xA3 b",			/* LDY */
	0xDE, "\x9D b",			/* LDX */
	0xDF, "\x90 b",			/* LDS */

	0xE0, "\x9BB i",		/* SUBB */
	0xE1, "\x87B i",		/* CMPB */
	0xE2, "\x98B i",		/* SBCB */
	0xE3, "\x81D i",		/* ADDD */
	0xE4, "\x82B i",		/* ANDB */
	0xE5, "\x85B i",		/* BITB */
	0xE6, "\x8FB i",		/* LDAB */
	0xE7, "\x9C i",			/* TST */
	0xE8, "\x8BB i",		/* EORB */
	0xE9, "\x80B i",		/* ADCB */
	0xEA, "\x93B i",		/* ORAB */
	0xEB, "\x81B i",		/* ADDB */
	0xEC, "\x9F i",			/* LDD */
	0xED, "\xA3 i",			/* LDY */
	0xEE, "\x9D i",			/* LDX */
	0xEF, "\x90 i",			/* LDS */

	0xF0, "\x9BB w",		/* SUBB */
	0xF1, "\x87B w",		/* CMPB */
	0xF2, "\x98B w",		/* SBCB */
	0xF3, "\x81D w",		/* ADDD */
	0xF4, "\x82B w",		/* ANDB */
	0xF5, "\x85B w",		/* BITB */
	0xF6, "\x8FB w",		/* LDAB */
	0xF7, "\x9C w",			/* TST */
	0xF8, "\x8BB w",		/* EORB */
	0xF9, "\x80B w",		/* ADCB */
	0xFA, "\x93B w",		/* ORAB */
	0xFB, "\x81B w",		/* ADDB */
	0xFC, "\x9F w",			/* LDD */
	0xFD, "\xA3 w",			/* LDY */
	0xFE, "\x9D w",			/* LDX */
	0xFF, "\x90 w",			/* LDS */

	0x00, "FCB o" };		/* End of table */

/*
 * 0x18 SHIFT table
 */
unsigned char itable1[] = {
	0x00, "iw|\xABW #w;i",	/* MOVW */
	0x01, "iw|\xABW w;i",	/* MOVW */
	0x02, "\xABW i;i",		/* MOVW */
	0x03, "\xABW #w;w",		/* MOVW */
	0x04, "\xABW w;w",		/* MOVW */
	0x05, "\xABW i;w",		/* MOVW */
	0x06, "ABA",
	0x07, "DAA",
	0x08, "ib|\xABB #b;i",	/* MOVB */
	0x09, "iw|\xABB w;i",	/* MOVB */
	0x0A, "\xABB i;i",		/* MOVB */
	0x0B, "\xABB #b;w",		/* MOVB */
	0x0C, "\xABB w;w",		/* MOVB */
	0x0D, "\xABB i;w",		/* MOVB */
	0x0E, "TAB",
	0x0F, "TBA",
	0x10, "I\xAF",
	0x11, "F\xAF",
	0x12, "EMACS w",
	0x13, "E\xAES",
	0x14, "E\xAFS",
	0x15, "I\xAFS",
	0x16, "SBA",
	0x17, "CBA",
	0x18, "\xACA i",		/* MAXA */
	0x19, "\xADA i",		/* MINA */
	0x1A, "E\xACD i",		/* EMAXD */
	0x1B, "E\xADD i",		/* EMIND */
	0x1C, "\xACM i",		/* MAXM */
	0x1D, "\xADM i",		/* MINM */
	0x1E, "E\xACM i",		/* EMAXM */
	0x1F, "E\xADM i",		/* EMINM */

	0x20, "LBRA t",
	0x21, "LBRN t",
	0x22, "LBHI t",
	0x23, "LBLS t",
	0x24, "LBCC t",
	0x25, "LBCS t",
	0x26, "L\xA8 t",
	0x27, "L\xA7 t",
	0x28, "LBVC t",
	0x29, "LBVS t",
	0x2A, "LBPL t",
	0x2B, "LBMI t",
	0x2C, "LBGE t",
	0x2D, "LBLT t",
	0x2E, "LBGT t",
	0x2F, "LBLE t",

	0x3A, "REV",
	0x3B, "REVW",
	0x3C, "WAV",
	0x3D, "TBL i",
	0x3E, "STOP",
	0x3F, "ETBL i",

	0x00, "TRAP o" };

/*
 * 0xB7 TFR/SEX/EXG table
 */
unsigned char opcode, *inst_ptr;
unsigned vtable[4], vindex;

char *tfrreg[] = { "A", "B", "CCR", "TMP", "D", "X", "Y", "SP" };
char *dbeq[] = { "DBEQ", "DBNE", "TBEQ", "TBNE", "IBEQ", "IBNE", "?", "?" };

/*
 * Get a word from memory in the proper "endian"
 * 6812 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned v, i;
	unsigned char c, *ptr;

	vindex = 0;
	length = 1;
	inst_ptr = itable;
	if((opcode = *memory) == 0x18) {		/* Shifted opcode */
		inst_ptr = itable1;
		opcode = memory[length++] = getbyte(); }

	/* Search instruction table for this opcode */
	while(*inst_ptr++ != opcode) {
		while(*inst_ptr++);
		if(!*inst_ptr) {
			++inst_ptr;
			break; }  }

	ptr = inst_ptr;
	while(*ptr) switch(*ptr++) {
		case '|' :				/* Special - out of order */
			v = vtable[vindex-1];
			for(i=vindex-1; i; --i)
				vtable[i] = vtable[i-1];
			vtable[0] = v;
			inst_ptr = ptr;
			return;
		case 'b' :
			v = memory[length++] = getbyte();
		set8:
			if(dflag)
				set_symbol(v, BIT8);
			vtable[vindex++] = v;
			continue;
		case 'w' :
			v = (memory[length++] = getbyte()) << 8;
			v += memory[length++] = getbyte();
		set16:
			if(dflag)
				set_symbol(v, BIT16);
			vtable[vindex++] = v;
			continue;
		case 'r' :
			c = memory[length++] = getbyte();
			v = address + length + (int)c;
			goto set16;
		case 's' :
			c = memory[length++] = getbyte();
			v = memory[length++] = getbyte();
			if(c & 0x10)
				v |= 0xFF00;
			v = address + length + v;
			goto set16;
		case 't' :
			v = (memory[length++] = getbyte()) << 8;
			v += (memory[length++] = getbyte());
			v = address + length + v;
			goto set16;
		case 'x' :
			memory[length++] = getbyte();
			break;
		case 'i' :
			vtable[vindex++] = c = memory[length++] = getbyte();
			if(!(c & 0x20)) {	/* 5 bit address */
				if((v = c & 0x1F) & 0x10)
					v |= 0xFFF0;
				goto set8; }
			if((c & 0xE4) == 0xE0) {
				v = memory[length++] = getbyte();
				if(!(c & 0x02)) {
					if(c & 0x01)
						v |= 0xFF00;
					goto set16; }
				v = (v << 8) + (memory[length++] = getbyte());
				goto set16; } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	char c, *ptr;

	i = vindex = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case 'b' :		/* Byte operand */
			i += printv(vtable[vindex++], BIT8);
			break;
		case 'w' :		/* Word operand */
		case 'r' :		/* 8 bit Relative address */
		case 's' :		/* 9 bit relative address */
		case 't' :		/* 16 bit relative offset */
			i += printv(vtable[vindex++], BIT16);
			break;
		case 'v' :		/* DBEQ family opcode */
			i += outstr(dbeq[memory[1] >> 5]);
			break;
		case 'x' :		/* Register from upper nibble */
			i += outstr(tfrreg[(memory[1] >> 4) & 7]);
			break;
		case 'y' :		/* Register from lower nibble */
			i += outstr(tfrreg[memory[1] & 7]);
			break;
		case 'z' :		/* EXG/SEX/TFR opcode */
			if((c = memory[1]) & 0x80)
				i += outstr("EXG");
			else if(	((c & 0x0E) >= 0x04) && ((c & 0x70) < 3) )
				i += outstr("SEX");
			else
				i += outstr("TFR");
			break;
		case 'i' :		/* Indexed */
			i += index(vtable[vindex++]);
			break;
		default:
			if(c & 0x80) {
				ptr = inames + (c & 0x7F)*3;
				putc(*ptr++, stdout);
				putc(*ptr++, stdout);
				i += 2;
				c = *ptr; }
			++i;
			putc(c, stdout); }

	return i;
}

/*
 * Decode indexed operands
 */
index(unsigned char p)
{
	unsigned r, t, l;
	char i, a, x;
	static char *rtable[4] = { "X", "Y", "SP", "PC" };

	i = a = x = t = l = 0;
	if(!(p & 0x20)) {		/* 5 bit indexed */
		r = p >> 6;
		l = printf("%d", vtable[vindex++]); }
	else {
		r = p >> 3;
		switch(p & 0xE7) {
			default:
				r = p & 0x07;
				if(p & 0x08) {
					x = '-';
					r = 8 - r; }
				else {
					x = '+';
					++r; }
				l = printf("%u", r);
				r = p >> 6;
				break;
			case 0xE4 :	a = 'A';	break;
			case 0xE5 : a = 'B';	break;
			case 0xE6 :	a = 'D';	break;
			case 0xE7 :	a = 'D';	putc(i = '[', stdout);	break;
			case 0xE3 :	putc(i = '[', stdout);
			case 0xE2 :
				l += printv(vtable[vindex++], BIT16);
				break;
			case 0xE0 :
			case 0xE1 :
				l += printf("%d", vtable[vindex++]); } }

	if(a) {
		putc(a, stdout);
		++l; }
	putc(',', stdout);
	if(x) {
		++l;
		if(!(p & 0x10)) {
			putc(x, stdout);
			x = 0; } }
	l += outstr(rtable[r & 3]);
	if(x)
		putc(x, stdout);
	if(i) {
		putc(']', stdout);
		l += 2; }

	return l+1;
}

/*
 * Write a string
 */
outstr(char *p)
{
	unsigned l;
	l = 0;
	while(*p) {
		putc(*p++, stdout);
		++l; }
	return l;
}
