/*
 * Tables/Functions to implement the 8008 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "8008";		/* Name of CPU */
char _cmd_[]	= "DIS88";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/* Table of instruction opcodes: MASK, COMPARE, TYPE/LENGTH, TEXT */
char itable[] = {
	0xFE, 0x00, 0x01, 'H', 'L', 'T', 0,
	0xFF, 0xFF, 0x01, 'H', 'L', 'T', 0,

	0xC0, 0xC0, 0x01, 'L', 'd', 's', 0,
	0xC7, 0x06, 0x02, 'L', 'd', 'I', ' ', 0,
	0xC7, 0x00, 0x01, 'I', 'N', 'd', 0,
	0xC7, 0x01, 0x01, 'D', 'C', 'd', 0,
	0xF8, 0x80, 0x01, 'A', 'D', 's', 0,
	0xF8, 0x88, 0x01, 'A', 'C', 's', 0,
	0xF8, 0x90, 0x01, 'S', 'U', 's', 0,
	0xF8, 0x98, 0x01, 'S', 'B', 's', 0,
	0xF8, 0xA0, 0x01, 'N', 'D', 's', 0,
	0xF8, 0xA8, 0x01, 'X', 'R', 's', 0,
	0xF8, 0xB0, 0x01, 'O', 'R', 's', 0,
	0xF8, 0xB8, 0x01, 'C', 'P', 's', 0,

	0xFF, 0x04, 0x02, 'A', 'D', 'I', ' ', 0,
	0xFF, 0x0C, 0x02, 'A', 'C', 'I', ' ', 0,
	0xFF, 0x14, 0x02, 'S', 'U', 'I', ' ', 0,
	0xFF, 0x1C, 0x02, 'S', 'B', 'I', ' ', 0,
	0xFF, 0x24, 0x02, 'N', 'D', 'I', ' ', 0,
	0xFF, 0x2C, 0x02, 'X', 'R', 'I', ' ', 0,
	0xFF, 0x34, 0x02, 'O', 'R', 'I', ' ', 0,
	0xFF, 0x3C, 0x02, 'C', 'P', 'I', ' ', 0,

	0xFF, 0x02, 0x01, 'R', 'L', 'C', 0,
	0xFF, 0x0A, 0x01, 'R', 'R', 'C', 0,
	0xFF, 0x12, 0x01, 'R', 'A', 'L', 0,
	0xFF, 0x1A, 0x01, 'R', 'A', 'R', 0,

	0xC7, 0x44, 0x03, 'J', 'M', 'P', ' ', 0,
	0xC7, 0x40, 0x03, 'J', 'c', ' ', 0,
	0xC7, 0x46, 0x03, 'C', 'A', 'L', ' ', 0,
	0xC7, 0x42, 0x03, 'C', 'c', ' ', 0,
	0xC7, 0x07, 0x01, 'R', 'E', 'T', 0,
	0xC7, 0x03, 0x01, 'R', 'c', 0,

	0xC7, 0x05, 0x01, 'R', 'S', 'T', ' ', 'r', 0,
	0xF1, 0x41, 0x01, 'I', 'N', 'P', ' ', 'i', 0,
	0xC1, 0x41, 0x01, 'O', 'U', 'T', ' ', 'o', 0,

	/*  This entry always matches invalid opcodes */
	0x00, 0x00, 0x01, 'D', 'B', ' ', 'x', 0 };

/* Tables to convert register index into actual names */
char regtab[]	= { 'A', 'B', 'C', 'D', 'E', 'H', 'L', 'M' };
char *cctab[] = { "FC", "FZ", "FS", "FP", "TC", "TZ", "TS", "TP" };

unsigned char opcode, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 8085 uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while((*inst_ptr++ & opcode) != *inst_ptr++) {
		++inst_ptr;
		while(*inst_ptr++); }
	length = *inst_ptr++;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	/* Add any operand to symbol table */
	if(dflag) {
		if(length > 2) {
			set_symbol(get_word(1), BIT16);
			return; }
		if((length > 1) && bflag)
			set_symbol(memory[1], BIT8); }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	unsigned char c;
	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case 'r' :		/* Interrupt vector */
			putc(((opcode >> 3) & 7) + '0', stdout);
			goto iplus;
		case 'd' :		/* Destination register */
			putc(regtab[(opcode >> 3) & 7], stdout);
			goto iplus;
		case 's' :		/* Source register */
			putc(regtab[opcode & 7], stdout);
			goto iplus;
		case 'c' :		/* Condition */
			fputs(cctab[(opcode >> 3) & 7], stdout);
			i += 2;
			break;
		case 'i' :		/* Input port */
			i += printf("%u", (opcode >> 1) & 0x07);
			break;
		case 'o' :		/* Output port */
			i += printf("%u", ((opcode >> 1) & 0x1F) - 8);
			break;
		case 'x' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		default:
			putc(c, stdout);
		iplus:
			++i; }

	switch(length) {
		case 2 :		/* Single byte operand */
			i += printv(memory[1], BIT8);
			break;
		case 3 :		/* Double byte operand */
			i += printv(get_word(1), BIT16); }

	return i;
}
