/*
 * Tables/Functions to implement the 8051 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "8051/52";	/* Name of CPU */
char _cmd_[]	= "DIS51";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/* Table of duplicated instruction names */
unsigned char inames[] = {
	'A','D','D',	/* 80 */
	'A','N','L',	/* 81 */
	'C','L','R',	/* 82 */
	'C','P','L',	/* 83 */
	'D','E','C',	/* 84 */
	'I','N','C',	/* 85 */
	'J','M','P',	/* 86 */
	'J','N','E',	/* 87 */
	'J','N','Z',	/* 88 */
	'C','A','L',	/* 89 */
	'M','O','V',	/* 8A */
	'O','R','L',	/* 8B */
	'R','E','T',	/* 8C */
	'S','E','T',	/* 8D */
	'S','U','B',	/* 8E */
	'X','C','H',	/* 8F */
	'X','R','L' }; 	/* 90 */

/* Table of instruction opcodes: MASK, COMPARE, TEXT */
unsigned char itable[] = {
	0x1F,	0x11,	"A\x89L a",
	0xF8,	0x28,	"\x80 A,r",
	0xFF,	0x25,	"\x80 A,d",
	0xFE,	0x26,	"\x80 A,i",
	0xFF,	0x24,	"\x80 A,m",
	0xF8,	0x38,	"\x80C A,r",
	0xFF,	0x35,	"\x80C A,d",
	0xFE,	0x36,	"\x80C A,i",
	0xFF,	0x34,	"\x80C A,m",
	0x1F,	0x01,	"A\x86 a",
	0xF8,	0x58,	"\x81 A,r",
	0xFF,	0x55,	"\x81 A,d",
	0xFE,	0x56,	"\x81 A,i",
	0xFF,	0x54,	"\x81 A,m",
	0xFF,	0x52,	"\x81 d,A",
	0xFF,	0x53,	"\x81 d,m",
	0xFF,	0x82,	"\x81 C,b",
	0xFF,	0xB0,	"\x81 C,/b",
	0xFF,	0xB5,	"C\x87 A,d,j",
	0xFF,	0xB4,	"C\x87 A,m,j",
	0xF8,	0xB8,	"C\x87 r,m,j",
	0xFE,	0xB6,	"C\x87 i,m,j",
	0xFF,	0xE4,	"\x82 A",
	0xFF,	0xC3,	"\x82 C",
	0xFF,	0xC2,	"\x82 b",
	0xFF,	0xF4,	"\x83 A",
	0xFF,	0xB3,	"\x83 C",
	0xFF,	0xB2,	"\x83 b",
	0xFF,	0xD4,	"DA A",
	0xFF,	0x14,	"\x84 A",
	0xF8,	0x18,	"\x84 r",
	0xFF,	0x15,	"\x84 d",
	0xFE,	0x16,	"\x84 i",
	0xFF,	0x84,	"DIV AB",
	0xF8,	0xD8,	"D\x88 r,j",
	0xFF,	0xD5,	"D\x88 d,j",
	0xFF,	0x04,	"\x85 A",
	0xF8,	0x08,	"\x85 r",
	0xFF,	0x05,	"\x85 d",
	0xFE,	0x06,	"\x85 i",
	0xFF,	0xA3,	"\x85 DPTR",
	0xFF,	0x20,	"JB b,j",
	0xFF,	0x10,	"JBC b,j",
	0xFF,	0x40,	"JC j",
	0xFF,	0x73,	"\x86 [A+DPTR]",
	0xFF,	0x30,	"JNB b,j",
	0xFF,	0x50,	"JNC j",
	0xFF,	0x70,	"\x88 j",
	0xFF,	0x60,	"JZ j",
	0xFF,	0x12,	"L\x89L x",
	0xFF,	0x02,	"L\x86 x",
	0xF8,	0xE8,	"\x8A A,r",
	0xFF,	0xE5,	"\x8A A,d",
	0xFE,	0xE6,	"\x8A A,i",
	0xFF,	0x74,	"\x8A A,m",
	0xF8,	0xF8,	"\x8A r,A",
	0xF8,	0xA8,	"\x8A r,d",
	0xF8,	0x78,	"\x8A r,m",
	0xFF,	0xF5,	"\x8A d,A",
	0xF8,	0x88,	"\x8A d,r",
	0xFF,	0x85,	"\x8A e,f",
	0xFE,	0x86,	"\x8A d,i",
	0xFF,	0x75,	"\x8A d,m",
	0xFE,	0xF6,	"\x8A i,A",
	0xFE,	0xA6,	"\x8A i,d",
	0xFE,	0x76,	"\x8A i,m",
	0xFF,	0xA2,	"\x8A C,b",
	0xFF,	0x92,	"\x8A b,C",
	0xFF,	0x90,	"\x8A DPTR,#x",
	0xFF,	0x93,	"\x8AC A,[A+DPTR]",
	0xFF,	0x83,	"\x8AC A,[A+PC]",
	0xFE,	0xE2,	"\x8AX A,i",
	0xFF,	0xE0,	"\x8AX A,[DPTR]",
	0xFE,	0xF2,	"\x8AX i,A",
	0xFF,	0xF0,	"\x8AX [DPTR],A",
	0xFF,	0xA4,	"MUL AB",
	0xFF,	0x00,	"NOP",
	0xF8,	0x48,	"\x8B A,r",
	0xFF,	0x45,	"\x8B A,d",
	0xFE,	0x46,	"\x8B A,i",
	0xFF,	0x44,	"\x8B A,m",
	0xFF,	0x42,	"\x8B d,A",
	0xFF,	0x43,	"\x8B d,m",
	0xFF,	0x72,	"\x8B C,b",
	0xFF,	0xA0,	"\x8B C,/b",
	0xFF,	0xD0,	"POP d",
	0xFF,	0xC0,	"PUSH d",
	0xFF,	0x22,	"\x8C",
	0xFF,	0x32,	"\x8CI",
	0xFF,	0x23,	"RL A",
	0xFF,	0x33,	"RLC A",
	0xFF,	0x03,	"RR A",
	0xFF,	0x13,	"RRC A",
	0xFF,	0xD3,	"\x8DB C",
	0xFF,	0xD2,	"\x8DB b",
	0xFF,	0x80,	"S\x86 j",
	0xF8,	0x98,	"\x8EB A,r",
	0xFF,	0x95,	"\x8EB A,d",
	0xFE,	0x96,	"\x8EB A,i",
	0xFF,	0x94,	"\x8EB A,m",
	0xFF,	0xC4,	"SWAP A",
	0xF8,	0xC8,	"\x8F A,r",
	0xFF,	0xC5,	"\x8F A,d",
	0xFE,	0xC6,	"\x8F A,i",
	0xFE,	0xD6,	"\x8FD A,i",
	0xF8,	0x68,	"\x90 A,r",
	0xFF,	0x65,	"\x90 A,d",
	0xFE,	0x66,	"\x90 A,i",
	0xFF,	0x64,	"\x90 A,m",
	0xFF,	0x62,	"\x90 d,A",
	0xFF,	0x63,	"\x90 d,m",
	0x00,	0x00,	"DB o" };

/* Direct SFR address/names */
unsigned char *sfrs[] = {
	"\xE0ACC",
	"\xF0B",
	"\xD0PSW",
	"\x81SP",
	"\x82DPL",
	"\x83DPH",
	"\x80P0",
	"\x90P1",
	"\xA0P2",
	"\xB0P3",
	"\xB8IP",
	"\xA8IE",
	"\x89TMOD",
	"\xC8T2CON",
	"\x88TCON",
	"\x8CTH0",
	"\x8ATL0",
	"\x8DTH1",
	"\x8BTL1",
	"\xCDTH2",
	"\xCCTL2",
	"\xCBRCAP2H",
	"\xCARCAP2L",
	"\x98SCON",
	"\x99SBUF",
	"\x87PCON",
	0 };

unsigned char opcode, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 8051 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, j, k;
	unsigned char *ptr, *ptr1, c;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while((*inst_ptr++ & opcode) != *inst_ptr++) {
		while(*inst_ptr++); }

	/* Determine length of instruction & load into memory */
	ptr = inst_ptr;
	length = j = 1;
	while(c = *ptr++) switch(c) {
		case 'x' :
			memory[length++] = getbyte();
		case 'd' :
		case 'm' :
		case 'j' :
		case 'a' :
		case 'b' :
		case 'e' :
		case 'f' :
			memory[length++] = getbyte(); }

	/* Generate set symbols */
	if(dflag) {
		ptr = inst_ptr;
		while(c = *ptr++) switch(c) {
			case 'x' :
				set_symbol(get_word(j), BIT16);
				continue;
			case 'j' :
				set_symbol(address+(int)(char)memory[j++]+j, BIT16);
				continue;
			case 'a' :
				i = (((unsigned)opcode << 3) & 0x0700) + memory[j++];
				set_symbol((address & 0xF800) + i, BIT16);
				continue;
			case 'm' :	i = memory[j++];		goto set8bit;
			case 'e' :	i = memory[++j];		goto setsfr;
			case 'f' :	i = memory[j++ - 1];	goto setsfr;
			case 'd' :	i = memory[j++];		goto setsfr;
			case 'b' :	
				if((i = memory[j++]) & 0x80)
					i &= ~7;
				else
					i = (i >> 3) + 0x20;
			setsfr:
				for(k=0; ptr1 = sfrs[k]; ++k)
					if(i == *ptr1)
						goto skip_def8;
			set8bit:
				if(bflag)
					set_symbol(i, BIT8);
			skip_def8: } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i, j, k;
	unsigned char *ptr, c;

	i = 0;
	j = 1;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case 'r' :		/* Register id */
			putc('R', stdout);
			putc((opcode & 7) + '0', stdout);
			i += 2;
			continue;
		case 'i' :		/* Indirect register id */
			i += printf("[R%u]", opcode & 1);
			continue;
		case 'd' :		/* Direct address */
			i += printd(memory[j++]);
			continue;
		case 'b' :		/* Bit address */
			if((k = memory[j]) & 0x80)
				k &= ~7;
			else
				k = (k >> 3) + 0x20;
			i += printd(k);
			i += printf(".%u", memory[j++] & 0x07);
			continue;
		case 'm' :		/* Immediate value */
			putc('#', stdout);
			i += printv(memory[j++], BIT8) + 1;
			continue;
		case 'x' :		/* Extended address */
			i += printv(get_word(j), BIT16);
			continue;
		case 'j' :		/* Realtive jump address */
			i += printv(address + (int)(char)memory[j++]+j, BIT16);
			continue;
		case 'a' :		/* Absolute jump address */
			i += printv((address & 0xF800) +
			(((unsigned)opcode << 3) & 0x0700) + memory[j++], BIT16);
			continue;
		case 'e' :
			i += printd(memory[++j]);
			continue;
		case 'f' :
			i += printd(memory[j++ - 1]);
			continue;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		default:
			if(c & 0x80) {
				ptr = inames + (c & 0x7F)*3;
				putc(*ptr++, stdout);
				putc(*ptr++, stdout);
				i += 2;
				c = *ptr; }
			putc(c, stdout);
			++i; }

	return i;
}

/*
 * Display a direct address
 */
printd(value)
	unsigned value;
{
	int i;
	char *ptr;

	for(i=0; ptr = sfrs[i]; ++i) {
		if(*ptr++ == value) {
			i = 0;
			while(*ptr) {
				putc(*ptr++, stdout);
				++i; }
			return i; } }
	return printv(value, BIT8);
}
