/*
 * Tables/Functions to implement the 8085 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "8085";		/* Name of CPU */
char _cmd_[]	= "DIS85";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/* Table of instruction opcodes: MASK, COMPARE, TYPE/LENGTH, TEXT */
char itable[] = {
	0xFF, 0xFE, 0x02, 'C', 'P', 'I', ' ', 0,
	0xFF, 0x3A, 0x03, 'L', 'D', 'A', ' ', 0,
	0xFF, 0x32, 0x03, 'S', 'T', 'A', ' ', 0,
	0xFF, 0x2A, 0x03, 'L', 'H', 'L', 'D', ' ', 0,
	0xFF, 0x22, 0x03, 'S', 'H', 'L', 'D', ' ', 0,
	0xFF, 0xF5, 0x01, 'P', 'U', 'S', 'H', ' ', 'P', 'S', 'W', 0,
	0xFF, 0xF1, 0x01, 'P', 'O', 'P', ' ', 'P', 'S', 'W', 0,
	0xFF, 0x27, 0x01, 'D', 'A', 'A', 0,
	0xFF, 0x76, 0x01, 'H', 'L', 'T', 0,
	0xFF, 0xFB, 0x01, 'E', 'I', 0,
	0xFF, 0xF3, 0x01, 'D', 'I', 0,
	0xFF, 0x37, 0x01, 'S', 'T', 'C', 0,
	0xFF, 0x3F, 0x01, 'C', 'M', 'C', 0,
	0xFF, 0x2F, 0x01, 'C', 'M', 'A', 0,
	0xFF, 0xEB, 0x01, 'X', 'C', 'H', 'G', 0,
	0xFF, 0xE3, 0x01, 'X', 'T', 'H', 'L', 0,
	0xFF, 0xF9, 0x01, 'S', 'P', 'H', 'L', 0,
	0xFF, 0xE9, 0x01, 'P', 'C', 'H', 'L', 0,
	0xFF, 0xDB, 0x02, 'I', 'N', ' ', 0,
	0xFF, 0xD3, 0x02, 'O', 'U', 'T', ' ', 0,
	0xFF, 0x07, 0x01, 'R', 'L', 'C', 0,
	0xFF, 0x0F, 0x01, 'R', 'R', 'C', 0,
	0xFF, 0x17, 0x01, 'R', 'A', 'L', 0,
	0xFF, 0x1F, 0x01, 'R', 'A', 'R', 0,
	0xFF, 0xC6, 0x02, 'A', 'D', 'I', ' ', 0,
	0xFF, 0xCE, 0x02, 'A', 'C', 'I', ' ', 0,
	0xFF, 0xD6, 0x02, 'S', 'U', 'I', ' ', 0,
	0xFF, 0xDE, 0x02, 'S', 'B', 'I', ' ', 0,
	0xFF, 0xE6, 0x02, 'A', 'N', 'I', ' ', 0,
	0xFF, 0xF6, 0x02, 'O', 'R', 'I', ' ', 0,
	0xFF, 0xEE, 0x02, 'X', 'R', 'I', ' ', 0,
	0xFF, 0x00, 0x01, 'N', 'O', 'P', 0,
	/*  8085 specific instructions */
	0xFF, 0x20, 0x01, 'R', 'I', 'M', 0,
	0xFF, 0x30, 0x01, 'S', 'I', 'M', 0,
	/*  Jumps, Calls & Returns */
	0xFF, 0xC3, 0x0B, 'J', 'M', 'P', ' ', 0,
	0xFF, 0xCA, 0x43, 'J', 'Z', ' ', 0,
	0xFF, 0xC2, 0x4B, 'J', 'N', 'Z', ' ', 0,
	0xFF, 0xDA, 0x13, 'J', 'C', ' ', 0,
	0xFF, 0xD2, 0x1B, 'J', 'N', 'C', ' ', 0,
	0xFF, 0xEA, 0x23, 'J', 'P', 'E', ' ', 0,
	0xFF, 0xE2, 0x2B, 'J', 'P', 'O', ' ', 0,
	0xFF, 0xFA, 0x83, 'J', 'M', ' ', 0,
	0xFF, 0xF2, 0x8B, 'J', 'P', ' ', 0,
	0xFF, 0xCD, 0x0B, 'C', 'A', 'L', 'L', ' ', 0,
	0xFF, 0xCC, 0x43, 'C', 'Z', ' ', 0,
	0xFF, 0xC4, 0x4B, 'C', 'N', 'Z', ' ', 0,
	0xFF, 0xDC, 0x13, 'C', 'C', ' ', 0,
	0xFF, 0xD4, 0x1B, 'C', 'N', 'C', ' ', 0,
	0xFF, 0xEC, 0x23, 'C', 'P', 'E', ' ', 0,
	0xFF, 0xE4, 0x2B, 'C', 'P', 'O', ' ', 0,
	0xFF, 0xFC, 0x83, 'C', 'M', ' ', 0,
	0xFF, 0xF4, 0x8B, 'C', 'P', ' ', 0,
	0xFF, 0xC9, 0x05, 'R', 'E', 'T', 0,
	0xFF, 0xC8, 0x45, 'R', 'Z', 0,
	0xFF, 0xC0, 0x4D, 'R', 'N', 'Z', 0,
	0xFF, 0xD8, 0x15, 'R', 'C', 0,
	0xFF, 0xD0, 0x1D, 'R', 'N', 'C', 0,
	0xFF, 0xE8, 0x25, 'R', 'P', 'E', 0,
	0xFF, 0xE0, 0x2D, 'R', 'P', 'O', 0,
	0xFF, 0xF8, 0x85, 'R', 'M', 0,
	0xFF, 0xF0, 0x8D, 'R', 'P', 0,
	/*  Register based instructions */
	0xC0, 0x40, 0x01, 'M', 'O', 'V', ' ', 'd', ',', 's', 0,
	0xC7, 0x06, 0x02, 'M', 'V', 'I', ' ', 'd', ',', 0,
	0xF8, 0x90, 0x01, 'S', 'U', 'B', ' ', 's', 0,
	0xF8, 0x98, 0x01, 'S', 'B', 'B', ' ', 's', 0,
	0xF8, 0x80, 0x01, 'A', 'D', 'D', ' ', 's', 0,
	0xF8, 0x88, 0x01, 'A', 'D', 'C', ' ', 's', 0,
	0xF8, 0xA0, 0x01, 'A', 'N', 'A', ' ', 's', 0,
	0xF8, 0xB0, 0x01, 'O', 'R', 'A', ' ', 's', 0,
	0xF8, 0xA8, 0x01, 'X', 'R', 'A', ' ', 's', 0,
	0xF8, 0xB8, 0x01, 'C', 'M', 'P', ' ', 's', 0,
	0xC7, 0x04, 0x01, 'I', 'N', 'R', ' ', 'd', 0,
	0xC7, 0x05, 0x01, 'D', 'C', 'R', ' ', 'd', 0,
	/*  Register pair instructions */
	0xCF, 0x01, 0x03, 'L', 'X', 'I', ' ', 'p', ',', 0,
	0xEF, 0x0A, 0x01, 'L', 'D', 'A', 'X', ' ', 'p', 0,
	0xEF, 0x02, 0x01, 'S', 'T', 'A', 'X', ' ', 'p', 0,
	0xCF, 0x03, 0x01, 'I', 'N', 'X', ' ', 'p', 0,
	0xCF, 0x0B, 0x01, 'D', 'C', 'X', ' ', 'p', 0,
	0xCF, 0x09, 0x01, 'D', 'A', 'D', ' ', 'p', 0,
	0xCF, 0xC5, 0x01, 'P', 'U', 'S', 'H', ' ', 'p', 0,
	0xCF, 0xC1, 0x01, 'P', 'O', 'P', ' ', 'p', 0,
	/*  Restart instruction */
	0xC7, 0xC7, 0x01, 'R', 'S', 'T', ' ', 'v', 0,
	/*  This entry always matches invalid opcodes */
	0x00, 0x00, 0x01, 'D', 'B', ' ', 'o', 0 };

/* Tables to convert register index into actual names */
char regtab[]	= { 'B','C','D','E','H','L','M','A' };
char rptab[]	= { 'B','D','H','S' };

unsigned char opcode, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * 8085 uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while((*inst_ptr++ & opcode) != *inst_ptr++) {
		++inst_ptr;
		while(*inst_ptr++); }
	length = *inst_ptr++ & 3;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	/* Add any operand to symbol table */
	if(dflag) {
		if(length > 2) {
			set_symbol(get_word(1), BIT16);
			return; }
		if((length > 1) && bflag)
			set_symbol(memory[1], BIT8); }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	unsigned char c;
	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case 'v' :		/* Interrupt vector */
			putc(((opcode >> 3) & 7) + '0', stdout);
			goto iplus;
		case 'p' :		/* Register PAIR */
			putc(c = rptab[(opcode >> 4) & 3], stdout);
			if(c == 'S') {
				putc('P', stdout);
				++i; }
			goto iplus;
		case 'd' :		/* Destination register */
			putc(regtab[(opcode >> 3) & 7], stdout);
			goto iplus;
		case 's' :		/* Source register */
			putc(regtab[opcode & 7], stdout);
			goto iplus;
		case 'o' :		/* Output opcode byte */
			i += printf("$%02x", opcode);
			break;
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		default:
			putc(c, stdout);
		iplus:
			++i; }

	switch(length) {
		case 2 :		/* Single byte operand */
			i += printv(memory[1], BIT8);
			break;
		case 3 :		/* Double byte operand */
			i += printv(get_word(1), BIT16); }

	return i;
}
