/*
 * Tables/Functions to implement the 8086 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "8086";		/* Name of CPU */
char _cmd_[]	= "DIS86";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/*
 * Disassembly opcode table: flag, opcode, "text+operands"
 *
 * Flag byte:
 * 7	= '<>' not needed in EA
 * 6	= 'REG' field of opcode is used
 * 5-3	= REG field of E/A should equal this value
 * 2	= 'REG' = E/A field of opcode
 * 1	= 'S' is used
 * 0	= 'W' is used
 *
 * Operands bytes:
 * $8n	= REG (xxxREGxxx) Offset n
 * $9n	= EA  (MODxxxR/M) Offset n
 * $An	= 16 Bit jump displacement
 * $Bn	= 16 Bit value
 * $Cn	= 8 bit value
 * $Dn	= 8 bit jump displacement
 * $En	= Segment register
 * $F0	= 16 bit register from bits (0-2) of opcode
 * $F1	= 8 bit Register from bits (0-2) of opcode
 * $F2	= Accumulator (AX or AL)
 * $F3	= Immediate value (SI adjusted to last offset)
 * $FE	= Set 'W' bit
 * $FF	= Kill ',' separator
 */
unsigned char itable[] = {
	0x00,0x37,"AAA",
	0x00,0x3F,"AAS",
	0x80,0xD5,"AAD\xC1",
	0x80,0xD4,"AAM\xC1",
	0x81,0x10,"ADC\x91\x81",
	0x81,0x12,"ADC\x81\x91",
	0x17,0x80,"ADC\x91\xF3",
	0x81,0x14,"ADC\xF2\xF3",
	0x81,0x00,"ADD\x91\x81",
	0x81,0x02,"ADD\x81\x91",
	0x07,0x80,"ADD\x91\xF3",
	0x81,0x04,"ADD\xF2\xF3",
	0x81,0x20,"AND\x91\x81",
	0x81,0x22,"AND\x81\x91",
	0x25,0x80,"AND\x91\xF3",
	0x81,0x24,"AND\xF2\xF3",
	0x80,0xE8,"CALL\xA1",
	0x94,0xFF,"CALL\xFE\x91",	//>
	0x80,0x9A,"CALL\xB3\xFF:\xB1",
	0x9C,0xFF,"CALL\xFE0:\x91",	//FAR:>
	0x00,0x98,"CBW",
	0x00,0xF8,"CLC",
	0x00,0xFC,"CLD",
	0x00,0xFA,"CLI",
	0x00,0xF5,"CMC",
	0x81,0x38,"CMP\x91\x81",
	0x81,0x3A,"CMP\x81\x91",
	0x3F,0x80,"CMP\x91\xF3",
	0x81,0x3C,"CMP\xF2\xF3",
	0x00,0xA6,"CMPSB",
	0x00,0xA7,"CMPSW",
	0x00,0x99,"CWD",
	0x00,0x27,"DAA",
	0x00,0x2F,"DAS",
	0x0D,0xFE,"DEC\x91",
	0x40,0x48,"DEC\xF0",
	0x35,0xF6,"DIV\x91",
	0x40,0xD8,"ESC\xF4\x91",
	0x00,0xF4,"HLT",
	0x3D,0xF6,"IDIV\x91",
	0x2D,0xF6,"IMUL\x91",
	0x01,0xE4,"IN\xF2\xC1",
	0x01,0xEC,"IN\xF2DX",	//,
	0x05,0xFE,"INC\x91",
	0x40,0x40,"INC\xF0",
	0x00,0xCD,"INT\xC1",
	0x00,0xCC,"INT\xFF3",
	0x00,0xCE,"INTO",
	0x00,0xCF,"IRET",
	0x00,0x77,"JA\xD1",
	0x00,0x73,"JNC\xD1",
	0x00,0x72,"JC\xD1",
	0x00,0x76,"JBE\xD1",
	0x00,0xE3,"JCXZ\xD1",
	0x00,0x74,"JZ\xD1",
	0x00,0x7F,"JG\xD1",
	0x00,0x7D,"JGE\xD1",
	0x00,0x7C,"JL\xD1",
	0x00,0x7E,"JLE\xD1",
	0x00,0x75,"JNZ\xD1",
	0x00,0x71,"JNO\xD1",
	0x00,0x79,"JNS\xD1",
	0x00,0x7B,"JNP\xD1",
	0x00,0x70,"JO\xD1",
	0x00,0x7A,"JP\xD1",
	0x00,0x78,"JS\xD1",
	0x00,0xE9,"JMP\xA1",
	0x00,0xEB,"JMP\xFF<\xD1",
	0xA4,0xFF,"JMP\xFE\x91",	//>
	0x80,0xEA,"JMP\xB3\xFF:\xB1",
	0xAC,0xFF,"JMP\xFE0:\x91",	//FAR:
	0x00,0x9F,"LAHF",
	0x00,0xC5,"LDS\xFE\x81\x91",	//,
	0x00,0x8D,"LEA\xFE\x81\x91",	//,
	0x00,0xC4,"LES\xFE\x81\x91",	//,
	0x00,0xF0,"LOCK",
	0x00,0xAC,"LODSB",
	0x00,0xAD,"LODSW",
	0x00,0xE2,"LOOP\xD1",
	0x00,0xE1,"LOOPZ\xD1",
	0x00,0xE0,"LOOPNZ\xD1",
	0x81,0x88,"MOV\x91\x81",
	0x81,0x8A,"MOV\x81\x91",
	0x05,0xC6,"MOV\x91\xF3",
	0x40,0xB0,"MOV\xF1\xF3",
	0x40,0xB8,"MOV\xF0\xF3",
	0x01,0xA0,"MOV\xF2\xB1",
	0x81,0xA2,"MOV\xB1\xF2",
	0x00,0x8E,"MOV\xFE\xE1\x91",	//,
	0x80,0x8C,"MOV\xFE\x91\xE1",	//,
	0x00,0xA4,"MOVSB",
	0x00,0xA5,"MOVSW",
	0x25,0xF6,"MUL\x91",
	0x1D,0xF6,"NEG\x91",
	0x00,0x90,"NOP",
	0x15,0xF6,"NOT\x91",
	0x81,0x08,"OR\x91\x81",
	0x81,0x0A,"OR\x81\x91",
	0x0D,0x80,"OR\x91\xF3",
	0x81,0x0C,"OR\xF2\xF3",
	0x01,0xE6,"OUT\xC1\xF2",
	0x01,0xEE,"OUT\xFFDX,\xF2",
	0x04,0x8F,"POP\xFE\x91",
	0x40,0x58,"POP\xF0",
	0x00,0x07,"POP\xE0",
	0x00,0x0F,"POP\xE0",
	0x00,0x17,"POP\xE0",
	0x00,0x1F,"POP\xE0",
	0x00,0x9D,"POPF",
	0x34,0xFF,"PUSH\xFE\x91",
	0x40,0x50,"PUSH\xF0",
	0x00,0x06,"PUSH\xE0",
	0x00,0x0E,"PUSH\xE0",
	0x00,0x16,"PUSH\xE0",
	0x00,0x1E,"PUSH\xE0",
	0x00,0x9C,"PUSHF",
	0x15,0xD0,"RCL\x911",	//,
	0x15,0xD2,"RCL\x91CL",	//,
	0x1D,0xD0,"RCR\x911",	//,
	0x1D,0xD2,"RCR\x91CL",	//,
	0x00,0xF2,"REPNZ",
	0x00,0xF3,"REPZ",
	0x00,0xC3,"RET",
	0x00,0xC2,"RET\xB1",
	0x00,0xCB,"RETF",
	0x00,0xCA,"RETF\xB1",
	0x05,0xD0,"ROL\x911",	//,
	0x05,0xD2,"ROL\x91CL",	//,
	0x0D,0xD0,"ROR\x911",	//,
	0x0D,0xD2,"ROR\x91CL",	//,
	0x00,0x9E,"SAHF",
	0x3D,0xD0,"SAR\x911",	//,
	0x3D,0xD2,"SAR\x91CL",	//,
	0x81,0x18,"SBB\x91\x81",
	0x81,0x1A,"SBB\x81\x91",
	0x1F,0x80,"SBB\x91\xF3",
	0x81,0x1C,"SBB\xF2\xF3",
	0x00,0xAE,"SCASB",
	0x00,0xAF,"SCASW",
	0x25,0xD0,"SHL\x911",	//,
	0x25,0xD2,"SHL\x91CL",	//,
	0x2D,0xD0,"SHR\x911",	//,
	0x2D,0xD2,"SHR\x91CL",	//,
	0x00,0xF9,"STC",
	0x00,0xFD,"STD",
	0x00,0xFB,"STI",
	0x00,0xAA,"STOSB",
	0x00,0xAB,"STOSW",
	0x81,0x28,"SUB\x91\x81",
	0x81,0x2A,"SUB\x81\x91",
	0x2F,0x80,"SUB\x91\xF3",
	0x81,0x2C,"SUB\xF2\xF3",
	0x81,0x84,"TEST\x91\x81",
	0x05,0xF6,"TEST\x91\xF3",
	0x81,0xA8,"TEST\xF2\xF3",
	0x00,0x9B,"WAIT",
	0x01,0x86,"XCHG\x81\x91",
	0x40,0x90,"XCHG\xFE\xF2\xF0",
	0x00,0xD7,"XLAT",
	0x81,0x30,"XOR\x91\x81",
	0x81,0x32,"XOR\x81\x91",
	0x35,0x80,"XOR\x91\xF3",
	0x81,0x34,"XOR\xF2\xF3",
	0x00,0x26,"ES:",
	0x00,0x2E,"CS:",
	0x00,0x36,"SS:",
	0x00,0x3E,"DS:",
	0x00,0x00,"???" };

/* Register identifiers */
char *ea_reg16[] = {	/* 16 bit */
	"AX", "CX", "DX", "BX", "SP", "BP", "SI", "DI" };
char *ea_reg8[] = {		/* 8 bit */
	"AL", "CL", "DL", "BL", "AH", "CH", "DH", "BH" };
char *ea_seg[] = {		/* Segment */
	"ES", "CS", "SS", "DS", "??", "??", "??", "??" };
char *ea_rm[] = {		/* R/M modes */
	"[BX+SI]", "[BX+DI]", "[BP+SI]", "[BP+DI]", "[SI]", "[DI]", "[BP]", "[BX]" };

unsigned char opcode, *inst_ptr, ea, op, tf, sf;

/*
 * Get a word from memory in the proper "endian"
 * 8086 uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned e;
	unsigned char c, o, o1, g, *ptr;

	opcode = *memory;				/* Record opcode for later */
	g = 0;

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	for(;;) {
		tf = *inst_ptr++;			/* Get type byte from table */
		o = *inst_ptr++;			/* Get opcode byte from table */
		if(!(o | tf))				/* Not found - unknown */
			break;
		o1 = opcode;
		if(tf & 0x40)				/* Register bits used? */
			o1 &= 0xF8;				/* Yes, remove them */
		else
			o1 &= ((~tf) | 0xFC);		/* Mask 'S' and 'W' bits */
		if(o == o1)				/* We found it */
			break;
	gonext:
		while(*inst_ptr++); }

	/* Process EA if required */
	if(tf & 0x04) {					/* EA required */
		if(!g)						/* If not already obtained */
			memory[g = 1] = ea = getbyte();	/* Get EA byte */
		if((ea & 0x38) != (tf & 0x38))
			goto gonext; }
	/* We have the correct table entry for this opcode */
	op = (opcode | 0xFC) & tf;
	ptr = inst_ptr;
	while(c = *ptr++) {
		if(c & 0x80) switch(c & 0xF0) {
			case 0x80 :		/* EA field REG */
				if(!g)
					memory[g=1] = ea = getbyte();
				break;
			case 0x90 :		/* EA field MOD/RM */
				if(!g)
					memory[++g] = ea = getbyte();
				switch(ea & 0xC0) {	/* select by mode */
					case 0x00 :		/* extended EA */
						if((ea & 7) != 6)
							break;
					case 0x80 :		/* 16 bit */
						memory[++g] = getbyte();
					case 0x40 :		/* 8 bit */
						memory[++g] = getbyte(); }
				break;
			case 0xA0 :		/* 16 bit jump displacement */
			case 0xB0 :		/* 16 bit value */
				memory[++g] = getbyte();
			case 0xC0 :		/* 8 bit value */
			case 0xD0 :		/* 8 bit jump displacement */
				memory[++g] = getbyte();
				break;
			case 0xF0 :		/* Special cases */
				switch(c) {
					case 0xF0:
						op |= 1;
						break;
					case 0xF3:
						memory[++g] = getbyte();
						if(op & 0x01) {
							if((tf & 0x02) & (opcode & 0x02))
								break;
							memory[++g] = getbyte(); }
						break;
					case 0xFE :
						op |= 0x01; } } }
	length = g+1;

	ptr = inst_ptr;
	if(dflag) while(c = *ptr++) {
		switch(c & 0xF0) {
			case 0x90 :	/* MOD/RM field */
				e = c & 15;
				switch(ea & 0xC0) {		/* Select by mode */
					case 0x00 :
						if((ea & 7) == 6)
							set_symbol(get_word(e+1), BIT16);
						break;
					case 0x40 :			/* 8 bit displacement */
						if(bflag)
							set_symbol(memory[e+1], BIT8);
						break;
					case 0x80 :			/* 16 bit displacement */
						set_symbol(get_word(e+1), BIT16); }
				break;
			case 0xA0 :	/* 16 bit jump displacement */
				e = get_word(c & 15);
			dojmp:
				set_symbol(address+length+e, BIT16);
				break;
			case 0xB0 :	/* 16 bit value */
				set_symbol(get_word(c & 15), BIT16);
				break;
			case 0xC0 :	/* 8 bit value */
				if(bflag)
					set_symbol(memory[c & 15], BIT8);
				break;
			case 0xD0 :	/* 8 bit jump displacement */
				e = (int)(char)memory[c&15];
				goto dojmp;
				break;
			case 0xF0 :	/* Specials */
				switch(c) {
					case 0xF3 :	/* Immediate value */
						if(op & 1) {
							if((tf & 0x02) & (opcode & 0x02)) {
								if(bflag)
									set_symbol(memory[length-1], BIT8);
								break; }
							set_symbol(get_word(length-2), BIT16); }
						else
							set_symbol(memory[length-1], BIT8); } } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i, e;
	unsigned char c, **ptr1;
	i = sf = 0;
	while(c = *inst_ptr) {
		if(c & 0x80) {
			do
				putc(' ', stdout);
			while(++i < 8);
			break; }
		putc(c, stdout);
		++inst_ptr;
		++i; }

	while(c = *inst_ptr++) {
		switch(c & 0xF0) {
			case 0x80 :	/* REG field */
				sf = -1;
				e = memory[c & 15];
				ptr1 = (op & 1) ? ea_reg16 : ea_reg8;
				i += printf(ptr1[(e >> 3) & 7]);
				break;
			case 0x90 :	/* MOD/RM field */
				e = c & 15;
				switch(ea & 0xC0) {		/* Select by mode */
					case 0x00 :
						i += show_size();
						if((ea & 7) == 6) {
							i += printv(get_word(e+1), BIT16);
							goto normreg; }
						break;
					case 0xC0 :			/* RM treated as REG field */
						sf = -1;
						ptr1 = (op & 1) ? ea_reg16 : ea_reg8;
						i += printf(ptr1[memory[e] & 7]);
						goto normreg;
					case 0x40 :			/* 8 bit displacement */
						i += show_size();
						i += printv(memory[e+1], BIT8);
						break;
					case 0x80 :			/* 16 bit displacement */
						i += show_size();
						i += printv(get_word(e+1), BIT16); }
				i += printf(ea_rm[memory[e] & 7]);
		normreg: break;
			case 0xA0 :	/* 16 bit jump displacement */
				e = get_word(c & 15);
			dojmp:
				i += printv(address+length+e, BIT16);
				break;
			case 0xB0 :	/* 16 bit value */
				i += show_size();
				i += printv(get_word(c & 15), BIT16);
				break;
			case 0xC0 :	/* 8 bit value */
				i += show_size();
				i += printv(memory[c & 15], BIT8);
				break;
			case 0xD0 :	/* 8 bit jump displacement */
				e = (int)(char)memory[c&15];
				goto dojmp;
			case 0xE0 :	/* Segment register */
				sf = -1;
				e = memory[c & 15];
				i += printf(ea_seg[(e >> 3) & 0x07]);
				break;
			case 0xF0 :	/* Specials */
				switch(c) {
					case 0xF0 :	/* 16 bit register from opcode */
						sf = -1;
						op |= 0x01;
						i += printf(ea_reg16[opcode & 7]);
						break;
					case 0xF1 :	/* 8 bit register from opcode */
						sf = -1;
						i += printf(ea_reg8[opcode & 0x07]);
						break;
					case 0xF2 :	/* Accumulator */
						sf = -1;
						i += printf((op & 1) ? "AX" : "AL");
						break;
					case 0xF3 :	/* Immediate value */
						putc('#', stdout);
						++i;
						if(op & 1) {
							if((tf & 0x02) & (opcode & 0x02)) {
								i += printv(memory[length-1], BIT8);
								break; }
							i += printv(get_word(length-2), BIT16); }
						else
							i += printv(memory[length-1], BIT8);
						break;
					case 0xF4 :	/* ESC vector */
						i += printf("%u", ((opcode & 7) << 3) | ((memory[1] >> 3) & 7));
						break;
					case 0xFE :	op |= 0x01;
					case 0xFF : continue; };
				break;
			default:
				putc(c, stdout);
				++i;
				continue; }
		if(*inst_ptr) {
			if(*inst_ptr == 0xFF)
				continue;
			putc(',', stdout);
			++i; } };

		return i;
}

/*
 * Show the size of an operand value
 */
show_size()
{
	if(sf || (tf & 0x80))
		return 0;
	putc((op & 1) ? '>' : '<', stdout);
	sf = -1;
	return 1;
}
