/*
 * Tables/Functions to implement the 8096 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "8096";		/* Name of CPU */
char _cmd_[]	= "DIS96";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/*
 * Table of instruction opcodes: FLAG, OPCODE, "text"
 * Prefix:		80 = Word OP
 *				40 = Byte OP
 *				20 = Bit/Addr field
 *
 * Operands:	$8x - Addressable operand
 *				$9x - Register op
 *				$Ax - 8-bit displacement
 *				$Bx - 16-bit displacement
 *				$Cx - Special register (immediate for shifts)
 *				$Dx - 11 bit displacement
 *				$Ex - bit address
 */
unsigned char itable[] = {
	0x83,0x64,"ADD \x92,\x81",
	0x84,0x44,"ADD \x93,\x92,\x81",
	0xC3,0x74,"ADDB \x92,\x81",
	0xC4,0x54,"ADDB \x93,\x92,\x81",
	0x83,0xA4,"ADDC \x92,\x81",		//*
	0xC3,0xB4,"ADDCB \x92,\x81",	//*
	0x83,0x60,"AND \x92,\x81",
	0x84,0x40,"AND \x93,\x92,\x81",
	0xC3,0x70,"ANDB \x92,\x81",
	0xC4,0x50,"ANDB \x93,\x92,\x81",
	0x02,0xE3,"BR \x91",
	0x02,0x01,"CLR \x91",
	0x02,0x11,"CLRB \x91",
	0x01,0xF8,"CLRC",
	0x01,0xFC,"CLRVT",
	0x83,0x88,"CMP \x92,\x81",
	0xC3,0x98,"CMPB \x92,\x81",
	0x02,0x05,"DEC \x91",
	0x02,0x15,"DECB \x91",
	0x01,0xFA,"DI",
	0x83,0x8C,"DIVU \x92,\x81",
	0xC3,0x9C,"DIVUB \x92,\x81",
	0x03,0xE0,"DJNZ \x91,\xA2",
	0x01,0xFB,"EI",
	0x02,0x06,"EXT \x91",
	0x02,0x16,"EXTB \x91",
	0x02,0x07,"INC \x91",
	0x02,0x17,"INCB \x91",
	0x23,0x30,"JBC \x91,\xE0,\xA2",
	0x23,0x38,"JBS \x91,\xE0,\xA2",
	0x02,0xDB,"JC \xA1",
	0x02,0xDF,"JE \xA1",
	0x02,0xD6,"JGE \xA1",
	0x02,0xD2,"JGT \xA1",
	0x02,0xD9,"JH \xA1",
	0x02,0xDA,"JLE \xA1",
	0x02,0xDE,"JLT \xA1",
	0x02,0xD3,"JNC \xA1",
	0x02,0xD7,"JNE \xA1",
	0x02,0xD1,"JNH \xA1",
	0x02,0xD0,"JNST \xA1",
	0x02,0xD5,"JNV \xA1",
	0x02,0xD4,"JNVT \xA1",
	0x02,0xD8,"JST \xA1",
	0x02,0xDD,"JV \xA1",
	0x02,0xDC,"JVT \xA1",
	0x03,0xEF,"LCALL \xB1",
	0x83,0xA0,"LD \x92,\x81",
	0xC3,0xB0,"LDB \x92,\x81",
	0xC3,0xBC,"LDBSE \x92,\x81",
	0xC3,0xAC,"LDBZE \x92,\x81",
	0x03,0xE7,"LJMP \xB1",
	0x83,0x6C,"MULU \x92,\x81",
	0x84,0x4C,"MULU \x93,\x92,\x81",
	0xC3,0x7C,"MULUB \x92,\x81",
	0xC4,0x5C,"MULUB \x93,\x92,\x81",
	0x02,0x03,"NEG \x91",
	0x02,0x13,"NEGB \x91",
	0x01,0xFD,"NOP",
	0x03,0x0F,"NORML \x92,\x91",
	0x02,0x02,"NOT \x91",
	0x02,0x12,"NOTB \x91",
	0x83,0x80,"OR \x92,\x81",
	0xC3,0x90,"ORB \x92,\x81",
	0x82,0xCC,"POP \x81",
	0x01,0xF3,"POPF",
	0x82,0xC8,"PUSH \x81",
	0x01,0xF2,"PUSHF",
	0x01,0xF0,"RET",
	0x01,0xFF,"RST",
	0x22,0x28,"SCALL \xD1",
	0x01,0xF9,"SETC",
	0x03,0x09,"SHL \x92,\xC1",
	0x03,0x19,"SHLB \x92,\xC1",
	0x03,0x0D,"SHLL \x92,\xC1",
	0x03,0x08,"SHR \x92,\xC1",
	0x03,0x18,"SHRB \x92,\xC1",
	0x03,0x0C,"SHRL \x92,\xC1",
	0x03,0x0A,"SHRA \x92,\xC1",
	0x03,0x1A,"SHRAB \x92,\xC1",
	0x03,0x0E,"SHRAL \x92,\xC1",
	0x22,0x20,"SJMP \xD1",
	0x02,0x00,"SKIP \x91",
	0x83,0xC0,"ST \x92,\x81",
	0xC3,0xC4,"STB \x92,\x81",
	0x83,0x68,"SUB \x92,\x81",
	0x84,0x48,"SUB \x93,\x92,\x81",
	0xC3,0x78,"SUBB \x92,\x81",
	0xC4,0x58,"SUBB \x93,\x92,\x81",
	0x83,0xA8,"SUBC \x92,\x81",
	0xC3,0xB8,"SUBCB \x92,\x81",
	0x01,0xF7,"TRAP",
	0x83,0x84,"XOR \x92,\x81",
	0xC3,0x94,"XORB \x92,\x81",
	0, 0, "???" };

/* Table of shifted opcodes */
unsigned char itable1[] = {
	0x83,0x8C,"DIV \x92,\x81",
	0xC3,0x9C,"DIVB \x92,\x81",
	0x83,0x6C,"MUL \x92,\x81",
	0x84,0x4C,"MUL \x93,\x92,\x81",
	0xC3,0x7C,"MULB \x92,\x81",
	0xC4,0x5C,"MULB \x93,\x92,\x81",
	0, 0, "???" };
	

unsigned shift_offset;
unsigned char opcode, *inst_ptr, ea, flg;

/*
 * Get a word from memory in the proper "endian"
 * 8096 uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, cx;
	unsigned char o;
	int s;

	shift_offset = 0;
	inst_ptr = itable;
	length = 1;
	if((opcode = *memory) == 0xFE) {	/* Shifted opcode ? */
		shift_offset = 1;
		inst_ptr = itable1;
		opcode = memory[length++] = getbyte(); }

	for(;;) {
		o = opcode;
		flg = *inst_ptr++;		/* Get flag byte */
		if(flg & 0x20)			/* Bit field */
			o &= 0xF8;
		else if(flg & 0x80)		/* Address field */
			o &= 0xFC;
// printf("%04x %02x %02x %s\n", inst_ptr, *inst_ptr, o, inst_ptr+1);
		if(*inst_ptr++ == o)	/* Match */
			break;
		while(*inst_ptr++);		/* Skip to end of this one */
		if(!*inst_ptr) {
			inst_ptr += 2;
			++flg;
			break; } }

// printf("f=%04x o=%02x op=%04x ip=%s ", flg, o, opcode, inst_ptr);
	if(flg & 0x80) {			/* Special length calculation */
		switch(opcode & 0x03) {	/* Addressing mode */
			case 0x01 :			/* Immediate */
				if(!(flg & 0x40))
					flg = (flg + 1) | 0x10;
				break;
			case 0x03 :			/* Indexed */
				ea = memory[length++] = getbyte();
				if(ea & 1)
					flg = (flg+1) | 0x20;	/* Indicate +2 */
				flg = (flg+1) | 0x10; } }		/* Indicate +1 */

	/* Load insstruction into memory */
	i = (flg & 0x0F) + shift_offset;
	while(length < i)
		memory[length++] = getbyte();

	if(dflag) while(o=*inst_ptr++) if(o & 0x80) {	/* Special operand */
		cx = (o & 0x0F) + shift_offset;
		switch(o & 0xF0) {					/* Select by operand */
			case 0x80 :						/* Addressable operand */
				switch(opcode & 0x03) {		/* Select by addressing */
					case 0x00 :				/* Register direct */
						if(bflag)
							set_symbol(memory[cx], BIT8);
						break;
					case 0x01 :				/* Immediate */
						if(flg & 0x10)
							set_symbol(get_word(cx), BIT16);
						else if(bflag)
							set_symbol(memory[cx], BIT8);
						break;
					case 0x02 :				/* Indirect */
						if(bflag)
							set_symbol(memory[cx] & 0xFE, BIT8);
						break;
					case 0x03 :				/* Indexed */
						if(flg & 0x20)		/* Long */
							set_symbol(get_word(cx+1), BIT16);
						else if(bflag)		/* Short */
							set_symbol(memory[cx+1], BIT8);
						if(bflag)
							set_symbol(memory[cx] & 0xFE, BIT8); }
				continue;
			case 0x90 :						/* Register op */
				if(flg & 0x10) {			/* Adjust length */
					++cx;
					if(flg & 0x20)
						++cx; }
				if(bflag)
					set_symbol(memory[cx], BIT8);
				continue;
			case 0xA0 :						/* 8 bit displacement */
				s = (char)memory[cx];
				set_symbol(address + cx + s + 1, BIT16);
				continue;
			case 0xB0 :						/* 16 bit displacement */
				s = get_word(cx);
				set_symbol(address + cx + s + 2, BIT16);
				continue;
			case 0xC0 :						/* Special register */
				if((o = memory[cx]) < 16) {	/* Immediate */
					if(bflag)
						set_symbol(o, BIT8) + 1; }
				else if(bflag)
					set_symbol(o, BIT8);
				continue;
			case 0xD0 :						/* 11 bit displacement */
				s = ((int)(opcode & 0x07) << 8) + memory[cx];
				if(s & 0x400)
					s |= 0xF800;
				set_symbol(address + cx + s + 1, BIT16);
				continue; } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i, cx;
	unsigned char c;
	int s;

	i = 0;
	while(c = *inst_ptr) {
		++inst_ptr;
		if(c == ' ') {
			do
				putc(' ', stdout);
			while(++i < 8);
			break; }
		putc(c, stdout);
		++i; }

	while(c=*inst_ptr++) if(c & 0x80) {		/* Special operand */
		cx = (c & 0x0F) + shift_offset;
// printf("Spc: cx=%04x c=%02x", cx, c);
		switch(c & 0xF0) {					/* Select by operand */
			case 0x80 :						/* Addressable operand */
				switch(opcode & 0x03) {		/* Select by addressing */
					case 0x00 :				/* Register direct */
						i += printr(cx, 0);
						break;
					case 0x01 :				/* Immediate */
						putc('#', stdout);
						++i;
						if(flg & 0x10)
							i += printv(get_word(cx), BIT16);
						else
							i += printv(memory[cx], BIT8);
						break;
					case 0x02 :				/* Indirect */
						i += printr(cx, -1);
						if(memory[cx] & 1) {
							putc('+', stdout);
							++i; }
						break;
					case 0x03 :				/* Indexed */
						if(flg & 0x20)		/* Long */
							i += printv(get_word(cx+1), BIT16);
						else				/* Short */
							i += printv((char)memory[cx+1], BIT16);
						i += printr(cx, 1); }
				continue;
			case 0x90 :						/* Register op */
				if(flg & 0x10) {			/* Adjust length */
					++cx;
					if(flg & 0x20)
						++cx; }
				i += printr(cx, 0);
				continue;
			case 0xA0 :						/* 8 bit displacement */
				s = (char)memory[cx];
				i += printv(address + cx + s + 1, BIT16);
				continue;
			case 0xB0 :						/* 16 bit displacement */
				s = get_word(cx);
				i += printv(address + cx + s + 2, BIT16);
				continue;
			case 0xC0 :						/* Special register */
				if((c = memory[cx]) < 16) {	/* Immediate */
					putc('#', stdout);
					i += printv(c, BIT8) + 1; }
				else
					i += printr(cx, 0);
				continue;
			case 0xD0 :						/* 11 bit displacement */
				s = ((int)(opcode & 0x07) << 8) + memory[cx];
				if(s & 0x400)
					s |= 0xF800;
				i += printv(address + cx + s + 1, BIT16);
				continue;
			case 0xE0 :						/* Bit address */
				putc((opcode & 7) + '0', stdout);
				++i;
				continue; } }
		else {
			putc(c, stdout);
			++i; }

	return i;
}

/*
 * Display a register value
 */
printr(reg, ind)
	unsigned reg;
	char ind;
{
	reg = memory[reg];

	if(ind) {
		if((ind == 1) && !(reg & 0xFE))
			return 0;
		putc('[', stdout);
		ind = printv(reg & 0xFE, BIT8);
		putc(']', stdout);
		return (reg & 1) + ind + 2; }

	return printv(reg, BIT8);
}
