/*
 * Tables/Functions to implement the CFLEA cross disassembler
 */

#define	XBIT16	0x10
#define	PCREL	0x20

/* Text strings that the disassembler needs */
char _cpu_[]	= "CFLEA";		/* Name of CPU */
char _cmd_[]	= "DISCF";		/* Command name */
char _byte_[]	= "FCB     ";	/* Directive to encode bytes */
char _word_[]	= "FDB     ";	/* Directive to encode words */
char _string_[]	= "FCC     ";	/* Directive to encode strings */

unsigned char inames[] = {
	'L','D','B',	/* 80 */
	'A','D','D',	/* 81 */
	'S','U','B',	/* 82 */
	'M','U','L',	/* 83 */
	'D','I','V',	/* 84 */
	'A','N','D',	/* 85 */
	'O','R','B',	/* 86 */
	'X','O','R',	/* 87 */
	'C','M','P',	/* 88 */
	'L','D','I',	/* 89 */
	'L','E','A',	/* 8A */
	'S','T','B',	/* 8B */
	'S','T','I',	/* 8C */
	'S','H','R',	/* 8D */
	'S','H','L',	/* 8E */
	'J','M','P',	/* 8F */
	'J','N','Z',	/* 90 */
	'A','L','L',	/* 91 */
	'P','U','S'		/* 92 */
};

/* Table of CFLEA opcodes: OPCODE, FLAG/LENGTH, TEXT */
/* FLAG = 0x80 if second opcode byte required. */
unsigned char itable[] = {
	0x00,0x03,"LD #\xD1",
	0x01,0x03,"LD \xD1",
	0x02,0x01,"LD I",
	0x03,0x02,"LD \xC1,I",
	0x04,0x02,"LD \xC1,S",
	0x05,0x01,"LD S+",
	0x06,0x01,"LD [S+]",
	0x07,0x01,"LD [S]",
	0x08,0x02,"\x80 #\xC1",		/* LDB */
	0x09,0x03,"\x80 \xD1",		/* LDB */
	0x0A,0x01,"\x80 I",			/* LDB */
	0x0B,0x02,"\x80 \xC1,I",	/* LDB */
	0x0C,0x02,"\x80 \xC1,S",	/* LDB */
	0x0D,0x01,"\x80 S+",		/* LDB */
	0x0E,0x01,"\x80 [S+]",		/* LDB */
	0x0F,0x01,"\x80 [S]",		/* LDB */
	0x10,0x03,"\x81 #\xD1",		/* ADD */
	0x11,0x03,"\x81 \xD1",		/* ADD */
	0x12,0x01,"\x81 I",			/* ADD */
	0x13,0x02,"\x81 \xC1,I",	/* ADD */
	0x14,0x02,"\x81 \xC1,S",	/* ADD */
	0x15,0x01,"\x81 S+",		/* ADD */
	0x16,0x01,"\x81 [S+]",		/* ADD */
	0x17,0x01,"\x81 [S]",		/* ADD */
	0x18,0x02,"\x81B #\xC1",	/* ADDB */
	0x19,0x03,"\x81B \xD1",		/* ADDB */
	0x1A,0x01,"\x81B I",		/* ADDB */
	0x1B,0x02,"\x81B \xC1,I",	/* ADDB */
	0x1C,0x02,"\x81B \xC1,S",	/* ADDB */
	0x1D,0x01,"\x81B S+",		/* ADDB */
	0x1E,0x01,"\x81B [S+]",		/* ADDB */
	0x1F,0x01,"\x81B [S]",		/* ADDB */
	0x20,0x03,"\x82 #\xD1",		/* SUB */
	0x21,0x03,"\x82 \xD1",		/* SUB */
	0x22,0x01,"\x82 I",			/* SUB */
	0x23,0x02,"\x82 \xC1,I",	/* SUB */
	0x24,0x02,"\x82 \xC1,S",	/* SUB */
	0x25,0x01,"\x82 S+",		/* SUB */
	0x26,0x01,"\x82 [S+]",		/* SUB */
	0x27,0x01,"\x82 [S]",		/* SUB */
	0x28,0x02,"\x82B #\xC1",	/* SUBB */
	0x29,0x03,"\x82B \xD1",		/* SUBB */
	0x2A,0x01,"\x82B I",		/* SUBB */
	0x2B,0x02,"\x82B \xC1,I",	/* SUBB */
	0x2C,0x02,"\x82B \xC1,S",	/* SUBB */
	0x2D,0x01,"\x82B S+",		/* SUBB */
	0x2E,0x01,"\x82B [S+]",		/* SUBB */
	0x2F,0x01,"\x82B [S]",		/* SUBB */
	0x30,0x03,"\x83 #\xD1",		/* MUL */
	0x31,0x03,"\x83 \xD1",		/* MUL */
	0x32,0x01,"\x83 I",			/* MUL */
	0x33,0x02,"\x83 \xC1,I",	/* MUL */
	0x34,0x02,"\x83 \xC1,S",	/* MUL */
	0x35,0x01,"\x83 S+",		/* MUL */
	0x36,0x01,"\x83 [S+]",		/* MUL */
	0x37,0x01,"\x83 [S]",		/* MUL */
	0x38,0x02,"\x83B #\xC1",	/* MULB */
	0x39,0x03,"\x83B \xD1",		/* MULB */
	0x3A,0x01,"\x83B I",		/* MULB */
	0x3B,0x02,"\x83B \xC1,I",	/* MULB */
	0x3C,0x02,"\x83B \xC1,S",	/* MULB */
	0x3D,0x01,"\x83B S+",		/* MULB */
	0x3E,0x01,"\x83B [S+]",		/* MULB */
	0x3F,0x01,"\x83B [S]",		/* MULB */
	0x40,0x03,"\x84 #\xD1",		/* DIV */
	0x41,0x03,"\x84 \xD1",		/* DIV */
	0x42,0x01,"\x84 I",			/* DIV */
	0x43,0x02,"\x84 \xC1,I",	/* DIV */
	0x44,0x02,"\x84 \xC1,S",	/* DIV */
	0x45,0x01,"\x84 S+",		/* DIV */
	0x46,0x01,"\x84 [S+]",		/* DIV */
	0x47,0x01,"\x84 [S]",		/* DIV */
	0x48,0x02,"\x84B #\xC1",	/* DIVB */
	0x49,0x03,"\x84B \xD1",		/* DIVB */
	0x4A,0x01,"\x84B I",		/* DIVB */
	0x4B,0x02,"\x84B \xC1,I",	/* DIVB */
	0x4C,0x02,"\x84B \xC1,S",	/* DIVB */
	0x4D,0x01,"\x84B S+",		/* DIVB */
	0x4E,0x01,"\x84B [S+]",		/* DIVB */
	0x4F,0x01,"\x84B [S]",		/* DIVB */
	0x50,0x03,"\x85 #\xD1",		/* AND */
	0x51,0x03,"\x85 \xD1",		/* AND */
	0x52,0x01,"\x85 I",			/* AND */
	0x53,0x02,"\x85 \xC1,I",	/* AND */
	0x54,0x02,"\x85 \xC1,S",	/* AND */
	0x55,0x01,"\x85 S+",		/* AND */
	0x56,0x01,"\x85 [S+]",		/* AND */
	0x57,0x01,"\x85 [S]",		/* AND */
	0x58,0x02,"\x85B #\xC1",	/* ANDB */
	0x59,0x03,"\x85B \xD1",		/* ANDB */
	0x5A,0x01,"\x85B I",		/* ANDB */
	0x5B,0x02,"\x85B \xC1,I",	/* ANDB */
	0x5C,0x02,"\x85B \xC1,S",	/* ANDB */
	0x5D,0x01,"\x85B S+",		/* ANDB */
	0x5E,0x01,"\x85B [S+]",		/* ANDB */
	0x5F,0x01,"\x85B [S]",		/* ANDB */
	0x60,0x03,"OR #\xD1",
	0x61,0x03,"OR \xD1",
	0x62,0x01,"OR I",
	0x63,0x02,"OR \xC1,I",
	0x64,0x02,"OR \xC1,S",
	0x65,0x01,"OR S+",
	0x66,0x01,"OR [S+]",
	0x67,0x01,"OR [S]",
	0x68,0x02,"\x86 #\xC1",		/* ORB */
	0x69,0x03,"\x86 \xD1",		/* ORB */
	0x6A,0x01,"\x86 I",			/* ORB */
	0x6B,0x02,"\x86 \xC1,I",	/* ORB */
	0x6C,0x02,"\x86 \xC1,S",	/* ORB */
	0x6D,0x01,"\x86 S+",		/* ORB */
	0x6E,0x01,"\x86 [S+]",		/* ORB */
	0x6F,0x01,"\x86 [S]",		/* ORB */
	0x70,0x03,"\x87 #\xD1",		/* XOR */
	0x71,0x03,"\x87 \xD1",		/* XOR */
	0x72,0x01,"\x87 I",			/* XOR */
	0x73,0x02,"\x87 \xC1,I",	/* XOR */
	0x74,0x02,"\x87 \xC1,S",	/* XOR */
	0x75,0x01,"\x87 S+",		/* XOR */
	0x76,0x01,"\x87 [S+]",		/* XOR */
	0x77,0x01,"\x87 [S]",		/* XOR */
	0x78,0x02,"\x87B #\xC1",	/* XORB */
	0x79,0x03,"\x87B \xD1",		/* XORB */
	0x7A,0x01,"\x87B I",		/* XORB */
	0x7B,0x02,"\x87B \xC1,I",	/* XORB */
	0x7C,0x02,"\x87B \xC1,S",	/* XORB */
	0x7D,0x01,"\x87B S+",		/* XORB */
	0x7E,0x01,"\x87B [S+]",		/* XORB */
	0x7F,0x01,"\x87B [S]",		/* XORB */
	0x80,0x03,"\x88 #\xD1",		/* CMP */
	0x81,0x03,"\x88 \xD1",		/* CMP */
	0x82,0x01,"\x88 I",			/* CMP */
	0x83,0x02,"\x88 \xC1,I",	/* CMP */
	0x84,0x02,"\x88 \xC1,S",	/* CMP */
	0x85,0x01,"\x88 S+",		/* CMP */
	0x86,0x01,"\x88 [S+]",		/* CMP */
	0x87,0x01,"\x88 [S]",		/* CMP */
	0x88,0x02,"\x88B #\xC1",	/* CMPB */
	0x89,0x03,"\x88B \xD1",		/* CMPB */
	0x8A,0x01,"\x88B I",		/* CMPB */
	0x8B,0x02,"\x88B \xC1,I",	/* CMPB */
	0x8C,0x02,"\x88B \xC1,S",	/* CMPB */
	0x8D,0x01,"\x88B S+",		/* CMPB */
	0x8E,0x01,"\x88B [S+]",		/* CMPB */
	0x8F,0x01,"\x88B [S]",		/* CMPB */
	0x90,0x03,"\x89 #\xD1",		/* LDI */
	0x91,0x03,"\x89 \xD1",		/* LDI */
	0x92,0x01,"\x89 I",			/* LDI */
	0x93,0x02,"\x89 \xC1,I",	/* LDI */
	0x94,0x02,"\x89 \xC1,S",	/* LDI */
	0x95,0x01,"\x89 S+",		/* LDI */
	0x96,0x01,"\x89 [S+]",		/* LDI */
	0x97,0x01,"\x89 [S]",		/* LDI */
	0x98,0x02,"//?
	0x99,0x03,"\x8AI \xD1",		/* LEAI */
	0x9A,0x01,"\x8AI I",		/* LEAI */
	0x9B,0x02,"\x8AI \xC1,I",	/* LEAI */
	0x9C,0x02,"\x8AI \xC1,S",	/* LEAI */
	0x9E,0x01,"\x8AI [S+]",		/* LEAI */
	0x9F,0x01,"\x8AI [S]",		/* LEAI */
	0xA1,0x03,"ST \xD1",
	0xA2,0x01,"ST I",
	0xA3,0x02,"ST \xC1,I",
	0xA4,0x02,"ST \xC1,S",
	0xA6,0x01,"ST [S+]",
	0xA7,0x01,"ST [S]",
	0xA9,0x03,"\x8B \xD1",		/* STB */
	0xAA,0x01,"\x8B I",			/* STB */
	0xAB,0x02,"\x8B \xC1,I",	/* STB */
	0xAC,0x02,"\x8B \xC1,S",	/* STB */
	0xAE,0x01,"\x8B [S+]",		/* STB */
	0xAF,0x01,"\x8B [S]",		/* STB */
	0xB1,0x03,"\x8C \xD1",		/* STI */
	0xB2,0x01,"\x8C I",			/* STI */
	0xB3,0x02,"\x8C \xC1,I",	/* STI */
	0xB4,0x02,"\x8C \xC1,S",	/* STI */
	0xB6,0x01,"\x8C [S+]",		/* STI */
	0xB7,0x01,"\x8C [S]",		/* STI */
	0xB8,0x02,"\x8D #\xC1",		/* SHR */
	0xB9,0x03,"\x8D \xD1",		/* SHR */
	0xBA,0x01,"\x8D I",			/* SHR */
	0xBB,0x02,"\x8D \xC1,I",	/* SHR */
	0xBC,0x02,"\x8D \xC1,S",	/* SHR */
	0xBD,0x01,"\x8D S+",		/* SHR */
	0xBE,0x01,"\x8D [S+]",		/* SHR */
	0xBF,0x01,"\x8D [S]",		/* SHR */
	0xC0,0x02,"\x8E #\xC1",		/* SHL */
	0xC1,0x03,"\x8E \xD1",		/* SHL */
	0xC2,0x01,"\x8E I",			/* SHL */
	0xC3,0x02,"\x8E \xC1,I",	/* SHL */
	0xC4,0x02,"\x8E \xC1,S",	/* SHL */
	0xC5,0x01,"\x8E S+",		/* SHL */
	0xC6,0x01,"\x8E [S+]",		/* SHL */
	0xC7,0x01,"\x8E [S]",		/* SHL */
	0xC8,0x01,"LT",
	0xC9,0x01,"LE",
	0xCA,0x01,"GT",
	0xCB,0x01,"GE",
	0xCC,0x01,"ULT",
	0xCD,0x01,"ULE",
	0xCE,0x01,"UGT",
	0xCF,0x01,"UGE",
	0xD0,0x03,"\x8F \xD1",		/* JMP */
	0xD1,0x03,"JZ \xD1",
	0xD2,0x03,"\x90 \xD1",		/* JNZ */
	0xD3,0x02,"S\x8F \xE1",		/* SJMP */
	0xD4,0x02,"SJZ \xE1",
	0xD5,0x02,"S\x90 \xE1",		/* SJNZ */
	0xD6,0x01,"I\x8F",			/* IJMP */
	0xD7,0x01,"SWITCH",
	0xD8,0x03,"C\x91 \xD1",		/* CALL */
	0xD9,0x01,"RET",
	0xDA,0x02,"\x91OC \xC1",	/* ALLOC */
	0xDB,0x02,"FREE \xC1",
	0xDC,0x01,"\x92HA",			/* PUSHA */
	0xDD,0x01,"\x92HI",			/* PUSHI */
	0xDE,0x01,"TAS",
	0xDF,0x01,"TSA",
	0xE0,0x01,"CLR",
	0xE1,0x01,"COM",
	0xE2,0x01,"NEG",
	0xE3,0x01,"NOT",
	0xE4,0x01,"INC",
	0xE5,0x01,"DEC",
	0xE6,0x01,"TAI",
	0xE7,0x01,"TIA",
	0xE8,0x01,"ADAI",
	0xE9,0x01,"ALT",
	0xEA,0x02,"OUT \xC1",
	0xEB,0x02,"IN \xC1",
	0xEC,0x01,"NATIVE",

	/*  This entry always matches invalid opcodes */
	0x00, 0x00, "FCB \xC0" };

unsigned char opcode, opcode1, *inst_ptr;

/*
 * Get a word from memory in the proper "endian"
 * CFLEA uses "little endian", 1st byte is LOW, 2nd byte is HIGH
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index+1] << 8) | memory[index];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i, type;
	unsigned char c, *ptr;
	char d;

	opcode = *memory;		/* Record opcode for later */

	/* Search instruction table for this opcode */
	inst_ptr = itable;
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {	/* 0 length = FCB */
			length = 1;
			return; }
		++inst_ptr;		/* Skip possible second byte */
		while(*inst_ptr++); }
	length = (type = *inst_ptr++) & 0x07;

	/* Load instruction into memory */
	for(i=1; i < length; ++i)
		memory[i] = getbyte();

	if(dflag) {
		ptr = inst_ptr;
		while(*ptr != ' ')
			if(!*ptr++)
				return;
		while(c = *ptr++) if((c & 0xC0) == 0xC0) {
			if(c & PCREL) {
				if(c & XBIT16) {
					set_symbol(get_word(c & 0x07)+address+2, BIT16);
					continue; }
				else {
					d = memory[c & 0x07];
					set_symbol((address+2)+(int)d, BIT16);
					continue; } }
			if(c & XBIT16) {
				set_symbol(get_word(c & 0x07), BIT16);
				continue; }
			if(bflag)
				set_symbol(memory[c & 0x07], BIT8); } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	unsigned char c, *ptr;
	char d;

	/* Display instruction name */
	i = 0;
	while((c = *inst_ptr++) != ' ') {
		if(!c)
			return i;
		if(c & 0x80) {
			ptr = inames + (c & 0x7F)*3;
			putc(*ptr++, stdout);
			putc(*ptr++, stdout);
			i += 2;
			c = *ptr; } 
		++i;
		putc(c, stdout); }

	/* Space if operands */
	do
		putc(' ', stdout);
	while(++i < 8);

	/* Display operands */
	while(c = *inst_ptr++) {
		if(c & 0x80) {
			if(c & PCREL) {
				if(c & XBIT16) {
					i += printv(get_word(c & 0x07)+address+2, BIT16);
					continue; }
				else {
					d = memory[c & 0x07];
					i += printv((address+2)+(int)d, BIT16);
					continue; } }
			if(c & XBIT16) {
				i += printv(get_word(c & 0x07), BIT16);
				continue; }
			i += printv(memory[c & 0x07], BIT8);
			continue; }
		++i;
		putc(c, stdout); }

	return i;
}
