/*
 * Tables/Functions to implement the H8 cross disassembler
 */

/* Text strings that the disassembler needs */
char _cpu_[]	= "H8";			/* Name of CPU */
char _cmd_[]	= "DISH8";		/* Command name */
char _byte_[]	= "DB      ";	/* Directive to encode bytes */
char _word_[]	= "DW      ";	/* Directive to encode words */
char _string_[]	= "STR     ";	/* Directive to encode strings */

/*
 s = Source  8 bit (2nd byte Bits 7-4)
 d = Dest    8 bit (2nd byte Bits 3-0)
 t = Source 16 bit (2nd byte Bits 6-4)
 e = Dest   16 bit (2nd byte Bits 2-0)
 f = Dest    8 bit (1st byte Bits 3-0)
 a = Byte address  (1st byte Bits 7-0)
 b = Byte value    (1st byte Bits 7-0)
 x = Bit immdiate  (2nd byte Bits 6-4)
 y = 8-bit reg     (4th byte bits 7-3)
 z = Bit immediate (4th byte Bits 6-4)
 w = Word value    (Bytes 3 & 4)
 o = Opcode value  (Bytes 1 and 2)
 r = relative address
 c = 'C' for normal opcodes
*/

/* Table of instruction opcodes: MASK, COMPARE, TYPE/LEN, TEXT
 * First byte of text is FLAG:LENGTH, if B7 is set, then instruction
 * uses extended lookup table, and next byte is table index. */
char *itable[] = {
	0xFFF8, 0x6D70, "\x02POP e",
	0xFFF8, 0x6DF0, "\x02PUSH e",

	0xFF00, 0x0C00, "\x02MOV.B s,d",
	0xFF88, 0x0D00, "\x02MOV.W t,e",

	0xF000, 0xF000, "\x02MOV.B #b,f",
	0xFF80, 0x6800, "\x02MOV.B [t],d",
	0xFF80, 0x6E00, "\x04MOV.B {w,t},d",
	0xFF80, 0x6C00, "\x02MOV.B [t+],d",
	0xF000, 0x2000, "\x02MOV.B @<a,f",
	0xFFF0, 0x6A00, "\x04MOV.B @w,d",

	0xFFF8, 0x7900, "\x04MOV.W #w,e",
	0xFF88, 0x6900, "\x02MOV.W [t],e",
	0xFF88, 0x6F00, "\x04MOV.W {w,t},e",
	0xFF88, 0x6D00, "\x02MOV.W [t+],e",
	0xFFF8, 0x6B00, "\x04MOV.W @w,e",

	0xFF80, 0x6880, "\x02MOV.B d,[t]",
	0xFF80, 0x6E80, "\x04MOV.B d,{w,t}",
	0xFF80, 0x6C80, "\x02MOV.B d,[-t]",
	0xF000, 0x3000, "\x02MOV.B f,@<a",
	0xFFF0, 0x6A80, "\x04MOV.B d,@w",

	0xFF88, 0x6980, "\x02MOV.W e,[t]",
	0xFF88, 0x6F80, "\x04MOV.W e,{w,t}",
	0xFF88, 0x6D80, "\x02MOV.W e,[-t]",
	0xFFF8, 0x6B80, "\x04MOV.W e,@w",

	0xF000, 0x8000, "\x02ADD.B #b,f",
	0xFF00, 0x0800, "\x02ADD.B s,d",
	0xFF88, 0x0900, "\x02ADD.W t,e",

	0xFFF8, 0x0B00, "\x02ADDS #1,e",
	0xFFF8, 0x0B80, "\x02ADDS #2,e",

	0xF000, 0x9000, "\x02ADDX #b,f",
	0xFF00, 0x0E00, "\x02ADDX s,d",

	0xFF00, 0x1800, "\x02SUB.B s,d",
	0xFF88, 0x1900, "\x02SUB.W t,e",

	0xFFF8, 0x1B00, "\x02SUBS #1,e",
	0xFFF8, 0x1B80, "\x02SUBS #2,e",

	0xF000, 0xB000, "\x02SUBX #b,f",
	0xFF00, 0x1E00, "\x02SUBX s,d",

	0xF000, 0xE000, "\x02AND #b,f",
	0xFF00, 0x1600, "\x02AND s,d",

	0xF000, 0xC000, "\x02OR #b,f",
	0xFF00, 0x1400, "\x02OR s,d",

	0xF000, 0xD000, "\x02XOR #b,f",
	0xFF00, 0x1500, "\x02XOR s,d",

	0xFF00, 0x0600, "\x02ANDc #b,CCR",
	0xFF00, 0x0400, "\x02ORc #b,CCR",
	0xFF00, 0x0500, "\x02XORc #b,CCR",

	0xFF00, 0x4000, "\x02BRA r",
	0xFF00, 0x4100, "\x02BRN r",
	0xFF00, 0x4200, "\x02BHI r",
	0xFF00, 0x4300, "\x02BLS r",
	0xFF00, 0x4400, "\x02BCC r",
	0xFF00, 0x4500, "\x02BCS r",
	0xFF00, 0x4600, "\x02BNE r",
	0xFF00, 0x4700, "\x02BEQ r",
	0xFF00, 0x4800, "\x02BVC r",
	0xFF00, 0x4900, "\x02BVS r",
	0xFF00, 0x4A00, "\x02BPL r",
	0xFF00, 0x4B00, "\x02BMI r",
	0xFF00, 0x4C00, "\x02BGE r",
	0xFF00, 0x4D00, "\x02BLT r",
	0xFF00, 0x4E00, "\x02BGT r",
	0xFF00, 0x4F00, "\x02BLE r",
	0xFF00, 0x5500, "\x02BSR r",

	0xFF80, 0x7600, "\x02BAND #x,d",
	0xFF80, 0x7200, "\x02BCLR #x,d",
	0xFF00, 0x6200, "\x02BCLR s,d",
	0xFF80, 0x7680, "\x02BIAND #x,d",
	0xFF80, 0x7780, "\x02BILD #x,d",
	0xFF80, 0x7480, "\x02BIOR #x,d",
	0xFF80, 0x6780, "\x02BIST #x,d",
	0xFF80, 0x7580, "\x02BIXOR #x,d",
	0xFF80, 0x7700, "\x02BLD #x,d",
	0xFF80, 0x7100, "\x02BNOT #x,d",
	0xFF00, 0x6100, "\x02BNOT s,d",
	0xFF80, 0x7400, "\x02BOR #x,d",
	0xFF80, 0x7000, "\x02BSET #x,d",
	0xFF00, 0x6000, "\x02BSET s,d",
	0xFF80, 0x6700, "\x02BST #x,d",
	0xFF80, 0x7300, "\x02BTST #x,d",
	0xFF00, 0x6300, "\x02BTST s,d",
	0xFF80, 0x7500, "\x02BXOR #x,d",

	0xF000, 0xA000, "\x02CMP.B #b,f",
	0xFF00, 0x1C00, "\x02CMP.B s,d",
	0xFF88, 0x1D00, "\x02CMP.W t,e",

	0xFFF0, 0x0A00, "\x02INC d",
	0xFFF0,	0x0F00, "\x02DAA d",
	0xFFF0, 0x1F00, "\x02DAS d",
	0xFFF0, 0x1A00, "\x02DEC d",
	0xFFF0, 0x1780, "\x02NEG d",
	0xFFF0, 0x1700, "\x02NOT d",
	0xFFF0, 0x1280, "\x02ROTL d",
	0xFFF0, 0x1380, "\x02ROTR d",
	0xFFF0, 0x1200, "\x02ROTXL d",
	0xFFF0, 0x1300, "\x02ROTXR d",
	0xFFF0, 0x1080, "\x02SHAL d",
	0xFFF0, 0x1180, "\x02SHAR d",
	0xFFF0, 0x1000, "\x02SHLL d",
	0xFFF0, 0x1100, "\x02SHLR d",

	0xFF08, 0x5100, "\x02DIVXU s,e",

	0xFF8F, 0x5900, "\x02JMP [t]",
	0xFFFF, 0x5A00, "\x04JMP @w",
	0xFF00, 0x5B00, "\x02JMP @[<b]",
	0xFF8F, 0x5D00, "\x02JSR [t]",
	0xFFFF, 0x5E00, "\x04JSR @w",
	0xFF00, 0x5F00, "\x02JSR @[<b]",

	0xFF00, 0x0700, "\x02LDC #b,CCR",
	0xFFF0, 0x0300, "\x02LDC d,CCR",
	0xFFF0, 0x0200, "\x02STC CCR,d",

	0xFF08, 0x5000, "\x02MULXU s,e",

	0xFFFF, 0x5470, "\x02RTS",
	0xFFFF, 0x5670,	"\x02RTE",
	0xFFFF, 0x0000, "\x02NOP",
	0xFFFF, 0x0180, "\x02SLEEP",

	0xFFFF, 0x7B5C, "\x84\x00",
	0xFF8F, 0x7C00, "\x84\x01",
	0xFF8F, 0x7D00, "\x84\x02",
	0xFF00, 0x7E00, "\x84\x03",
	0xFF00, 0x7F00, "\x84\x04",

	/*  This entry always matches invalid opcodes */
	0x0000, 0x0000, "\x02DW o" };

char *reg8[] = {	/* 8 bit registers */
	"R0H", "R1H", "R2H", "R3H", "R4H", "R5H", "R6H", "R7H",
	"R0L", "R1L", "R2L", "R3L", "R4L", "R5L", "R6L", "R7L" };
char *reg16[] = {	/* 16 bit registers */
	"R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7" };

char *itable7B[] = {	/* 7B extended table */
	0xFFFF, 0x598F,	"EEPMOV",
	0x0000, 0x0000, "DW o,w" };

char *itable7C[] = {	/* 7C extended table */
	0xFF8F, 0x7600, "BAND #z,[t]",
	0xFF8F, 0x7680, "BIAND #z,[t]",
	0xFF8F, 0x7780, "BILD #z,[t]",
	0xFF8F, 0x7480, "BIOR #z,[t]",
	0xFF8F, 0x7580, "BIXOR #z,[t]",
	0xFF8F, 0x7700, "BLD #z,[t]",
	0xFF8F, 0x7400, "BOR #z,[t]",
	0xFF8F, 0x7300, "BTST #z,[t]",
	0xFF0F, 0x6300, "BTST y,[t]",
	0xFF8F, 0x7500, "BXOR #z,[t]",
	0x0000, 0x0000, "DW o,w" };

char *itable7D[] = {	/* 7D extended table */
	0xFF8F, 0x7200, "BCLR #z,[t]",
	0xFF0F, 0x6200, "BCLR y,[t]",
	0xFF8F, 0x6780, "BIST #z,[t]",
	0xFF8F, 0x7100, "BNOT #z,[t]",
	0xFF0F, 0x6100, "BNOT y,[t]",
	0xFF8F, 0x7000, "BSET #z,[t]",
	0xFF0F, 0x6000, "BSET y,[t]",
	0xFF8F, 0x6700, "BST #z,[t]",
	0x0000, 0x0000, "DW o,w" };

char *itable7E[] = {	/* 7E extended table*/
	0xFF8F, 0x7600, "BAND #z,<a",
	0xFF8F, 0x7680, "BIAND #z,<a",
	0xFF8F, 0x7780, "BILD #z,<a",
	0xFF8F, 0x7480, "BIOR #z,<a",
	0xFF8F, 0x7580, "BIXOR #z,<a",
	0xFF8F, 0x7700, "BLD #z,<a",
	0xFF8F, 0x7400, "BOR #z,<a",
	0xFF8F, 0x7300, "BTST #z,<a",
	0xFF0F, 0x6300, "BTST y,<a",
	0xFF8F, 0x7500, "BXOR #z,<a",
	0x0000, 0x0000, "DW o,w" };

char *itable7F[] = {	/* 7F extended table */
	0xFF80, 0x7200, "BCLR #z,<a",
	0xFF0F, 0x6200, "BCLR y,<a",
	0xFF8F, 0x6780, "BIST #z,<a",
	0xFF8F, 0x7100, "BNOT #z,<a",
	0xFF0F, 0x6100, "BNOT y,<a",
	0xFF8F, 0x7000, "BSET #z,<a",
	0xFF0F, 0x6000, "BSET y,<a",
	0xFF8F, 0x6700, "BST #z,<a",
	0x0000, 0x0000, "DW o,w" };

/* Index of extended tables */
char **etables[] = { &itable7B, &itable7C, &itable7D, &itable7E, &itable7F };

unsigned opcode, opcode1, *i_ptr;
unsigned char *inst_ptr;

#define	CPU_OPTION
char
	opt_std = -1;
char cpu_help[] = { "\
	-s	- enable Standard H8 style output\n" };
int cpu_option(int c)
{
	switch(c) {
	default: return 0;
	case 'S' :
		opt_std = 0;
		break; }
	return -1;
}

/*
 * Get a word from memory in the proper "endian"
 * H8 uses "Big endian", 1st byte is HIGH, 2nd byte is LOW
 */
unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Load an instruction into memory, determine it length, and
 * identify any symbol addresses it references.
 *
 * Entry:
 *	'memory' contains the first byte of the instruction only
 *	additional bytes may be obtained by calling getbyte()
 *	'dflag' indicates that symbol definitions should be performed.
 *	'bflag' indicates that byte symbols are to be defined.
 * Exit:
 *	'memory' must be filled with the remainder of the instruction
 *	'length' must be set to the length of the instruction
 *	set_symbol() must be called for values or addresses referenced
 *	in the operands of the instruction.
 */
load_instruction()
{
	unsigned i;
	char *ptr, c;

	memory[1] = getbyte();
	opcode = get_word(0);

	/* Search instruction table for this opcode */
	i_ptr = itable;
	while((*i_ptr++ & opcode) != *i_ptr++)
		++i_ptr;
	inst_ptr = *i_ptr;

	length = *inst_ptr & 15;

	/* Load instruction into memory */
	for(i=2; i < length; ++i)
		memory[i] = getbyte();

	if(*inst_ptr++ & 0x80) {
		opcode1 = get_word(2);
		i_ptr = etables[*inst_ptr];
		while((*i_ptr++ & opcode1) != *i_ptr++)
			++i_ptr;
		inst_ptr = *i_ptr; }

	if(dflag) {
		ptr = inst_ptr;
		for(;;) switch(*ptr++) {
		case 'a' :
			set_symbol((opcode & 255) | 0xFF00, BIT16);
			continue;
		case 'b' :
			set_symbol(opcode & 255, BIT8);
			continue;
		case 'w' :
			set_symbol(get_word(2), BIT16);
			continue;
		case 'r' :
			c = memory[1];
			set_symbol((address+2)+(int)c, BIT16);
			continue;
		case 0 :
			return; } }
}

/*
 * Display an instruction on the standard output stream
 *
 * Entry:
 *	'memory' - Contains the instruction data
 *	'length' - Contains length of instruction
 * Exit:
 *	instruction is written to stdout
 *	return value is length of string written
 *
 *	'printv()' can be used to output symbolic values
 */
display_instruction()
{
	unsigned i;
	char c;

	i = 0;
	while(*inst_ptr) switch(c = *inst_ptr++) {
		case 's' :
			i += printf(reg8[(opcode >> 4) & 0x0F]);
			break;
		case 'd' :
			i += printf(reg8[opcode & 0x0F]);
			break;
		case 'y' :
			i += printf(reg8[(memory[3] >> 4) & 0x0F]);
			break;
		case 't' :
			i += printf(reg16[(opcode >> 4) & 0x07]);
			break;
		case 'e' :
			i += printf(reg16[opcode & 0x07]);
			break;
		case 'f' :
			i += printf(reg8[(opcode >> 8) & 0x0F]);
			break;
		case 'a' :
			i += printv((opcode & 255) | (opt_std ? 0 : 0xFF00), BIT16);
			break;
		case 'b' :
			i += printv(opcode & 255, BIT8);
			break;
		case 'x' :
			i += printf("$%u", (opcode >> 4) & 0x07);
			break;
		case 'z' :
			i += printf("$%u", (memory[3] >> 4) & 0x07);
			break;
		case 'w' :
			i += printv(get_word(2), BIT16);
			break;
		case 'r' :		/* Relative address */
			c = memory[1];
			i += printv((address+2)+(int)c, BIT16);
			break;
		case 'o' :
			i += printf("$%04x", opcode);
			break;
		case ' ' :		/* Separator */
			do
				putc(' ', stdout);
			while(++i < 8);
			break;
		/* Special option handlers */
		case '[' :		/* Opening brace */
			if(!opt_std)
				c = '@';
			goto output;
		case ']' :		/* Closing brace */
		case '<' :		/* Direct indicator */
			if(!opt_std)
				continue;
			goto output;
		case '@' :		/* Indirect for extended references */
			if(opt_std)
				continue;
			goto output;
		case 'c' :		/* 'C' on CCR instructions */
			if(opt_std)
				continue;
			c = 'C';
			goto output;
		case '{' :		/* Indexed Indirection start */
			c = '[';
			if(!opt_std) {
				putc('@', stdout);
				++i;
				c = '(' ; }
			goto output;
		case '}' :		/* Indexed indirection close */
			c = opt_std ? ']' : ')' ;
			goto output;
		case '.' :		/* .B or .W suffix */
			if(opt_std) {
				if((*inst_ptr == 'B') || (*inst_ptr == 'W')) {
					++inst_ptr;
					continue; } }
		default:
		output:
			putc(c, stdout);
			++i; }

	return i;
}
