/*
 * ccc DMS -pofm
 * cc UDBASE -pofm MODE=2
 * cc TARGET -pofm
 * lc DMS UDBASE TARGET
 */
// #define	MEMCHK
// #define	DEMO	CFLEA
// #define	DEMOID	0xCFCF

#define	VERSION	1.0a
#include <stdio.h>
#include <window.h>
#include <file.h>
#include "dmscpu.h"

#define	POOL_SIZE	10000		// Size of data pool
#define	MAX_WATCH	19			// Maximum number of watches
#define	MAX_FILES	255			// Maximum number of files
#define	MAX_BREAK	8			// Maxumum number of breakpoints
#define	MAX_MEM		6			// Maximum number of memory types
#define	INDEX		128			// Index points

#define	MODE_EDIT	0x01		// Display edit information
#define	MODE_DIS	0x02		// Disassemble all the time

// Information in UDBASE loaded from the CPU definition file
extern unsigned
	kernel_id,					// Kernel ID
	com_speed,					// Interface speed
	Timeout,					// COM timeout
	sym_seg,					// Symbol segment
	sym_top,					// Symbol segment top marker
	sym_ptr,					// Symbol search pointer
	cpu_flags,					// Capabilities flags
	cpu_jmp_address,			// Alternate target address
	cpu_num_mem,				// Number of memory blocks
	cpu_num_reg,				// Number of registers defined
	cpu_num_func,				// Number of functions defined
	cpu_num_name;				// Number of instruction names defined
extern unsigned char
	*ARGV[],					// Global ARGV pointer
	com_port,					// Comm port
	cpu_bp_length,				// Length of a breakpoint
	cpu_id_flag,				// Indicate special instruction modes
	cpu_kc[],					// CPU kernel commands
	**cpu_memory,				// Address of memory block table
	**cpu_registers,			// Address of register name index table
	**cpu_functions,			// Address of function index table
	**cpu_inst_names,			// Address of instruction name index table
	*cpu_inst_data,				// Address of instruction data index table
	*cpu_reg_size,				// Register size table
	*cpu_default_regs;			// Address of default register values

unsigned char
	*cpu_name,					// Name of CPU to load
	*map_name,					// Name of MAP file to load
	Type,						// Last type selected
	Mode,						// Operational mode flags
	Dis_mode,					// Last display mode
	Dis_buffer[128],			// Disassembly buffer
	Dis_tabs[MAX_FILES],		// Disassembly tab size
	*parse_ptr,					// General parsing pointer
	**reg_datap,				// Register data pointers
	*files[MAX_FILES],			// Storage for file names
	case_mode = -1,				// Case insensitive
	break_store[MAX_BREAK][4],	// Breakpoint storage
	filename[65],				// File to read/write
	pool[POOL_SIZE];			// Misc. data storage pool
unsigned
	Dis_top,					// Address of top of display
	Dis_addr,					// Current display address
	Dis_offset,					// Display - X - offset
	Dis_tab = 8,				// Display tab size
	Dis_last,					// Last address displayed
	Dis_load = -1,				// Last buffer loaded
	Symbol,						// Last symbol selected
	watch_top,					// Size of watch list
	file_top,					// Filename top
	file_seg,					// File index segment
	file_lines[MAX_FILES],		// Max lines for file
	break_addr[MAX_BREAK],		// Breakpoint addresses
	break_top,					// Number of breakpoints
	index_top,					// Top of indexed lines
	open_file = -1,				// Indicates file open
	number_base = 2,			// Default number base
	index_l[(65535/INDEX)+2],	// Index points low (+maxlines)
	index_h[(65535/INDEX)+1],	// Index points high
	Step_pc,					// PC for stepping
	pool_top;					// Top of misc data pool

FILE
	*open_fp;					// Open file

struct WATCH_ENTRY {
	unsigned char watch_type;	// Memory type to watch
	unsigned char watch_mode;	// Type of display
	unsigned watch_address;		// Address to watch
	unsigned char watch_length;	// Watch length
	unsigned watch_symbol;		// Indicates symbol watching
	} watch_list[MAX_WATCH];	// List of watch points

/*dwatch()
{
	unsigned i;
	struct WATCH_ENTRY *w;
	for(i=0; i < watch_top; ++i) {
		w = watch_list[i];
		printf("\n%u %u %04x %u %u", w->watch_type, w->watch_mode,
		w->watch_address, w->watch_length, w->watch_symbol); }
}*/

// Window color definitions
#define	CO_REG		0			// Color index for register background
#define	CO_REGD		1			// Color index for register data
#define	CO_EDIT		2			// Color index for edit windows
#define	CO_PROMPT	3			// Color index for prompt windows
#define	CO_SOURCE	4			// Color index for source window
#define	CO_WATCH	5			// Color index for watch window
#define	CO_DEBUG	6			// Color index for debug window
#define	CO_WARN		7			// Color index for warning windows
#define	CO_ADDR		8			// Color index for current address
#define	CO_BREAK	9			// Color index for breakpoint
#define	CO_HELPS	10			// Color index for Help screen
#define	CO_HELPL	11			// Color index for help links
#define	CO_HELPC	12			// Color index for help cursor
unsigned char colors[] = {
	0x70,						// 0: Register data
	0x76,						// 1: Register name
	0x67,						// 2: Edit
	0x70,						// 3: Prompts
	0x17,						// 4: Source
	0x57,						// 5: Watch
	0x30,						// 6: Debug
	0x4E,						// 7: Warnings
	0x6E,						// 8: Current address
	0x1A,						// 9: Breakpoint
	0x70,						//10: Help screen
	0x71,						//11: Help links
	0x17 };						//12: Help cursor

struct WINDOW
	*rwin,						// Register window
	*swin,						// Source window
	*wwin,						// Watch window
	*dwin;						// Debug window

// Constant tables
#define	WT_DUMP		0
#define	WT_BDEC		1
#define	WT_WDECL	2
#define	WT_WDECB	3
#define	WT_STRING	4
char *watch_mode_table[] = {
	"Hex bytes",
	"Decimal bytes",
	"LE Decimal words",
	"BE Decimal words",
	"String",
	0 };

// Available numeric base modes
char *number_base_table[] = {
	"Binary (2)",
	"Octal (8)",
	"Decimal (10)",
	"Hexidecimal (16)",
	0 };

char cmd_help[] = { "\nDunfield MicroScope v"#VERSION" - "#__DATE__"\n\n\
For help, use: DMS ?\n\n\
?COPY.TXT 2001-2005 Dave Dunfield\n**See COPY.TXT**.\n" };

#include "help.c"

/*
 * Scale to a fractional value
 */
scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Extract a 0-terminated string from a high-bit terminated string
 */
void extracth(unsigned char *d, unsigned char *s)
{
	do {
		if((*d = *s & 0x7F) < ' ')
			break;
		++d; }
	while(!(*s++ & 0x80));
	*d = 0;
}

/*
 * Display symbol type
 */
char *get_sym_type(char *buf, unsigned t)
{
	static char *sym_type_text[] = {
		"Constant",	// 0
		"?1",		// 1
		"?2",		// 2
		"?3",		// 3
		"?4",		// 4
		"?5",		// 5
		"?6",		// 6
		"Stack" };	// 7

	if(t && (t <= cpu_num_mem)) {
		extracth(buf, cpu_memory[t-1]+4);
		return buf; }
	return sym_type_text[t];
}

/*
 * Indicate a function not supported
 */
int test_support(unsigned test, char *m)
{
	if(!test) {
		wprintf("\n%s not supported by this target!", m);
		return -1; }
	return 0;
}

/*
 * Skip blanks in input stream
 */
int skip_blanks(void)
{
	while(isspace(*parse_ptr))
		++parse_ptr;
	return *parse_ptr;
}

/*
 * Test for a valid symbol character
 */
int issymbol(unsigned char c)
{
	return	((c >= 'A') && (c <= 'Z'))
		||	((c >= 'a') && (c <= 'z'))
		||	((c >= '0') && (c <= '9'))
		||	(c == '_') || (c == '?');
}

/*
 * Display a title on a window
 */
void title(unsigned char *ptr)
{
	char buffer[8];
	memcpy(buffer, W_OPEN, sizeof(buffer));
	*W_OPEN = REVERSE;
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, sizeof(buffer));
}

/*
 * Test for single HEX digit return value (-1 if not)
 */
unsigned gethex(int c)
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	return -1;
}

/*
 * Get a single value element from the input line
 */
int get_value_element(unsigned *dest)
{
	unsigned i, j, b, v, l;
	unsigned char *p, c, rf, temp[50], temp1[50];
	static unsigned char xbase[] = { 2, 8, 10, 16 };

	Symbol = v = l = rf = 0;
	p = temp;

	switch(skip_blanks()) {
	case '$' :	b = 16;	goto donum;			// Hex
	case '@' :	b = 8;	goto donum;			// Octal
	case '%' :	b = 2;	goto donum;			// Binary
	case '.' :	b = 10; goto donum;
	case '0' :	case '1' :	case '2' :	case '3' :	case '4' :
	case '5' :	case '6' :	case '7' :	case '8' :	case '9' :
		--parse_ptr;
		b = xbase[number_base];
	donum:
		++parse_ptr;
		while((i = gethex(*parse_ptr)) < b) {
			v = (v * b) + i;
			++parse_ptr;
			++l; }
		if(l) {
			*dest =  v;
			return 0; }
		return -1;
	case '\'':								// Character value
		while(*++parse_ptr != '\'') {
			if((++l > 2) || !*parse_ptr)
				return -1;
			v = (v << 8) + *parse_ptr; }
		++parse_ptr;
		if(!l)
			return -1;
		*dest = v;
		return 0;
	case '=' :								// Register
		++parse_ptr;
		rf = -1; }

	while(issymbol(c = *parse_ptr)) {
		*p++ = (rf|case_mode) ? toupper(c) : c ;
		++parse_ptr;
		++l; }
	if(!l)
		return -1;
	*p = 0;

	if(rf) {				// Lookup register
		for(i=0; i < cpu_num_reg; ++i) {
			extracth(temp1, cpu_registers[i]);
			if(!strcmp(temp, temp1))
				goto fnd_reg; }
		return -1;
	fnd_reg:
		if(!(j = cpu_reg_size[i] & 0x07))
			return -1;
		p = reg_datap[i] + j;
		do {
			v = (v << 8) | *--p; }
		while(--j);
		*dest = v;
		return 0; }

	// Lookup symbol
	sym_ptr = 0;
	while(Symbol = lookup_symbol(temp)) {
		i = peek(sym_seg, sym_ptr);
		v = peekw(sym_seg, Symbol);
		wopen(0, 10, 80, 3, WSAVE|WCOPEN|WBOX1|colors[CO_WARN]);
		wprintf("%s [%s] %04x(%u)", temp, get_sym_type(temp1, i >> 5), v, v);
//wclose();
//skip_symbol();
//continue;
		if(peek(sym_seg, sym_ptr+3) & 0x80) {
			i = sym_ptr + (i & 0x1F);
			wprintf(" in-scope %04x-%04x", peekw(sym_seg, i+3), peekw(sym_seg, i+5)); }
		wprintf(" Ok(Y/N)?", temp, i, v, v);
	xx:	switch(wgetc()) {
			default: goto xx;
			case 'y' :
			case 'Y' :
				wclose();
				*dest = v;
				return 0;
			case 0x1B :
				wclose();
				return -1;
			case 'n' :
			case 'N' :
				wclose(); }
		skip_symbol(); }

	return -1;
}

/*
 * Get a value element
 */
unsigned get_value(char *prompt, unsigned *dest)
{
	unsigned i, p, v, v1, s;
	unsigned char c, buffer[80];

	wopen(0, 10, 80, 3, WSAVE|WCOPEN|WBOX1|colors[CO_PROMPT]);
	wcursor_line();
reprompt:
	wputs(prompt);
	p = Type = 0;
top:
	switch(i = wgetc()) {
	case _K1 :
		help(HELP_VALUE);
		goto top;
	case 0x1B :		// Abort
		wclose();
		return -1;
	case _KHO :
	case _KEN :
		wclwin();
		goto reprompt;
	case _KBS :
	case _KLA :
		if(p) {
			wputs("\b \b");
			--p;
			goto top; }
	default:
		if((p < 50) && (i > ' ') && (i < 0x7F)) {
			wputc(i);
			buffer[p++] = i;
			goto top; }
	error:
		beep(600, 100);
		goto top;
	case '\n' : }
	if(!p)
		goto error;
	buffer[p] = 0;

	parse_ptr = buffer;
	if(get_value_element(&v))
		goto error;
	if(s = Symbol)
		Type = peek(sym_seg, sym_ptr) >> 5;
	for(;;) {
		if(!(c = skip_blanks())) {
			*dest = v;
			wclose();
			return 0; }
		++parse_ptr;
		if(get_value_element(&v1))
			goto error;
		if(!s)
			s = Symbol;
		switch(c) {
		case '+' : v += v1;	continue;
		case '-' : v -= v1;	continue;
		case '*' : v *= v1; continue;
		case '/' : v /= v1; continue;
		case '%' : v %= v1; continue;
		case '&' : v &= v1; continue;
		case '|' : v |= v1; continue;
		case '^' : v ^= v1; continue;
		case '>' : v >>= v1; continue;
		case '<' : v <<= v1; continue; }
		goto error; }
}

/*
 * Get a string 
 */
int get_string(char *prompt, char *dest, unsigned l)
{
	unsigned i;
	wopen(0, 10, 80, 3, WSAVE|WBOX3|WCOPEN|colors[CO_PROMPT]);
	i = strlen(prompt);
	wputs(prompt);
	for(;;) switch(wgets(i, 0, dest, l)) {
		case '\n' :
			wclose();
			return 0;
		case 0x1B :
			wclose();
			return -1;
		case _K1 :
			help(HELP_SEDIT); }
}

/*
 * Update the title window
 */
void update_status(unsigned char Md)
{
	unsigned i, w;
	unsigned char c, d, e, *data, temp[50], *p, *p1;

	c = Mode & MODE_EDIT;
	if((Md & MODE_EDIT) != c)
		return;

	w = W_OPEN;
	W_OPEN = rwin;
	wgotoxy(0, 0);

	if(c) {
		if(Dis_mode)
			wprintf("%-65s %5u %7u", files[Dis_mode-1], Dis_top+1, index_l[index_top]);
		else
			wprintf("%-65s %04x", "$Disassembly$", Dis_top);
		wcleol();
		W_OPEN = w;
		return; }

	for(i=0; i < cpu_num_reg; ++i) {
		if(d = (c = cpu_reg_size[i]) & 0x07) {
			if(c & 0x08)
				continue;
			p = cpu_registers[i];
			data = reg_datap[i] + d;
			// Extract name and truncate if necessary
			p1 = temp;
			do {
				if((e = *p & 0x7F) < ' ')
					break;
				*p1++ = e; }
			while(!(*p++ & 0x80));
			p = temp;
			if(e = (c & 0x30) >> 4) do {
				--p1; }
				while(--e);
			if(e = (c & 0xC0) >> 6) do {
				++p; }
				while(--e);
			*p1 = 0;
			*W_OPEN = colors[CO_REGD];
			wputs(p);
			*W_OPEN = colors[CO_REG];
			while(d--)
				wprintf("%02x", *--data);
			wputc(' '); } }
	W_OPEN = w;
}

/*
 * Seek to a line in the file
 */
int seek_line(void)
{
	int c;
	unsigned l;

	// Index file and seek to line
	if(!index_file(Dis_mode-1))
		return -1;
	if(!(l = Dis_top >> 7))
		rewind(open_fp);
	else {
		if(l > index_top) {
	eof: 	w_printf(dwin, "\nLine %u not found!", Dis_top+1);
			return -1; }
		fseek(open_fp, index_h[l-1], index_l[l-1], 0); }
	l = (file_lines[Dis_mode-1] = Dis_top) & (INDEX-1);
	while(l) {
		if((c = getc(open_fp)) == EOF)
			goto eof;
		if(c == '\n')
			--l; }
	return 0;
}

/*
 * Update the source code display window from Dis_top
 */
void update_display(void)
{
	int c;
	unsigned i, j, t, w, wy, l, lb, ox, btop, bline[MAX_BREAK];

	w = W_OPEN;						// Switch to display window
	W_OPEN = swin;
	wy = W_OPEN->WINheight;			// Save current window size

	if(!Dis_mode)
		goto do_dis;				// Disassembly is active

	// Locate any lines with breakpoints in this file
	for(i=btop=0; i < break_top; ++i) {
		if(peek(file_seg, j = break_addr[i]) == Dis_mode) {
			bline[btop++] =
				(peek(file_seg+4096, j) | (peek(file_seg+8192, j) << 8))-1; } }

	// If file active, get line number for current address
	l = -1;
	if(peek(file_seg, j = *(unsigned*)reg_datap[0]) == Dis_mode) {
		l = (peek(file_seg+4096, j) |
			(peek(file_seg+8192, j) << 8)) - 1; }

	if(seek_line())			// Seek to line
		goto do_exit;

	// Display the source file
	ox = Dis_offset + 80;
	lb = Dis_top;
	if(!(t = Dis_tabs[Dis_mode-1]))
		t = Dis_tab;
	for(i=0; i < wy; ++i) {
		wgotoxy(0, i);
		for(j=0; j < btop; ++j) {
			if(bline[j] == lb)
				*W_OPEN = colors[CO_BREAK]; }
		if(lb == l)
			*W_OPEN = colors[CO_ADDR];
		j = 0;
		while((c = getc(open_fp)) != '\n') {
			if(c == EOF)
				break;
			if(c == '\r')
				continue;
			if(c == '\t') {
				do {
					if((j >= Dis_offset) && (j < ox))
						wputc(' '); }
				while(++j % t); }
			else {
				if((j >= Dis_offset) && (j < ox))
					wputc(c);
				++j; } }
		if(W_OPEN->WINcurx < 79)
			wcleol();
		++lb;
		*W_OPEN = colors[CO_SOURCE]; }
	goto do_exit;

// No source file available - Display as a disassembly
do_dis:
	if(Dis_load != Dis_top)
		target_read_memory(0, Dis_load = Dis_top, sizeof(Dis_buffer), Dis_buffer);

	// Display disassemly output
	Dis_last = Dis_top;
	i = lb = 0;
	for(;;) {
		wgotoxy(0, i);
		for(j=0; j < break_top; ++j)
			if(break_addr[j] == Dis_last)
				*W_OPEN= colors[CO_BREAK];
		if(Dis_last == *(unsigned*)reg_datap[0])
			*W_OPEN = colors[CO_ADDR];
		l = disassemble(Dis_buffer+lb, Dis_last, -1);
		wcleol();
		*W_OPEN = colors[CO_SOURCE];
		if(++i >= wy)
			break;
		lb += l;
		Dis_last += l; }

do_exit:
	W_OPEN = w;
	update_status(MODE_EDIT);
	return;
}

/*
 * Adjust value for big/little endian
 */
unsigned adjust_value(unsigned v)
{
	if(cpu_flags & CPU_BENDIAN)
		return (v << 8) | (v >> 8);
	return v;
}

/*
 * Edit the registers
 */
void edit_registers(void)
{
	unsigned i, j, k, pl, pp, rn, ys, yl, ds, dv;
	unsigned char a1, a2, c, d, *p1, *p2, *dp, temp[50];
	static unsigned char
		mdsize,		// Memory dump size
		cy,			// Y cursor position
		py;			// Y page position

	a1 = colors[CO_EDIT];
	a2 = (a1 >> 4) | (a1 << 4);

	for(rn = i = 0; i < cpu_num_reg; ++i)
		if(cpu_reg_size[i] & 0x07)
			++rn;
	yl = (ys = (rn < 20) ? rn : 20) -1;
	wopen(20, 1, 40, ys+2, WSAVE|WCOPEN|WBOX1|colors[CO_EDIT]);
	title("REGISTERS");
	wcursor_off();
top:
	wgotoxy(0, 0);
	for(i = pp = 0; i < cpu_num_reg; ++i) {
		if(!(d = (c = cpu_reg_size[i]) & 0x07))
			continue;				// Not displayed at all
		if(pp++ < py)				// Not displayed yet
			continue;
		pl = (pp - py)-1;
		if(pl >= ys)			// Past bottom of page
			break;
		wgotoxy(1, pl);			// Position on screen
		p1 = temp;
		p2 = cpu_registers[i];
		do {					// Extract register name
			if((c = *p2 & 0x7F) < ' ')
				break;
			*p1++ = c; }
		while(!(*p2++ & 0x80));
		*p1 = 0;
		p1 = reg_datap[i] + d;	// Extract data values
		if(pl == cy) {
			*W_OPEN = a2;
			ds = d;
			dp = reg_datap[i]; }
		wprintf("%s=", temp);
		dv = 0;					// Display the data
		while(d--) {
			dv = (dv << 8) | *--p1;
			wprintf("%02x", *p1); }
		*W_OPEN = a1;
		wcleol();
		if(c < ' ') {			// Special display mode
			j = W_OPEN->WINwidth;
			if(c == R_BIT) {		// Bit pattern
				k = (d = cpu_reg_size[i] & 0x07) * 8;
				p1 = reg_datap[i] + d;
				wgotoxy(j-k, pl);
				for(j=0; j < k; ++j) {
					d = 0x80 >> (j & 7);
					if(*(p1-((j/8)+1)) & d)
						*W_OPEN = a2;
					wputc(p2[j+1] & 0x7F);
					*W_OPEN = a1; } }
			else {					// Memory pointer
				if(k = mdsize) {
					c -= R_PTR;
					wgotoxy(j-(k*3), pl);
					target_read_memory(c, dv, k, p2 = temp);
					while(k--)
						wprintf(" %02x", *p2++); } } } }

	switch(c = wgetc()) {
	case _K1 :
		help(HELP_REGE);
		goto top;
	case '+' :
		if(mdsize < 8)
			++mdsize;
		goto top;
	case '-' :
		if(mdsize)
			--mdsize;
		goto top;
	case 0x1B :
	case '\n' :
		wclose();
		update_status(0);
		return;
	case _KUA :
		if(cy)
			--cy;
		else if(py)
			--py;
		goto top;
	case _KDA :
		if((cy+py+1) >= rn)
			goto top;
		if(cy < yl)
			++cy;
		else
			++py;
		goto top;
	case _KPU :
		if(cy) {
			cy = 0;
			goto top; }
		if(py > ys) {
			py -= ys;
			goto top; }
	case _KHO :
		cy = py = 0;
		goto top;
	case _KPD :
		if(cy < yl) {
			cy = yl;
			goto top; }
		if((py += ys) < (rn-ys))
			goto top;
	case _KEN :
		py = rn-ys;
		cy = yl;
		goto top;
	case ' ' :
		if(!get_value("Value? ", &i)) {
			while(ds--) {
				*dp++ = i;
				i >>= 8; } }
		goto top; }

	if((c = gethex(c)) <= 15) {		// Insert hex digits
		while(ds--) {
			d = *dp >> 4;
			*dp = (*dp << 4) | c;
			++dp;
			c = d; } }
	goto top;
}

/*
 * Display a line for the editor
 */
void draw_edit_line(unsigned addr, unsigned char *buffer, unsigned offset)
{
	unsigned int i, c;

	wprintf("%04x  ", addr+offset);
	for(i=0; i < 16; ++i)
		wprintf("%02x ", buffer[i]);
	wputc(' ');
	for(i=0; i < 16; ++i) {
		c = buffer[i];
		wputc(((c >= ' ') && (c <= 0x7E)) ? c : '.'); }
}

/*
 * Get memory type
 */
unsigned get_memory_type(void)
{
	unsigned i;
	unsigned char *p, *l[MAX_MEM+1];
	static unsigned t;

	// Do not prompt when no type is required
	if(cpu_num_mem <= 1)
		return 0;

	// Extract the memory type names
	p = &pool[pool_top];
	for(i=0; i < cpu_num_mem; ++i) {
		extracth(l[i] = p, cpu_memory[i]+4);
		while(*p++); }
	l[i] = 0;

	return wmenu(20, 7, WSAVE|WBOX1|WCOPEN|colors[CO_PROMPT], l, &t)
		? -1 : t;

}

/*
 * Edit global memory
 */
void edit_memory(void)
{
	unsigned
		i, j,			// Misc variables
		type,			// Type of memory to edit
		dt,				// Display top
		dc,				// Display cursor
		*edparm,		// Pointer to edit parameters
		offset,			// Offset to start of memory addresses
		top,			// Top of memory (from 0 offset)
		esize,			// Size of edit screen
		editp;			// Current edit position
	unsigned char
		buffer[256],	// Modification buffer
		editm;			// Edit mode (Hex/ASCII)
	static unsigned
		Xeditp[MAX_MEM];	// Saved edit positions
	static unsigned char
		Xeditm[MAX_MEM];	// Saved edit modes

	if((type = get_memory_type()) & 0xFF00)
		return;

	edparm = cpu_memory[type];
	editp = Xeditp[type];
	editm = Xeditm[type];
	offset = edparm[0];
	top = edparm[1] - offset;
	esize = (top > 256) ? 256 : (top+1);

	Dis_load = -1;		// Insure disassembler flushed

rewin:
	i = (esize / 16) + 2;
	if(esize & 0x0F)
		++i;
	wopen(3, 3, 73, i, WSAVE|WBOX1|WCOPEN|colors[CO_EDIT]);

	extracth(buffer, &edparm[2]);
	strcat(buffer, " Memory");
	title(buffer);
	memset(buffer, 0, sizeof(buffer));

recursor:
	wcursor_line();
redraw:
	dt = editp & 0xFF00;
	target_read_memory(type, dt+offset, esize, buffer);

	// Display the current edit screen
	for(i=j=0; i < 16; ++i, j += 16) {
		wgotoxy(0, i);
		if((dt+j) > top) {
			wcleow();
			break; }
		draw_edit_line(dt+j, buffer+j, offset); }

prompt:
	if(editp > top)
		editp = (editp >= (65535-15)) ? top : 0;
	if((editp & 0xFF00) != dt)
		goto redraw;
	dc = editp & 0xFF;
	wgotoxy(editm ? (dc%16)+55 : ((dc%16)*3)+6, dc/16);
	switch(toupper(i = wgetc())) {
		case 0x1B :		/* Exit editor */
			wclose();
			Xeditp[type] = editp;
			Xeditm[type] = editm;
			return;
		case _K1 :		/* Help request */
			help(HELP_EDIT);
			break;
		case _K2 :		/* Go to address */
			get_value("Edit address? ", &editp);
			editp -= offset;
			break;
		case _K3 :		/* Toggle mode */
			editm = !editm;
			break;
		case _K4 :		/* Fill memory */
			if(!cpu_kc[KCMD_WRITE1+type])
				goto prompt;
			if(get_value("Starting address? ", &dc))
				break;
			if((dc -= offset) > top)
				break;
			if(get_value("Ending address? ", &dt))
				break;
			if(dt < dc) {
				i = dt;
				dt = dc;
				dc = i; }
			if(((dt -= offset) > top) || (dt < dc))
				break;
			if(get_value("Fill with? ", &i))
				break;
			wcursor_off();
			memset(buffer, i, sizeof(buffer));
			wclose();
			wprintf("\nFilling: %04x-", dc+offset);
			j = W_OPEN->WINcurx;
			do {
				wprintf("%04x", dc+offset);
				W_OPEN->WINcurx = j;
				if(wtstc() == 0x1B)
					break;
				if((i = dt - dc) > 255)
					i = 255;
				++i;
				target_write_memory(type, dc+offset, i, buffer); }
			while(((dc += i)-1) < dt);
			wprintf("%04x", dc-1);
			goto rewin;
		case _K5 :		/* Refresh screen */
			goto recursor;
		case _KLA :
			--editp;
			break;
		case _KPU :
			editp -= 256;
			break;
		default:
			if(!cpu_kc[KCMD_WRITE1+type])
				goto prompt;
			if(editm) {
				if(i >= 128)
					break;
				buffer[dc] = i; }
			else {
				if((j = gethex(i)) >= 16)
					break;
				wputc(i);
				if((i = gethex(wgetc())) >= 16) {
					--editp;
					beep(600, 100);
					goto fixline; }
				buffer[dc] = (j<<4)+i; }
			target_write_memory(type, editp+offset, 1, &buffer[dc]);
		fixline:
			wgotoxy(0, i = dc/16);
			i *= 16;
			draw_edit_line(dt+i, buffer+i, offset);
		case _KRA :
			++editp;
			break;
		case _KPD :
			editp += 256;
			break;
		case _KHO :
			editp = (editp-1) & 0xFFF0;
			break;
		case _KUA :
			editp -= 16;
			break;
		case _KEN :
			editp = (editp+1) | 0x000F;
			break;
		case _KDA :
			editp += 16;
			break;
		case _CPU :
			editp = 0;
			break;
		case _CPD :
			editp = top; }
	goto prompt;
}

/*
 * Open the working windows
 */
void open_windows(char *m)
{
	unsigned x, s, d, o;

	x = 24 - watch_top;
	s = (x * 3) / 4;
	d = x - s;
	o = watch_top + 1;
	if(watch_top)
		wwin = wopen(0, 1, 80, watch_top, WSAVE|WCOPEN|colors[CO_WATCH]);
	swin = wopen(0,  o, 80, s, WSAVE|WCOPEN|colors[CO_SOURCE]);
	dwin = wopen(0, o+s, 80,  d, WSAVE|WCOPEN|WSCROLL|colors[CO_DEBUG]);
	wcursor_off();
	wputs(m);
}

/*
 * Close the working windows
 */
void close_windows(void)
{
	wclose();
	wclose();
	if(watch_top)
		wclose();
}

/*
 * Add an entry to the watch table
 */
void add_watch(void)
{
	unsigned a, l, *mp;
	struct WATCH_ENTRY *wp;
	static unsigned m;
	static unsigned rwlen[] = {
		16,			// Byte dump
		16,			// Byte decimal
		8,			// Word LE
		8,			// Word BE
		60 };		// String

	if(watch_top >= MAX_WATCH) {
		wprintf("\nWatch table full");
		return; }
	wp = watch_list[watch_top];

	if(get_value("Location? ", &a))
		return;

	wp->watch_address = a;
	wp->watch_symbol = Symbol;

	if(--Type >= cpu_num_mem)
		if((Type = get_memory_type()) >= MAX_MEM)
			return;

	mp = cpu_memory[wp->watch_type = Type];
	if((mp[0] > a) || (mp[1] < a)) {
		wprintf("\nOut of range!");
		return; }

	if(wmenu(20, 7, WSAVE|WCOPEN|WBOX1|colors[CO_PROMPT], watch_mode_table, &m))
		return;
	wp->watch_mode = m;

reget:
	if(get_value("Length to watch (ESC=max)? ", &l))
		l = rwlen[m];
	if(l > rwlen[m]) {
		wprintf("\nLength too long, max=%u", rwlen[m]);
		goto reget; }
	wp->watch_length = l;

	close_windows();
	++watch_top;
	open_windows("Watch added");
	update_display();
}

/*
 * Remove a watch
 */
void delete_watch(void)
{
	unsigned p;

	if(!watch_top)
		return;

	wopen(0, 0, 80, 1, WSAVE|WCOPEN|colors[CO_WARN]);
	wputs(" Select watch to erase");

	p = 0;
	wcursor_block();
	for(;;) {
		w_gotoxy(0, p, wwin);
		switch(w_getc(wwin)) {
		case _KUA :
			p = p ? p-1 : watch_top-1;
			continue;
		case _KDA :
			if(++p >= watch_top)
				p = 0;
			continue;
		case '\n' :
			wclose();
			while(++p < watch_top)
				memcpy(watch_list[p-1], watch_list[p], sizeof(struct WATCH_ENTRY));
			close_windows();
			--watch_top;
			open_windows("Watch removed");
			update_display();
			wcursor_off();
			return;
		case 0x1B :
			wclose();
			wcursor_off();
			return; } }
}

/*
 * Update watches
 */
void update_watch(void)
{
	unsigned a, i, s, l, l1, ls;
	unsigned char t, m, buffer[60], a1, a2, *p;
	struct WINDOW *sw;

	struct WATCH_ENTRY *w;

	a1 = *(sw = W_OPEN);
	W_OPEN = wwin;
	a2 = (a1 >> 4) | (a1 << 4);

	ls = cpu_num_mem ? 15 : 16;

	for(i=0; i < watch_top; ++i) {
		wgotoxy(0, i);
		w = watch_list[i];
		a = w->watch_address;
		t = w->watch_type;
		m = w->watch_mode;

		if(cpu_num_mem)
			wprintf("%u", t+1);

		if(s = w->watch_symbol) {
			l1 = l = peek(sym_seg, s-1) & 0x1F;	// Get length
			while(l1 < ls) {
				wputc(' ');
				++l1; }
			if(l > ls)
				l = ls;
			++s;							// Skip value
			while(l--)
				wputc(peek(sym_seg, ++s) & 0x7F); }
		else {
			l = ls-5;
			do {
				wputc(' '); }
			while(--l);
			wprintf("$%04x", a); }

		l = w->watch_length;
		target_read_memory(t, a, ((m==WT_WDECB)||(m==WT_WDECL))?l*2:l, p = buffer);
//w_printf(dwin, "\nR:%u %04x %u/%u %u", t, a, l,l1, p);

		switch(m) {
		case WT_DUMP :
			do {
				wprintf("  %02x", m = *p++); }
			while(--l);
			break;
		case WT_BDEC :
			do {
				wprintf("%4u", *p++); }
			while(--l);
			break;
		case WT_WDECL :
			do {
				s = *p++;
				s = (*p++ << 8) + s;
				wprintf("%8u", s); }
			while(--l);
			break;
		case WT_WDECB :
			do {
				s = *p++;
				s = (s << 8) + *p++;
				wprintf("%8u", s); }
			while(--l);
			break;
		case WT_STRING :
			wputc(' ');
			do {
				if(!(m = *p++))
					break;
				if((m <= ' ') || (m >= 0x7F))
					m = 0xA8;
				wputc(m); }
			while(--l);
			wcleol(); } }

	W_OPEN = sw;
}

/*
 * Read value from parse_ptr
 */
unsigned read_value(unsigned b)
{
	unsigned v, c;
	skip_blanks();
	v = 0;
	while((c = gethex(*parse_ptr)) < b) {
		v = (v * b) + c;
		++parse_ptr; }
	return v;
}
void load_symbolics(char *filename)
{
	unsigned f, v, sh, sl;
	unsigned char t, buffer[100], temp[50], *p;
	FILE *fp;

	fp = fopen(filename, "rvq");
	while(fgets(parse_ptr = buffer, sizeof(buffer)-1, fp)) {
		switch(skip_blanks()) {
		case 'C' :		// Variable definitions
		case '0' : t = 0; goto dosym;
		case '1' : t = 1; goto dosym;
		case '2' : t = 2; goto dosym;
		case '3' : t = 3; goto dosym;
		case '4' : t = 4; goto dosym;
		case '5' : t = 5; goto dosym;
		case '6' : t = 6; goto dosym;
		case 'S' :
		case '7' : t = 7;
		dosym:
			++parse_ptr;
			p = temp;
			while(issymbol(*parse_ptr)) {
				v = *parse_ptr++;
				*p++ = case_mode ? toupper(v) : v; }
			*p = 0;
			f = read_value(10);
			v = read_value(16);
			sh = read_value(16);
			sl = read_value(16);
			if(define_symbol(temp, t, v, sh, sl))
				abort("Symbol table overflow");
//printf("Symbol '%s' %u %04x %x-%x\n", temp, t, v, sh, sl);
		case ';' :
		case 0 :
			continue;
		case 'F' :		// File definition
			if(file_top >= MAX_FILES)
				abort("Too many files");
			files[file_top] = &pool[pool_top];
			++parse_ptr;
			while(*parse_ptr && !isspace(*parse_ptr))
				pool[pool_top++] = toupper(*parse_ptr++);
			pool[pool_top++] = 0;
//printf("File %u : %s\n", file_top, files[file_top]);
			++file_top;
			continue;
		case 'A' :		// Address line mapping
			++parse_ptr;
			v = read_value(16);
			if(!(f = read_value(10)))
				abort("Bad address record\n");
			sh = read_value(10);
			poke(file_seg, v, f);
			poke(file_seg+4096, v, sh);
			poke(file_seg+8192, v, sh >> 8);
//printf("Addr:%04x %u %u\n", v, f, sh);
			continue; }
	fputs(buffer, stderr);
	abort("\nUnknown map file input\n"); }
}

/*
 * Index a file to determine line number positions
 */
int index_file(unsigned file)
{
	int c;
	unsigned sl, sh, lc;
	unsigned char *p;

	if(file != open_file) {
		index_top = sl = sh = lc = 0;
		open_file = file;
		if(open_fp)
			fclose(open_fp);
		w_printf(dwin, "\nSource file: %s", p = files[file]);
		if(open_fp = fopen(p, "rb")) {
			while((c = getc(open_fp)) != EOF) {
				if(!++sl)
					++sh;
				if(c == '\n') {
					if(!(++lc & (INDEX-1))) {
						index_l[index_top] = sl;
						index_h[index_top++] = sh; } } }
			index_l[index_top] = lc;
			return open_fp; }
		w_puts(" - unable to open", dwin);
		return 0; }
	return open_fp;
}

/*
 * Move the source code display to a specific address
 */
void display_address(unsigned a)
{
	unsigned i, wy, f, l, lb, ox;

	wy = swin->WINheight;

	// Disassembly only mode selected
	if(Mode & MODE_DIS)
		goto do_dis;

	// If no source file for this address - disassemble
	if(!(f = peek(file_seg, Dis_addr = a)))
		goto do_dis;

	// Determine line number in file
	l = (peek(file_seg+4096, a) | (peek(file_seg+8192, a) << 8)) - 1;

	// If not already in this file, display from here
	if(Dis_mode != f) {
		Dis_top = l;
		Dis_mode = f; }

	// Determine if current line is visible on display
	if((l - Dis_top) >= wy)			// Outside of range
		Dis_top = l;

	goto exit1;

// No source file available - Display as a disassembly
do_dis:

	if(Dis_mode) {		// File currently on display
		Dis_top = a;
		Dis_mode = 0; }

	if(Dis_load != Dis_top)
		target_read_memory(0, Dis_load = Dis_top, sizeof(Dis_buffer), Dis_buffer);

	// Scan for this address from current top
	ox = Dis_top;
	for(i=lb=0; i < wy; ++i) {
		if(ox == a)
			goto exit1;
		l = disassemble(Dis_buffer+lb, ox, 0);
		lb += l;
		ox += l; }

	// Does not fit - reset address
	if(Dis_load != a)
		target_read_memory(0, Dis_load = Dis_top = a, sizeof(Dis_buffer), Dis_buffer);
exit1:
//wprintf("\nDtop=%04x Daddr=%04x Dmode=%u", Dis_top, Dis_addr, Dis_mode);
}

/*
 * Calculate higher address
 */
void scroll_up(unsigned n)
{
	unsigned a, b, o, l, i, addr[sizeof(Dis_buffer)];

	o = a = i = 0;
	if(Dis_top > sizeof(Dis_buffer))
		a = Dis_top - sizeof(Dis_buffer);

	if(Dis_load != a)
		target_read_memory(0, Dis_load = a, sizeof(Dis_buffer), Dis_buffer);

	do {
		addr[i++] = b = a;
		l = disassemble(Dis_buffer+o, a, 0);
		if((a += l) < b) // Wrap around
			break;
		o += l; }
	while(a < Dis_top);
	Dis_top = (i >= n) ? addr[i-n] : 0;
}

/*
 * Display breakpoints
 */
display_breaks(char c)
{
	unsigned i;
	wputc(c);
	wputs("Brk: ");
	for(i=0; i < break_top; ++i)
		wprintf("%04x ", break_addr[i]);
	if(!break_top)
		wputs("None");
}

/*
 * Set a breakpoint
 */
set_breakpoint()
{
	unsigned i, a;
	unsigned char f;

	if(test_support(cpu_kc[KCMD_BREAK], "Breakpoint"))
		return;

	display_breaks('\n');

	if(break_top >= MAX_BREAK) {
		wputs("-Full!");
		return; }
	if(get_value("Set breakpoint at? ", &a))
		return;
	for(i=f=0; i < break_top; ++i) {
		if(break_addr[i] == a) {
			wputs("-is Set!");
			return; }
		if(((break_addr[i] - a) < cpu_bp_length)
		|| ((a - break_addr[i]) < cpu_bp_length))
			f = -1; }
	if(f) {
		wputs("-Conflict!");
		return; }
	break_addr[break_top++] = a;
	display_breaks('\r');
}

/*
 * Remove a breakpoint
 */
delete_breakpoint()
{
	unsigned i, a;

	display_breaks('\n');
	if(!break_top)
		return;
	if(get_value("Remove breakpoint at? ", &a))
		return;
	for(i=0; i < break_top; ++i) {
		if(break_addr[i] == a)
			goto found_brk; }
	wputs("-Not Found!");
	return;
found_brk:
	while(++i < break_top) {
		break_addr[i-1] = break_addr[i];
		memcpy(break_store[i-1], break_store[i], sizeof(break_store[0])); }
	--break_top;
	display_breaks('\r');
	wcleol();
}

/*
 * Read file into memory image
 */
char *lptr;	/* Global ptr to load arguments */
static unsigned get_byte() { return (gethex(*lptr++) << 4) + gethex(*lptr++); }
void load_file(unsigned type)
{
	unsigned a, b, chksum, addr, length, count, size, i, low, high, xpc;
	char buffer[80], databuf[256];
	FILE *fp;

	wprintf("\n%s: ", filename);
	if(!(fp = fopen(filename,"r"))) {
		wputs("Unable to access");
		return; }

	low = *(unsigned*)cpu_memory[type];
	high = *(unsigned*)(cpu_memory[type]+2);
	xpc = -1;

	for(size = 0; fgets(lptr = buffer, 80, fp);) {
		again: switch(*lptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*lptr == '9') goto quitload;
				if(*lptr++ != '1') continue;
				length = count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				addr = (a << 8) + b;
				if((addr < low) || ((addr+length) > high))
					goto quitrange;
				if(!size)
					xpc = addr;
				for(i=0; i < count; ++i)
					chksum += databuf[i] = get_byte();
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				target_write_memory(type, addr, count, databuf);
				addr += count;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = get_byte())) goto quitload;
				chksum = (a = get_byte()) + length;
				chksum += b = get_byte();
				addr = (a << 8) + b;
				if((addr < low) || ((addr+length) > high))
					goto quitrange;
				chksum += get_byte();
				if(!size)
					xpc = addr;
				for(i=0; i < count; ++i)
					chksum += databuf[i] = get_byte();
				if((255 & -chksum) != get_byte())
					goto quitbad;
				target_write_memory(type, addr, count, databuf);
				addr += count;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				wputs("Invalid record format");
				goto quit; }
		size += length; }

quitload:
	wprintf("%u bytes loaded", size);
	goto quit;
quitrange:
	wprintf("Out of range!");
	goto quit;
quitbad:
	wputs("Bad checksum!");
quit:
	fclose(fp);
	if((xpc != -1) && !type) {
		*(unsigned*)reg_datap[0] = xpc;
		display_address(xpc);
		update_status(0); }
}

/*
 * Execute a sub-shell
 */
void doshell()
{
	char comspec[65];

	if(!getenv("COMSPEC", comspec)) {
		wputs("Cannot locate COMSPEC environment variable");
		return 0; }

	wopen(0, 0, 80, 25, WSAVE|WCOPEN|0x07);
	wputs("Type 'EXIT' to return to MicroScope");
	wcursor_line();
	wupdatexy();
	exec(comspec, "");
	wclose();
}

/*
 * Install breakpoints in target system
 */
int insert_breakpoints()
{
	unsigned i;

	for(i=0; i < break_top; ++i) {
		if(target_set_break(break_addr[i], break_store[i])) {
			while(i--)	// Remove any installed so far!
				target_write_memory(0, break_addr[i], cpu_bp_length, break_store[i]);
			return -1; } }

	return 0;
}

/*
 * Remove breakpoints from target system
 */
void remove_breakpoints()
{
	unsigned i;

	for(i=0;i < break_top; ++i)
		target_write_memory(0, break_addr[i], cpu_bp_length, break_store[i]);
}

// 0=Step "into" 1=Step "over"
int ss_one_inst(unsigned char mode)
{
	unsigned a1, r, e, e2;
	unsigned char savea1[4], savea2[4], s2, sf;

	Step_pc = *(unsigned*)reg_datap[0];

	// If not already loaded, load disassembly buffer
	if((Step_pc < Dis_load) || (Step_pc > (Dis_load+(sizeof(Dis_buffer)-8))))
		target_read_memory(0, Dis_load = Step_pc, sizeof(Dis_buffer), Dis_buffer);

	// Calculate following address
	a1 = Step_pc +
	(Type = disassemble(Dis_buffer + (Step_pc - Dis_load), Step_pc, 1));

	// See if target step support is available
	if(sf = cpu_kc[KCMD_STEP]) {		// Fast step available?
		if(!mode) {								// Step "into"?
	step_target:
			return target_launch(0); } }

	// Get lengths for range check
	e = Step_pc + cpu_bp_length;
	e2 = cpu_jmp_address + cpu_bp_length;

	s2 = 0;
	switch(cpu_id_flag) {	// Handle special instruction
	case 0x83 :				// Conditional - CALL
		if(mode)				// Stepping over - ignore condition
			goto step_call;
	case 0x82 :				// Conditional - Dual possible targets
		if((e2 > Step_pc) && (cpu_jmp_address < a1))
			goto bad_align;
		if(target_set_break(cpu_jmp_address, savea2))	// Alt address
			return -1;
		s2 = -1;
		break;
	case 0x81 :				// Unconditional - CALL
		if(mode) {				// Stepping over
		step_call:
			if(target_set_break(a1, savea1))
				return -1;
			r = target_launch(3);
			goto remove_breaks; }
	case 0x80 :				// Unconditional transfer
		if(sf)					// Can do fast?
			goto step_target;
		if((e2 > Step_pc) && (cpu_jmp_address < a1))
			goto bad_align;
		a1 = cpu_jmp_address;
		break;
	default:				// All other instructions
		if(sf)					// Can do fast?
			goto step_target; }

	if(target_set_break(a1, savea1))	// Install primary breakpoint
		return -1;
	r = target_launch(2);				// Run with short timeout

	// Remove the breakpoints
remove_breaks:
	target_write_memory(0, a1, cpu_bp_length, savea1);
	if(s2)
		target_write_memory(0, cpu_jmp_address, cpu_bp_length, savea2);

	return r;
bad_align:
	wputs("Can't step self-reference!");
	return -1;
}
/*
 * Single step
 */
void single_step(unsigned char flag)
{
	unsigned f, *pc, sm, sx, m;
	unsigned char c;
	static unsigned mode,
	static char *step_type_text[] = {
	"1-Into instruction",
	"2-Into this file",
	"3-Into any  file",
	"4-Into HL   file",
	"5-Over instruction",
	"6-Over this file",
	"7-Over any  file",
	"8-Over HL   file",
	0 };

	// If we don't do breakpoints, truncate the command list */
	if(!cpu_kc[KCMD_BREAK]) {
		if(test_support(cpu_kc[KCMD_STEP], "Stepping"))
			return;
		step_type_text[4] = 0; }

	// Prompt for step type/number
	m = 1;
	if(flag) {
		if(wmenu(20, 5, WSAVE|WCOPEN|WBOX1|colors[CO_PROMPT], step_type_text, &mode))
			return;
		if(get_value("How many? ", &m))
			return; }

	wprintf("\n%04x %-5u: Step %s ... ", *(pc = (unsigned*)reg_datap[0]),
		m, step_type_text[mode]+2);

	switch(mode&3) {
	case 1 :	// Current file
		if(!(f = peek(file_seg, *pc))) {
			wputs("Current address is not in a file");
			return; }
	case 2 :	// Any file
	case 3 :	// HL file
		if(!file_top) {
			wputs("No files loaded");
			return; } }

	// Select stepping mode
	sm = (mode > 3) ? -1 : 0;

	sx = dwin->WINcurx;
	c = 0;
	// And actually perform the step
	switch(mode&3) {
	case 0 :		// Step one instruction
		do {
			if(ss_one_inst(sm))
				goto fail;
			if(!(++c & 15)) {
				if(wtstc() == 0x1B)
					goto stop;
				wprintf("\r%04x %-5u", *pc, m);
				dwin->WINcurx = sx; } }
		while(--m);
		break;
	case 1 :		// Step to address in same file
		do {
			do {
				if(ss_one_inst(sm))
					goto fail;
				if(!(++c & 15)) {
					if(wtstc() == 0x1B)
						goto stop;
					wprintf("\r%04x %-5u", *pc, m);
					dwin->WINcurx = sx; } }
			while(peek(file_seg, *pc) != f); }
		while(--m);
		break;
	case 2 :		// Step to address in any file
		do {
			do {
				if(ss_one_inst(sm))
					goto fail;
				if(!(++c & 15)) {
					if(wtstc() == 0x1B)
						goto stop;
					wprintf("\r%04x %-5u", *pc, m);
					dwin->WINcurx = sx; } }
			while(!peek(file_seg, *pc)); }
		while(--m);
		break;
	case 3 :		// Step to address in any HL file
		do {
			do {
				if(ss_one_inst(sm))
					goto fail;
				if(!(++c & 15)) {
					if(wtstc() == 0x1B)
						goto stop;
					wprintf("\r%04x %-5u", *pc, m);
					dwin->WINcurx = sx; } }
			while(peek(file_seg, *pc) <= 1); }
		while(--m); }

	wprintf("\r%04x %-5u", *pc, m);
	return;

stop: wputs("Stopped!");
fail:
}

/*
 * Process command line options
 */
void process_option(char *ptr)
{
	unsigned i;
	unsigned char x0, x1, x2, *p, buffer[80];
	FILE *fp;
	static char vchars[] = { "SREPDWMACBHLX" };

	switch( ((x0 = toupper(ptr[0])) << 8) | (x1 = toupper(ptr[1])) ) {
	case '?' << 8 :
	case '-?' :
	case '/?' :
	case '-H' :
	case '/H' :
		help(HELP_CMD);
		abort(cmd_help); }

	x2 = toupper(ptr[2]);
	p = ptr+3;

	if(*ptr == '-') {
		switch((x1 << 8) | x2) {
		case 'CS' : case_mode = 0;				return;
		case 'CI' : case_mode =-1;				return;
		case 'RH' : number_base = 3;			return;
		case 'RD' : number_base = 2;			return;
		case 'RO' : number_base = 1;			return;
		case 'RB' :	number_base = 0;			return; }
		if(!cpu_name) {
			cpu_name = &pool[pool_top];
			++ptr;
		copy_name:
			while(*ptr && !isspace(*ptr))
				pool[pool_top++] = toupper(*ptr++);
			pool[pool_top++] = 0;
			return; } }
	else if(ptr[2] == '=') switch((x0 << 8) | x1) {
		case 'CP': com_port = atoi(p);
			if((com_port < 1) || (com_port > 4))
				abort("Com port must be 1-4");
			return;
		case 'CT': Timeout = atoi(p) / 55;
			if((Timeout < 9) || (Timeout > 182))
				abort("Timeout must be 500-10000");
			return;
		case 'CS': com_speed = scale(57600, 2, atoi(p));	return;
		case 'TW' : Dis_tab = atoi(p);						return;
		default: if(x0 == 'V') {
				for(i=0; x0 = vchars[i]; ++i)
					if(x0 == x1) {
						colors[i] = atox(p);
						return; } } }
	else {
		if(!cpu_name) {
			if(!(fp = fopen(ptr, "rv")))
				goto xfail;
			while(fgets(parse_ptr = buffer, sizeof(buffer), fp)) {
				switch(skip_blanks()) {
				case ';' :
				case 0 :	continue; }
				process_option(parse_ptr); }
			fclose(fp);
			return; }
		if(!*filename) {
			concat(filename, ptr);
			return; }
		if(!map_name) {
			map_name = &pool[pool_top];
			goto copy_name; } }
	printf("Unknown option: %s\n", ptr);
xfail:
	abort(cmd_help);
}

/*
 * Select a name from a long list
 */
unsigned select_list(char *names[], int top)
{
	int p, y, z;
	unsigned i, x;
	unsigned char buffer[65], *xnames[MAX_FILES], xindex[MAX_FILES];
	static int l;

	y = (top > 20) ? 20 : top;
	wopen(5, 11-(y/2), 67, y+2, WSAVE|WCOPEN|WBOX1|0x70);
	wcursor_off();

	for(i=0; i < top; ++i) {
		xindex[i] = i;
		xnames[i] = names[i]; }
	for(i=0; i < top; ++i) {
		for(x=i+1; x < top; ++x) {
			if(strcmp(xnames[x], xnames[i]) < 0) {
				z = xindex[i];
				xindex[i] = xindex[x];
				xindex[x] = z;
				z = xnames[i];
				xnames[i] = xnames[x];
				xnames[x] = z; } } }

	p = x = 0;
	for(;;) {
		if(p > (top-y)) {
			p = top-y;
			l = top-1; }
		if(p < 0)
			l = p = 0;
		for(i=0; i < y; ++i) {
			wgotoxy(0, i);
			if((p+i) == l)
				*W_OPEN = 0x07;
			wputs(xnames[p+i]);
			wcleol();
			*W_OPEN = 0x70; }
		z = l - p;
		i = wgetc();
		// If ASCII characgter, perform manual search
		if((i > ' ') && (i < 0x7F) && (x < (sizeof(buffer)-1))) {
			buffer[x++] = toupper(i);
			buffer[x] = 0;
			for(i=0; i < top; ++i) {
				if(strbeg(xnames[i], buffer))
					goto found; }
			beep(600, 100);
			--x;
			continue;
	found:	l = i;
			if((p > l) || ((p+y) <= l))
				p = l;
			if(p > (top-y))
				p = top-y;
			continue; }

		x = 0;
		switch(i) {
		case _KUA :
			if(l) {
				--l;
				if(!z)
					--p; }
			continue;
		case _KDA :
			if(l < (top-1)) {
				++l;
				if(z >= (y-1))
					++p; }
			continue;
		case _KPU :
			if(!z)
				p -= y;
			l = p;
			continue;
		case _KPD :
			if(z >= (y-1))
				p += y;
			l = (y-1) + p;
			continue;
		case _KHO :
			p = -1;
			continue;
		case _KEN :
			p = 32767;
			continue;
		case '\n' :
			wclose();
			return xindex[l]+1;
		case 0x1B :
			wclose();
			return 0; } }
}

/*
 * Search the current file for a string
 */
do_search(unsigned flag)
{
	int c;
	unsigned l, h, t, t1, s;
	unsigned char buffer[64], *p;
	static unsigned char search_string[64];

	if(!Dis_mode) {
		wputs("\nNo file active");
		return; }

	if(flag) {
		if(get_string("Search? ", search_string, sizeof(search_string)-1))
			return; }

	if(!*search_string)
		return;

	if(seek_line())
		return;

	wprintf("\nSearching '%s': ", search_string);
	s = dwin->WINcurx;

	if(!flag) {
		while((c = getc(open_fp)) != '\n') {
			if(c == -1)
				goto notfound; }
		++Dis_top; }
	wprintf("%u ", (l = Dis_top)+1);

restart:
	h = t = 0;
next_c:
	if((c = getc(open_fp)) != -1) {
		if(c == '\n') {
			dwin->WINcurx = s;
			wprintf("%u ", (++l)+1);
			goto restart; }
		buffer[h] = c;
		h = (h + 1) & (sizeof(buffer)-1);
rescan:
		t1 = t;
		p = search_string;
		while(*p) {
			if(t1 == h)
				goto next_c;
			if(buffer[t1] != *p++) {
				t = (t+1) & (sizeof(buffer)-1);
				goto rescan; }
			t1 = (t1 + 1) & (sizeof(buffer)-1); }
		Dis_top = l;
		wputs("Found!");
		return; }
	dwin->WINcurx = s;
notfound:
	wputs("Not found!");
}

main(int argc, char *argv[])
{
	unsigned i, hl, ws, a;

#ifdef MEMCHK
	// Track memory usage
	static unsigned memlow;
	free(parse_ptr = memlow = malloc(1));
	parse_ptr += 10;
	hl = (&i) - 100;
	while(++parse_ptr < hl)
		*parse_ptr = 0xA5;
#endif

	// Truncate program name to home directory
	parse_ptr = argv[a = 0];
	while(*parse_ptr) {
		switch(*parse_ptr++) {
		case '\\' :
			a = parse_ptr; } }
	*(char*)a = 0;

	// Handle command line arguments
	for(i=1; i < argc; ++i)
		process_option(argv[i]);

	// Allocate segments & zero file/symbol table
	if(!(sym_seg = alloc_seg(4096*4)))
		abort("Out of memory");
	file_seg = sym_seg + 4096;
	i = 0;
	do {
		pokew(sym_seg, i, 0);
		pokew(file_seg, i, 0); }
	while(i += 2);

	IOB_size = 1024;

	if(!cpu_name) {
		printf("Target/CPU name required\n");
		abort(cmd_help); }

#ifdef DEMO
	if(strcmp(cpu_name, ""#DEMO""))
		abort("This versiom of DMS works only with target: -"#DEMO"");
#endif

	concat(Dis_buffer, argv[0], cpu_name, ".DMS");
	if(i = load_cpu(Dis_buffer))
		abort(i);

#ifdef DEMOID
	if(cpu_jmp_address != DEMOID)
		abort("This versiom of DMS works only with target: -"#DEMO"");
#endif

	// Allocate space for register data & store index addresses
	ws = &pool[pool_top];
	for(i=0; i < cpu_num_reg; ++i)
		pool_top += cpu_reg_size[i] & 0x07;
	reg_datap = &pool[pool_top];
	for(i=0; i < cpu_num_reg; ++i) {		// Generate indexes
		pool[pool_top++] = ws;
		pool[pool_top++] = ws >> 8;
		ws += cpu_reg_size[i] & 0x07; }

	// Load symbolic definitions
	if(map_name)
		load_symbolics(map_name);


	// Open the display windows
	rwin = wopen( 0,  0, 80,  1, WSAVE|WCOPEN|colors[CO_REG]);
	open_windows("Dunfield MicroScope - "#VERSION"   ");

//upd_r:
	if(target_init(/*-1*/)) {
quit_udb:
		wputs("\n\nPress ENTER...");
		while(wgetc() != '\n');
		close_windows();
		wclose();
		return; }
	if(kernel_id != cpu_jmp_address) {
		wprintf("\n\nTarget %04x required", cpu_jmp_address);
		goto quit_udb; }

	wputs("   - Press F1 for help");

	if(*filename)
		load_file(0);

upd_r:
	if(*cpu_default_regs) {
		memcpy(reg_datap[0], cpu_default_regs+1, *cpu_default_regs);
		if(cpu_flags & CPU_RVECTOR) {
			target_read_memory(0, *(unsigned*)reg_datap[0], sizeof(i), &i);
			*(unsigned*)reg_datap[0] = adjust_value(i); }
		display_address(*(unsigned*)reg_datap[0]); }
	update_status(Mode);
	target_update();
upd_d:
	update_display();

	// Main command loop
	for(;;) {
		if(cpu_flags & CPU_WORDALN) {
			if(*reg_datap[0] & 1) {
				*reg_datap[0] &= -2;
				wputs("\nPC misalignment corrected");
				goto upd_r; } }
		hl = index_l[index_top];
		ws = swin->WINheight;
		switch(i = toupper(wgetc())) {
		case _KLA :		// Scroll left
			if(Dis_offset)
				--Dis_offset;
			goto upd_d;
		case _KRA :		// Scroll right
			++Dis_offset;
			goto upd_d;
		case _KHO :		// Home to margin
			Dis_offset = 0;
			goto upd_d;
		case _KUA :		// Scroll up
			if(!Dis_mode)
				scroll_up(1);
			else {
				--Dis_top;
				if(Dis_top == 0xFFFF)
					Dis_top = 0; }
			goto upd_d;
		case _KDA :		// Scroll down
			if(!Dis_mode) {
				Dis_top += disassemble(Dis_buffer, Dis_top, 0);
				goto upd_d; }
			if(hl <= ws)
				goto goend;
			if(++Dis_top >= (hl-ws))
				goto goend;
			goto upd_d;
		case _KPU :		// Page up
			if(!Dis_mode) {
				scroll_up(ws-1);
				goto upd_d; }
			else if(Dis_top > ws) {
				Dis_top -= ws;
				goto upd_d; }
		case _CPU :		// Home display
			Dis_top = 0;
			goto upd_d;
		case _KPD :		// Page down
			if(!Dis_mode) {
				Dis_top = Dis_last;
				goto upd_d; }
			if(hl <= ws)
				goto goend;
			if((Dis_top += ws) < (hl-ws))
				goto upd_d;
		case _CPD :		// End of display
		goend:
			if(Dis_mode)
				Dis_top = (hl > ws) ? hl-ws : 0;
			goto upd_d;

		case 'B' :		// Set breakpoint
			set_breakpoint();
			goto upd_d;
			continue;
		case 'C' :		// Calculator
			if(!get_value("Expression? ", &a)) {
				wprintf("\nResult: $%04x %u ", a, a);
				if(a & 0x8000)
					wprintf("(%d) ", a);
				wprintf("@%o %c%b", a, 37, a); }
			continue;
		case 'D' :		// Disassemble/display
			if(!get_value("Location? ", &a)) {
				display_address(a);
				goto upd_d; }
			continue;
		case 'E' :		// Erase (remove) a watchpoint
			delete_watch();
			goto upd_w;
		case 'F' :
			if(file_top) {
				if(i = select_list(files, file_top)) {
					Dis_top = file_lines[(Dis_mode = i)-1];
					goto upd_d; }
				continue; }
			wputs("\nNo files loaded");
			continue;
		case '?' :		// Search
			do_search(-1);
			goto upd_d;
		case '/' :		// Search again
			do_search(0);
			goto upd_d;
		case '\t' :		// Set tab size
			i = Dis_mode ? Dis_tabs[hl=Dis_mode-1] : Dis_tab;
			wprintf("\n%s tab size:%u", Dis_mode ? files[hl] : "Default", i);
			if(!get_value("Tab size? ", &i)) {
				if(!i)
					continue;
				if(Dis_mode)
					Dis_tabs[hl] = i;
				else
					Dis_tab = i;
				wprintf(" now:%u", i); }
			goto upd_d;
		case 'G' :		// Execute
			if(test_support(cpu_kc[KCMD_EXEC], "Execute"))
				continue;
		test_bp:	// Step till clear of breakpoints
			ws = (Step_pc = *(unsigned*)reg_datap[0]) + cpu_bp_length;
			for(i=0; i < break_top; ++i) {
				hl = (a = break_addr[i]) + cpu_bp_length;
				if((ws > a) && (Step_pc < hl)) {
					wprintf("\nStepping over breakpoint at %04x ... ", a);
					if(ss_one_inst(0))
						goto upd_step;
					goto test_bp; } }
			if(insert_breakpoints())
				continue;
			target_launch(1);
			remove_breakpoints();
			update_status(0);
			display_address(*(unsigned*)reg_datap[0]);
			goto upd_wd;
		case 'L' :		// Load a file
			if((hl = get_memory_type()) & 0xFF00)
				continue;
			if(test_support(cpu_kc[KCMD_WRITE1+hl], "Loading this memory"))
				continue;
			if(!get_string("Load file? ", filename, 49)) {
				load_file(hl);
				Dis_load = -1;
				goto upd_wd; }
			continue;
		case 'M' :		// Edit memory
			edit_memory();
		upd_wd:
			update_watch();
			goto upd_d;
		case 'N' :		// Go to line number
			if(!Dis_mode) {
				wputs("\nNo file active");
				continue; }
			if(get_value("Line?", &i))
				continue;
			Dis_top = i-1;
			goto upd_d;
		case 'O' :		// Open new file
			*(parse_ptr = &pool[pool_top]) = 0;
			if(get_string("Filename? ", parse_ptr, 64))
				continue;
			files[file_top] = parse_ptr;
			if(index_file(file_top)) {
				while(pool[pool_top++]);
				Dis_mode = ++file_top;
				Dis_top = 0;
				goto upd_d; }
			continue;
		case 'R' :		// Edit registers
			edit_registers();
			target_update();
			goto upd_d;
		case 'S' : single_step(-1); goto do_ss;
		case ' ' : single_step(0); do_ss:
		upd_step:
			update_status(0);
			display_address(*(unsigned*)reg_datap[0]);
			goto upd_wd;
		case 'K' :		// Remove breakpoint
			delete_breakpoint();
			goto upd_d;
		case 'T' :		// TTY with target
			target_tty();
			continue;
		case _K1 : help(HELP_MAIN); continue;
		case _K2 :
			wputs("\nReset");
//			target_reset();
			if(target_init(/*0*/)) goto quit_udb;
			goto upd_r;
		case _K3 :		// Set title line
			Mode ^= MODE_EDIT;
			w_clwin(rwin);
			update_status(Mode);
			continue;
		case _K4 :		// Source debug mode
			wprintf("\nSource display: %s", ((Mode ^= MODE_DIS) & MODE_DIS)
				?	"OFF" : "ON");
			display_address(Dis_addr);
			goto upd_d;
		case 'W' :		// Set a watchpoint
			add_watch();
		case _K5 :		// Update the watchpoints
		upd_w:
			update_watch();
			continue;
		case _K6 :		// Set number base
			wmenu(20, 7, WSAVE|WCOPEN|WBOX1|colors[CO_PROMPT],
				number_base_table, &number_base);
			continue;
#ifdef MEMCHK
		case _ALT+'M' :	// Debugging aid - to be removed in final version
			free(i=malloc(1));
			wprintf("\nMemory: %04x %04x/%u", memlow, i, hl = (&i)-i);
			parse_ptr = ws = i + (hl/2);
			while(*parse_ptr-- == 0xA5);
			wprintf(" %04x-", hl = parse_ptr+1);
			parse_ptr = ws;
			while(*parse_ptr++ == 0xA5);
			wprintf("%04x/%u", parse_ptr-1, parse_ptr - hl);
			continue;
#endif
		case _K9 :		// DOS shell
			doshell();
			continue;
		case _K10 :		// Terminate
			wopen(28, 10, 25, 3, WSAVE|WCOPEN|WBOX2|colors[CO_WARN]);
			wputs("Exit MicroScope (Y/N)?");
			loop: switch(wgetc()) {
				default: goto loop;
				case 'y' :
				case 'Y' :
				case '\n' :
					wclose();
					target_close();
					close_windows();
					wclose();
					return;
				case 'n' :
				case 'N' :
				case 0x1B : }
			wclose();
			continue; }
		beep(600, 100); }
}
