/*
 * Target interface for DDS Universal Debugger
 */
#include <stdio.h>
#include <window.h>
#include <comm.h>
#include "dmscpu.h"
#define	NOHELP
#include "help.c"

#define	Debug(x)			// wprintf x;

#define	CO_DEBUG	6		// Color index for debug screen
#define	CO_WARN		7		// Color index for warning

// Constants used in marking target comms waiting
#define	BIOS_TICK	peekw(0x40,0x6C)
#define	TIME_MARK	10		// Timeout for marker (must be > TIME_CMD/STEP)
// B15=!Allow Stop B14=!Allow interactive
#define	TIME_CMD	0xC009	// Timeout for standard commands
#define	TIME_STEP	0xC009	// Timeout for STEP (allow Stop)
#define	TIME_LSTEP	0x6A30	// Allow step for 10 minutes (allow Stop)
#define	TIME_RUN	0x4000	// Timeout for RUN (allow immed)
#define	TIME_FLUSH	18		// Time for buffer flush

#define	VIDEO_ADDR	159		// Address for marker block
#define	VIDEO_MARK	0x4E	// Attribute to display

extern unsigned char
	cpu_bp_length,			// Cpu breakpoint length
	*cpu_cmd_prefix,		// Command prefix
	*cpu_break_list,		// Break list
	*cpu_launch_list,		// Launch list
	*cpu_update_list,		// Update list
	*cpu_reg_size,			// Sizes of registers
	*cpu_name,				// Name of CPU
	cpu_id_flag,			// Special instruction flags
	cpu_kc[],				// Kernel commands
	cpu_kernel_mem[],		// Kernel memory block
	**reg_datap,			// Register data pointers
	Type,					// Used to pass inst length
	colors[],				// Color values
	*ARGV[],				// Argument list
	pool[];					// Memory pool
extern unsigned
	irq = 0x60,				// Irq for target interface
	pool_top,				// Memory pool allocation boundary
	cpu_flags,				// CPU flags
	cpu_jmp_address,		// for kernel ID
	cpu_mh_mask,			// Memory hardware access
	cpu_num_reg;			// Number of registers
extern struct WINDOW
	*rwin,					// Status window
	*dwin;					// Debug window

unsigned
	kernel_id,				// Kernel ID from query
	com_addr,				// Com port address
	com_speed = _19200,		// Communications baud rate
	Timeout = 18;			// Comm timeout
unsigned char
	com_port = 1;			// Comm port
static unsigned
	ma,						// Address of memory block
	me,						// End of memory block
	ms;						// Size of memory block
static unsigned char
	kernel_version,			// Kernel version from query
	zoom,					// Window zoomed
	com_mask,				// Out mask
	mt = -1,				// Memory block type
	*mp;					// Memory pointer

//extern Csignals();
asm "	EXTRN _Csignals:NEAR";

/* Function to control DTR/RTS lines to target */
static void drop_rts(void)
{
	out(com_addr, (cpu_flags & CPU_INVRTS) ? com_mask |= (SET_RTS|OUTPUT_2) : com_mask &= ~SET_RTS);
}
static void raise_rts(void)
{
	out(com_addr, (cpu_flags & CPU_INVRTS) ? com_mask &= ~SET_RTS : com_mask |= (SET_RTS|OUTPUT_2));
}
static void set_dtr(unsigned char state)
{
	static unsigned char last_state;
	// If state change - wait a bit to insure last command finished
	if(state != last_state) {
		asm {
			XOR		DX,DX				; Get zero
			SUB		DX,DGRP:_com_speed	; Compute speed factor
			SHR		DX,1				; /2
			SHR		DX,1				; /4
			SHR		DX,1				; /8
			SHR		DX,1				; /16
			XOR		CX,CX				; Zero high
			MOV		AH,86h				; BIOS delay function
			INT		15h					; Call BIOS
		} }
	if(last_state = state)
		out(com_addr, (cpu_flags & CPU_INVDTR) ? com_mask &= ~SET_DTR : com_mask |= (SET_DTR|OUTPUT_2));
	else
		out(com_addr, (cpu_flags & CPU_INVDTR) ? com_mask |= (SET_DTR|OUTPUT_2) : com_mask &= ~SET_DTR);
}
static int can_abort()
{
	return (cpu_flags & CPU_DTRSTOP) | cpu_kc[KCMD_ABORT];
}
static void zoom_off(char co)
{
	if(zoom) {
		zoom = 0;
		wclose(); }
	if(co)
		wcursor_off();
}

/*
 * TTY session with target system
 */
void target_tty(void)
{
	int c;
	char flag;
	static char eflag;

	wputs("\nTTY: F8=Zoom F9=Echo F10=Exit ");

zoom_up:
	wopen(0, 1, 80, 24, WSAVE|WCOPEN|WSCROLL|WWRAP|colors[CO_DEBUG]);
	flag = zoom = -1;
set_cursor:
	eflag ? wcursor_block() : wcursor_line();
	for(;;) {
		if((c = Ctestc()) != -1) {
			wputc(c);
			flag = -1; }
		switch(c = wtstc()) {
		case _K1 :		// Help
			help(HELP_TTY);
			continue;
		case _K8 :		// Zoom window
			if(!zoom)
				goto zoom_up;
			zoom_off(0);
			flag = -1;
			goto set_cursor;
		case _K9 :		// Echo mode
			eflag = eflag ? 0 : -1;
			goto set_cursor;
		case _K10 :		// Close session
			zoom_off(-1);
			wputs("Done!");
			return;
		case 0 :		// No character
			if(flag) {
				wupdatexy();
				flag = 0; }
			continue;
		case _KDL :		// Delete
			c = 0x7F;
			goto dochar;
		case _KBS :		// Backspace
			c = '\b';
		default:
		dochar:
			if(!(c & 0xFF00)) {
				Cputc((c == '\n') ? '\r' : c);
				if(eflag) {
					wputc(c);
					flag = -1; } } } }

}

target_flush()
{
	unsigned ct, ts, t;
	// First flush out the incoming stream
	ct = 0;
	ts = BIOS_TICK;
	do {
		t = BIOS_TICK;
		if(Ctestc() != -1) {
			if(++ct > 500)
				return -1;
			ts = t; } }
	while((t - ts) <= TIME_FLUSH);
	return 0;
}

/*
 * Reset the target system
 */
void target_reset(void)
{
	unsigned i, to, oc, cs;
	unsigned char c, e, buffer[65];
	FILE *fp;

	wopen(30, 10, 20, 3, WSAVE|WCOPEN|WBOX1|REVERSE);
	wcursor_off();
	wputs("Resetting...");

	drop_rts();
	delay(500);
	raise_rts();

	// Determine home directory & append filename
	concat(buffer, *ARGV, cpu_name, ".DML");
	if(fp = fopen(buffer, "rb")) {
		wclwin(); wputs("Loading kernel");
		oc = e = 0;
		to = Timeout;
		cs = com_speed;
	next:
		while((i = getc(fp)) != -1) {
			if(oc) {
				Cputc(c=i);
				--oc;
				if(e) goto echo;
				continue; }
			switch(i) {
			case 0x00 :	goto end;	// End of file
			case 0xFB :				// Flush file
				c = getc(fp);
			restart:
				i = peekw(0x40, 0x6C);
				do {
					if(Ctestc() != -1)
						goto restart; }
				while((peekw(0x40, 0x6C) - i) <= c);
				continue;
			case 0xFC :				// Wait for character
				c = getc(fp);
			echo:
				i = peekw(0x40, 0x6C);
				do {
					if(Ctestc() == c)
						goto next; }
				while((peekw(0x40,0x6C) - i) <= to);
				wclwin(); wputs("No response!");
				delay(1000);
				goto end;
			case 0xFD : e = 0;	continue;			// Disable expect-echo
			case 0xFE :	e = -1;	continue;			// Enable  expect-echo
			case 0xFA :				// Set baudrate
				if(!(com_speed = getc(fp)))
					com_speed = cs;
				out(com_addr-1, (c = in(com_addr-1)) | 0x80);
				out(com_addr-4, com_speed);
				out(com_addr-3, com_speed >> 8);
				out(com_addr-1, c);
				continue;
			case 0xFF : to = getc(fp); continue; }	// Set timeout
			oc = i; }
	end:
		fclose(fp); }

//	wclwin(); wputs("Clear COM...");
//	target_flush();
	wclose();
}

/*
 * Write a byte to the target system
 */
static void target_write_byte(unsigned char b)
{
#ifdef FAKE
	if(dwin->WINcurx > 75)
		w_putc('\n', dwin);
	w_printf(dwin, "%02x-", b);
#else
		Cputc(b);
#endif
}

/*
 * Send prefix bytes to system
 */
void send_command(unsigned cmd, unsigned dl)
{
	unsigned size;
	unsigned char *s;

	while(Ctestc() != -1);	// Flush input

	size = *(s = cpu_cmd_prefix);
	while(size--) {
		target_write_byte(*++s);
		if(dl)
			delay(dl); }
	target_write_byte(cmd);
}

/*
 * Send an address to the system
 */
static send_address(unsigned a)
{
	if(cpu_flags & CPU_SADDRL) {
		target_write_byte(a);
		target_write_byte(a >> 8); }
	else {
		target_write_byte(a >> 8);
		target_write_byte(a); }
}

/*
 * Synchronize to the target system
 */
int target_sync()
{
	int c;
	unsigned ts, t, ct;
	unsigned char response[5], r1, r2;
	r1 = cpu_kc[KCMD_RESP1];
	r2 = cpu_kc[KCMD_RESP2];

retry:
	if(target_flush())
		goto tfail;

	// Send query command & wait for response
	send_command(cpu_kc[KCMD_QUERY], 50);
	ct = 0;
	ts = BIOS_TICK;
	do {
		t = BIOS_TICK;
		if((c = Ctestc()) != -1) {
			if(ct > sizeof(response))
				goto tfail;
			response[ct++] = c;
			ts = t; } }
	while((t - ts) <= Timeout);

	if((response[0] != r1) || (response[1] != r2))
		goto tfail;
	kernel_id = (response[2] << 8) | response[3];
	kernel_version = response[4];
	return -1;

tfail:
	wopen(10, 10, 60, 3, WSAVE|WCOPEN|WBOX2|colors[CO_WARN]);
	wputs("Unable to locate target... (Abort/Retry/Target-reset)?");
	for(;;) switch(wgetc()) {
		case 't' :
		case 'T' :
			target_reset();
		case 'r' :
		case 'R' :
			wclose();
			goto retry;
		case 'a' :
		case 'A' :
			wclose();
			return 0; }
}

/*
 * Read a byte from the target system
 */
static int target_read_byte(unsigned time)
{
	int c, k;
	unsigned v, ct, ts1, ts2, tf, tv, sc, timeout;

	timeout = time & 0x3FFF;

retry:
	if((c = Ctestc()) != -1)			// Fast return if chars ready
		return c;

	ts1 = ts2 = BIOS_TICK;
	v = ct = tf = sc = k = 0;
	for(;;) {
		if((c = Ctestc()) != -1) {		// Test for character from target
			if(sc)
				poke(W_BASE, VIDEO_ADDR, sc);
			return c; }
		if(!(time & 0xC000)) {			// Abort/Stop enabled
			switch(k = wtstc()) {		// Test for keyboard character
			case _KDL :	k = 0x7F;	break;
			case _KBS :	k = '\b';	break;
			case _K10 :					// Abort local
				zoom_off(0);
				goto target_fail;
			case _K8 :					// Zoom
				if(!timeout) {			// Full exec only
					if(sc)
						poke(W_BASE, VIDEO_ADDR, sc);
					return -3; }
				break;
			case _K9 :					// Stop target
				if(!(time & 0x8000)) {		// Stop enabled?
					if(can_abort()) {
						zoom_off(0);
						wputs("\nAttempting to stop target... ");
						wupdatexy(); }
					if(cpu_flags & CPU_DTRSTOP)
						set_dtr(0);
					if(c = cpu_kc[KCMD_ABORT])
						send_command(c, 500); } } }
		tv = BIOS_TICK;
		if((tv - ts1) > TIME_MARK) {	// Update marker on timeout
			ts1 = tv;
			if(sc) {
				poke(W_BASE, VIDEO_ADDR, sc);
				sc = 0; }
			else {
				time &= ~0x4000;			// Enable abort
				sc = peek(W_BASE, VIDEO_ADDR);
				poke(W_BASE, VIDEO_ADDR, VIDEO_MARK); } }
		if(!timeout) {					// Interaction allowed
			if(k && !(k & 0xFF00)) {
				target_write_byte((k == '\n') ? '\r' : k);
				k = 0; }
			continue; }
		if((tv - ts2) >= timeout)		// Abort on command timeout
			goto target_fail; }

target_fail:
	if(sc)
		poke(W_BASE, VIDEO_ADDR, sc);
	if(!timeout) {	// Exec - reset to abort
		target_reset();
		target_flush();
		return -1; }
	wopen(10, 10, 60, 3, WSAVE|WCOPEN|WBOX2|colors[CO_WARN]);
	wcursor_off();
	wputs("Target communication failure... (Abort/Retry/Wait)?");
	for(;;) switch(wgetc()) {
		case 'r' :
		case 'R' :
			wclose();
			return -2;
		case 'a' :
		case 'A' :
			wclose();
			return -1;
		case 'w' :
		case 'W' :
			wclose();
			goto retry; }
}

/*
 * Terminate the target system interface
 */
void target_close(void)
{
	Cclose();
}

/*
 * Initialize target system interface
 */
int target_init(/* int flag */)
{
	com_mask = SET_RTS|SET_DTR|OUTPUT_2;
	if(cpu_flags & CPU_INVRTS) com_mask &= ~SET_RTS;
	if(cpu_flags & CPU_INVDTR) com_mask &= ~SET_DTR;

	if(Copen(com_port, com_speed, PAR_NO|DATA_8|STOP_2, com_mask)) {
		wputs(" Cannot open COM port");
		return -1; }
	Cflags |= TRANSPARENT;

	// trick! - peek at MOV instruction in Csignals to get com address ptr
	asm {
		MOV	BX,CS:word ptr _Csignals+2	; Address function
		MOV	AX,[BX]					; Read variable
		DEC	AX						; Back to com port
		MOV	DGRP:_com_addr,AX		; Write the address
	}

//	if(flag)
		target_reset();

	// Attempt to locate target system
	wputs(" Target: ");

#ifdef FAKE
	kernel_id = cpu_jmp_address;
#else
	if(!target_sync()) {
		wputs("Not found");
		target_close();
		return -1; }
#endif

	wprintf("%04x v%u", kernel_id, kernel_version);

	// Obtain memory block information
	if(!mp) {
		if(ms = *cpu_kernel_mem) {
			mt = cpu_kernel_mem[1];
			ma = cpu_kernel_mem[3] << 8;
			ma |= cpu_kernel_mem[2];
			me = (ma + ms) - 1;
			mp = &pool[pool_top];
			pool_top += ms; } }

	return 0;
}

/*
 * Read memory from the target system
 */
target_read_memory(unsigned type, unsigned address, unsigned size, char *dest)
{
	unsigned s, end;
	unsigned char *d;

	if(!size)	// Handle zero-length reads
		return;

//w_printf(dwin, "\nRD: %04x %u", address, size);
#ifdef FAKE
	s = address >> 8;
	while(size--)
		*dest++ = s++;
	return;
#endif

	if((0x100 << type) & cpu_mh_mask)
		set_dtr(-1);
	else
		set_dtr(0);

	// Split wrapped accesses into two non-wrapped accesses
	if((end = (address+size)-1) < address) {
		target_read_memory(type, address, 0-address, dest);
		target_read_memory(type, 0, end+1, (0-address)+dest);
		return; }
	if((type == mt) && mp) {	// Possible block conflict
		if((address <= me) && (end >= ma)) {	// Overlap
			if((address < ma) && (end >= ma))	// Section below
				target_read_memory(type, address, ma-address, dest);
			if((address <= me) && (end > me))	// Section above
				target_read_memory(type, me+1, end-me, ((me - address) + dest)+1);
			if(address < ma) {	// Clip to start
				dest += (ma - address);
				address = ma; }
			if(end > me)		// Clip to end
				end = me;
			size = (end - address)+1;
			end = address - ma;
			while(size--)
				*dest++ = mp[end++];
			return; } }

	// Proceed with standard read
retry:
	send_command(cpu_kc[KCMD_READ1+type], 0);
	send_address(address);
	target_write_byte(size);
	s = size;
	d = dest;
	while(s--) {
		switch(end = target_read_byte(TIME_CMD|Timeout)) {
		case -2 : if(target_sync()) goto retry;
		case -1 : return; }
		*d++ = end; }
}

/*
 * Write memory to the target system
 */
target_write_memory(unsigned type, unsigned address, unsigned size, char *dest)
{
	unsigned end;

	if(!size)		// Handle zero-length writes
		return;

// w_printf(dwin, "\nWR: %04x %u", address, size);
	if((0x001 << type) & cpu_mh_mask)
		set_dtr(-1);
	else
		set_dtr(0);

	// Split wrapped accesses into two non-wrapped accesses
	if((end = (address+size)-1) < address) {
		target_write_memory(type, address, 0-address, dest);
		target_write_memory(type, 0, end+1, (0-address)+dest);
		return; }
	if((type == mt) && mp) {	// Possible block conflict
		if((address <= me) && (end >= ma)) {	// Overlap
			if((address < ma) && (end >= ma))	// Section below
				target_write_memory(type, address, ma-address, dest);
			if((address <= me) && (end > me))	// Section above
				target_write_memory(type, me+1, end-me, ((me - address) + dest)+1);
			if(address < ma) {	// Clip to start
				dest += (ma - address);
				address = ma; }
			if(end > me)		// Clip to end
				end = me;
			size = (end - address)+1;
			end = address - ma;
			while(size--)
				mp[end++] = *dest++;
			return; } }

	// Proceed with standard write
	send_command(cpu_kc[KCMD_WRITE1+type], 0);
	send_address(address);
	target_write_byte(size);
	while(size--)
		target_write_byte(*dest++);
}

/*
 * Set a breakpoint on the target
 */
target_set_break(unsigned address, unsigned char *save)
{
	unsigned i;
	unsigned char l, *s;

	if(cpu_mh_mask & 0x001)
		set_dtr(-1);
	else
		set_dtr(0);

	// Send the breakpoint command
retry:
	send_command(cpu_kc[KCMD_BREAK], 0);
	send_address(address);

	// Read the current memory content
	s = save;
	for(l=0; l < cpu_bp_length; ++l) {
		switch(i = target_read_byte(TIME_CMD|Timeout)) {
		case -2 : if(target_sync()) goto retry;
		case -1 : goto fail; }
		*s++ = i; }
	// And the response code
	switch(i = target_read_byte(TIME_CMD|Timeout)) {
	case -2 : if(target_sync()) goto retry;
	case -1 : goto fail; }
	if(!i)
		return 0;
fail:
	wprintf("\nBreak at %04x failed (%d)", address, i);
	return -1;
}

/*
 * Send a control list to the target
 */
void target_send_list(unsigned char *s, unsigned mode)
{
	unsigned size, j;
	unsigned char c, *p;

	size = *s++;

	while(size--) {
		if((c = *s++) & 0x80) {			// Constant value
			target_write_byte(c);
			continue; }
		switch(c) {						// Special cases
		case L_IL : j = Type;			goto sendstep;	// Inst length
		case L_IF : j = cpu_id_flag;	goto sendstep;	// Inst flags
		case L_JL : j = cpu_jmp_address;goto sendstep;	// Jump addr low
		case L_JH : j = cpu_jmp_address >> 8;			// Jump addr high
		sendstep: if(!mode) target_write_byte(j);
			continue;
		case L_KM :						// Kernel memory block
			for(j=0; j < ms; ++j)
				target_write_byte(mp[j]);
			continue; }
		if(c >= L_BIG) {				// Big endian
			j = cpu_reg_size[c -= L_BIG]&7;	// Register data
			p = reg_datap[c] + j;
			while(j--)
				target_write_byte(*--p); }
		else {							// Little endian
			j = cpu_reg_size[c] & 7;	// Register data
			p = reg_datap[c];
			while(j--)
				target_write_byte(*p++); } }
}

/*
 * Update the target registers
 */
void target_update()
{
	unsigned char c;
	if(c = cpu_kc[KCMD_UPDATE]) {
		send_command(c, 0);
		target_send_list(cpu_update_list, 0); }
}

/*
 * Launch the target into a particular mode
 *
 * 0=Step 1=Run 2=Run/No-prompt/Short-timeout 3=Run/No-prompt/Long-timeout
 */
target_launch(unsigned mode)
{
	unsigned i, j, size, to, h, t;
	unsigned char buffer[8], *s, *p, c, pca;

	Debug(("\nLaunch"))

	set_dtr(-1);

retry:
	to = TIME_STEP|Timeout;
	if(mode) {
		if(mode == 1) {
			wputs("\nRunning ... F8=Zoom");
			if(can_abort())
				wputs(" F9=Stop");
			wputs(" F10=Abort ... ");
			to = TIME_RUN;
			wcursor_line();
			wupdatexy(); }
		else if(mode == 3)
			to = TIME_LSTEP;
		send_command(cpu_kc[KCMD_EXEC], 0); }
	else
		send_command(cpu_kc[KCMD_STEP], 0);

	target_send_list(cpu_launch_list, mode);

	// Wait for target to return
	h = t = zoom = 0;
	size = *(s = cpu_break_list);
	Debug(("\nWait "))
wait_for_char:
	// If data received, insert into buffer
	switch(i = target_read_byte(to)) {
	case -3:
		if(zoom)
			zoom_off(0);
		else {
			zoom = -1;
			wopen(0, 1, 80, 24, WSAVE|WCOPEN|WSCROLL|WWRAP|colors[CO_DEBUG]);
			wupdatexy(); }
		goto wait_for_char;
	case -2: zoom_off(-1); if(target_sync()) goto retry;
	case -1: zoom_off(-1); goto abort_load; }
	buffer[h] = i;
	h = (h+1) & (sizeof(buffer)-1);
scan_for_sig:
	// See if this data matches
	s = cpu_break_list;
	for(i=0; *++s & 0x80; ++i) {
		if((j = (t+i) & (sizeof(buffer)-1)) == h)
			goto wait_for_char;
	Debug(("{%02x-%02x}", buffer[j], *s))
		if((c = buffer[j]) != *s) {
			wputc(buffer[t]);
			t = (t+1) & (sizeof(buffer)-1);
			wupdatexy();
			goto scan_for_sig; } }
	size -= i;

	zoom_off(-1);

	Debug(("\n[OK]"))
	// Record execution result code
	switch(i = target_read_byte(TIME_CMD|Timeout)) {
		case -2 : if(target_sync()) goto retry;
		case -1 : goto abort_load; }
	pca = i;
	// Read break data
	while(size--) {
		if((c = *s++) == L_KM) {			// Kernel Memory block
			for(j=0; j < ms; ++j) {
				Debug(("[MEM-%u]", j))
				switch(i = target_read_byte(TIME_CMD|Timeout)) {
				case -2: if(target_sync()) goto retry;
				case -1: goto abort_load; }
				mp[j] = i; }
			continue; }
		if(c >= L_BIG) {					// Big endian
			j = cpu_reg_size[c -= L_BIG]&7;	// Register data
			Debug(("[BReg-%u-%u]", c, j))
			p = reg_datap[c] + j;
			while(j--) {
				switch(i = target_read_byte(TIME_CMD|Timeout)) {
				case -2: if(target_sync()) goto retry;
				case -1: goto abort_load; }
				*--p = i; } }
		else {							// Little endian
			j = cpu_reg_size[c] & 7;	// Register data
			Debug(("[LReg-%u-%u]", c, j))
			p = reg_datap[c];
			while(j--) {
				switch(i = target_read_byte(TIME_CMD|Timeout)) {
				case -2: if(target_sync()) goto retry;
				case -1: goto abort_load; }
				*p++ = i; } } }
	if(mode == 1)
		wputs("Stopped!");

	*(unsigned*)reg_datap[0] -= pca;
	return 0;

abort_load:
//	wputs((mode == 1) ? "Aborted!" : "Failed!");
	wputs("Aborted!");
	return -1;
}
