/*
 * Univrsal Debugger - base CPU/Symbol-table management
 */
#include <stdio.h>
#include <window.h>
#include <file.h>
#include "dmscpu.h"

#ifdef MODE
	#if ((MODE < 1) || (MODE > 3))
		#error MODE parameter must be 1-3
	#endif
#else
	#define	MODE	1
#endif

#if MODE==1		// Stand-alone test base
	#define	CPU_debug(x)	//printf x;
	#define	CPU_debug1(x)	//printf x;
	#define	Xputc(c)	putc(c, stdout)
	#define	Xprintf		printf
	unsigned char *reg_datap[64];
	unsigned char reg_data[256];
	void target_read_memory(unsigned t, unsigned a, unsigned s, char *p)
	{
		CPU_debug1(("READ: t=%u a=%04x s=%u p=%04x\n", t, a, s, p))
		*++p = a;
		*p = t;
	}
#endif

#if MODE==2		// Linkable module
	#define	CPU_debug(x)
	#define	CPU_debug1(x)
	#define	Xputc(c)	wputc(c)
	#define	Xprintf		wprintf
	extern char **reg_datap;
#endif

#if MODE==3		// Includable file
	#define	CPU_debug(x)
	#define	CPU_debug1(x)
#endif

// Information loaded from the CPU definition file
unsigned
	sym_seg,			// Symbol segment
	sym_top,			// Symbol segment top marker
	sym_ptr,			// Current symbol pointer
	cpu_flags,			// Capability flags
	cpu_mh_mask,		// Memory hardware access mask
	cpu_jmp_address,	// Alternate jump address
	cpu_num_mem,		// Number of memory blocks
	cpu_num_reg,		// Number of registers defined
	cpu_num_func,		// Number of functions defined
	cpu_num_name;		// Number of instruction names defined
unsigned char
	cpu_bp_length,		// Size of breakpoints
	cpu_id_flag,		// Set for special ID modes
	cpu_kc[KCMD_SIZE],	// Kernel commands
	cpu_kernel_mem[4],	// Kernel memory
	**cpu_memory,		// Address of memory block descriptors
	**cpu_registers,	// Address of register name index table
	**cpu_functions,	// Address of function index table
	**cpu_inst_names,	// Address of instruction name index table
	*cpu_inst_data,		// Address of instruction data index table
	*cpu_reg_size,		// Register size table
	*cpu_cmd_prefix,	// Address of command prefix list
	*cpu_break_list,	// Address of break sequence list
	*cpu_launch_list,	// Address of launch sequence list
	*cpu_update_list,	// Address of register update list
	*cpu_default_reg;	// Cpu default register data

/*
 * Lookup a symbol by name
 * Format:	tttlllll	<= Type/Length of name
 *			llllllll	<= Low byte of value
 *			hhhhhhhh	<= High byte of value
 *			....		<= Symbol name (high bit set if scope info
 *			[rangl-l]	<= Low byte of low scope range
 *			[rangl-h]	<= High byte of low scope range
 *			[rangh-l]	<= Low byte of high scope range
 *			[rangh-h]	<= High byte of high scope range
 */
unsigned lookup_symbol(char *s)
{
	unsigned p1, l, t1, l1;
	unsigned char *p2;

	l = strlen(s);

	if(sym_ptr >= sym_top)
		return sym_ptr = 0;

	do {
		if((t1 = peek(sym_seg, sym_ptr) & 0x1F) == l) {		/* Length correct? */
			p1 = sym_ptr+2;
			p2 = s;
			l1 = t1;
			while((peek(sym_seg, ++p1) & 0x7F) == *p2++) {
				if(!--l1)
					return sym_ptr+1; } }
			if(peek(sym_seg, sym_ptr+3) & 0x80)	// Scope bytes
				sym_ptr += 4; }
	while((sym_ptr += (t1+3)) < sym_top);

	return sym_ptr = 0;
}

/*
 * Skip symbol in symbol table
 */
skip_symbol()
{
	unsigned s;
	s = (peek(sym_seg, sym_ptr) & 0x1F) + 3;
	if(peek(sym_seg, sym_ptr+3) & 0x80)			// Skip scope bytes
		s += 4;
	sym_ptr += s;
}

/*
 * Defing a symbol (Does not check for dups)
 */
define_symbol(unsigned char *s, unsigned char t, unsigned v, unsigned sl, unsigned sh)
{
	unsigned l, p, x;

	l = strlen(s);
	poke(sym_seg, x = sym_top++, (t << 5) | l);
	poke(sym_seg, sym_top++, v);
	poke(sym_seg, sym_top++, v >> 8);
	p = sym_top;
	while(l--)
		poke(sym_seg, sym_top++, *s++);
	if(sl | sh) {
		poke(sym_seg, p, peek(sym_seg, p) | 0x80);
		poke(sym_seg, sym_top++, sl);
		poke(sym_seg, sym_top++, sl >> 8);
		poke(sym_seg, sym_top++, sh);
		poke(sym_seg, sym_top++, sh >> 8); }
	poke(sym_seg, sym_top, 0);
	return (sym_top < x) ? -1 : 0;
}

/*
 * Display a symbol from it's value and type
 * v=Value t=Memory-type p=Current-PC m=mode
 */
int print_symbol(unsigned v, unsigned char t, unsigned pc, unsigned char m)
{
	unsigned p, p1, l, t1;

	t <<= 5;

	p = 0;
	while(p < sym_top) {
		p1 = p + ((t1 = peek(sym_seg, p)) & 0x1F);
		if(peek(sym_seg, p+3) & 0x80) {		// Out of scope?
			p1 += 4;
			if((pc < peekw(sym_seg, p1-1)) || (pc > peekw(sym_seg, p1+1)))
				goto next_sym; }
		if((peekw(sym_seg, p+1) == v) && ((t1 & 0xE0) == t)) {	// Match
			l = t1 & 0x1F;
			p += 2;
			if(m) {
				while(l--)
					Xputc(peek(sym_seg, ++p) & 0x7F); }
			return -1; }
	next_sym:
		p = p1 + 3; }
	return 0;
}

/*
 * Load the CPU definition file
 */
char *load_cpu(char *filename)
{
#ifndef GSIZE
	unsigned size;
#endif
	unsigned v;
	unsigned char t, l, *p, temp[50];
	FILE *fp;

	fp = fopen(filename, "rvqb");

	// Read fixed values
	fget(&cpu_jmp_address, sizeof(cpu_jmp_address), fp);
	fget(&cpu_flags, sizeof(cpu_flags), fp);
	fget(&cpu_mh_mask, sizeof(cpu_mh_mask), fp);
	fget(cpu_kernel_mem, sizeof(cpu_kernel_mem), fp);
	fget(cpu_kc, sizeof(cpu_kc), fp);
	cpu_bp_length = getc(fp);

	// Read and assign symbols
	while(t = getc(fp)) {
		l = t & 0x1F;
		v = getc(fp);
		v |= (getc(fp) << 8);
		p = temp;
		while(l--)
			*p++ = getc(fp);
		*p = 0;
		define_symbol(temp, (t>>5)&7, v, 0, 0); }

	// Read CPU item sizes */
	fget(&size, sizeof(size), fp);						// Get data block size
	fget(&cpu_num_mem, sizeof(cpu_num_mem), fp);		// Get # memory blocks
	fget(&cpu_num_reg, sizeof(cpu_num_reg), fp);		// Get # registers
	fget(&cpu_num_func, sizeof(cpu_num_func), fp);		// Get # functions
	fget(&cpu_num_name, sizeof(cpu_num_name), fp);		// Get # instruction names

	// Allocate memory and read data block
	if(!(cpu_memory = malloc(size))) {					// Allocate space
		fclose(fp);
		return "Out of memory"; }
	if(fget(cpu_memory, size, fp) != size) {			// Read the block
		fclose(fp);
		return "File format error 1"; }
	if(getc(fp) != -1) {								// Check format
		fclose(fp);
		return "File format error 2"; }
	fclose(fp);

	// Initialize pointers to data block entries
	cpu_registers = cpu_memory + (cpu_num_mem*2);
	cpu_functions = cpu_registers + (cpu_num_reg*2);
	cpu_inst_names = cpu_functions + (cpu_num_func*2);
	cpu_reg_size = cpu_inst_names + (cpu_num_name*2);
	cpu_cmd_prefix = cpu_reg_size + cpu_num_reg;
	cpu_break_list = cpu_cmd_prefix  + (*cpu_cmd_prefix  + 1 );
	cpu_launch_list = cpu_break_list + (*cpu_break_list  + 1);
	cpu_update_list = cpu_launch_list+ (*cpu_launch_list + 1);
	cpu_default_reg = cpu_update_list+ (*cpu_update_list + 1);
	cpu_inst_data = cpu_default_reg  + (*cpu_default_reg + 1);

	// Fixup data block pointers to relocate
	v = cpu_num_mem + cpu_num_reg + cpu_num_func + cpu_num_name;
	while(v--)
		cpu_memory[v] += (unsigned)cpu_memory;
}

/*
 * Print out a high-bit terminated value from memory
 */
void printh(unsigned char *u, unsigned w)
{
	unsigned i;
	unsigned char c;
	i = 0;
	do {
		++i;
		if((c = *u & 0x7f) < ' ')
			break;
		Xputc(c); }
	while(!(*u++ & 0x80));
	while(i < w) {
		Xputc(' ');
		++i; }
}

#ifndef NODIS
/*
 * Perform an disassembly operand function
 * mode: 0 = Calculate alternate target addresses
 *		!0 = Display operands
 */
static void cpu_function(unsigned f, unsigned pc, unsigned char *opcode_data, unsigned mode)
{
	unsigned v, v1, i;
	unsigned char *fp, *p, *lp, c, o;
	static unsigned var1, var2, var3;

	CPU_debug(("F%u", f))
	fp = cpu_functions[f];
	o = 0;
	for(;;) {
		c = *fp++;
		CPU_debug(("[%02x]", c))
		if(c < F_NUM) {
			v = c;
		do_op:
			CPU_debug(("v1=%04x v=%04x o=%04x ", v1, v, o))
			switch(o) {
			case F_ADD : v += v1; 		break;
			case F_SUB : v = v1 - v;	break;
			case F_MUL : v *= v;		break;
			case F_DIV : v = v1 / v;	break;
			case F_MOD : v = v1 % v;	break;
			case F_AND : v &= v1;		break;
			case F_OR  : v |= v1;		break;
			case F_XOR : v ^= v1;		break;
			case F_SHR : v = v1 >> v;	break;
			case F_SHL : v = v1 << v;	break;
			case F_CMP : v = v1 == v;	}
			CPU_debug(("r=%04x\n", v))
			o = 0;
			continue; }
		if((c >= F_OPCODE) & (c < F_OPEND)) {	// Opcode retreival
			v = opcode_data[c - F_OPCODE];
			goto do_op; }
		if((c >= F_ADD) && (c <= F_CMP)) {		// Arithmetic operation
			v1 = v;
			o = c;
			continue; }
		if((c >= F_SYM0) && (c <= F_SYM7)) {
			if(print_symbol(v, c-F_SYM0, pc, mode))
				++fp;
			continue; }
		switch(c) {
		case F_COND1 :	// Start conditional
			if(!v) {
				while(*fp != F_COND3) {
					if(*fp++ == F_COND2)
						break; } }
			continue;
		case F_COND2 :	// Else conditional
			while(*fp != F_COND3) {
				if(*fp++ == F_END)
					return; }
		case F_COND3 :	// End conditional
			continue;
		case F_NUM :	// Extended length number
			v = *fp++;
			v |= (*fp++ << 8);
			goto do_op;
		case F_PC :		// Current PC address
			v = pc;
			goto do_op;
		case F_LOW :	// Low byte only
			v &= 255;
			continue;
		case F_HIGH:	// High byte only
			v &= 0xFF00;
			continue;
		case F_SWAP:	// Swap high and loe
			v = (v << 8) | (v >> 8);
			continue;
		case F_EXTEND:	// Sign extensions
			if(v & 0x80)
				v |= 0xFF00;
			else
				v &= 0x00FF;
			continue;
		case F_PUT1 :	// Save variable 1
			var1 = v;
			continue;
		case F_GET1 :	// Read variable 1
			v = var1;
			goto do_op;
		case F_PUT2 :	// Save variable 2
			var2 = v;
			continue;
		case F_GET2 :	// Read variable 2
			v = var2;
			goto do_op;
		case F_PUT3	:	// Save variable 3
			var3 = v;
			continue;
		case F_GET3 :	// Read variable 3
			v = var3;
			goto do_op;
		case F_LOOPS:	// Loop start
			lp = fp;
			continue;
		case F_LOOPE:	// Loop end
			fp = lp;
			continue;
		case F_LINK :	// Link to function
			fp = cpu_functions[v];
			continue;
		case F_CHAR :	// Output as ASCII character
			if(mode) Xputc(v);
			continue;
		case F_DEC :	// Output as signed decimal number
			if(mode) Xprintf("%d", (int)v);
			continue;
		case F_UDEC :	// Output as unsigned decimal number
			if(mode) Xprintf("%u", v);
			continue;
		case F_BYTE :	// Output as hex byte
			if(mode) Xprintf("$%02x", v & 255);
			continue;
		case F_WORD :	// Output as hex word
			if(mode) Xprintf("$%04x", v);
			continue;
		case F_REG :	// Output as register name
			if(mode) printh(cpu_registers[v],0);
			continue;
		case F_END:
		case F_EXIT:
			return; }
	if(!mode) switch(c) {
		case F_REGV :	// Current register content
			i = cpu_reg_size[v];
			p = reg_datap[v] + i;
			CPU_debug1(("Reg: %u size:%u ptr:%04x", v, i, p))
			v = 0;
			while(i--)
				v = (v << 8) | *--p;
			CPU_debug1((" value:%04x\n", v))
			continue;
		case F_JUMP:	// Set jump address
			cpu_jmp_address = v;
			v = 0x80;
		case F_ID :		// Set special id
			cpu_id_flag |= v;
			continue;
		case F_MEM1:
		case F_MEM2:
		case F_MEM3:
		case F_MEM4:
		case F_MEM5:
		case F_MEM6:
			target_read_memory(c - F_MEM1, v, 2, &v);
			continue;
		default:
			if(mode) Xprintf("Unknown: %02x\n", c); } }
}

/*
 * Disassemble instruction
 *
 * Mode: 0 = Get length only
 * Mode: 1 = Get length + alternate execution address (cpu_jmp_address)
 * Mode: 2 = Get length + display instruction in current window
 */
unsigned disassemble(unsigned char *data, unsigned addr, char mode)
{
	unsigned i, l, x;
	unsigned char *p, *d, c, o, m;

	CPU_debug(("Data:%02x %02x %02x %02x\n", data[0], data[1], data[2], data[3]))
	p = cpu_inst_data;

	// Test opcode against this instruction table entry
	x = 0;
next:
	CPU_debug(("test: %02x %02x %02x %02x : ", p[0], p[1], p[2], p[3]))
	if((o = *p++) == 0xFF) {		// End of table
		return 1; }
	l = (o & 0x03) + 1;					// Length of compare butes
	m = o & 0xF0;						// Compare masking flags
	// Test this entry
	d = data;
	for(i=0; i < l; ++i) {				// Check all bytes
		c = *d++;						// Get data from opcode
		CPU_debug(("%02x", c))
		if((0x80 >> i) & m) {			// Mask if enabled for this byte
			CPU_debug(("&%02x=%02x", *p, c & *p))
			c &= *p++; }
		CPU_debug((" == %02x\n", *p))
		if(c != *p++) {					// If not match ...
			while(++i < l) {			// Advance to end of entry
				if((0x80 >> i) & m)		// Including mask bytes
					++p;
				++p; }
			p += *p;					// Skip text
			++x;
			goto next; } }				// And test the next one

	l += (o >> 2) & 3;
	CPU_debug(("Inst=%u Len=%u\n", x, l))

	switch(mode) {
	case 1 :							// Alternate address only
		cpu_id_flag = 0;
		i = *p++ - 1;						// Get text size
		++p;								// Skip inst name
		while(--i) {
			c = *p++;
			if(c & 0x80) {					// Special case
				if(c < I_REG)				// register name
					cpu_function(c & 0x7F, addr, data, 0); } }
	case 0 :							// Length only
		return l; }

	// Display the starting address & data
	Xprintf("%04x", addr);
	for(i=0; i < 6; ++i) {
		if(i < l)
			Xprintf(" %02x", data[i]);
		else
			Xprintf("   "); }
	Xprintf((l > 6) ? ".. " : "   ");

	// Process the text decription and display
	i = *p++ - 1;						// Get text size
	if(x = *p++)
		printh(cpu_inst_names[x-1], 8);
	while(--i) {
		c = *p++;
		if(c & 0x80) {					// Special case
			if(c < I_REG)				// register name
				cpu_function(c & 0x7F, addr, data, -1);
			else						// operand function
				printh(cpu_registers[c - I_REG], 0); }
		else
			Xputc(c); }
#if MODE == 1
	Xputc('\n');
#endif
	return l;
}
#endif

#if MODE==1
/*
 * Dump the content of the CPU definition file
 */
dump_cpu()
{
	unsigned i, j;
	unsigned char *p, *p1, c, d;

	printf("Kernel ID: %04x\n", cpu_jmp_address);
	printf("CPU flags: %04x\n", cpu_flags);
	printf("Breakpoint length: %u\n", cpu_bp_length);
	printf("Memory Hardware Mask: %08b\n", cpu_mh_mask);
	printf("Kernel memory block: %u %u %02x%02x\n", *cpu_kernel_mem,
		cpu_kernel_mem[1], cpu_kernel_mem[3], cpu_kernel_mem[2]);
	printf("Kernel cmds:");
	for(i=0; i < KCMD_SIZE; ++i)
		printf(" %02x", cpu_kc[i]);
	putc('\n', stdout);

	i = 0;
	while(i < sym_top) {
		j = (d = peek(sym_seg, i)) & 0x1F;
		printf("Symbol %04x t=%02x v=%04x: ", i, d, peekw(sym_seg, i+1));
		c = peek(sym_seg, i += 3);
		while(j--)
			putc(peek(sym_seg, i++) & 0x7F, stdout);
		if(c & 0x80) {
			printf(" in-scope %04x-%04x", peekw(sym_seg, i), peekw(sym_seg, i+2));
			i += 4; }
		putc('\n', stdout); }

	for(i=0; i < cpu_num_mem; ++i) {
		p = cpu_memory[i];
		printf("Memory %u : %04x-%04x : ", i, *(unsigned*)p, *(unsigned*)(p+2));
		printh(p+4, 0);
		putc('\n', stdout); }

	for(i=0; i < cpu_num_reg; ++i) {
		printf("Register %u : ", i);
		printh(cpu_registers[i], 0);
		printf(".%u\n", cpu_reg_size[i]); }

	for(i=0; i < cpu_num_func; ++i) {
		printf("Func  %u :", i);
		p = cpu_functions[i];
		do {
			printf(" %02x", *p); }
		while(*p++ < F_END);
		putc('\n', stdout); }

	for(i=0; i < cpu_num_name; ++i) {
		printf("Inst name %u : ", i);
		printh(cpu_inst_names[i], 0);
		putc('\n', stdout); }

	p = cpu_inst_data;
	i = 0;
	do {
		printf("Idata %u :", i++);
		p1 = p;
		printf(" %02x", *p++);
		c = (*p1 & 0x03)+1;
		d = *p1 & 0xF0;
		for(j=0; j < c; ++j) {
			if((0x80 >> j) & d)
				printf(" %02x", *p++);
			printf(" %02x", *p++); }
		printf(" %02x", c = *p++);
		while(--c)
			printf(" %02x", *p++);
		putc('\n', stdout); }
	while(p1[1] || p1[2]);

	printf("Cmd  prefix:");
	for(i=0; i <= *cpu_cmd_prefix; ++i)
		printf(" %02x", cpu_cmd_prefix[i]);
	putc('\n', stdout);

	printf("Break  list:");
	for(i=0; i <= *cpu_break_list; ++i)
		printf(" %02x", cpu_break_list[i]);
	putc('\n', stdout);

	printf("Launch list:");
	for(i=0; i <= *cpu_launch_list; ++i)
		printf(" %02x", cpu_launch_list[i]);
	putc('\n', stdout);

	printf("Update list:");
	for(i=0; i <= *cpu_update_list; ++i)
		printf(" %02x", cpu_update_list[i]);
	putc('\n', stdout);

	printf("Default reg: ");
	for(i=0; i <= *cpu_default_reg; ++i)
		printf(" %02x", cpu_default_reg[i]);
	putc('\n', stdout);
}
static char temp[50], *pp;
/* char *parse()
{
	char *p;
	p = temp;
	while(*pp != ':') {
		if(!*pp)
			abort("Invalid DMSCPU setting");
		if((*p++ = *pp++) == '.') {
			--p;
			break; } }
	*p = 0;
	return temp;
} */
main(int argc, char *argv[])
{
	unsigned a, i, l;
	unsigned char data[50], *p;
	static char
		initreg	= 0,
		dumpreg = 0;

	if(!getenv("DMSCPU", pp = data))
		abort("DMSCPU environment variable not set!");

	while(*pp) {
		if(*pp == ':') {
			*pp++ = 0;
			while(*pp) switch(*pp++) {
				case 'R' : dumpreg = -1;			continue;
				case 'I' : initreg = -1;			continue;
				default: abort("Invalid DMSCPU setting"); }
			break; }
		++pp; }

next:
	if(!(sym_seg = alloc_seg(4096)))
		abort("Out of memory");

	concat(temp, data, ".DMS");
	if(i = load_cpu(temp))
		abort(i);

	if(argc < 2) {
		dump_cpu();
		return; }

	// Set register data
	memcpy(p = reg_data, cpu_default_regs+1, *cpu_default_regs);
	if(initreg) {
		a = 0;
		i = *cpu_default_regs;
		while(i < sizeof(reg_data))
			reg_data[i++] = ++a; }
	for(i=0; i < cpu_num_reg; ++i) {
		reg_datap[i] = p;
		p += cpu_reg_size[i]; }

/*	for(i=0; i < 256; ++i) {
		if(!(i %16))
			printf("\n%04x", i);
		printf(" %02x", peek(sym_seg, i)); }
	putc('\n', stdout); */

//	printf("%u", lookup_symbol(argv[2], 1));

	for(i=1; i < argc; ++i)
		data[i-1] = atox(argv[i]);
	data[i-1] = 0x12;
	data[i] = 0x34;
	data[i+1] = 0x56;
	data[i+2] = 0x78;
	data[i+3] = 0x9A;

	if(dumpreg) {
		for(i=0; i < cpu_num_reg; ++i) {
			if(l = cpu_reg_size[i]) {
				if(i) putc(' ', stdout);
				p = reg_datap[i] + l;
				printh(cpu_registers[i], 0);
				putc('=', stdout);
				while(l) {
					printf("%02x", *--p);
					--l; } } }
		putc('\n', stdout); }

	p = data;
	a = *(unsigned*)reg_data;
	for(i=0; i < 1; ++i) {
		disassemble(p, a, -1);
		l = disassemble(p, a, 1);
		if(cpu_id_flag) {
			printf("Flags %02x:", cpu_id_flag);
			if(cpu_id_flag & 0x80)
				printf(" JmpAddr=%04x", cpu_jmp_address);
			if(cpu_id_flag & 0x02)
				printf(" COND");
			if(cpu_id_flag & 0x01)
				printf(" CALL");
			putc('\n', stdout); }
		a += l;
		p += l;  }
}
#endif
