/*
 * DDSPELL - A tiny/fast spelling checker
 *
 * ?COPY.TXT 2001-2005 Dave dunfield
 *  -- see COPY.TXT --.
 */
#include <stdio.h>
#include <window.h>
#include <file.h>

#define	BLOCK	4096	// Line/Block size
#define	WORD	256		// Maximum word size
#define	MSCR	20		// Size of main screen
#define	LISTMAX	36		// Maximum words in list
#define	EWIDTH	77		// Width of editor
#define	EXTRA	500		// Maximum extra words
#define	MICHR	28		// Maximum word index character
#define	FILES	25		// Maximum number of input files

#define	_MAIN	0		// Main window attribute index
#define	_CMD	1		// Command window attribute index
#define	_LINE	2		// Line display window attribute index
#define	_ERROR	3		// Error window attribute index

#define	E_CONT	1		// Allow continue in error()
#define	E_DELT	2		// Delete temporary in error()
#define	E_DUMP	4		// Allow dumping dictionary in error()

unsigned
	Changes,			// Count of changes made
	line,				// Line count
	icount,				// Input file count
	list[LISTMAX],		// List of offered words
	listi[LISTMAX],		// List of offered word indexes
	lcount,				// Count of offered words
	elist[EXTRA],		// Extra word list
	eindex[EXTRA],		// Extra word indexes
	ecount,				// Extra word count
	iseg,				// Index segment
	tseg,				// Text segment
	lseg,				// Low  link segment
	hseg,				// High link segment
	ttop = 1,			// Text segment top
	Mlen = 2,			// Pass words under this length
	windex,				// Index for parsed word
	wlen;				// Length of parsed word

char
	Dchange,			// Indicate dictionary changed
	Echange,			// Extra files have changed
	Prefix = -1,		// Control prefix handling
	Prefix_control,		// Prefix controlling flag
	*Dfile="DDSPELL.DCT", // Dictionary read file
	*Efile,				// Extra word input file
	*Hdir="",			// Home directory
	Dupwarn = -1,		// Duplicate warning
	*infiles[FILES],	// Input files
	buffer[BLOCK],		// General buffer
	buffer1[BLOCK],		// "" ""
	buffer2[BLOCK],		// ""
	word[WORD],			// Parsed word
	temp[65],			// Temporary file
	*work_ptr,			// Working position pointer
	*parse_ptr;			// General parsing pointer

FILE
	*fpi,				// Input file pointer
	*fpo;				// Output file pointer

struct WINDOW
	*twin,				// Title window
	*mwin,				// Main windows
	*lwin,				// Line window
	*cwin;				// Command window

unsigned char
	C_color[] = { 0x67, 0x20, 0x17, 0x4E },
	M_color[] = { NORMAL, REVERSE, NORMAL, REVERSE },
	*color = &C_color;

/*
 * Prefix's for word
 */
char *prefix[] = {
	"auto-", "non-", "no-", "un-", "re-", "in-", "uni-", "bi-", "pre-",
	"post-", "cross-", "hand-", "single-", "double-", "triple-", "near-",
	"less-", "on-", "off-", "multi-",
	0 };

/*
 * Common suffix's
 */
char *suffix[] = { "s", "'s", "ing", "ed", "es", "er", "or", "ly",
	"age", 0 };

/*
 * Help text
 */
static char help[] = { "\n\
use:	ddspell	filenames... [options]\n\n\
opts:	/D	= Do not warn dictionary duplicates\n\
	/M	= Force Monochrome display\n\
	/P	= Disable auto-prefix\n\
	D=file	= specify Dictionary filename\n\
	E=file	= read Extra words from file\n\
	M=#	= Minimum word length\n\
\n?COPY.TXT 2001-2005 Dave Dunfield\n -- see COPY.TXT --.\n" };

static char help1[] = { "\n\
  0-9   = Select offered replacement word\n\
  Enter = Accept entered replacement word" };

static char help2[] = { "\n\
  A     = Accept word (this document only)\n\
  B     = Back to start of line\n\
  D     = accept word and add to Dictionary\n\
  E     = Edit word\n\
  L     = edit entire Line\n\
  R     = Replace word entirely\n\
  Space = leave word unchanged\n\
  F10   = terminate DDSPELL" };

static char help3[] = { "\n\
ENTER = Exit and save changes\n\
F10   = Exit and discard changes\n\
\x1B/\x1A   = Move back/forward by one character\n\
\x18/\x19   = Move back/forward by "#EWIDTH" Characters\n\
Home  = Move to start of line\n\
End   = Move to end of line\n\
PgUp  = Clear entire line\n\
PgDn  = Clear to end of line\n\
Ins   = Toggle Insert On/Off\n\
Del   = Delete character under cursor\n\
Backsp= Delete preceeding character\n\
<text>= Enter text into data" };

static char help4[] = { "\n\nPress ENTER to proceed." };

static char dhelp1[] = { "\n\
You have make changes to the EXTRA WORD LIST:" };

static char dhelp2[] = { "\n\
You have made changes to the MAIN DICTIONARY:" };

static char dhelp3[] = { "\n\n\n\
If you would like to save the file with the new\n\
words, edit the filename below, and press ENTER.\n\n\
Press F10 to continue without saving." };


/*
 * Report an unrecoverable error
 */
register error(unsigned args)
{
	char buffer[81];
	unsigned l;

	l = _format_(nargs() * 2 + &args, buffer);
	if((l < 35) && (args & E_CONT))
		l = 35;
	wopen(40-(l/2), 10, l+2, 4, WSAVE|WBOX3|WCOPEN|color[_ERROR]);
	wcursor_off();
	wputs(buffer);
	wputc('\n');
	wputs("Press F10 to exit");
	if(args && E_CONT)
		wputs(" or F9 to continue");
	for(;;) {
		switch(wgetc()) {
		case _K10 :
			wclose();
			if(fpo)
				fclose(fpo);
			if(fpi)
				fclose(fpi);
			if(args & E_DUMP)
				dump_dictionary();
			wclose();
			wclose();
			wclose();
			wclose();
			if(args & E_DELT)
				delete(temp);
			exit(-1);
		case _K9 :
			if(args & E_CONT) {
				wclose();
				return; } }
		beep(1000, 500); }
}

/*
 * Test for a valid word symbol
 */
int iswsym(int c)
{
	return	((c >= 'a') && (c <= 'z'))
		||	((c >= 'A') && (c <= 'Z'))
		||	(c == '\'')
		||	(c == '-');
}

/*
 * Compute the word index character
 */
unsigned word_index(unsigned c)
{
	switch(c) {
	case '\'':	return 27;
	case '-' :	return 28;
	case 0 :	return 0; }
	return c - ('a'-1);
}

/*
 * Retreive character from index value
 */
unsigned index_char(unsigned i)
{
	switch(i) {
	case 0 :	return 0;
	case 27 :	return '\'';
	case 28 :	return '-'; }
	return i + ('a'-1);
}

/*
 * Parse a word from the source line
 */
unsigned parse_word()
{
	char c;

	wlen = 0;
	while(!isalpha(*parse_ptr)) {	/* Skip leading blanks */
		if(!*parse_ptr)
			return 0;
		++parse_ptr; }
	work_ptr = parse_ptr;
	while(iswsym(c = *parse_ptr)) {
		word[wlen++] = tolower(c);
		++parse_ptr; }
	while(wlen && !isalpha(word[wlen-1])) {
		--wlen;
		--parse_ptr; }
	word[wlen] = 0;
	return wlen;
}

/*
 * Add the word to the dictionary
 */
int add_word()
{
	unsigned n, w, t;
	char *p;

	t = ttop;
	if(wlen > 2) {
		if(wlen > 31) {
			error("Word is too long (max 31 chars).", E_CONT);
			return 0; }
		p = word + 1;
		w = wlen - 2;
		do {
			if(!ttop) {
				ttop = t;
				error("Out of dictionary memory", E_CONT|E_DUMP);
				return 0; }
			poke(tseg, ttop++, *++p); }
		while(--w); }

	n = peekw(iseg, windex);
	pokew(iseg, windex, t);
	poke(hseg, t, n >> 8);
	poke(lseg, t, n);

	return -1;
}

/*
 * Match a word with an entry
 */
int match(offset) asm
{
		MOV		CX,DGRP:_wlen			; Get word length
		SUB		CX,2					; Convert lengh
		JBE		loop4					; < matches	
		MOV		SI,4[BP]				; Get offset
		MOV		ES,DGRP:_tseg			; Get text segment
		MOV		BX,OFFSET DGRP:_word+2	; Point to word
loop:	MOV		AL,[BX]					; Get character
		CMP		AL,ES:[SI]				; Does it match?
		JNZ		loop2					; Yes, its OK
loop1:	INC		BX						; Next source
		INC		SI						; Next dest
		LOOP	loop					; Test them all
loop4:	MOV		AX,-1					; Indicate success
		JMP	SHORT loop3					; And exit
loop2:	CMP		AL,'?'					; Wildcard?
		JZ		loop1					; Pretend it matched
		XOR		AX,AX					; Indicate failure
loop3:
}
	
/*
 * Lookup word in internal dictionary
 */
unsigned lookup_word()
{
	unsigned w, t;
	char *p;

	/* Fail to find words that are too large */
	if((w = wlen) > 31)
		return 0;

	/* Search dictionary for matching word */
	windex = ((word_index(*word) << 10) | (word_index(word[1]) << 5) | w) << 1;
	if(t = peekw(iseg, windex)) do {
		if(match(t))
			return t; }
	while(t = (peek(hseg, t) << 8) | peek(lseg, t));

	if(Prefix_control) {
		for(t=0; p = prefix[t]; ++t) {
			w = 0;
			while(*p) {
				if(word[w++] != *p++)
					goto nextp; }
			parse_ptr = work_ptr + w;
			return -1;
		nextp: } }

	return 0;
}

/*
 * Dump the word list to a file
 */
int dump_dictionary()
{
	unsigned i, j, t, t1, w;
	unsigned char c, c1;

	/* If selected, write out extra words */
	if((int)Echange && Efile) {
		strcpy(buffer, Efile);
	Eagain:
		wclwin();
		wputs(dhelp1);
		wputs(dhelp3);
		if(edit(buffer, 64, 15)) {
			w_clwin(twin);
			w_printf(twin, "Writing: %s", buffer);
			if(!(fpo = fopen(buffer, "w"))) {
				w_clwin(cwin);
				w_printf(cwin, "Cannot WRITE: %s", buffer);
				beep(1000, 500);
				goto Eagain; }
			for(i=0; i < ecount; ++i) {
				t1 = elist[i];
				j = eindex[i];
				if(c = (j >> 11) & 31)
					putc(index_char(c), fpo);
				if(c = (j >> 6) & 31)
					putc(index_char(c), fpo);
				if((j = (j >> 1) & 31) > 2) {
					do {
						putc(peek(tseg, t1++), fpo); }
					while(--j > 2); }
				putc('\n', fpo); }
			fclose(fpo);
			fpo = 0; } }

	/* Write new dictionary if changed */
	if(Dchange) {
		/* Eliminate any "extra" words */
		for(i=0; i < ecount; ++i) {
			j = eindex[i];
			if(((j >> 1) & 31) > 2)
				poke(tseg, elist[i], 0);
			else
				pokew(iseg, j, 0); }
		sprintf(buffer1, "%s%s", Hdir, Dfile);

	Dagain:
		wclwin();
		wputs(dhelp2);
		wputs(dhelp3);
		if(edit(buffer1, 64, 15)) {
			w_clwin(twin);
			w_printf(twin, "Writing: %s", buffer1);
			if(!(fpo = fopen(buffer1, "w"))) {
				w_clwin(cwin);
				w_printf(cwin, "Cannot WRITE: %s", buffer1);
				beep(1000, 500);
				goto Dagain; }
			/* Traverse the dictionary and dump all defined words */
			i = 0;
			do {
				if(t = peekw(iseg, i)) {
					c = (i >> 11) & 31;
					c1 = (i >> 6) & 31;
					if((w = (i >> 1) & 31) <= 2) {
						if(c)
							putc(index_char(c), fpo);
						if(c1)
							putc(index_char(c1), fpo);
						putc('\n', fpo); }
					else {
						do {
							if(peek(tseg, t1 = t)) {
								putc(index_char(c), fpo);
								putc(index_char(c1), fpo);
								j = w - 2;
								do {
									putc(peek(tseg, t1++), fpo); }
								while(--j);
								putc('\n', fpo); } }
						while(t = (peek(hseg, t) << 8) | peek(lseg, t)); } } }
			while(i += 2);
			fclose(fpo);
			fpo = 0; } }
}

/*
 * Replace word in current line
 */
void replace_word(int uf)
{
	unsigned o;
	char *p, *p1;

	o = -1;
	if(!++Changes)
		Changes = -1;

	/* Determine capitalization of new word */
	if(isupper(*work_ptr) && uf) {
		p = work_ptr;
		while(++p < parse_ptr) {
			if(islower(*p))
				o = 0; }
		*word = toupper(*word);
		if(o) {			/* Capitalize entire word */
			p = word;
			while(*++p)
				*p = toupper(*p); } }

	p = buffer1;

	/* Copy current buffer up to working word */
	p1 = buffer;
	while(p1 < work_ptr)
		*p++ = *p1++;

	/* Copy in replaced word */
	p1 = word;
	while(*p1)
		*p++ = *p1++;
	o = p - buffer1;

	/* Copy in remainder of line */
	p1 = parse_ptr;
	while(*p1)
		*p++ = *p1++;
	*p = 0;

	strcpy(buffer, buffer1);
	parse_ptr = buffer + o;
}

/*
 * Lookup any close words & add to list if not already found
 */
void record_close_words()
{
	unsigned t, i;

	if(t = peekw(iseg, windex)) do {
		if(match(t)) {
			for(i=0; i < lcount; ++i) {
				if(list[i] == t)
					goto noadd; }
			if(lcount < LISTMAX) {
				listi[lcount] = windex;
				list[lcount++] = t; }
			noadd: } }
	while((t = (peek(hseg, t) << 8) | peek(lseg, t)) && (wlen > 2));
}

/*
 * Test for close word in list and add to list if not already listed
 */
void test_close_word()
{
	unsigned w, c, d;

	if((w = wlen) > 31)
		w = 31;

//wprintf("\nTest: '%s'", word);
//wgetc();
	if(*word == '?') {		/* First character wildcard */
		d = (word_index(word[1]) << 5) | w;
		for(c=1; c < MICHR; ++c) {
			windex = ((c << 10) | d) << 1;
			record_close_words(); }
		return; }

	if(word[1] == '?') {	/* Second character wildcard */
		d = (word_index(*word) << 10) | w;
		for(c=1; c < MICHR; ++c) {
			windex = ((c << 5) | d) << 1;
			record_close_words(); }
		return; }

	windex = ((word_index(*word) << 10) | (word_index(word[1]) << 5) | w) << 1;
	record_close_words();
}

/*
 * Simple line/word editor
 */
int edit(char *block, unsigned length, unsigned yo)
{
	unsigned a, i, x, y;
	int pos, l;
	char *p, c, changed;
	static char insert;

	strcpy(buffer2, block);

	if(insert)
		wcursor_block();
	else
		wcursor_line();

	a = *W_OPEN;
	pos = changed = 0;
redraw:
	wgotoxy(0, yo);
	p = buffer2;
	x = y = l = 0;
	while(c = *p) {
		wputc(c);
		++p;
		++l;
		if(++x >= EWIDTH) {
			wcleol();
			*W_OPEN = (a>>4)|(a<<4);
			wputc('<');
			*W_OPEN = a;
			wgotoxy(x=0, ++y + yo); } }
	wcleow();
	*W_OPEN = (a>>4)|(a<<4);
	wputc('<');
	*W_OPEN = a;
recmd:
	w_printf(cwin, "\r%-4u  %-9u%-9s", pos, l, insert ? "[INS]" : "");
	wgotoxy(pos%EWIDTH, (pos/EWIDTH)+yo);
	switch(i = wgetc()) {
	case _KUA :		/* Move UP */
		pos = (pos > EWIDTH) ? (pos-EWIDTH) : 0;
		goto recmd;
	case _KDA :		/* Move DOWN */
		pos = (pos < (l-EWIDTH)) ? (pos+EWIDTH) : l;
		goto recmd;
	case _KLA :		/* Move LEFT */
		if(pos)
			--pos;
		goto recmd;
	case _KRA :		/* Move RIGHT */
		if(pos < l)
			++pos;
		goto recmd;
	case _KHO :		/* Home */
		pos = 0;
		goto recmd;
	case _KEN :		/* End */
		pos = l;
		goto recmd;
	case _KPU :	/* Clear line */
		buffer2[pos=l=0] = 0;
		goto rechange;
	case _KPD :	/* Clear to end of line */
		buffer2[l = pos] = 0;
		changed = -1;
		goto rechange;
	case _KIN :		/* Insert toggle */
		if(insert) {
			insert = 0;
			wcursor_line(); }
		else {
			insert = -1;
			wcursor_block(); }
		goto recmd;
	case _KBS :		/* Backspace & delete */
		if(!pos)
			goto recmd;
		--pos;
	case _KDL :		/* Delete */
		for(x=pos; x < l; ++x)
			buffer2[x] = buffer2[x+1];
		goto rechange;
	case '\n' :		/* Exit and save */
		if(changed) {
			strcpy(block, buffer2);
			return -1; }
		return 1;
	case _K10 :		/* Exit without saving */
		return 0;
	case _K1 :		/* Help request */
		wclwin();
		wputs(help3);
		wputs(help4);
		while(wgetc() != '\n');
		goto redraw;
	case 0 :		/* Disallowed characters */
		i = -1; }
	if(i & 0xFF00) {
		beep(1000, 500);
		goto recmd; }
	if(insert) {
		if((x=l) >= length) {
			beep(1000, 500);
			goto recmd; }
		buffer2[x+1] = 0;
		while(x > pos) {
			buffer2[x] = buffer2[x-1];
			--x; } }
	if(pos < length) {
		if(pos >= l)
			buffer2[pos+1] = 0;
		buffer2[pos++] = i; }
rechange:
	changed = -1;
	goto redraw;
}

/*
 * Handle unrecognized words in the source file
 */
int unrecognized_word()
{
	unsigned i, j, sw, si, t, v, x;
	unsigned char *p, c, sa;

	if(wlen < Mlen)
		return 0;

	sa = *lwin;

	w_gotoxy(40, 0, twin);
	w_printf(twin, "%-5u'%s'", line, word);
	w_cleol(twin);
	w_clwin(lwin);
	for(p=buffer; c = *p; ++p) {
		if(p == work_ptr)
			*lwin = (sa>>4)|(sa<<4);
		else if(p == parse_ptr)
			*lwin = sa;
		if(c == '\t')
			c = ' ';
		else if(c < ' ')
			c = '.';
		w_putc(c, lwin); }
	*lwin = sa;

	lcount = 0;

	sw = wlen;
	si = windex;
	strcpy(buffer1, word);

	/* Try words with a wrong character */
	for(i=0; i < sw; ++i) {
		p = word;
		for(j=0; j < sw; ++j)
			*p++ = (j == i) ? '?' : buffer1[j];
		*p = 0;
		test_close_word(); }

	/* Try words with transposed characters */
	for(i=1; i < sw; ++i) {
		strcpy(word, buffer1);
		c = word[i];
		word[i] = word[i-1];
		word[i-1] = c;
		test_close_word(); }
		
	/* Try words with a missing character */
	wlen = sw-1;
	for(i=0; i < sw; ++i) {
		p = word;
		for(j=0; j < sw; ++j) {
			if(j != i)
				*p++ = buffer1[j]; }
		*p = 0;
		test_close_word(); }

	/* Try words with an extra character */
	wlen = sw+1;
	for(i=0; i < wlen; ++i) {
		p = word;
		for(j=0; j < wlen; ++j) {
			if(j == i)
				*p++ = '?';
			*p++ = buffer1[j]; }
		*p = 0;
		test_close_word(); }

	/*
	 * Try removing common suffixs
	 */
	for(i=0; p = suffix[i]; ++i) {
		if((j = strlen(p)) < sw) {
			strcpy(word, buffer1);
			word[wlen = sw - j] = 0;
			test_close_word(); } }

redraw_choices:
	wclwin();
	if(lcount) {
		sa = *mwin;
		for(i=0; i < lcount; ++i) {
			wgotoxy((i/(MSCR-2))*40, i % (MSCR-2));
			j = listi[i];
			p = buffer2;
			if(c = (j >> 11) & 31)
				*p++ = index_char(c);
			if(c = (j >> 6) & 31)
				*p++ = index_char(c);
			if((v = (j >> 1) & 31) > 2) {
				j = list[i];
				do {
					*p++ = peek(tseg, j++); }
				while(--v > 2); }
			for(j=0; x = suffix[j]; ++j) {
				strcpy(p, x);
				if(!strcmp(buffer1, buffer2))
					*mwin = (sa >> 4) | (sa << 4); }
			*p = 0;
			wprintf("%2u-", i);
			*mwin = sa;
			wputc(' ');
			wputs(buffer2);
			wcleol(); } }
	else {
		wputs("\nNo near matches!\n");
		wputs(help2); }

	x = v = 0;
	for(;;) {
		w_clwin(cwin);
		w_gotoxy(40, 0, cwin);
		w_puts("Press F1 for help", cwin);
		w_gotoxy(75, 0, cwin);
		w_printf(cwin, "%5u\r", 0-ttop);
		if(x)
			w_printf(cwin, "%u", v);
		if(isdigit(j = w_getc(cwin))) {	/* Numeric input */
			++x;
			v = (v * 10) + (j - '0');
			continue; }
		switch(toupper(j)) {
		case _KBS :			/* Delete numeric input */
			if(x) {
				--x;
				v /= 10;
				continue; }
			break;
		case '\n' :			/* Accept numeric input */
			if(x && (v < lcount)) {
				j = listi[v];
				p = word;
				if(c = (j >> 11) & 31)
					*p++ = index_char(c);
				if(c = (j >> 6) & 31)
					*p++ = index_char(c);
				if((j = (j >> 1) & 31) > 2) {
					t = list[v];
					do {
						*p++ = peek(tseg, t++); }
					while(--j > 2); }
				*p = 0;
				replace_word(-1);
				return 0; }
			break;
		case 'E' :		/* Edit word */
			j = 0;
			p = work_ptr;
			while(p < parse_ptr)
				word[j++] = *p++;
			word[j] = 0;
		doedit:
			if(edit(word, sizeof(word)-2, 0) == -1) {
				replace_word(0);
				parse_ptr = work_ptr;
				return -1; }
			goto redraw_choices;
		case 'R' :		/* Replace word */
			*word = 0;
			goto doedit;
		case ' ' :		/* Ignore word */
			return 0;
		case 'L' :		/* Edit line */
			if(edit(buffer, sizeof(buffer)-1, 0) == -1) {
				if(!++Changes)
					Changes = -1;
				parse_ptr = buffer;
				return -1; }
			goto redraw_choices;
		case 'B' :		/* Back to start of line */
			parse_ptr = buffer;
			return -1;
		case 'A' :		/* Accept word (this document only) */
			if(ecount >= EXTRA) {
				error("Extra word list is full!", E_CONT);
				break; }
			strcpy(word, buffer1);
			wlen = sw;
			eindex[ecount] = windex = si;
			elist[ecount] = ttop;
			if(add_word()) {
				++ecount;
				Echange = -1; }
			return 0;
		case 'D' :		/* Accept and add to dictionary */
			strcpy(word, buffer1);
			wlen = sw;
			windex = si;
			if(add_word())
				Dchange = -1;
			return 0;
		case '?' :		/* Help request */
		case _K1 :
			wclwin();
			wputs(help1);
			wputs(help2);
			wputs(help4);
			while(wgetc() != '\n');
			goto redraw_choices;
		case _K10 :
			error("User requested termination", E_DELT|E_CONT|E_DUMP);
			continue; }
		beep(1000, 500); }
}

/*
 * Read dictionary as a file
 */
void read_dictionary(char *filename, char nflag)
{
	unsigned dcount;

	/* Read in dictionary words */
	w_clwin(twin);
	w_printf(twin, "Reading dictionary: %s", filename);

	if(!(fpi = fopen(filename, "r"))) {
		error("Cannot READ: %s", filename, E_CONT);
		return; }
	dcount = 0;
	while(fgets(parse_ptr = buffer, sizeof(buffer)-1, fpi)) {
		while(parse_word()) {
			if(lookup_word()) {
				++dcount;
				if(Dupwarn)
					error("Duplicate: %s", word, E_CONT); }
			else {
				if(nflag) {
					if(ecount >= EXTRA) {
						error("Extra word list is full!", E_CONT);
						goto quit; }
					eindex[ecount] = windex;
					elist[ecount++] = ttop; }
				add_word(); } } }
	quit:
	fclose(fpi);
	fpi = 0;
	if(dcount) {
		if(nflag)
			Echange = -1;
		else
			Dchange = -1;
		wprintf("\n%u duplicated dictionary entries.. Press ENTER...", dcount);
		while(wgetc() != '\n'); }
}

main(int argc, char *argv[])
{
	unsigned i, j;
	char *ptr;

	IOB_size = 4096;

	/* Accept command line options */
	for(i=1; i < argc; ++i) {
		parse_ptr = argv[i];
		switch((toupper(*parse_ptr++) << 8) | toupper(*parse_ptr++)) {
		case '/D' :
		case '-D' : Dupwarn = 0;			continue;
		case '/P' :
		case '-P' : Prefix = 0;				continue;
		case '/M' :
		case '-M' : color = M_color;		continue;
		case 'D=' : Dfile = parse_ptr;		continue;
		case 'E=' : Efile = parse_ptr;		continue;
		case 'M=' : Mlen = atoi(parse_ptr);	continue;
		case '?'<<8:
		case '/?' :
		case '-?' : abort(help); }
		infiles[icount++] = argv[i]; }

	if(!icount)
		abort(help);

	i = 0;
	if(getenv("TEMP", temp)) {
		if(temp[(i = strlen(temp))-1] != '\\') {
			temp[i++] = '\\'; } }
	strcpy(temp+i, "$DDSPEL$.TMP");

	/* Allocate and clear far segments */
	if(!(iseg = alloc_seg(4096*4)))
		abort("Cannot allocate memory");
	tseg = iseg + (4096*1);
	lseg = iseg + (4096*2);
	hseg = iseg + (4096*3);
	i = 0;
	do {
		pokew(iseg, i, 0);
		pokew(tseg, i, 0);
		pokew(lseg, i, 0);
		pokew(hseg, i, 0); }
	while(i += 2);

	/* Determine if dictionary file has it's own path */
	ptr = Dfile;
	while(*ptr) {
		switch(*ptr++) {
		case ':' :
		case '\\':
			goto nohome; } }

	/* Pre-pend our location to dictionary file */
	parse_ptr = ptr = Hdir = argv[0];
	while(*ptr) {
		switch(*ptr++) {
		case '\\' :
		case ':' :
		parse_ptr = ptr; } }
	*parse_ptr = 0;
nohome:
	sprintf(buffer, "%s%s", Hdir, Dfile);

	twin = wopen(0, 0, 80, 1, WSAVE|WCOPEN|REVERSE);
	if(W_BASE == 0xB000)
		color = M_color;
	lwin = wopen(0, 1, 80, 23-MSCR, WSAVE|WWRAP|WCOPEN|color[_LINE]);
	cwin = wopen(0, 24, 80, 1, WSAVE|WCOPEN|color[_CMD]);
	mwin = wopen(0, 24-MSCR, 80, MSCR, WSAVE|WCOPEN|WBOX1|WSCROLL|color[_MAIN]);

	read_dictionary(buffer, 0);
	if(Efile)
		read_dictionary(Efile, -1);

	Prefix_control = Prefix;	/* Enable prefix processing */

	for(i=0; i < icount; ++i) {
		w_printf(twin, "\r%s", ptr = infiles[i]);
		w_cleol(twin);
		line = Changes = 0;
		if(!(fpi = fopen(ptr, "r")))
			error("Cannot READ input file: %s", ptr, 0);
	retemp:
		if(!(fpo = fopen(temp, "w"))) {
			error("Cannot WRITE temp file: %s", temp, E_CONT);
			wclwin();
			wputs("Please edit filename to try again");
			edit(temp, sizeof(temp)-1, 5);
			goto retemp; }
		while(fgets(parse_ptr = buffer, sizeof(word)-1, fpi)) {
			++line;
			while(parse_word()) {
				if(!lookup_word())
					unrecognized_word(); }
			fputs(buffer, fpo);
			putc('\n', fpo); }
		fclose(fpo);
		fclose(fpi);
		fpo = fpi = 0;

		w_clwin(lwin);
		w_puts("File: ", lwin);
		w_puts(ptr = infiles[i], lwin);
		wclwin();
		wprintf("\n%u change%s made to this file.\n\nPress ENTER ", Changes, (Changes == 1) ? " was" : "s were");
		if(Changes) {
			wputs("to save or F10 to leave file unchanged.");
		reprompt:
			if((j = wgetc()) == _K10)
				continue;
			if(j != '\n')
				goto reprompt;
			if(!(fpi = open(temp, F_READ)))
				error("Cannot READ temp file: %s", temp, E_DUMP);
		rewrite:
			if(!(fpo = open(infiles[i], F_WRITE))) {
				error("Cannot RE-WRITE file: %s", infiles[i], E_CONT|E_DUMP);
				wclwin();
				wputs("Please edit filename to try again");
				edit(infiles[i], 65, 5);
				goto rewrite; }
			do {
				if(j = read(buffer, (BLOCK*3), fpi))
					write(buffer, j, fpo); }
			while(j == (BLOCK*3));
			close(fpo);
			close(fpo);
			fpi = fpo = 0; }
		else {
			while(wgetc() != '\n'); } }

	dump_dictionary();

	wclose();
	wclose();
	wclose();
	wclose();

	delete(temp);
}
