/*
 * Syncronize two directory trees.
 *
 * ?COPY.TXT 1998-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 */
#include <stdio.h>
#include <file.h>

#define	MAXFILES	1424			/* Maximum that fits in 32k */
#define	ALLFILES	0x3F			/* Find ALL files */
#define	BUFFER_SIZE	4096			/* Size of copy buffer */

/* Allowable copy operations */
#define	DEL_SRC		0x0001			/* Delete from source */
#define	DEL_DST		0x0002			/* Delete from dest */
#define	CPY_S2D		0x0008			/* Copy source to dest */
#define	CPY_D2S		0x0004			/* Copy dest to source */
#define	AUTOCOPY	0x0100			/* Autocopy allowed */
#define	PROMPT		0x0200			/* Prompt to be sure */

/* DOS return codes */
#define	RC_NOFILE	2				/* File not found */
#define	RC_NODIR	3				/* Directory not found */
#define	RC_NOMORE	18				/* No more files */

/* Structure of file info block */
struct FI_block {
	char		FI_attrib;			/* file Attributes */
	unsigned	FI_time;			/* file Time stamp */
	unsigned	FI_date;			/* file Date stamp */
	unsigned	FI_size[2];			/* Size of file */
	char		FI_name[13];		/* Name of file */
	char 		FI_hit; };			/* File was processed */

/* buffer to record source entries */
struct FI_block sfiles[MAXFILES];
unsigned sfile = 0, sdir = MAXFILES;

/* Source and destination paths */
char source[81], dest[81], *send, *dend, *path;

/* Exclude directory list */
char *efile_list[100]; unsigned efile_top = 0;

char text_pool[5000], *text_ptr = &text_pool;

char debug = 0, verbose = -1, autocopy = 0, autodir = 0, autoskip = 0,
	safe = -1, bypass = 0, fast = -1, chktime = -1, xdir = 0, Hour = 255;

char help[] = { "\n\
Use: SYNC source_path dest_path [opts]\n\n\
opts:	/Copy	- Auto. copy new files\n\
	/Dir	- Auto. create dest. directories\n\
	/Hour	- update timstamp if exactly 1 Hour difference\n\
	/Ignore - Ignore Time/Date, compare size/content only\n\
	/Quiet	- Reduce informational output\n\
	/Skip	- Auto. skip safe copies\n\
	/Test	- Test content of files\n\
	/Verbose- Display files scanned\n\
	/Yes	- Inhibit prompting on unsafe operations\n\
	/Xclude	- ignore new dirs on dest.\n\
	E=file	- Specify file of files to exclude\n\n\
?COPY.TXT 1998-2005 Dave Dunfield\n -- see COPY.TXT --.\n" };

char help1[] = { "?\n\n\
A	- Auto. Copy new files for rest of this directory\n\
C	- Copy file from source to dest\n\
K *	- Copy file from dest to source\n\
D *	- Delete file from destination\n\
E *	- Delete file from source\n\
S	- Skip this file (do nothing)\n\
B	- Bypass remainder of this directory\n\
L	- Leave this directory and all subdirs\n\
<esc>	- Enter error recovery menu.\n\n\
Commands marked '*' are considered unsafe (may lose data)!\n\n" };

char help2[] = { "?\n\n\
A	- Abort SYNC & return to DOS\n\
C	- Continue with current operation\n\
D	- Execute DOS sub-prompt\n\n" };

/*
 * Show details of a file
 */
show_file(struct FI_block *file)
{
	unsigned t, d;
	char s[11];
	static char attribs[] = { '?','?','A','D','V','S','H','R' };

	t = file->FI_time;
	d = file->FI_date;
	ltoa(file->FI_size, s, 10);
	printf("%-13s %10s  %02u/%02u/%02u%3u:%02u:%02u  ",
		file->FI_name, s,
		d&0x001F, (d>>5)&0x0F, (((d>>9)&0x7f)+80) % 100,
		(t>>11)&0x1F, (t>>5)&0x3F, (t&0x1F)*2 );
	t = 0x80;
	for(d=0; d < 8; ++d) {
		putc((file->FI_attrib & t) ? attribs[d] : '.', stdout);
		t >>= 1; }
	putc('\n', stdout);
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	int i;
	char *ptr;

#ifdef SHOWMEM
	asm " MOV AX,SP";
	i = nargs();
	ptr = malloc(10);
	printf("%04x %04x %u", ptr, i, i - ptr);
#endif

	if(argc < 3)
		abort(help);

	send = stpcpy(source, argv[1]);
	dend = stpcpy(dest, argv[2]);
	switch(*(send-1)) {
		default: *send++ = '\\';
		case ':' :
		case '\\' : }
	switch(*(dend-1)) {
		default: *dend++ = '\\';
		case ':' :
		case '\\' : }
	path = send;

	for(i=3; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++)<<8) | toupper(*ptr++)) {
			case '/Y' :		// Suppress "Are you sure"
			case '-Y' : safe = 0;		continue;
			case '/V' :		// debug messages
			case '-V' : debug = 255;	continue;
			case '/Q' :		// quiet mode
			case '-Q' : verbose = 0;	continue;
			case '/S' :		// Auto skip
			case '-S' : autoskip = 255;	continue;
			case '/D' :		// Auto create directory
			case '-D' : autodir = 255;	continue;
			case '/H' :		// one Hour fix
			case '-H' : Hour = 0;		continue;
			case '/C' :		// Auto copy
			case '-C' :	autocopy = 255;	continue;
			case '/T' :		// Test each file contents
			case '-T' : fast = 0;		continue;
			case '/I' :		// Ignore timestamps
			case '-I' : chktime = 0;	continue;
			case '/X' :		// Ignore extra DEST dirs
			case '-X' :	 xdir = 255;	continue;
			case 'E=' : exclude(ptr);	continue;
		}
		printf("Unknown option: %s\n", argv[i]);
		exit(-1); }

/*	strupr(source);
	strupr(dest); */

	if(verbose) {
		printf("Source: %s\n", source);
		printf("Dest  : %s\n", dest); }

	sync_dir();
}

/*
 * Test return code & display failure
 */
test_rc(int rc)
{
	if(rc)
		printf("Operation failed, return code=%d\n", rc);
	return rc;
}

int chk1hour(struct FF_block *dir, struct FI_block *sinfo)
{
	unsigned t, h, mi, s, H, MI, S;

	if(Hour)
		return 0;

	// If size different - not 1-hour fix
	if(	(dir->FF_size[0] != sinfo->FI_size[0]) ||
		(dir->FF_size[1] != sinfo->FI_size[1])) {
		return 0; }

	// Read timestamp
	t	= sinfo->FI_time;
	h	= (t>>11)&0x1F;
	mi	= (t>>5)&0x3F;
	s	= (t&0x1F)*2;
	t	= dir->FF_time;
	H	= (t>>11)&0x1F;
	MI	= (t>>5)&0x3F;
	S	= (t&0x1F)*2;

	if((mi != MI) || (s != S))			// Minites/seconds mismatch
		return 0;

	// If dates differ - not 1-hour fix
	if(dir->FF_date != sinfo->FI_date) {
		if( ((h - H) == 23) || ((H - h) == 23) )
			goto dofix;
		return 0; }

	if( ((h - H) != 1) && ((H - h) != 1) )	// Exactly 1 hours
		return 0;

dofix:
	printf("Update TS: %s\n", dest);
	if(!(h = open(dest, F_WRITE|F_APPEND|F_BINARY))) {
		printf("Cannot WRITE: %s\n", dest);
		alert();
		return; }
	touch(h, sinfo->FI_time, sinfo->FI_date);
	close(h);

	return 255;
}

/*
 * Syncronize this directory and all subdirectories
 */
sync_dir()
{
	unsigned i, j;
	unsigned bdir;
	char *sbase, *dbase;
	static int rc;
	static struct FF_block dir;
	static struct FI_block *sinfo;
	static char *ptr, *ptr1;

	bdir = sdir;
	sbase = send;
	dbase = dend;
	sfile = bypass = 0;
	if(autocopy == 1)
		autocopy = 0;

	if(verbose)
		printf("Path: %s\n", path);
	stpcpy(send, "*.*");
	stpcpy(dend, "*.*");

	if(rc = findfirst(source, dir, ALLFILES)) {
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			printf("Unable to access source path, code=%u\n", rc);
			severe_error();
			return; } }
	else do {		/* Read all files & handle it */
		if(dir.FF_attrib & VOLUME)		/* Ignore volume labels */
			continue;
		if(*dir.FF_name == '.')
			continue;
		stpcpy(send, dir.FF_name);
		for(i=0; i < efile_top; ++i)
			if(fmatch(path, efile_list[i])) {
				printf("Skip: %s\n", path);
				goto skipsource; }
		if(dir.FF_attrib & DIRECTORY) {
			if(debug) printf("Recording source directory: %s\n", dir.FF_name);
			sfiles[--sdir].FI_hit = 0;
			memcpy(sfiles[sdir], dir+sizeof(dir.FF_reserved),
				sizeof(struct FI_block)-1);
			continue; }
		if(debug) printf("Recording source file: %s\n", dir.FF_name);
		sfiles[sfile].FI_hit = 0;
		memcpy(sfiles[sfile++], dir+sizeof(dir.FF_reserved),
			sizeof(struct FI_block)-1);
skipsource: }
	while(!findnext(dir));

retry1:
	if(rc = findfirst(dest, dir, ALLFILES)) {
		if(rc == RC_NODIR) {
			i = *--dend;
			*dend = 0;
			printf("DEST path does not exist:\n ");
			fputs(dest, stdout);
			putc('\n', stdout);
			if(autodir || yesno("Create")) {
				rc = mkdir(dest);
				*dend++ = i;
				if(!test_rc(rc))
					goto retry1;
				severe_error(); }
			return; }
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			printf("Unable to access dest path, code=%u\n", rc);
			severe_error();
			return; }
		goto checknew; }

	do {
		if(dir.FF_attrib & VOLUME)
			continue;
		stpcpy(dend, ptr = dir.FF_name);
		ptr1 = stpcpy(send, ptr);
		for(i=0; i < efile_top; ++i)
			if(fmatch(path, efile_list[i]))
				goto skipdest;
		if(dir.FF_attrib & DIRECTORY) {		/* Scan for directory */
			if(*dir.FF_name == '.')
				continue;
			for(i=sdir; i < bdir; ++i)
				if(!strcmp(sfiles[i].FI_name, ptr)) {
					sfiles[i].FI_hit = -1;
					goto dfound; }
			stpcpy(ptr1, "\\");
			if(xdir) continue;
			printf("Directory exists in DEST, but not in SOURCE\n ");
			show_file(dir+21);
			severe_error();
			dfound: continue; }
		if(bypass == 1) continue;
		if(bypass == 2) goto doexit;
		if(debug) printf("Testing dest file: %s\n", dest);
		for(i = 0; i < sfile; ++i)
			if(!strcmp(sfiles[i].FI_name, ptr))
				goto found;
		/* File from dest is not found on source */
		alert();
		printf("File exists in DEST directory, but not in SOURCE\n ");
		show_file(dir+21);
		file_action(CPY_D2S|DEL_DST, dir, dir);
		continue;

found:
	sfiles[i].FI_hit = -1;
	sinfo = sfiles[i];
	if(chktime) {
		if(dir.FF_date < sinfo->FI_date) goto older;
		if(dir.FF_date > sinfo->FI_date) goto younger;
		if(dir.FF_time < sinfo->FI_time) goto older;
		if(dir.FF_time > sinfo->FI_time) goto younger; }

	if(	(dir.FF_size[0] == sinfo->FI_size[0]) &&
		(dir.FF_size[1] == sinfo->FI_size[1])) {
		if(!fast) {
			if(fcompare(source, dest)) {
				printf("Files TIME/DATE and SIZE match, but content differs:\n Src: ");
				show_file(sinfo); printf(" Dst: "); show_file(dir+21);
				alert();
				file_action(CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|PROMPT, sinfo, dir); } }
		continue; }		// No changes

/* Timestamps are same, but file size is DIFFERENT */
	printf("Files TIME/DATE match, but SIZE is different:\n Src: ");
	show_file(sinfo); printf(" Dst: "); show_file(dir+21);
	alert();
	file_action(CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|PROMPT, sinfo, dir);
	continue;
younger:
	if(!chk1hour(dir, sinfo)) {
		printf("Destination file is NEWER than SOURCE:\n Src: ");
		show_file(sinfo); printf(" Dst: "); show_file(dir+21);
		alert();
		file_action(CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|PROMPT, sinfo, dir); }
	continue;
older:
	if(!chk1hour(dir, sinfo)) {
		printf("Destination file is OLDER than SOURCE:\n Src: ");
		show_file(sinfo); printf(" Dst: "); show_file(dir+21);
		file_action(CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|AUTOCOPY, sinfo, dir); }
skipdest: }
	while(!findnext(dir));

checknew:
	/* Check for extra files in source */
	for(i = 0; i < sfile; ++i) {
		if((sinfo = sfiles[i])->FI_hit)
			continue;
		if(bypass == 1) break;
		if(bypass == 2) goto doexit;
		stpcpy(send, sinfo->FI_name);
		stpcpy(dend, sinfo->FI_name);
		printf("File exists in SOURCE directory, but not in DEST\n ");
		show_file(sinfo);
		file_action(CPY_S2D|DEL_SRC|AUTOCOPY, sinfo, sinfo); }

/* Recurse into subdirs if present */
	if(bypass == 2) goto doexit;
	j = sdir;
	i = bdir;
	while(i > j) {
		sinfo = sfiles[--i];
		send = stpcpy(stpcpy(send, sinfo->FI_name), "\\");
		dend = stpcpy(stpcpy(dend, sinfo->FI_name), "\\");
		sync_dir();
		send = sbase;
		dend = dbase; }

doexit:
	sdir = bdir;
}

/*
 * Sound the alert
 */
alert()
{
	beep(1500, 250);
}

/*
 * Take action on a file discrepency
 */
file_action(unsigned p, struct FI_block *sinfo, struct FF_block *dinfo)
{
	if(autoskip && (p & AUTOCOPY)) goto doskip;
	if(autocopy && (p & AUTOCOPY)) goto docopy;
loop1:
	if(p & CPY_S2D) printf("All Copy(s->d) ");
	if(p & CPY_D2S) printf("Kopy(d->s) ");
	if(p & DEL_DST) printf("Delete(d) ");
	if(p & DEL_SRC) printf("Erase(s) ");
	printf("Skip Bypass Leave?");
loop2: switch(toupper(kbget())) {
		case 0x1B :
			putc('\n', stdout);
			severe_error();
			goto loop1;
		case '?' :
			fputs(help1, stdout);
			goto loop1;
		default: goto loop2;
		case 'A' : if(!(p & CPY_S2D)) goto loop2;
			autocopy = 1;
			printf("All");
		case 'C' : if(!(p & CPY_S2D)) goto loop2;
		docopy:
			printf("Copy\n");
			if(p & PROMPT) {
				if(safe && !yesno("Are you sure"))
					goto loop1; }
			copy_file(source, dest, sinfo->FI_time, sinfo->FI_date);
			break;
		case 'K' : if(!(p & CPY_D2S)) goto loop2;
			printf("Kopy\n");
			if(safe && !yesno("Are you sure"))
				goto loop1;
			copy_file(dest, source, dinfo->FF_time, dinfo->FF_date);
			break;
		case 'D' : if(!(p & DEL_DST)) goto loop2;
			printf("Delete\n");
			if(safe && !yesno("Are you sure"))
				goto loop1;
			test_rc(delete(dest));
			break;
		case 'E' : if(!(p & DEL_SRC)) goto loop2;
			printf("Erase\n");
			if(safe && !yesno("Are you sure"))
				goto loop1;
			test_rc(delete(source));
			break;
		case 'B' : bypass = 1; printf("Bypass\n"); break;
		case 'L' : bypass = 2; printf("Leave\n"); break;
		case 'S' :
		doskip: printf("Skip\n"); }
}

/*
 * Copy file & update timestamp
 */
copy_file(char *source, char *dest, unsigned time, unsigned date)
{
	HANDLE s, d;
	unsigned n;
	char buffer[BUFFER_SIZE];

	if(!(s = open(source, F_READ))) {
		printf("Cannot READ: %s\n", source);
		alert();
		return; }

	if(!(d = open(dest, F_WRITE))) {
		printf("Cannot WRITE: %s\n", dest);
		alert();
		return; }

	do {
	if(n = read(buffer, sizeof(buffer), s))
		if(test_rc(write(buffer, n, d)))
			return; }
	while(n == sizeof(buffer));

	touch(d, time, date);
	close(d);
	close(s);
}

/*
 * Touch the file with encoded date
 */
touch(handle, time, date) asm
{
		MOV		DX,4[BP]	; Get date
		MOV		CX,6[BP]	; Get time
		MOV		BX,8[BP]	; Get file handle
		MOV		AX,5701h	; Set date & time function
		INT		21h			; Ask DOS
		MOV		AX,1
		JC		tx1
		XOR		AX,AX
tx1:
}

/*
 * Ask a YES/NO question
 */
yesno(char *prompt)
{
loop1:
	printf("%s (Y/N)?", prompt);
	for(;;) switch(kbget()) {
		case 'Y' :
		case 'y' :
			printf("Yes\n");
			return -1;
		case 'N' :
		case 'n' :
			printf("No\n");
			return 0;
		case 0x1B :
			putc(' ', stdout);
			severe_error();
			goto loop1; }
}

/*
 * Handle a severe error condition
 */
severe_error()
{
	alert();
loop:
	printf("Abort Continue Dos-shell ?");
	for(;;) switch(toupper(kbget())) {
		case '?' :
			fputs(help2, stdout);
			goto loop;
		case 'A' :
			printf("Abort\n");
			exit(-1);
		case 'C' :
			printf("Continue\n");
			return;
		case 'D' :
			exec("C:\\COMMAND.COM", "");
			goto loop; }
}

/*
 * Build an exclude list of files
 */
exclude(char *file)
{
	char buffer[81], *ptr, c;
	HANDLE fh;

	fh = -1;
	ptr = buffer;
	while(c = *file++) {
		if(c == '.')
			fh = 0;
		*ptr++ = c; }
	*ptr = 0;
	if(fh) stpcpy(ptr, ".SNC");

	if(!(fh = open(buffer, F_READ))) {
		printf("Cannot READ: %s\n", buffer);
		exit(-1); }
	while(lgets(ptr = buffer, sizeof(buffer)-1, fh)) {
		while(isspace(*ptr)) ++ptr;
		if(!*ptr)
			continue;
		efile_list[efile_top++] = text_ptr;
		while(c = *ptr++) {
			if(!isspace(c))
				*text_ptr++ = toupper(c); }
		*text_ptr++ = 0; }
	close(fh);
}

/*
 * Compare one file against another
 */
fcompare(char *source, char *dest)
{
	HANDLE fs, fd;
	int r;
	unsigned n;
	char sbuffer[BUFFER_SIZE/2], dbuffer[BUFFER_SIZE/2];

	if(!(fs = open(source, F_READ))) {
		printf("Cannot READ: %s\n", source);
		severe_error();
		return 0; }
	if(!(fd = open(dest, F_READ))) {
		close(fs);
		printf("Cannot READ: %s\n", dest);
		severe_error();
		return 0; }

	r = 0;
	do {
		n = read(sbuffer, sizeof(sbuffer), fs);
		if(read(dbuffer, sizeof(dbuffer), fd) != n) {
			r = -1;
			break; }
		if(n && mcompare(sbuffer, dbuffer, n)) {
			r = -1;
			break; } }
	while(n == sizeof(sbuffer));
	close(fs);
	close(fd);
	return r;
}

/*
 * High speed compare of two block of memory
 */
mcompare(block1, block2, size)
asm {
		PUSH	DS				; Save DATA segment
		POP		ES				; Set EXTRA segment
		MOV		SI,8[BP]		; Get first string
		MOV		DI,6[BP]		; Get second string
		MOV		CX,4[BP]		; Get count
		XOR		AX,AX			; Assume success
	REPE CMPSB					; Do the compare
		JZ		cok				; Its ok
		DEC		AX				; Set -1 (fail)
cok:
}

/*
 * Match filenames with wildcards
 */
fmatch(char *f, char *p)
{
	char c;
	for(;;) switch(c = *p++) {
		case '?' :
			if(*f) ++f;
			continue;
		case '*' :
			switch(*p) {
				case 0 :
					return 1;
				case '\\' :
					while((*f != '\\') && *f)
						++f;
					continue; }
			while((*f != '.') && (*f != '\\') && *f)
				++f;
			continue;
		case 0 :
			return !*f;
		default:
			if(c != *f++)
				return 0; }
}
