/*
 * EDT: a small/simple text editor
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile with Turbo-C 2.0: tcc -ms edt.c video.asm
 */
#include <stdio.h>				/* Standard I/O definitions */
#include <ctype.h>				/* Character classification macros */
#include "tty.h"				/* Special chars for tty interface */

#define	FULL					/* Use line25 for display */

#define BUFFER_SIZE	51201		/* size of text buffer */
#define EDIT_SIZE	256			/* max. size of line to edit */
#define CMD_SIZE	50			/* max. size of input commands */
#define LINES		25			/* number of lines on terminal */
#define COLS		80			/* number of columns on terminal */
#define TAB_WIDTH	4			/* Default tab width */

extern char *refresh_line();

/*	Global variables:
 *	t_start		- Start of edit buffer
 *	t_end		- End of text in memory
 *	txtpos		- Position of current line
 *	scrtop		- Address of top of screen
 *	actpos		- Address of actual current line (txtpos or edit_buff)
 *	tag			- Pointer to start of tagged lines
 *	tag1		- Pointer to end of tagged lines
 *	changed		- Line changed flag
 *	CHANGED		- File changed flag
 *	refresh		- Screen refresh required flag
 *	newpos		- New cursor position
 *	errmsg		- Error message to output flag
 *	insflg		- Insert mode flag
 *	eolflg		- End of line display flag
 *	video		- Video attribute on/off indicator
 *	window		- Window edit mode enable flag
 *	fname		- Name of file being edited
 *	command		- Command line input buffer
 *	edit_buff	- Line edit buffer
 *	del_buff	- Line deleted buffer
 *	cx			- Virtual cursor 'x' position
 *	cy			- Virtual cursor 'y' position
 *	horz		- Real cursor 'x' position
 *	offset		- Offset of screen from start of line
 *	edlen		- Length of line being edited (0 = no line edit)
 *	delen		- Length of line in delete buffer
 */

char *t_start, *t_end, *txtpos, *scrtop, *actpos, *tag = 0, *tag1 = 0,
	*errmsg = 0, *newpos = 0, *ep;
char changed = 0, CHANGED = 0, refresh = 0, insflg = 0, eolflg=0, video = 0,
	window = -1, wblock = 0;
char fname[50], command[CMD_SIZE+1], edit_buff[EDIT_SIZE], del_buff[EDIT_SIZE];
int cx, cy, horz, offset, edlen, delen = 0, tab_width = TAB_WIDTH;

char buffer[BUFFER_SIZE];		/* text edit buffer in ram */

char *ext8[] = { "ASM", "MAC", "TCL" };

/*
 * Display a line of text
 */
char *display_line(ptr)
	char *ptr;
{
	register int h;
	register char c;

	h = 0;

	do {
		if((c = *ptr++) == 9) {
			do
				putchr(' ');
			while(++h % tab_width); }
		else {
			putchr(c);
			++h; } }
	while(c != '\n');
	return ptr;
}

/*
 * Main program
 */
main(argc, argv)
	int argc;
	char *argv[];
{
/*	Main pgm variables:
 *	i			- General unsigned int
 *	j			- Same as above
 *	tmptr		- General temporary pointer
 *	edtpos		- Misc pointer used in editing functions
 *	chr			- General purpose character variable
 *	chr1		- Same as above
 *	cchr		- Character entered from keyboard (command or text)
 */
	unsigned i, j;
	FILE *fp;
	char *tmptr, *edtpos, chr, chr1, cchr;

	if(argc < 2) {
	help:
		window = 0;
		x_help();
		exit(-1); }

/* Read in the file to be edited */
	t_end = t_start = buffer;
	edtpos = 0;
	tmptr = fname;
	do {
		chr = *argv[1]++;
		*tmptr++ = ((chr >= 'a') && (chr <= 'z')) ? chr - 0x20 : chr;
		if(chr == '.')
			edtpos = tmptr; }
	while(chr);
	if(edtpos) {
		for(i=0; i < (sizeof(ext8)/sizeof(ext8[0])); ++i) {
			if(partial_match(ext8[i], edtpos))
				tab_width = 8; } }
	if(fp = fopen(fname, "rt")) {
		t_end += fread(t_start, 1, BUFFER_SIZE - 1, fp);
		if(!feof(fp)) {
			wblock = 255;
			errmsg = "File too large\007"; }
		fclose(fp); }
	else
		errmsg = "New file";

	*t_end = '\n';

	for(i=2; i < argc; ++i) {
		tmptr = argv[i];
		switch((*tmptr++ << 8) | *tmptr++) {
		case ('-'<<8)|'v':
		case ('-'<<8)|'V' : window = 0;	continue;
		case ('c'<<8)|'=' :
		case ('C'<<8)|'=' : ep = tmptr;	continue; }
		goto help; }

	if(window)
		v_init();

home:	cx = cy = offset = horz = 0;
		scrtop = txtpos = t_start;
		--refresh;

		if(ep) {
			execute(ep);
			ep = 0; }

cmd: if(window) {
		if(newpos) {
			reposition(newpos);
			newpos = 0; }
		j = position_cursor();
		if(refresh)			/* refresh screen if nessary */
			refresh_screen(scrtop, refresh = 0);
		if(errmsg) {		/* issue error message if any */
			error_message(errmsg);
			errmsg = 0; }
		v_gotoxy(horz - offset, cy);
		cchr = v_getc();
		if((cchr != _KDO) && (cchr != _KUP))
			cx = j;
		actpos += cx;
		if(cchr < 0) {		/* special command */
		switch(cchr) {
			case _KHO:					/* start of file */
				update_changes();
				goto home;
			case _KEN:					/* end of file */
				update_changes();
				scrtop = txtpos = t_end;
				back_page(LINES-1);
				newpos = t_end;
				break;
			case _KPU:					/* page up */
				back_page(LINES);
				break;
			case _KND:					/* cursor forward */
				fwd_chr();
				break;
			case _KDO:					/* down key */
				fwd_line();
				break;
			case _KPD:					/* page down */
				fwd_page();
				break;
			case _KBS:					/* backspace key */
				back_chr();
				break;
			case _KUP:					/* up key */
				back_line();
				break;
			case _K6:					/* current line to top of screen */
				update_changes();
				scrtop = txtpos;
				cy = 0;
				--refresh;
				break;
			case _K3:					/* toggle insert flag */
				if(insflg = !insflg)
					errmsg = "Overwrite";
				else
					errmsg = "Insert";
				break;
			case _KPL:					/* cursor to start of line */
				if(!cx)
					back_line();
				cx = 0;
				break;
			case _KPR:					/* cursor to end of line */
				cx = 32767;
				if(*actpos == '\n')
					fwd_line();
				break;
			case _K11:					/* word right */
				if(*actpos == '\n')
					fwd_chr();
				else {
					while(*actpos > ' ') {
						++actpos;
						++cx; }
					while(((chr = *actpos) != '\n') && (chr <= ' ')) {
						++actpos;
						++cx; } }
				break;
			case _K12:					/* word left */
				if(cx) {
					while((cx > 0) && (*--actpos <= ' '))
						--cx;
					while((cx > 0) && (*actpos > ' ')) {
						--actpos;
						--cx; } }
				else
					back_chr();
				break;
			case _KDP:					/* backspace and delete */
				back_chr();
				cx = position_cursor();
				v_gotoxy(horz-offset, cy);
			case _KDC:					/* delete character key */
				if(txtpos < t_start)
					break;
				start_edit();
				chr = *(tmptr = edtpos = (edit_buff + cx));
				for(i=cx; i <= edlen; ++i)
					*tmptr++ = *++edtpos;
				--edlen;
				if(chr == '\n') {
					update_changes();
					if(!refresh)
						refresh_screen(txtpos, cy); }
				else
					refresh_line(edit_buff + cx);
				break;
			case _K7:			/* tag lines(s) command */
				update_changes();
				if(!tag) {
					tag = tag1 = txtpos;
					v_gotoxy(horz = 0, cy);
					refresh_line(txtpos); }
				else {
					if(txtpos < tag)
						tag = txtpos;
					else if(txtpos > tag1)
						tag1 = txtpos;
					else
						tag = tag1 = 0;
					--refresh; }
				break;
			case _K8:			/* delete line */
				update_changes();
				if((edtpos = txtpos + cx) < t_end) {
					tmptr = del_buff;
					delen = 0;
					do {
						*tmptr++ = (chr = *edtpos++);
						++delen; }
					while(chr != '\n');
					delete(txtpos + cx, delen);
					refresh_screen(txtpos, cy); }
				break;
			case _K9:			/* delete to end of line */
				update_changes();
				tmptr = del_buff;
				edtpos = txtpos + cx;
				for(delen = 0; (chr = *edtpos++) != '\n'; ++delen)
					*tmptr++ = chr;
				delete(txtpos + cx, delen);
				putchr(_CE);
				break;
			case _K10:			/* insert deleted */
				update_changes();
				insert(txtpos + cx, delen, del_buff, delen);
				refresh_screen(txtpos, cy);
				break;
			case _K4:			/* toggle eol display on/off */
				eolflg = !eolflg;
			case _KCL:			/* refresh screen */
				update_changes();
				--refresh;
				break;
			case _K5:			/* Display cursor position */
				v_gotoxy(0, LINES-1);
				putchr(_SO);
				v_printf(" Cursor: %u down, %u over, at character %u in line ",
					cy + 1, horz + 1, cx + 1);
				putchr(_SE);
				break;
#ifdef DEBUG
			case _K13:
				v_gotoxy(0, LINES-1);
				putchr(_SO);
				v_printf("S=%04x E=%04x P=%04x A=%04x, EL=%u DL=%u c=%u y=%u j=%u, H=%u o=%u",
					t_start, t_end, txtpos, actpos, edlen, delen, cx, cy, j, horz, offset);
				putchr(_SE);
				putchr(_CE);
				break;
#endif
			case _K1:			/* line mode command */
			case _K15:
				v_gotoxy(0, LINES-1);
				putchr(_CD);
				get_input("Command: ", command, CMD_SIZE);
			case _K2:		/* execute last command */
			case _K14:
				v_gotoxy(0, LINES-1);		/* display command line */
				putchr(_SO);
				video = -1;
				for(tmptr = command; *tmptr; ++tmptr)
					display_chr(*tmptr);
				putchr(_SE);
				video = 0;
				putchr(_CE);
				fflush(stdout);
				update_changes();
				execute(command);
				if(!errmsg)				/* clear displayed command */
					errmsg = "";
				break;
			default:
				errmsg = "Bad KEY\007"; } }
	else {					/* normal key pressed */
		start_edit();		/* grab the line to edit */
		tmptr = edit_buff + cx;
		if((!insflg) || (*tmptr == '\n') || (cchr == '\n')) {
			chr = *(edtpos = tmptr);
			do {
				chr1 = chr;
				chr = *++edtpos;
				*++tmptr = chr1; }
			while(chr1 != '\n');
			++edlen;
			tmptr = edit_buff + cx; }
		*tmptr = cchr;
		++cx;
		if(cchr == '\n') {
			update_changes();
			refresh_screen(txtpos, cy);
			cx = 0;
			fwd_line(); }
		else
			refresh_line(tmptr); } }
	else {
		if(errmsg) {
			v_printf("%s\n",errmsg);
			errmsg = 0; }
		if(newpos) {
			reposition(newpos);
			--refresh;
			newpos = 0; }
		if(refresh) {
			if((tmptr = txtpos) >= t_end)
				v_printf("*EOF*\n");
			else display_line(tmptr);
			refresh = 0; }
		get_input("* ", command, CMD_SIZE);
		execute(command); }
	goto cmd;
}

/* backup up one character */
back_chr() {
	if(0 > --cx) {
		cx = 32767;
		back_line(); }
}

/* backup a line */
back_line() {
	register unsigned i;

	update_changes();
	--txtpos;
	while((txtpos >= t_start) && (*--txtpos != '\n'));
	++txtpos;
	if(0 > --cy) {
		for(i=0; i < (LINES/2); ++i)
			while((scrtop >= t_start) && (*--scrtop != '\n'));
		++scrtop;
		for(i=0; (i < cx) && (*txtpos != '\n'); ++i)
			++txtpos;
		newpos = txtpos;
		--refresh; }
}

/* move back a page */
back_page(n)
	unsigned n;
{
	register unsigned i;

	update_changes();
#ifdef FULL
	for(i=0; i < n; ++i)
#else
	for(i=0; i < (LINES-1); ++i)
#endif
		while((scrtop >= t_start) && (*--scrtop != '\n'));
	txtpos = ++scrtop;
	cy = 0;
	--refresh;
}

/* move forward one character */
fwd_chr()
{
	if(*actpos != '\n')
		++cx;
	else if(fwd_line())
		cx = 0;
}

/* move forward one line */
fwd_line()
{
	register char *ptr;

	update_changes();
	ptr = txtpos;
	do  {
		if(ptr >= t_end)
			return(0); }
	while(*ptr++ != '\n');

	txtpos = ptr;

	if(cy < (LINES-2))				/* next line in on screen */
		++cy;
	else {							/* advance screen a line */
		while(*scrtop++ != '\n');
		v_gotoxy(0, LINES-1);
		putchr(_CE);
	/* some terminals may not perform forward scrolling when _DO is */
	/* printed on the bottom line.... if so, replace next two lines */
	/* with code similar to that found in 'back_line'.              */
		putchr(_DO);
		refresh_screen(txtpos, cy); }
	return(-1);
}

/* move forward one page */
fwd_page() {
	register unsigned i;

	update_changes();
#ifdef FULL
	for(i=0; i < (LINES-1); ++i)
#else
	for(i=0; i < (LINES-2); ++i)
#endif
		while((scrtop < t_end) && (*scrtop++ != '\n'));
	txtpos = scrtop;
	cy = 0;
	--refresh;
}

/* function to position the cursor at the right location	*/
/* and sets up the global variable 'actpos' to point to		*/
/* the actual character under the cursor (in the text file	*/
/* or edit buffer. Returns actual size of line 				*/
int position_cursor() {
	register char *tmptr;
	register int i;

	horz = 0;
	if(changed)
		actpos = tmptr = edit_buff;
	else
		actpos = tmptr = txtpos;

	for(i=0; (i < cx) && (*tmptr != '\n'); ++i) {
		++horz;
		if(*tmptr++ == 9)
			while(horz % tab_width)
				++horz; }

	if(horz < offset) {		/* scroll screen right */
		update_changes();
		while(horz < offset)
			offset -= COLS/2;
		--refresh; }
	else if(horz >= (offset + COLS)) {	/* scroll screen left */
		update_changes();
		while(horz >= (offset + COLS))
			offset += COLS/2;
		--refresh; }

	return(i);
}

/* position cursor at an address */
reposition(addr)
	char *addr;
{
	register char *ptr;

/* calculate new 'Y' address */
	cy=0;
	for(ptr = scrtop; ptr < addr; ++ptr) {
		if(*ptr == '\n')
			if(++cy > LINES-2)
				ptr = addr; }		/* no sence looking farther */

/* calculate new 'X' address */
	cx = 0;
	txtpos = addr;
	while((txtpos > t_start) && (*(txtpos-1) != '\n')) {
		--txtpos;
		++cx; }

	if((addr < scrtop) || (cy > LINES-2)) {
		scrtop = txtpos;
		cy = 0;
		--refresh; }
}

/* function to refresh the screen */
refresh_screen(ptr, vert)
	char *ptr;
	unsigned vert;

{
	unsigned savhorz;

	savhorz = horz;
	do {
		v_gotoxy(horz = 0, vert);
		if(ptr >= t_end) {			/* end of file reached */
			putchr(_SO);
			putstr("*EOF*");
			putchr(_SE);
			break; }
		ptr = refresh_line(ptr); }	/* display the line */
#ifdef FULL
	while(++vert < LINES);
#else
	while(++vert < (LINES-1));
#endif
	putchr(_CD);
	horz = savhorz;
}

/* refresh a line */
char *refresh_line(ptr)
	char *ptr;
{
	register unsigned eol;

	eol = offset + COLS;

/* if within tagged lines, display in inverse video */
	if((ptr >= tag) && (ptr <= tag1)) {
		video = -1;
		putchr(_SO); }

/* skip any data which preceeds the horizontal scrolling window */
	while((horz < offset) && (*ptr != '\n')) {
		if(*ptr == 9)						/* skip over tab */
			while(++horz % tab_width);
		else
			++horz;
		++ptr; }

/* output any data which is within the horizontal scrolling window */
	if(horz >= offset) {
		while(horz < eol) {
			if(*ptr == '\n') {					/* newline */
				if(eolflg)
					display_chr('\n');
				break; }
			else if(*ptr == 9) {				/* tab */
				do
					putchr(' ');
				while((horz < eol) && (++horz % tab_width)); }
			else								/* all others */
				display_chr(*ptr);
			++ptr; } }

/* skip past the window to the end of the line */
	while(*ptr++ != '\n');

/* if lines were tagged, turn off special video */
	if(video) {
		video = 0;
		putchr(_SE); }

/* if not at end, clear end of line */
	if(horz < eol)
		putchr(_CE);

	return(ptr);
}


/* display character in special video modes */
display_chr(chr)
	unsigned char chr;
{
	if(chr < ' ') {
		putchr((video) ? _SE : _SO);
		putchr(chr+'@');
		putchr((video) ? _SO : _SE); }
	else
		putchr(chr);
	++horz;
}

/* display error message */
error_message(text)
	char *text;
{
	v_gotoxy(0, LINES-1);
	putchr(_SO);
	putstr(text);
	putchr(_SE);
	putchr(_CE);
}

/* grab current line for editing */
start_edit()
{
	register char *ptr, *ptr1;

	if(!changed) {		/* starting update for a new line */
		CHANGED = changed = -1;
		edlen = 0;
		ptr = edit_buff;
		ptr1 = txtpos;
		do {
			++edlen;
			if(ptr1 >= t_end) {
				*ptr = '\n';
				refresh_screen(t_end, cy+1);
				v_gotoxy(horz-offset, cy);
				break; }
			*ptr++ = *ptr1; }
		while(*ptr1++ != '\n'); }
}

/* update edited line to text file if required */
update_changes()
{
	unsigned buflen;
	register char *ptr;

	if(changed) {
		buflen = 0;
		ptr = txtpos;
		for(ptr = txtpos; ptr < t_end; ++ptr) {
			++buflen;
			if(*ptr == '\n')
				break; }
	/* insert or delete space as required */
		if(buflen < edlen) 	/* have to insert space */
			insert(txtpos, edlen - buflen, edit_buff, edlen);
		else {
			if(buflen > edlen)
				delete(txtpos, buflen - edlen);
			cpymem(txtpos, edit_buff, edlen); }
		edlen = changed = 0; }
}

/* delete space from the text file */
delete(pos, len)
	char *pos;
	unsigned len;
{
	register char *ptr;

	ptr = pos + len;

	if(pos < txtpos) {		/* adjust pointers if deleting below */
		if((txtpos -= len) < pos)
			txtpos = pos;
		newpos = txtpos; }
	if(pos < scrtop) {
		if((scrtop -= len) < pos) { 	/* top of screen deleted */
			scrtop = pos;
			newpos = txtpos;
			--refresh; } }
	if(pos < tag)
		tag -= len;
	if(pos < tag1)
		tag1 -= len;

	while(ptr < t_end)
		*pos++ = *ptr++;
	*(t_end = pos) = '\n';
	CHANGED = -1;

}

/* insert space into the text file */
insert(pos, len, src, slen)
	char *pos, *src;
	unsigned len, slen;
{
	register char *ptr, *ptr1;

	if((t_end+len) >= t_start + BUFFER_SIZE) {		/* ran out of memory */
		wblock = 255;
		errmsg = "Out of memory!\007";
		--refresh;
		return; }

	if(pos < scrtop)			/* adjust pointers if inserting below */
		scrtop += len;
	if(pos < txtpos)
		txtpos += len;
	if(pos < tag)
		tag += len;
	if(pos < tag1)
		tag1 += len;
	if((src > pos) && (src <= t_end))
		src += len;

	ptr = t_end;			/* move text to make room */
	ptr1 = t_end += len;
	while(ptr > pos)
		*--ptr1 = *--ptr;
	while(slen--)			/* copy in destination string */
		*pos++ = *src++;
	*t_end = '\n';
	CHANGED = -1;
}

/* copy memory */
cpymem(dest, src, len)
	char *dest, *src;
	unsigned len;
{
	while(len--)
		*dest++ = *src++;
}

/* locate a line by number */
find_line(num)
	unsigned num;
{
	register char *ptr;

	if(num) {
		ptr = t_start;
		while(--num)
			while((ptr < t_end) && (*ptr++ != '\n')); }
	else
		ptr = t_end;
	return(ptr);
}

/* execute line mode commands */
execute(cmd)
	char *cmd;
{
	char *start = 0, *end, *optr, *tmptr;
	unsigned i, j;
	int k;
	char deflg, tgflg = 0, chr;
	FILE *fp;

/* get input line range */
	do {
		while((chr = *cmd++) == ' ');		/* skip leading banks */
		tmptr = txtpos;
		deflg = end = i = 0;
		switch(chr) {
			case '=':		/* tagged lines */
				if(!(tmptr = tag)) {
					errmsg = "No tagged lines\007";
					return; }
				end = tag1;
				--tgflg;
				break;
			case '/':		/* entire file */
				tmptr = t_start;
				end = t_end;
			case '*':		/* current line (already set) */
				break;
			default:		/* unknown could be numeric */
				--cmd;
				if(isdigit(chr)) {
					while(isdigit(*cmd))
						i = (i * 10) + (*cmd++ - '0');
					tmptr = find_line(i); }
				else
					--deflg; }

/* handle '+' and '-' from range */
		while((chr = *cmd++) == ' ');		/* skip leading blanks */
		if((chr == '+') || (chr == '-')) {
			i = 0;
			while(isdigit(*cmd))
				i = (i * 10) + (*cmd++ - '0');
			if(chr == '+') {
				while((tmptr < t_end) && i)
					if(*tmptr++ == '\n')
						--i; }
			else {
				++i;
				while((tmptr >= t_start) && i)
					if(*--tmptr == '\n')
						--i;
				++tmptr; }
			end = tmptr;
			deflg = 0;
			while((chr = *cmd++) == ' '); }

		if(!start)
			start = tmptr;
		if(!end)
			end = tmptr; }
	while(chr == ',');

	if(end < start) {
		tmptr = start;
		start = end;
		end = tmptr; }
	while((end < t_end) && (*end++ != '\n'));
	j = end - start;

/* get command character */
	optr = cmd;							/* pointer to operands */
	while(*cmd == ' ')					/* skip trailing blanks */
		++cmd;
	switch(chr = tolower(chr)) {
		case 0 :			/* goto line */
			newpos = start;
			break;
		case '?':			/* find */
			if(deflg) {		/* default to cursor position */
				start = txtpos + cx + 1;
				end = t_end; }
			while(start < end) {
				if(partial_match(optr, start))			/* found it */
					end = newpos = start;
				++start; }
			if(!newpos)
				errmsg = "Not found";
			break;
		case 's':			/* replace */
			tmptr = edit_buff;
			chr = *cmd++;
			while(*cmd && (*cmd != chr))
				*tmptr++ = *cmd++;
			*tmptr = 0;
			if(!*cmd++) {
				errmsg = "Invalid search string\007";
				break; }
			i = strlen(edit_buff);
			j = strlen(cmd);
			while(start < end) {
				if(partial_match(edit_buff, start)) {		/* found it */
					if(i < j) {
						insert(start, k=j-i, cmd, j);
						end += k; }
					else {
						if(j < i) {
							delete(start, k=i-j);
							end -= k; }
						cpymem(start, cmd, j); }
					newpos = start;		/* point to last one found */
					start += j;
					CHANGED = -1; }
				else					/* not here, advance to next */
					++start; }
			if(newpos)
				--refresh;
			else
				errmsg = "Not found\007";
			break;
		case 't' :		/* tag lines */
			tag = start;
			tag1 = end - 1;
			tgflg = 0;			/* incase tagged were used */
			--refresh;
			break;
		case 'c' :		/* copy lines */
			if((txtpos > start) && (txtpos < end)) {
				errmsg = "Invalid destination\007";
				break; }
			insert(txtpos, j, start, j);
			--refresh;
			break;
		case 'm' :		/* move lines */
			if((txtpos >= start) && (txtpos < end)) {
				errmsg = "Invalid destination\007";
				break; }
			insert(txtpos, j, start, j);
			if(start >= txtpos)
				start += j;
		case 'd' :		/* delete lines */
			delete(start, j);
			--refresh;
			break;
		case '$':		/* shell command */
			start_output();
			if(window) {
				v_printf("$ %s\n", cmd);
				v_gotoxy(0, 1); }
			system(cmd);
			end_output();
			break;
		case 'f' :		/* display file statistics */
			start_output();
			i = j = 0;
			for(tmptr = t_start; tmptr <= t_end; ++tmptr) {
				if(tmptr == start)
					j = i;
				if(*tmptr == '\n')
					++i; }
			v_printf("Filename: %s, %u Lines, %u Characters\n",
				fname, i, t_end - t_start);
			i = 0;
			for(tmptr = start; tmptr <= end; ++tmptr) {
				if(*tmptr == '\n')
					++i; }
			v_printf("Position: %u, %u Lines, %u Characters\n",
				j+1, i, end - start);
			v_printf("There are%s unsaved changes.\n",
				(CHANGED) ? "" : " no");
			end_output();
			break;
		case 'l':		/* list lines */
			start_output();
			while(start < end)
				start = display_line(start);
			end_output();
			break;
		case 'p':		/* print lines */
			start_output();
			i = 1;
			for(tmptr = t_start; tmptr < end;) {
				if(tmptr >= start) {
					if(tmptr == txtpos)
						chr ='*';
					else if((tmptr >= tag) && (tmptr <= tag1))
						chr = '=';
					else
						chr = ' ';
					v_printf("%c%5u ", chr, i);
					display_line(tmptr); }
				while(*tmptr++ != '\n');
				++i; }
			end_output();
			break;
		case 'i':			/* input lines */
			start_output();
			v_printf("Input:\n");
			while(i=get_input("",edit_buff, EDIT_SIZE - 1)) {
				edit_buff[i++] = '\n';
				insert(start, i, edit_buff, i);
				start += i; }
			end_output();
			break;
		case 'r':			/* read file */
			if(fp = fopen(cmd, "rt")) {
				do {
					i = fread(edit_buff, 1, EDIT_SIZE, fp);
					insert(start, i, edit_buff, i);
					start += i; }
				while( i == EDIT_SIZE);
				fclose(fp);
				--refresh; }
			else
				errmsg = "Can't open file\007";
			break;
		case 'w':			/* write file */
		case 'x':			/* write file & exit */
			if(wblock) {
				errmsg = "Blocked-'h' to clear";
				break; }
			if(deflg) {		/* default to entire file */
				start = t_start;
				end = t_end; }
			if(!*cmd)
				cmd = fname;
			if(fp = fopen(cmd, "wt")) {
				fwrite(start, 1, end - start, fp);
				fclose(fp);
				CHANGED = 0; }
			else {
				errmsg = "Can't open file\007";
				break; }
			if(chr != 'x')
				break;
		case 'q' :			/* quit command */
			if(CHANGED && ('q' != tolower(*optr)))
				errmsg = "Unsaved changes, 'qq' to quit anyway\007";
			else {
				if(window)
					putchr(_CL);
				exit(0); }
			break;
		case 'v':		/* change visual modes */
			cx = cy = 0;
			newpos = start;
			--refresh;
			if(window = !window)
				v_init();
			else
				v_putc(_CL);
			break;
		case 'h':		/* Horizontal TAB size + help request */
			if(i) {			/* Set htab size */
				tab_width = i;
				--refresh; }
			else {			/* Help request */
				wblock = 0;
				start_output();
				/* v_help() displays the help text from the code segment */
				/* thereby avoiding using up the data segment for help */
				v_help();
				end_output(); }
			/*	errmsg = "Invalid tab size\007"; */
			break;
		default:
			errmsg = "Unknown command\007"; }

	if(tgflg && !errmsg) {		/* clear tags if no errors */
			tag = tag1 = 0;
			refresh = window; }
}

/* prepare for large output command */
start_output()
{
	if(window)
		putchr(_CL);
}

/* terminate large output command */
end_output()
{
	if(window) {
		v_gotoxy(0, LINES-1);
		putstr("Press any key to continue... ");
		v_getc();
		--refresh; }
}

/* get a command line */
get_input(prompt, dest, length)
	char *prompt, dest[];
	unsigned length;
{
	register unsigned i;
	register char chr;

	i = 0;
	v_printf("%s", prompt);
	if(window) {		/* full screen mode */
		do {
			if(_KDP == (chr = v_getc())) {
				if(i) {
					putstr("\010 \010");
					--i; } }
			if((chr >= 0) && (i < length))
				display_chr(dest[i++] = chr); }
		while((chr != _K1) && (chr != _K15));
		dest[i] = 0;
		if(!*prompt)
			putchr('\n'); }
	else {				/* line by line mode */
		fflush(stdout);
		fgets(dest, length, stdin);
		while(dest[i]) {
			if(dest[i] == '\n') {
				dest[i] = 0;
				break; }
			++i; } }
	return(i);
}

/*
 * Detect a partial match
 */
partial_match(string, source)
	char *string, *source;
{
	while(*string)
		if(*string++ != *source++)
			return 0;
	return 1;
}

/*
 * Write a character to the output device
 */
putchr(chr)
	char chr;
{
	if(window)
		v_putc(chr);
	else
		putc(chr, stdout);
}

/*
 * Write a string to the output device
 */
putstr(ptr)
	char *ptr;
{
	register char c;

	while(c = *ptr++)
		putchr(c);
}

/*
 * Shared format routine, format spec. and operands are passed
 * as a pointer to the calling functions argument list.
 */
_format_(optr, outptr)
	unsigned *optr;
	char *outptr;
{
	char outstk[17], *ptr, *format, justify, zero, minus, chr;
	unsigned width, value, i;

	format = (char *) *optr;

	while(chr = *format++) {
		if(chr == '%') {					/* format code */
			chr = *format++;
			*(ptr = &outstk[16]) = justify = minus = width = value = i = 0;
			zero = ' ';
			if(chr == '-') {				/* left justify */
				--justify;
				chr = *format++; }
			if(chr == '0')					/* leading zeros */
				zero = '0';
			while(isdigit(chr)) {			/* field width specifier */
				width = (width * 10) + (chr - '0');
				chr = *format++; }

			value = *++optr;				/* get parameter value */

			switch(chr) {
				case 'd' :					/* decimal number */
					if(value > 32767) {
						value = 0 - value;
						++minus; }
				case 'u' :					/* unsigned number */
					i = 10;
					break;
				case 'x' :					/* hexidecimal number */
					i = 16;
					break;
				case 'o' :					/* octal number */
					i = 8;
					break;
				case 'b' :					/* binary number */
					i = 2;
					break;
				case 'c' :					/* character data */
					*--ptr = value;
					break;
				case 's' :					/* string */
					ptr = (char *) value;
					break;
				default:					/* all others */
					*--ptr = chr;
					--optr; }

			if(i)		/* for all numbers, generate the ASCII string */
				do {
					if((chr = (value % i) + '0') > '9')
						chr += 7;
					*--ptr = chr; }
				while(value /= i);

/* output sign if any */
			if(minus) {
				*outptr++ = '-';
				if(width)
					--width; }

/* pad with 'zero' value if right justify enabled  */
			if(width && !justify) {
				for(i = strlen(ptr); i < width; ++i)
					*outptr++ = zero; }

/* move in data */
			i = 0;
			value = width - 1;
			while((*ptr) && (i <= value)) {
				*outptr++ = *ptr++;
				++i; }

/* pad with 'zero' value if left justify enabled */
			if(width && justify) {
				while(i < width) {
					*outptr++ = zero;
					++i; } } }
		else
/* not a format code, simply display the character */
			*outptr++ = chr; }

	*outptr = 0;
}

/*
 * Formatted print to console device
 */
v_printf(args)
	unsigned args;
{
	char buffer[100];

	_format_(&args, buffer);
	putstr(buffer);
}
