#include <stdio.h>
#include <file.h>

#define LFN_FIND
#define	LFN_OPEN
#include <lfn.ch>
#include <segstore.ch>

unsigned
	FNseg[2], FNidx,		// Filename segment + index
	DNseg[2], DNidx,		// Directory segment + index
	Count,					// Count of samefiles
	Parent = -1,			// Current directory parent
	Dseg = 1,				// Directory segments
	Fseg = 3,				// File segments
	Xseg,					// External segment identifier
	Xptr,					// External segment pointer
	Dmask,					// Masked directories
	Fmask,					// Masked files
	DupSize[2];				// Duplicated size
FILE
	*Fh1,
	*Fh2;
unsigned char
	*Pattern,				// Search pattern
	*Dptr,					// Directory position marker	
	Case = 255,				// Set case
	Test = 255,				// Test file content
	First = 255,			// Display first file
	Index,					// Display index numbers
	Size,					// Display size
	Iname,					// Ignore names
	Verbose,				// Verbose mode
	Ddot,					// Drop '.' directories
	Directory[256],			// Current directory
	Temp1[1024],			// ""
	Temp2[1024];			// ""
struct FENTRY {
	unsigned		Fsize[2];
	unsigned		Fparent;
	unsigned char	Name[256]; } Fentry;
struct DENTRY {
	unsigned		Dparent;
	unsigned char	Dname[256]; } Dentry;

struct LFN Lfn;

unsigned char Help[] = { "\n\Find Duplicate files in current directory tree.\n\n\
use:	FINDUP	[pattern] [options]\n\n\
opts:	/C		= Case sensitive name match\n\
	/F		= do not show First file found\n\
	/I		= include file Index numbers\n\
	/N		= ignore Names (size/content only)\n\
	/S		= include file Size\n\
	/T		= do not Test file content\n\
	/V		= Verbose: show statistics\n\
	D=1+		= Directory segments			[1]\n\
	F=1+		= Filename  segments			[3]\n\
	I=...		= Ignore:\n\
		H	- Hidden files\n\
		S	- System files\n\
		D	- Hidden/System directories\n\
		.	- directories beginning with '.'\n\
\nDave Dunfield - "#__DATE__"\n" };

register error(unsigned args)
{
	unsigned char buf[256];
	_format_(nargs() * 2 + &args, buf);
	fputs(buf, stdout);
	if(Fh1) close(Fh1);
	if(Fh2) close(Fh2);
	exit(-1);
}

// Show status
void showseg(unsigned sp[])
{
	unsigned s, n, i, h, l;
	s = *sp;
	n = sp[1];
	do {
		i = peekw(s, 0);
		l = (i*2) + 6;
		h = peekw(s, 2);
		printf("%6u/%04x-%04x=%04x", i, h, l, h-l);
		s += 4096; }
	while(--n);
	putc('\n', stdout);
}

void showstat(unsigned char *t)
{
	printf("D:%6u %-6u", DNidx, Dseg);	showseg(DNseg);
	printf("F:%6u %-6u", FNidx, Fseg);	showseg(FNseg);
	printf("Top of mem: %04x\n", malloc(1));
	if(t)
		error("%s\n", t);
}

// Add text to segment
void addseg(unsigned *sp, unsigned char *text, unsigned len)
{
	unsigned i;
	if((i = AddSeg(sp, text, len)) == 0xFFFF)
		showstat("Segment overflow");
	sp[2] = i+1;
}

// Get text from segment
void getseg(unsigned char *d, unsigned *sp, unsigned index, unsigned char flag)
{
	unsigned i;
	if(!(i = GetSeg(sp, index, d)))
		showstat("Bad index");
	if(flag) asm {
		MOV		DGRP:_Xseg,ES
		SUB		SI,DX
		MOV		DGRP:_Xptr,SI
	}
	d[i] = 0;
// printf("%u '%s'\n", index, d);
}

/*
 * Match a filename against a pattern using unix rules
 * - '?' matches any single character
 * - '*' matches any substring
 * - '.' is treated like any other character
 */
int fmatch(unsigned char *name, unsigned char *pattern)
{
	unsigned char c;

	for(;;) switch(c = *pattern++) {
	case 0 : return *name == 0;
	case '?' :	// Single wildcard
		if(!*name++)
			return 0;
		break;
	case '*' :	// Multiple wildcard
		if(!*pattern)
			return 1;
		while(*name) {
			if(fmatch(name, pattern))
				return 1;
			++name; }
		return 0;
	default:
		if(*name++ != c)
			return 0; }
}

// Display parent directory
unsigned char *showparent(unsigned p)
{
	unsigned l;
	unsigned char *d;
	struct DENTRY t;
	static unsigned char temp[256];
	d = temp+(sizeof(temp)-1);
	*d = 0;
	while(p != -1) {
		getseg(t, DNseg, p, 0);
		l = strlen(t.Dname);
		d -= l+1;
		strcpy(d, t.Dname);
		d[l] = '\\';
		p = t.Dparent; }
	return d;
}

// Display file information
void show(unsigned i, unsigned p, unsigned char *n, unsigned s[2])
{
	if(Index)
		printf("%-6u", i);
	if(Size) {
		ltoa(s, Temp1, 10);
		printf("%-11s", Temp1); }
	fputs(showparent(p), stdout);
	fputs(n, stdout);
	putc('\n', stdout);
}

// Compare two files
int compare(unsigned char *file1, unsigned char *file2)
{
	unsigned s1, s2, r;
	if(!(Fh1 = lfn_open(file1, F_READ)))
		error("Can't open: %s\n", file1);
	if(!(Fh2 = lfn_open(file2, F_READ)))
		error("Can't open: %s\n", file2);
	r = 255;
	for(;;) {
		s1 = read(Temp1, sizeof(Temp1), Fh1);
		s2 = read(Temp2, sizeof(Temp2), Fh2);
		if(s1 != s2)
			break;
		if(!s1) {
			r = 0;
			break; }
		if(memcmp(Temp1, Temp2, s1))
			break; }
	close(Fh1);
	close(Fh2);
	Fh1 = Fh2 = 0;
	return r;
}

// Process directory & sibdirectories
void process_dir(void)
{
	unsigned i, j;
	unsigned sdi, sp;
	unsigned char *sdp;
	static unsigned char *p;

	sdi = DNidx;
	sp = Parent;
	strcpy(sdp = Dptr, "*.*");

	if(lfn_find_first(Directory, 0x3F, Lfn))
		return;
	do {
		if(Lfn.Attrib & VOLUME)		// Ignore volume label
			continue;
		if(Lfn.Attrib & DIRECTORY) {
			if(Lfn.Attrib & Dmask)	// Masked
				continue;
			if(*Lfn.Lname == '.') {
				if(Ddot)			// Drop .* directories
					continue;
				switch(Lfn.Lname[1]) {
				case 0 :						// .
				case '.': continue; } }			// ..
			i = DNidx;
			strcpy(Dentry.Dname, Lfn.Lname);
			Dentry.Dparent = Parent;
			addseg(DNseg, Dentry, strlen(Dentry.Dname)+2);
			continue; }
		if(Lfn.Attrib & Fmask)		// Masked
			continue;
		if(Case)
			strupr(Lfn.Lname);
		if(!fmatch(Lfn.Lname, Pattern))			// Does not match
			continue;
		for(i=0; i < FNidx; ++i) {
			getseg(Fentry, FNseg, i, 255);
			*Fentry.Name &= 0x7F;
			if(Iname || !strcmp(Lfn.Lname, Fentry.Name)) {	// Name match
				if((Fentry.Fsize[0] != Lfn.SizeL[0]) || (Fentry.Fsize[1] != Lfn.SizeL[1]))
					continue;
				if(Test) {
					concat(Temp1, showparent(Fentry.Fparent), Fentry.Name);
					concat(Temp2, showparent(Parent), Lfn.Lname);
					if(compare(Temp1, Temp2))
						continue;
					concat(Temp1, showparent(Fentry.Fparent), Fentry.Name);
					concat(Temp2, showparent(Parent), Lfn.Lname); }
				if(First) {
					asm {
						MOV		ES,DGRP:_Xseg
						MOV		SI,DGRP:_Xptr
						MOV		AL,ES:6[SI]
						AND		AL,AL
						JS		nos
						OR		AL,80h
						MOV		ES:6[SI],AL
					}
					show(i, Fentry.Fparent, Fentry.Name, Fentry.Fsize);
					asm {
nos:
					} }
				show(i, Parent, Lfn.Lname, Lfn.SizeL);
				longadd(DupSize, Lfn.SizeL);
				++Count;
				goto found; } }
		i = FNidx;
		strcpy(Fentry.Name, Lfn.Lname);
		longcpy(Fentry.Fsize, Lfn.SizeL);
		Fentry.Fparent = Parent;
		addseg(FNseg, Fentry, strlen(Fentry.Name)+6);
	found: }
	while !lfn_find_next(Lfn);
	lfn_find_close();

	j = DNidx;
	for(i = sdi; i < j; ++i) {
		getseg(Dentry, DNseg, Parent = i, 0);
		p = Dentry.Dname;
		while(*p)
			*Dptr++ = *p++;
		*Dptr++ = '\\';
		process_dir();
		Dptr = sdp; }

	Parent = sp;
}

main(int argc, char *argv[])
{
	unsigned i;
	for(i=1; i < argc; ++i) {
		Dptr = argv[i];
		switch((toupper(*Dptr++)<<8) | toupper(*Dptr++)) {
		case '-C' :
		case '/C' : Case = 0;					continue;	// Case sensitive
		case '-F' :
		case '/F' : First = 0;					continue;	// Display first
		case '-I' :
		case '/I' : Index = 255;				continue;	// Display index
		case '-N' :
		case '/N' : Iname = 255;				continue;	// Ignore names
		case '-S' :
		case '/S' : Size = 255;					continue;	// Include size
		case '-T' :
		case '/T' : Test = 0;					continue;	// Do not test data
		case '-V' :
		case '/V' : Verbose = 255;				continue;
		case 'D=' : Dseg = atoi(Dptr);			continue;	// Dir  segments
		case 'F=' : Fseg = atoi(Dptr);			continue;	// File segments
		case 'I=' :
			while(*Dptr) switch(toupper(*Dptr++)) {
			default: abort(Help);
			case 'H' : Fmask |= HIDDEN;				continue;
			case 'S' : Fmask |= SYSTEM;				continue;
			case 'D' : Dmask |= (HIDDEN|SYSTEM);	continue;
			case '.' : Ddot = 255; }
			continue;
		case '?'<<8:
		case '-?' :
		case '/?' : abort(Help); }
		if(Pattern)
			abort(Help);
		Pattern = Dptr - 2; }
	if(!(Fseg && Dseg))
		abort(Help);

	if(!Pattern)
		Pattern = "*";
	else if(Case)
		strupr(Pattern);

	if(SegAlloc(FNseg, Fseg) || SegAlloc(DNseg, Dseg))
		error("Out of memory");

	lfn_init();

	Dptr = Directory;
	process_dir();

	printf("%u files ", Count);
	if(Count) {
		ltoa(DupSize, Temp1, 10);
		printf("(%s bytes) ", Temp1); }
	fputs("are duplicated.\n", stdout);

	if(Verbose)
		showstat(0);
}
