/*
 * Hexidecimal screen editor:
 * NOTE: Must be compiled in SMALL model!
 *
 *	Special Keys:
 *		PgDn		= Page forward
 *		PgUp		= Page backward
 *		CTRL-PgUp	= Goto start of file
 *		CTRL-PgDn	= Goto end of file
 *		Home		= Goto start of line
 *		End			= Goto end of line
 *		Ins			= Insert a byte at cursor location
 *		Del			= Delete a byte at cursor location
 *		F1			= Toggle between HEX/ASCII editing
 *		CTRL-Home	= Redraw screen
 *	F10/Keypad '+'	= Enter command
 *	 F9/Keypad '-'	= Re-execute last command
 *	Commands:
 *		G<xxxx>		= Goto address xxxx
 *		Q			= Quit editor
 *		QQ			= Quit even if unsaved changes
 *		W[file]		= Write file
 *		X[file]		= Exit and write file
 *		?xx ...		= Search for HEX byte pattern
 *		/text...	= Search for ASCII text
 *
 * ?COPY.TXT 1983-2011 Dave Dunfield
 *  -- see COPY.TXT --
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc hexed -fop m=s
 */
#include <stdio.h>
#include <video.h>
#include <file.h>

unsigned
	Seg;

unsigned char
	*cmdptr,
	cmdbuf[50],
	searchbuf[50];

/*
 * Skip to the next non-blank character
 */
unsigned char skip()
{
	while(*cmdptr == ' ')
		++cmdptr;
	return *cmdptr;
}

/*
 * Display a character in printable form.
 * Non-printable characters display as '.'
 */
void putprint(char c)
{
	vputc(((c >= ' ') && (c < 0x7f)) ? c : '.');
}

/*
 * Write a string to the output device
 */
void putstr(char *ptr)
{
	char c;

	while(c = *ptr++)
		vputc(c);
}

/*
 * Display an error message on the bottom line of the screen
 */
void error(char *string)
{
	vgotoxy(0,24);
	V_ATTR = REVERSE;
	putstr(string);
	V_ATTR = NORMAL;
	vcleos();
	vputc(7);
}

/*
 * Test for a valid hexidecimal digit
 */
int ishex(char c)
{
	return ((c >= '0') && (c <= '9')) || ((c >= 'A') && (c <= 'F'));
}

/*
 * Get a hex value from the input line
 */
unsigned gethex(void)
{
	unsigned i;
	char chr, flag;

	flag = i = 0;
	while(ishex(chr = toupper(*cmdptr))) {
		++cmdptr;
		flag = -1;
		if(chr > '9')
			chr -= 7;
		i = (i*16) + (chr - '0'); }
	if(!flag) {
		error("Invalid HEX digit");
		return -1; }
	return i;
}

/*
 * Compare two strings for the given length
 */
int compare(unsigned char *str1, unsigned str2, unsigned len)
{
	while(len--) {
		if(peek(Seg, str2++) != *str1++)
			return 0; }
	return 1;
}

main(argc, argv)
	unsigned argc;
	char *argv[];
{
	int cursor, chr;
	unsigned i, j, k, pos, size;
	unsigned char c, mode, modified;
	FILE *fp;

	cursor = pos = mode = modified = 0;
	if(argc<2)
		abort("\nUse: hexed <file>\n\n?COPY.TXT 1983-2011 Dave Dunfield\n -- see COPY.TXT --.\n");

	Seg = alloc_seg(4096);
	fp = fopen(argv[1], "rvqb");
	size = 0;
	while((i = getc(fp)) != -1) {
		poke(Seg, size++, i);
		if(!size)
			abort("File to big!\n"); }
	fclose(fp);
	
	vopen();

draw:		/* draw entire screen */
	vgotoxy(0,0);
	for(i=0; i<30; ++i) vputc('-');
	V_ATTR = REVERSE;
	putstr(" Hex-Ed Version 1.4 ");
	V_ATTR = NORMAL;
	for(i=0; i<30; ++i) vputc('-');
	vgotoxy(0,2);
	vprintf("File: %s",argv[1]);
	vgotoxy(59,2);
	vprintf("Size: %u, ($%04x)",size, size);

redraw:		/* re-draw text area of screen */
	for(i=0; i < 16; ++i) {		/* for each line */
		vgotoxy(0, i+4);
		if(size <= (k=pos+(i*16)))
			break;
		vprintf("%04x  ",k=pos+(i*16));
		for(j = k; j < (k+16) ; ++j) {
			if(j<size)
				vprintf(" %02x", peek(Seg, j));
			else {
				vcleol();
				break; } }
		vgotoxy(63,i+4);
		for(j = k; (j<(k+16)) && (j<size); ++j)
			putprint(peek(Seg, j)); }
	vcleos();

prompt:
	if(mode)
		vgotoxy(63 + cursor%16, 4 + cursor/16);
	else
		vgotoxy(7+3*(cursor%16), 4+cursor/16);
	switch(chr = vgetc()) {
		case _KHO :		/* backup to beginning of field */
			if(cursor & 15) {
				cursor = cursor & 0xff0;
				break; }
		case _KUA :		/* backup a line (16 bytes) */
			if((cursor = cursor - 16) < 0) {
				cursor += 16;
			if(pos > 15) {
				pos -= 16;
				goto redraw; }
		case _KPU :		/* backup a page (256 bytes) */
			if(pos > 255) {
				pos -= 256;
				goto redraw; }
			else if(pos) {
				pos = 0;
				goto redraw; } }
			break;
		case _KEN :		/* advance to end of field */
			if((cursor & 15) != 15) {
				cursor |= 15;
				goto pd1; }
		case _KDA :		/* advance a line (16 bytes) */
			if((cursor + pos + 16) >= size)
				break;
			if(cursor <= (255-16)) {
				cursor += 16;
				break; }
			if((pos + 16) < size)
				pos += 16;
				goto pd1;
		case _KPD :		/* advance a page (256 bytes) */
			if((pos+256) < size)
				pos += 256;
pd1:		if((pos + cursor) > size)
				cursor = size - pos - 1 ;
			goto redraw;
		case _KLA :		/* backup one byte */
			if(cursor)
				--cursor;
			else if(pos) {
				--pos;
				goto redraw; }
			break;
		case _KRA :		/* advance one byte */
			if(size <= (pos + ++cursor)) --cursor;
			else if(255 < cursor) {
				--cursor;
				if(pos < (size - 255)) ++pos;
				goto redraw; }
			break;
		case _CPD :		/* goto end of file */
			if(size > 255) {
				cursor = 255;
				pos = size - 256; }
			else {
				cursor = size-1;
				pos = 0; }
			goto redraw;
		case _CPU :		/* goto start of file */
			cursor = pos = 0;
			goto redraw;
		case _K1 :		/* swap editing field */
			mode = ~mode;
			break;
		case _CHO :		/* re-draw the screen */
			vclscr();
			goto draw;
		case _KDL:		/* Delete a byte */
			i = pos + cursor;
			while(i < size)
				poke(Seg, i, peek(Seg, ++i));
			if((pos + cursor) >= --size) {
				if(cursor)
					--cursor;
				else if(pos)
					--pos; }
			modified = -1;
			goto redraw;
		case _KIN:		/* Insert a byte */
			i = pos + cursor;
			if(size < 65535) {
				chr = peek(Seg, i);
				while(i < size) {
					c = peek(Seg, ++i);
					poke(Seg, i, chr);
					chr = c; }
				++size; }
			modified = -1;
			goto redraw;
		case _K10 :		/* execute line mode command */
		case _KKP :
			vgotoxy(0,22);
			vcleos();
			vputs("Command: ");
			i = 0;
			do {
				if((chr = vgetc()) == _KBS) {
					if(i) {
						putstr("\010 \010");
						--i; } }
				else if(chr >= 0)
					vputc(cmdbuf[i++] = chr); }
			while((chr != _KKP) && (chr != _K10));
			cmdbuf[i] = 0;
		case _K9 :		/* repeat last command */
		case _KKM :
			vgotoxy(0, 22);
			V_ATTR = REVERSE;
			putstr(cmdbuf);
			V_ATTR = NORMAL;
			vcleos();
			cmdptr = cmdbuf;
			switch(chr = toupper(skip())) {
				case 'G' :		/* goto address */
					++cmdptr;
					skip();
					i = 0;
					if((i = gethex()) == -1)
						break;
					cursor = 0;
					pos = (i<size) ? i : size;
					goto redraw;
				case '/' :		/* Search for ASCII */
					i = 0;
					while(*++cmdptr)
						searchbuf[i++] = *cmdptr;
					if(!i)
						break;;
					goto do_search;
				case '?' :		/* search for HEX */
					++cmdptr;
					skip();
					i = 0;
					do
						searchbuf[i++] = j = gethex();
					while((j != -1) && skip());
					if(j == -1)
						break;
		do_search:	j = pos + cursor + 1;
					while(!compare(searchbuf, j, i)) {
						if(++j >= size) {
							error("Not found");
							break; } }
					if(j < size) {
						if((cursor = j - pos) & 0xff00) {
							pos = j;
							cursor = 0;
							goto redraw; } }
					break;
				case 'X' :		/* Exit and write file */
				case 'W' :		/* write file */
					++cmdptr;
					if(!skip())
						cmdptr = argv[1];
					if(fp=fopen(cmdptr, "wb")) {
						for(i=0; i < size;  ++i)
							putc(peek(Seg, i), fp);
						modified = 0;
						fclose(fp); }
					else {
						error("Can't open output file");
						break; }
					if(chr != 'X')
						break;
				case 'Q' :		/* quit (Exit) */
					if(modified && (toupper(*(cmdptr+1)) != 'Q'))
						error("Unsaved changes, 'qq' to quit anyway");
					else {
						vclscr();
						exit(0); }
					break;
				case 0 :		/* null command, do nothing */
					break;
				default:
					error("Unknown command"); }
			vgotoxy(0,22);
			vcleol();
			break;
		default:
			if(chr & 0x80)
				break;
			k = peek(Seg, pos + cursor);
			if(mode) {
				poke(Seg, pos+cursor, k = chr);
				modified = chr = -1; }
			else {
				if(ishex(chr = toupper(chr))) {
					vputc(chr);
					if(chr > '9') chr -= 7;
					i = (chr-'0')*16;
					vputc('-');
					vputc('\b');
					if(ishex(chr=toupper(vgetc()))) {
						if(chr > '9') chr -= 7;
						poke(Seg, pos+cursor, k =(chr-'0') + i);
						modified = chr = -1; } } }
				vgotoxy(7+3*(cursor%16), 4+cursor/16);
				vprintf("%02x ",k);
				vgotoxy(63 + cursor%16, 4 + cursor/16);
				putprint(k);
				if(chr == -1) {
					if(size <= (pos + ++cursor)) --cursor;
					else if(cursor > 255) {
						if(pos < (size-256)) {
							pos += 256;
							cursor = 0; }
						else 
							pos = size-(pos + cursor);
					goto redraw; } } }
	goto prompt;
}
