#include <stdio.h>
#include <setjmp.h>

#define TINY				// Quoted-args in TINY model
#define	RECALL		256		// Size of recall buffer
#define	ISTACK		25		// Size of input stack
#define	PSTACK		100		// Parameter stack
#define	LABELS		1000	// Lable list
#define	SYMBOLS		2500	// Number of symbols (divisible by 8)

// non-ASCII operators
#define	SHR			0x80	// <<
#define	SHL			0x81	// >>
#define	LE			0x82	// <=
#define	GE			0x83	// >=
#define	LAND		0x84	// &&
#define	LOR			0x85	// ||

// Bits in SymFlags
#define	SYM_CHANGE	0x80	// Symbol has been changed
#define	SYM_MACRO	0x40	// Symbol is a macro

// Provide global access to argc and argv (via Micro-C/PC runtime library)
extern int ARGC;
extern char *ARGV[];

#ifdef SMALL
	extern unsigned PSP;
	#define GET_TAIL peek(PSP, tptr)
	#define GET_TAIL_INC peek(PSP, tptr++)
#endif

#ifdef TINY
	#define GET_TAIL *tptr
	#define GET_TAIL_INC *tptr++
#endif

unsigned
	Seg,					// External name segment
	SEtop,					// Top of external segment
	SYtop,					// Top of symbol list
	Base = 10,				// Default number base
	Rsp,					// Recall stack pointer
	Psp,					// Parameter stack pointer
	Pbase,					// Parameter stack base
	Isp,					// Input stack pointer
#define	ISTK	16			// Number of bytes to stack
	Ibase,					// Input base pointer
	Iname,					// Input name
	Ifunc,					// Input function
	Iline,					// Line number for error file
	Iptr,					// Input pointer
	Lbase,					// Base of label list
	Ltop,					// Top of label list
	SEend,					// End of external segment
//
	L1[2],					// Working accumulator
	LT[2],					// Long-temp
	Rstack[RECALL][2],		// Recall stack
	Pstack[PSTACK][2],		// Parameter stack
	LabName[LABELS],		// Lable names
	LabValue[LABELS],		// Lable values
	SymName[SYMBOLS],		// Symbol name pointers
	SymValue[SYMBOLS][2];	// Symbol values
FILE
	*fp;
unsigned char
	Itype,					// Input type
	Istack[ISTACK][11],		// Input stack
	SymFlags[SYMBOLS],		// Symbol change flags
	Buffer[256],			// Input buffer
	Temp[128];				// Temp buffer
static unsigned
	L256[] = { 1, 0 };		// 256 long

extern unsigned
	Longreg[2];				// Remainder after division

jmp_buf
	Ljmp;					// for setjmp/longjmp

unsigned char Chelp[] = { "\nUse: PC [[\"]calculation[\"]] | file...\n" };

unsigned char Ihelp[] = { "\
unary: - ~ !	binary: + - * / % & | ^ == != < <= > >= << >> && ||\n\
%0-1		binary number\n\
@0-7		octal number\n\
.0-9		decimal number\n\
$0-F / 0x0-F	hexidecimal number\n\
'..' / \"..\"	character constant\n\
[n]		macro parameter / interactive result from line n\n\
[]		last interactive result\n\
(..)		use brackets to force precedence\n\
symbol[(..)]	insert symbol value / call function\n\
symbol = v	assign value to symbol\n\
#symbol ..	assign function to symbol\n\
BASE [v]	set default number base\n\
BIN [v]		show value in binary with bit positions\n\
CLEAR		clear to fresh workspace\n\
GOTO v		transfer execution within function\n\
IF v : ..	conditional\n\
MEM		show memory usage\n\
PRINT {f}v,...	formatted print: {f} = {[-][0][width]B|O|D|U|H}\n\
QUIT		exit\n\
RETURN [v]	return value from function\n\
SAVEA file	save all symbols to file\n\
SAVEC file	save changed/added symbols to file\n\
" };

// Forward references
int gii(void);
void Ipop(void);
unsigned char get_seg(unsigned char *p, unsigned a);
int expression(unsigned L[2], unsigned e);
int doline(unsigned L[2]);

#ifdef GET_TAIL
/*
 * Re-parse the command line arguments to obtain values in quotations
 *
 * New arguments are parsed into the same holding buffer as the original
 * ones, which means that the value of ARGV will not change. The global
 * ARGC is updated by this function, and the new argc value is returned
 * so that local copies may be easily updated as well.
 */
int parse_quoted_args(void)
{
	unsigned char *tptr, *aptr;
	unsigned l;
	char c, flag;

	tptr = 0x80;					/* Command tail (TINY model only) */
	aptr = ARGV[ARGC=1];
	l = GET_TAIL_INC;
	flag = 0;
	while(l--) {
		c = GET_TAIL_INC;
		if(!flag) {					/* Looking for next argument */
			if(isspace(c))
				continue;
			ARGV[ARGC++] = aptr;
			if(c == '"') {			/* Quoted argument */
				flag = 2;
				continue; }
			flag = 1; }
		switch(c) {
			case '"' :				/* Quote is special case */
				if(flag != 2)		/* Not processing within quotes */
					goto norchar;
				if(GET_TAIL == '"') {	/* Double quotes, insert a single one */
					++tptr;
					--l;
					goto norchar; }
			/* Not a double quote, fall through & end this argument */
			case 0x0D :				/* Secondary end of line indicator */
				flag = 1;
			case ' ' :				/* Space between normal arguments */
			case '\t' :
				if(flag == 1) {		/* Not quoted, end this argument */
					*aptr++ = flag = 0;
					continue; }
			default:				/* Normal character, add to arg */
			norchar:
				*aptr++ = c; } }
	*aptr = 0;
	return ARGC;
}
#endif

// Reset system variable to interactive (nothing active)
void reset()
{
	Pbase = Psp = Isp = Iname = Ifunc = Iline = Itype =
	Lbase = Ltop = 0;
	SEend = 65535;
}

// Report an error and restart
register error(unsigned args)
{
	unsigned ip;
	unsigned char buf[80], c;
	_format_(nargs() * 2 + &args, buf);
	switch(Itype) {
	case 0x01 :		// Named local
		printf("%s(%u): ", Iname, Iline);
		break;
	case 0x80 :		// External
		get_seg(Temp, Iname);
		printf("%s(%u): ", Temp, Iline); }
	fputs(buf, stdout);
	putc('\n', stdout);
	ip = Iptr;
	Iptr = Ibase;
	Isp = 0;	// Prevent pop
	while((c = gii()) && (c != '\n'))
		putc(c, stdout);
	putc('\n', stdout);
	Iptr = Ibase;
	while(++Iptr < ip)
		putc(' ', stdout);
	fputs("^\n", stdout);
	longjmp(Ljmp, 255);
}

// Get byte from input source
int gi(void) asm
{
		XOR		AH,AH			// Zero high result
		MOV		BX,DGRP:_Iptr	// Get input pointer
		MOV		AL,DGRP:_Itype	// Get type
		AND		AL,80h			// External?
		JZ		ein1			// No - normal
		MOV		ES,DGRP:_Seg	// Set ES
		MOV		AL,ES:[BX]		// Get data
		POP		BP
		RET
ein1:	MOV		AL,[BX]			// Get data
}

// Get byte from input & increment
int gii(void) asm
{
		CALL	_gi
		INC		WORD PTR DGRP:_Iptr
}

// Get buffer from external segment
unsigned char get_seg(unsigned char *p, unsigned a)
{
	while(*p = peek(Seg, a++))
		++p;
	return p;
}

// Add buffer to external segment
void add_seg(unsigned char *p)
{
	unsigned e;
	e = SEtop;
	do {
		poke(Seg, e++, *p);
		if(e >= SEend)
			error("Out of memory"); }
	while(*p++);
	p = SEtop;
	SEtop = e;
}

// Push current input position
void Ipush()
{
	unsigned char *p;
	if(Isp >= ISTACK)
		error("Too many levels");
	p = Istack[Isp++];
	memcpy(p, &Ibase, ISTK);
	p[ISTK] = Itype;
}

// Pop current input position
void Ipop()
{
	unsigned char *p;
	p = Istack[--Isp];
	memcpy(&Ibase, p, ISTK);
	Itype = p[ISTK];
}

// Skip to non-blank
int skip()
{
	int c;
	while(((c = gi()) == ' ') || (c == '\t'))
		++Iptr;
	return c;
}
int more()
{
	int c;
	switch(c = skip()) {
	case '\n':
	case ';' : c = 0; }
	return c;
}

// Parse a symbol name
unsigned symbol()
{
	unsigned i;
	unsigned char c;
	i = 0;
	while(isalnum(c = gi())) {
		Temp[i++] = toupper(c);
		++Iptr; }
	Temp[i] = 0;
	return i;
}

// match name in extra segment
unsigned match(index) asm
{
		MOV		ES,DGRP:_Seg			// Get extra segment
		MOV		SI,OFFSET DGRP:_Temp	// Offset to temp
		MOV		DI,4[BP]				// Get dest pointer
ma1:	MOV		AL,[SI]					// Get from source
		CMP		AL,ES:[DI]				// Match with dest
		JNZ		ma2						// No match
		INC		SI						// Next in source
		INC		DI						// Next in dest
		AND		AL,AL					// End of string?
		JNZ		ma1						// Keep looking
		MOV		AX,1					// Indicate found
		POP		BP
		RET
ma2:	XOR		AX,AX					// 0 = no match
}

// Lookup a symbol [Temp]
unsigned lookup()
{
	unsigned i;
	for(i=0; i < SYtop; ++i) {
		if(match(SymName[i]))
			return i; }
	return -1;
}

// Scan for labels in a function
void scanlab()
{
	unsigned c, i, l, s;
	unsigned char *p;
	l = 1;
	Lbase = Ltop;
	do {
		Ibase = Iptr;
		if(isalnum(skip())) {
			s = symbol()+1;
			if(skip() == ':') {
				if(Ltop >= LABELS)
					error("Label stack exhausted");
				i = (SEend -= s);
//				printf("'%s' = %u %x\n", Temp, l, i);
				p = Temp;
				do {
					poke(Seg, i++, *p); }
				while(*p++);
				LabName[Ltop] = SEend;
				LabValue[Ltop++] = l; } }
		while((c = gii()) && (c != '\n'));
		++l; }
	while(c == '\n');
}

// Get a value element
void value(unsigned L[2])
{
	unsigned b, c, f, t[2], t1[2];

	longset(L, f=0);
	skip();
	switch(toupper(gii())) {
	case '%' : b = 2;	goto donum;
	case '@' : b = 8;	goto donum;
	case '.' : b = 10;	goto donum;
	case '$' : dohex:	b = 16;	donum:
	donum:
		longset(t1, b);
		for(;;) {
			c = gi();
			if((c >= '0') && (c <= '9'))
				c -= '0';
			else if((c >= 'a') && (c <= 'f'))
				c -= ('a'-10);
			else if((c >= 'A') && (c <= 'F'))
				c -= ('A'-10);
			else
				c = 255;
			if(c >= b) {
				if(!f)
					error("Bad number");
				return; }
			longmul(L, t1);
			longset(t, c);
			longadd(L, t);
			++Iptr;
			++f; }
	case '-' :	value(t); longsub(L, t);					return;
	case '~' :	value(L); L[1] ^= 0xFFFF; *L ^= 0xFFFF;		return;
	case '!' :	value(t); if(!longtst(t)) longset(L, 1);	return;
	case '(' :	expression(L, ')');							return;
	case '[' :
		if(skip() == ']') {
			++Iptr;
			longcpy(L, Rstack[(Rsp-1) & (RECALL-1)]);
			return; }
		expression(t, ']');
		if(Ifunc) {			// Function parameter
			c = *t + Pbase;
			if(t[1] || (c >= Psp))
				error("Must be [%u]-[%u]", 0, Psp-Pbase);
			longcpy(L, Pstack[c]);
			return; }
		c = Rsp - *t;
		if(t[1] || (c >= RECALL) || !c)
			error("Must be [%u]-[%u]",
				(Rsp < RECALL) ? 0 : Rsp-RECALL,
				Rsp ? Rsp-1 : 0);
		longcpy(L, Rstack[*t & (RECALL-1)]);
		return;
	case '\'':
	case '"' :
		--Iptr;
		f = gii();
		while((c = gi()) != f) {
			if(!c)
				error("Improper string");
			L[1] = (L[1] << 8) | (*L >> 8);
			*L = (*L << 8) | c;
			++Iptr; }
		++Iptr;
		return;
	case '0' :
		if(gi() == 'x') {
			++Iptr;
			goto dohex; }
	default:
		--Iptr;
		if(isdigit(gi())) {		// Decimal
			b = Base;
			goto donum; }
		if(isalpha(gi())) {			// Symbol
			b = Isp;
			Isp = 0;
			symbol();
			Isp = b;
			for(c=Lbase; c < Ltop; ++c) {
				if(match(LabName[c])) {
					longset(L, LabValue[c]);
					return; } }
			if((c = lookup()) == -1)
				error("Unknown symbol '%s'", Temp);
			if(SymFlags[c] & SYM_MACRO) {
				b = Pbase;
				Pbase = Psp;
				if(skip() == '(') {		// Parameterized
					++Iptr;
					Pbase = Psp;
					do {
						f = expression(t, ',)');
						if(Psp >= PSTACK)
							error("Parameter stack overflow");
						longcpy(Pstack[Psp++], t); }
					while(f == ','); }
				Ipush();
				Ifunc = c + 1;
				Iname = SymName[c];
				Iline = 0;
				Itype = 0x80;
				Iptr = f = *SymValue[c];
				scanlab();
				Iptr = f;
xline:			++Iline;
				Ibase = Iptr;
				doline(L);
				if(gii() == '\n')
					goto xline;
				Ipop();
				Psp = Pbase;
				Pbase = b;
				return; }
			else
				longcpy(L, SymValue[c]);
			return; }
		error("Unknown value"); }
}

// Evaluate an expression
int expression(unsigned L[2], unsigned e)
{
	unsigned o, a, t[2];

	value(L);		// Get value
	for(;;) {
		skip();
		switch(o = a = gii()) {
		case '<' :		// <= <<
			switch(gi()) {
			case '<' : ++Iptr; o = SHL;	goto doval;
			case '=' : ++Iptr; o = LE;	}
			goto doval;
		case '>' :		// >= >>
			switch(gi()) {
			case '>' : ++Iptr; o = SHR;	goto doval;
			case '=' : ++Iptr; o = GE;	}
			goto doval;
		case '&' : if(gi() == '&') { ++Iptr; o = LAND; } goto doval;
		case '|' : if(gi() == '|') { ++Iptr; o = LOR; } goto doval;
		case '!' :		// !=
		case '=' :		// ==
			if(gii() != '=')
				goto syntax;
		case '+' :
		case '-' :
		case '*' :
		case '/' :
		case '%' :
		case '^' :
doval:		value(t);
			switch(o) {
			case '+' : longadd(L, t);							continue;
			case '-' : longsub(L, t);							continue;
			case '*' : longmul(L, t);							continue;
			case '&' : L[1] &= t[1]; *L &= *t;					continue;
			case '|' : L[1] |= t[1]; *L |= *t;					continue;
			case '^' : L[1] ^= t[1]; *L ^= *t;					continue;
			case '/' :
				if(!longtst(t))
					error("Divide by zero");
				longdiv(L, t);
				continue;
			case '%' :
				if(!longtst(t))
					error("Divide by zero");
				longdiv(L, t);
				longcpy(L, Longreg);
				continue;
			case LAND:
				longset(L, (longtst(L) && longtst(t)) ? 1 : 0);
				continue;
			case LOR :
				longset(L, (longtst(L) || longtst(t)) ? 1 : 0);
				continue;
			case '=' :	// ==
				longset(L, longcmp(L, t) ? 0 : 1);
				continue;
			case '!' :	// !=
				longset(L, longcmp(L, t) ? 1 : 0);
				continue;
			case '<' :	// Less than
				longset(L, (longcmp(L, t) == -1) ? 1 : 0);
				continue;
			case '>' : // Greater than
				longset(L, (longcmp(L, t) == 1) ? 1 : 0);
				continue;
			case LE :	// Less than or equal
				longset(L, (longcmp(L, t) != 1) ? 1 : 0);
				continue;
			case GE :	// Greater than or equal
				longset(L, (longcmp(L, t) != -1) ? 1 : 0);
				continue;
			case SHR :
				while(*t) {
					--*t;
					longshr(L); }
				continue;
			case SHL :
				while(*t) {
					--*t;
					longshl(L); }
				continue; }
		case ';' :
			*t = Isp;
			Isp = 0;
			while((o = gii()) && (o != '\n'));
			Isp = *t;
		case '\n' :
			o = 0;
		default:
			if((e & 255) == o) {
				if(!o)
					--Iptr;
				return a; }
			if((e >>= 8) && (e == o))
				return a;
			--Iptr;
syntax:		error("Syntax error"); } }
}

// Save symbols to file
void savesym(unsigned char f)
{
	unsigned i, j, c, ct;

	if(!more())
		error("No filename");
	i = 0;
	while((c = gi()) && !isspace(c)) {
		Temp[i++] = c;
		++Iptr; }
	Temp[i] = 0;
	for(fp=i=ct=0; i < SYtop; ++i) {
		if(f || (SymFlags[i] & SYM_CHANGE)) {
			if(!fp) {
				if(!(fp = fopen(Temp, "wv")))
					return; }
			get_seg(Temp, SymName[i]);
			if(SymFlags[i] & SYM_MACRO) {
				fprintf(fp, "#%s ", Temp);
				j = *SymValue[i];
				while(c = peek(Seg, j++)) {
					if(c == '\n')
						putc('\\', fp);
					putc(c, fp); }
				putc('\n', fp); }
			else
				fprintf(fp, "%s = $%04x%04x\n", Temp, SymValue[i][1], *SymValue[i]);
			++ct; } }
	fclose(fp);
	printf("%u symbols written\n", ct);
}

// Insure running within function
void infunc()
{
	if(!Ifunc)
		error("Function only");
}

// Scan input line to remove whitespace and detect continuation
int xscan(unsigned char *buf)
{
	unsigned char c, q, r, w, *p, *p1;

	q = r = w = 0;
	p = p1 = buf;
	for(;;) switch(c = *p++) {
		case '\'':
		case '"' :
			if(!q)
				q = c;
			else if(c == q)
				q = 0;
		default: save:
			if(w) {
				if(isalnum(c) && (p1 > buf) && isalnum(*(p1-1)))
					*p1++ = ' ';
				w = 0; }
			*p1++ = c;
			continue;
		case ' ' :
		case '\t' :
			if(q) goto save;
			w = 255;
			continue;
		case ';' :
			if(q) goto save;
			while(*p && (*p != '\n'))
				++p;
			if(*(p-1) == '\\')
				r = 255;
			continue;
		case 0 : xr1:
			while((p1 > buf) && isspace(*(p1-1)))
				--p1;
			if((p1 > buf) && *(p1-1) == '\\') {
				r = 255;
				--p1;
				goto xr1; }
			*p1 = 0;
// printf("[%s]\n", buf);
			return r; }
}

// Process an input line
int doline(unsigned L[2])
{
	unsigned i, j, w;
	unsigned char c, f, m, n, z, *p;
	static unsigned char *reserved[] = {
		"IF",
		"GOTO",
		"RETURN",
		"BIN",
		"MEM",
		"HELP",
		"QUIT",
		"CLEAR",
		"SAVEA",
		"SAVEC",
		"BASE",
		"PRINT" };
reline:
#if 0
	printf("line %u %x(%u): ", Itype, Iptr, Iline);
	p = Temp;
	i = Iptr;
	while((j = gii()) && (j != '\n'))
		*p++ = j;
	*p = 0;
	Iptr = i;
	printf(" %s\n", Temp);
#endif
	switch(toupper(more())) {
	case 0 : return 255;		// EOL
	case '#' :					// Define
		++Iptr;
		if(!isalpha(skip()))
			error("Bad symbol");
		symbol();
		skip();
		if((i = lookup()) == -1) {				// New symbol
			if(SYtop >= SYMBOLS)
				error("Too many symbols");
			SymName[i = SYtop] = SEtop;
			add_seg(Temp);
			++SYtop; }
		longset(SymValue[i], SEtop);
		SymFlags[i] = SYM_CHANGE|SYM_MACRO;
xl1:
		i = xscan(Iptr);
		if(j=*(unsigned char *)Iptr)
			add_seg(Iptr);
		if(i) {
			if(!fgets(Iptr = Buffer, sizeof(Buffer), fp))
				error("EOF in function");
			if(j)
				poke(Seg, SEtop-1, '\n');
			goto xl1; }
		return 255; }

	if(isalpha(gi())) {		// Possible label, assigment or command
		p = Iptr;
		symbol();
		switch(skip()) {
		case ':' :	// Line label
			infunc();
			++Iptr;
			goto reline;
		case '=' :	// Symbol assignment
			++Iptr;
			if(gi() != '=') {
				if((i = lookup()) == -1) {				// New symbol
					if(SYtop >= SYMBOLS)
						error("Too many symbols");
					SymName[i = SYtop] = SEtop;
					add_seg(Temp);
					++SYtop; }
				expression(L, 0);
				longcpy(SymValue[i], L);
				SymFlags[i] = SYM_CHANGE;
				return 255; } }
		// Check possible commands
		for(i=0; i < (sizeof(reserved)/2); ++i)
			if(!strcmp(reserved[i], Temp)) switch(i) {
				case 0 :		// IF
					expression(L, ':');
					if(longtst(L))
						goto reline;
					while((i = gi()) && (i != '\n'))
						++Iptr;
					return 255;
				case 1 :		// GOTO
					infunc();
					expression(L, 0);
					Iptr = Ibase = *SymValue[Ifunc-1];
					Iline = 1;
					while(--*L) {
						while((i = gii()) && (i != '\n'));
						if(!i)
							error("GOTO beyond end");
						++Iline;
						Ibase = Iptr; }
					goto reline;
				case 2 :		// RETURN
					infunc();
					if(more())
						expression(L, 0);
					Itype = 0;
					Iptr = Ibase = "";
					return 255;
				case 3 :		// BIN
					if(more())
						expression(L, 0);
					else
						longcpy(L, Rstack[(Rsp-1) & (RECALL-1)]);
					j = 0;
					memset(Buffer, ' ', 120);
					for(i = 32;;) {
						--i;
						Buffer[j+80] = '-';
						Buffer[j+40] = (i % 10) + '0';
						Buffer[j++] = (i / 10) + '0';
						if(!i)
							break;
						if(!(i & 3))
							++j; }
					Buffer[39] = Buffer[79] = '\n';
					Buffer[119] = 0;
					fputs(Buffer, stdout);
					for(i=0; i < 8; ++i) {
						putc(i ? ' ' : '\n', stdout);
						printf("%04b", L[1] >> 12);
						longshl(L);
						longshl(L);
						longshl(L);
						longshl(L); }
					putc('\n', stdout);
					return 255;
				case 4 :		// MEM
					free(p = malloc(1));
					printf("%04x-%04x (%u free) %x\n", SEtop, SEend, SEend-SEtop, p);
					printf("%u of "#SYMBOLS" symbols\n", SYtop);
					printf("%u of "#PSTACK" parameters\n", Psp);
					printf("%u of "#LABELS" labels\n", Ltop);
					return 255;
				case 5 :		// HELP
					fputs(Ihelp, stdout);
					return 255;
				case 6 :		// QUIT
					exit(0);
				case 7 :		// CLEAR
					reset();
					SYtop = SEtop = 0;
					return 255;
				case 8 :		// SAVEA
					savesym(255);
					return 255;
				case 9 :		// SAVEC
					savesym(0);
					return 255;
				case 10 :		// BASE
					if(more()) {
						expression(L, 0);
						Base = *L; }
					else
						printf("%u\n", Base);
					return 255;
				case 11:		// PRINT
					f = 0;
ap:					m = n = w = 0;
					z = ' ';
					j = Base;
					switch(c = skip()) {
					case '\'':
					case '"' :
						++Iptr;
						f = 0;
						while((i = gii()) != c) {
							if((i == '\n') || !i)
								error("Improper string");
							putc(i, stdout); }
						if(skip() == ',') {
							++Iptr;
							goto ap; }
						goto ae; }
					if(f)
						putc(' ', stdout);
					switch(c=more()) {
					case 0 :
						return 255;
					case '{' :
						++Iptr;
						if(gi() == '-') {
							++Iptr;
							m = 255; }
						if(gi() == '0')
							z = '0';
						while(isdigit(i = gi())) {
							w = (w * 10) + (i - '0');
							++Iptr; }
						switch(toupper(gii())) {
						default: error("Bad format");
						case 'B' : j = 2;	break;
						case 'O' : j = 8;	break;
						case 'D' : n = 255;
						case 'U' : j = 10;	break;
						case 'H' : j = 16; }
						if(gii() != '}')
							error("Bad format"); }
					f = 255;
					c = expression(L, ','<<8);
					if(n) {
						if(L[1] & 0x8000) {
							longcpy(LT, L);
							longset(L, 0);
							longsub(L, LT); }
						else
							n = 0; }
					longset(LT, j);
					j = 0;
					do {
						longdiv(L, LT);
						if(*Longreg > 9)
							*Longreg += 7;
						Temp[j++] = *Longreg+'0'; }
					while(longtst(L));
					if(n)
						Temp[j++] = '-';
					i = j;
					if(!m) {		// Right justified
						while(j++ < w)
							putc(z, stdout); }
					while(i)
						putc(Temp[--i], stdout);
					if(m) {			// Left justified
						while(j++ < w)
							putc(z, stdout); }
					if(c == ',')
						goto ap;
ae:					putc('\n', stdout);
					return 255;
		}
		Iptr = p; }
	expression(L, 0);
	return 0;
}

// Display result of calculation
void result()
{
	unsigned t[2];
	if(L1[1] & 0x8000) {
		longset(t, 0);
		longsub(t, L1);
		ltoa(t, Temp, 10);
		printf("-%s ", Temp); }
	ltoa(L1, Temp, 10);
	ltoa(L1, Temp+32, 8);
	printf("%s $%04x%04x @%s %%%016b%016b\n", Temp, L1[1], *L1, Temp+32, L1[1], *L1);
}

// Determine if argument is a filename
int isfile(unsigned char *p)
{
	int c;
	unsigned char f;
	f = 255;
	for(;;)  switch(c = *p++) {
		case 0 : return 255;
		case '.' : continue;
		default:
			if(!(f ? isalpha(c) : isalnum(c)))
				return 0;
			f = 0; }
}

main(int argc, char *argv[])
{
	unsigned i;
	unsigned char *Ptr, f;

#ifdef GET_TAIL
	argc = parse_quoted_args();
#endif
	printf("Programmers Calculator - Dave Dunfield - "#__DATE__"\n");
	reset();
	Seg = alloc_seg(4096);
	if(setjmp(Ljmp))
		exit(-1);
	for(i=f=1; i < argc; ++i) {
		Ptr = argv[i];
		switch((toupper(*Ptr++)<<8) | toupper(*Ptr++)) {
		case '?'<<8:
		case '-?' :
		case '/?' : abort(Chelp);
		}
		if(isfile(Ptr -= 2)) {
			// Process file as input
			fp = fopen(Iname = argv[i], "rvq");
			Itype = 1;
			while(fgets(Iptr = Ibase = Buffer, sizeof(Buffer)-1, fp)) {
				++Iline;
				doline(L1); }
			fclose(fp);
			Iline = 0;
			continue; }
		Itype = f = 0;
		Iptr = Ibase = Ptr;
		if(doline(L1))
			continue;
		result(); }

	if(!f)
		return;

	for(i=0; i < SYtop; ++i)	// Clear changed flags
		SymFlags[i] &= ~SYM_CHANGE;

	setjmp(Ljmp);
	for(;;) {
		reset();
		fp = stdin;
		printf("%u> ", Rsp);
		fgets(Iptr = Ibase = Buffer, sizeof(Buffer)-1, stdin);
		if(doline(L1))
			continue;
		longcpy(Rstack[Rsp++ & (RECALL-1)], L1);
		result(); }
}
