/*
 * Replace symbols in file
 *
 * DEF file:
 *	; and null lines ignored
 *	symbol	replace-text
 *				$0	= insert original symbol name
 *				$1+	= insert function argument
 *				$$	= Insert '$'
 *				$	= Insert space ($ followed by space)
 *				$n	= Insert newline
 *				$t	= Insert tab
 *				$;	= Insert nothing (end $1+ if needed)
 * If $1+ exists, function arguments are parsed.
 * You may have differing definitions for the same symbol if they use
 * a different number of arguments (determined by max $1+ value)
 */
#include <stdio.h>

#define	NAMES	500

unsigned
	Line,				// Input line number
	Seg,				// External segment
	Stop,				// Top of external segment
	Ntop,				// Top of listed names
	Atop,				// Top of argument list
	Ptop,				// Top of memory pool
	Rcount,				// Count of replacements
	Names[NAMES],		// Listed names
	Alist[128];			// Argument offsets
FILE
	*fpi,				// Input file
	*fpo;				// Output file
unsigned char
	*Ptr,				// General pointer
	*Ifile,				// Input file
	*Ofile,				// Output file
	*Dfile="SUBSYM.DEF",// Definitions files
	Trim,				// Trim arguments
	Inline[1024],		// Input buffer
	Temp[256],			// Temp buffer
	Args[NAMES],		// Argument counts
	Pool[32768];		// Memory pool

// Report error and terminate
register error(unsigned args)
{
	int c;
	unsigned char *p;
	_format_(nargs() * 2 + &args, Pool);
	if(Line)
		printf("%u: ", Line);
	fputs(Pool, stdout);
	putc('\n', stdout);
	p = Inline;
	while(c = *p++)
		putc((c >= ' ') ? c : ' ', stdout);
	putc('\n', stdout);
	p = Inline;
	while(++p < Ptr)
		putc(' ', stdout);
	fputs("^\n", stdout);
	exit(-1);
}

// Skip to non-blank
int skip(void)
{
	while(isspace(*Ptr))
		++Ptr;
	return *Ptr;
}

void expect(int c)
{
	if(skip() != c)
		error("'%c' expected", c);
	++Ptr;
}

// Test for valid symbol character
int issymbol(int c)
{
	return(	((c >= 'a') && (c <= 'z'))
		||	((c >= 'A') && (c <= 'Z'))
		||	(c == '_') );
}

// Parse symbol from input line
unsigned parse_symbol(unsigned char *dest)
{
	unsigned l;
	l = 0;
	while(issymbol(*Ptr) || isdigit(*Ptr))
		dest[l++] = *Ptr++;
	dest[l] = 0;
	return l;
}

void outmem() { error("Segment overflow"); }
// Write character to external segment
void segw(b) asm
{
	MOV		AX,4[BP]			// Get data
	MOV		ES,DGRP:_Seg		// Get segment
	MOV		BX,DGRP:_Stop		// Top of segment
	MOV		ES:[BX],AL			// Write to segment
	INC		BX					// Advance
	JZ		_outmem				// Report out of memory
	MOV		DGRP:_Stop,BX		// Resave
}

// Get character from external segment
int segr(offset) asm
{
	MOV		ES,DGRP:_Seg		// Get segment
	MOV		BX,4[BP]			// Get offset
	XOR		AH,AH				// Zero high
	MOV		AL,ES:[BX]			// Get data
}

// Write string to external segment
void segs(unsigned char *s)
{
	do {
		segw(*s); }
	while(*s++);
}

// Compare string with external segment
// On match, returns following external address
unsigned segc(unsigned offset, unsigned char *s)
{
	do {
		if(segr(offset++) != *s)
			return 0; }
	while(*s++);
	return offset;
}

// Parse arguments to function
void parse_args(void)
{
	int c, b1, b2;
	unsigned char q;

	Ptop = Atop = b1 = b2 = q = 0;
	if(skip() != '(')
		error("expected '('");
	++Ptr;
again:
	if(Trim)
		skip();
	Alist[Atop] = Ptop;
	for(;;) {
		switch(c = *Ptr++) {
		case '\\' : 	// Escape
			Pool[Ptop++] = c;
			if(c = *Ptr++)	// Not EOL
				break;
		case 0 :		// End of line
			Pool[Ptop++] = '\n';
			if(!fgets(Ptr = Inline, sizeof(Inline)-1, fpi))
				error("EOF in arg parse");
			++Line;
			continue;
		case '[' :	++b2;	break;
		case ']' :	--b2;	break;
		case '(' :	++b1;	break;
		case ')' :		// Could be nested brace or end of arguments
			if(b1) {
				--b1;
				break; }
		case ',' :		// Another argument?
			if(q || b1 || b2)		// Nested in something
				break;
			if(Trim) {
				while((Ptop > Alist[Atop]) && isspace(Pool[Ptop-1]))
					--Ptop; }
			Pool[Ptop++] = 0;
			++Atop;
			if(c == ')')
				return;
			goto again;
		case '\'': 		// Single quote
		case '"' :		// Double quote
			if(c == q)
				q = 0;
			else if(!q)
				q = c; }
		Pool[Ptop++] = c; }
}

/* Open a file with optional extension
FILE *openf(unsigned char *fn, unsigned char *ext, unsigned char *opt)
{
	unsigned char d, *p;
	p = Temp;
	d = 0;
	for(;;) switch(*p++ = *fn++) {
		case '.' : d = 255;			continue;
		case ':' :
		case '\\': d = 0;			continue;
		case 0 :
			if(!d)
				strcpy(p-1, ext);
			return fopen(Temp, opt); }
} */

static unsigned char Help[] = { "\n\Use: SUBSYM <infile> [outfile] [options]\n\n\
opts:	D=filename	= set Definition file		[SUBSUM.DEF]\n\
	-T		= Trim function arguments\n\
\nDave Dunfield - "#__DATE__"\n" };

main(int argc, char *argv[])
{
	int c;
	unsigned i, j, m;
	unsigned char f;

	for(i=1; i < argc; ++i) {
		Ptr = argv[i];
		switch((toupper(*Ptr++) << 8) | toupper(*Ptr++)) {
		case '-T' :
		case '/T' : Trim = 255;		continue;
		case 'D=' :	Dfile = Ptr;	continue;
		} Ptr -= 2;
		if(!Ifile)
			Ifile = Ptr;
		else if(!Ofile)
			Ofile = Ptr;
		else
			goto help; }

	if(!Ifile) { help:
		abort(Help); }

	Seg = alloc_seg(4096);

	// Read definitions file into memory
	fpi = fopen(Dfile, "rvq");
	while(fgets(Ptr = Inline, sizeof(Inline)-1, fpi)) {
		++Line;
		switch(skip()) {
		case ';' :				// Comment
		case 0 :				// Blank line
			continue; }
		if(!issymbol(*Ptr))
			error("Bad symbol");
		parse_symbol(Temp);
		if(Ntop >= NAMES)
			error("Too many symbols");
		Names[Ntop] = Stop;
		segs(Temp);
		skip();
		m = 0;
		while(c = *Ptr++) {
			f = 255;
			if(c == '$') {			// Special '$' command
				switch(c = *Ptr++) {
				case ' ' :
				case '$':				goto sw;
				case 't' :	c = '\t';	goto sw;
				case 'n' :	c = '\n';	goto sw;
				case 0 :	// Another line
					if(!fgets(Ptr = Inline, sizeof(Inline)-1, fpi))
						error("No continuation line");
					++Line;
					skip();
				case ';' :
					continue;
				case '#' :
					f = 0;
					c = *Ptr++; }
				if(!isdigit(c))
					error("Unknown \escape");
				i = c - '0';
				while(isdigit(*Ptr))
					i = (i * 10) + (*Ptr++ - '0');
				c = i | 0x80;
				if(i > m)
					m = i; }
	sw:		if(f)
				segw(c); }
		for(i=0; i < Ntop; ++i) {
			if(segc(Names[i], Temp) && (Args[i] == m))
				error("Duplicate: %s(%u arguments)", Temp, m); }
		Args[Ntop++] = m;
		segw(0); }		// End of definition
	fclose(fpi);

#if 1
	fpo = fopen("R:B", "wvqb");
	for(i=0; i < Stop; ++i)
		putc(segr(i), fpo);
	fclose(fpo);
#endif

	Line = 0;
	fpi = fopen(Ifile, "rvq");
	fpo = (Ofile) ? fopen(Ofile, "wvq") : stdout;
	while(fgets(Ptr = Inline, sizeof(Inline)-1, fpi)) {
		++Line;
		while(c = *Ptr) {
			if(!issymbol(c)) {
				putc(*Ptr++, fpo);
				continue; }
			parse_symbol(Temp);
			i = f = 0;
	lknext:	while(i < Ntop) {
				if(j = segc(Names[i], Temp))
					goto found;
				++i; }
			fputs(Temp, fpo);
			if(f) {		// Arguments were parsed
				putc('(', fpo);
				for(i=0; i < Atop;) {
					fputs(Alist[i]+Pool, fpo);
					putc((++i == Atop) ? ')' : ',', fpo); } }
			continue;
	found:	if(Args[i]) {		// Arguments expected
				if(!f)
					parse_args();
				f = 255;
				if(Args[i] != Atop) {
					++i;
					goto lknext; } }
			// Inject replacement text
			++Rcount;
			while(c = segr(j++)) {
				if(c & 0x80) {
					if(!(c &= 0x7F))
						fputs(Temp, fpo);
					else
						fputs(Alist[c-1]+Pool, fpo);
					continue; }
				putc(c, fpo); } }
		putc('\n', fpo); }
	fclose(fpo);
	fclose(fpi);
	if(Ofile)
		printf("%s->%s : %u replacements\n", Ifile, Ofile, Rcount);
}
