#include <stdio.h>
#include <file.h>
#include <window.h>

#define	SSIZE		24			// Screen size (#entries)
#define	A_BROWSE	0x17		// Viewing file
#define	A_HILITE	0x71		// Control-Character display
#define	A_STATUS	0x70		// Status line

unsigned
	Base,					// Base offset
	Cursor,					// Cursor position
	Tab = 7,				// TAB size
	Handle,					// File handle
	Bptr,					// Block pointer
	Bsize,					// Block size
	Sh, Sl,					// High/Low size
	Ph, Pl,					// High/Low position
	Line,					// Current line
	Lines,					// # lines in file
	Llist[2048][2];			// Line offset list
unsigned char
	*File,					// File to view
	Sflag,					// Status display flag
	Buffer[1024];			// I/O buffer

unsigned char Help[] = { "\n\
File Viewer - Dave Dunfield - "#__DATE__"\n\n\
use: VIEW filename\n" };

/*
 * Display status message
 */
register status(unsigned args)
{
	unsigned char a, buf[81];

	_format_(nargs() * 2 + &args, buf);
	wgotoxy(Sflag=0, 24);
	a = *W_OPEN;
	*W_OPEN = A_STATUS;
	wcleol();
	wputs(buf);
	*W_OPEN = a;
}

/*
 * Get byte from file
 */
int Fgetc(void)
{
	unsigned char c;

	if(!Bptr)
		Bsize = read(Buffer, sizeof(Buffer), Handle);
	if(Bptr >= Bsize)
		return -1;
	c = Buffer[Bptr++];
	if(!++Pl)
		++Ph;
	if(Bptr >= sizeof(Buffer))
		Bptr = 0;
	return c;
}

/*
 * Get string from input file
 */
int Fgets(unsigned char *dest, unsigned l)
{
	int c;
	unsigned char e;

	e = 0;
	while((c = Fgetc()) != -1) {
		e = 255;
		if(c == '\n')
			break;
		if(c != '\r') {
			*dest++ = c;
			if(!--l)
				break; } }
	*dest = 0;
	return e;
}
	
/*
 * Goto a particular line
 */
void goline(unsigned l)
{
	int c;
	unsigned b;

	if(l > Lines)
		Line = l = Lines;
	b = l >> 5;
	l &= 0x1F;
	lseek(Handle, Ph=Llist[b][1], Pl=Llist[b][0], Bptr=0);
	while(l--) {
		while((c = Fgetc()) != '\n') {
			if(c == -1)
				return; } }
}

/*
 * Display character in window
 */
void Putc(int c)
{
	if((Base <= Cursor) && ((Base+80) > Cursor))
		wputc(c);
	++Cursor;
}

/*
 * 32-bit compare of two pairs of unsigned
 */
int Lcmp(unsigned h1, unsigned l1, unsigned h2, unsigned l2)
{
	if(h1 < h2) return -1;
	if(h1 > h2) return 1;
	if(l1 < l2) return -1;
	if(l1 > l2) return 1;
	return 0;
}

/*
 * File viewer
 */
int main(int argc, char *argv[])
{
	int c;
	unsigned h, i, j, k, l, m, n;
	unsigned char f, buf[16];

	if(argc < 2)
		abort(Help);

	if(!(Handle = open(File = argv[1], F_READ|F_BINARY))) {
		printf("Unable to open: %s\n", File);
		return; }

	wopen(0, 0, 80, SSIZE+1, WSAVE|WCOPEN|A_BROWSE);

	*W_OPEN = A_BROWSE;
	wclwin();

	Base = Line = Lines = Bsize = Bptr = Sh = Sl = h = l = m = n = 0;

	// Build line index
	status("Indexing '%s'...", File);
	while(i = read(Buffer, sizeof(Buffer), Handle)) {
		for(j=0; j < i; ++j) {
			if(!++Sl)
				++Sh;
			switch(c = Buffer[j]) {
			default:
				if((c < ' ') || (c > 0x7E)) {
					if(!++n)
						++m; }
			case '\t' :
			case '\r' : continue;
			case '\n' :
				++Lines;
				if(!(Lines & 0x1F)) {	// 64-line boundary
					k = Lines >> 5;
					if(k >= 2048) {
						close(Handle);
						wclose();
						abort("File exceeds 64k lines\n"); }
					Llist[k][0] = Sl;
					Llist[k][1] = Sh; } } } }

	wcursor_off();

	// If 1/8 or more of file is unprintable, assume HEX
	i = Sh & 7;
	j = (Sl >> 3) | (i << 13);
	i = Sh >> 3;
	if(Lcmp(m, n, i, j) >= 0)
		goto rehex;

	// ASCII viewer
retext:
	if(Base & 0x8000)
		Base = 0;
	if((Line+SSIZE) > Lines)
		Line = Lines-(SSIZE-1);
	if((Line & 0x8000) || (Lines < (SSIZE-1)))
		Line = 0;
	goline(Line);
	h = Ph;
	l = Pl;
	for(i=f=0; i < SSIZE; ++i) {
		wgotoxy(Cursor=0, i);
		if(!f) {
			while((c = Fgetc()) != '\n') {
				switch(c) {
				case -1:
					*W_OPEN = A_HILITE;
					Putc('E'); Putc('O'); Putc('F');
					*W_OPEN = A_BROWSE;
					f = 255;
					goto clr;
				case '\t' :
					do {
						Putc(' '); }
					while(Cursor & Tab);
				case '\r' : continue; }
				if(c < ' ') {
					*W_OPEN = A_HILITE;
					c += '@'; }
				Putc(c);
				*W_OPEN = A_BROWSE; } }
clr:	if((Cursor-Base) < 80)
			wcleol(); }

	status("%-12s  Lines %u-%u of %u, Columns %u-%u, Tab %u",
		File, Line+1, Line+24, Lines, Base+1, Base+80, Tab);
	for(;;) {
		switch(wgetc()) {
		default: status("\
\x18\x19:\xF1Line Pg\x18\x19:\xf1Page \x1B\x1A:\xF1Col Home:Col1 ^Home:Top ^End:Bottom Tab F1:Mode F10:Exit");
			continue;
		case _KUA:	--Line;			goto retext;
		case _KDA:	++Line;			goto retext;
		case _KPU:	Line -= 24;		goto retext;
		case _KPD:	Line += 24;		goto retext;
		case _CHO:	Line = 0;		goto retext;
		case _CEN:	Line = 32767;	goto retext;
		case _KLA:	--Base;			goto retext;
		case _KRA:	++Base;			goto retext;
		case _KHO:	Base = 0;		goto retext;
		case '\t':
			if(!(Tab >>= 1))
				Tab = 7;
			goto retext;
		case _K1:					goto rehex;
		case _K10:
		case 0x1B:
quit:		close(Handle);
			wclose();
			return; } }

	// Hexidecimal viewer
rehex:
	if((Sl <= 384) && !Sh)
		h = l = 0;
	i=Sh; j = Sl - (23*16);
	if(j > Sl)
		--i;
	if(Lcmp(h, l, i, j) >= 0) {
		h = i;
		l = j; }
	lseek(Handle, Ph=h, Pl=l, Bptr=0);
	for(i=f=0; i < 24; ++i) {
		wgotoxy(0, i);
		m = Ph;
		n = Pl;
		for(j=k=0; j < 16; ++j) {
			if(!f) {
				if((c = Fgetc()) == -1)
					f=255;
				else
					buf[k++] = c; } }
		wcleol();
		if(k) {
			wprintf("%04x%04x ", m, n);
			for(j=0; j < 16; ++j) {
				if(!(j&3))
					wputc(' ');
				if(j < k)
					wprintf(" %02x", buf[j]);
				else
					wputs("   "); }
			wputs("   ");
			for(j=0; j < k; ++j) {
				c = buf[j];
				wputc( ((c >= ' ') && (c <= 0x7E)) ? c : '.'); } } }
	if(Ph|Pl) {
		i = Pl;
		if(--Pl > i)
			--Ph; }
	status("%-12s  %04x%04x-%04x%04x of %04x%04x", File, h, l, Ph, Pl, Sh, Sl);
	for(;;) {
		switch(wgetc()) {
		default: status("\
\x1B\x1a:\xF11 \x18\x19:\xF116 Pg\x18\x19:\xF1384 Home:%%16 ^Home:Top ^End:Bottom           F1:Mode F10:Exit");
			continue;
		case _KRA: i=1;	goto doadd;
		case _KLA: i=1;	goto dosub;
		case _KUA: i = 16; dosub:
			j = l;
			if((l -= i) > j)  {
				if(!h--)
					h=l=0; }
			goto rehex;
		case _KDA: i = 16; doadd:
			j = l;
			if((l += i) < j)
				++h;
			goto rehex;
		case _KPU: i = (24*16);		goto dosub;
		case _KPD: i = (24*16);		goto doadd;
		case _KHO: l &= 0xFFF0;		goto rehex;
		case _CHO: h=l=0;			goto rehex;
		case _CEN: h=Sh; l=Sl;		goto rehex;
		case _K1 :					goto retext;
		case _K10:
		case 0x1B:					goto quit; } }
}
