#include <stdio.h>
#include <window.h>
#include <file.h>

#define	NAME	XC
#define	FILES	4		// Number of files supported
#define	LMASK	32		// Line record mask (must be power of two)
#define	LSIZE	4096	// Maximum size of input line

#define	SWHITE	0x08	// Group whitespace
#define	STAIL	0x04	// Trim trailing blanks
#define	SLEAD	0x02	// Trim leading blanks
#define	SCASE	0x01	// Case insensitive

unsigned
	Tab,				// Tab spacing
	Lseg,				// Line index segment
	Ltop,				// Line index top marker
	File,				// Current file indicator
	Files,				// # of open files
	Wheight,			// Calculated height of window
	Wwidth,				// Calculated width of window
	Horz[FILES],		// Horizontal offset
	Hmax[FILES],		// Horizontal maximum
	Fstart[FILES],		// Offset to first line indicator
	Fline[FILES],		// Current line in file
	Flines[FILES];		// # files in each file

FILE
	*Fp[FILES];			// Open file pointers

unsigned char
	Scan,				// Scan parameters
	Sscan,				// Search parameters
	Dscan,				// Difference parameters
	Mode,				// Display mode
	search_string[51],	// String to search
	*Emsg,				// Error message
	*Names[FILES],		// File names
	*Smsg[FILES],		// Status line messages
	buffer[LSIZE];		// General buffer

struct WINDOW
	*status,			// Status window
	*Win[FILES];		// File view windows


char *search_form[] = {		// Form for string search entry
	64<<8|3,
	"\x00\x00\x32Search for:",
	0 };

unsigned char Wattr[] = {	// Video attributes
	0x70, 0x17, 0x47, 0x17, 0x47 };

/*
 * Index file & record line number positions
 */
void index(unsigned char *file)
{
	unsigned l, oh, ol;
	int c;
	FILE *fp;

	if(Files >= FILES)
		abort("Too many files");
	printf("Indexing: %s... ", Names[Files] = file);
	Fp[Files] = fp = fopen(file, "rvqb");
	Fstart[Files] = Ltop;

	l = oh = ol = 0;
	/* Build table of line numbers .vs. file offsets */
	while((c = getc(fp)) != EOF) {
		if(!++ol)
			++oh;
		if(c == '\n') {
			if(!(++l & (LMASK-1))) {
				pokew(Lseg, Ltop, ol);
				pokew(Lseg, Ltop+2, oh);
				Ltop += 4; } } }

	Flines[Files++] = l;
	printf(" Done\n");
}

/*
 * Close open data windows
 */
void closewin(void)
{
	unsigned i;
	for(i=0; i < Files; ++i)
		w_close(Win[i]);
}

/*
 * Position file to line
 */
int gotoline(unsigned f, unsigned l)
{
	int c;
	unsigned o;
	FILE *fp;

	if(l >= Flines[f])		// Too big
		return -1;

	fp = Fp[f];
	if(o = l / LMASK) {		// Perform seek
		o = ((o-1) * 4) + Fstart[f];
		fseek(fp, peekw(Lseg, o+2), peekw(Lseg, o), 0); }
	else					// First LMASK lines
		rewind(fp);

	o = l & (LMASK-1);
	while(o) {
		if((c = getc(fp)) == EOF)
			return -1;
		if(c == '\n')
			--o; }
	return 0;
}

/*
 * Read a buffer with input translations (for difference search)
 */
int getbuf(unsigned char *b, unsigned f)
{
	unsigned char c, *p, *p1;

	if(!fgets(p = p1 = b, LSIZE, Fp[f]))
		return 255;

	if(*p == '\r')
		++p;
	if(Scan & SLEAD) {
		while(isspace(*p))
			++p; }
	while(c = *p++) {
		if(c == '\r')
			continue;
		if(isspace(c)) {		// White space
			if(Scan & SWHITE) {
				while(isspace(*p))
					++p;
				c = ' '; } }
		else if(islower(c)) {	// Lower case
			if(Scan & SCASE)
				c &= 0xDF; }
		*p1++ = c; }
	if(Scan & STAIL) {
		while((p1 > b) && isspace(*(p1-1)))
			--p1; }
	return *p1 = 0;
}

/*
 * Update file display
 */
void draw_file(unsigned f)
{
	int c;
	unsigned i, j, h, hw;
	unsigned char *p;
	FILE *fp;

	W_OPEN = Win[f];
	fp = Fp[f];

	if(gotoline(f, Fline[f])) {
		wclwin();
		wputs("*EOF*");
		return; }

	hw = Horz[f] + Wwidth;
	Hmax[f] = 0;
	for(i=0; i < Wheight; ++i) {
		wgotoxy(h=0, i);
		p = buffer;
rdnxt:	switch(c = getc(fp)) {
		case EOF:
			strcpy(p, "*EOF*");
			wcleow();
			i = Wheight;
			break;
		case '\t' :
			do {
				*p++ = ' '; }
			while(++h % Tab);
		case '\r' :
			goto rdnxt;
		default:
			*p++ = (c < ' ') ? 0xA8 : c;
			++h;
			goto rdnxt;
		case '\n' :
			*p = 0; }
		wcleol();
		p = buffer;
		if(h > Hmax[f])
			Hmax[f] = h;
		if(j = Horz[f]) {
			while(j-- && *p)
				++p; }
		h = 0;
		while(c = *p++) {
			if(h >= Wwidth)
				break;
			wputc(c);
			++h; } }
}

/*
 * Update status line
 */
void showstat(void)
{
	unsigned i, w;
	W_OPEN = status;
	wclwin();
	if(Emsg) {
		wputs(Emsg);
		Emsg = 0;
		return; }
	w = 80 / Files;
	for(i=0; i < Files; ++i) {
		wgotoxy(i*w, 0);
		if(i == File)
			*W_OPEN = 0x07;
		if(Smsg[i]) {
			wputs(Smsg[i]);
			Smsg[i] = 0; }
		else {
			wprintf("%s:%u", Names[i], Fline[i]+1);
			if(Horz[i])
				wprintf("-%u", Horz[i]+1); }
		*W_OPEN = *Wattr; }
}

/*
 * Test for string occuring within a line.
 * We could easily do this in 'C', however its a good chance to show
 * off inline assembly language, and get a slight speed improvement.
 */
inline(line, string) asm
{
		MOV		SI,6[BP]		; Get line
inl1:	MOV		DI,4[BP]		; Get string
inl2:	MOV		AL,[DI]			; Get char from string
		MOV		AH,[SI]			; Get char from line
		AND		AL,AL			; End of string?
		JZ		inl3			; Yes, we have match
		INC		SI				; Advance line
		INC		DI				; Advance string
		CMP		AL,AH			; *line == *string?
		JZ		inl2			; Yes, keep looking
		AND		AH,AH			; End of line?
		JNZ		inl1			; No, keep trying
; End of line... string was not found
		XOR		AX,AX			; 0 = Not found
		JMP SHORT inl4			; And exit
; Found string
inl3:	MOV		AX,-1			; 1 = Success
inl4:	POP		BP				; Restore
		RET
}

/*
 * Search files for string
 */
void search(unsigned f, unsigned l)
{

	Scan = Sscan;
	gotoline(f, l);
	while(!getbuf(buffer, f)) {
		if(inline(buffer, search_string)) {
			Fline[f] = l;
			return; }
		++l; }
	Smsg[f] = "Not found";
}

/* xout(unsigned char *p)
{
	int c;
	W_OPEN = status;
	wclwin();
	while(c = *p++) {
		if((c < ' ') || (c > 0x7E))
			c = '?';
		wputc(c); }
	wputc('|');
	wgetc();
} */

/*
 * Scan forward for differences
 */
void scan(void)
{
	unsigned i;
	unsigned char f, b[LSIZE];

	Scan = Dscan;
	// Reset all file positions
	for(i=f=0; i < Files; ++i) {
		if(i != File) {
			++f;
			gotoline(i, Fline[i]); } }

	if(f < 2) {
		Emsg = "Must scan at least two files";
		return; }

	do {
		for(i=f=0; i < Files; ++i) {
			if(i != File) {
				++Fline[i];
				if(!f) {
					f = 15;
					if(getbuf(buffer, i))
						f = 255;
					continue; }
				if(getbuf(b, i)) {
					f = 255;
					continue; }
				if(strcmp(buffer, b))
					f = 255; } } }
	while(!(f & 0xF0));
//	xout(buffer);
//	xout(b);
	for(i=0; i < Files; ++i) {
		if(i != File)
			--Fline[i]; }
}

/*
 * Perform an operation over all open files
 */
int fileop(unsigned k)
{
	unsigned i;
	for(i=0; i < Files; ++i) {
		if((File == i) || (File >= Files)) switch(k) {
			default:
				Emsg = "Unknown command - use '?' for help.";
				return 0;
			case _KRA:
				++Horz[i];
				continue;
			case _KLA:
				if(Horz[i])
					--Horz[i];
				continue;
			case _KUA:
				if(Fline[i])
					--Fline[i];
				continue;
			case _KDA:
				++Fline[i];
				continue;
			case _KPU:
				Fline[i] = (Fline[i] >= Wheight) ? Fline[i] - Wheight : 0;
				continue;
			case _KPD:
				Fline[i] += Wheight;
				continue;
			case _KEN :
				if(Hmax[i] > Wwidth) {
					Horz[i] = Hmax[i] - Wwidth;
					continue; }
			case _KHO :
				Horz[i] = 0;
				continue;
			case _CPU:
				Fline[i] = 0;
				continue;
			case _CPD:
				Fline[i] = Flines[i];
				continue;
			case _K3:
				search(i, Fline[i]);
				continue;
			case _K4:
				search(i, Fline[i]+1);
	} }
	return 255;
}

static char help[] = { "\neXpress Compare - Dave Dunfield - "#__DATE__"\n\n\
Use:	"#NAME"	<file>... [opts]\n\n\
Opts:	/DC	- ignore letter Case when scanning for differences\n\
	/DL	- ignore Leading  whitespace when scanning\n\
	/DT	- ignore Trailing whitespace when scanning\n\
	/DW	- treat all Whitespace as single space when scanning\n\
	/SC	- ignore letter Case when searching\n\
	/SL	- trim Leading  whitespace when searching\n\
	/ST	- trim Trailing whitespace when searching\n\
	/SW	- treat all Whitespace as single space when searching\n\
	T=n	- display Tab size		[4]\n\
\n?COPY.TXT 2008 Dave Dunfield\n -- see COPY.TXT --.\n" };

static char keys[] = { "\
"#NAME" - ?COPY.TXT 2008 Dave Dunfield -  -- see COPY.TXT --.\n\n\
	TAB		-	Select file\n\
   Up/Down	-*	Back/Fwd line\n\
 PgUp/PgDn	-*	Back/Fwd page\n\
 Left/Right	-*	Shift display left/right (within line)\n\
   Home		-*	Move to start of line\n\
	End		-*	Display rightmost portion of lines on screen\n\
^PgUp/^PgDn	-*	Move to Start/End of file\n\
	F1		-	Toggle display mode (Row or Column)\n\
	F2		-$	Scan for difference (from current position)\n\
	F3		-*	Search for string\n\
	F4		-*	Repeat last search\n\
	F5		-	Adjust scan parameters\n\
	F6		-	Adjust search parameters\n\
	ESC		-	Exit\n\
\n\
*	Command keys apply to hilighted file - if no filename is hilighted,\n\
	command applies to all files.\n\n\
$	Hilighted file is ignored during F2 scan.\n\
\nPress any key to proceed." };

scanparm(unsigned char *p, unsigned char *d)
{
	unsigned char c, v;
	while(c = *p++) {
		switch(toupper(c)) {
			default: abort(help);
			case 'L' : v = SLEAD;	break;
			case 'T' : v = STAIL;	break;
			case 'W' : v = SWHITE;	break;
			case 'C' : v = SCASE; }
		*d |= v; }
}

main(int argc, char *argv[])
{
	int c;
	unsigned i;
	unsigned char *p;

	if(!(Lseg = alloc_seg(4096)))
		abort("No memory");

	IOB_size = 2048;
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case '-D' :
		case '/D' : scanparm(p, &Dscan);	continue;
		case '-S' :
		case '/S' : scanparm(p, &Sscan);	continue;
		case 'T=' :
			Tab = atoi(p);
			continue; }
		index(p-2); }

	if(!Files)
		abort(help);

	if(!Tab)
		Tab = 4;

	status = wopen(0, 0, 80, 1, WSAVE|WCOPEN|Wattr[0]);
	wcursor_off();
	File = Files;
remode:
	if(!Mode) {
		Wheight = 24;
		Wwidth = 80 / Files;
		for(i=0; i < Files; ++i)
			Win[i] = wopen(i * Wwidth, 1, Wwidth, Wheight, WSAVE|WCOPEN|Wattr[i+1]); }
	else {
		Wheight = 24 / Files;
		Wwidth = 80;
		for(i=0; i < Files; ++i)
			Win[i] = wopen(0, (i*Wheight)+1, Wwidth, Wheight, WSAVE|WCOPEN|Wattr[i+1]); }
redraw:
	for(i=0; i < Files; ++i) {
		if(Fline[i] > Flines[i])
			Fline[i] = Flines[i];
		draw_file(i); }

stat:
	showstat();
	for(;;) switch(i=wgetc()) {
	case _K1:
		closewin();
		Mode = Mode ? 0 : 255;
		goto remode;
	case _K2: scan();	goto redraw;
	case _K3:
		if(wform(7, 10, WSAVE|WCOPEN|WBOX1|REVERSE, search_form, search_string))
			continue;
		if(Sscan & SCASE)
			strupr(search_string);
	default:
		if(fileop(i))
			goto redraw;
	case ' ' :
	case '\n':
		goto stat;
	case _K5: setparms(&Dscan, "Scan parameters");		goto stat;
	case _K6: setparms(&Sscan, "Search parameters");	goto stat;
	case '\t':
		if(++File > Files)
			File = 0;
		goto stat;
	case '?' :
		closewin();
		wopen(i=0, 1, 80, 24, WSAVE|WCOPEN|NORMAL);
		p = keys;
		while(*p) switch(c=*p++) {
			case '\t' :
				do {
					wputc(' '); }
				while(++i & 3);
				continue;
			case '\n' : i = -1;
			default: wputc(c);
				++i; }
		wgetc();
		wclose();
		goto remode;
	case 0x1B:
		closewin();
		w_close(status);
		for(i=0; i < Files; ++i)
			fclose(Fp[i]);
		return; }
}

setparms(unsigned char *d, unsigned char *m)
{
	unsigned i;
	unsigned char v;
	static unsigned char *n[] = {
		"C)ase insensitive",
		"L)eading whitespace",
		"T)railing whitespace",
		"all W)hitespace",
		0 };
	W_OPEN = status;
	wclwin();
	wputs(m);
	wputs(" - Enter to save, ESC to discard");
	wopen(25, 10, 30, 6, WSAVE|WBOX1|Wattr[0]);
	v = *d;
redraw:
	wclwin();
	for(i=0; n[i]; ++i) {
		wgotoxy(0, i);
		wprintf("%-26s%c", n[i], ((1<<i) & v) ? 'Y' : 'N'); }
	for(;;) switch(toupper(wgetc())) {
		case 'C' : v ^= SCASE;	goto redraw;
		case 'L' : v ^= SLEAD;	goto redraw;
		case 'T' : v ^= STAIL;	goto redraw;
		case 'W' : v ^= SWHITE;	goto redraw;
		case '\n':
			*d = v;
		case 0x1B:
			wclose();
			return; }
}
