/*
 * Skeleton for directory walking function
 */
#include <stdio.h>
#include <file.h>

// Output flags - dependant on application
#define	O_DIR		0x0001		// Directory display
#define	O_STAT		0x0002		// Memory statistics
#define	O_MATCH		0x0004		// Matching input files
#define	O_OK		0x0008		// Match OK
#define	O_FAIL		0x0010		// Match fail
#define	O_NF		0x0020		// Not found
#define	O_N1		0x0040		// Not found once


static unsigned char Oflags[] = { "DSIMFN1" };

unsigned
	Sh,					// Size-High
	Sl,					// Size-Low
	A,					// Attributes
	T,					// Time
	D,					// Date
	Ptop,				// Top of pool
	Ftop,				// Top of filename string
	Phigh,				// Pool high-water mark
	Fhigh,				// Filename high-water mark
	Output = O_FAIL|O_N1,
	Lines = 24;			// Lines/page
unsigned char
	*ptr,				// Global pointer
	*Directory,			// Starting directory
	Pool[16384],		// String pool
	Filename[128],		// Filename string
	Name[13];			// Filename

register Printf(unsigned args);

/* ----- Begin user application ----- */

#define	ATTRS	DIRECTORY		// HIDDEN|SYSTEM|DIRECTORY

unsigned
	Seg,						// Segment
	Stop,						// Segment top
	Icount,						// Count of located files
	Ecount,						// Exact match count
	Fcount,						// Failed match count
	Tcount;						// Total files processed
unsigned char
	*Pattern,					// Pattern to lookup
	Exact,						// Exact matches
	Size = 255,					// Match size
	Time = 255;					// Match time

unsigned char Help[] = { "\n\
eXpress Treesearch - Dave Dunfield - "#__DATE__"\n\n\
Use: XT <input-pattern> [tree-path] [options]\n\n\
opts:	/E		- only count Exact matches\n\
	/S		- do not compare Size\n\
	/T		- do not compare Timestamp\n\
	L=lines		- set # Lines in output		[24]\n\
	O=[+-]DIMFN1S	- set Output mask		[F1]\n\
		D = Directory being scanned\n\
		S = Statistics\n\
		I = # Input files\n\
		M = Matching files\n\
		F = Failed (mismatch) files\n\
		N = files Not in input\n\
		1 = files not occuring once in tree-path\n\
" };

void init_app(void)
{
	unsigned char *n;

	if(!Pattern)
		abort(Help);

	if(!(Seg = alloc_seg(4096)))
		abort("Out of memory");
	A = 0;
	do {
		pokew(Seg, A, 0); }
	while(A -= 2);

	if(find_first(Pattern, ATTRS, Name, &Sh, &Sl, &A, &T, &D))
		abort("No files found");

	do {
		if(A & DIRECTORY)
			continue;
		++Icount;
		A = Stop;
		n = Name;
		do {
			poke(Seg, Stop++, *n); }
		while(*n++);
		pokew(Seg, Stop, Sh);
		pokew(Seg, Stop+2, Sl);
		pokew(Seg, Stop+4, T);
		pokew(Seg, Stop+6, D);
		pokew(Seg, Stop+8, 0);
		Stop += 10;
		if(Stop < A)
			abort("Ext seg. exhausted"); }
	while(!find_next(Name, &Sh, &Sl, &A, &T, &D));
	Printf(O_MATCH, "Input files=%u", Icount);
	if(!Icount)
		exit(0);
}

extern char Longreg[];
unsigned char *disp10(unsigned high, unsigned low)
{
	unsigned l[2];
	unsigned char *p;
	static unsigned L10[] = { 10, 0 };
	static unsigned char x, buffer[2][12];

	l[1] = high;
	l[0] = low;
	p = buffer[++x & 1] + 12;
	*--p = 0;
	do {
		longdiv(l, L10);
		*--p = *Longreg + '0'; }
	while(longtst(l));
	return p;
}

void process_file(void)
{
	unsigned sh, sl, t, d;
	unsigned c, ct, sp;
	unsigned char name[13], *n, f;

	++Tcount;
	for(c = sp = 0; c < Icount; ++c) {
		n = name;
		while(*n++ = peek(Seg, sp++));
		if(strcmp(Name, name)) {
			sp += 10;
			continue; }
		sh = peekw(Seg, sp);
		sl = peekw(Seg, sp+2);
		t = peekw(Seg, sp+4);
		d = peekw(Seg, sp+6);
		ct = peekw(Seg, sp+8);
		sp += 10;
		if(!Exact)
			pokew(Seg, sp-2, ct+1);
		f = 0;
		if(Size) {
			if((Sh != sh) || (Sl != sl)) {
//				Printf(O_FAIL, "%s = Size mismatch %x%04x %x%04x", Filename, Sh,Sl,sh,sl);
				Printf(O_FAIL, "%s = Size mismatch %s %s", Filename, disp10(Sh,Sl), disp10(sh,sl));
				f = 255; } }
		if(Time) {
			if((T != t) || (D != d)) {
				Printf(O_FAIL, "%s = Time mismatch", Filename);
				f = 255; } }
		if(f)
			++Fcount;
		else {
			++Ecount;
			if(Exact)
				pokew(Seg, sp-2, ct+1);
			Printf(O_OK, "%s = OK", Name); }
		return; }
	Printf(O_NF, "%s = Not found", Name);
}

void end_app()
{
	unsigned sh, sl, t, d;
	unsigned c, ct, sp;
	unsigned char name[13], *n;

	for(c = sp = 0; c < Icount; ++c) {
		n = name;
		while(*n++ = peek(Seg, sp++));
		sh = peekw(Seg, sp);
		sl = peekw(Seg, sp+2);
		t = peekw(Seg, sp+4);
		d = peekw(Seg, sp+6);
		ct = peekw(Seg, sp+8);
		sp += 10;
		if(ct != 1)
			Printf(O_N1, "%s occured %u times", name, ct); }

	Printf(O_STAT, "FILES: Input: %u Scanned:%u Matched:%u Failed:%u",
		Icount, Tcount, Ecount, Fcount);
}

/* ----- End user application  ----- */

register Printf(unsigned args)
{
	unsigned char buf[256];
	unsigned *ptr, m;
	static unsigned Lcount;
	static unsigned char clear[] = { "\r                  \r" };

	m = *(ptr = (nargs() - 1) * 2 + &args);
	if((m & Output) || !m) {
		_format_(ptr, buf);
		fputs(buf, stdout);
		putc('\n', stdout);
		if(Lines) if(++Lcount >= Lines) {
			fputs("<ENTER to proceed>", stderr);
			for(;;) switch(kbget()) {
			case 0x1B:
			case 0x03:
				fputs(clear, stderr);
				abort("<ABORTED>");
			case '\r' :
				fputs(clear, stderr);
				Lcount = 0;
				return; } } }
}

/*
 * Add a file to the string pool
 */
unsigned char *add_pool(unsigned char *n)
{
	unsigned p;
	p = Ptop;
	while(Pool[Ptop++] = *n++);
	if(Ptop > sizeof(Pool))
		abort("String pool exhausted");
	return Pool+p;
}

/*
 * Walk directory tree and process files
 * Argument of 0 (NULL) means current directory
 */
void do_dir(unsigned char *n)
{
	unsigned Dt, Pt, L, H;

	Dt = Ftop;
	Pt = Ptop;

	if(n) {
		while(Filename[Ftop++] = *n++);
		switch(Filename[Ftop-2]) {
		default: Filename[Ftop-1] = '\\';	break;
		case ':' :
		case '\\' : --Ftop; } }
	Filename[Ftop] = 0;
	Printf(O_DIR, "Dir: %s", Filename);
	strcpy(Filename+Ftop, "*.*");

	if(find_first(Filename, ATTRS, Name, &Sh, &Sl, &A, &T, &D)) {
		return; }

	do {
		if(A & DIRECTORY) {
			if(*Name == '.')
				continue;
			add_pool(Name);
			continue; }
		strcpy(Filename+Ftop, Name);
		process_file(); }
	while(!find_next(Name, &Sh, &Sl, &A, &T, &D));

	L = Pt;
	H = Ptop;
	while(L < H) {
		do_dir(Pool+L);
		while(Pool[L++]); }

	if(Ftop > Fhigh)
		Fhigh = Ftop;
	if(Ptop > Phigh)
		Phigh = Ptop;

	Ftop = Dt;
	Ptop = Pt;
}

unsigned option(unsigned Flag, unsigned char *Text)
{
	unsigned i, v;
	unsigned char c, m;
	m = 0;
	if(!*ptr)
		Flag = 0;
	while(c = toupper(*ptr++)) {
		switch(c) {
		case '+' : m = 1;	continue;
		case '-' : m = 2;	continue; }
		for(i=0; Text[i] != c; ++i) {
			if(!Text[i]) {
				printf("Bad option flag: '%c'\n", c);
				exit(-1); } }
		v = 1 << i;
		switch(m) {
		case 0 : Flag = v; m = 1;	continue;
		case 1 : Flag |= v;			continue;
		case 2 : Flag &= ~v; } }
	return Flag;
}

main(int argc, char *argv[])
{
	unsigned i;

	/* Argument parsing - modify as required */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
		case ('L'<<8)|'=' : Lines = atoi(ptr);					continue;
		case ('O'<<8)|'=' : Output = option(Output, Oflags);	continue;
		// Application qualifiers
		case ('/'<<8)|'E' :
		case ('-'<<8)|'E' : Exact = 255;						continue;
		case ('/'<<8)|'S' :
		case ('-'<<8)|'S' : Size = 0;							continue;
		case ('/'<<8)|'T' :
		case ('-'<<8)|'T' : Time = 0;							continue;
		} ptr -= 2;
		if(!Pattern) {
			Pattern = ptr;
			continue; }
		if(!Directory) {
			Directory = ptr;
			continue; }
		abort(Help); }

	init_app();

	do_dir(Directory);

	end_app();

	Printf(O_STAT, "MEM: File:%u Pool=%u Seg=%u", Fhigh, Phigh, Stop);
}
