/*
 * EMILY: An 8051/8052 simulator/debugger
 *
 * ?COPY.TXT 1991-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <window.h>

/* Misc simulator definitions */
#define	PODID	0x55	/* Init response of hardware pod */
#define	TRCSIZ	4096	/* Size of trace buffer */
#define	NUMBRK	10		/* Number of breakpoints */
#define	SFROFF	19		/* Offset into SFR table to skip registers */

/* Index's into window attribute table */
#define	DATTR	0		/* Index for disassembler window attributes */
#define	RATTR	1		/* Index for register window attributes */
#define	IATTR	2		/* Index for IRAM window attributes */
#define	MATTR	3		/* Index for message window attributes */
#define	EATTR	4		/* Index for editor window(s) attributes */

/* Breakpoint flags */
#define	B_SET	0xFF00	/* Breakpoint is set */
#define	B_ENAB	0x0040	/* Breakpoint enabled */
#define	B_WATCH	0x0020	/* Watch breakpoint */
#define	B_PAUSE	0x0010	/* Pause at breakpoint */
#define	B_KEY	0x0008	/* Wait for keypress */
#define	B_DEC	0x0004	/* Decrement counter */
#define	B_MASK	0x0003	/* Counter mask */

/* Version marker (for EMSETUP) */
char version = 0x16;
/* 8052 special functions register table */
char sfrs[1024] = {
	0xE0,	'A',
	0xF0,	'B',
	0xD0,	'P','S','W',
	0x81,	'S','P',
	0x82,	'D','P','L',
	0x83,	'D','P','H',
	0x80,	'P','0',
	0x90,	'P','1',
	0xA0,	'P','2',
	0xB0,	'P','3',
	0xB8,	'I','P',
	0xA8,	'I','E',
	0x89,	'T','M','O','D',
	0x88,	'T','C','O','N',
	0xC8,	'T','2','C','O','N',
	0x8C,	'T','H','0',
	0x8A,	'T','L','0',
	0x8D,	'T','H','1',
	0x8B,	'T','L','1',
	0xCD,	'T','H','2',
	0xCC,	'T','L','2',
	0xCB,	'R','C','A','P','2','H',
	0xCA,	'R','C','A','P','2','L',
	0x98,	'S','C','O','N',
	0x99,	'S','B','U','F',
	0x87,	'P','C','O','N',
	0 };

/* 8250 uart configuration tables */
unsigned baud[] = { 384, 384, 96, 48, 24, 12, 4, 0 };
char baud_rate = 0, parity = 4, dbits = 3, sbits = 0;

/* Command line options */
char filename[50] = { ".HEX" };
unsigned hbaud = 4, com1 = 0x3FD, com2 = 0x2FD, adelay = 0, bdelay = 1000,
	iomlo, iomhi = -1;
char reset = -1, IOM = 0;

int *attributes,		/* Pointer to active attribute set */
	Cattrs[] = {		/* Attributes for color monitors */
		WSAVE|WBOX1|WCOPEN|0x17,					/* Disassembler window */
		WSAVE|WBOX1|WCOPEN|0x57,					/* Register window */
		WSAVE|WBOX1|WCOPEN|0x47,					/* Internal RAM window */
		WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|0x30,		/* Message window */
		WSAVE|WBOX2|WCOPEN|0x67 },					/* Edit window(s) */
	Mattrs[] = {		/* Attributes for monochrome monitors */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Disassembler window */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Register window */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Internal RAM window */
		WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|NORMAL,	/* Message window */
		WSAVE|WBOX2|WCOPEN|NORMAL };				/* Edit window(s) */

extern int PSP;		/* To determine our own segment */

/* Misc. flags & variables */
char oflag = 0, fflag = 0, mflag = 0, local_flag;
unsigned LOAD_PC = 0, BRKCNT[B_MASK+1], edit_seg;

/* 8052 simulator variables (defined in SIM.ASM) */
extern unsigned char A, B, PSW, SP, INTRAM[384], POD, Aflag, Dflag;
extern int CSEG, DSEG, DPTR, PC, BRKPTS[NUMBRK+1], BRKFLG[NUMBRK],
	TRCPTR, TRCBUF[TRCSIZ], Sseg;

/* Window/Display control variables */
char *Dwin, *Rwin, *Iwin, *Mwin;
unsigned Dpos, Dhigh, Rpos = 0;

/* Memory editor argument strings */
/* Edit address, highest address, mode, title */
char *Iargs[] = { 0, 0x00FF, 0, "Internal memory" };
char *Dargs[] = { 0, 0xFFFF, 0, "Data memory" };
char *Cargs[] = { 0, 0xFFFF, 0, "Program memory" };

/* 8052 register modification tables */
char Rchars[] = { "01234567ABDSWP" };
char *Rnames[] = {
	"R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7",
	"A", "B", "DPTR", "SP", "PSW", "PC" };

/* 8052 interrupt vector tables */
char *intnames[] = {
	"External 0",
	"External 1",
	"Timer 0 overflow",
	"Timer 1 overflow",
	"Timer 2 overflow",
	"Timer 2 external",
	"Serial Receive",
	"Serial Transmit",
	0 };
unsigned char intvec[] = {		/* Interrupt vectors */
	0x03, 0x13, 0x0B, 0x1B, 0x2B, 0x2B, 0x23, 0x23 };
unsigned char intbit[] = {		/* Bits to set */
	0x00, 0x00, 0x00, 0x00, 0xCF, 0xCE, 0x98, 0x99 };
int IV = 0;

/* Help screens */
char *mhelp[] = {		/* Main emulator help */
	"Emulator commands",
	"A    - Animate execution       F2   - Set disassembler address",
	"B    - set a Breakpoint        F3   - Setup serial port",
	"C    - Change a register       F4   - Breakpoint counters",
	"D    - edit Data memory        F5   - Reset processor",
	"F    - Function registers      F10  - Shell to DOS",
	"G    - Go: execute program     space- Step one instruction",
	"I    - edit Internal memory    Up   - Backup  IRAM one line",
	"J    - Jump to PC display      Down - Advance IRAM one line",
	"K    - Kill all breakpoints    PgUp - Backup  IRAM one page",
	"L    - Load program file       PgDn - Advance IRAM one page",
	"N    - Next code page          Esc  - Terminate EMILY52",
	"P    - edit Program memory",
	"R    - Remove a breakpoint",
	"T    - Traceback viewer",
	"U    - scroll code page Up",
	"V    - Vector to interrupt",
	0 };

char *ehelp[] = {		/* Ram editor help */
	"Editor commands",
	"F2       - Set edit address",
	"F3       - Toggle HEX/ASCII edit",
	"F4       - Block fill memory",
	"Left     - Backup  cursor  one byte",
	"Right    - Advance cursor  one byte",
	"Up       - Backup  cursor  one line",
	"Down     - Advance cursor  one line",
	"PgUp     - Backup  display one page",
	"PgDn     - Advance display one page",
	"Home     - Backup  to start of line",
	"End      - Advance to end of line",
	"Ctl-PgUp - Position to start of memory",
	"Ctl-PgDn - Position to end of memory",
	"Esc      - Terminate EDIT",
	0 };

char *chelp[] = {		/* Register change help */
	"Change commands",
	"A   - change A accumulator    (A)",
	"B   - change B accumulator    (B)",
	"D   - change Data pointer     (DPTR)",
	"S   - change Stack poiner     (SP)",
	"P   - change Program counter  (PC)",
	"W   - change processor status Word (PSW)",
	"0-7 - change general register (Rn)",
	"Esc - Abort change command",
	0 };

char *shelp[] = {		/* String input help */
	"String input",
	"Left - Move cursor left",
	"Right- Move cursor right",
	"Bs   - Backspace & delete",
	"Del  - Delete character under cursor",
	"Ins  - Toggle Insert/Overwrite mode",
	"Home - Move Cursor to start of string",
	"End  - Move cursor to end of string",
	"PgUp - Clear entire field",
	"PgDn - Clear from cursor to end of field",
	"Enter- Accept entry",
	"Esc  - Abort input",
	0 };

char *thelp[] = {		/* Traceback help */
	"Traceback commands",
	"Up   - Display previous instruction",
	"Down - Display next instruction",
	"PgUp - Backup  50 instructions",
	"PgDn - Advance 50 instructions",
	"Home - Position to start of buffer",
	"End  - Position to end of buffer",
	"F2   - Set traceback position",
	"F3   - Search backward (in time)",
	"F4   - Search forward  (in time)",
	"Esc  - Terminate TRACEBACK",
	0 }

char *welcome[] = {		/* Startup message */
	">>> EMILY-52 <<<",
	"",
	"A PC based 8051/52 simulator.",
	"",
	"Release 1.7",
	"",
	"Use: EMILY52 [filename] [options]",
	"",
	"Options:",
	"",
	"/Io[:low-high] /Hardware[:baud] /Mono /Overlay /Reset /Swapcom S=sfr_code",
	"",
	"?COPY.TXT 1991-2005 Dave Dunfield",
	"**See COPY.TXT**.",
//	"", "", "", "*** Pre-Release - NOT FOR DISTRIBUTION ***",
	0 }

char *count_form[] = {
	21<<8|6,
	"\x01\x00\x85Counter 1 :",
	"\x01\x01\x85Counter 2 :",
	"\x01\x02\x85Counter 3 :",
	"\x01\x03\x85Delay (ms):",
	0 };

/*
 * Main program - Initialize and present main windows
 */
main(argc, argv)
	int argc;
	char *argv[];
{
	unsigned c, i, j;
	char *ptr;

/* Process command line parameters */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case '/O' :		/* Overlaid I/O space */
			case '-O' :
				oflag = -1;
				break;
			case '/M' :		/* Force monochrome */
			case '-M' :
				mflag = -1;
				break;
			case '/I' :		/* IO redirection */
			case '-I' :
				IOM = -1;
				while(*ptr) {
					if(*ptr++ == ':') {
						iomlo = atox(ptr);
						while(*ptr) {
							if(*ptr++ == '-') {
								iomhi = atox(ptr);
								break; } }
						break; } }
				break;
			case '/H' :		/* Hardware simulation */
			case '-H' :
				POD = -1;
				while(*ptr) {
					if(*ptr++ == ':') {
						hbaud = scale(57600, 2, atoi(ptr));
						break; } }
				break;
			case '/R' :		/* Reverse reset */
			case '-R' :
				reset = !reset;
				break;
			case '/S' :		/* Swap COMM ports */
			case '-S' :
				c = com1;
				com1 = com2;
				com2 = c;
				break;
			case 'S=' :		/* SFR emulation */
				Sseg = alloc_seg(4096);
				Mwin = fopen(ptr, "rvqb");
				j = 0;
				while(!((c = getc(Mwin)) & 0xFF00))
					poke(Sseg, j++, c);
				fclose(Mwin);
				break;
			default:
				fflag = -1;
				strcpy(filename, argv[i]); } }

/* Open the welcome screen & output startup message */
	wopen(0, 0, 80, 25, WSAVE|WBOX2|WCOPEN|REVERSE);
	wcursor_off();
	for(i=0; ptr = welcome[i]; ++i) {
		wgotoxy((78-strlen(ptr))/2, i+4);
		wputs(ptr); }

	attributes = (mflag || (W_BASE == 0xB000)) ? Mattrs : Cattrs;

	CSEG = alloc_seg(4096);
	DSEG = oflag ? CSEG : alloc_seg(4096);
	init_serial();
	out(com2-2, 0x83);			/* Access divisor latch */
	out(com2-5, hbaud);			/* Baudrate divisor (low) */
	out(com2-4, hbaud>>8);		/* Baudrate divisor (high) */
	out(com2-2, 0x03);			/* 8 Data, NO parity. 1 Stop */
	out(com2-4, 0);				/* Disable all interrupts */
	out(com2-1, reset ? 3 : 1);	/* Reset with correct mode */

/* Zero all code and data memory */
	i = 0;
	do {
		poke(CSEG, i, 0);
		poke(DSEG, i, 0); }
	while(++i);

/* Initialize internal memory & SFR's, breakpoint table & tracebuffer */
/*	memset(BRKPTS, 0, sizeof(BRKPTS)+sizeof(INTRAM)+sizeof(TRCBUF)); */
	reset_target(0);
	i = 51;
	while(--i && !wtstc())
		delay(100);
	wclose();

/* Draw the working menus */
	Dwin = wopen(0,  0, 45, 18, attributes[DATTR]);
	Rwin = wopen(45, 0, 16, 18, attributes[RATTR]);
	Iwin = wopen(61, 0, 19, 18, attributes[IATTR]);
	Mwin = wopen(0, 18, 80,  7, attributes[MATTR]);
	wcursor_off();

/* Check for POD and set mode */
	while(POD | IOM) {
		rdpod();
		rdpod();
		podcmd(0xFF);
		wcleol();
		if(podcmd(0xFE) == PODID)
			break;
		wputs("Target system is not responding!!!\r");
		if(wtstc() == 0x1B)
			POD = IOM = 0; }

	if(IOM)
		patchio();

	wprintf("Code and data are %s, Hardware emulation is %sabled%s",
		oflag ? "overlaid" : "separate",
		POD ? "en" : "dis", IOM ? " (IO)" : "");

	if(Sseg)
		SFRinit();
	Dflag = -1;

/* Load file & setup PC */
	if(fflag)
		load_file(CSEG);
	Dpos = PC;

/* Prompt for and get commands */
	show_dis();
	update();
	wputs("\nPress F1 for help");

	for(;;) {
		switch(c = toupper(wgetc())) {
		case 'A' :	/* Animate */
			i = -1;
			if((c = get_number("Animate how many instructions (CR=Non-stop)? ", 10, &i)) != 0x1B) {
				w_printf(Mwin, "\nAnimating at %04x", PC);
				while(i) {
					if(!c)
						--i;
					single_step();
					if((wtstc() == 0x1B) || !Aflag)
						goto stop;
					if(j = isbreak(PC)) {
						if(!handle_breakpoint(j))
							break; }
					if((j & (B_ENAB|B_WATCH)) != (B_ENAB|B_WATCH)) {
						show_pc();
						update(); }
					if(adelay)
						delay(adelay); }
				goto stop; }
			continue;
		case 'B' :	/* Set breakpoint */
			w_putc('\n', Mwin);
			show_brks();
			j = PC;
			if((get_number("Breakpoint at (CR=PC)? ", 16, &j) != 0x1B) && j) {
				for(i=0; i < NUMBRK; ++i) {
					if(BRKPTS[i] == j) {
						w_puts("Exists!", Mwin);
						BRKFLG[i] = get_bflags(BRKFLG[i]);
						show_brks();
						goto sbrk; }
					if(!BRKPTS[i]) {
						BRKPTS[i] = j;
						BRKFLG[i] = get_bflags(B_SET|B_ENAB);
						show_brks();
						goto sbrk; } }
				w_puts("All used!", Mwin);
			sbrk:
				show_dis(); }
			continue;
		case 'C' :	/* Change register */
			change_reg();
			break;
		case 'D' :	/* Data memory */
			edit_seg = DSEG;
			edit(Dargs, 0);
			show_dis();
			break;
		case 'F' :	/* Special function registers */
			edit_sfr(sfrs+SFROFF);
			break;
		case 'I' :	/* Internal memory */
			edit_seg = PSP;
			edit(Iargs, INTRAM);
			break;
		case ' ' :	/* Step program */
			single_step();
		case 'J' :	/* Jump to PC */
			show_pc();
			break;
		case 'K' :	/* Kill ALL breakpoints */
			if(yesno("Kill all breakpoints")) {
				memset(BRKPTS, 0, sizeof(BRKPTS));
				w_putc('\n', Mwin);
				show_brks();
				show_dis(); }
			continue;
		case 'L' :	/* Load file */
			wopen(10, 10, 60, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
			wputs("Load file? ");
again:		if((i=wgets(11, 0, filename, 49)) == '\n') {
				wclose();
				load_file(CSEG);
				show_pc();
				break; }
			if(i == _K1) {
				help(shelp);
				goto again; }
			wclose();
			break;
		case 'N' :	/* Next page */
			Dpos = Dhigh;
			show_dis();
			continue;
		case 'U' :	/* Up a page */
			scroll_up();
			show_dis();
			continue;
		case 'P' :	/* Program memory */
			edit_seg = CSEG;
			edit(Cargs, 0);
			show_dis();
			break;
		case 'R' :	/* Remove breakpoint */
			w_putc('\n', Mwin);
			show_brks();
			c = PC;
			if(get_number("Remove breakpoint at (CR=PC)? ", 16, &c) != 0x1B) {
				for(i=0; i < NUMBRK; ++i) {
					if(BRKPTS[i] == c) {
						while(i < NUMBRK) {
							BRKFLG[i] = BRKFLG[i+1];
							BRKPTS[i] = BRKPTS[++i]; } } }
				show_brks();
				show_dis(); }
			continue;
		case 'G' :	/* Go */
			i = 0;
			mflag = -1;
			if((c = get_number("Execute how many instructions (CR=Non-stop)? ", 10, &i)) != 0x1B) {
				w_printf(Mwin, "\nExecuting at %04x", PC);
				if(!baud_rate)
					wcursor_line();
		go:
				wupdatexy();
				if(i || c) {
					multi_step(i);
					if(i) {
						asm {
							MOV		AX,DGRP:?temp
							}
						c = i = nargs(); }
					if(handle_breakpoint(isbreak(PC)) && mflag)
						goto go; }
		stop:
				wcursor_off();
				show_pc();
				w_printf(Mwin, ", Stopped at %04x", PC); }
			break;
		case 'T' :	/* Trace back */
			traceback();
			continue;
		case 'V' :	/* Vector to interrupt */
			if(!wmenu(31, 6, attributes[EATTR], intnames, &IV)) {
				push16(PC);
				PC = intvec[IV];
				if(i = intbit[IV])
					set_bit(i);
				show_pc(); }
			break;
		case _K1 :	/* Help */
			help(mhelp);
			continue;
		case _K2 :	/* Show */
			get_number("Code address? ", 16, &Dpos);
			show_dis();
			continue;
		case _K3 :	/* Set COMM parameters */
			setup_serial();
			continue;
		case _K4 :	/* Breakpoint counters */
			wform(30, 10, WSAVE|WBOX2|WCOPEN|REVERSE, count_form,
				&BRKCNT[1], &BRKCNT[2], &BRKCNT[3], &bdelay);
			continue;
		case _K5 :	/* Reset processor */
			reset_target(-1);
			show_pc();
			break;
		case _K10 :	/* Shell to DOS */
			doshell();
			break;
		case _KUA :	Rpos -= 4;	break;
		case _KDA :	Rpos += 4;	break;
		case _KPU : Rpos -= 64;	break;
		case _KPD :	Rpos += 64;	break;
		case 0x1B :	/* Exit */
			if(yesno("Terminate EMILY52")) {
				wclose();
				wclose();
				wclose();
				wclose();
				return; } }
		update(); }
}

/*
 * Display a disassembler address
 */
show_dis()
{
	unsigned int a, i, j, k, at, ra;
	char buffer[50];

	a = Dpos;
	ra = reverse(at = attributes[DATTR]);
	for(i=0; i < 16; ++i) {
		w_gotoxy(0, i, Dwin);
		*Dwin = (a == PC) ? ra : at;
		j = disass(CSEG, a, buffer);
		w_printf(Dwin, "%04x", Dhigh = a);
		for(k=0; k < 3; ++k)
			w_printf(Dwin, k < j ? " %02x" : "   ", peek(CSEG, a+k));
		w_printf(Dwin, "   %-26s%c", buffer, show_breakpoint(isbreak(a)));
		a += j; }
}

/*
 * Calculate higher address
 */
scroll_up()
{
	unsigned a, i, addr[48];
	char buffer[50];

	a = i = 0;
	if(Dpos > 48)
		a = Dpos - 48;

	do {
		addr[i++] = a;
		a += disass(CSEG, a, buffer); }
	while(a < Dpos);
	Dpos = (i > 14) ? addr[i-15] : 0;
}

/*
 * Show DIS at PC
 */
show_pc()
{
	int a, i;
	char buffer[50];

	a = Dpos;
	for(i=0; i < 16; ++i) {
		if(a == PC) {
			show_dis();
			return; }
		a += disass(CSEG, a, buffer); }
	Dpos = PC;
	show_dis();
}

/*
 * Update the register & RAM display
 */
update()
{
	int i, j;

	w_gotoxy(0, 1, Rwin);
	w_printf(Rwin, "A   :%02x\nB   :%02x\nDPTR:%04x [%02x]\nSP  :%02x   [%02x]\nPSW :%02x",
		A, B, DPTR, peek(DSEG, DPTR), SP, INTRAM[SP], PSW);
	w_printf(Rwin, " %c%c%c%c%c%c",
		PSW & 0x80 ? 'C' : '-',
		PSW & 0x40 ? 'A' : '-',
		PSW & 0x20 ? 'F' : '-',
		((PSW & 0x18) >> 3) + '0',
		PSW & 0x04 ? 'O' : '-',
		PSW & 0x01 ? 'P' : '-');
	w_printf(Rwin, "\nPC  :%04x\nR0  :%02x   [%02x]\nR1  :%02x   [%02x]", PC,
		i = INTRAM[j = PSW & 0x18], INTRAM[i], i=INTRAM[j+1], INTRAM[i]);
	for(i=2; i < 8; ++i)
		w_printf(Rwin, "\nR%u  :%02x", i, INTRAM[i+j]);

	w_gotoxy(0, 0, Iwin);
	for(i=0; i < 64; ++i) {
		if(!(i%4))
			w_printf(Iwin, "%s%04x:", i ? "\n": "", (i+Rpos) & 255);
		w_printf(Iwin, " %02x", INTRAM[(i+Rpos) & 255]); }
}

/*
 * Write memory with IO redirection (if enabled)
 */
write_memory(unsigned addr, int v)
{
	if((edit_seg == DSEG) && IOM) {
		if((addr >= iomlo) && (addr <= iomhi)) {
			wrpod(0x63);
			wrpod(1);
			wrpod(addr);
			wrpod(addr >> 8);
			wrpod(v);
			return; } }
	poke(edit_seg, addr, v);
}

/*
 * Edit global memory
 */
edit(args, offset)
	unsigned args[], offset;
{
	unsigned dt, dc, i, j, k;

	wopen(3, 3, 73, 18, attributes[EATTR]);
	title(args[3]);
	wcursor_line();
redraw:
	dt = *args & 0xFF00;
	for(i=0; i < 16; ++i) {
		wgotoxy(0, i);
		draw_line(dt+(i*16), offset); }
prompt:
	if(*args > args[1])
		*args &= 0x00FF;
	if((*args & 0xFF00) != dt)
		goto redraw;
	dc = *args & 0xFF;
	wgotoxy(args[2] ? (dc%16)+55 : ((dc%16)*3)+6, dc/16);
	switch(toupper(i = wgetc())) {
		case 0x1B :
			wclose();
			return;
		case _K1 :		/* Help request */
			help(ehelp);
			break;
		case _K2 :		/* Go to address */
			get_number("Edit address? ", 16, args);
			break;
		case _K3 :		/* Toggle mode */
			args[2] = !args[2];
			break;
		case _K4 :		/* Fill memory */
			if(get_number("Starting address? ", 16, &dc))
				break;
			if(dc > args[1])
				break;
			if(get_number("Ending address? ", 16, &dt))
				break;
			if((dt > args[1]) || (dt < dc))
				break;
			if(get_number("Fill with? ", 16, &i))
				break;
			do
				write_memory(dc + offset, i);
			while(dc++ < dt);
			break;
		case _KLA :
			--*args;
			break;
		case _KPU :
			*args -= 256;
			break;
		default:
			if(args[2]) {
				if(i >= 128)
					break;
				write_memory(*args+offset, i); }
			else {
				if((j = gethex(i)) >= 16)
					break;
				wputc(i);
				if((k = gethex(wgetc())) < 16)
					write_memory(*args+offset, (j<<4)+k); }
			wgotoxy(0, i = dc/16);
			draw_line(dt+(i*16), offset);
		case _KRA :
			++*args;
			break;
		case _KPD :
			*args += 256;
			break;
		case _KHO :
			*args = (*args-1) & 0xFFF0;
			break;
		case _KUA :
			*args -= 16;
			break;
		case _KEN :
			*args = (*args+1) | 0x000F;
			break;
		case _KDA :
			*args += 16;
			break;
		case _CPU :
			*args = 0;
			break;
		case _CPD :
			*args = args[1]; }
	goto prompt;
}

/*
 * Display a line for the editor
 */
draw_line(addr, ptr)
	unsigned addr, ptr;
{
	unsigned int i, c;
	unsigned char buffer[16];

	ptr += addr;

	/* If remote I/O is enabled, access target */
	if((edit_seg == DSEG) && IOM) {
		if((ptr >= iomlo) && (ptr <= iomhi)) {
			if((ptr + 15) <= iomhi) {
				wrpod(0x62);
				wrpod(16);
				wrpod(ptr);
				wrpod(ptr >> 8);
				for(i=0; i < 16; ++i)
					buffer[i] = podrd();
				goto havemem; }
			for(i=0; i < 16; ++i) {
				if(ptr <= iomhi) {
					wrpod(0x62);
					wrpod(1);
					wrpod(ptr);
					buffer[i] = podcmd(ptr >> 8); }
				else
					buffer[i] = peek(edit_seg, ptr);
				++ptr; }
			goto havemem; } }
	for(i=0; i < 16; ++i)
		buffer[i] = peek(edit_seg, ptr++);

havemem:
	wprintf("%04x  ", addr);
	for(i=0; i < 16; ++i)
		wprintf("%02x ", buffer[i]);
	wputc(' ');
	for(i=0; i < 16; ++i) {
		c = buffer[i];
		wputc(((c >= ' ') && (c <= 0x7E)) ? c : '.'); }
}

/* Test for HEX digit & get value*/
gethex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	return -1;
}

/*
 * Display a title on a window
 */
title(ptr)
	char *ptr;
{
	char buffer[8];
	memcpy(buffer, W_OPEN, sizeof(buffer));
	*W_OPEN = REVERSE;
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, sizeof(buffer));
}

/*
 * Get an numeric value at a prompt
 */
get_number(prompt, base, var)
	char *prompt;
	int base, *var;
{
	unsigned int i, j, x, value, len;

	i = (len = (base == 10) ? 5 : 4) + strlen(prompt) + 3;
	wopen((80-i)/2, 10, i, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wputs(prompt);
	wcursor_line();
	value = x = 0;
	for(;;) {
		switch(i = wgetc()) {
			case '\n' :
				if(x) {
					*var = value;
					wclose();
					return 0; }
			case 0x1B :
				wclose();
				return i;
			case _KBS :
			case _KLA :
				if(x) {
					--x;
					wputs("\b \b");
					value /= base; }
				break;
			default:
				if(isdigit(j = toupper(i)))		/* convert numeric digits */
					j -= '0';
				else if(j >= 'A')				/* convert alphabetics */
					j -= ('A' - 10);
				else							/* not a valid "digit" */
					break;
				if((x < len) && (j < base)) {
					wputc(i);
					++x;
					value = (value * base) + j; } } }
}

/*
 * Change register contents
 */
change_reg()
{
	int c, i;
	char buffer[50];

	wopen(30, 10, 20, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wputs("Change register? ");
prompt:
	if((c = toupper(wgetc())) == 0x1B) {
		wclose();
		return; }
	if(c == _K1) {
		help(chelp);
		goto prompt; }
	for(i=0; Rchars[i] != c; ++i)
		if(!Rchars[i])
			goto prompt;
	wclose();
	sprintf(buffer, "New value for %s? ", Rnames[i]);
	if(!get_number(buffer, 16, &c))	switch(i) {
		case 8 : A = c;				break;
		case 9 : B = c;				break;
		case 10: DPTR = c;			break;
		case 11: SP = c;			break;
		case 12: PSW = c;			break;
		case 13: PC = c; show_pc();	break;
		default: INTRAM[(PSW & 0x18)+i] = c; }
}

char *lptr;	/* Global ptr to load arguments */
/*
 * Read file into memory image
 */
load_file(seg)
	int seg;
{
	unsigned a, b, chksum, addr, length, count, size;
	char buffer[80];
	FILE *fp;

	w_printf(Mwin, "\n%s: ", filename);
	if(!(fp = fopen(filename,"r"))) {
		w_puts("Unable to access", Mwin);
		return; }

	for(size = 0; fgets(lptr = buffer, 80, fp);) {
		again: switch(*lptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*lptr == '9') goto quitload;
				if(*lptr++ != '1') continue;
				length = count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				addr = (a << 8) + b;
				if(!size)
					LOAD_PC = PC = addr;
				while(count--) {
					chksum += a = get_byte();
					poke(seg, addr++, a); }
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = get_byte())) goto quitload;
				chksum = (a = get_byte()) + length;
				chksum += b = get_byte();
				addr = (a << 8) + b;
				chksum += get_byte();
				if(!size)
					LOAD_PC = PC = addr;
				while(count--) {
					chksum += a = get_byte();
					poke(seg, addr++, a); }
				if((255 & -chksum) != get_byte())
					goto quitbad;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				w_puts("Invalid record format", Mwin);
				goto quit; }
		size += length;
		}

quitload:
	w_printf(Mwin, "%u bytes loaded", size);
	goto quit;
quitbad:
	w_puts("Bad checksum", Mwin);
quit:
	fclose(fp);
	show_pc();
}

/* Load a byte from the hex file */
get_byte()
{
	return (gethex(*lptr++) << 4) + gethex(*lptr++);
}

/*
 * Display a help screen
 */
help(ptr)
	char *ptr[];
{
	int i;
	for(i=1; ptr[i]; ++i);
	wopen(7, (24-i)/2, 66, i+1, WSAVE|WBOX2|WCOPEN|REVERSE);
	title(ptr[0]);
	wcursor_off();
	for(i=1; ptr[i]; ++i) {
		wgotoxy(1, i-1);
		wputs(ptr[i]); }
	while(wgetc() != 0x1B);
	wclose();
}

/*
 * Initialize uart after allowing change to parameters
 */
setup_serial()
{
	unsigned b;
	char buffer[6];
	static char *partab[] = { "Odd", "Even", "Mark", "Space", "None", 0 };

	wopen(30, 10, 20, 6, attributes[EATTR]);
	title("COMM Setup");
	for(;;) {
		wgotoxy(0, 0);
		sprintf(buffer, "%u", scale(57600, 2, b = baud[baud_rate]));
		wprintf("(B)audrate:  %5s\n", baud_rate ? buffer : "Local");
		wprintf("(P)arity:    %5s\n", partab[parity]);
		wprintf("(D)ata bits: %5u\n", dbits+5);
		wprintf("(S)top bits: %5u", sbits+1);
		switch(toupper(wgetc())) {
			case _KUA :
				--b;
				goto setbaud;
			case _KDA :
				++b;
			setbaud:
				if(baud_rate)
					baud[baud_rate] = (b < 2) ? 2 : b;
				break;
			case 'B' :
				if(!baud[++baud_rate])
					baud_rate = 0;
				break;
			case 'P' :
				if(!partab[++parity])
					parity = 0;
				break;
			case 'D' :
				if(++dbits > 3)
					dbits = 0;
				break;
			case 'S' :
				if(++sbits > 1)
					sbits = 0;
				break;
			case 0x1B:
				init_serial();
				wclose();
				return; } }
}

/*
 * Initialize PC serial port for simulated processor
 */
init_serial()
{
	unsigned i, r;

	i = ((parity << 4) & 0x30)	|	/* parity bits */
		(dbits & 0x03)			|	/* # data bits */
		((sbits << 2) & 0x04)	|	/* # stop bits */
		((parity < 4) << 3) ;
	out(com1-2, i | 0x80);
	out(com1-5, r = baud[baud_rate]);
	out(com1-4, r >> 8);
	out(com1-2, i);
	out(com1-4, 0);	/* disable all interrupts */
	out(com1-1, 3);	/* Enable DTR and RTS */
}

/*
 * Display the breakpoints
 */
show_brks()
{
	int i, j;

	w_puts("\rBreak: ", Mwin);
	for(i=0; BRKPTS[i]; ++i) {
		if((j = show_breakpoint(BRKFLG[i])) != 'b')
			w_putc(j, Mwin);
		w_printf(Mwin, "%04x ", BRKPTS[i]); }
	w_cleol(Mwin);
	return i;
}

/*
 * Edit the SFR's
 */
edit_sfr(table)
	char *table;
{
	unsigned i, j, c, n;
	unsigned char buffer[10], buffer1[50], *ptr, *ptr1;
	static int bank = 0;

	n = 0;
	for(ptr=table; *ptr; ++ptr)
		if(*ptr & 0x80)
			++n;
redraw:
	wopen(6, 1, ((((n-1)/20)+1)*17)+2, 22, attributes[EATTR]);
	title("SFRs");
	for(;;) {
		ptr = table;
		i = 0;
		do {
			wgotoxy(((i/20)*17)+1, i % 20);
			j = *ptr++;
			for(ptr1 = buffer; (c = *ptr) && !(c & 0x80); ++ptr)
				*ptr1++ = c;
			*ptr1 = 0;
			wprintf((i/20 == bank) ? "%c:" : "  ", (i%20)+'A');
			wprintf(" %-8s -%02x", buffer, read_direct(j));
			++i; }
		while(*ptr);

		if((j = toupper(wgetc())) == 0x1B) {
			wclose();
			return; }

		if(j == ' ')
			bank = (bank + 1) % ((i+19)/20);
		if((j < 'A') || (j > 'T'))
			continue;

		i = (j - 'A') + (bank * 20);
		ptr = table;
		while(*ptr) {
			j = *ptr++;
			for(ptr1 = buffer; (c = *ptr) && !(c & 0x80); ++ptr)
				*ptr1++ = c;
			*ptr1 = 0;
			if(!i--) {
				sprintf(buffer1, "New value for %s? ", buffer);
				if(!get_number(buffer1, 16, &c)) {
					wclose();
					write_direct(j, c);
					goto redraw; }
				break; } } }
}

/*
 * Reverse color attributes
 */
reverse(attr)
	int attr;
{
	return (attr & 0xff00) | ((attr & 0xf0) >> 4) | ((attr & 0x0f) << 4);
}

/*
 * Prompt for YES/NO response
 */
yesno(ptr)
	char *ptr;
{
	int i;

	i = strlen(ptr)+6;
	wopen((80-i)/2, 10, i, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wprintf("%s ? Y", ptr);
	for(;;) switch(toupper(wgetc())) {
		case 'N' :
		case 0x1B :
			wclose();
			return 0;
		case '\n' :
		case 'Y' :
			wclose();
			return 1; }
}

/*
 * Perform a traceback display
 */
traceback()
{
	unsigned int savepc, i, j, tbase;
	int k;

	TRCBUF[tbase = TRCPTR/2] = savepc = PC;
	wopen(51, 14, 26, 3, attributes[EATTR]);
	i = 0;
	for(;;) {
		wgotoxy(0, 0);
		wprintf("Traceback position: %4u", i);
		PC = TRCBUF[(tbase-i) & (TRCSIZ-1)];
		show_pc();
		switch(wgetc()) {
			case 0x1B :
				PC = savepc;
				wclose();
				show_dis();
				return;
			case _K1 :
				help(thelp);
				break;
			case _K2 :
				j = -1;
				get_number("New position? ", 10, &j);
				if(j < TRCSIZ)
					i = j;
				break;
			case _K3 :
				j = PC;
				if(get_number("Search backward for (CR=Current address)? ", 16, &j) != 0x1B) {
					for(k=i+1; k < TRCSIZ; ++k) {
						if(TRCBUF[(tbase-k) & (TRCSIZ-1)] == j) {
							i = k;
							goto found; } }
					goto notfound; }
				break;
			case _K4 :
				j = PC;
				if(get_number("Search forward for (CR=Current address)? ", 16, &j) != 0x1B) {
					for(k = i-1; k >= 0; --k) {
						if(TRCBUF[(tbase-k) & (TRCSIZ-1)] == j) {
							i = k;
							goto found; } }
			notfound:
					w_printf(Mwin, "\n%04x: Not found", j); }
			found:
				break;
			case _KPU :
				if(i < (TRCSIZ-51)) {
					i+= 50;
					break; }
			case _KHO :
				i = TRCSIZ-1;
				break;
			case _KPD :
				if(i > 50) {
					i -= 50;
					break; }
			case _KEN :
				i = 0;
				break;
			case _KUA :
				if(i < (TRCSIZ-1))
					++i;
				break;
			case _KDA :
				if(i)
					--i; } }
}

/*
 * Scale to a fractional value
 */
scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Function to execute when ESCAPE key pressed in LOCAL mode
 */
local_menu()
{
	static int i = 0;
	static char *menu[] = {
		"End simulation",
		"Send ESCAPE",
		"Zoom window", 0 }

	if(!wmenu(32,9,WSAVE|WCOPEN|WBOX3|REVERSE,menu,&i)) switch(i) {
		case 0 :	/* Stop simulation */
			return -1;
		case 1 :	/* Send ESCAPE */
			return 1;
		case 2 :	/* Zoom window */
			if(local_flag = !local_flag)
				wopen(0,0,80,25,attributes[MATTR]);
			else
				wclose();
			wupdatexy(); }
	return 0;
}

#ifdef DEBUG
char DTEXT[] = { "\nPC=%04x AX=%04x BX=%04x CX=%04x DX=%04x SI=%04x DI=%04x" };
debug() asm
{
	MOV		DGRP:?temp,AX
	MOV		AX,OFFSET DGRP:_DTEXT
	PUSH	AX
	MOV		AX,2[BP]
	PUSH	AX
	MOV		AX,DGRP:?temp
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	MOV		AX,8
	CALL	_wprintf
	POP		DI
	POP		SI
	POP		DX
	POP		DX
	POP		BX
	POP		AX
	ADD		SP,4
}
#endif
/*
 * Execute a sub-shell
 */
doshell()
{
	char comspec[65];

	if(!getenv("COMSPEC", comspec)) {
		wputs("Cannot locate COMSPEC environment variable");
		return; }
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|0x07);
	wputs("Type 'EXIT' to return to EMILY52.");
	wcursor_line();
	wupdatexy();
	exec(comspec, "");
	wclose();
}

/*
 * Reset the target system
 */
reset_target(prompt)
	char prompt;
{
	if(prompt & 0xF0) {
		if(!yesno("Reset"))
			return 0; }

	if(POD) {
		out(com2-1, reset ? 1 : 3);	/* Reset with correct mode */
		delay(300); }

	memset(INTRAM+256, 0, sizeof(INTRAM)-256);
	A = B = PSW = DPTR = 0;
	SP = 7;
	PC = LOAD_PC;
	INTRAM[0x100] = INTRAM[0x110] = INTRAM[0x120] = INTRAM[0x130] = 0xFF;
	out(com2-1, reset ? 3 : 1);	/* Reset with correct mode */

	if(prompt)
		w_puts("\nRESET!", Mwin);
	return -1;
}

/*
 * Handle a breakpoint exception
 */
handle_breakpoint(unsigned b)
{
	unsigned i;

	if(!(b & B_ENAB))
		return -1;

	if(i = b & B_MASK) {
		if(b & B_DEC) {				/* Decrement till quit */
			if(!BRKCNT[i])
				return mflag = 0;
			else
				--BRKCNT[i]; }
		else {						/* Increment count */
			if(!++BRKCNT[i])
				BRKCNT[i] = -1; } }

	if(b & B_WATCH) {				/* Watch point */
		show_pc();
		update();
		i = -1; }

	if(b & B_PAUSE) {
		wupdatexy();
		delay(bdelay);
		i = -1; }

	if(b & B_KEY) {
		i = -1;
		wputs("\nSPACE to proceed... ESC to halt:");
		for(;;) switch(wgetc()) {
			case ' ' :	return -1;
			case 0x1B :	return 0; } }

	return i;
}

/*
 * Get breakpoint flags
 */
get_bflags(unsigned f)
{
	wopen(24, 9, 32, 8, WSAVE|WCOPEN|WBOX2|REVERSE);
	for(;;) {
		wgotoxy(0, 0);
		wprintf("(E)nable        : %s\n", (f & B_ENAB) ? "YES" : "NO ");
		wprintf("(W)atchpoint    : %s\n", (f & B_WATCH)  ? "YES" : "NO ");
		wprintf("(K)eypress      : %s\n", (f & B_KEY) ? "YES" : "NO ");
		wprintf("(P)ause         : %s",   (f & B_PAUSE) ? "YES" : "NO ");
		if(f & B_PAUSE)
			wprintf(" (%u)", bdelay);
		wcleol();
		wputs("\n(C)ounter       : ");
		if(f & B_MASK) {
			wprintf("%u   (%u)", f & B_MASK, BRKCNT[f & B_MASK]);
			wcleol();
			wprintf("\n(D)ecrement/stop: %s", (f & B_DEC) ? "YES" : "NO "); }
		else {
			wputs("NO ");
			wcleow(); }
		switch(toupper(wgetc())) {
			case 0x1B :
			case '\n' :
				wclose();
				return f;
			case 'E' :	f ^= B_ENAB;	break;
			case 'W' :	f ^= B_WATCH;	break;
			case 'K' :	f ^= B_KEY;		break;
			case 'P' :	f ^= B_PAUSE;	break;
			case 'D' :	f ^= B_DEC;		break;
			case 'C' :	f = (f & ~B_MASK) | (++f & B_MASK); } }
}

/*
 * Get correct character to show for breakpoint
 */
show_breakpoint(unsigned f)
{
	char c;
	c = ' ';
	if(f) {
		if(!(f & B_ENAB))
			return 'x';
		c = 'b';
		if(f & B_WATCH)
			c = 'w';
		if(f & B_PAUSE)
			c = 'p';
		if(f & B_KEY)
			c = 'k';
		if(f & B_MASK) {
			c = '+';
			if(f & B_DEC)
				c = '-'; } }
	return c;
}
