/*
 * MONICA: An 8051 simulator/debugger
 *
 * ?COPY.TXT 1991-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <window.h>
#include <comm.h>
#define	write_target	COMputc

/* Reset types */
#define	RESET_PROMPT	0x01
#define	RESET_DISPLAY	0x02
#define	RESET_CPU		0x04

/* Kernal commands */
#define	PODIDLE	0xFF	/* Idle command to pod */
#define	PODIDEN	0xFE	/* Request ID */
#define	PODRI	0x60	/* Read INTERNAL memory */
#define	PODWI	0x61	/* Write INTERNAL memory */
#define	PODRE	0x62	/* Read EXTERNAL memory */
#define	PODWE	0x63	/* Write EXTERNAL memory */
#define	PODRP	0x64	/* Read PROGRAM memory */
#define	PODWP	0x65	/* Write PROGRAM memory */
#define	PODDS5	0x66	/* DS5000 memory setup */
#define	PODBRK	0xE0	/* Set a breakpoint */
#define	PODEXEC	0xE1	/* Begin execution */
#define	PODSTEP	0xE2	/* Single-Step one instruction */

/* Misc emulator commands */
#define	PODID	0x55	/* Init response of hardware pod */
#define	PODEID1	0xA5	/* Execution completed flag 1 */
#define	PODEID2	0x5A	/* Execution completed flag 2 */
#define	PODEID3	0x99	/* Execution completed flag 3 */
#define	NUMBRK	10		/* Number of breakpoints */
#define	SFROFF	19		/* Offset into SFR table to skip registers */

/* Offsets to window attributes */
#define	DATTR	0		/* Index for disassembler window attributes */
#define	RATTR	1		/* Index for register window attributes */
#define	IATTR	2		/* Index for IRAM window attributes */
#define	MATTR	3		/* Index for message window attributes */
#define	EATTR	4		/* Index for editor window(s) attributes */

/* Breakpoint flags */
#define	B_SET	0x80	/* Breakpoint is set */
#define	B_ENAB	0x40	/* Breakpoint is enabled */
#define	B_WATCH	0x20	/* Watch breakpoint */
#define	B_PAUSE	0x10	/* Pause at breakpoint */
#define	B_KEY	0x08	/* Wait for key */
#define	B_DEC	0x04	/* Decrement counter */
#define	B_MASK	0x03	/* Counter mask */

/* Version marker (for EMSETUP) */
char version = 0x16;
/* Special functions register table */
char sfrs[1024] = {
	0xE0,	'A',
	0xF0,	'B',
	0xD0,	'P','S','W',
	0x81,	'S','P',
	0x82,	'D','P','L',
	0x83,	'D','P','H',
	0x80,	'P','0',
	0x90,	'P','1',
	0xA0,	'P','2',
	0xB0,	'P','3',
	0xB8,	'I','P',
	0xA8,	'I','E',
	0x89,	'T','M','O','D',
	0x88,	'T','C','O','N',
	0xC8,	'T','2','C','O','N',
	0x8C,	'T','H','0',
	0x8A,	'T','L','0',
	0x8D,	'T','H','1',
	0x8B,	'T','L','1',
	0xCD,	'T','H','2',
	0xCC,	'T','L','2',
	0xCB,	'R','C','A','P','2','H',
	0xCA,	'R','C','A','P','2','L',
	0x98,	'S','C','O','N',
	0x99,	'S','B','U','F',
	0x87,	'P','C','O','N',
	0 };

/* 8250 uart configuration tables */
unsigned baud[] = { 384, 384, 96, 48, 24, 12, 4, 0 };
char baud_rate = 0, parity = 4, dbits = 3, sbits = 0;

/* Command line options */
char filename[50] = { ".HEX" };
unsigned hbaud = 4, com1 = 0x3FD, com2 = 0x2FD, adelay = 0, bdelay = 1000;
char reset = -1;

int *attributes,		/* Pointer to active attribute set */
	Cattrs[] = {		/* Attributes for color monitors */
		WSAVE|WBOX1|WCOPEN|0x17,					/* Disassembler window */
		WSAVE|WBOX1|WCOPEN|0x57,					/* Register window */
		WSAVE|WBOX1|WCOPEN|0x47,					/* Internal RAM window */
		WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|0x30,		/* Message window */
		WSAVE|WBOX2|WCOPEN|0x67 },					/* Edit window(s) */
	Mattrs[] = {		/* Attributes for monochrome monitors */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Disassembler window */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Register window */
		WSAVE|WBOX1|WCOPEN|NORMAL,					/* Internal RAM window */
		WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|NORMAL,	/* Message window */
		WSAVE|WBOX2|WCOPEN|NORMAL };				/* Edit window(s) */


/* Misc. flags & variables */
char fflag = 0, mflag = 0, ds5000 = 0;
unsigned char ds5000_mcon = 0xF0;
unsigned LOAD_PC = 0, BRKCNT[B_MASK+1];

/* 8051 Monitor variables */
unsigned char A=0, B=0, PSW=0, SP=7, INTREG[32];
unsigned int DPTR=0, PC=0x0800, BRKPTS[NUMBRK+1];
unsigned char BREAK_SAVE[NUMBRK][3], BRKFLG[NUMBRK];

/* Window/Display control variables */
char *Dwin, *Rwin, *Iwin, *Mwin;

/* Disassembler variables */
unsigned int Dpos, Dhigh, Daddr, Rpos = 0;
unsigned char Dbuffer[64];

/* Memory editor argument strings */
char *Iargs[] = { 0x0000, 0x00FF, 0, "Internal memory" };
char *Dargs[] = { 0x0000, 0xFFFF, 0, "Data memory" };
char *Cargs[] = { 0x0800, 0xFFFF, 0, "Program memory" };

/* Register modification tables */
char Rchars[] = { "01234567ABDSWP" };
char *Rnames[] = {
	"R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7",
	"A", "B", "DPTR", "SP", "PSW", "PC" };

/* Help screens */
char *mhelp[] = {		/* Main monitor help */
	"Monitor commands",
	"A    - Animate execution       F2   - Set disassembler address",
	"B    - set a Breakpoint        F3   - Setup serial port",
	"C    - Change a register       F4   - Breakpoint counters",
	"D    - edit Data memory        F5   - Reset processor",
	"F    - Function registers      F6   - Reset hardware only",
	"G    - Go: execute program     F7   - Setup DS5000 options",
	"I    - edit Internal memory    F10  - Shell to DOS",
	"J    - Jump to PC display      Space- Step one instruction",
	"K    - Kill all breakpoints    Up   - Backup  IRAM one line",
	"L    - Load program file       Down - Advance IRAM one line",
	"N    - Next code page          PgUp - Backup  IRAM one page",
	"P    - edit Program memory     PgDn - Advance IRAM one page",
	"R    - Remove a breakpoint     Esc  - Terminate MONICA52",
	"U    - scroll code page Up",
	0 };

char *ehelp[] = {		/* Ram editor help */
	"Editor commands",
	"F2       - Set edit address",
	"F3       - Toggle HEX/ASCII edit",
	"F4       - Block fill memory",
	"F5       - Refresh display from target",
	"Left     - Backup  cursor  one byte",
	"Right    - Advance cursor  one byte",
	"Up       - Backup  cursor  one line",
	"Down     - Advance cursor  one line",
	"PgUp     - Backup  display one page",
	"PgDn     - Advance display one page",
	"Home     - Backup  to start of line",
	"End      - Advance to end of line",
	"Ctl-PgUp - Position to start of memory",
	"Ctl-PgDn - Position to end of memory",
	"Esc      - Terminate EDIT",
	0 };

char *chelp[] = {		/* Register change help */
	"Change commands",
	"A   - change A accumulator    (A)",
	"B   - change B accumulator    (B)",
	"D   - change Data pointer     (DPTR)",
	"S   - change Stack poiner     (SP)",
	"P   - change Program counter  (PC)",
	"W   - change processor status Word (PSW)",
	"0-7 - change general register (Rn)",
	"Esc - Abort change command",
	0 };

char *shelp[] = {		/* String input help */
	"String input",
	"Left - Move cursor left",
	"Right- Move cursor right",
	"Bs   - Backspace & delete",
	"Del  - Delete character under cursor",
	"Ins  - Toggle Insert/Overwrite mode",
	"Home - Move Cursor to start of string",
	"End  - Move cursor to end of string",
	"PgUp - Clear entire field",
	"PgDn - Clear from cursor to end of field",
	"Enter- Accept entry",
	"Esc  - Abort input",
	0 };

char *ds5000_help[] = {	/* DS5000 setup help */
	"DS5000 Setup",
	"Up    - Increase data address",
	"Down  - Decrease data address",
	"F2    - Toggle CE2 ON/OFF",
	"Esc   - Exit",
	0 };

char *welcome[] = {		/* Startup message */
	">>> MONICA-52 <<<",
	"",
	"A PC hosted 8051/52 Monitor.",
	"",
	"Release 1.7",
	"",
	"Use: MONICA52 [filename] [/DS5000 /Hardware:baud /Mono /Reset /Swapcom]",
	"",
	"?COPY.TXT 1991-2005 Dave Dunfield",
	"**See COPY.TXT**.",
	0 }

/* Go menu */
char *gomenu[] = {
	"Reset target",
	"Halt at break",
	"Send ESCAPE",
	"Zoom window",
	0 }

char *count_form[] = {
	21<<8|6,
	"\x01\x00\x85Counter 1 :",
	"\x01\x01\x85Counter 2 :",
	"\x01\x02\x85Counter 3 :",
	"\x01\x03\x85Delay (ms):",
	0 };

/*
 * Main program - Initialize and present main windows
 */
main(argc, argv)
	int argc;
	char *argv[];
{
	unsigned int c, i, j, k;
	char *ptr;

/* Process command line parameters */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case '/D' :		/* DS5000 mode */
			case '-D' :
				ds5000 = -1;
				break;
			case '/M' :		/* Force monochrome */
			case '-M' :
				mflag = -1;
				break;
			case '/H' :		/* Hardware baudrate */
			case '-H' :
				while(*ptr) {
					if(*ptr++ == ':') {
						hbaud = scale(57600, 2, atoi(ptr));
						break; } }
				break;
			case '/R' :		/* Reverse reset */
			case '-R' :
				reset = !reset;
				break;
			case '/S' :		/* Swap COMM ports */
			case '-S' :
				c = com1;
				com1 = com2;
				com2 = c;
				break;
			default:
				fflag = -1;
				strcpy(filename, argv[i]); } }

/* Open the welcome screen & output startup message */
	wopen(0, 0, 80, 25, WSAVE|WBOX2|WCOPEN|REVERSE);
	wcursor_off();
	for(i=0; ptr = welcome[i]; ++i) {
		wgotoxy((78-strlen(ptr))/2, i+6);
		wputs(ptr); }

	attributes = (mflag || (W_BASE == 0xB000)) ? Mattrs : Cattrs;
	init_serial();

/* Initialize internal memory & SFR's
	for(i=0; i < sizeof(INTREG); ++i)
		INTREG[i] = 0; */
/* Clear breakpoints
	for(i=0; i < (NUMBRK+1); ++i)
		BRKPTS[i] = 0; */

	if(!(c = COMopen(com2, hbaud, PAR_NO|DATA_8|STOP_1, SET_DTR|OUTPUT_2)))
		reset_target(0);		/* Reset the system */

	i = 10;
	do
		short_pause(9);
	while(--i && !wtstc());

	wclose();

	if(c)
		abort("Cannot access serial port!");

/* Draw the working menus */
	Dwin = wopen(0,  0, 45, 18, attributes[DATTR]);
	Rwin = wopen(45, 0, 16, 18, attributes[RATTR]);
	Iwin = wopen(61, 0, 19, 18, attributes[IATTR]);
	Mwin = wopen(0, 18, 80,  7, attributes[MATTR]);
	wcursor_off();

	for(;;) {
		read_target();
		write_target(PODIDLE);
		wcleol();
		if(!read_target()) {
			write_target(PODIDEN);
			if(read_target() == PODID)
					break; }
		wputs("Target system is not responding!!!\r");
		if(wtstc() == 0x1B)
			goto quit; }

	wputs("Communication with Target established");
	if(ds5000)
		wputs(", DS5000 mode enabled");

/* Load file & setup PC */
	if(fflag)
		load_file();
	Daddr = (Dpos = PC) - 1;

/* Prompt for and get commands */
	show_dis();
	update();
	wputs("\nPress F1 for help");
	for(;;) {
		switch(c = toupper(wgetc())) {
		case 'A' :	/* Animate */
			i = -1;
			if((c = get_number("Animate how many instructions (CR=Non-stop)? ", 10, &i)) != 0x1B) {
				wprintf("\nAnimating at %04x", PC);
				while(i) {
					if(!c)
						--i;
					if(single_step())
						goto stop1;
					if(wtstc() == 0x1B)
						goto stop;
					if(j = isbreak(PC)) {
						if(!handle_breakpoint(j))
							break; }
					if((j & (B_ENAB|B_WATCH)) != (B_ENAB|B_WATCH)) {
						show_pc();
						update(); }
					if(adelay)
						delay(adelay); }
				goto stop; }
			continue;
		case 'B' :	/* Set breakpoint */
			wputc('\n');
			show_brks();
			j = PC;
			if((get_number("Breakpoint at (CR=PC)? ", 16, &j) != 0x1B) && j) {
				for(i=0; i < NUMBRK; ++i) {
					if((k = BRKPTS[i]) == j) {
						wputs("Exists!");
						BRKFLG[i] = get_bflags(BRKFLG[i]);
						show_brks();
						goto sbrk; }
					if(!k) {
						BRKPTS[i] = j;
						BRKFLG[i] = get_bflags(B_SET|B_ENAB);
						show_brks();
						goto sbrk; }
					if(((k-2) <= j) && ((k+3) > j)) {
						wputs("Conflict!");
						goto sbrk; } }
					wputs("All used!");
			sbrk:
				show_dis(); }
			continue;
		case 'C' :	/* Change register */
			change_reg();
			break;
		case 'D' :	/* Data memory */
			edit(1, Dargs);
			show_dis();
			break;
		case 'F' :	/* Special function registers */
			edit_sfr(sfrs+SFROFF);
			break;
		case 'I' :	/* Internal memory */
			edit(0, Iargs);
			break;
		case ' ' :	/* Step program */
			single_step();
		case 'J' :	/* Jump to PC */
			show_pc();
			break;
		case 'K' :	/* Kill ALL breakpoints */
			if(yesno("Kill all breakpoints")) {
				memset(BRKPTS, 0, sizeof(BRKPTS));
				wputc('\n');
				show_brks();
				show_dis(); }
			continue;
		case 'L' :	/* Load file */
			wopen(10, 10, 60, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
			wputs("Load file? ");
again:		if((i=wgets(11, 0, filename, 49)) == '\n') {
				wclose();
				load_file();
				Daddr = Dpos - 1;
				show_pc();
				break; }
			if(i == _K1) {
				help(shelp);
				goto again; }
			wclose();
			break;
		case 'N' :	/* Next page */
			Dpos = Dhigh;
			show_dis();
			continue;
		case 'U' :	/* Scroll Up */
			scroll_up();
			show_dis();
			continue;
		case 'P' :	/* Program memory */
			edit(2, Cargs);
			show_dis();
			break;
		case 'R' :	/* Remove breakpoint */
			wputc('\n');
			show_brks();
			c = PC;
			if(get_number("Remove breakpoint at (CR=PC)? ", 16, &c) != 0x1B) {
				for(i=0; i < NUMBRK; ++i) {
					if(BRKPTS[i] == c) {
						while(i < NUMBRK) {
							BRKFLG[i] = BRKFLG[i+1];
							BRKPTS[i] = BRKPTS[++i]; } } }
				show_brks();
				show_dis(); }
			continue;
		case 'G' :	/* Go */
			if(!yesno("Begin execution"))
				continue;
			mflag = -1;
			c = 0;
		go:
			while(install_breakpoints()) {
				if(!c) {
					wprintf("\nStepping over breakpoint at %04x", PC);
					if(wtstc() == 0x1B) {
						remove_breakpoints();
						goto stop1; } }
				single_step(); }
			if(!c)
				wprintf("\nExecuting at %04x", PC);
			write_registers(PODEXEC);
			if(baud_rate ? go_remote() : go_local()) {
				reset_target(RESET_DISPLAY);
				remove_breakpoints();
				goto stop1; }
			read_registers();
			remove_breakpoints();
			if(c = isbreak(PC-3)) {
				PC -= 3;
				if(handle_breakpoint(c) && mflag)
					goto go; }
		stop:
			wprintf(", Stopped at %04x", PC);
		stop1:
			wcursor_off();
			show_pc();
			break;
		case _K1 :	/* Help */
			help(mhelp);
			continue;
		case _K2 :	/* Show */
			get_number("Code address? ", 16, &Dpos);
			show_dis();
			continue;
		case _K3:	/* Setup serial */
			setup_serial();
			continue;
		case _K4 :	/* Breakpoint counters */
			wform(30, 10, WSAVE|WBOX2|WCOPEN|REVERSE, count_form,
				&BRKCNT[1], &BRKCNT[2], &BRKCNT[3], &bdelay);
			continue;
		case _K5 :	/* Reset CPU */
			reset_target(RESET_PROMPT|RESET_DISPLAY|RESET_CPU);
			show_pc();
			break;
		case _K6 :	/* Reset hardware */
			reset_target(RESET_PROMPT|RESET_DISPLAY);
			break;
		case _K7:	/* DS5000 */
			setup_ds5000();
			break;
		case _K10:	/* Shell */
			doshell();
			break;
		case _KUA :	Rpos -= 4;	break;
		case _KDA :	Rpos += 4;	break;
		case _KPU : Rpos -= 64;	break;
		case _KPD :	Rpos += 64;	break;
		case 0x1B :	/* Exit */
			if(yesno("Terminate MONICA52")) {
		quit:	wclose();
				wclose();
				wclose();
				wclose();
				COMclose();
				return; } }
		update(); }
}

/*
 * Display a disassembler address
 */
show_dis()
{
	unsigned int a, b, i, j, k, at, ra;
	char buffer[50];

	if(Dpos != Daddr)
		read_program(Daddr = Dpos, 64, Dbuffer);

	a = Dpos;
	b = 0;
	ra = reverse(at = attributes[DATTR]);
	for(i=0; i < 16; ++i) {
		w_gotoxy(0, i, Dwin);
		*Dwin = (a == PC) ? ra : at;
		j = disass(a, Dbuffer+b, buffer);
		w_printf(Dwin, "%04x", Dhigh = a);
		for(k=0; k < 3; ++k)
			w_printf(Dwin, k < j ? " %02x" : "   ", Dbuffer[b+k]);
		w_printf(Dwin, "   %-26s%c", buffer, show_breakpoint(isbreak(a)));
		a += j;
		b += j; }
}

/*
 * Calculate higher address
 */
scroll_up()
{
	unsigned a, b, i, j, addr[48];
	char buffer[50];

	b = i = a = 0;
	if(Dpos > 48)
		a = Dpos - 48;
	read_program(Daddr = a, 64, Dbuffer);

	do {
		addr[i++] = a;
		a += (j = disass(a, Dbuffer + b, buffer));
		b += j; }
	while(a < Dpos);
	Dpos = (i > 14) ? addr[i-15] : 0;
}

/*
 * Show DIS at PC
 */
show_pc()
{
	int a, b, i, j;
	char buffer[50];

	a = Dpos;
	b = 0;
	for(i=0; i < 16; ++i) {
		if(a == PC) {
			show_dis();
			return; }
		j = disass(a, Dbuffer+b, buffer);
		a += j;
		b += j; }
	Dpos = PC;
	show_dis();
}

/*
 * Update the register & RAM display
 */
update()
{
	int i, j;
	unsigned char sp, r0, r1, dptr, intbuf[64];

	read_internal(INTREG[j = PSW & 0x18], 1, &r0);
	read_internal(INTREG[j+1], 1, &r1);
	read_internal(SP, 1, &sp);
	read_external(DPTR, 1, &dptr);
	read_internal(Rpos, 64, intbuf);

	w_gotoxy(0, 1, Rwin);
	w_printf(Rwin, "A   :%02x\nB   :%02x\nDPTR:%04x [%02x]\nSP  :%02x   [%02x]\nPSW :%02x",
		A, B, DPTR, dptr, SP, sp, PSW);
	w_printf(Rwin, " %c%c%c%c%c%c",
		PSW & 0x80 ? 'C' : '-',
		PSW & 0x40 ? 'A' : '-',
		PSW & 0x20 ? 'F' : '-',
		((PSW & 0x18) >> 3) + '0',
		PSW & 0x04 ? 'O' : '-',
		PSW & 0x01 ? 'P' : '-');
	w_printf(Rwin, "\nPC  :%04x\nR0  :%02x   [%02x]\nR1  :%02x   [%02x]", PC,
		INTREG[j], r0, INTREG[j+1], r1);
	for(i=2; i < 8; ++i)
		w_printf(Rwin, "\nR%u  :%02x", i, INTREG[i+j]);

	w_gotoxy(0, 0, Iwin);
	for(i=0; i < 64; ++i) {
		if(!(i%4))
			w_printf(Iwin, "%s%04x:", i ? "\n": "", (Rpos+i) & 255);
		w_printf(Iwin, " %02x", intbuf[i]); }
}

/*
 * Edit global memory 
 */
edit(type, args)
	unsigned int type, args[];
{
	unsigned int dt, dc, i, j;
	unsigned char buffer[256];

	wopen(3, 3, 73, 18, attributes[EATTR]);
	title(args[3]);
recursor:
	wcursor_line();
redraw:
	dt = *args & 0xFF00;
	switch(type) {
		case 0 : read_internal(dt, 256, buffer);	break;
		case 1 : read_external(dt, 256, buffer);	break;
		case 2 : read_program(dt, 256, buffer); }

	for(i=j=0; i < 16; ++i, j += 16) {
		wgotoxy(0, i);
		draw_line(dt+j, buffer+j); }

prompt:
	if(*args > args[1])
		*args &= 0x00FF;
	if((*args & 0xFF00) != dt)
		goto redraw;
	dc = *args & 0xFF;
	wgotoxy(args[2] ? (dc%16)+55 : ((dc%16)*3)+6, dc/16);
	switch(toupper(i = wgetc())) {
		case 0x1B :		/* Exit editor */
			wclose();
			return;
		case _K1 :		/* Help request */
			help(ehelp);
			break;
		case _K2 :		/* Go to address */
			get_number("Edit address? ", 16, args);
			break;
		case _K3 :		/* Toggle mode */
			args[2] = !args[2];
			break;
		case _K4 :		/* Fill memory */
			if(get_number("Starting address? ", 16, &dc))
				break;
			if(dc > args[1])
				break;
			if(get_number("Ending address? ", 16, &dt))
				break;
			if((dt > args[1]) || (dt < dc))
				break;
			if(get_number("Fill with? ", 16, &i))
				break;
			wcursor_off();
			for(j=0; j < sizeof(buffer); ++j)
				buffer[j] = i;
			do {
				if((i = dt - dc) > 254)
					i = 254;
				++i;
				switch(type) {
					case 0 :	write_internal(dc, i, buffer);	break;
					case 1 :	write_external(dc, i, buffer);
								Daddr = Dpos - 1; break;
					case 2 :	write_program(dc, i, buffer);
								Daddr = Dpos - 1; } }
			while(((dc += i)-1) < dt);
		case _K5 :		/* Refresh screen */
			goto recursor;
		case _KLA :
			--*args;
			break;
		case _KPU :
			*args -= 256;
			break;
		default:
			if(args[2]) {
				if(i >= 128)
					break;
				buffer[dc] = i; }
			else {
				if((j = gethex(i)) >= 16)
					break;
				wputc(i);
				if((i = gethex(wgetc())) < 16)
					buffer[dc] = (j<<4)+i; }
			switch(type) {
				case 0 :	write_internal(*args, 1, &buffer[dc]);	break;
				case 1 :	write_external(*args, 1, &buffer[dc]);
							Daddr = Dpos - 1; break;
				case 2 :	write_program(*args, 1, &buffer[dc]);
							Daddr = Dpos - 1; }
			wgotoxy(0, i = dc/16);
			i *= 16;
			draw_line(dt+i, buffer+i);
		case _KRA :
			++*args;
			break;
		case _KPD :
			*args += 256;
			break;
		case _KHO :
			*args = (*args-1) & 0xFFF0;
			break;
		case _KUA :
			*args -= 16;
			break;
		case _KEN :
			*args = (*args+1) | 0x000F;
			break;
		case _KDA :
			*args += 16;
			break;
		case _CPU :
			*args = 0;
			break;
		case _CPD :
			*args = args[1]; }
	goto prompt;
}

/*
 * Display a line for the editor
 */
draw_line(addr, buffer)
	int addr;
	unsigned char buffer[];
{
	unsigned int i, c;

	wprintf("%04x  ", addr);
	for(i=0; i < 16; ++i)
		wprintf("%02x ", buffer[i]);
	wputc(' ');
	for(i=0; i < 16; ++i) {
		c = buffer[i];
		wputc(((c >= ' ') && (c <= 0x7E)) ? c : '.'); }
}

/* Test for HEX digit & get value */
gethex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	return -1;
}

/*
 * Display a title on a window
 */
title(ptr)
	char *ptr;
{
	char buffer[8];
	memcpy(buffer, W_OPEN, sizeof(buffer));
	*W_OPEN = REVERSE;
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, sizeof(buffer));
}

/*
 * Get an numeric value at a prompt
 */
get_number(prompt, base, var)
	char *prompt;
	int base, *var;
{
	unsigned int i, j, x, value, len;

	i = (len = (base == 10) ? 5 : 4) + strlen(prompt) + 3;
	wopen((80-i)/2, 10, i, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wputs(prompt);
	wcursor_line();
	value = x = 0;
	for(;;) {
		switch(i = wgetc()) {
			case '\n' :
				if(x) {
					*var = value;
					wclose();
					return 0; }
			case 0x1B :
				wclose();
				return i;
			case _KBS :
			case _KLA :
				if(x) {
					--x;
					wputs("\b \b");
					value /= base; }
				break;
			default:
				if(isdigit(j = toupper(i)))		/* convert numeric digits */
					j -= '0';
				else if(j >= 'A')				/* convert alphabetics */
					j -= ('A' - 10);
				else							/* not a valid "digit" */
					break;
				if((x < len) && (j < base)) {
					wputc(i);
					++x;
					value = (value * base) + j; } } }
}

/*
 * Change register contents
 */
change_reg()
{
	int c, i;
	char buffer[50];

	wopen(30, 10, 20, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wputs("Change register? ");
prompt:
	if((c = toupper(wgetc())) == 0x1B) {
		wclose();
		return; }
	if(c == _K1) {
		help(chelp);
		goto prompt; }
	for(i=0; Rchars[i] != c; ++i)
		if(!Rchars[i])
			goto prompt;
	wclose();
	sprintf(buffer, "New value for %s? ", Rnames[i]);
	if(!get_number(buffer, 16, &c))	switch(i) {
		case 8 : A = c;				break;
		case 9 : B = c;				break;
		case 10: DPTR = c;			break;
		case 11: SP = c;			break;
		case 12: PSW = c;			break;
		case 13: PC = c; show_pc();	break;
		default: INTREG[(PSW & 0x18)+i] = c; }
}

char *lptr;	/* Global ptr to load arguments */
/*
 * Read file into memory image
 */
load_file()
{
	unsigned a, b, chksum, addr, length, count, size, i;
	char buffer[80], databuf[256];
	FILE *fp;

	w_printf(Mwin, "\n%s: ", filename);
	if(!(fp = fopen(filename,"r"))) {
		w_puts("Unable to access", Mwin);
		return; }

	for(size = 0; fgets(lptr = buffer, 80, fp);) {
		again: switch(*lptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*lptr == '9') goto quitload;
				if(*lptr++ != '1') continue;
				length = count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				addr = (a << 8) + b;
				if(!size)
					LOAD_PC = PC = addr;
				for(i=0; i < count; ++i)
					chksum += databuf[i] = get_byte();
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				write_program(addr, count, databuf);
				addr += count;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = get_byte())) goto quitload;
				chksum = (a = get_byte()) + length;
				chksum += b = get_byte();
				addr = (a << 8) + b;
				chksum += get_byte();
				if(!size)
					LOAD_PC = PC = addr;
				for(i=0; i < count; ++i)
					chksum += databuf[i] = get_byte();
				if((255 & -chksum) != get_byte())
					goto quitbad;
				write_program(addr, count, databuf);
				addr += count;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				w_puts("Invalid record format", Mwin);
				goto quit; }
		size += length; }

quitload:
	w_printf(Mwin, "%u bytes loaded", size);
	goto quit;
quitbad:
	w_puts("Bad checksum", Mwin);
quit:
	fclose(fp);
	show_pc();
}

/* Load a byte from the hex file */
get_byte()
{
	return (gethex(*lptr++) << 4) + gethex(*lptr++);
}

/*
 * Display a help screen
 */
help(ptr)
	char *ptr[];
{
	int i;
	for(i=1; ptr[i]; ++i);
	wopen(7, (24-i)/2, 66, i+1, WSAVE|WBOX2|WCOPEN|REVERSE);
	title(ptr[0]);
	wcursor_off();
	for(i=1; ptr[i]; ++i) {
		wgotoxy(1, i-1);
		wputs(ptr[i]); }
	while(wgetc() != 0x1B);
	wclose();
}

/*
 * Initialize uart after allowing change to parameters
 */
setup_serial()
{
	unsigned b;
	char buffer[6];
	static char *partab[] = { "Odd", "Even", "Mark", "Space", "None", 0 };

	wopen(30, 10, 20, 6, attributes[EATTR]);
	title("COMM Setup");
	for(;;) {
		wgotoxy(0, 0);
		sprintf(buffer, "%u", scale(57600, 2, b = baud[baud_rate]));
		wprintf("(B)audrate:  %5s\n", baud_rate ? buffer : "Local");
		wprintf("(P)arity:    %5s\n", partab[parity]);
		wprintf("(D)ata bits: %5u\n", dbits+5);
		wprintf("(S)top bits: %5u", sbits+1);
		switch(toupper(wgetc())) {
			case _KUA :
				--b;
				goto setbaud;
			case _KDA :
				++b;
			setbaud:
				if(baud_rate)
					baud[baud_rate] = (b < 2) ? 2 : b;
				break;
			case 'B' :
				if(!baud[++baud_rate])
					baud_rate = 0;
				break;
			case 'P' :
				if(!partab[++parity])
					parity = 0;
				break;
			case 'D' :
				if(++dbits > 3)
					dbits = 0;
				break;
			case 'S' :
				if(++sbits > 1)
					sbits = 0;
				break;
			case 0x1B:
				init_serial();
				wclose();
				return; } }
}

/*
 * Initialize PC serial port for simulated processor
 */
init_serial()
{
	unsigned i, r;

	i = ((parity << 4) & 0x30)	|	/* parity bits */
		(dbits & 0x03)			|	/* # data bits */
		((sbits << 2) & 0x04)	|	/* # stop bits */
		((parity < 4) << 3) ;
	out(com1-2, i | 0x80);
	out(com1-5, r = baud[baud_rate]);
	out(com1-4, r >> 8);
	out(com1-2, i);
	out(com1-4, 0);	/* disable all interrupts */
	out(com1-1, 3);	/* Enable DTR and RTS */
}

/*
 * Display the breakpoints
 */
show_brks()
{
	int i, j;

	w_puts("\rBreak: ", Mwin);
	for(i=0; BRKPTS[i]; ++i) {
		if((j = show_breakpoint(BRKFLG[i])) != '*')
			w_putc(j, Mwin);
		w_printf(Mwin, "%04x ", BRKPTS[i]); }
	w_cleol(Mwin);
	return i;
}

/*
 * Edit the SFR's
 */
edit_sfr(table)
	char *table;
{
	unsigned i, j, c, n;
	unsigned char buffer[10], buffer1[50], *ptr, *ptr1;
	static int bank = 0;

	n = 0;
	for(ptr=table; *ptr; ++ptr)
		if(*ptr & 0x80)
			++n;
redraw:
	wopen(6, 1, ((((n-1)/20)+1)*17)+2, 22, attributes[EATTR]);
	title("SFRs");
	for(;;) {
		ptr = table;
		i = 0;
		do {
			wgotoxy(((i/20)*17)+1, i % 20);
			j = *ptr++;
			for(ptr1 = buffer; (c = *ptr) && !(c & 0x80); ++ptr)
				*ptr1++ = c;
			*ptr1 = 0;
			wprintf((i/20 == bank) ? "%c:" : "  ", (i%20)+'A');
			wprintf(" %-8s -%02x", buffer, read_direct(j));
			++i; }
		while(*ptr);

		if((j = toupper(wgetc())) == 0x1B) {
			wclose();
			return; }

		if(j == ' ')
			bank = (bank + 1) % ((i+19)/20);
		if((j < 'A') || (j > 'T'))
			continue;

		i = (j - 'A') + (bank * 20);
		ptr = table;
		while(*ptr) {
			j = *ptr++;
			for(ptr1 = buffer; (c = *ptr) && !(c & 0x80); ++ptr)
				*ptr1++ = c;
			*ptr1 = 0;
			if(!i--) {
				sprintf(buffer1, "New value for %s? ", buffer);
				if(!get_number(buffer1, 16, &c)) {
					wclose();
					write_direct(j, c);
					goto redraw; }
				break; } } }
}

/*
 * Setup the DS5000 chip options
 */
setup_ds5000()
{
	int ram, ce2;

	ram = ds5000_mcon & 0xF0;
	ce2 = ds5000_mcon & 0x04;

	wopen(25, 10, 30, 4, attributes[EATTR]);
	wcursor_off();
	if(!ds5000) {
		wputs("DS5000 option not active!\nPress any key to exit.");
		wgetc();
		wclose();
		return; }
	title("DS5000 Setup");
	for(;;) {
		wgotoxy(0, 0);
		wprintf("Data memory   : %04x\n", (ram + ((ram == 0xF0) ? 0x10 : 0)) * 128);
		wprintf("CE2 (32k data): %sabled ", ce2 ? "En" : "Dis");
		switch(toupper(wgetc())) {
			case _K1 :
				help(ds5000_help);
				continue;
			case _KUA :
				if((ram += 0x10) > 0xF0)
					ram = 0x10;
				continue;
			case _KDA :
				if((ram -= 0x10) <= 0)
					ram = 0xF0;
				continue;
			case _K2 :
				ce2 ^= 0x04;
			default :
				continue;
			case 0x1B : }
		wclose();
		ds5000_mcon = ram | ce2;
		write_target(PODDS5);
		write_target(ds5000_mcon);
		return; }
}

/*
 * Reverse color attributes
 */
reverse(attr)
	int attr;
{
	return (attr & 0xff00) | ((attr & 0xf0) >> 4) | ((attr & 0x0f) << 4);
}

/*
 * Prompt for YES/NO response
 */
yesno(ptr)
	char *ptr;
{
	int i;

	i = strlen(ptr)+6;
	wopen((80-i)/2, 10, i, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wprintf("%s ? Y", ptr);
	for(;;) switch(toupper(wgetc())) {
		case 'N' :
		case 0x1B :
			wclose();
			return 0;
		case '\n' :
		case 'Y' :
			wclose();
			return 1; }
}

/*
 * Test for breakpoints
 */
isbreak(addr)
	unsigned addr;
{
	unsigned i, a;

	for(i=0; a = BRKPTS[i]; ++i) {
		if(a == addr)
			return BRKFLG[i]; }
	return 0;
}

/*
 * Single-Step one instruction
 */
single_step()
{
	unsigned char c1, c2, c3;
	c2 = c3 = 0;
	write_registers(PODSTEP);
	do {
		c1 = c2;
		c2 = c3;
		if(!(c3 = read_target())) {
			if(reset_target(RESET_PROMPT|RESET_DISPLAY))
				return -1; } }
	while((c1 != PODEID1) || (c2 != PODEID2) || (c3 != PODEID3));

	read_registers();
	return 0;
}

/*
 * Install breakpoints into the user program
 */
install_breakpoints()
{
	unsigned i, j;
	for(i=0; (i < NUMBRK) && (j = BRKPTS[i]); ++i) {
		if((j <= PC) && ((j+3) > PC))
			return -1;
		read_program(j, 3, BREAK_SAVE[i]); }
#ifdef DEBUG
w_printf(Mwin,"\nSet brkpt:");
#endif
	set_mode(1);		/* Access program storage */
	for(i=0; (i < NUMBRK) && (j = BRKPTS[i]); ++i) {
		if(BRKFLG[i] & B_ENAB) {
			write_target(PODBRK);
			write_address(j); } }
	return 0;
}

/*
 * Remove breakpoints from the user program
 */
remove_breakpoints()
{
	unsigned i, j;

	set_mode(1);	/* Access program storage */
	for(i=0; (i < NUMBRK) && (j = BRKPTS[i]); ++i)
		if(BRKFLG[i] & B_ENAB)
			write_program(j, 3, BREAK_SAVE[i]);
}

/*
 * Scale to a fractional value
 */
scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Read a byte from the target system
 */
read_target() asm
{
		MOV		AX,40h			; Address BIOS data area
		MOV		ES,AX			; Point to it
		MOV		CX,18			; Timeout in ticks
readt1:	MOV		DX,ES:6Ch		; Point to 
readt2:	CALL	_COMtestc		; Character ready?
		CMP		AX,-1			; If so, return with it
		JNZ		readt3			; Yes, exit
		CMP		DX,ES:6Ch		; Tick passed?
		JZ		readt2			; No, keep trying
		LOOP	readt1			; Do all ticks
		XOR		AX,AX			; Return ZERO on timeout
readt3:
}

/*
 * POD interface routines
 */
/* Read direct address */
read_direct(addr)
	unsigned addr;
{
#ifdef DEBUG
w_printf(Mwin,"\nRead direct: %04x", addr);
#endif
	write_target(addr | 0x80);
	return read_target();
}

/* Write direct address */
write_direct(addr, data)
	unsigned addr, data;
{
#ifdef DEBUG
w_printf(Mwin,"\nWrite direct: %04x = %02x", addr, data);
#endif
	write_target(addr & 0x7F);
	write_target(data);
}

/* Read internal memory */
read_internal(addr, size, ptr)
	unsigned char addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nRead internal: %04x %u %04x", addr, size, ptr);
#endif
	while(addr < 0x20) {
		*ptr++ = INTREG[addr++];
		if(!--size)
			return; }
	write_target(PODRI);
	write_target(size);
	write_target(addr);
	do
		*ptr++ = read_target();
	while(--size);
}

/* Write internal memory */
write_internal(addr, size, ptr)
	unsigned char addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nWrite internal: %04x %u %04x", addr, size, ptr);
#endif
	while(addr < 32) {
		INTREG[addr++] = *ptr++;
		if(!--size)
			return; }

	write_target(PODWI);
	write_target(size);
	write_target(addr);
	do
		write_target(*ptr++);
	while(--size);
}

/* Read external memory */
read_external(addr, size, ptr)
	unsigned addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nRead external: %04x %u %04x", addr, size, ptr);
#endif
	set_mode(0);	/* Access user data */
	write_target(PODRE);
	write_target(size);
	write_address(addr);
	do
		*ptr++ = read_target();
	while(--size);
}

/* Write external memory */
write_external(addr, size, ptr)
	unsigned addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nWrite external: %04x %u %04x", addr, size, ptr);
#endif
	set_mode(0);	/* Access user data */
	write_target(PODWE);
	write_target(size);
	write_address(addr);
	do
		write_target(*ptr++);
	while(--size);
}

/* Read program memory */
read_program(addr, size, ptr)
	unsigned addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nRead program: %04x %u %04x", addr, size, ptr);
#endif
	set_mode(0);	/* Read in user mode */
	write_target(PODRP);
	write_target(size);
	write_address(addr);
	do
		*ptr++ = read_target();
	while(--size);
}

/* Write program memory */
write_program(addr, size, ptr)
	unsigned addr;
	unsigned char size, *ptr;
{
#ifdef DEBUG
w_printf(Mwin,"\nWrite program: %04x %u %04x", addr, size, ptr);
#endif
	set_mode(1);		/* Access program storage */
	write_target(PODWP);
	write_target(size);
	write_address(addr);
	do
		write_target(*ptr++);
	while(--size);
}

/* Write address to system */
write_address(addr)
	unsigned addr;
{
	write_target(addr & 255);
	write_target(addr >> 8);
}

/*
 * Write registers to target
 */
write_registers(cmd)
	unsigned cmd;
{
	int i;
#ifdef DEBUG
w_printf(Mwin,"\nWrite regs: %02x", cmd);
#endif
	set_mode(0);		/* Run in user mode */
	write_target(cmd);
	for(i=0; i < 32; ++i)
		write_target(INTREG[i]);
	write_target(SP);
	write_address(PC);
	write_target(A);
	write_target(B);
	write_address(DPTR);
	write_target(PSW);
}

/*
 * Load registers from target
 */
read_registers()
{
	unsigned i;
#ifdef DEBUG
w_printf(Mwin,"\nRead regs:");
#endif
	PSW = read_target();
	A = read_target();
	B = read_target();
	i = read_target();
	PC = read_target() + (i << 8);
	i = read_target();
	DPTR = read_target() + (i << 8);
	SP = read_target();
	for(i=0; i < 32; ++i)
		INTREG[i] = read_target();
}

/*
 * Reset the target system
 */
reset_target(prompt)
	char prompt;
{
	int i;
	if(prompt & RESET_PROMPT) {
		if(!yesno((prompt & RESET_CPU) ? "Reset" : "Reset hardware"))
			return 0; }
	set_mode(2);	/* reset */
	short_pause(3);
	set_mode(0);	/* Normal */
	short_pause(3);
	COMopen(com2, hbaud, PAR_NO|DATA_8|STOP_1, (reset && SET_RTS)|(SET_DTR|OUTPUT_2));
	for(i=5000; read_target() && i; --i);

	if(prompt & RESET_CPU) {
		A = B = PSW = DPTR = 0;
		SP = 7;
		PC = LOAD_PC; }

	if(prompt)
		w_puts("\nRESET!", Mwin);
	return -1;
}

/*
 * Set up the board access mode
 */
char sysmode = 0;
set_mode(mode)
	char mode;
{ asm {	/* Mode: 0=Pseudo-ROM, 1=Standard, 2=Reset */
		EXTRN	_COMa:WORD
		MOV		BL,4[BP]		; Get system mode
		CMP		BL,DGRP:_sysmode; Same?
		JNZ		setm0			; No, handle it
		MOV		SP,BP			; Clean stack
		POP		BP				; Restore caller
		RET
setm0:	MOV		DGRP:_sysmode,BL; Set new mode
; Wait for transmit character to shift out
		MOV		DX,DGRP:_COMa	; Get COM address
setm1:	IN		AL,DX			; Read status
		TEST	AL,40h			; Tx shifted out?
		JZ		setm1			; No, its not
		MOV		AX,2			; Get 2
		PUSH	AX				; Pass parameter
		CALL	_short_pause	; Wait a bit
		POP		AX				; Clean up stack
		DEC		DX				; Backup to control
		MOV		AH,DGRP:_reset	; Get reset flag
		AND		AH,AH			; Test for reverse
		JNZ		setm2			; Normal RESET
; Reversed reset line
		MOV		AL,09h			; DTR=1, RTS=0
		AND		BL,BL			; Normal mode?
		JZ		setm3			; Its OK
		MOV		AL,08h			; DTR=0, RTS=0
		DEC		BL				; System mode?
		JZ		setm3			; and proceed
		MOV		AL,03h			; Reset: DTR=1, RTS=1
		JMP		SHORT setm3		; And proceed
; Normal reset line
setm2:	MOV		AL,0Bh			; DTR=1, RTS=1
		AND		BL,BL			; Normal mode?
		JZ		setm3			; Its OK
		MOV		AL,0Ah			; DTR=0, RTS=1
		DEC		BL				; System mode?
		JZ		setm3			; and proceed
		MOV		AL,01h			; Reset: DTR=1, RTS=0
setm3:	OUT		DX,AL			; Write to port
}
	if(ds5000) switch(mode) {
		case 0 :
			write_target(PODDS5);
			write_target(ds5000_mcon);
			break;
		case 1 :
			write_target(PODDS5);
			write_target(0x18); }
}	

/*
 * Wait for short pause
 */
short_pause(count) asm
{
		MOV		AX,40h			; Address BIOS data area
		MOV		ES,AX			; Point to it
pause1:	MOV		AX,ES:6Ch		; Get timer tick
pause2:	CMP		AX,ES:6Ch		; Does it match?
		JZ		pause2			; Yes, wait for it
		DEC		WORD PTR 4[BP]	; Reduce count
		JNZ		pause1			; Keep going
}

/*
 * Handle I/O to debug window during 'Go'
 */
go_local()
{
	static int gms = 0;
	char rxc, zoom;
	int c;
	unsigned state;

	wcursor_line();
	rxc = zoom = state = 0;
	for(;;) {
		/* Handle characters received from system */
		if((c = COMtestc()) != -1) {
			switch(state) {
			case 1 :
				if(c == PODEID2) {
					state = 2;
					continue; }
				wputc(PODEID1);
				goto exstate;
			case 2 :
				if(c == PODEID3) {
					if(zoom)
						wclose();
					return 0; }
				wputc(PODEID1);
				wputc(PODEID2);
			default:
			exstate:
				state = 0; }
			if(c == PODEID1) {
				state = 1;
				continue; }
			wputc(c);
			rxc = 0;
			continue; }

		if(!rxc) {
			wupdatexy();
			rxc = -1; }

		/* Handle characters received from terminal */
		if(c = wtstc()) {
			if(c == 0x1B) {
				if(!wmenu(33,9,WSAVE|WCOPEN|WBOX3|REVERSE,gomenu,&gms)) switch(gms) {
					case 0 :	/* Reset target */
						if(zoom)
							wclose();
						return -1;
					case 1 :	/* Halt at watch */
						mflag = 0;
						continue;
					case 2 :	/* Send ESCAPE */
						write_target(0x1B);
						continue;
					case 3 :	/* Zoom window */
						if(zoom = !zoom)
							wopen(0,0,80,25,attributes[MATTR]);
						else
							wclose();
						rxc = 0; }
					continue; }
			write_target((c == '\n') ? '\r' : c); } }
}

/*
 * Execute a sub-shell
 */
doshell()
{
	char comspec[65];

	if(!getenv("COMSPEC", comspec)) {
		wputs("Cannot locate COMSPEC environment variable");
		return; }
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|0x07);
	wputs("Type 'EXIT' to return to MONICA52.");
	wcursor_line();
	wupdatexy();
	exec(comspec, "");
	wclose();
}

/*
 * Handle a breakpoint exception
 */
handle_breakpoint(unsigned b)
{
	unsigned i;

	if(!(b & B_ENAB))
		return -1;

	if(i = b & B_MASK) {
		if(b & B_DEC) {				/* Decrement till quit */
			if(!BRKCNT[i])
				return mflag = 0;
			else
				--BRKCNT[i]; }
		else {						/* Increment count */
			if(!++BRKCNT[i])
				BRKCNT[i] = -1; } }

	if(b & B_WATCH) {				/* Watch point */
		show_pc();
		update();
		i = -1; }

	if(b & B_PAUSE) {
		wupdatexy();
		delay(bdelay);
		i = -1; }

	if(b & B_KEY) {
		i = -1;
		wputs("\nSPACE to proceed... ESC to halt:");
		for(;;) switch(wgetc()) {
			case ' ' :	return -1;
			case 0x1B :	return 0; } }

	return i;
}

/*
 * Get breakpoint flags
 */
get_bflags(unsigned f)
{
	wopen(24, 9, 32, 8, WSAVE|WCOPEN|WBOX2|REVERSE);
	for(;;) {
		wgotoxy(0, 0);
		wprintf("(E)nable        : %s\n", (f & B_ENAB) ? "YES" : "NO ");
		wprintf("(W)atchpoint    : %s\n", (f & B_WATCH)  ? "YES" : "NO ");
		wprintf("(K)eypress      : %s\n", (f & B_KEY) ? "YES" : "NO ");
		wprintf("(P)ause         : %s",   (f & B_PAUSE) ? "YES" : "NO ");
		if(f & B_PAUSE)
			wprintf(" (%u)", bdelay);
		wcleol();
		wputs("\n(C)ounter       : ");
		if(f & B_MASK) {
			wprintf("%u   (%u)", f & B_MASK, BRKCNT[f & B_MASK]);
			wcleol();
			wprintf("\n(D)ecrement/stop: %s", (f & B_DEC) ? "YES" : "NO "); }
		else {
			wputs("NO ");
			wcleow(); }
		switch(toupper(wgetc())) {
			case 0x1B :
			case '\n' :
				wclose();
				return f;
			case 'E' :	f ^= B_ENAB;	break;
			case 'W' :	f ^= B_WATCH;	break;
			case 'K' :	f ^= B_KEY;		break;
			case 'P' :	f ^= B_PAUSE;	break;
			case 'D' :	f ^= B_DEC;		break;
			case 'C' :	f = (f & ~B_MASK) | (++f & B_MASK); } }
}

/*
 * Get correct character to show for breakpoint
 */
show_breakpoint(unsigned f)
{
	char c;
	c = ' ';
	if(f) {
		if(!(f & B_ENAB))
			return 'x';
		c = 'b';
		if(f & B_WATCH)
			c = 'w';
		if(f & B_PAUSE)
			c = 'p';
		if(f & B_KEY)
			c = 'k';
		if(f & B_MASK) {
			c = '+';
			if(f & B_DEC)
				c = '-'; } }
	return c;
}
