/*
 * Script compiler
 */
#include <stdio.h>

#ifdef _MICROC_
	//#define STKMON
	#define	U8	unsigned char
	#define	U16	unsigned
#else
	#include <stdlib.h>
	#include <string.h>
	#include <ctype.h>
	typedef unsigned char U8;
	typedef unsigned short U16;
#endif

// System parameters
#define	EBUFFER		128		// Size of expression buffer
#define	SYMBOLS		2000	// Number of symbols
#define	MAXMAP		100		// Number of mapped opcodes
#define	CLIST		500		// Number of case list elements
#define	FIXUPS		500		// Number of fixups
#define	ESTACK		32		// Execution stack size
#define	IDEPTH		3		// Include stack depth
#define	POOL		16384	// Size of string pool
#define	EEXT		".esl"	// Extension type

// Option bits
#define	OP_DEBUG	0x0001	// Debug output
#define	OP_QUIET	0x0002	// Quiet operation
#define	OP_COUT		0x0004	// C format output
#define	OP_ELIM		0x0010	// OPT: redundant code elimination
#define	OP_CONST	0x0020	// OPT: constant expression reduction
#define	OP_JMP		0x0040	// OPT: jump optimization
#define	OP_INDEX	0x0080	// OPT: Index optimization
#define	OP_DMPSYM	0x0100	// Dump symbol table
#define	OP_DEFAULT	(OP_ELIM | OP_CONST | OP_JMP | OP_INDEX)

// Symbol types
#define	SYMVAR		0		// Variable
#define	SYMCON		1		// Numeric Constant
#define	SYMFIX		2		// Fixed String
#define	SYMSTR		3		// String
#define	SYMLAB		4		// Defined label
#define	SYMULAB		5		// Undefined label
#define	SYMKEY		6		// Keyword
#define	SYMUSR		7		// User defined
#define	SYMSYS		8		// System value

// GETSYMBOL flags
#define	GS_NEW		0x01	// Symbol must be new
#define	GS_OLD		0x02	// Symbol must exist
#define	GS_VALID	0x04	// Symbol must be valid

// Output token types
#define	OTVAL		0		// Output is a number
#define	OTVAR		1		// Output is a variable
#define	OTSYS		2		// Output is a System variable
#define	OTASS		3		// Output is an assigned variable
#define	OTOPR		4		// Output is an operator
#define	OTIDX		5		// Output is an indexed variable

#define	OPKILL		0xFE	// Kill opcode
#define	OPKASS		0xFF	// Kill but indicate assignment

#include "eslopc.h"			// Opcode definitions

U16
	Cwidth = 70,			// C output line width
	FindAdr = -1,			// Find address
	Options = OP_DEFAULT,	// Command options
	NumVar = 32,			// Number of numeric variables
	NumStr = 4,				// Number of string variables
	Line,					// Line number
	Seg,					// External segment
	Oseg,					// Output segment
	Stop,					// Top of external segment
	Otop,					// Top of output segment
	Etop,					// Top of Expression buffer
	CLtop,					// Top of case-list
	Ptop,					// Top of string pool
	Mtop,					// Top of MAP list
	SymTop,					// Symbol index
	Sbase,					// Symbol search base address
	SymBase,				// Symbol index base address
	FixTop,					// Top of fixup list
	Itop,					// Top of include stack
	Wcount,					// Warning count
	OCcount,				// OpCode count
	Emax,					// Tracks maximum expression buffer depth
	Smax,					// Tracks maximum expression stack depth
	IFbase1,				// IF fixup 1
	IFbase2,				// IF fixup 2
	Noffset,				// Offset to filename
	Eoffset,				// Offset to extension
	MAPopcode = -1,			// Map this opcode
	Wline = -1,				// Last warning line
	L[2],					// Temp long value
	F[2],					// Function encoding
	Iline[IDEPTH],			// Include stack line
	CLvalue[CLIST],			// Case-List value
	FixValue[FIXUPS],		// Fixup value
	MAPlist[MAXMAP],		// Mapped opcode list
	SymValue[SYMBOLS][2],	// Symbol value
	Evalue[EBUFFER][2];		// Expression buffer - value
FILE
	*Fp,					// Working file pointer
	*Lfp,					// Listing file pointer
	*Ifp[IDEPTH];			// Include file stack
U8
	*Ptr,					// General pointer
	*Iname,					// Input filename
	*Oname,					// Output filename
	*Lname,					// Listing name
	*Efile,					// Extension file
	*Aptr,					// Argument pointer
	*Nptr,					// Filename pointer
	Bit32 = 255,			// Output 32-bit values
	LEflag,					// Extensions loaded flag
	Lst,					// Generate listing
	Lst1,					// Saved for includes
	Lopcode,				// Last opcode
	UnReach,				// Code is unreachable
	SymType[SYMBOLS],		// Symbol type
	CLtype[CLIST],			// Case-List type
	Etype[EBUFFER],			// Expression buffer - type
	Efixup[EBUFFER],		// Expression buffer - fixups
	Name[256],				// Encoded name
	Buffer[256],			// Input buffer
	Pool[POOL];				// String pool

U8 *KeyWords[] = {
	"IF",			// 1
	"ELSE",			// 2
	"END",			// 3
	"SWITCH",		// 4
	"CASE",			// 5
	"WHILE",		// 6
	"DO",			// 7
	"CALL",			// 8
	"GOTO",			// 9
	"RETURN",		// 10
	"RESET",		// 11
	"VARIABLE",		// 12
	"CONSTANT",		// 13
	"FIXED",		// 14
	"STRING",		// 15
	"SET",			// 16
	"STOP",			// 17
	"INCLUDE",		// 18
	"NOLIST",		// 19
	"SECTION"};		// 20
#define	IF			1
#define	ELSE		2
#define	END			3
#define	SWITCH		4
#define	CASE		5
#define	WHILE		6
#define	DO			7
#define	CALL		8
#define	GOTO		9
#define	RETURN		10
#define	RESET		11
#define	VARIABLE	12
#define	CONSTANT	13
#define	FIXED		14
#define	STRING		15
#define	SET			16
#define	STOP		17
#define	INCLUDE		18
#define	NOLIST		19
#define	SECTION		20

U8 *Stype[] = {
	"VARIABLE", "CONSTANT", "FIXED", "STRING", "LABEL", "LABEL",
	"KEYWORD", "FUNCTION", "SYSVAR", "???" };

U8 *Hindex[] = {
	".ES",											// 0 Extension
	".ESB",											// 1 Binary output
	".ESC",											// 2 C output
	".ESS",											// 3 String file
	"Extensible Script Language (ESL)",				// 4 Name
	0,												// 5 Command
	EEXT };											// 6 extension extension
#define	CMD		5

U8 Help[] = { "\n\
use:	\x85 infile[\x80] [outfile[\x81/\x82]] [options]\n\n\
opts:	/C[width]	- output C [\x82], otherwise BINARY [\x81]\n\
	/D		- enable Debug output\n\
	/L[I]		- enable Listing	I=+Included\n\
	/Q		- Quiet: reduce output\n\
	/O... 		- suppress Optimizations:\n\
		C = Constant expressions\n\
		I = Index\n\
		J = Jump\n\
		R = Redundant code elimination\n\
	/S		- display Symbol table\n\
	E=file[\x86]	- specify Extension file	[\x85\x86]\n\
	L=file[.LST]	- specify Listing file		[infile.LST]\n\
	F=address	- Find source line at address	[none]\n\
	name=value	- preset constant symbol	[none]\n\
\nExtensible Script Language - ?COPY.TXT 2005-2019 Dave Dunfield.\n" };

// Prototype functions called before definition
void value_expression(int t);
void vlist(U16 r[]);
U16 showerr(U8 *buf);

#ifdef _MICROC_				// DDS Micro-C/PC compiler
extern Longreg[2];			// Remainder after division

// Output debug message
register Debug(U16 args)
{
	U16 a;
	U8 buf[128];
	a = nargs();
	if(Options & OP_DEBUG) {
		_format_(a * 2 + &args, buf);
		fputs(buf, stdout); }
}
// Display critical-error message and terminate
register error(U16 args)
{
	U8 buf[81];
	_format_(nargs() * 2 + &args, buf);
	showerr(buf);
	exit(-1);
}

// Display warning message
register warn(U16 args)
{
	U8 buf[81];
	_format_(nargs() * 2 + &args, buf);
	if(Wline != Line) {
		Wline = Line;
		showerr(buf);
		if(++Wcount >= 5) {
			Line = 0;
			error("!Compile failed"); } }
	if(!Line)
		exit(-1);
}

void segfull()	// Report external segment exhausted
{
	error("Out of memory");
}
/*
 * Add name to external segment
 */
void SegAdd(name) asm
{
		MOV		ES,DGRP:_Seg		// External segment
		MOV		DI,DGRP:_Stop		// Top of external segment
		PUSH	DI					// Save
		MOV		SI,4[BP]			// Get name
		INC		DI					// Leave space for length
sadd1:	AND		DI,DI				// Segment overflow?
		JZ		_segfull			// Exit
		MOV		AL,[SI]				// Get from source
		MOV		ES:[DI],AL			// Save in seg
		AND		AL,AL				// End of string?
		JZ		sadd2				// Yes, top
		INC		SI					// Next in source
		INC		DI					// Next in dest
		JMP short sadd1				// And continue
sadd2:	MOV		AX,DI				// Get end marker
		MOV		DGRP:_Stop,AX		// Save new top
		POP		DI					// DI = length byte
		SUB		AX,DI				// Calculate length
		DEC		AX					// -1 for length
		MOV		ES:[DI],AL			// Save length
}

/*
 * Lookup symbol in external segment
 */
U16 SegLook(start, end, name) asm
{
		MOV		ES,DGRP:_Seg		// Get extra segment
		MOV		SI,8[BP]			// Get starting address
		XOR		BX,BX				//; Zero result
		XOR		DX,DX				// Keep count
		XOR		CH,CH				// Zero high
slok1:	CMP		SI,6[BP]			// At end?
		JAE		slok4				// Yes - stop looking
		INC		DX					// Next
		MOV		CL,ES:[SI]			// Get length
		MOV		DI,4[BP]			// Get local string
		INC		SI					// Skip length
slok2:	MOV		AL,ES:[SI]			// Get from seg
		CMP		AL,[DI]				// Does it match?
		JNZ		slok3				// No - skip to next
		INC		SI					// Next source
		INC		DI					// Next dest
		DEC		CL					// Reduce count
		JNZ		slok2				// Keep looking
; Matches - check term
		MOV		AL,[DI]				// Get terminator
		AND		AL,AL				// Is this is?
		JNZ		slok1				// No - try again
		MOV		BX,DX				// Save index
		JMP short slok1				// And keep looking
; Did not match - skip to next
slok3:	ADD		SI,CX				// Offset to next
		JMP short slok1				// And look again
slok4:	MOV		AX,BX				// Get final result
}

// Get a string from external segment
void SegGet(start, index, dest) asm
{
		MOV		ES,DGRP:_Seg		// Point to extra segment
		MOV		SI,8[BP]			// Get starting point
		MOV		DX,6[BP]			// Get index
		XOR		CH,CH				// Zero high
sget1:	MOV		CL,ES:[SI]			// Get length
		INC		SI					// Skip length
		AND		DX,DX				// At end?
		JZ		sget2				// We have it
		ADD		SI,CX				// Skip to next
		DEC		DX					// Reduce
		JMP short sget1				// And proceed
sget2:	MOV		DI,4[BP]			// Put it here
sget3:	MOV		AL,ES:[SI]			// Get from source
		MOV		[DI],AL				// Write to dest
		INC		SI					// Next source
		INC		DI					// Next dest
		LOOP	sget3				// Do them all
		MOV		[DI],CL				// Zero terminate
}

// Copy data within segment
void SegCopy(to, from, length) asm
{
		MOV		ES,DGRP:_Seg		// Get segment
		MOV		CX,4[BP]			// Length
		AND		CX,CX				// Data to move?
		JZ		scpy2				// No
		MOV		SI,6[BP]			// From
		MOV		DI,8[BP]			// To
scpy1:	MOV		AL,ES:[SI]			// Get from source
		MOV		ES:[DI],AL			// Write to dest
		INC		SI					// Next source
		INC		DI					// Next dest
		LOOP	scpy1				// Copy all
scpy2:
}

#else							// LCCWIN32 compiler
#include "lincomp.c"

void Debug(char *fmt, ...)
{
	char buf[128];
	va_list a;
	if(Options & OP_DEBUG) {
		va_start(a, fmt);
		_format_(buf, 127, fmt, a);
		va_end(a);
		fputs(buf, stdout); }
}
void error(char *fmt, ...)
{
	char buf[128];
	va_list a;
	va_start(a, fmt);
	_format_(buf, 127, fmt, a);
	va_end(a);
	showerr(buf);
	exit(-1);
}
void warn(char *fmt, ...)
{
	U16 l;
	U8 buf[81];
	va_list a;
	va_start(a, fmt);
	_format_(buf, 127, fmt, a);
	va_end(a);
	if(Wline != Line) {
		Wline = Line;
		showerr(buf);
		if(++Wcount >= 5) {
			Line = 0;
			error("!Compile failed"); } }
	if(!Line)
		exit(-1);
}

void SegAdd(U8 *n)
{
	U16 o;
	o = Stop++;
	do {
		if(!Stop)
			error("Out of memory");
		poke(Seg, Stop++, *n); }
	while(*n++);
	--Stop;		// Don't include null
	poke(Seg, o, (Stop-o) - 1);
}
int SegLook(U16 pos, U16 end, U8 *name)
{
	U16 l, r, c;
	U8 *n;
	r = c = 0;
	while(pos < end) {
		++c;
		l = peek(Seg, pos++);
		n = name;
		while(peek(Seg, pos) == *n) {
			++pos;
			++n;
			if(!--l) {
				if(!*n)
					r = c; } }
		pos += l; }
	return r;
}
void SegGet(U16 pos, U16 index, U8 *dest)
{
	unsigned l;
	for(;;) {
		l = peek(Seg, pos++);
		if(!index--)
			break;
		pos += l; }
	do {
		*dest++ = peek(Seg, pos++); }
	while(--l);
	*dest = 0;
}
void SegCopy(U16 to, U16 from, U16 length)
{
	while(length) {
		--length;
		poke(Seg, to++, peek(Seg, from++)); }
}
#endif

// Display error output
U16 showerr(U8 *buf)
{
	U16 i, l;
	U8 *p;
	if(l=Line) {
		if(LEflag)
			fputs(Iname, stdout);
		putc('(', stdout);
		for(i=0; i < Itop; ++i)
			printf("%u-", Iline[i]);
		printf("%u): ", Line); }
	p = buf;
ag:	switch(*p++) {
	case '*' :	fputs("Warning - ", stdout);	// Warning
				--Wcount;
	case '!' :	l = 0;		goto ag;			// No line display
	case '~' :	--Wcount;
	case '@' :	Wline = -1; goto ag; }			// Reset line
	fputs(p-1, stdout);
	putc('\n', stdout);
	if(l) {
		p = Buffer;
		while(i = *p++)
			putc(((i == '\t') ? ' ' : i), stdout);
		putc('\n', stdout);
		if(l != -1) {
			p = Buffer;
			while(++p < Ptr)
				putc(' ', stdout);
			fputs("^\n", stdout); } }
	return l;
}

// Add character to pool
void padd(U8 c)
{
	if(Ptop >= sizeof(Pool))
		error("String space exhausted");
	Pool[Ptop++] = c;
}

// Skip to non-blank
int skip(void)
{
	while(isspace(*Ptr))
		++Ptr;
	switch(*Ptr) {
		case ';' : return 0;
		case '/' :
			if(Ptr[1] == '/')
				*Ptr = 0; }
	return *Ptr;
}

// Test for valid symbol character
int issymbol(int c)
{
	if((c >= 'a') && (c <= 'z'))
		return 255;
	if((c >= 'A') && (c <= 'Z'))
		return 255;
	return (c == '_');
}

// Get a symbol
int getsymbol(U8 f)
{
	U8 *p;
	U16 i;
	if(issymbol(skip())) {
		p = Name;
		while(issymbol(*Ptr) || isdigit(*Ptr))
			*p++ = toupper(*Ptr++);
		*p = 0;
		if(f & GS_OLD) {	// Must be old
			if(!(i = SegLook(0, Stop, Name)))
				warn("Symbol '%s' does not exist", Name);
			return i; }
		if(f & GS_NEW) {	// Must be new
			if(SegLook(Sbase, Stop, Name))
				warn("Duplicate symbol: %s", Name);
			SegAdd(Name); }
		return 255; }
	if(f & GS_VALID)
		warn("Symbol expected");
	return 0;
}

// Create a symbol
void newsymbol(U16 t)
{
	if(SymTop >= SYMBOLS)
		error("Too many symbols");
	SymType[SymTop++] = t;
}

// Place RPN stream into output buffer
// Resolve any compile time computations
void output(U8 t, U16 v[])
{
	U16 l[2], l1[2];
	if(Etop >= EBUFFER) {
		warn("Expression overflow");
		return; }
	if((t == OTOPR) && (Options & OP_CONST) && (Etype[Etop-1] == OTVAL)) {
		longcpy(l, Evalue[Etop-1]);
		switch(*v) {
		case OMNEG:
			longset(l1, 0);
			longsub(l1, l);
			goto sm;
		case OMNOT:
			longset(l1, !longtst(l));
			goto sm;
		case OMCOM:
			*l1 = *l ^ 0xFFFF;
			l1[1] = l[1] ^ 0xFFFF;
			goto sm; }
		if(Etype[Etop-2] == OTVAL) {
			longcpy(l1, Evalue[Etop-2]);
			switch(*v) {
			case ODADD:	longadd(l1, l);	goto sb;
			case ODSUB:	longsub(l1, l);	goto sb;
			case ODMUL:	longmul(l1, l);	goto sb;
			case ODDIV:
			case ODMOD:
				if(!longtst(l)) {
					warn("Divide by zero");
					*l = 1; }
				longdiv(l1, l);
				if(*v == ODMOD)
					longcpy(l1, Longreg);
			sb:	--Etop;
			sm:	longcpy(Evalue[Etop-1], l1);
				return;
			case ODEQ: longset(l1, longcmp(l1, l) == 0);	goto sb;
			case ODNE: longset(l1, longcmp(l1, l) != 0);	goto sb;
			case ODLT: longset(l1, longcmp(l1, l) < 0);		goto sb;
			case ODGT: longset(l1, longcmp(l1, l) > 0);		goto sb;
			case ODLE: longset(l1, longcmp(l1, l) <= 0);	goto sb;
			case ODGE: longset(l1, longcmp(l1, l) >= 0);	goto sb;
			case ODBAND:	*l1 &= *l; l1[1] &= l[1];		goto sb;
			case ODBOR:		*l1 |= *l;	l1[1] |= l[1];		goto sb;
			case ODBXOR:	*l1 ^= *l;	l1[1] ^= l[1];		goto sb;
			case ODLAND: if(longtst(l1)) longcpy(l1, l);	goto sb;
			case ODLOR: if(!longtst(l1)) longcpy(l1, l);	goto sb;
			case ODSL: while((*l)--) longshl(l1);			goto sb;
			case ODSR: while((*l)--) longshr(l1);			goto sb;
	} } }

	longcpy(Evalue[Etop], v);
	Etype[Etop++] = t;
	if(Etop > Emax)
		Emax = Etop;
}

// Get a value element from the input stream
void value_element(void)
{
	U16 i, c, v[2], b[2], l[2];
	U8 f;

	longset(b, 10);
	switch(c=toupper(skip())) {
	default:
		if(isdigit(c))		// Number
			break;
		if(getsymbol(GS_VALID)) {
			if(c = SegLook(0, Stop, Name)) {
				longcpy(l, SymValue[--c]);
				switch(SymType[c]) {
				case SYMSYS: output(OTSYS, l);	return;
				case SYMCON: output(OTVAL, l);	return;
				case SYMVAR:
					output(OTVAR, l);
					if(skip() == '[') {
						++Ptr;
						i = (c = Etop) - 1;
						value_expression(']');
						++Ptr;
						if(Options & OP_INDEX) {
							// If index is constant, convert to direct reference
							if(((Etop-c) == 1) && !Etype[c]) {
								longcpy(l, Evalue[c]);
								longadd(l, Evalue[i]);
								if(l[1] || (*l >= NumVar))
									warn("Out of range");
								--Etop;					// Remove constant
								longcpy(Evalue[i], l);	// set new value
								return; }
							if(!*Evalue[i]) {	// From 0 - switch to monadic index
								Etype[i] = OPKILL;
								c = OMIDX;
								goto op; } }
						Etype[i] = OTIDX;	// Indicate indexed variable
						c = ODIDX;
				op:		Efixup[Etop] = i;	// Record position for '=' adjust
						longset(l, c);
						output(OTOPR, l); }
					return;	} } }
		warn("Numeric value expected");
		output(OTVAL, b);
		return;
	case '(' :							// Nested expression
		++Ptr;
		value_expression(')');
		++Ptr;
		return;
	case '[' :							// Variable list
		++Ptr;
		vlist(l);
		goto outl;
	case '\'':							// Character
		++Ptr;
		longset(b, 256);
		longset(l, 0);
		while((c = *Ptr++) != '\'') {
			if(!c) {
				warn("Unterminated character constant");
				break; }
			longmul(l, b);
			*l |= c; }
outl:	output(OTVAL, l);
		return;
	case '&' :							// Address of variable
		++Ptr;
		if(c = getsymbol(GS_VALID|GS_OLD)) switch(SymType[--c]) {
			default: warn("Variable expected");
			case SYMVAR:
			case SYMSTR:
				longset(l, *SymValue[c]); }
		goto outl;
	case '-' : c = OMNEG;	goto doop;	// Monadic negation
	case '!' : c = OMNOT;	goto doop;	// Monadic not
	case '~' : c = OMCOM;	doop:		// Monadic compliment
		++Ptr;
		value_element();
		longset(l, c);
		output(OTOPR, l);
		return;
	case '0' :							// Number
		switch(toupper(Ptr[1])) {
		case 'B' : *b = 2;	goto xb2;
		case 'O' : *b = 8;	goto xb2;
		case 'X' : *b = 16;	xb2:
			Ptr += 2; }
		break;
	case '%' : *b = 2;	goto xb1;		// Binary number
	case '@' : *b = 8;	goto xb1;		// Octal number
	case '$' : *b = 16;					// Hexidecimal number
		xb1: ++Ptr; }

	longset(v, f = 0);
	for(;;) {
		c = toupper(*Ptr);
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else
			break;
		if(c >= *b)
			break;
		longmul(v, b);
		longset(l, c);
		longadd(v, l);
		++Ptr;
		f = 255; }
	if(!f)
		warn("Numeric value expected");
	output(OTVAL, v);
}

// Resolve an expression
void value_expression(int t)
{
	U16 c, op;

	value_element();
	for(;;) {
		if((c = skip()) == t)
			return;
		if(!c) {
			if(t != ';')
				warn("Bad expression");
			return; }
		switch(c = (c << 8) | *++Ptr) {
			case ('='<<8)|'=' : op = ODEQ;		goto op1;
			case ('!'<<8)|'=' : op = ODNE;		goto op1;
			case ('<'<<8)|'=' : op = ODLE;		goto op1;
			case ('>'<<8)|'=' : op = ODGE;		goto op1;
			case ('<'<<8)|'<' : op = ODSL;		goto op1;
			case ('>'<<8)|'>' : op = ODSR;		goto op1;
			case ('&'<<8)|'&' : op = ODLAND;	goto op1;
			case ('|'<<8)|'|' : op = ODLOR;		op1:
				++Ptr;
				break;
			default: switch(c >>= 8) {
				default: warn("Unknown operator '%c'", c);
				case '+' : op = ODADD;	break;
				case '-' : op = ODSUB;	break;
				case '*' : op = ODMUL;	break;
				case '/' : op = ODDIV;	break;
				case '%' : op = ODMOD;	break;
				case '<' : op = ODLT;	break;
				case '>' : op = ODGT;	break;
				case '&' : op = ODBAND;	break;
				case '|' : op = ODBOR;	break;
				case '^' : op = ODBXOR;	break;
				case '=' :		// Assignment
					switch(Etype[c=Etop-1]) {
					case OTOPR:
						switch(*Evalue[c]) {
						case ODIDX:	// Convert dyadic indexed into assignment
							*Evalue[c] = ODADD;			// To simple ADD
							Etype[Efixup[c]] = OTASS;	// And assignment
							break;
						case OMIDX:	// Convert monadic index into assignment
							Etype[Efixup[c]] = OPKASS;	// Indicate assign
							--Etop;	}					// Remove @
						break;
					default: warn("Non-assignable %u", Etype[c]); break;
					case OTVAR:	Etype[c] = OTASS;	}	// Indicate assigned
					value_expression(t);
					op = ODASS;
					goto noget;
				case '[' :		// Indexed variable
					warn("Incorrect index");
		} }
		value_element();
noget:	longset(L, op);
		output(OTOPR, L);
	}
}

// Write a byte to the output segment
void ob(U8 b)
{
	poke(Oseg, Otop++, b);
	if(!Otop)
		error("Output segment overflow");
}
// Write and record opcode byte
void oc(U8 b)
{
	ob(b);
	++OCcount;
	switch(UnReach) {
	case 0 :
		switch(b) {
		case XBRANCH :
		case XSTOP :
		case XRETURN :
			if(Options & OP_ELIM)
				UnReach = 1; }
		return;
	case 1 :
		if(b == MAPopcode) {
			UnReach = 0;
			return; }
		warn("*@Unreachable");
		UnReach = 255; }
}
// Write a word to the output segment
void ow(U16 w)
{
	ob(w);
	ob(w >> 8);
}
// Get word from output segment
U16 gw(U16 o)
{
	return (peek(Oseg, o+1) << 8) | peek(Oseg, o);
}
// Patch word in output buffer
void pw(U16 o, U16 w)
{
	if(o) {
		poke(Oseg, o, w);
		poke(Oseg, o+1, w >> 8); }
}
// Patch code with current location
void ph(U16 o)
{
	pw(o, Otop);
	UnReach = 0;
}

/*
 * Output the results of an expression
 */
void output_expression(void)
{
	U16 i, l[2], o, s;
	U8 b;
	o = Etop - 1;
	Etype[Etop] = 255;
//for(i=0; i < Etop; ++i)
//	printf("%u: T=%u V=%x%04x\n", i, Etype[i], Evalue[i][1], *Evalue[i]);

	for(i=s=0; i < Etop; ++i) {
		if(s > Smax)
			Smax = s;
		if(s > ESTACK)
			warn("Expression stack overflow");
		b = (i < o) ? 0x80 : 0x00;
		longcpy(l, Evalue[i]);
		switch(Etype[i]) {
		case OTASS :	// Assigned variable (output simple value)
		if(Options & OP_INDEX) {
			if((Etype[i+1] == OTVAL) && (Etype[i+2] == OTOPR) && (*Evalue[i+2] == ODADD)) {
//printf("Opt1\n");
				longadd(l, Evalue[i+1]);
				i += 2; } }
		case OTVAL :	// Value
			++s;
			if(l[1] || (*l & 0xE000)) {	// 32-bit
				ob(b | 0x60);
				ow(*l);
				if(Bit32)
					ow(l[1]);
				continue; }
	v5:		if(*l & 0xFFE0) {			// 13-bit value
				ob((*l & 0x1F) | 0x20 | b);
				ob(*l >> 5);
				continue; }
			ob(*l | b);					// 5-bit value
			continue;
		case OTIDX :	// Indexed variable  ("")
		if(Options & OP_INDEX) {
			if((Etype[i+1] == OTVAL) && (Etype[i+2] == OTOPR) && (*Evalue[i+2] == ODIDX)) {
//printf("Opt2\n");
				longadd(l, Evalue[i+1]);
				++s;
				if(*l & 0xFFE0) {
					ob((*l & 0x1F) | 0xA0);
					++s;
					ob(*l >> 5); }
				else
					ob(*l | 0x80);
				*l = OMIDX;
				goto op; } }
			++s;
			goto v5;
//			ob(*l | b);					// 5-bit value
//			continue;
		case OTSYS :	// System variable
			++s;
			ob((*l | b) + 0x61);
			continue;
		case OTVAR :	// Variable (retrieval
			++s;
			if(*l < 32) {
				ob(*l | 0x40 | b);
				continue; }
			ob((*l & 0x1F) | 0xA0);
			ob(*l >> 5);
			*l = OMIDX;
		case OTOPR :	// Operator
		op:	ob((*l | b) + 0x68);
			if(*l >= ODADD)
				--s;
			continue; } }
}

/*
 * Resolve an expression
 */
void expression(int t)
{
	Etop = 0;
	value_expression(t);
	if(t)
		++Ptr;
	output_expression();
}

/*
 * Resolve a constant expression
 */
U16 constant(U16 l[], U8 t)
{
	Etop = 0;
	value_expression(t);
	if((Etop != 1) || *Etype)
		warn("Constant expression required");
	if(t)
		++Ptr;
	if(l)
		longcpy(l, Evalue[0]);
	return *Evalue[0];
}

/*
 * Resolve a conditional
 */
U16 conditional(U8 n, U16 label)
{
	U16 o, t;
	U8 b;
	o = Otop++;
	Etop = b = 0;
	value_expression(0);
	if(Etype[Etop-1] == OTOPR) {
		t = Etop-2;
		switch(*Evalue[Etop-1]) {
		case ODNE:	// != 0
			if((Etype[t] == OTVAL) && !longtst(Evalue[t]))
				Etop = t;
			break;
		case ODEQ:	// == 0
			if((Etype[t] != OTVAL) || longtst(Evalue[t]))
				break;
			--Etop;			// Remove value (fall through)
		case OMNOT:	// !
			--Etop;			// Remove operator
			b = 255; } }		// Switch to XBTRUE
	output_expression();	// Generate expression
	if(n)
		t = b ? XBFALSE : XBTRUE;
	else
		t = b ? XBTRUE : XBFALSE;
	poke(Oseg, o, t);		// Output C-BRANCH opcode
	o = Otop;				// Branch address position
	ow(label);				// Branch address (fixup later)
	return o;
}

// Apply fixups
void fixup(unsigned index)
{
	unsigned i, ii, j, l;
	for(i=ii=0; i < FixTop; ++i) {
		if(gw(j = FixValue[i]) == index) {
			l = *SymValue[index];
			Debug("Fixup: %04x <- %04x %u\n", j, l, index);
			pw(j, l);
			continue; }
		FixValue[ii++] = j; }
	FixTop = ii;
}

// Get a line from the input stream
U16 get_line(void)
{
	U16 i;
	U8 *sp;

	if(skip())
		warn("Extra data on line - ignored");
	if(*Ptr == ';')
		++Ptr;
	else {
ag:		if(!fgets(Ptr = Buffer, sizeof(Buffer), Fp)) {
			if(Itop) {
				fclose(Fp);
				Fp = Ifp[--Itop];
				Line = Iline[Itop];
				if(Lst & 0xF0) putc('\n', Lfp);
				Lst = Lst1;
				goto ag; }
			return 0; }
		++Line;
		if(Lst) {
			fprintf(Lfp, "%04x%6u\t", Otop, Line);
			fputs(Buffer, Lfp);
			putc('\n', Lfp); } }
	if(!skip()) goto ag;
	sp = Ptr;

	if(getsymbol(0)) {
		if(skip() == ':') {		// Line label
			if(i = SegLook(Sbase, Stop, Name)) {	// Already exists
				if(SymType[i=(i+SymBase)-1] != SYMULAB)		// Not fwd ref
					warn("Duplicate symbol: %s", Name);
				SymType[i] = SYMLAB;
				longset(SymValue[i], Otop);
				fixup(i); }
			else {				// New label
				SegAdd(Name);
				longset(SymValue[SymTop], Otop);
				newsymbol(SYMLAB); }
			UnReach = 0;
			IFbase2 = -1;
			sp = ++Ptr;
			if(!skip()) goto ag;
			if(!getsymbol(0))
				return -1; }
		if(i = SegLook(0, Stop, Name)) {
			longcpy(F, SymValue[--i]);
			switch(SymType[i]) {
			case SYMKEY: return *F + 1;
			case SYMUSR: return 0xFFFE; } }
		Ptr = sp; }
	return 0xFFFF;
}

// Process a string
void string(U8 f)
{
	int c, v, o;
	U8 t;

	do {
		if(getsymbol(0)) {
			if(c = SegLook(0, Stop, Name)) switch(SymType[--c]) {
				case SYMSTR:
					if(f) { csr:
						error("Constant string required"); }
					ob(0x80 | *SymValue[c]);
					goto nxts;
				case SYMFIX:
					o = *SymValue[c];
					while(c = Pool[o++])
						ob(c);
					goto nxts;
				case SYMCON:
					longcpy(L, SymValue[c]);
					goto ov; }
			warn("String variable expected"); }
		switch(t = *Ptr) {
		default: warn("String expected");
		case '(' :	// Numeric conversion
			if(f) goto csr;
			++Ptr;
			o = Otop++;
			expression(')');
			v = 0;
			while(isdigit(c = *Ptr++))
				v = (v * 10) + (c - '0');
			if(v)
				--v;
			switch(toupper(c)) {
			default: warn("Bad format '%c'", c);
			case 'B' : v |= 0xA0;	break;
			case 'D' :
			case 'U' : v |= 0xC0;	break;
			case 'X' : v |= 0xE0;	}
			poke(Oseg, o, v);
			break;
		case '{' :
			++Ptr;
			constant(L, '}');
		ov:	while(longtst(L)) {
				if(c = L[1] >> 8) {
					if(c & 0x80)
						warn("Bad character (>0x80)");
					ob(c); }
				v = (*L >> 8) & 255;
				L[1] = (L[1] << 8) | v;
				*L <<= 8; }
			break;
		case ',' :	goto ex;
		case '"' :
		case '\'':
		case '`' :
		case '/' :
		case '\\':
			++Ptr;
			while((c = *Ptr++) != t) {
				if(!c) {
					warn("Unterminated string");
					--Ptr;
					break; }
				ob(c); } } nxts: ;
	} while(skip());
ex:	ob(0);
}

// Encode a variable list
void addmask(U16 v[], U16 i)
{
	U16 l[2];
	longset(l, 1);
	while(i--)
		longshl(l);
	v[1] |= l[1];
	*v |= *l;
}

void vlist(U16 r[])
{
	U16 i, j, m;

	m = Bit32 ? 31 : 15;
	longset(r, 0);
	for(;;) {
		if(i = getsymbol(GS_VALID|GS_OLD)) switch(SymType[--i]) {
		default: ba: warn("Bad vlist '%s'", Name); break;
		case SYMKEY:
			if(*SymValue[i] != (SECTION-1))
				goto ba;
			for(i=SymBase; i < SymTop; ++i) {
				if(SymType[i] == SYMVAR) {
					if((j = *SymValue[i]) <= m)
						addmask(r, j);
					else {
						SegGet(0, i, Name);
						warn("*@Variable '%s' omitted", Name); } } }
			break;
		case SYMVAR:
			if((j = *SymValue[i]) > m) {
				warn("Variable '%s' out of range (0-%u)", Name, m); }
			addmask(r, j); }

		switch(skip()) {
		default: warn("']' expected"); --Ptr;
		case ']' :
			++Ptr;
			return;
		case ',' : ++Ptr; } }
}

// Encode a numeric variable
void getvar(U8 ns)
{
	U16 i, t, l[2];

	t = ns ? SYMSTR : SYMVAR;
	if(getsymbol(GS_VALID)) {
		if(i = SegLook(0, Stop, Name)) {
			if(SymType[--i] == t) {
				if(skip() == '[') {
					++Ptr;
					constant(l, ']');
					longadd(l, SymValue[i]);
					if(l[1] || (*l >= (ns ? NumStr : NumVar)))
						warn("Out of range");
					ob(*l); }
				else
					ob(*SymValue[i]);
				return; } } }
	warn("%s variable expected", ns ? "String" : "Numeric");
}

// Place a value on the case list
void docase(U8 t)
{
	if(CLtop >= CLIST) {
		warn("CASE overflow");
		return; }
	CLvalue[CLtop] = Otop;
	CLtype[CLtop++] = t;
}

// Generate a code label reference
void dolabel(void)
{
	U16 i;

	getsymbol(GS_VALID);

	if(i = SegLook(Sbase, Stop, Name))
		i += SymBase;
	else {
		SegAdd(Name);
		longset(SymValue[SymTop], Line);
		newsymbol(SYMULAB);
		i = SymTop; }
	switch(SymType[--i]) {
	default: warn("Label expected");
	case SYMLAB : i = *SymValue[i];	break;		// Symbol exists
	case SYMULAB:					// Unknown symbol - add fixup
		if(FixTop >= FIXUPS)
			error("Too many fixups");
		FixValue[FixTop++] = Otop; }
	// If the symbol is unresolved, 'i' will contain the symbol table
	// index to be resolved the fixup list contains the address.
	ow(i);
}

/*
 * Clean section symbol table and promote
 * unresolved labels to the previous section
 */
void close_section(U16 oSbase, U16 oSymBase)
{
	U16 st, sst;
	U16 i, j, k, l;

	st = SymTop;			// Save section symbol top
	SymTop = SymBase;		// Save section symbol base, restore previous top
	SymBase = oSymBase;		// Restore previous base
	sst = Stop = Sbase;		// Save section base, restore TOS
	Sbase = oSbase;			// Restore previous section base

	for(i=SymTop; i < st; ++i) {
		if(SymType[i] == SYMULAB) {		// New unresolved symbol
			SegGet(0, i, Name);						// Get name
			if(j = SegLook(Sbase, sst, Name)) {		// Exist previously?
				switch(SymType[j=(j+SymBase)-1]) {
				case SYMLAB:		// Previous defined
				case SYMULAB:		// Previous undefined
					for(l=0; l < FixTop; ++l) {
						if(peekw(Oseg, k = FixValue[l]) == i)
							pokew(Oseg, k, j); }
					if(SymType[j] == SYMLAB)
						fixup(j); }
				continue; }
			// Symbol does not exist
			SegAdd(Name);								// Place at end
			longset(SymValue[SymTop], *SymValue[i]);	// Copy value
			for(l=0; l < FixTop; ++l) {		// Adjust fixups
				if(peekw(Oseg, j=FixValue[l]) == i)
					pokew(Oseg, j, SymTop); }
			newsymbol(SYMULAB); } }
}

// Process a block of statements
// type:	0 = Main block
//			1 = ELSE,END		(IF)
//			2 = only END		(WHILE, ELSE, SECTION)
//			3 = CASE,END		(SWITCH)
//			4 = WHILE			(DO)
int statement(U8 type)
{
	U16 i, a, b, c;
	static unsigned abase;

	for(;;) {
		if(Otop > FindAdr) {
			warn("~Find %04x-%04x", abase, Otop-1);
			FindAdr = -1; }
		abase = Otop;
		switch(get_line()) {
		case 0 :
			if(type)
				error("Unexpected EOF");
			return 0;
		case ELSE :
			if(type != 1) {
				warn("Unexpected ELSE");
				continue; }
			return 2;
		case END :
			if(type == 4)
				warn("Unexpected END");
			return 1;
		case VARIABLE:
			a = SYMVAR;
			b = NumVar;
	defs:	getsymbol(GS_VALID|GS_NEW);
			constant(L, 0);
			longcpy(SymValue[SymTop], L);
			newsymbol(a);
			if(b) {
				if(L[1] || (*L >= b))
					warn("Out of range"); }
			continue;
		case CONSTANT:
			a = SYMCON;
			b = 0;
			goto defs;
		case FIXED:
			getsymbol(GS_VALID|GS_NEW);
			longset(SymValue[SymTop], Ptop);
			newsymbol(SYMFIX);
			a = Otop;
			string(255);
			for(b=0; b < Otop; ++b)
				padd(peek(Oseg, a+b));
			Otop = a;
			continue;
		case STRING:
			a = SYMSTR;
			b = NumStr;
			goto defs;
		case SET:
			oc(XSET);
			getvar(255);
			string(0);
			continue;
		case CASE :
			if(type != 3) {
				warn("Unexpected CASE");
				continue; }
			if(!UnReach) {
				oc(XBRANCH);
				docase(1);
				ow(0); }
			docase(UnReach = 0);
			expression(0);
			continue;
		case IF :
			IFbase1 = Otop;
			a = conditional(0, 0);
			IFbase2 = Otop;
			b = 0;
			if(statement(1) == 2) {	// Else
				if(!UnReach) {
					oc(XBRANCH);
					b = Otop;
					ow(0); }
				ph(a);
				a = b;
				statement(2); }
			ph(a);
			if(((Otop - IFbase2) == 3) && (peek(Oseg, IFbase2) == XBRANCH) && (Options & OP_JMP)) {
				switch(b=peek(Oseg, IFbase1)) {
				default: error("?branch %04x %02x", IFbase1, b);
				case XBTRUE : b = XBFALSE;	break;
				case XBFALSE : b = XBTRUE; }
				poke(Oseg, IFbase1, b);
				pokew(Oseg, a, peekw(Oseg, ++IFbase2));
				if(FixTop && (FixValue[FixTop-1] == IFbase2))
					FixValue[FixTop-1] = a;
				Otop -= 3;
				--OCcount; }
			IFbase2 = -1;
			continue;
		case SWITCH:
			oc(XSWITCH);
			expression(0);
		ds:	a = CLtop;
			b = Otop;				// Index array
			ow(c=0);
			statement(3);
			if(!UnReach) {
				oc(XBRANCH);
				c = Otop;			// Exit location
				ow(0); }
			pw(b, Otop);			// Point to table
			for(i=a; i < CLtop; ++i) {	// Fill in table
				if(!CLtype[i])
					ow(CLvalue[i]); }
			ow(0);
			ph(c);
			b = Otop;
			for(i=a; i < CLtop; ++i) {	// Fixup case exits
				if(CLtype[i] == 1)
					pw(CLvalue[i], b); }
			CLtop = a;
			continue;
		case WHILE:
			if(type == 4)
				return 1;
			a = Otop;
			b = conditional(0, 0);
			statement(2);
			if(!UnReach) {
				oc(XBRANCH);
				ow(a); }
			ph(b);
			continue;
		case DO:
			a = Otop;
			statement(4);
			conditional(255, a);
			continue;
		case CALL:
			oc(XCALL);
			dolabel();
			continue;
		case GOTO:
			oc(XBRANCH);
			dolabel();
			continue;
		case RETURN:
			oc(XRETURN);
			continue;
		case RESET:
			oc(XRESET);
			expression(0);
			continue;
		case STOP:
			oc(XSTOP);
			continue;
		case INCLUDE:
			a = b = Otop;
			string(255);
			c = 0;
			while(b < Otop)
				Name[c++] = peek(Oseg, b++);
			Otop = a;
			if(Itop >= IDEPTH)
				error("INCLUDE too deep");
			Iline[Itop] = Line;
			Ifp[Itop] = Fp;
			if(!(Fp = fopen(Name, "r")))
				error("Cannot open: %s", Name);
			Line = 0;
			++Itop;
			if(!(Lst & 0xF0))
				Lst = 0;
			continue;
		case NOLIST:
			Lst = 0;
			continue;
		case SECTION:		// Local block
			a = Sbase;			// Save symbol table base address
			b = SymBase;		// Save symbol table base index
			c = Ptop;			// Save string pool top
			Sbase = Stop;
			SymBase = SymTop;
			statement(2);
			close_section(a, b);
			Ptop = c;
			continue;
		case 0xFFFF :		// Possible expression for side-effects
			oc(XEVAL);
			value_expression(Etop = 0);
			if(Options & OP_ELIM) {
				for(i=a=b=0; i < Etop; ++i) {	// Remove no-effect operations
					switch(Etype[i]) {
					case OTASS:
					case OPKASS:
						++b; }
					if(b) {
						longcpy(Evalue[a], Evalue[i]);
						Etype[a++] = Etype[i]; }
					if((Etype[i] == OTOPR) && (*Evalue[i] == ODASS))
						--b; }
				if(a != Etop) {
					warn("*@Superfluous operations removed");
					if(!(Etop = a))
						--Otop; } }
			output_expression();
			continue;
		default:			// User Defined
			if((*F & 255) == MAPopcode) {
				if(Mtop >= MAXMAP)
					error("MAP list full");
				MAPlist[Mtop++] = Otop+1; }
			if((*F & 0x700) != 0x700)
				oc(*F);
			b = c = 0;
			while(i = (*F & 0x700)>>8) {
				longshr(F);
				longshr(F);
				longshr(F);
				a = (*F & 0x700) ? ',' : 0;
				switch(i) {
				case 1 : expression(a);	continue;		// Value
				case 2 : string(0);		break;			// String
				case 3 : getvar(0);		break;			// Nvar
				case 4 : getvar(255);	break;			// Svar
				case 5 : dolabel();		break;			// Label
				case 6 : c = 1;			continue; }		// SWITCH
				if(a) {		// More parameters
					if(skip() != ',')
						warn("',' expected");
					else
						++Ptr; } }
			if(c == 1)
				goto ds; 		// switch
		continue;
	} }
}

// Display expression output buffer
#if 0
dumpout()
{
	U16 i;
	U8 temp[16];
	static U8 *ops[] = {
	"m-", "m!", "m~", "@",
	"+", "-", "*", "/", "%", "==", "!=", "<", ">", "<=", ">=",
	"<<", ">>", "&", "|", "^", "&&", "||", "=", "[]", "[=]",
	"?", "?", "?", "?" };

	for(i=0; i < Etop; ++i) {
		if(i)
			putc(' ', stdout);
		switch(Etype[i]) {
		case OTVAL :
			ltoa(Evalue[i], temp, 10);
			printf("%s", temp);
			continue;
		case OTVAR :
			printf("V%u", *Evalue[i]);
			continue;
		case OTASS :
			printf("A%u", *Evalue[i]);
			continue;
		case OTIDX :
			printf("I%u", *Evalue[i]);
			continue;
		case OTOPR :
			printf("%s", ops[*Evalue[i]]);
			continue;
		} error("Unknown type %u", Etype[i]); }
	putc('\n', stdout);
}
#endif

/*
 * Open file with default extension
 * On exit: Ptr points to filename (without extension)
 */
FILE *openf(U8 hdir, U8 *name, U8 *ext, U8 *opt)
{
	U16 i;
	U8 *p, *ep, df, of, pf;
	FILE *fp;

	p = Name;
	if(hdir) {
		strcpy(p, Aptr);
		p += Noffset; }
	Nptr = p;
	df = pf = of = 0;
	if(name) {		// Name supplied
		for(;;) switch(*p++ = *name++) {
			case 0 : goto x1;
			case '.': df = 255; ep = p-1; continue;
			case ':' :
			case '/' :
			case '\\':
				if(hdir) {
					*p = 0;
					strcpy(Name, Name+Noffset);
					p -= Noffset; }
				pf = 255;
				df = hdir = 0;
				Nptr = p; }
	x1:		if(!df)
				strcpy(ep = p-1, ext); }
	else {
		strcpy(p, Aptr+Noffset);
		switch(p[(i = Eoffset - Noffset)-1]) {
		case 'C' :
		case 'c' :
			--i; }
		strcpy(ep = p+i, ext); }
	for(;;) {
		switch(*opt) {
		default: goto s3;
		case 'P' : if(pf) goto s1;		// Error if PATH
		case 'p' : if(pf) goto s2;		// Verbose if PATH
			break;
		case 'N' : if(name) goto s1;	// Error if NAME
		case 'n' : if(name) goto s2;	// Verbose if NAME
			break;
//		case 'E' : if(df) goto s1;
//		case 'e' : if(df) goto s2;
//			break;
		case 'V' : s1: of = 3;			// Error unconditionally
		case 'v' : s2: of |= 1;	}		// Verbose unconditionally
		++opt; } s3:
	if(!(fp = fopen(Name, opt))) {
		if(of & 1)
			printf("Unable to open: %s\n", Name);
		if(of & 2)
			exit(-1); }
	*ep = 0;
	return fp;
}

// Lookup keyword
U16 lookup(void)
{
	U16 i, j, t;
	static U8 *kw[] = {
		"EXT", "BEXT", "CEXT", "SEXT", "NAME", "BIT16",
		"NUMVAR", "NUMSTR", "CONSTANT", "SYSVAR", "FUNCTION",
		"VALUE", "STRING", "NVARIABLE", "SVARIABLE", "LABEL", "SWITCH",
		"MAP" };
	for(i=t=0; i < (sizeof(kw)/sizeof(kw[0])); ++i) {
		if(strbeg(kw[i], Name)) {
			j = i;
			++t; } }
	return (t == 1) ? j : -1;
}
#define	EXT			0
#define	BEXT		1
#define	CEXT		2
#define	SEXT		3
#define	NAME		4
#define	EBIT16		5	// Here and above are 2nd pass only
#define	ENUMVAR		6
#define	ENUMSTR		7
#define	ECONSTANT	8
#define	ESYSVAR		9
#define	EFUNCTION	10
#define	EVALUE		11	// Here and above are function operand types
#define	ESTRING		12
#define	ENVARIABLE	13
#define	ESVARIABLE	14
#define	ELABEL		15
#define	ESWITCH		16
#define	EMAP		17

// Encode function parameters
void load_extensions(U8 lf)
{
	U16 i, p, ft, l[2];
	static U8 loaded;

	if(!lf) {
		if(loaded)
			return;
		loaded = 255; }

	if(!(Fp = openf(0, Efile, EEXT, "Pr")))
		if(!(Fp = openf(1, Efile, EEXT, "Nr")))
			return;
	ft = XUSER;
	while(fgets(Ptr = Buffer, sizeof(Buffer)-1, Fp)) {
		++Line;
		if(!skip())
			continue;
		getsymbol(GS_VALID);			// Get keyword
		if((i = lookup()) < EBIT16) {	// Set name
			if(!lf) {
				Hindex[i] = Pool + Ptop;
				if(!(p = skip())) { bs:
					warn("Bad string");
					continue; }
				while((i = *++Ptr) != p) {
					if(!i) goto bs;
					padd(i); }
				padd(0); }
			continue; }
		if(lf) switch(i) {
			default: bk: warn("Unknown keyword: %s", Name);
			case EBIT16  :	Bit32 = 0;					continue;
			case ENUMVAR :	NumVar = constant(L, 0);	continue;
			case ENUMSTR :	NumStr = constant(L, 0);	continue;
			case ECONSTANT: i = SYMCON; p = 0;	dosym:
				getsymbol(GS_VALID|GS_NEW);
				constant(L, 0);
				if(p) {
					if(L[1] || (*L >= p))
						warn("Out of range"); }
				longcpy(SymValue[SymTop], L);
				newsymbol(i);
				continue;
			case ESYSVAR :		i = SYMSYS; p = 7;		goto dosym;
			case ENVARIABLE:	i = SYMVAR; p = NumVar;	goto dosym;
			case ESVARIABLE:	i = SYMSTR; p = NumStr;	goto dosym;
			case EFUNCTION:
				if(skip() == '(') {
					++Ptr;
					ft = constant(L, ')'); }
				getsymbol(GS_VALID|GS_NEW);
				Debug("%5u %-5u %s\n", ft, ft-XUSER, Name);
				longset(L, ft++);
				p = 8;
				while(skip()) {
					getsymbol(GS_VALID);
					if((i = lookup()) < EVALUE)
						warn("Bad option");
					if(i == EMAP) {
						if(MAPopcode != -1)
							warn("Can only MAP 1 opcode");
						MAPopcode = ft-1;
						continue; }
					if(i > ESWITCH) goto bk;
					longset(l, i-(EVALUE-1));
					if(p >= 32)
						warn("Too many options");
					i = p;
					while(i--)
						longshl(l);
					L[1] |= l[1];
					*L |= *l;
					p += 3; }
				longcpy(SymValue[SymTop], L);
				newsymbol(SYMUSR); }
	}
	fclose(Fp);
	Line = 0;
}

unsigned char *Xname(unsigned char *name)
{
	if(!name) {
		strcpy(name = Buffer, Iname);
		Ptr = 0;
		for(;;) switch(*name++) {
			case 0 :
				if(Ptr)
					*Ptr = 0;
				return Buffer;
			case '.' : Ptr = name-1;	continue;
			case ':' :
			case '/' :
			case '\\': Ptr = 0; } }
	return name;
}

/*
 * Process a language statement
 */
int main(int argc, char *argv[])
{
	U16 i, j, k, l;
	U8 *p;
	static U16 ct;

#ifdef STKMON
	asm {
		EXTRN	?heap:near
		MOV		SI,offset DGRP:?heap + 16
		MOV		DI,SP
		SUB		DI,256
fml:	MOV		byte ptr [SI],0A5h
		INC		SI
		CMP		SI,DI
		JB		fml
	}
#endif

	// Get home directory
	Aptr = argv[0];
	i = 0;
	for(;;) switch(Aptr[i++]) {
		case 0 :
			if(!Eoffset)
				Eoffset = i-1;
			goto xx;
		case '.' :
			Eoffset = i-1;
			continue;
		case ':' :
		case '/' :
		case '\\':
			Noffset = i; Eoffset = 0; }
xx:	strcpy(Pool, Aptr + Noffset);
//	if(Pool[(i = Eoffset - Noffset)-1] == 'C')
//		--i;
	Pool[i = Eoffset - Noffset] = 0;
	Ptop += (i+1);

	for(i=1; i < argc; ++i) {
		Ptr = argv[i];
		switch((toupper(*Ptr++) << 8) | toupper(*Ptr++)) {
		case ('-'<<8)|'O' :
		case ('/'<<8)|'O' :
			for(;;) {
				switch(toupper(*Ptr++)) {
				default: goto help;
				case 0 : goto cont;
				case 'C' : j = OP_CONST;		break;
				case 'I' : j = OP_INDEX;		break;
				case 'J' : j = OP_JMP;			break;
				case 'R' : j = OP_ELIM;			}
				Options &= ~j; }
		case ('-'<<8)|'D' :
		case ('/'<<8)|'D' : j = OP_DEBUG;				goto setop;
		case ('-'<<8)|'Q' :
		case ('/'<<8)|'Q' : j = OP_QUIET;				goto setop;
		case ('-'<<8)|'S' :
		case ('/'<<8)|'S' : j = OP_DMPSYM;				goto setop;
		case ('-'<<8)|'C' :
		case ('/'<<8)|'C' :
			j = OP_COUT;
			if(*Ptr)
				Cwidth = constant(L, 0);
	setop:	Options |= j;
	cont:	continue;
		case ('L'<<8)|'=' : Lname = Ptr;
		case ('/'<<8)|'L' :
		case ('-'<<8)|'L' :
			Lst = Lst1 = (toupper(*Ptr) == 'I') ? 0xFF : 0x0F;
			continue;
		case ('E'<<8)|'=' : Efile = Ptr;				continue;
		case ('F'<<8)|'=' :	FindAdr = constant(0, 0);	continue;
		}
		if(issymbol(*(Ptr -= 2))) {		// Command line assign
			p = Ptr;
			while(issymbol(*++Ptr));
			if(*Ptr == '=') {
				CLvalue[ct++] = i;
				continue; }
			Ptr = p; }
		if(!Iname) {
			Iname = Ptr;
			continue; }
		if(!Oname) {
			Oname = Ptr;
			continue; }
		goto help; }

	load_extensions(0);
	if(!(Options & OP_QUIET))
#ifndef _MICROC_
		printf("%s Compiler - "__DATE__"\n", Hindex[NAME]);
#else
		printf("%s Compiler - "#__DATE__"\n", Hindex[NAME]);
#endif

	if(!Iname) { help:
		Hindex[CMD] = Pool;
		p = Help;
		while(i = *p++) {
			if(i & 0x80) {
				fputs(Hindex[i&7], stdout);
				if(i == (CMD|0x80))
					Pool[(Eoffset - Noffset)-1] = 0;
				continue; }
			putc(i, stdout); }
		exit(0); }

	Oseg = (Seg = alloc_seg(8192)) + 4096;
	i = 0; do {
		pokew(Seg, i, 0);
		pokew(Oseg, i, 0);
		i += 2; }
	while(i);

	// Add fixed keywords to symbols dictionary
	for(i=0; i < (sizeof(KeyWords)/sizeof(KeyWords[0])); ++i) {
		SegAdd(KeyWords[i]);
		longset(SymValue[SymTop], i);
		newsymbol(SYMKEY); }

	// Add user-defined keywords
	load_extensions(255);
	LEflag = 255;

	// Add command-line constant assignments
	for(i=0; i < ct; ++i) {
		Ptr = argv[CLvalue[i]];
		getsymbol(GS_VALID|GS_NEW);
		++Ptr;
		constant(SymValue[SymTop], 0);
		newsymbol(SYMCON); }

	if(Lst) {
		Lname = Xname(Lname);
		Lfp = openf(0, Lname, ".LST", "Vw"); }

	// Compile script
	Fp = openf(0, Iname, Hindex[EXT], "Vr");
	Ptr = "";
	if(statement(0)) {
		if(get_line())
			warn("*Code after final END"); }
	if(!UnReach)
		oc(XSTOP);
	if(Lfp)
		fclose(Lfp);
	fclose(Fp);

	// Check for unresolved symbols
	while(i < SymTop) {
		if(SymType[i] == SYMULAB) {
			SegGet(0, i, Name);
			Line = *SymValue[i];
			warn("!@Unresolved: %s", Name); }
		++i; }
	Line = 0;

	if(Wcount)
		error("Compile failed");

	// Display summary
	if(!(Options & OP_QUIET)) {
		printf("%5u of %-5u Symbols\n", SymTop, SYMBOLS);
		printf("%5u of %-5u Max expression buffer depth\n", Emax, EBUFFER);
		printf("%5u of %-5u Max expression stack  depth\n", Smax, ESTACK);
//		printf("%5u of %-5u Fixups\n", FixTop, FIXUPS);
//		printf("%5u of %-5u Mapped opcodes\n", Mtop, MAXMAP);
		printf("%5u of 65536 Symbol table\n", Stop);
		printf("%u opcodes generated, image size is %u\n", OCcount, Otop); }

	Oname = Xname(Oname);

	if(!(Options & OP_COUT)) {		// Write BINARY output file
		Fp = openf(0, Oname, Hindex[BEXT], "Vwb");
		if(MAPopcode != -1) {
			for(i=0; i < Mtop; ++i) {
				putc((j=MAPlist[i]) & 255, Fp);
				putc(j>>8, Fp); }
			putc(0, Fp);
			putc(0, Fp); }
		for(i=0; i < Otop; ++i)
			putc(peek(Oseg, i), Fp); }
	else {							// Write C output file
		Fp = openf(0, Oname, Hindex[CEXT], "Vw");
		if(MAPopcode != -1) {
			fprintf(Fp, "unsigned %s_MAP[] = {", Nptr);
			for(i=j=0; i < Mtop; ++i) {
				if(!j)	fputs("\n\t", Fp);
				j += fprintf(Fp, "%u,", MAPlist[i]);
				if(j >= Cwidth)
					j = 0; };
			fputs("0 };\n", Fp); }
		fprintf(Fp, "unsigned char %s[] = {", Nptr);
		for(i=j=0; i < Otop; ++i) {
			if(i)	putc(',', Fp);
			if(!j)	fputs("\n\t", Fp);
			j += fprintf(Fp, "%u", peek(Oseg, i)) + 1;
			if(j >= Cwidth)
				j = 0; }
		fputs(" };\n", Fp); }
	fclose(Fp);
#if 0
	Fp = fopen("R:POOL", "wvqb");
	fput(Pool, Ptop, Fp);
	fclose(Fp);
#endif

	if(Options & OP_DMPSYM) {		// Dump symbol table
		Smax = i = 0; do {			// Initial ascending index list
			pokew(Oseg, i+i, i); }
		while(++i < SymTop);

		for(i=1; i < SymTop; ++i) {	// Sort by name
			j = i - 1;
			SegGet(0, i, Name);
			if((k = strlen(Name)) > Smax)
				Smax = k;
			for(k=0; k < j; ++k) {
				SegGet(0, peekw(Oseg, k+k), Buffer);
				if(strcmp(Name, Buffer) < 0) {
					l = peekw(Oseg, i+i);
					pokew(Oseg, i+i, peekw(Oseg, k+k));
					pokew(Oseg, k+k, l);
					strcpy(Name, Buffer); } } }

		for(i=0; i < SymTop; ++i) {	// Output list
			j = peekw(Oseg, i+i);
			SegGet(0, j, Name);
			if( (k = SymType[j]) >= (sizeof(Stype)/sizeof(Stype[0])) )
				k = (sizeof(Stype)/sizeof(Stype[0])) - 1;
			p = Name;
			l = Smax; do {
				putc(*p ? *p++ : ' ', stdout); }
			while(--l);
			printf(" %-9s%04x%04x\n", Stype[k], SymValue[j][1], *SymValue[j]); } }

#ifdef STKMON
	asm {
		MOV		SI,offset DGRP:?heap + 16	// Bottom of free memory
		MOV		DI,SP						// Top of free memory
		XOR		CX,CX						// Zero max count
al1:	XOR		BX,BX						// Zero block count
al2:	CMP		SI,DI						// At end?
		JA		al4							// Exit
		MOV		AL,[SI]						// Get value
		INC		SI							// Advance
		CMP		AL,0A5H						// Match clear value?
		JNZ		al3							// No - new block
		INC		BX							// Advance block size
		JMP short al2						// And continue
al3:	CMP		BX,CX						// Is this block larger?
		JBE		al1							// No
		MOV		CX,BX						// Save max block
		JMP		al1							// And look again
al4:	MOV		DGRP:_Line,CX				// Save largest block
	}
	printf("\n[%u]\n", Line);
#endif
}
