/*
 * Script Decompiler
 */
#include <stdio.h>

#define	EEXT	".esl"

#define	USER	100				// Number of user supplied functions

#define	U8		unsigned char
#define	U16		unsigned

#include "eslopc.h"

#define		F_DATA		0x01	// Switch data
#define		F_CASE		0x02	// Switch case statement
#define		F_MAP		0x04	// Mapped address

U16
	Seg,				// External segment
	Fseg,				// Flag segment
	Stop,				// Top of external segment
	Ptop,				// Top of string pool
	Utop,				// Top of user list
	Offset,				// Execution offset
	Loffset,			// Last execution offset
	Uopcode,			// User opcode
	Noffset,			// Offset to name
	Eoffset,			// Offset to extension
	Horz,				// Horizontal output position
	Width = 40,			// Display width
	Nwidth = 5-1,		// Newline width
	Wmax = 77-3,		// Maximum width
	MAPopcode = -1,		// Mapped opcode
	L[2],				// General long
	Uvalue[USER][2];	// User value
FILE
	*Fp,				// Input file
	*Fpo;				// Output file
U8
	*Ptr,				// General pointer
	*Ifile,				// Input file
	*Efile,				// Code file
	*Ofile,				// Output file
	*Aptr,				// argv[0] pointer
	*Uname[USER],		// User function names
	MAPopt,				// Map options
	Xout = 255,			// Output flag
	Bit32 = 255,		// 32-bit output
	Quiet,				// Quiet mode
	DumpOpcode,			// Dump opcode bytee
	Method,				// Load method
	Name[128],			// Encoded name
	Temp[128],			// Temp buffer
	Pool[32768];		// Symbol pool

U8 *Hindex[] = {
	".ES",											// 0 Extension
	".ESB",											// 1 Binary output
	".ESC",											// 2 C output
	".ESS",											// 3 String file
	"Extensible Script Language (ESL)",				// 4 Name
	0,												// 5 Command
	EEXT };											// 6 extension extension
#define	CMD		5
U8 Help[] = { "\n\
use:	\x85	infile[\x81|\x82] [outfile] [options]\n\n\
opts:	/B		- load Binary file		[by extension]\n\
	/C		- load C format file		[by extension]\n\
	/I		- Identify mapped sections\n\
	/M		- show only Mapped opcodes\n\
	/O		- include Opcode display\n\
	/Q		- Quiet: reduce output\n\
	E=file[\x86]	- specify Extension file	[\x85\x86]\n\
	W=s,e,o		- set Opcode columns		[40,77,4]\n\
\nExtensible Script Language - ?COPY.TXT 2005-2019 Dave Dunfield.\n" };

// Report error and terminate
register error(U16 args)
{
	U8 buf[81];
	_format_(nargs() * 2 + &args, buf);
	fputs(buf, stdout);
	exit(-1);
}

Xputc(int c)			// Output string with horizontal tracking
{
	if(Xout) {
		putc(c, Fpo);
		++Horz; }
}
Xputs(U8 *s)
{						// Output string
	while(*s)
		Xputc(*s++);
}
register Xprintf(U16 args)		// Formatted output
{
	U8 buf[128];
	_format_(nargs() * 2 + &args, buf);
	Xputs(buf);
}

void Fset(o, v) asm
{
		MOV		ES,DGRP:_Fseg		// Get external segment
		MOV		BX,6[BP]			// Get offset
		MOV		AL,ES:[BX]			// Get value
		OR		AL,4[BP]			// Set bits
		MOV		ES:[BX],AL			// Resave
}

// Add to string pool
void padd(int c)
{
	if(Ptop >= sizeof(Pool))
		error("Out of memory");
	Pool[Ptop++] = c;
}

// Skip to non-blank
int skip()
{
	while(isspace(*Ptr))
		++Ptr;
	switch(*Ptr) {
	case ';' :
	case '/' : *Ptr = 0; }
	return *Ptr;
}

// get byte from code segment
U16 gb()
{
	if(Offset >= Stop)
		error("Unexpected EOF");
	return peek(Seg, Offset++);
}

// Get word from code segment
U16 gw()
{
	U16 v;
	v = gb();
	return (gb() << 8) | v;
}

// Evaluate expression
void expression(void)
{
	U8 b, c, v, f, l[4];

	static U8 *ops[] = {
	"m-", "m!", "m~", "@",
	"+", "-", "*", "/", "%", "==", "!=", "<", ">", "<=", ">=",
	"<<", ">>", "&", "|", "^", "&&", "||", "=", "[]" };

	f = 0;
	do {
		b = gb();
		c = b & 0x1F;
		if(f) Xputc(' ');
		f = 255;
		switch(b & 0x60) {
		case 0x00:
			Xprintf("%u", c);
			continue;
		case 0x20:
			v = gb();
			Xprintf("%u", (v << 5) | c);
			continue;
		case 0x40 :
			Xprintf("V%u", c);
			continue; }
		if(c & 0x18) {
			Xprintf("%s", ops[c - 0x08]);
			continue; }
		if(c &= 0x07) {
			Xprintf("Sy%u", c-1);
			continue; }
		l[0] = gb();
		l[1] = gb();
		if(Bit32) {
			l[2] = gb();
			l[3] = gb(); }
		else
			l[2] = l[3] = 0;
		ltoa(l, Temp, 10);
		Xprintf("%s", Temp); }
	while(b & 0x80);
}


// Pick a string delimiter
int delim(void)
{
	int c;
	U16 o;
	U8 s, v;
	o = Offset--;
	s = 255;
	while(c = gb()) {
		if(c & 0x80)
			break;
		switch(c) {
		default: continue;
		case '"' : v = ~0x01;	break;
		case '\'': v = ~0x02;	break;
		case '/' : v = ~0x04;	break;
		case '\\': v = ~0x08;	break;
		case '`' : v = ~0x10;	break;
		case '|' : v = ~0x20;	break;
		case '!' : v = ~0x40;	break;
		case '~' : v = ~0x80;	}
		s &= v; }
	Offset = o;
	if(s & 0x01) return '"';
	if(s & 0x02) return '\'';
	if(s & 0x04) return '/';
	if(s & 0x08) return '\\';
	if(s & 0x10) return '`';
	if(s & 0x20) return '|';
	if(s & 0x40) return '!';
	if(s & 0x80) return '~';
	return '?';
}

// Handle an encoded string
string()
{
	int c, b, d;
	U8 df, sf;
	df = sf = 0;
	while(c = gb()) {
		if(c & 0x80) {
			if(df) {
				Xputc(d);
				df = 0; }
			if(sf)
				Xputc(' ');
			sf = 255;
			switch(c & 0x60) {
			case 0x00:		// String variable
				Xprintf("S%u", c & 0x1F);
				continue;
			case 0x20 :	b = 'b'; 	break;
			case 0x40 :	b = 'd';	break;
			default:	b = 'x'; }
			Xputc('(');
			expression();
			Xputc(')');
			if(c &= 0x1F)
				Xprintf("%u", c+1);
			Xputc(b);
			continue; }
		if(c < ' ') {
			if(df) {
				Xputc(d);
				df = 0; }
			if(sf)
				Xputc(' ');
			Xprintf("{%u}", c);
			sf = 255;
			continue; }
		if(!df) {
			if(sf)
				Xputc(' ');
			Xputc(d = delim());
			df = sf = 255; }
		Xputc(c); }
	if(df)
		Xputc(d);
}

// Test for valid symbol character
int issymbol(int c)
{
	if((c >= 'a') && (c <= 'z'))
		return 255;
	if((c >= 'A') && (c <= 'Z'))
		return 255;
	return (c == '_');
}

// Get a symbol
int getsymbol()
{
	U8 *p;

	skip();
	p = Name;
	if(issymbol(*Ptr)) {
		while(issymbol(*Ptr) || isdigit(*Ptr))
			*p++ = toupper(*Ptr++); }
	*p = 0;
}

// Get a value element from the input stream
unsigned value_expression(int t);
unsigned value_element(void)
{
	U16 b, c, v;
	U8 f;
	b = 10;
	switch(skip()) {
	case '-' : ++Ptr; return -value_element();
	case '~' : ++Ptr; return ~value_element();
	case '!' : ++Ptr; return !value_element();
	case '(' :
		++Ptr;
		v = value_expression(')');
		++Ptr;
		return v;
	case '\'':
		++Ptr;
		v = 0;
		while((c = *Ptr++) != '\'') {
			if(!c)
				error("Unterminated character constant");
			v = (v << 8) | c; }
		return v;
	case '0' :
		switch(toupper(*++Ptr)) {
		case 'B' : goto bin;
		case 'O' : goto oct;
		case 'X' : goto hex; }
		--Ptr;
		break;
	case '%' : bin: b=2;	goto setb;
	case '@' : oct: b=8;	goto setb;
	case '$' : hex: b=16; 	setb:
		++Ptr; }
	v = f = 0;
	for(;;) {
		c = toupper(*Ptr);
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else
			break;
		if(c >= b)
			break;
		v = (v * b) + c;
		++Ptr;
		f = 255; }
	if(!f)
		error("Numeric value expected");
	return v;
}

// Resolve a numeric expression
unsigned value_expression(int t)
{
	U16 c, v, r;
	v = value_element();
	for(;;) {
		if((c = skip()) == t)
			return v;
		if(!c) {
			if(t != ';')
				error("Bad expression");
			return v; }
		switch(c = (c << 8) | *++Ptr) {
		case ('='<<8)|'=' :
		case ('!'<<8)|'=' :
		case ('<'<<8)|'=' :
		case ('>'<<8)|'=' :
		case ('<'<<8)|'<' :
		case ('>'<<8)|'>' :
		case ('&'<<8)|'&' :
		case ('|'<<8)|'|' :
			++Ptr;
			break;
		default: c >>= 8; }
		r = value_element();
		switch(c) {
		default: error("Bad operation");
		case '+' :			v += r;			continue;
		case '-' :			v -= r;			continue;
		case '*' :			v *= r;			continue;
		case '/' :			v /= r;			continue;
		case '%' :			v %= r;			continue;
		case '<' :			v = v < r;		continue;
		case '>' :			v = v > r;		continue;
		case '&' :			v &= r;			continue;
		case '|' :			v |= r;			continue;
		case '^' :			v ^= r;			continue;
		case ('='<<8)|'=' :	v = v == r;		continue;
		case ('!'<<8)|'=' :	v = v != r;		continue;
		case ('<'<<8)|'=' :	v = v <= r;		continue;
		case ('>'<<8)|'=' :	v = v >= r;		continue;
		case ('<'<<8)|'<' :	v <<= r;		continue;
		case ('>'<<8)|'>' :	v >>= r;		continue;
		case ('&'<<8)|'&' : v = v ? r : 0;	continue;
		case ('|'<<8)|'|' :	v = v ? v : r;	} }
}

/*
 * Open file with default extension
 * On exit: Ptr points to filename (without extension)
 */
FILE *openf(U8 hdir, U8 *name, U8 *ext, U8 *opt)
{
	U8 *p, df, pf, of;
	FILE *fp;
	p = Name;
	if(hdir) {
		strcpy(p, Aptr);
		p += Noffset; }
	df = pf = of = 0;
	if(name) {		// Name supplied
		for(;;) switch(*p++ = *name++) {
			case 0 : goto x1;
			case '.': df = 255; Ptr = p-1; continue;
			case ':' :
			case '/' :
			case '\\':
				pf = 255;
				if(hdir) {
					*p = 0;
					strcpy(Name, Name+Noffset);
					p -= Noffset; }
				df = hdir = 0; }
	x1:		if(!df)
				strcpy(Ptr = p-1, ext); }
	else {
		strcpy(p, Aptr+Noffset);
		switch(p[(hdir = Eoffset-Noffset)-1]) {
		case 'D' :
		case 'd' :
			--hdir; }
		strcpy(Ptr = p+hdir, ext); }
	for(;;) {
		switch(*opt) {
		default: goto s3;
		case 'P' : if(pf) goto s1;
		case 'p' : if(pf) goto s2;
			break;
		case 'N' : if(name) goto s1;
		case 'n' : if(name) goto s2;
			break;
		case 'E' : if(df) goto s1;
		case 'e' : if(df) goto s2;
			break;
		case 'V' : s1: of = 3;
		case 'v' : s2: of |= 1;	}
		++opt; } s3:
	if(!(fp = fopen(Name, opt))) {
		if(of & 1)
			printf("Unable to open: %s\n", Name);
		if(of & 2)
			exit(-1); }
	return fp;
}

// Lookup keyword
U16 lookup()
{
	U16 i, j, t;
	static U8 *kw[] = {
		"EXT", "BEXT", "CEXT", "SEXT", "NAME", "BIT16",
		"NUMVAR", "NUMSTR", "CONSTANT", "SYSVAR", "FUNCTION", "VALUE",
		"STRING", "NVARIABLE", "SVARIABLE", "LABEL", "SWITCH", "MAP" };
	for(i=t=0; i < (sizeof(kw)/sizeof(kw[0])); ++i) {
		if(strbeg(kw[i], Name)) {
			j = i;
			++t; } }
	if(t != 1)
		error("Unknown keyword: %s", Name);
	return j;
}
#define	EXT			0
#define	BEXT		1
#define	CEXT		2
#define	SEXT		3
#define	NAME		4
#define	EBIT16		5	// Here and above are 2nd pass only
#define	ENUMVAR		6
#define	ENUMSTR		7
#define	ECONSTANT	8
#define	ESYSVAR		9
#define	EFUNCTION	10
#define	EVALUE		11	// Here and above are function operand types
#define	ESTRING		12
#define	ENVARIABLE	13
#define	ESVARIABLE	14
#define	ELABEL		15
#define	ESWITCH		16
#define	EMAP		17

// Encode function parameters
void load_extensions(U8 lf)
{
	U16 i, p, l[2];
	static U8 loaded;

	if(!lf) {
		if(loaded)
			return;
		loaded = 255; }

	if(!(Fp = openf(0, Efile, EEXT, "Pr")))
		if(!(Fp = openf(1, Efile, EEXT, "Nr")))
			return;

	while(fgets(Ptr = Temp, sizeof(Temp)-1, Fp)) {
		if(!skip())
			continue;
		getsymbol();
		if((i = lookup()) < EBIT16) {	// Set name
			if(!lf) {
				Hindex[i] = Pool + Ptop;
				if(!(p = skip())) { bs:
					error("Bad string");
					continue; }
				while((i = *++Ptr) != p) {
					if(!i) goto bs;
					padd(i); }
				padd(0); }
			continue; }
		if(lf) switch(i) {
			case EBIT16	: Bit32 = 0;						continue;
//			case EOPCODE: skip();	Uopcode = atoi(Ptr);	continue;
			case EFUNCTION:
				if(Utop >= USER)
					error("Too many user-supplied");
				if(skip() == '(') {
					++Ptr;
					Uopcode = value_expression(')');
					++Ptr; }
				getsymbol();
				Uname[Utop] = Pool + Ptop;
				i = 0;
				do {
					padd(Name[i]); }
				while(Name[i++]);
				longset(L, Uopcode++);
				p = 8;
				while(skip()) {
					getsymbol();
					if((i = lookup()) < EVALUE)
						error("Bad option");
					if(i == EMAP) {
						MAPopcode = Uopcode - 1;
						continue; }
					longset(l, i-(EVALUE-1));
					if(p >= 32)
						error("Too many options");
					i = p;
					while(i--)
						longshl(l);
					L[1] |= l[1];
					*L |= *l;
					p += 3; }
				longcpy(Uvalue[Utop++], L); } }
	fclose(Fp);
}

// Issue newline
void nl()
{
	U16 w;

	w = Width;
	if((((Loffset - Offset)*3)+Width) >= Wmax) {
//		w = Nwidth;
		Horz = Width; }
	if(DumpOpcode && (Loffset != Offset) && Xout) {
		if(Horz >= Width) {
		nl:	Xputc('\n');
			w = Nwidth;
			Horz = 0; }
		while(Horz < w)
			Xputc(' ');
		do {
			if(Horz >= Wmax)
				goto nl;
			Xprintf(" %02x", peek(Seg, Loffset++)); }
		while(Loffset < Offset); }
	Loffset = Offset;
	Xputc('\n');
	Horz = 0;
}

void quit()
{
	int c;
	load_extensions(0);
	Ptr = Help;
	Hindex[CMD] = Pool;
	while(c = *Ptr++) {
		if(c & 0x80) {
			fputs(Hindex[c&=7], stdout);
			if(c == CMD)
				Pool[(Eoffset-Noffset)-1] = 0;
			continue; }
		putc(c, stdout); }
	exit(0);
}

U16 num(void)
{
	U16 c, v;
	U8 f;
	v = f = 0;
	for(;;) {
		if((c = *Ptr - '0') > 9)
			break;
		v = (v * 10) + c;
		f = 255;
		++Ptr; }
	if(!f)
		quit();
	return v;
}

int match(U16 index)
{
	U8 *a, *b;
	a = Ptr;
	b = Hindex[index];
	do {
		if(toupper(*a++) != toupper(*b))
			return 0; }
	while(*b++);
	return 255;
}

unsigned get8(void)
{
	int c;
	if((c = getc(Fp)) == EOF)
		error("Unexpected EOF");
	return c;
}

main(int argc, char *argv[])
{
	int c;
	U16 i, j, k;

	// Get home directory
	Aptr = argv[0];
	i = 0;
	for(;;) switch(Aptr[i++]) {
		case 0 :
			if(!Eoffset)
				Eoffset = i-1;
			goto xx;
		case '.' :
			Eoffset = i-1;
			continue;
		case ':' :
		case '/' :
		case '\\':
			Noffset = i; Eoffset = 0; }
xx:	strcpy(Pool, Aptr + Noffset);
	Pool[i = Eoffset - Noffset] = 0;
	Ptop += (i+1);

	for(i=1; i < argc; ++i) {
		Ptr = argv[i];
		switch((toupper(*Ptr++) << 8) | toupper(*Ptr++)) {
		case '-B' :
		case '/B' : Method |= 1;				continue;
		case '-C' :
		case '/C' : Method |= 2;				continue;
		case '-O' :
		case '/O' : DumpOpcode = 255;			continue;
		case '-Q' :
		case '/Q' : Quiet = 255;				continue;
		case '-I' :
		case '/I' : MAPopt = 255;				continue;
		case '-M' :
		case '/M' : MAPopt = 0x55;				continue;
		case 'E=' : Efile = Ptr;				continue;
		case 'W=' :
			Width = num();
			if(*Ptr++ == ',') {
				Wmax = num()-3;
				if(*Ptr++ == ',') {
					if(!(Nwidth = num()-1))
							Nwidth = 1; } }
			if(Wmax <= Width)
				Wmax = Width + 1;
			continue;
		} Ptr -= 2;
		if(!Ifile)
			Ifile = Ptr;
		else if(!Ofile)
			Ofile = Ptr;
		else
			quit(); }
	load_extensions(0);
	if(!Quiet)
#ifndef _MICROC_
		printf("%s Decompiler - "__DATE__"\n", Hindex[NAME]);
#else
		printf("%s Decompiler - "#__DATE__"\n", Hindex[NAME]);
#endif

	if(!Ifile)
		quit();

	Fseg = (Seg = alloc_seg(8192)) + 4096;
	i = 0; do {
		pokew(Seg, i, 0);
		pokew(Fseg, i, 0); }
	while(i -= 2);

	// Read user defined function definitions
	Uopcode = XUSER;
	load_extensions(255);

	// Open input file
	Fp = 0;
	if(Method != 2)
		Fp = openf(0, Ifile, Hindex[BEXT], "Evrb");
	if((Method != 1) && !Fp)
		Fp = openf(0, Ifile, Hindex[CEXT], "Vrb");
	if(!Fp)
		exit(0);

	switch(Method) {
	default: quit();
	case 0 :
		if(match(BEXT))
			goto mb;
		else if(match(CEXT))
			goto mc;
		error("Unknown type '%s' - use /B or /C\n", Ptr);
	case 1 :	// Binary
mb:		if(MAPopcode != -1) for(;;) {
			i = get8();
			if(!(i |= (get8() << 8)))
				break;
			Fset(i-1, F_MAP); }
		while((c = getc(Fp)) != EOF)
			poke(Seg, Stop++, c);
		break;
	case 2 :	// C
mc:		j = 0;
		i = (MAPopcode != -1) ? 0 : 2;
		while((c = getc(Fp)) != EOF) {
			if(c <= ' ')
				continue;
			switch(k=i) {
			case 0 :		// Waiting for MAPlist
			case 2 :		// Waiting for DATA
				if(c == '{')
					++i;
				continue;
			case 1 :		// MAPlist
			case 3 :		// DATA
				if((c >= '0') && (c <= '9')) {
					j = (j * 10) + (c - '0');
					continue; }
				switch(c) {
				case '}' :
					++i;
				case ',' :
					if(k == 1)
						Fset(j-1, F_MAP);
					else
						poke(Seg, Stop++, j);
					j = 0; } } } }
	fclose(Fp);

	if(Ofile)
		Fpo = fopen(Ofile, "wvq");
	else
		Fpo = stdout;

	while(Offset < Stop) {
		if(MAPopt == 0x55) Xout = 0;
		if((i = peek(Fseg, Offset)) & F_CASE) {
			Xprintf("%04x-CASE ", Offset);
			expression();
			nl();
			continue; }
		else if(i & F_DATA) {
			Xprintf("%04x-[%04x]", Offset, peekw(Seg, Offset));
			/* Loffset = */ Offset += 2;
			nl();
			continue; }
		if(i & F_MAP) {
			if(MAPopt) {
				nl();
				Xout = 255; } }
		Xprintf("%04x-", Offset);
		switch(c = peek(Seg, Offset++)) {
		default:
			for(i=j=0; i < Utop; ++i) {
				longcpy(L, Uvalue[i]);
				if((*L & 255) == c)
					goto fu; }
			error("Unknown opcode: %d-%02x\n", c, c);
		fu:	Xputs(Uname[i]);
			c = ' ';
			while(i = (*L & 0x700) >> 8) {
				longshr(L);
				longshr(L);
				longshr(L);
				if(i == 6) {		// switch
					j = 255;
					continue; }
				Xputc(c);
				c = ',';
				switch(i) {
				case 1 : expression();			continue;
				case 2 : string();				continue;
				case 3 : Xprintf("v%u", gb());	continue;
				case 4 : Xprintf("s%u", gb());	continue;
				case 5 : Xprintf("l%04x", gw());	} }
			if(j) goto ds;
			nl();
			continue;
		case XEVAL :
			Xprintf("EVAL ");
			expression();
			nl();
			continue;
		case XSWITCH :
			Xprintf("SWITCH ");
			expression();
		ds:	Xprintf(" [%04x]", i = gw());
			for(;;) {
				Fset(i, F_DATA);
//				Fset(i+1, F_DATA);
				if(!(j = peekw(Seg, i)))
					break;
				Fset(j, F_CASE);
				if((i += 2) >= Stop)
					error("Out of bounds"); }
			nl();
			continue;
		case XRETURN:
			Xprintf("RETURN");
			nl();
			continue;
		case XRESET:
			Xprintf("RESET ");
			expression();
			nl();
			continue;
		case XSTOP:
			Xprintf("STOP");
			nl();
			continue;
		case XSET:
			Xprintf("SET S%u ", gb());
			string();
			nl();
			continue;
		case XCALL: Xprintf("CALL");		goto db;
		case XBRANCH: Xprintf("BRANCH");	goto db;
		case XBTRUE: Xprintf("BTRUE ");	goto dc;
		case XBFALSE: Xprintf("BFALSE ");
		dc:	expression();
		db:	Xprintf(" ->%04x", gw());
			nl();
			continue;
	} }
	Xout = 255;
	Xprintf("%04x-END\n", Offset);
	fclose(Fpo);
}
