/*
 * Sample ESL-32 runtime implementation in GNU-C
 *
 * This is a standard C 32-bit compiler.
 *
 * Tested runtime platform is linux. Since this uses the linux time
 * library (for the user defined system variable TICK), you must
 * compile with: gcc eslgcc.c -lrt
 *
 * Dave Dunfield
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "eslopc.h"

// Script execution parameters
#define	VARS		32			// Number of variables
#define	STRS		4			// Number of string buffers
#define	STACK		32			// Evaluation stack size
#define	CSTACK		8			// Call stack size

// User defined system variables
#define	SYS_TICK	0			// 1ms clock tick

// User defined functions
#define	XPRINT		XUSER+0		// PRINT string

unsigned
	Pc,							// Program counter
	Sp,							// Evaluation stack pointer
	Csp,						// Call stack pointer
	Ctop,						// Upper code address
	Stack[STACK],				// Evaluation stack
	Cstack[CSTACK],				// Call stack
	Vars[VARS];					// Numeric variables
unsigned char
	*Code,						// Code buffer pointer
	Strings[STRS][128],			// String variables
	Temp[256];					// Temp storage (string buffer)

// Macros to read byte/words from code buffer
#define	gb(x)	Code[x]
#define	gw(x)	((Code[x+1] << 8) | Code[x])

/*
 * Evaluate an encoded RPN numeric expression
 *
 * If argument 'p' is non-zero, it contains the address to process,
 *    ending address is returned and expression result is: Stack[Sp-1]
 * If argument 'p' is zero, decode at PC (and update), result is returned.
 */
unsigned eval(unsigned p)
{
	unsigned v, l, r, *t;
	unsigned char c, f;

	Sp = f = 0;
	if(!p) {		// No offset, assume PC in and out
		p = Pc;
		f = 255; }

	do {
		v = (c = gb(p++)) & 0x1F;
		switch(c & 0x60) {				// Element type
		case 0x00:						// 5-bit value
		s1:	Stack[Sp] = v;				// Stack value
			if(Sp++ >= STACK)			// Test stack overflow
				error("Stack overflow");
			continue;
		case 0x20:						// 13-bit value
			v = (gb(p++) << 5) | v;
			goto s1;
		case 0x40:						// Variable
			v = Vars[v];
			goto s1; }
		if(!(v & 0x18)) switch(v) {		// Special value elements
			default: error("?SYSVAR%u [%x]", v-1, p-1);
			case 0 :		// 32-bit value
				v = gb(p++);
				v |= gb(p++) << 8;
				v |= gb(p++) << 16;
				v |= gb(p++) << 24;
				goto s1;

			//
			// Place handlers for user-defined system variables here
			//
			case SYS_TICK+1 :		// 1-ms tick
			{	struct timespec tv;
				clock_gettime(CLOCK_MONOTONIC, &tv);
				v = (tv.tv_sec * 1000ul) + (tv.tv_nsec / 1000000ul); }
				goto s1; }

		// Arithmetic operation
		r = *(t=&Stack[Sp-1]);			// Get right operand
		if((v -=8) >= ODADD)			// Dyadics require left operand
			l = *(t = &Stack[--Sp - 1]);
		switch(v) {						// Operators
		default: error("Bad op %02x [%04x]", c, p-1);
		case OMNEG:	*t = -r;				break;
		case OMNOT:	*t = !r;				break;
		case OMCOM:	*t = ~r;				break;
	// GCC warns that 'l' may not be initialized, however it will be
	// for all operators >= ODADD (which is where it is used).
		case ODADD:	*t = l + r;				break;
		case ODSUB:	*t = l - r;				break;
		case ODMUL:	*t = l * r;				break;
		case ODDIV:	*t = l / r;				break;
		case ODMOD:	*t = l % r;				break;
		case ODEQ:	*t = l == r;			break;
		case ODNE:	*t = l != r;			break;
		case ODLT:	*t = l < r;				break;
		case ODGT:	*t = l > r;				break;
		case ODLE:	*t = l <= r;			break;
		case ODGE:	*t = l >= r;			break;
		case ODSL:	*t = l << r;			break;
		case ODSR:	*t = l >> r;			break;
		case ODBAND:*t = l & r;				break;
		case ODBOR:	*t = l | r;				break;
		case ODBXOR:*t = l ^ r;				break;
		case ODLAND:if(l)  *t = r;			break;
		case ODLOR:	if(!l) *t = r;			break;
		case ODASS:	*t = Vars[l] = r;		break;
		case ODIDX:	*t = Vars[l+r];			} }
	while(c & 0x80);		// More elements

	if(f) {					// No offset, save PC and return value
		Pc = p;
		return Stack[Sp-1]; }

	return p;				// Return new offset
}

/*
 * Process a string at the current position
 */
unsigned char *string()
{
	unsigned c, csp, v, b;
	unsigned char *p, *p1, cs[33];

	p = Temp;
	while(c = gb(Pc++)) {
		switch(c & 0xE0) {		// Type of string operation
		default:					// Normal character
			*p++ = c;
			continue;
		case 0x80 :					// Insert string buffer
			p1 = Strings[c & 0x1F];
			while(*p1)
				*p++ = *p1++;
			continue;
		case 0xA0 :	b = 2;		break;	// Special (binary in example)
		case 0xC0 :	b = 10;		break;	// Decimal
		case 0xE0 : b = 16;		}		// Hexidecimal
		v = eval(csp = 0);		// Eval expression, reset stack
		do {					// Stack digits in reverse order
			if((cs[csp] = v % b) > 9)
				cs[csp] += 7;	// Adjust for hex digits
			++csp; }
		while(v /= b);
		c = (c & 0x1F) + 1;		// Get fill width
		while(c-- > csp)		// Fill if specified
			*p++ = '0';
		while(csp)				// Output value in ASCII
			*p++ = cs[--csp] + '0'; }
	*p = 0;
	return p;
}

/*
 * Execute the script
 *
 * This is defined as an 'int' function which returns 0 when the script
 * terminates. Other return values can be used by user-defined extensions
 * to halt execution and inform the system the reason (for example, you
 * might be waiting for an event - when the event occurs, the system
 * would simply loop and call "execute()" again to continue the script
 * from where it left off.
 */
int execute()
{
	unsigned i, j, k, v;

	for(;;)	{
		if(Pc >= Ctop)
			error("EXEC above code");
		switch(gb(Pc++)) {
		case XRETURN:				// Return from subroutine
			if(Csp) {					// if stack not empty
				Pc = Cstack[--Csp];			// set PC to return address
				continue; }					// and proceed
			// Fall through to STOP if stack empty
		case XSTOP :				// Halt execution
			return 0;
		case XRESET:				// Reset control stack position
			Csp = eval(0);				// set stack pointer to value
			continue;
		case XEVAL:					// Evaluate (for side effects)
			eval(0);					// process expression, toss result
			continue;
		case XSET :					// Set string variable
			i = gb(Pc++);				// get string variable index
			string();					// process the string
			strcpy(Strings[i], Temp);	// and copy it in
			continue;
		case XCALL:					// Call subroutine
			if(Csp >= CSTACK)			// stack overflow?
				error("Call stack overflow");
			Cstack[Csp++] = Pc + 2;		// save return address
			// Fall into BRANCH to effect transfer to subroutine
		case XBRANCH:				// Transfer to new addres (GOTO)
		xb:	Pc = gw(Pc);				// set PC to following word
			continue;
		case XBFALSE:				// Transfer is false
			if(!eval(0))				// if expression is 0
				goto xb;				// perform branch
			Pc += 2;					// skip ignored transfer address
			continue;
		case XBTRUE:				// Transfer if true
			if(eval(0))					// if expression is !0
				goto xb;				// perform branch
			Pc += 2;					// skip ignored transfer address
			continue;
		case XSWITCH:				// Perform a switch
			v = eval(0);				// get selection value
			i = gw(Pc); Pc += 2;		// get address of case index table
			while(j=gw(i)) {			// get case address, test if !0 (end)
				k = eval(j);			// Evaluate case value
				if(Stack[Sp-1] == v) {	// if match...
					Pc = k;				//   set new pc
					break; }			//   and stop
				i += 2; }				// Skip to next transfer address
			continue;

		//
		// Place handlers for user defined extensions here
		//
		// To receive parameters:
		//	value		: eval(0)				: Numeric expression
		//	string		: string()				: String (in 'Temp')
		//	nvariable	: gb(Pc++)				: Numeric variable index
		//	svariable	: gb(Pc++)				: String variable index
		//	label		: gw(Pc); Pc += 2;		: Code address
		//
		case XPRINT:				// Print a string
			string();					// process string
			fputs(Temp, stdout);		// display
			putc('\n', stdout);			// end with newline
			continue;

		default:
			error("Unknown opcode %02x\n", gb(--Pc)); } }
}	

main(int argc, char *argv[])
{
	unsigned i, j;
	FILE *fp;

	if(argc < 2) {
		fputs("\nUse: ESLGCC <binary-script-filename>\n", stdout);
		return; }

	if(!(fp = fopen(argv[1], "rb"))) {
		printf("Unable to open: %s\n", argv[1]);
		return; }

	// Determine size of code & allocate memory
	Ctop = j = 0;
	while((i = getc(fp)) != EOF)
		++Ctop;
	if(!(Code = malloc(Ctop))) {
		printf("malloc(%u) failed!\n", Ctop);
		fclose(fp);
		return; }

	// Load code into memory
	rewind(fp);
	while((i = getc(fp)) != EOF)
		Code[j++] = i;
	fclose(fp);

	// Note, if you wish to pass start-up parameters to the script,
	// one simple way is to preset values in the global Numeric and
	// String variable slots.
	execute();
}
