/*
 * Sample ESL-16 runtime implementation in DDS Micro-C
 *
 * This is a 16-bit (only) C compiler.
 * 32-bit arithmetic is not required for a 16-bit ESL implementation.
 * Compile with: CC ESLMC16 -fop
 *
 * Dave Dunfield - 2005-2017
 */
#include <stdio.h>

#include "eslopc.h"

// Script execution parameters
#define	VARS		64			// Number of variables
#define	STRS		4			// Number of string buffers
#define	STACK		32			// Evaluation stack size
#define	CSTACK		8			// Call stack size

// User defined system variables
#define	SYS_TICK	0			// 1ms clock tick

// User defined functions
#define	XPRINT		XUSER+0		// PRINT string

unsigned
	Seg,						// External segment
	Stop,						// Top of external segment
	Pc,							// Program counter
	Sp,							// Evaluation stack pointer
	Csp,						// Call stack pointer
	Stack[STACK],				// Evaluation stack
	Cstack[CSTACK],				// Call stack
	Vars[VARS];					// Numeric variables
unsigned char
	Strings[STRS][128],			// String variables
	Temp[256];					// Temp storage (string buffer)

// To be compatible with the 32-bit versions, the 16-bit engine should
// return the tick in MS - however a 16-bit value will overflow every
// 65 seconds - to insure clean timing (at least for the first 65 seconds
// of script execution), we adjust the tick relative to launch time.
unsigned
	BaseTick;					// Base tick value at launch.

// Report and error and terminate
register error(unsigned args)
{
	unsigned char buf[81];
	_format_(nargs() * 2 + &args, buf);
	printf("%04x: %s\n", Pc, buf);
	exit(0);
}

// Get a byte from the external code segment
unsigned gb(off) asm
{
	MOV		ES,DGRP:_Seg	// Get segment
	MOV		BX,4[BP]		// Get offset
	MOV		AL,ES:[BX]		// Retrieve byte
	XOR		AH,AH			// Clear high
}

// Get a 16-bit word from the external code segment
unsigned gw(off) asm
{
	MOV		ES,DGRP:_Seg	// Get segment
	MOV		BX,4[BP]		// Get offset
	MOV		AX,ES:[BX]		// Retrieve word
}

/*
 * Evaluate an encoded RPN numeric expression
 *
 * If argument 'p' is non-zero, it contains the address to process,
 *    ending address is returned and expression result is: Stack[Sp-1]
 * If argument 'p' is zero, decode at PC (and update), result is returned.
 */
unsigned eval(unsigned p)
{
	unsigned v, l, r, *t;
	unsigned char c, f;

	Sp = f = 0;
	if(!p) {		// No offset, assume PC in and out
		p = Pc;
		f = 255; }

	do {
		v = (c = gb(p++)) & 0x1F;
		switch(c & 0x60) {				// Element type
		case 0x00:						// 5-bit value
		s1:	Stack[Sp] = v;				// Stack value
			if(Sp++ >= STACK)			// Test stack overflow
				error("Stack overflow");
			continue;
		case 0x20:						// 13-bit value
			v = (gb(p++) << 5) | v;
			goto s1;
		case 0x40:						// Variable
			v = Vars[v];
			goto s1; }
		if(!(v & 0x18)) switch(v) {		// Special value elements
			default: error("?SYSVAR%u [%x]", v-1, p-1);
			case 0 :		// 16-bit value
				v = gw(p); p += 2;
				goto s1;

			//
			// Place handlers for user-defined system variables here
			//
			case SYS_TICK+1 :		// 55-ms tick
				v = (peekw(0x40, 0x6C) - BaseTick) * 55;
				goto s1; }

		// Arithmetic operation
		r = *(t=&Stack[Sp-1]);			// Get right operand
		if((v -=8) >= ODADD)			// Dyadics require left operand
			l = *(t = &Stack[--Sp - 1]);
		switch(v) {
		default: error("Bad op %02x [%04x]", c, p-1);
		case OMNEG:	*t = -r;				break;
		case OMNOT:	*t = !r;				break;
		case OMCOM:	*t = ~r;				break;
		case ODADD:	*t = l + r;				break;
		case ODSUB:	*t = l - r;				break;
		case ODMUL:	*t = l * r;				break;
		case ODDIV:	*t = l / r;				break;
		case ODMOD:	*t = l % r;				break;
		case ODEQ:	*t = l == r;			break;
		case ODNE:	*t = l != r;			break;
		case ODLT:	*t = l < r;				break;
		case ODGT:	*t = l > r;				break;
		case ODLE:	*t = l <= r;			break;
		case ODGE:	*t = l >= r;			break;
		case ODSL:	*t = l << r;			break;
		case ODSR:	*t = l >> r;			break;
		case ODBAND:*t = l & r;				break;
		case ODBOR:	*t = l | r;				break;
		case ODBXOR:*t = l ^ r;				break;
		case ODLAND:if(l)  *t = r;			break;
		case ODLOR:	if(!l) *t = r;			break;
		case ODASS:	*t = Vars[l] = r;		break;
		case ODIDX:	*t = Vars[l+r];			} }
	while(c & 0x80);		// More elements

	if(f) {					// No offset, save PC and return value
		Pc = p;
		return Stack[Sp-1]; }

	return p;				// Return new offset
}

/*
 * Process a string at the current position
 */
unsigned char *string()
{
	unsigned c, csp, v, b;
	unsigned char *p, *p1, cs[33];

	p = Temp;
	while(c = gb(Pc++)) {
		switch(c & 0xE0) {			// Type of string operation
		default:						// Normal character
			*p++ = c;
			continue;
		case 0x80 :						// Insert string buffer
			p1 = Strings[c & 0x1F];
			while(*p1)
				*p++ = *p1++;
			continue;
		case 0xA0 :	b = 2;		break;	// Special (binary)
		case 0xC0 :	b = 10;		break;	// Decimal
		case 0xE0 : b = 16;		}		// Hexidecimal
		v = eval(csp = 0);			// Eval expression, reset stack
		do {						// Stack digits in reverse order
			if((cs[csp] = v % b) > 9)
				cs[csp] += 7;		// Adjust for hex digits
			++csp; }
		while(v /= b);
		c &= 0x1F;					// Get fill width
		while(c-- > csp)			// Fill if specified
			*p++ = '0';
		while(csp)					// Output value in ASCII
			*p++ = cs[--csp] + '0'; }
	*p = 0;
	return p;
}

/*
 * Execute the script
 *
 * This is defined as an 'int' function which returns 0 when the script
 * terminates. Other return values can be used by user-defined extensions
 * to halt execution and inform the system the reason (for example, you
 * might be waiting for an event - when the event occurs, the system
 * would simply loop and call "execute()" again to continue the script
 * from where it left off.
 */
int execute()
{
	unsigned i, j, k, v;

	for(;;)	{
		if(Pc >= Stop)
			error("EXEC above code");
		switch(gb(Pc++)) {
		case XRETURN:				// Return from subroutine
			if(Csp) {					// if stack not empty
				Pc = Cstack[--Csp];		// set new PC address
				continue; }				// and proceed
			// If stack empty, fall through to STOP
		case XSTOP :				// Halt script
			return 0;
		case XRESET:				// Reset control stack
			Csp = eval(0);				// get value & set stack pointer
			continue;
		case XEVAL:					// Evaluate expression (side effects)
			eval(0);					// evaluate and discard result
			continue;
		case XSET :					// Set string variable
			i = gb(Pc++);				// get string buffer index
			string();					// process the string
			strcpy(Strings[i], Temp);	// and copy it in
			continue;
		case XCALL:					// Call subroutine
			if(Csp >= CSTACK)			// check stack overflow
				error("Call stack overflow");
			Cstack[Csp++] = Pc + 2;		// set return address on stack
			// Fall through to BRANCH to effect transfer to subroutine
		case XBRANCH:				// Unconditional transfer
		xb:	Pc = gw(Pc);				// get address and set new PC
			continue;
		case XBFALSE:				// Transfer if FALSE
			if(!eval(0))				// if value is zero..
				goto xb;					// perform transfer
			Pc += 2;					// skip unused transfer address
			continue;
		case XBTRUE:				// Transfer if TRUE
			if(eval(0))					// if value is NOT zero
				goto xb;					// perform transfer
			Pc += 2;					// skip unused transfer address
			continue;
		case XSWITCH:				// Perform switch
			v = eval(0);				// get selection value
			i = gw(Pc); Pc += 2;		// get case index table address
			while(j=gw(i)) {			// Get case address, 0==end/default
				k = eval(j);			// evaluate case value
				if(Stack[Sp-1] == v) {	// If this is the one...
					Pc = k;					// Set new execution address
					break; }				// and proceed
				i += 2; }				// skip to next case
			continue;
		//
		// Place handlers for user defined extensions here
		//
		// To receive parameters:
		//	value		: eval(0)				: Numeric expression
		//	string		: string()				: String (in 'Temp')
		//	nvariable	: gb(Pc++)				: Numeric variable index
		//	svariable	: gb(Pc++)				: String variable index
		//	label		: gw(Pc); Pc += 2;		: Code address
		//
		case XPRINT:				// Print a string
			string();					// process the string
			fputs(Temp, stdout);		// display
			putc('\n', stdout);			// end with newline
			continue;
		default:
			error("Unknown opcode %02x\n", gb(--Pc)); } }
}	

main(int argc, char *argv[])
{
	unsigned i;
	FILE *fp;

	if(argc < 2) {
		fputs("\nUse: ESLMC16 <binary-script-filename>\n", stdout);
		return; }

	Seg = alloc_seg(4096);	// 64k segment for script code

	// Read script file into memory
	fp = fopen(argv[1], "rvqb");
	while((i = getc(fp)) != EOF)
		poke(Seg, Stop++, i);
	fclose(fp);

	// Note, if you wish to pass start-up parameters to the script,
	// one simple way is to preset values in the global Numeric and
	// String variable slots.
	BaseTick = peekw(0x40, 0x6C);	// Establish base tick
	execute();
}
