/*
 * Sample ESL-32 runtime implementation in DDS Micro-C
 *
 * This is a 16-bit (only) C compiler that uses library functions to
 * perform the 32-bit arithmetic required by a 32-bit ESL implementation.
 * Compile with: CC ESLMC32 -fop
 *
 * Dave Dunfield - 2005-2017
 */
#include <stdio.h>

#include "eslopc.h"				// Get opcode values

// Script execution parameters
#define	VARS		64			// Number of variables
#define	STRS		4			// Number of string buffers
#define	STACK		32			// Evaluation stack size
#define	CSTACK		8			// Call stack size

// User defined system variables
#define	SYS_TICK	0			// System clock tick
// User defined functions
#define	XPRINT		XUSER+0		// PRINT string

unsigned
	Seg,						// External segment
	Stop,						// Top of external segment
	Pc,							// Program counter
	Sp,							// Evaluation stack pointer
	Csp,						// Call stack pointer
	Stack[STACK][2],			// Evaluation stack
	Cstack[CSTACK],				// Call stack
	Vars[VARS][2];				// Numeric variables
unsigned char
	Strings[STRS][128],			// String variables
	Temp[256];					// Temp storage (string buffer)

extern unsigned Longreg[];		// Contains remainder after division

// Report and error and terminate
register error(unsigned args)
{
	unsigned char buf[81];
	_format_(nargs() * 2 + &args, buf);
	printf("%04x: %s\n", Pc, buf);
	exit(0);
}

// Get a byte from the external code segment
unsigned gb(off) asm
{
	MOV		ES,DGRP:_Seg	// Get segment
	MOV		BX,4[BP]		// Get offset
	MOV		AL,ES:[BX]		// Retrieve byte
	XOR		AH,AH			// Clear high
}

// Get a 16-bit word from the external code segment
unsigned gw(off) asm
{
	MOV		ES,DGRP:_Seg	// Get segment
	MOV		BX,4[BP]		// Get offset
	MOV		AX,ES:[BX]		// Retrieve word
}

/*
 * Evaluate an encoded RPN numeric expression
 *
 * Decodes at passed address, returns next address.
 * Evaluation result is located in: Stack[Sp-1]
 */
unsigned eval(unsigned p)
{
	unsigned v, *l, *r, t[2];
	unsigned char c;

	Sp = 0;
	do {
		v = (c = gb(p++)) & 0x1F;
		switch(c & 0x60) {		// Element type
		case 0x00:				// 5-bit constant
			longset(Stack[Sp], v);
	stkl:	if(Sp++ >= STACK)
				error("Stack overflow [%04x]", p);
			continue;
		case 0x20:				// 13-bit constant
			longset(Stack[Sp], (gb(p++) << 5) | v);
			goto stkl;
		case 0x40:				// Variable
			longcpy(Stack[Sp], Vars[v]);
			goto stkl; }
		if(!(v & 0x18)) switch(v) {	// Special value elements
			default: error("?SYSVAR%u [%04x]", v-1, p-1);
			case 0 :			// 32-bit constant
				*Stack[Sp] = gw(p);
				Stack[Sp][1] = gw(p+2);
				p += 4;
				goto stkl;

			//
			// Place handlers for user-defined system variables here
			//
			case SYS_TICK+1:	// Clock TICK
				Stack[Sp][1] = peekw(0x40, 0x6E);
				Stack[Sp][0] = peekw(0x40, 0x6C);
				longset(t, 55);	// 55ms per tick
				longmul(Stack[Sp], t);
				goto stkl; }

		// Arithemetic operation
		r = Stack[Sp-1];
		if((v -= 0x08) >= ODADD)	// Dyadics require left operand
			l = Stack[--Sp - 1];
		switch(v) {
		default: error("Bad op %02x [%04x]", c, p-1);
		case OMNEG:	longset(t, 0); longsub(t, r); longcpy(r, t);	break;
		case OMNOT:	longset(r, !longtst(r));						break;
		case OMCOM:	*r = ~*r; r[1] = ~r[1];							break;
		case OMIDX:	longcpy(r, Vars[*r]);							break;
		case ODADD:	longadd(l, r);									break;
		case ODSUB:	longsub(l, r);									break;
		case ODMUL:	longmul(l, r);									break;
		case ODDIV:	longdiv(l, r);									break;
		case ODMOD:	longdiv(l, r); longcpy(l, Longreg); 			break;
		case ODEQ:	longset(l, longcmp(l, r) == 0);					break;
		case ODNE:	longset(l, longcmp(l, r) != 0);					break;
		case ODLT:	longset(l, longcmp(l, r) < 0);					break;
		case ODGT:	longset(l, longcmp(l, r) > 0);					break;
		case ODLE:	longset(l, longcmp(l, r) <= 0);					break;
		case ODGE:	longset(l, longcmp(l, r) >= 0);					break;
		case ODSL:	while((*r)--) longshl(l);						break;
		case ODSR:	while((*r)--) longshr(l);						break;
		case ODBAND: *l &= *r; l[1] &= r[1];						break;
		case ODBOR:	*l |= *r; l[1] |= r[1];							break;
		case ODBXOR: *l ^= *r; l[1] ^= r[1];						break;
		case ODLAND: if(longtst(l)) longcpy(l, r);					break;
		case ODLOR:	if(!longtst(l))	longcpy(l, r);					break;
		case ODASS:	longcpy(Vars[*l], r); longcpy(l, r);			break;
		case ODIDX:	longadd(l, r); longcpy(l, Vars[*l]);			} }
	while(c & 0x80);	// More elements
	return p;
}

/*
 * Process a string at the current position
 */
void string()
{
	unsigned c, csp, *s, l[2];
	unsigned char *p, *p1, cs[33];
	p = Temp;
	while(c = gb(Pc++)) {
		switch(c & 0xE0) {			// Type of string operation?
		default:						// Normal character
			*p++ = c;
			continue;
		case 0x80 :						// Insert string buffer
			p1 = Strings[c & 0x1F];
			while(*p1)
				*p++ = *p1++;
			continue;
		case 0xA0 :	csp = 2;			// Special (binary in example)
		case 0xC0 : csp = 10;	break;	// Decimal
		case 0xE0 :	csp = 16;	}		// Hexidecimal
		Pc = eval(Pc);				// Evaluate expression
		s = Stack[Sp-1];			// Get result
		longset(l, csp);			// Set long base
		csp = 0;					// Reset stack
		do {						// Stack digits in reverse order
			longdiv(s, l);
			if(*Longreg > 9)
				*Longreg += 7;
			cs[csp++] = *Longreg + '0'; }
		while(longtst(s));
		c = (c & 0x1F) + 1;			// Get fill width
		while(c-- > csp)			// Fill to width
			*p++ = '0';
		do {						// Output to string
			*p++ = cs[--csp]; }
		while(csp); }
	*p = 0;
}

/*
 * Execute the script
 *
 * This is defined as an 'int' function which returns 0 when the script
 * terminates. Other return values can be used by user-defined extensions
 * to halt execution and inform the system the reason (for example, you
 * might be waiting for an event - when the event occurs, the system
 * would simply loop and call "execute()" again to continue the script
 * from where it left off.
 */
int execute()
{
	unsigned i, j, k, l[2];

	for(;;)	{
		if(Pc >= Stop)
			error("EXEC above code");
		switch(gb(Pc++)) {			// Opcode from current PC address
		case XRETURN:				// Return from subroutine
			if(Csp) {					// if stack not empty..
				Pc = Cstack[--Csp];			// set new PC address
				continue; }					// and proceed
			// Fall through to STOP if stack is empty
		case XSTOP :				// Halt execution
			return 0;
		case XRESET:				// Reset stack to specified level
			Pc = eval(Pc);
			Csp = *Stack[Sp-1];
			continue;
		case XEVAL:					// Evaluate expression (for side effects)
			Pc = eval(Pc);
			continue;
		case XSET :					// Set string variable
			i = gb(Pc++);				// get string buffer index
			string();					// process the string
			strcpy(Strings[i], Temp);	// and copy into buffer
			continue;
		case XCALL:					// Call subroutine
			if(Csp >= CSTACK)			// check stack overflow
				error("Call stack overflow");
			Cstack[Csp++] = Pc + 2;		// set return address
			// Fall through to BRANCH to effect transfer to subroutine
		case XBRANCH:				// Unconditional transfer
		xb:	Pc = gw(Pc);				// retrieve and set new PC
			continue;
		case XBFALSE:				// Transfer if FALSE
			Pc = eval(Pc);				// evaluate condition expression
			if(!longtst(Stack[Sp-1]))	// if == ZERO
				goto xb;					// perform branch
			Pc += 2;					// skip unused transfer address
			continue;
		case XBTRUE:				// Transfer if TRUE
			Pc = eval(Pc);				// evaluate condition expression
			if(longtst(Stack[Sp-1]))	// if != ZERO
				goto xb;					// perform branch
			Pc += 2;					// skip unused transfer address
			continue;
		case XSWITCH:				// Perform a  switch
			Pc = eval(Pc);				// evaluate selection value
			longcpy(l, Stack[Sp-1]);	// set 'l' to result (selection value)
			i = gw(Pc); Pc += 2;		// get case index table address
			while(j=gw(i)) {			// get case address, test !0 (end)
				k = eval(j);			// evaluate case value
				if(!longcmp(Stack[Sp-1], l)) {	// if matches
					Pc = k;						// set new PC
					break; }					// and stop
				i += 2; }				// skip to next case
			continue;

		//
		// Place handlers for user defined extensions here
		//
		// To receive parameters:
		//	value		: Pc=eval(Pc); Stack[Sp-1];	: Numeric expression
		//	string		: string()					: String (in 'Temp')
		//	nvariable	: gb(Pc++)					: Numeric variable index
		//	svariable	: gb(Pc++)					: String variable index
		//	label		: gw(Pc); Pc += 2;			: Code address
		//
		case XPRINT:				// Print a string
			string();					// process string
			fputs(Temp, stdout);		// display
			putc('\n', stdout);			// end with newline
			continue;

		default:
			error("Unknown opcode %02x\n", gb(--Pc)); } }
}	

main(int argc, char *argv[])
{
	unsigned i;
	FILE *fp;

	if(argc < 2) {
		fputs("\nUse: ESLMC32 <binary-script-filename>\n", stdout);
		return; }

	Seg = alloc_seg(4096);	// 64k segment for script code

	// Read script file into memory
	fp = fopen(argv[1], "rvqb");
	while((i = getc(fp)) != EOF)
		poke(Seg, Stop++, i);
	fclose(fp);

	// Note, if you wish to pass start-up parameters to the script,
	// one simple way is to preset values in the global Numeric and
	// String variable slots.
	execute();
}
