/*
 * Linux (32-bit) compatibility functions.
 *
 * These emulate the Micro-C "long" math and other functions, enabling code
 * written for 16-bit Micro-C to be compiled under a 32-bit ANSI compiler.
 *
 * This file is normally #included from another source.
 */
#include <stdarg.h>

typedef unsigned U32;

/*
 * Emulate Micro-C "long" math functions which perform 32-bit arithmetic
 * operations by manipulating 2-element arrays of 16-bit elements.
 */
U16 Longreg[2];		// Library places remainder after division

void longadd(U16 l[2], U16 r[2])		// l += r
{
	U32 v1, v2;
	v1 = (l[1] << 16) | l[0];
	v2 = (r[1] << 16) | r[0];
	v1 += v2;
	l[1] = v1 >> 16;
	l[0] = v1;
}
void longsub(U16 l[2], U16 r[2])		// l -= r
{
	U32 v1, v2;
	v1 = (l[1] << 16) | l[0];
	v2 = (r[1] << 16) | r[0];
	v1 -= v2;
	l[1] = v1 >> 16;
	l[0] = v1;
}
void longmul(U16 l[2], U16 r[2])		// l *= r
{
	U32 v1, v2;
	v1 = (l[1] << 16) | l[0];
	v2 = (r[1] << 16) | r[0];
	v1 *= v2;
	l[1] = v1 >> 16;
	l[0] = v1;
}
void longdiv(U16 l[2], U16 r[2])		// Longreg = l % r; l /= r
{
	U32 v1, v2, v3;
	v1 = (l[1] << 16) | l[0];
	v2 = (r[1] << 16) | r[0];
	v3 = v1 % v2;
	v1 /= v2;
	l[1] = v1 >> 16;
	l[0] = v1;
	Longreg[1] = v3 >> 16;
	Longreg[0] = v3;
}
void longshr(U16 l[2])					// l >>= 1
{
	unsigned v;
	v = (l[1] << 16) | l[0];
	v >>= 1;
	l[1] = v >> 16;
	l[0] = v;
}
void longshl(U16 l[2])					// l <<= 1
{
	unsigned v;
	v = (l[1] << 16) | l[0];
	v <<= 1;
	l[1] = v >> 16;
	l[0] = v;
}
void longcpy(U16 l[2], U16 r[2])		// l = r
{
	l[1] = r[1];
	l[0] = r[0];
}
void longset(U16 l[2], U16 v)			// l = (value 16)
{
	l[0] = v;
	l[1] = 0;
}
int longtst(U16 l[2])					// l ? 1 : 0
{
	return (l[1] | l[0]) ? 1 : 0;
}
int longcmp(U16 l[2], U16 r[2])
{
	unsigned v1, v2;
	v1 = (l[1] << 16) | l[0];
	v2 = (r[1] << 16) | r[0];
	if(v1 > v2)
		return 1;
	else if(v1 < v2)
		return -1;
	return 0;
}

/*
 * Emulate Micro-C memory allocation/access functions - Micro-C runs in a
 * real-mode "DOS" model which uses memory "segments".
 */
unsigned char *LINseg;		// Track allocation base address
U16 alloc_seg(U32 size)		// Allocate segment in 16-byte chunks
{
	if(!(LINseg = malloc(size * 16))) {
		printf("malloc failed");
		exit(-1); }
	return 0;
}
#define peek(s, o)  LINseg[(s<<4)+ o]
#define poke(s, o, v) LINseg[(s<<4)+ o] = v
/* U16 peek(unsigned s, unsigned o)	// Retrieve value8 from segment
{
	return LINseg[(s*16)+o];
} */
/*void poke(unsigned s, unsigned o, unsigned char d)	// Write value8 to segment
{
	LINseg[(s*16)+o] = d;
} */
U16 peekw(unsigned s, unsigned o)	// Retrieve value16 from segment
{
	unsigned v;
	o += (s*16);
	v = LINseg[o];
	v |= LINseg[o+1] << 8;
	return v;
}
static inline void pokew(unsigned s, unsigned o, U16 d)	// Write value16 to segment
{
	o += (s*16);
	LINseg[o] = d;
	LINseg[o+1] = d >> 8;
}

/*
 * Other misc. non-standard library functions available in Micro-C and used
 * by the main application.
 */

int strbeg(U8 *str1, U8 *str2)			// Test string1 beginning with string2
{
	while(*str2)
		if(*str1++ != *str2++)
			return 0;
	return 1;
}

// Get line from input file WITHOUT the newline at the end
unsigned char *Xfgets(unsigned char *buf, unsigned len, FILE *fp)
{
	unsigned c, l, f;
	l = f = 0;
	while(l < len) {
		if((c = getc(fp)) == '\n')
			break;
		if(c == EOF) {
			f = 255;
			break; }
		if(c != '\r')
			buf[l++] = c; }
	if(f && !l)
		return 0;
	buf[l] = 0;
	return buf;
}
#define	fgets	Xfgets

/*
 * Format to string routine, format operands are passed
 * as a pointer to the calling functions argument lists.
 */
unsigned _format_(char *output, unsigned len, const char *format, va_list optr)
{
	unsigned width, value, i, aoffset;
	char justify, zero, minus, comma, c, *ptr, *o;
	char outstk[43];

	o = output;
	while(c = *format++) {
		if(c == '%') {					/* format code */
			c = *format++;
			*(ptr = &outstk[42]) = justify = minus = comma = width = value = i = 0;
			zero = ' ';
			if(c == '-') {				/* left justify */
				--justify;
				c = *format++; }
			if(c == '0')					/* leading zeros */
				zero = '0';
			while((c >= '0') && (c <= '9')) {	/* Field width */
				width = (width * 10) + (c - '0');
				c = *format++; }

			value = va_arg(optr, unsigned);

again:		aoffset = 7;
			switch(c) {
			case 'd' :					/* decimal number */
				if(value & 0x80000000) {
					value = -(int)value;
					++minus; }
			case 'u' :					/* unsigned number */
				i = 10;
				break;
			case 'x' :					/* hexidecimal number (LC) */
				aoffset = 39;
			case 'X' :					/* hexidecimal number (UC) */
				i = 16;
				break;
			case 'o' :					/* octal number */
				i = 8;
				break;
			case 'b' :					/* binary number */
				i = 2;
				break;
			case 'c' :					/* character data */
				*--ptr = value;
				break;
			case 's' :					/* string */
				if(!(ptr = (char *) value))
					ptr = "";
				break;
			case ',' :					/* Comma delimiter */
				comma = 4;
				c = *format++;
				goto again;
			default:					/* all others */
				--optr;
				*--ptr = c; }

			if(i) do {	/* for all numbers, generate the ASCII string */
				if(comma) {
					if(!--comma) {
						*--ptr = ',';
						comma = 3; } }
				if((c = (value % i) + '0') > '9')
					c += aoffset;
				*--ptr = c; }
			while(value /= i);

/* output sign if any */
			if(minus) {
				if(len) {
					--len;
					*o++ = '-'; }
				if(width)
					--width; }

/* pad with 'zero' value if right justify enabled  */
			if(width && !justify) {
				for(i = strlen(ptr); i < width; ++i) {
					if(!len) break;
					--len;
					*o++ = zero; } }

/* move in data */
			i = 0;
			value = width - 1;
			while((*ptr) && (i <= value)) {
				if(!len) break;
				--len;
				*o++ = *ptr++;
				++i; }

/* pad with 'zero' value if left justify enabled */
			if(width && justify) {
				while(i < width) {
					if(!len) break;
					--len;
					*o++ = zero;
					++i; } } }
		else {
			if(!len) break;
			--len;
			*o++ = c; } }

	*o = 0;
	return (unsigned)o - (unsigned)output;
}

#if 0
/*
 * Formatted print
 */
void Xprintf(const char *format, ...)
{
	char buffer[150];
	va_list a;
	va_start(a, format);
	_format_(buffer, 149, format, a);
	va_end(a);
	fputs(buffer, stdout);
}
//#define printf Xprintf
void dump(unsigned address)
{
	unsigned c, i, j;
	unsigned char *p;
	for(i=0; i < 16; ++i) {
		printf("\n%08X", address);
		p = (unsigned char*)address;
		address += 16;
		for(j=0; j < 16; ++j) {
			if(!(j & 3))
				putc(' ', stdout);
			printf(" %02X", p[j]); }
		printf("  ");
		for(j=0; j < 16; ++j) {
			c = p[j];
			if((c < ' ') || (c > 0x7E))
				c = '.';
			putc(c, stdout); } }
}
#endif
